<?php
namespace AegisBackup\Libs;

defined( 'ABSPATH' ) || exit;

class AB_DB_Prefix_Migrator {

    public function apply_prefix_change( $prefix_current, $prefix_new, $prefix_mode = 'all' ) {
        global $wpdb;

        $prefix_current = (string) $prefix_current;
        $prefix_new     = (string) $prefix_new;
        $prefix_mode    = (string) $prefix_mode;

        if ( function_exists( 'current_user_can' ) && ! current_user_can( 'manage_options' ) ) {
            return array( 'success' => false, 'message' => 'Insufficient permissions.' );
        }

        if ( function_exists( 'is_multisite' ) && is_multisite() ) {
            return array( 'success' => false, 'message' => 'Prefix migration is disabled on Multisite (safety block).' );
        }

        if ( 'all' !== $prefix_mode ) {
            return array( 'success' => false, 'message' => 'Unsafe prefix mode blocked. Only full-prefix migrations are allowed.' );
        }

        $validation_error = $this->validate_new_prefix_value( $prefix_new, $prefix_current );
        if ( $validation_error ) {
            return array( 'success' => false, 'message' => $validation_error );
        }

        $like = $wpdb->esc_like( $prefix_current ) . '%';
        $rows = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) );

        if ( empty( $rows ) ) {
            return array( 'success' => false, 'message' => 'No tables matched the current prefix. Aborting.' );
        }

        $core_suffixes = $this->get_wp_core_table_suffixes();
        $rename_map    = array();

        foreach ( $rows as $tbl ) {
            $tbl = (string) $tbl;
            if ( 0 !== strpos( $tbl, $prefix_current ) ) {
                continue;
            }

            $suffix = substr( $tbl, strlen( $prefix_current ) );

            if ( 'core' === $prefix_mode ) {
                if ( ! in_array( $suffix, $core_suffixes, true ) ) {
                    continue;
                }
            }

            $rename_map[ $tbl ] = $prefix_new . $suffix;
        }

        if ( empty( $rename_map ) ) {
            return array( 'success' => false, 'message' => 'No tables qualified for renaming under the selected scope.' );
        }

        foreach ( $rename_map as $old => $new ) {
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new ) );
            if ( ! empty( $exists ) ) {
                return array( 'success' => false, 'message' => 'Destination table already exists: ' . $new . '. Aborting to avoid collisions.' );
            }
        }

        $results = array(
            'renamed_tables' => array(),
            'options_rows'   => 0,
            'usermeta_rows'  => 0,
            'wpconfig'       => array( 'updated' => false, 'writable' => false, 'path' => '' ),
            'prefix_current' => $prefix_current,
            'prefix_new'     => $prefix_new,
            'prefix_mode'    => $prefix_mode,
        );

        $renamed_new_to_old = array();

        foreach ( $rename_map as $old => $new ) {
            $q  = 'RENAME TABLE `' . str_replace( '`', '``', $old ) . '` TO `' . str_replace( '`', '``', $new ) . '`';
            $ok = $wpdb->query( $q ); 

            if ( false === $ok ) {
                foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                    $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                    $wpdb->query( $rq ); 
                }

                return array(
                    'success' => false,
                    'message' => 'Failed renaming table ' . $old . ' -> ' . $new . '. Rollback attempted.',
                    'details' => $results,
                );
            }

            $renamed_new_to_old[ $new ] = $old;
            $results['renamed_tables'][] = array( 'from' => $old, 'to' => $new );
        }

        $new_options = $prefix_new . 'options';
        $opt_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new_options ) );
        if ( ! empty( $opt_table_exists ) ) {
            $opt_like = $prefix_current . '%';
            $opt_rows = $wpdb->get_col(
                $wpdb->prepare(
                    'SELECT option_name FROM `' . str_replace( '`', '``', $new_options ) . '` WHERE option_name LIKE %s',
                    $opt_like
                )
            );

            if ( is_array( $opt_rows ) && ! empty( $opt_rows ) ) {
                $updated = 0;
                $skipped = 0;

                foreach ( $opt_rows as $old_key ) {
                    $old_key = (string) $old_key;
                    $suffix  = substr( $old_key, strlen( $prefix_current ) );
                    $new_key = $prefix_new . $suffix;

                    if ( $new_key === $old_key ) {
                        continue;
                    }

                    $exists = (int) $wpdb->get_var(
                        $wpdb->prepare(
                            'SELECT COUNT(*) FROM `' . str_replace( '`', '``', $new_options ) . '` WHERE option_name = %s',
                            $new_key
                        )
                    );

                    if ( $exists > 0 ) {
                        $wpdb->delete( $new_options, array( 'option_name' => $old_key ), array( '%s' ) );
                        $skipped++;
                        continue;
                    }

                    $ok = $wpdb->update(
                        $new_options,
                        array( 'option_name' => $new_key ),
                        array( 'option_name' => $old_key ),
                        array( '%s' ),
                        array( '%s' )
                    );

                    if ( false === $ok ) {
                        foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                            $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                            $wpdb->query( $rq ); 
                        }
                        return array( 'success' => false, 'message' => 'Failed while updating options prefix-bound keys.', 'details' => $results );
                    }

                    $updated++;
                }

                $results['options_rows'] = (int) $updated;
                if ( $skipped > 0 ) {
                    $results['options_collisions_skipped'] = (int) $skipped;
                }
            }
        }

        $new_usermeta = $prefix_new . 'usermeta';
        $um_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new_usermeta ) );
        if ( ! empty( $um_table_exists ) ) {
            $um_like = $prefix_current . '%';
            $um_rows = $wpdb->get_col(
                $wpdb->prepare(
                    'SELECT DISTINCT meta_key FROM `' . str_replace( '`', '``', $new_usermeta ) . '` WHERE meta_key LIKE %s',
                    $um_like
                )
            );

            if ( is_array( $um_rows ) && ! empty( $um_rows ) ) {
                $updated = 0;
                $skipped = 0;

                foreach ( $um_rows as $old_key ) {
                    $old_key = (string) $old_key;
                    $suffix  = substr( $old_key, strlen( $prefix_current ) );
                    $new_key = $prefix_new . $suffix;

                    if ( $new_key === $old_key ) {
                        continue;
                    }

                    $exists = (int) $wpdb->get_var(
                        $wpdb->prepare(
                            'SELECT COUNT(*) FROM `' . str_replace( '`', '``', $new_usermeta ) . '` WHERE meta_key = %s',
                            $new_key
                        )
                    );

                    if ( $exists > 0 ) {
                        $wpdb->delete( $new_usermeta, array( 'meta_key' => $old_key ), array( '%s' ) );
                        $skipped++;
                        continue;
                    }

                    $ok = $wpdb->update(
                        $new_usermeta,
                        array( 'meta_key' => $new_key ),
                        array( 'meta_key' => $old_key ),
                        array( '%s' ),
                        array( '%s' )
                    );

                    if ( false === $ok ) {
                        foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                            $rq = 'RENAME TABLE `' . str_replace( '`', '``', $rolled_new ) . '` TO `' . str_replace( '`', '``', $rolled_old ) . '`';
                            $wpdb->query( $rq ); 
                        }
                        return array( 'success' => false, 'message' => 'Failed while updating usermeta prefix-bound keys.', 'details' => $results );
                    }

                    $updated++;
                }

                $results['usermeta_rows'] = (int) $updated;
                if ( $skipped > 0 ) {
                    $results['usermeta_collisions_skipped'] = (int) $skipped;
                }
            }
        }

        $wpconfig = $this->try_update_wpconfig_table_prefix( $prefix_new );
        $results['wpconfig'] = $wpconfig;

        return array(
            'success' => true,
            'message' => 'Prefix change applied. IMPORTANT: ensure wp-config.php $table_prefix matches the NEW prefix.',
            'details' => $results,
            'manual_wpconfig' => "\$table_prefix = '{$prefix_new}';",
        );
    }

    public function validate_new_prefix_value( $prefix_new, $prefix_current = '' ) {
        $prefix_new     = (string) $prefix_new;
        $prefix_current = (string) $prefix_current;

        if ( '' === $prefix_new ) {
            return 'New prefix cannot be empty.';
        }

        if ( $prefix_new === $prefix_current ) {
            return 'New prefix is the same as the current prefix.';
        }

        if ( strlen( $prefix_new ) > 60 ) {
            return 'New prefix is too long.';
        }

        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $prefix_new ) ) {
            return 'New prefix contains invalid characters. Use letters, numbers, and underscore only.';
        }

        if ( '_' !== substr( $prefix_new, -1 ) ) {
            return 'New prefix should end with an underscore (_) for WordPress compatibility.';
        }

        return '';
    }

    public function try_update_wpconfig_table_prefix( $prefix_new ) {
        $prefix_new = (string) $prefix_new;

        $paths = array(
            ABSPATH . 'wp-config.php',
        );

        $parent = dirname( ABSPATH );
        if ( $parent && is_dir( $parent ) ) {
            $paths[] = trailingslashit( $parent ) . 'wp-config.php';
        }

        foreach ( $paths as $path ) {
            if ( ! file_exists( $path ) ) {
                continue;
            }

            $writable = is_writable( $path );
            $contents = file_get_contents( $path );
            if ( false === $contents ) {
                continue;
            }

            $updated = false;
            $new_contents = preg_replace(
                "/\$table_prefix\s*=\s*['\"]([^'\"]+)['\"];\s*/",
                "\$table_prefix = '{$prefix_new}';\n",
                $contents,
                1,
                $count
            );

            if ( $count > 0 && $writable ) {
                $ok = file_put_contents( $path, $new_contents );
                $updated = ( false !== $ok );
            }

            return array(
                'path' => $path,
                'writable' => (bool) $writable,
                'updated' => (bool) $updated,
            );
        }

        return array( 'path' => '', 'writable' => false, 'updated' => false );
    }

    public function get_wp_core_table_suffixes() {
        return array(
            'options',
            'users',
            'usermeta',
            'posts',
            'postmeta',
            'terms',
            'termmeta',
            'term_taxonomy',
            'term_relationships',
            'comments',
            'commentmeta',
            'links',
        );
    }
}
