<?php
namespace AegisBackup\Modules;

use AegisBackup\Libs\AB_DB_Prefix_Migrator;
use AegisBackup\Backup\AB_Backup_Manager;
use AegisBackup\AB_Plugin;

defined( 'ABSPATH' ) || exit;

require_once AEGISBACKUP_DIR . 'includes/libs/class-ab-db-prefix-migrator.php';
require_once AEGISBACKUP_DIR . 'includes/backup/class-ab-backup-manager.php';

class AB_Module_DB_Tools {

    const ACTIVITY_OPTION = 'aegisbackup_dbtools_activity';

    public function add_activity_log( $action, $result, $context = array() ) {
        $action = sanitize_key( (string) $action );
        $result = sanitize_text_field( (string) $result );
        if ( ! is_array( $context ) ) {
            $context = array();
        }

        $ts = (int) current_time( 'timestamp' );
        $row = array(
            'ts'      => $ts,
            'action'  => $action,
            'result'  => $result,
            'context' => $context,
        );

        $logs = get_option( self::ACTIVITY_OPTION, array() );
        if ( ! is_array( $logs ) ) {
            $logs = array();
        }

        array_unshift( $logs, $row );

        if ( count( $logs ) > 200 ) {
            $logs = array_slice( $logs, 0, 200 );
        }

        update_option( self::ACTIVITY_OPTION, $logs, false );
    }

    public function get_activity_logs( $limit = 50 ) {
        $limit = max( 1, (int) $limit );
        $logs  = get_option( self::ACTIVITY_OPTION, array() );
        if ( ! is_array( $logs ) ) {
            return array();
        }
        return array_slice( $logs, 0, $limit );
    }

    public function get_table_summary() {
        global $wpdb;

        $rows = $wpdb->get_results( 'SHOW TABLE STATUS', ARRAY_A );
        if ( ! is_array( $rows ) ) {
            return array(
                'tables' => array(),
                'totals' => array( 'bytes' => 0, 'human' => $this->format_bytes( 0 ), 'tables' => 0 ),
            );
        }

        $tables = array();
        $total_bytes = 0;

        foreach ( $rows as $r ) {
            $data_len   = isset( $r['Data_length'] ) ? (int) $r['Data_length'] : 0;
            $index_len  = isset( $r['Index_length'] ) ? (int) $r['Index_length'] : 0;
            $size_bytes = (int) ( $data_len + $index_len );
            $total_bytes += $size_bytes;

            $tables[] = array(
                'name'       => (string) ( $r['Name'] ?? '' ),
                'rows'       => (int) ( $r['Rows'] ?? 0 ),
                'size_bytes' => $size_bytes,
                'size'       => $this->format_bytes( $size_bytes ),
                'engine'     => isset( $r['Engine'] ) ? (string) $r['Engine'] : '',
                'collation'  => isset( $r['Collation'] ) ? (string) $r['Collation'] : '',
            );
        }

        return array(
            'tables' => $tables,
            'totals' => array(
                'bytes'  => $total_bytes,
                'human'  => $this->format_bytes( $total_bytes ),
                'tables' => count( $tables ),
            ),
        );
    }

    public function export_csv() {
        $summary = $this->get_table_summary();
        $lines = array();
        $lines[] = array( 'table', 'rows', 'size', 'engine', 'collation' );
        $tables = isset( $summary['tables'] ) && is_array( $summary['tables'] ) ? $summary['tables'] : ( is_array( $summary ) ? $summary : array() );
        foreach ( $tables as $r ) {
            $lines[] = array( $r['name'], (string) $r['rows'], $r['size'], $r['engine'], $r['collation'] );
        }
        $fh = fopen( 'php://temp', 'w+' );
        foreach ( $lines as $line ) {
            fputcsv( $fh, $line );
        }
        rewind( $fh );
        $csv = stream_get_contents( $fh );
        fclose( $fh );
        return (string) $csv;
    }

    public function run_table_op( $op, array $tables ) {
        global $wpdb;
        $op = (string) $op;

        $allowed = array(
            'analyze'  => 'ANALYZE TABLE',
            'optimize' => 'OPTIMIZE TABLE',
            'repair'   => 'REPAIR TABLE',
        );
        if ( ! isset( $allowed[ $op ] ) ) {
            return array( 'ok' => false, 'messages' => array( 'Invalid operation.' ) );
        }

        $tables = array_values( array_filter( array_map( 'sanitize_text_field', $tables ) ) );
        if ( empty( $tables ) ) {
            return array( 'ok' => false, 'messages' => array( 'No tables selected.' ) );
        }

		$msgs    = array();
		$results = array();
        foreach ( $tables as $t ) {
            $sql = $allowed[ $op ] . ' `' . str_replace( '`', '', $t ) . '`';
			$rows = $wpdb->get_results( $sql, ARRAY_A );
			if ( ! empty( $wpdb->last_error ) ) {
				$msgs[] = sprintf( '%s: %s', $t, $wpdb->last_error );
				$results[] = array(
					'table'    => $t,
					'status'   => 'error',
					'msg_type' => 'Error',
					'msg_text' => (string) $wpdb->last_error,
				);
				$wpdb->last_error = '';
				continue;
			}

			if ( is_array( $rows ) && ! empty( $rows ) ) {
				foreach ( $rows as $r ) {
					$results[] = array(
						'table'    => $t,
						'status'   => ( isset( $r['Msg_type'] ) && 'error' === strtolower( (string) $r['Msg_type'] ) ) ? 'error' : 'ok',
						'msg_type' => isset( $r['Msg_type'] ) ? (string) $r['Msg_type'] : 'OK',
						'msg_text' => isset( $r['Msg_text'] ) ? (string) $r['Msg_text'] : 'OK',
					);
				}

				$last = end( $rows );
				$mt = isset( $last['Msg_type'] ) ? (string) $last['Msg_type'] : 'OK';
				$mx = isset( $last['Msg_text'] ) ? (string) $last['Msg_text'] : 'OK';
				$msgs[] = sprintf( '%s: %s - %s', $t, $mt, $mx );
			} else {
				$results[] = array(
					'table'    => $t,
					'status'   => 'ok',
					'msg_type' => 'OK',
					'msg_text' => 'OK',
				);
				$msgs[] = sprintf( '%s: OK', $t );
			}
        }

		return array( 'ok' => true, 'messages' => $msgs, 'results' => $results );
    }

    public function apply_prefix_change( AB_Plugin $plugin, $new_prefix, $backup_first = true ) {
        global $wpdb;

        $new_prefix = (string) $new_prefix;
        $new_prefix = preg_replace( '/[^A-Za-z0-9_]/', '', $new_prefix );
        if ( '' === $new_prefix ) {
            return array( 'ok' => false, 'message' => 'New prefix is required.', 'details' => array() );
        }
        if ( '_' !== substr( $new_prefix, -1 ) ) {
            $new_prefix .= '_';
        }

        $current = (string) $wpdb->prefix;
        if ( $new_prefix === $current ) {
            return array( 'ok' => false, 'message' => 'New prefix matches the current prefix.', 'details' => array() );
        }

        $details = array();

        if ( $backup_first ) {
            $details[] = 'Creating pre-change DB backup...';
            $mgr = new AB_Backup_Manager();
            $job_id = $mgr->start_backup_job( array(
                'include_files'  => false,
                'include_db'     => true,
                'include_config' => false,
                'include_core'   => false,
                'db_export_mode'  => 'table',
                'excludes'       => array(),
            ) );

            $guard = 0;
            while ( $guard < 200 ) {
                $guard++;
                $step = $mgr->process_backup_job( $job_id );
                if ( isset( $step['done'] ) && $step['done'] ) {
                    $details[] = 'Pre-change DB backup created.';
                    break;
                }
                if ( isset( $step['error'] ) && $step['error'] ) {
                    $details[] = 'Pre-change DB backup failed: ' . $step['error'];
                    break;
                }
            }
        }

        $details[] = sprintf( 'Renaming tables from %s to %s ...', $current, $new_prefix );

        $migrator = new AB_DB_Prefix_Migrator();
        $result = $migrator->apply_prefix_change( $current, $new_prefix, 'all' );

        if ( is_array( $result ) && isset( $result['ok'] ) && $result['ok'] ) {
            $details = array_merge( $details, isset( $result['messages'] ) && is_array( $result['messages'] ) ? $result['messages'] : array() );
            return array( 'ok' => true, 'message' => 'Prefix change completed.', 'details' => $details );
        }

        $msg = 'Prefix change failed.';
        if ( is_array( $result ) && isset( $result['message'] ) ) {
            $msg = (string) $result['message'];
        }
        return array( 'ok' => false, 'message' => $msg, 'details' => $details );
    }

    protected function format_bytes( $bytes ) {
        $bytes = (int) $bytes;
        if ( $bytes < 1024 ) { return $bytes . ' B'; }
        $kb = $bytes / 1024;
        if ( $kb < 1024 ) { return number_format_i18n( $kb, 2 ) . ' KB'; }
        $mb = $kb / 1024;
        if ( $mb < 1024 ) { return number_format_i18n( $mb, 2 ) . ' MB'; }
        $gb = $mb / 1024;
        return number_format_i18n( $gb, 2 ) . ' GB';
    }

    public function acquire_run_lock( $key = 'default', $ttl = 600 ) {
        $k = 'aegisbackup_dbtools_lock_' . sanitize_key( $key );
        $now = time();
        $val = get_option( $k );
        if ( is_array( $val ) && isset( $val['expires'] ) && (int) $val['expires'] > $now ) {
            return false;
        }
        update_option( $k, array( 'expires' => $now + (int) $ttl ), false );
        return true;
    }

    public function maybe_run_weekly_optimization() {
        $settings = get_option( 'aegisbackup_dbtools_settings', array() );
        $enabled  = ! empty( $settings['weekly_optimize_enabled'] );
        if ( ! $enabled ) {
            return array( 'success' => false, 'message' => __( 'Weekly optimization is disabled.', 'aegisbackup' ) );
        }

        $last = isset( $settings['weekly_optimize_last'] ) ? (int) $settings['weekly_optimize_last'] : 0;
        if ( $last && ( time() - $last ) < WEEK_IN_SECONDS ) {
            return array( 'success' => false, 'message' => __( 'Weekly optimization already ran recently.', 'aegisbackup' ) );
        }

        $run = $this->run_optimization( 'auto' );
        if ( ! empty( $run['success'] ) ) {
            $settings['weekly_optimize_last'] = time();
            update_option( 'aegisbackup_dbtools_settings', $settings, false );
        }
        return $run;
    }

    public function run_manual_optimization() {
        return $this->run_optimization( 'manual' );
    }

    public function run_optimization( $mode = 'manual' ) {
        global $wpdb;

        if ( ! $this->acquire_run_lock( 'optimize', 900 ) ) {
            return array( 'success' => false, 'message' => __( 'Another DB Tools operation is already running.', 'aegisbackup' ) );
        }

        $summary = $this->get_table_summary();
        $tables  = isset( $summary['tables'] ) ? (array) $summary['tables'] : array();
        if ( empty( $tables ) ) {
            return array( 'success' => false, 'message' => __( 'No tables found.', 'aegisbackup' ) );
        }

        $done = array();
        foreach ( $tables as $t ) {
            $name = isset( $t['name'] ) ? (string) $t['name'] : '';
            if ( empty( $name ) ) { continue; }

            $wpdb->query( "ANALYZE TABLE `{$name}`" );
            $wpdb->query( "OPTIMIZE TABLE `{$name}`" );

            $done[] = $name;
        }

        return array(
            'success' => true,
            'message' => ( 'auto' === $mode ) ? __( 'Weekly optimization completed.', 'aegisbackup' ) : __( 'Optimization completed.', 'aegisbackup' ),
            'tables'  => $done,
        );
    }

    public function get_db_tools_backup_dir() {
        $upload = wp_upload_dir();
        $base   = isset( $upload['basedir'] ) ? (string) $upload['basedir'] : '';
        if ( empty( $base ) ) {
            return '';
        }

        $dir = trailingslashit( $base ) . 'aegisbackup/db-tools';

        if ( ! file_exists( $dir ) ) {
            wp_mkdir_p( $dir );
        }

        $index = trailingslashit( $dir ) . 'index.html';
        if ( ! file_exists( $index ) ) {
            @file_put_contents( $index, '' );
        }

        $htaccess = trailingslashit( $dir ) . '.htaccess';
        if ( ! file_exists( $htaccess ) ) {
            @file_put_contents( $htaccess, "Deny from all\n" );
        }

        return $dir;
    }

    public function create_db_backup_snapshot() {
        global $wpdb;

        $dir = $this->get_db_tools_backup_dir();
        if ( empty( $dir ) || ! is_dir( $dir ) || ! is_writable( $dir ) ) {
            return array(
                'success' => false,
                'message' => __( 'Backup directory is not writable.', 'aegisbackup' ),
            );
        }

        $timestamp = gmdate( 'Y-m-d_H-i-s' );
        $db_name   = defined( 'DB_NAME' ) ? (string) DB_NAME : 'database';
        $prefix    = isset( $wpdb->prefix ) ? (string) $wpdb->prefix : '';

        $safe_db   = preg_replace( '/[^A-Za-z0-9_\-]/', '_', $db_name );
        $safe_pref = preg_replace( '/[^A-Za-z0-9_\-]/', '_', $prefix );

        $filename = 'aegisbackup-db-backup-' . $safe_db . '-' . $safe_pref . '-' . $timestamp . '.sql';
        $path     = trailingslashit( $dir ) . $filename;

        $fh = @fopen( $path, 'w' );
        if ( ! $fh ) {
            return array( 'success' => false, 'message' => __( 'Unable to create backup file.', 'aegisbackup' ) );
        }

        @fwrite( $fh, "-- AegisBackup DB Tools Snapshot\n" );
        @fwrite( $fh, '-- Generated: ' . gmdate( 'c' ) . "\n\n" );

        $tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $prefix ) . '%' ) );
        if ( empty( $tables ) ) {
            @fclose( $fh );
            @unlink( $path );
            return array( 'success' => false, 'message' => __( 'No tables found for this site prefix.', 'aegisbackup' ) );
        }

        foreach ( $tables as $table ) {
            $table = (string) $table;
            @fwrite( $fh, "\n-- ----------------------------\n" );
            @fwrite( $fh, "-- Table structure for `{$table}`\n" );
            @fwrite( $fh, "-- ----------------------------\n" );
            $create = $wpdb->get_row( "SHOW CREATE TABLE `{$table}`", ARRAY_N );
            if ( isset( $create[1] ) ) {
                @fwrite( $fh, "DROP TABLE IF EXISTS `{$table}`;\n" );
                @fwrite( $fh, $create[1] . ";\n" );
            }

            @fwrite( $fh, "\n-- ----------------------------\n" );
            @fwrite( $fh, "-- Data for `{$table}`\n" );
            @fwrite( $fh, "-- ----------------------------\n" );

            $offset = 0;
            $limit  = 1000;
            while ( true ) {
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `{$table}` LIMIT %d OFFSET %d", $limit, $offset ), ARRAY_A );
                if ( empty( $rows ) ) {
                    break;
                }

                foreach ( $rows as $row ) {
                    $cols = array();
                    $vals = array();
                    foreach ( $row as $col => $val ) {
                        $cols[] = '`' . str_replace( '`', '``', (string) $col ) . '`';
                        $vals[] = $this->sql_escape_value( $val );
                    }
                    $sql = 'INSERT INTO `' . $table . '` (' . implode( ',', $cols ) . ') VALUES (' . implode( ',', $vals ) . ");\n";
                    @fwrite( $fh, $sql );
                }

                $offset += $limit;
            }
        }

        @fclose( $fh );

        return array(
            'success'  => true,
            'message'  => __( 'DB snapshot created.', 'aegisbackup' ),
            'file'     => $filename,
            'path'     => $path,
            'filesize' => @filesize( $path ),
        );
    }

    public function list_db_backups() {
        $dir = $this->get_db_tools_backup_dir();
        if ( empty( $dir ) || ! is_dir( $dir ) ) {
            return array();
        }

        $items = glob( trailingslashit( $dir ) . '*.sql' );
        if ( empty( $items ) ) {
            return array();
        }

        $out = array();
        foreach ( $items as $p ) {
            $p = (string) $p;
            $out[] = array(
                'file' => basename( $p ),
                'path' => $p,
                'size' => @filesize( $p ),
                'mtime'=> @filemtime( $p ),
            );
        }

        usort( $out, function( $a, $b ) {
            return (int) $b['mtime'] <=> (int) $a['mtime'];
        } );

        return $out;
    }

    public function delete_db_backup_snapshot( $file ) {
        $file = sanitize_file_name( (string) $file );
        if ( empty( $file ) || false === strpos( $file, '.sql' ) ) {
            return array( 'success' => false, 'message' => __( 'Invalid snapshot file.', 'aegisbackup' ) );
        }

        $dir = $this->get_db_tools_backup_dir();
        if ( empty( $dir ) ) {
            return array( 'success' => false, 'message' => __( 'Snapshot directory not available.', 'aegisbackup' ) );
        }

        $path = trailingslashit( $dir ) . $file;
        if ( ! file_exists( $path ) ) {
            return array( 'success' => false, 'message' => __( 'Snapshot file not found.', 'aegisbackup' ) );
        }

        if ( ! @unlink( $path ) ) {
            return array( 'success' => false, 'message' => __( 'Unable to delete snapshot file.', 'aegisbackup' ) );
        }

        return array( 'success' => true, 'message' => __( 'Snapshot deleted.', 'aegisbackup' ) );
    }

    public function restore_db_backup_snapshot( $file ) {
        global $wpdb;

        $file = sanitize_file_name( (string) $file );
        if ( empty( $file ) || false === strpos( $file, '.sql' ) ) {
            return array( 'success' => false, 'message' => __( 'Invalid snapshot file.', 'aegisbackup' ) );
        }

        if ( ! $this->acquire_run_lock( 'restore_snapshot', 1800 ) ) {
            return array( 'success' => false, 'message' => __( 'Another DB Tools operation is already running.', 'aegisbackup' ) );
        }

        $dir = $this->get_db_tools_backup_dir();
        if ( empty( $dir ) ) {
            return array( 'success' => false, 'message' => __( 'Snapshot directory not available.', 'aegisbackup' ) );
        }

        $path = trailingslashit( $dir ) . $file;
        if ( ! file_exists( $path ) || ! is_readable( $path ) ) {
            return array( 'success' => false, 'message' => __( 'Snapshot file not found or not readable.', 'aegisbackup' ) );
        }

        $fh = @fopen( $path, 'r' );
        if ( ! $fh ) {
            return array( 'success' => false, 'message' => __( 'Unable to open snapshot file.', 'aegisbackup' ) );
        }

        $messages = array();
        $messages[] = sprintf( 'Restoring snapshot: %s', $file );

        $buffer = '';
        $executed = 0;
        $errors = 0;

        while ( ( $line = fgets( $fh ) ) !== false ) {
            $line = (string) $line;
            if ( preg_match( '/^\s*(--|\/\*|#)/', $line ) ) {
                continue;
            }
            if ( trim( $line ) === '' ) {
                continue;
            }

            $buffer .= $line;
            if ( substr( rtrim( $line ), -1 ) !== ';' ) {
                continue;
            }

            $sql = trim( $buffer );
            $buffer = '';

            if ( '' === $sql ) {
                continue;
            }

            $wpdb->query( $sql );
            $executed++;
            if ( ! empty( $wpdb->last_error ) ) {
                $errors++;
                $messages[] = 'SQL error: ' . $wpdb->last_error;
            }
        }

        @fclose( $fh );

        $messages[] = sprintf( 'Statements executed: %d', (int) $executed );
        if ( $errors ) {
            $messages[] = sprintf( 'Errors: %d (see above)', (int) $errors );
            return array( 'success' => false, 'message' => __( 'Snapshot restore completed with errors.', 'aegisbackup' ), 'messages' => $messages );
        }

        return array( 'success' => true, 'message' => __( 'Snapshot restored successfully.', 'aegisbackup' ), 'messages' => $messages );
    }

    public function sql_escape_value( $val ) {
        if ( is_null( $val ) ) {
            return 'NULL';
        }
        if ( is_bool( $val ) ) {
            return $val ? '1' : '0';
        }
        if ( is_int( $val ) || is_float( $val ) ) {
            return (string) $val;
        }

        $s = (string) $val;
        $s = str_replace( "\0", "\\0", $s );
        $s = str_replace( "\n", "\\n", $s );
        $s = str_replace( "\r", "\\r", $s );
        $s = str_replace( "\x1a", "\\Z", $s );
        $s = str_replace( "'", "\\'", $s );
        $s = str_replace( '"', '\\"', $s );

        return "'{$s}'";
    }

    public function validate_new_prefix_value( $prefix_new ) {
        $prefix_new = (string) $prefix_new;
        if ( empty( $prefix_new ) ) {
            return array( 'valid' => false, 'message' => __( 'Prefix cannot be empty.', 'aegisbackup' ) );
        }
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $prefix_new ) ) {
            return array( 'valid' => false, 'message' => __( 'Prefix can only contain letters, numbers, and underscores.', 'aegisbackup' ) );
        }
        if ( '_' !== substr( $prefix_new, -1 ) ) {
            return array( 'valid' => false, 'message' => __( 'Prefix must end with an underscore (_).', 'aegisbackup' ) );
        }
        return array( 'valid' => true, 'message' => __( 'Prefix looks valid.', 'aegisbackup' ) );
    }

    public function build_prefix_preview_payload( $prefix_current, $prefix_new ) {
        global $wpdb;

        $prefix_current = (string) $prefix_current;
        $prefix_new     = (string) $prefix_new;

        if ( '' === $prefix_current ) {
            $prefix_current = (string) $wpdb->prefix;
        }

        $prefix_new = preg_replace( '/[^A-Za-z0-9_]/', '', $prefix_new );
        if ( '_' !== substr( $prefix_new, -1 ) ) {
            $prefix_new .= '_';
        }

        $val = $this->validate_new_prefix_value( $prefix_new );
        if ( empty( $val['valid'] ) ) {
            return array( 'ok' => false, 'message' => isset( $val['message'] ) ? $val['message'] : __( 'Invalid prefix.', 'aegisbackup' ) );
        }
        if ( $prefix_new === $prefix_current ) {
            return array( 'ok' => false, 'message' => __( 'The new prefix is the same as the current prefix.', 'aegisbackup' ) );
        }

        $like  = $wpdb->esc_like( $prefix_current ) . '%';
        $tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) );
        if ( empty( $tables ) ) {
            return array( 'ok' => false, 'message' => __( 'No tables matched the current prefix.', 'aegisbackup' ) );
        }

        $preview_tables = array();
        foreach ( $tables as $tbl ) {
            $tbl = (string) $tbl;
            if ( 0 !== strpos( $tbl, $prefix_current ) ) {
                continue;
            }
            $suffix = substr( $tbl, strlen( $prefix_current ) );
            $new    = $prefix_new . $suffix;
            $preview_tables[] = array(
                'current' => $tbl,
                'new'     => $new,
                'type'    => 'All',
            );
        }

        $collisions = array();
        foreach ( $preview_tables as $row ) {
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $row['new'] ) );
            if ( ! empty( $exists ) ) {
                $collisions[] = $row['new'];
            }
        }

        $preview_options = $this->build_prefix_option_keys_preview( $prefix_current, $prefix_new );
        $preview_um      = $this->build_prefix_usermeta_keys_preview( $prefix_current, $prefix_new );
        $wpconfig_info   = $this->get_wpconfig_prefix_info( $prefix_new );

        return array(
            'ok'        => empty( $collisions ),
            'message'   => empty( $collisions ) ? __( 'Preview generated.', 'aegisbackup' ) : __( 'Blocking issue: one or more destination tables already exist for the new prefix.', 'aegisbackup' ),
            'current'   => $prefix_current,
            'new'       => $prefix_new,
            'tables'    => $preview_tables,
            'options'   => $preview_options,
            'usermeta'  => $preview_um,
            'wpconfig'  => $wpconfig_info,
            'collisions'=> $collisions,
        );
    }

    protected function build_prefix_option_keys_preview( $prefix_current, $prefix_new ) {
        global $wpdb;

        $rows = array();
        $options_table = $prefix_current . 'options';
        $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $options_table ) );
        if ( $exists !== $options_table ) {
            return $rows;
        }

        $like = $wpdb->esc_like( $prefix_current ) . '%';
        $names = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM `" . str_replace( '`', '``', $options_table ) . "` WHERE option_name LIKE %s ORDER BY option_name ASC",
                $like
            )
        );

        if ( empty( $names ) ) {
            return $rows;
        }

        foreach ( $names as $name ) {
            $name = (string) $name;
            if ( 0 !== strpos( $name, $prefix_current ) ) {
                continue;
            }
            $rows[] = array(
                'current' => $name,
                'new'     => $prefix_new . substr( $name, strlen( $prefix_current ) ),
            );
        }

        $roles_key = $prefix_current . 'user_roles';
        if ( in_array( $roles_key, $names, true ) ) {
            $rows[] = array(
                'current' => $roles_key,
                'new'     => $prefix_new . 'user_roles',
            );
        }

        $uniq = array();
        $out  = array();
        foreach ( $rows as $r ) {
            $k = $r['current'] . '=>' . $r['new'];
            if ( isset( $uniq[ $k ] ) ) {
                continue;
            }
            $uniq[ $k ] = true;
            $out[] = $r;
        }

        return $out;
    }

    protected function build_prefix_usermeta_keys_preview( $prefix_current, $prefix_new ) {
        global $wpdb;

        $rows = array();
        $um_table = $prefix_current . 'usermeta';
        $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $um_table ) );
        if ( $exists !== $um_table ) {
            return $rows;
        }

        $like = $wpdb->esc_like( $prefix_current ) . '%';
        $keys = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT DISTINCT meta_key FROM `" . str_replace( '`', '``', $um_table ) . "` WHERE meta_key LIKE %s ORDER BY meta_key ASC",
                $like
            )
        );

        if ( empty( $keys ) ) {
            return $rows;
        }

        foreach ( $keys as $k ) {
            $k = (string) $k;
            if ( 0 !== strpos( $k, $prefix_current ) ) {
                continue;
            }
            $rows[] = array(
                'current' => $k,
                'new'     => $prefix_new . substr( $k, strlen( $prefix_current ) ),
            );
        }

        return $rows;
    }

    protected function get_wpconfig_prefix_info( $prefix_new ) {
        $info = array();

        $path = $this->detect_wpconfig_path();
        if ( ! $path ) {
            return $info;
        }

        $info['path']     = $path;
        $info['writable'] = is_writable( $path );
        $info['planned']  = (string) $prefix_new;

        $contents = @file_get_contents( $path ); 
        if ( is_string( $contents ) && preg_match( '/\$table_prefix\s*=\s*["\']([^"\']+)["\']\s*;/', $contents, $m ) ) {
            $info['current'] = isset( $m[1] ) ? (string) $m[1] : '';
        }

        $info['manual_line'] = "\$table_prefix = '{$prefix_new}';";
        return $info;
    }

    protected function detect_wpconfig_path() {
        $paths = array();
        if ( defined( 'ABSPATH' ) ) {
            $paths[] = rtrim( ABSPATH, '/\\' ) . '/wp-config.php';
            $paths[] = rtrim( dirname( ABSPATH ), '/\\' ) . '/wp-config.php';
        }
        foreach ( $paths as $p ) {
            if ( file_exists( $p ) ) {
                return $p;
            }
        }
        return '';
    }

    public function run_prefix_verification_scan_strict( $prefix_expected, $prefix_old = '' ) {
        global $wpdb;

        $prefix_expected = (string) $prefix_expected;
        $prefix_old      = (string) $prefix_old;

        if ( '' === $prefix_expected ) {
            return array( 'issues' => array( __( 'Expected prefix is required.', 'aegisbackup' ) ), 'warnings' => array(), 'samples' => array() );
        }

        $issues   = array();
        $warnings = array();
        $samples  = array();
        $expected_tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $prefix_expected ) . '%' ) );
        if ( empty( $expected_tables ) ) {
            $issues[] = __( 'No tables found for the expected prefix.', 'aegisbackup' );
        } else {
            $samples['expected_tables_count'] = count( $expected_tables );
        }

        $core_suffixes = array( 'options', 'users', 'usermeta', 'posts', 'postmeta' );
        foreach ( $core_suffixes as $suffix ) {
            $t = $prefix_expected . $suffix;
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $t ) );
            if ( $exists !== $t ) {
                $issues[] = sprintf( __( 'Missing core table under expected prefix: %s', 'aegisbackup' ), $t );
            }
        }

        if ( '' !== $prefix_old && $prefix_old !== $prefix_expected ) {
            $old_tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $prefix_old ) . '%' ) );
            if ( ! empty( $old_tables ) ) {
                $blocking = array();
                foreach ( $old_tables as $ot ) {
                    foreach ( $core_suffixes as $suffix ) {
                        if ( $ot === $prefix_old . $suffix ) {
                            $blocking[] = $ot;
                        }
                    }
                }
                if ( ! empty( $blocking ) ) {
                    $issues[] = __( 'Core tables still exist under the old prefix (migration likely incomplete).', 'aegisbackup' );
                    $samples['old_core_tables'] = array_slice( $blocking, 0, 25 );
                } else {
                    $warnings[] = sprintf( __( 'Some tables still exist under the old prefix (%d). Verify they are not needed or belong to another install.', 'aegisbackup' ), count( $old_tables ) );
                    $samples['old_tables'] = array_slice( $old_tables, 0, 25 );
                }
            }
        }

        $opt = $prefix_expected . 'options';
        $um  = $prefix_expected . 'usermeta';

        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $opt ) ) === $opt ) {
            $like_old = $wpdb->esc_like( $prefix_old ? $prefix_old : $wpdb->prefix ) . '%';
            $cnt = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `" . str_replace( '`', '``', $opt ) . "` WHERE option_name LIKE %s", $like_old ) );
            if ( $cnt > 0 ) {
                $issues[] = sprintf( __( 'Detected %d option_name rows still using the old prefix inside expected options table.', 'aegisbackup' ), $cnt );
                $samples['option_name_old_prefix_count'] = $cnt;
            }
        } else {
            $issues[] = __( 'Expected options table not found; cannot verify prefix-bound option keys.', 'aegisbackup' );
        }

        if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $um ) ) === $um ) {
            $like_old = $wpdb->esc_like( $prefix_old ? $prefix_old : $wpdb->prefix ) . '%';
            $cnt = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `" . str_replace( '`', '``', $um ) . "` WHERE meta_key LIKE %s", $like_old ) );
            if ( $cnt > 0 ) {
                $issues[] = sprintf( __( 'Detected %d meta_key rows still using the old prefix inside expected usermeta table.', 'aegisbackup' ), $cnt );
                $samples['meta_key_old_prefix_count'] = $cnt;
            }
        } else {
            $issues[] = __( 'Expected usermeta table not found; cannot verify prefix-bound usermeta keys.', 'aegisbackup' );
        }

        return array(
            'issues'   => $issues,
            'warnings' => $warnings,
            'samples'  => $samples,
        );
    }

    public function run_prefix_verification_scan( $prefix_expected ) {
        global $wpdb;
        $prefix_expected = (string) $prefix_expected;

        $results = array();
        $tables  = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $prefix_expected ) . '%' ) );
        if ( empty( $tables ) ) {
            $results[] = array( 'level' => 'error', 'message' => __( 'No tables found for expected prefix.', 'aegisbackup' ) );
        } else {
            $results[] = array( 'level' => 'ok', 'message' => sprintf( __( 'Found %d tables with expected prefix.', 'aegisbackup' ), count( $tables ) ) );
        }

        $opt = $prefix_expected . 'options';
        $um  = $prefix_expected . 'usermeta';

        $opt_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $opt ) );
        $um_exists  = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $um ) );

        if ( $opt_exists ) {
            $bad = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$opt}` WHERE option_name LIKE %s", $wpdb->esc_like( $wpdb->prefix ) . '%') );
            $results[] = array( 'level' => 'info', 'message' => sprintf( __( 'Options table present. (Scan note: %d prefixed option_name rows found for current runtime prefix.)', 'aegisbackup' ), $bad ) );
        }
        if ( $um_exists ) {
            $bad2 = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$um}` WHERE meta_key LIKE %s", $wpdb->esc_like( $wpdb->prefix ) . '%') );
            $results[] = array( 'level' => 'info', 'message' => sprintf( __( 'Usermeta table present. (Scan note: %d prefixed meta_key rows found for current runtime prefix.)', 'aegisbackup' ), $bad2 ) );
        }

        return array( 'success' => true, 'results' => $results );
    }

    public function try_update_wpconfig_table_prefix( $prefix_new ) {
        $prefix_new = (string) $prefix_new;

        $wp_config_path = ABSPATH . 'wp-config.php';
        if ( ! file_exists( $wp_config_path ) ) {
            return array( 'success' => false, 'message' => __( 'wp-config.php not found.', 'aegisbackup' ) );
        }
        if ( ! is_writable( $wp_config_path ) ) {
            return array( 'success' => false, 'message' => __( 'wp-config.php is not writable. Update $table_prefix manually.', 'aegisbackup' ) );
        }

        $contents = @file_get_contents( $wp_config_path );
        if ( false === $contents ) {
            return array( 'success' => false, 'message' => __( 'Unable to read wp-config.php.', 'aegisbackup' ) );
        }

        $pattern = '/\$table_prefix\s*=\s*[\"\']([^\"\']+)[\"\']\s*;/';
        if ( preg_match( $pattern, $contents ) ) {
            $contents2 = preg_replace( $pattern, "\$table_prefix = '{$prefix_new}';", $contents, 1 );
            if ( null === $contents2 ) {
                return array( 'success' => false, 'message' => __( 'Unable to update wp-config.php prefix.', 'aegisbackup' ) );
            }
            @file_put_contents( $wp_config_path, $contents2 );
            return array( 'success' => true, 'message' => __( 'wp-config.php updated.', 'aegisbackup' ) );
        }

        return array( 'success' => false, 'message' => __( 'Could not locate $table_prefix line in wp-config.php.', 'aegisbackup' ) );
    }

    public function maybe_check_growth( $force = false ) {
        $settings = get_option( 'aegisbackup_dbtools_settings', array() );
        $enabled  = ! empty( $settings['growth_monitor_enabled'] );
        if ( ! $enabled && ! $force ) {
            return array( 'success' => false, 'message' => __( 'Growth monitor is disabled.', 'aegisbackup' ) );
        }

        $summary = $this->get_table_summary();
        $total   = isset( $summary['totals']['bytes'] ) ? (int) $summary['totals']['bytes'] : 0;
        $prev    = isset( $settings['growth_last_total'] ) ? (int) $settings['growth_last_total'] : 0;
        $settings['growth_last_total'] = $total;
        $settings['growth_last_check'] = time();
        update_option( 'aegisbackup_dbtools_settings', $settings, false );

        $delta = $total - $prev;
        return array(
            'success' => true,
            'message' => __( 'Growth check recorded.', 'aegisbackup' ),
            'total'   => $total,
            'delta'   => $delta,
        );
    }

}
