<?php
namespace AegisBackup\Push;

defined( 'ABSPATH' ) || exit;

class AB_Push_Manager {
    const JOB_OPTION_PREFIX = 'aegisbackup_push_job_';

    public function start_push_job( array $args ) {
        $job_id = 'abp_' . wp_generate_password( 12, false, false );

        $token = isset( $args['token'] ) ? $args['token'] : array();
        $package_path = isset( $args['package_path'] ) ? (string) $args['package_path'] : '';
		$src_root = isset( $args['src_root'] ) ? (string) $args['src_root'] : '';
		$dst_root = isset( $args['dst_root'] ) ? (string) $args['dst_root'] : '';
		$paths    = isset( $args['paths'] ) && is_array( $args['paths'] ) ? (array) $args['paths'] : array();
		$purpose  = isset( $args['purpose'] ) ? sanitize_key( (string) $args['purpose'] ) : 'package';

        if ( empty( $token['endpoint'] ) || empty( $token['id'] ) || empty( $token['exp'] ) || empty( $token['sig'] ) ) {
            return array();
        }
        if ( empty( $package_path ) || ! is_file( $package_path ) ) {
            return array();
        }

        $size = (int) filesize( $package_path );
        $chunk_size = 512 * 1024; // 512KB per request (shared hosting friendly)
        $total_chunks = (int) ceil( $size / $chunk_size );
        $file_id = 'push_' . gmdate( 'YmdHis' ) . '_' . substr( $job_id, -6 );

        $state = array(
            'job_id' => $job_id,
            'status' => 'running',
            'created' => time(),
            'token' => $token,
            'package_path' => $package_path,
			'src_root' => $src_root,
			'dst_root' => $dst_root,
			'paths'    => $paths,
			'purpose'  => $purpose,
            'file_id' => $file_id,
            'size' => $size,
            'chunk_size' => $chunk_size,
            'total_chunks' => $total_chunks,
            'chunk_index' => 0,
            'offset' => 0,
            'progress' => 0,
            'phase' => 'send',
            'last_error' => '',
        );

        update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );

        return array(
            'job_id' => $job_id,
            'file_id' => $file_id,
            'total_chunks' => $total_chunks,
        );
    }

    public function process_push_job( $job_id ) {
        $state = get_option( self::JOB_OPTION_PREFIX . $job_id, array() );
        if ( empty( $state['job_id'] ) ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'Push job not found.' );
        }

        if ( 'done' === $state['phase'] ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'Push complete.' );
        }

        $token = (array) $state['token'];
        $endpoint = rtrim( (string) $token['endpoint'], '/' );
        $receive_url = $endpoint . '/wp-json/aegisbackup/v1/receive-chunk';
        $commit_url  = $endpoint . '/wp-json/aegisbackup/v1/commit';

        $package_path = (string) $state['package_path'];
        if ( ! is_file( $package_path ) ) {
            $state['phase'] = 'done';
            $state['status'] = 'error';
            update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
            return array( 'done' => true, 'progress' => (int) $state['progress'], 'log' => 'Package missing on source.' );
        }

        if ( 'send' === $state['phase'] ) {
            $chunk_index = (int) $state['chunk_index'];
            $chunk_size = (int) $state['chunk_size'];
            $offset = (int) $state['offset'];
            $size = (int) $state['size'];

            if ( $offset >= $size ) {
                $state['phase'] = 'commit';
                update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
                return array( 'done' => false, 'progress' => 98, 'log' => 'All chunks sent. Committing…' );
            }

            $fh = fopen( $package_path, 'rb' );
            if ( ! $fh ) {
                $state['status'] = 'error';
                $state['phase'] = 'done';
                update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
                return array( 'done' => true, 'progress' => (int) $state['progress'], 'log' => 'Failed to open package on source.' );
            }
            fseek( $fh, $offset );
            $data = fread( $fh, $chunk_size );
            fclose( $fh );

            if ( false === $data || '' === $data ) {
                $state['phase'] = 'commit';
                update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
                return array( 'done' => false, 'progress' => 98, 'log' => 'Chunk read complete. Committing…' );
            }

            $headers = array(
                'Content-Type' => 'application/octet-stream',
                'X-AB-Token-Id' => (string) $token['id'],
                'X-AB-Token-Exp' => (string) $token['exp'],
                'X-AB-Token-Sig' => (string) $token['sig'],
                'X-AB-File-Id' => (string) $state['file_id'],
                'X-AB-Chunk-Index' => (string) $chunk_index,
                'X-AB-Total-Chunks' => (string) $state['total_chunks'],
                'X-AB-Total-Size' => (string) $size,
				'X-AB-Purpose' => isset( $state['purpose'] ) ? (string) $state['purpose'] : 'package',
            );

            $resp = wp_remote_post(
                $receive_url,
                array(
                    'timeout' => 25,
                    'redirection' => 0,
                    'headers' => $headers,
                    'body' => $data,
                )
            );

            if ( is_wp_error( $resp ) ) {
                $state['last_error'] = $resp->get_error_message();
                update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
                return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'Push error: ' . $state['last_error'] );
            }

            $code = (int) wp_remote_retrieve_response_code( $resp );
            if ( $code < 200 || $code >= 300 ) {
                $body = (string) wp_remote_retrieve_body( $resp );
                $state['last_error'] = 'HTTP ' . $code . ' ' . substr( $body, 0, 200 );
                update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
                return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'Push rejected: ' . $state['last_error'] );
            }

            $state['chunk_index'] = $chunk_index + 1;
            $state['offset'] = $offset + strlen( $data );
            $state['progress'] = (int) ( ( $state['offset'] / max( 1, $size ) ) * 95 );
            update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );

            return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'Sent chunk ' . (int) $chunk_index . ' / ' . (int) $state['total_chunks'] );
        }

        if ( 'commit' === $state['phase'] ) {
            $headers = array(
                'Content-Type' => 'application/json',
                'X-AB-Token-Id' => (string) $token['id'],
                'X-AB-Token-Exp' => (string) $token['exp'],
                'X-AB-Token-Sig' => (string) $token['sig'],
            );

            $payload = array(
                'file_id' => (string) $state['file_id'],
                'total_chunks' => (int) $state['total_chunks'],
                'total_size' => (int) $state['size'],
				'purpose' => isset( $state['purpose'] ) ? (string) $state['purpose'] : 'package',
				'src_root' => isset( $state['src_root'] ) ? (string) $state['src_root'] : '',
				'dst_root' => isset( $state['dst_root'] ) ? (string) $state['dst_root'] : '',
				'paths'    => isset( $state['paths'] ) && is_array( $state['paths'] ) ? (array) $state['paths'] : array(),
            );

            $resp = wp_remote_post(
                $commit_url,
                array(
                    'timeout' => 25,
                    'redirection' => 0,
                    'headers' => $headers,
                    'body' => wp_json_encode( $payload ),
                )
            );

            if ( is_wp_error( $resp ) ) {
                $state['last_error'] = $resp->get_error_message();
                update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
                return array( 'done' => false, 'progress' => 98, 'log' => 'Commit error: ' . $state['last_error'] );
            }

            $code = (int) wp_remote_retrieve_response_code( $resp );
            $body = (string) wp_remote_retrieve_body( $resp );
            if ( $code < 200 || $code >= 300 ) {
                $state['last_error'] = 'HTTP ' . $code . ' ' . substr( $body, 0, 200 );
                update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );
                return array( 'done' => false, 'progress' => 98, 'log' => 'Commit rejected: ' . $state['last_error'] );
            }

			$commit_info = array();
			$decoded = json_decode( $body, true );
			if ( is_array( $decoded ) ) {
				$commit_info = $decoded;
			}
			$state['commit_response'] = $commit_info;

			$state['phase'] = 'done';
			$state['progress'] = 100;
			update_option( self::JOB_OPTION_PREFIX . $job_id, $state, false );

			$log_line = 'Push complete. Destination commit accepted.';
			if ( ! empty( $commit_info['purpose'] ) ) {
				$log_line .= ' Purpose=' . sanitize_key( (string) $commit_info['purpose'] ) . '.';
			}
			if ( ! empty( $commit_info['package_dir'] ) ) {
				$log_line .= ' Stored at: ' . (string) $commit_info['package_dir'];
			}

			return array( 'done' => true, 'progress' => 100, 'log' => $log_line );
        }

        return array( 'done' => false, 'progress' => (int) $state['progress'], 'log' => 'Working…' );
    }
}
