<?php
namespace AegisBackup\Restore;

defined( 'ABSPATH' ) || exit;

require_once AEGISBACKUP_DIR . 'includes/libs/class-ab-db-prefix-migrator.php';

use AegisBackup\Libs\AB_DB_Prefix_Migrator;

class AB_DB_Restorer {

    public function detect_backup_prefix( $package_dir ) {
        $db_dir = trailingslashit( (string) $package_dir ) . 'db';
        if ( ! is_dir( $db_dir ) ) {
            return '';
        }

        $candidates = glob( $db_dir . '/*.sql' );
        if ( empty( $candidates ) ) {
            $candidates = glob( $db_dir . '/*.sql.gz' );
        }
        if ( empty( $candidates ) ) {
            return '';
        }

        $file = (string) $candidates[0];
        $buf = '';
        if ( preg_match( '/\.gz$/i', $file ) ) {
            if ( function_exists( 'gzopen' ) ) {
                $h = @gzopen( $file, 'rb' );
                if ( $h ) {
                    $buf = (string) @gzread( $h, 200000 );
                    @gzclose( $h );
                }
            }
        } else {
            $h = @fopen( $file, 'rb' );
            if ( $h ) {
                $buf = (string) @fread( $h, 200000 );
                @fclose( $h );
            }
        }

        if ( '' === $buf ) {
            return '';
        }

        if ( preg_match( '/\b(?:CREATE\s+TABLE|INSERT\s+INTO|DROP\s+TABLE\s+IF\s+EXISTS)\s+`([^`]+)_options`/i', $buf, $m ) ) {
            return (string) $m[1] . '_';
        }
        if ( preg_match( '/\b(?:CREATE\s+TABLE|INSERT\s+INTO|DROP\s+TABLE\s+IF\s+EXISTS)\s+([A-Za-z0-9]+)_options\b/i', $buf, $m2 ) ) {
            return (string) $m2[1] . '_';
        }

        return '';
    }

    public function drop_tables_with_prefix_step( $prefix, array &$drop_state, $batch = 25 ) {
        global $wpdb;

        $prefix = (string) $prefix;
        if ( '' === $prefix ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'No prefix provided; skipping table drop.' );
        }

        if ( empty( $drop_state['init'] ) ) {
            $like = $wpdb->esc_like( $prefix ) . '%';
            $tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) );
            if ( ! is_array( $tables ) ) {
                $tables = array();
            }
            $drop_state['tables'] = array_values( $tables );
            $drop_state['idx'] = 0;
            $drop_state['init'] = true;
        }

        $tables = isset( $drop_state['tables'] ) ? (array) $drop_state['tables'] : array();
        $idx = isset( $drop_state['idx'] ) ? (int) $drop_state['idx'] : 0;

        if ( $idx >= count( $tables ) ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'Existing tables dropped (if any).' );
        }

        $chunk = array_slice( $tables, $idx, max( 1, (int) $batch ) );
        if ( empty( $chunk ) ) {
            $drop_state['idx'] = count( $tables );
            return array( 'done' => true, 'progress' => 100, 'log' => 'Existing tables dropped (if any).' );
        }

        foreach ( $chunk as $t ) {
            $t = preg_replace( '/[^A-Za-z0-9_\$]/', '', (string) $t );
            if ( '' === $t ) {
                continue;
            }
            $wpdb->query( 'DROP TABLE IF EXISTS `' . $t . '`' );
        }

        $drop_state['idx'] = $idx + count( $chunk );
        $pct = (int) floor( ( $drop_state['idx'] / max( 1, count( $tables ) ) ) * 100 );
        return array( 'done' => false, 'progress' => min( 99, max( 1, $pct ) ), 'log' => 'Dropping existing tables: ' . $drop_state['idx'] . '/' . count( $tables ) );
    }

    public function import_db_step( $package_dir, array &$import_state, $max_statements = 60, $max_seconds = 1.0, $old_prefix = '', $new_prefix = '' ) {
        global $wpdb;
        $package_dir = trailingslashit( (string) $package_dir );
        $old_prefix = (string) $old_prefix;
        $new_prefix = (string) $new_prefix;
        $db_dir = $package_dir . 'db/';

        if ( empty( $import_state['files'] ) ) {
            $files = array();

            $single = $db_dir . 'db.sql';
            if ( is_file( $single ) ) {
                $files[] = $single;
            }

            $tables = glob( $db_dir . 'tables/*.sql' );
            if ( $tables ) {
                sort( $tables );
                $files = array_merge( $files, $tables );
            }

            $direct = glob( $db_dir . '*.sql' );
            if ( $direct ) {
                foreach ( $direct as $f ) {
                    if ( basename( $f ) === 'db.sql' ) { continue; }
                    $files[] = $f;
                }
            }

            $files = array_values( array_unique( $files ) );

            $import_state['files'] = $files;
            $import_state['idx'] = 0;
            $import_state['pos'] = 0;
            $import_state['buffer'] = '';
            $import_state['statements_done'] = 0;
            $import_state['tables_done'] = 0;
        }

        $files = (array) $import_state['files'];
        $idx = (int) $import_state['idx'];

        if ( $idx >= count( $files ) ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'DB import complete.', 'statements' => isset($import_state['statements_done']) ? (int)$import_state['statements_done'] : 0, 'tables' => isset($import_state['tables_done']) ? (int)$import_state['tables_done'] : 0 );
        }

        $path = (string) $files[ $idx ];
        $import_state['current_file'] = basename( $path );
        if ( ! isset( $import_state['errors_count'] ) ) {
            $import_state['errors_count'] = 0;
        }
        if ( ! isset( $import_state['errors'] ) || ! is_array( $import_state['errors'] ) ) {
            $import_state['errors'] = array();
        }
        if ( ! is_file( $path ) ) {
            $import_state['idx'] = $idx + 1;
            $import_state['pos'] = 0;
            $import_state['buffer'] = '';
            return array( 'done' => false, 'progress' => $this->import_progress_pct( $import_state ), 'log' => 'Skipped missing SQL file.' );
        }

        $t0 = microtime( true );
        $statements = 0;

        $handle = fopen( $path, 'r' );
        if ( ! $handle ) {
            return array( 'done' => false, 'progress' => $this->import_progress_pct( $import_state ), 'log' => 'Failed to open SQL file.' );
        }

        $pos = (int) $import_state['pos'];
        if ( $pos > 0 ) {
            fseek( $handle, $pos );
        }

        $buffer = (string) $import_state['buffer'];

        while ( ! feof( $handle ) ) {
            $line = fgets( $handle );
            if ( false === $line ) {
                break;
            }

            if ( preg_match( '/^\s*(--|#)/', $line ) ) {
                continue;
            }

            $buffer .= $line;

            if ( preg_match( '/;\s*$/', trim( $line ) ) ) {
                $sql = trim( $buffer );
                $buffer = '';

                if ( '' !== $sql ) {
                    if ( '' !== (string) $old_prefix && '' !== (string) $new_prefix && (string) $old_prefix !== (string) $new_prefix ) {
                        $sql = $this->transform_sql_prefix( $sql, (string) $old_prefix, (string) $new_prefix );
                    }

                    $ok = $this->execute_sql_statement( $sql );
                    if ( ! $ok ) {
                        $import_state['errors_count'] = (int) $import_state['errors_count'] + 1;
                        $err = isset( $wpdb->last_error ) ? (string) $wpdb->last_error : '';
                        $err = trim( $err );
                        if ( '' !== $err ) {
                            if ( count( $import_state['errors'] ) < 25 ) {
                                $import_state['errors'][] = array(
                                    'file' => basename( $path ),
                                    'error' => $err,
                                    'sql' => substr( preg_replace( '/\s+/', ' ', $sql ), 0, 160 ),
                                    'statement_no' => (int) $import_state['statements_done'] + 1,
                                );
                            }
                        }
                    }
                    $statements++;
                    $import_state['statements_done'] = (int) $import_state['statements_done'] + 1;
                }

                if ( $statements >= (int) $max_statements ) {
                    break;
                }
                if ( ( microtime( true ) - $t0 ) >= (float) $max_seconds ) {
                    break;
                }
            }
        }

        $pos_now = ftell( $handle );
        $eof = feof( $handle );
        $file_size = (int) filesize( $path );

        $import_state['pos'] = $pos_now;
        $import_state['buffer'] = $buffer;
        fclose( $handle );

        if ( $eof || $pos_now >= $file_size ) {
            $import_state['idx'] = $idx + 1;
            $import_state['pos'] = 0;
            $import_state['buffer'] = '';
        }

        $progress = $this->import_progress_pct( $import_state );
        $log = 'DB import: ' . basename( $path ) . ' +' . $statements . ' statements';
        if ( ! empty( $import_state['errors_count'] ) ) {
            $log .= ' (errors: ' . (int) $import_state['errors_count'] . ')';
            $last_err = '';
            if ( ! empty( $import_state['errors'] ) && is_array( $import_state['errors'] ) ) {
                $last = end( $import_state['errors'] );
                if ( is_array( $last ) && ! empty( $last['error'] ) ) {
                    $last_err = (string) $last['error'];
                }
            }
            if ( '' !== $last_err ) {
                $log .= ' Last error: ' . $last_err;
            }
        }

        return array(
            'done' => false,
            'progress' => $progress,
            'log' => $log,
            'errors_count' => isset( $import_state['errors_count'] ) ? (int) $import_state['errors_count'] : 0,
            'errors' => ( isset( $import_state['errors'] ) && is_array( $import_state['errors'] ) ) ? $import_state['errors'] : array(),
            'current_file' => basename( $path ),
        );
    }

    private function import_progress_pct( array $import_state ) {
        $files = isset( $import_state['files'] ) ? (array) $import_state['files'] : array();
        $idx = isset( $import_state['idx'] ) ? (int) $import_state['idx'] : 0;
        $count = max( 1, count( $files ) );
        $pct = (int) floor( ( $idx / $count ) * 100 );
        return max( 1, min( 99, $pct ) );
    }

    protected function transform_sql_prefix( $sql, $old_prefix, $new_prefix ) {
        $sql = (string) $sql;
        $old_prefix = (string) $old_prefix;
        $new_prefix = (string) $new_prefix;

        if ( '' === $old_prefix || '' === $new_prefix || $old_prefix === $new_prefix ) {
            return $sql;
        }

        $sql = preg_replace( '/`' . preg_quote( $old_prefix, '/' ) . '([A-Za-z0-9_]+)`/i', '`' . $new_prefix . '$1`', $sql );

        $head = ltrim( $sql );
        if ( preg_match( '/^(CREATE\s+TABLE|DROP\s+TABLE|ALTER\s+TABLE|INSERT\s+INTO|UPDATE|LOCK\s+TABLES|UNLOCK\s+TABLES)/i', $head ) ) {
            $sql = preg_replace( '/\b' . preg_quote( $old_prefix, '/' ) . '([A-Za-z0-9_]+)\b/i', $new_prefix . '$1', $sql, 1 );
        }

        return $sql;
    }

protected function execute_sql_statement( $sql ) {
        global $wpdb;

        $sql = trim( (string) $sql );
        if ( '' === $sql ) {
            return true;
        }

        $r = $wpdb->query( $sql );
        return false !== $r;
    }

    public function apply_prefix_migration( $old_prefix, $new_prefix ) {
        $m = new AB_DB_Prefix_Migrator();
        return $m->apply_prefix_change( (string) $old_prefix, (string) $new_prefix, 'all' );
    }

    public function domain_replace_step( $old, $new, $prefix, array &$domain_state ) {
        global $wpdb;

        $old = trim( (string) $old );
        $new = trim( (string) $new );
        $prefix = (string) $prefix;

        if ( '' === $old || '' === $new || $old === $new ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'No domain replacement needed.' );
        }

        if ( empty( $domain_state['init'] ) ) {
            $wpdb->update( $prefix . 'options', array( 'option_value' => $new ), array( 'option_name' => 'home' ) );
            $wpdb->update( $prefix . 'options', array( 'option_value' => $new ), array( 'option_name' => 'siteurl' ) );

            $domain_state['targets'] = array(
                array( 'table' => $prefix . 'options', 'pk' => 'option_id', 'col' => 'option_value', 'where' => '' ),
                array( 'table' => $prefix . 'postmeta', 'pk' => 'meta_id', 'col' => 'meta_value', 'where' => '' ),
                array( 'table' => $prefix . 'usermeta', 'pk' => 'umeta_id', 'col' => 'meta_value', 'where' => '' ),
                array( 'table' => $prefix . 'posts', 'pk' => 'ID', 'col' => 'post_content', 'where' => '' ),
                array( 'table' => $prefix . 'posts', 'pk' => 'ID', 'col' => 'post_excerpt', 'where' => '' ),
            );

            $domain_state['tidx'] = 0;
            $domain_state['last_id'] = 0;
            $domain_state['updated'] = 0;
            $domain_state['replacements'] = 0;
            $domain_state['init'] = true;
        }

        $targets = (array) $domain_state['targets'];
        $tidx = (int) $domain_state['tidx'];

        if ( $tidx >= count( $targets ) ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'Domain replacement complete. Updated rows: ' . (int) $domain_state['updated'], 'rows_updated' => (int) $domain_state['updated'], 'replacements' => isset($domain_state['replacements']) ? (int) $domain_state['replacements'] : 0 );
        }

        $t = $targets[ $tidx ];
        $table = (string) $t['table'];
        $pk = (string) $t['pk'];
        $col = (string) $t['col'];
        $last_id = (int) $domain_state['last_id'];
        $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );
        if ( $exists !== $table ) {
            $domain_state['tidx'] = $tidx + 1;
            $domain_state['last_id'] = 0;
            return array( 'done' => false, 'progress' => $this->domain_progress_pct( $domain_state ), 'log' => 'Skipped missing table: ' . $table );
        }

        $batch = 200;
        $rows = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT {$pk} AS pk, {$col} AS val FROM {$table} WHERE {$pk} > %d ORDER BY {$pk} ASC LIMIT %d",
                $last_id,
                $batch
            ),
            ARRAY_A
        );

        if ( empty( $rows ) ) {
            $domain_state['tidx'] = $tidx + 1;
            $domain_state['last_id'] = 0;
            return array( 'done' => false, 'progress' => $this->domain_progress_pct( $domain_state ), 'log' => 'Completed: ' . $table . '.' . $col, 'rows_updated' => (int) $domain_state['updated'], 'replacements' => isset($domain_state['replacements']) ? (int) $domain_state['replacements'] : 0 );
        }

        foreach ( (array) $rows as $r ) {
            $id = (int) $r['pk'];
            $val = (string) $r['val'];

            $new_val = $this->serialized_safe_replace( $old, $new, $val );
            if ( $new_val !== $val ) {
                $wpdb->update( $table, array( $col => $new_val ), array( $pk => $id ) );
                $domain_state['updated'] = (int) $domain_state['updated'] + 1;
                $cnt = substr_count( $val, $old );
                if ( $cnt < 1 ) { $cnt = 1; }
                $domain_state['replacements'] = (int) $domain_state['replacements'] + (int) $cnt;
            }
            $domain_state['last_id'] = $id;
        }

        return array( 'done' => false, 'progress' => $this->domain_progress_pct( $domain_state ), 'log' => 'Domain replace: ' . $table . '.' . $col . ' updated=' . (int) $domain_state['updated'] );
    }

    private function domain_progress_pct( array $domain_state ) {
        $tidx = isset( $domain_state['tidx'] ) ? (int) $domain_state['tidx'] : 0;
        $targets = isset( $domain_state['targets'] ) ? (array) $domain_state['targets'] : array();
        $count = max( 1, count( $targets ) );
        $pct = (int) floor( ( $tidx / $count ) * 100 );
        return max( 1, min( 99, $pct ) );
    }

    private function serialized_safe_replace( $old, $new, $value ) {
        $old = (string) $old;
        $new = (string) $new;

        if ( '' === $value ) {
            return $value;
        }

        if ( function_exists( 'is_serialized' ) && ! is_serialized( $value ) ) {
            return str_replace( $old, $new, $value );
        }

        $un = @unserialize( $value );
        if ( false === $un && 'b:0;' !== $value ) {
            return str_replace( $old, $new, $value );
        }

        $replaced = $this->deep_replace( $old, $new, $un );
        if ( function_exists( 'maybe_serialize' ) ) {
            return maybe_serialize( $replaced );
        }
        return serialize( $replaced );
    }

    private function deep_replace( $old, $new, $data ) {
        if ( is_string( $data ) ) {
            return str_replace( $old, $new, $data );
        }
        if ( is_array( $data ) ) {
            $out = array();
            foreach ( $data as $k => $v ) {
                $nk = is_string( $k ) ? str_replace( $old, $new, $k ) : $k;
                $out[ $nk ] = $this->deep_replace( $old, $new, $v );
            }
            return $out;
        }
        if ( is_object( $data ) ) {
            foreach ( $data as $k => $v ) {
                $data->$k = $this->deep_replace( $old, $new, $v );
            }
            return $data;
        }
        return $data;
    }

    public function verify_checksums_step( $tmp_dir, array &$state, $max_items = 200, $max_seconds = 1.0 ) {
        $start = microtime( true );
        $tmp_dir = rtrim( (string) $tmp_dir, '/\\' );
        $file = $tmp_dir . '/checksums.sha256';

        if ( ! file_exists( $file ) ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'No checksums.sha256 found. Skipping integrity verification.' );
        }

        if ( empty( $state['lines'] ) ) {
            $lines = file( $file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES );
            $state['lines'] = is_array( $lines ) ? $lines : array();
            $state['i'] = 0;
            $state['total'] = count( $state['lines'] );
            $state['bad'] = 0;
            $state['skipped_large'] = 0;
            $state['skipped_large_bytes'] = 0;
            if ( 0 === (int) $state['total'] ) {
                return array( 'done' => true, 'progress' => 100, 'log' => 'Checksums file is empty. Skipping.' );
            }
        }

        $processed = 0;
        while ( (int) $state['i'] < (int) $state['total'] ) {
            $line = (string) $state['lines'][ (int) $state['i'] ];
            $state['i'] = (int) $state['i'] + 1;

            if ( ! preg_match( '/^([a-f0-9]{64})\s+(.+)$/i', trim( $line ), $m ) ) {
                continue;
            }

            $expected = strtolower( $m[1] );
            $rel = ltrim( str_replace( '\\', '/', $m[2] ), '/' );
            $path = $tmp_dir . '/' . $rel;

            if ( file_exists( $path ) && is_file( $path ) ) {
                $sz = @filesize( $path );
                if ( false !== $sz && (int) $sz > ( 100 * 1024 * 1024 ) ) {
                    $state['skipped_large'] = isset( $state['skipped_large'] ) ? ( (int) $state['skipped_large'] + 1 ) : 1;
                    $state['skipped_large_bytes'] = isset( $state['skipped_large_bytes'] ) ? ( (int) $state['skipped_large_bytes'] + (int) $sz ) : (int) $sz;
                } else {
                    $actual = strtolower( hash_file( 'sha256', $path ) );
                    if ( $actual !== $expected ) {
                        $state['bad'] = (int) $state['bad'] + 1;
                    }
                }
            } else {
                $state['bad'] = (int) $state['bad'] + 1;
            }

            $processed++;
            if ( $processed >= (int) $max_items ) {
                break;
            }
            if ( ( microtime( true ) - $start ) >= (float) $max_seconds ) {
                break;
            }
        }

        $pct = 0;
        if ( ! empty( $state['total'] ) ) {
            $pct = (int) floor( ( (int) $state['i'] / (int) $state['total'] ) * 100 );
            $pct = max( 0, min( 100, $pct ) );
        }

        $done = ( (int) $state['i'] >= (int) $state['total'] );
        $skipped = isset( $state['skipped_large'] ) ? (int) $state['skipped_large'] : 0;
        $log = $done
            ? sprintf( 'Integrity verification complete. %d issue(s) found.%s', (int) $state['bad'], ( $skipped > 0 ? ( ' (Skipped hashing ' . $skipped . ' large file(s) for speed.)' ) : '' ) )
            : sprintf( 'Verifying integrity… %d/%d (%d%%)%s', (int) $state['i'], (int) $state['total'], (int) $pct, ( $skipped > 0 ? ( ' (Skipped hashing ' . $skipped . ' large file(s) so far.)' ) : '' ) );

        return array( 'done' => $done, 'progress' => $pct, 'log' => $log, 'bad' => (int) $state['bad'], 'skipped_large' => $skipped );
    }

public function post_restore_cleanup() {

        if ( function_exists( 'flush_rewrite_rules' ) ) {
            flush_rewrite_rules( true );
        }
        if ( function_exists( 'wp_cache_flush' ) ) {
            wp_cache_flush();
        }
        if ( function_exists( 'wp_clear_scheduled_hook' ) ) {
        }
    }
}
