<?php
namespace AegisBackup\Restore;

defined( 'ABSPATH' ) || exit;

class AB_File_Restorer {
    const MAX_RETRIES = 2;
    const MAX_SAMPLES = 25;

    public function restore_files_step( $tmp_dir, array &$state, $max_files = 200, $max_seconds = 1.0 ) {
        $start = microtime( true );
        $tmp_dir = rtrim( (string) $tmp_dir, '/\\' );

        $files_root = $tmp_dir . '/files';
        if ( ! is_dir( $files_root ) ) {
            return array( 'done' => true, 'progress' => 100, 'log' => 'No files payload in package (files/ not found). Skipping file restore.', 'restored' => 0, 'bytes' => 0 );
        }

        if ( empty( $state['queue'] ) || ! is_array( $state['queue'] ) ) {
            $state['queue'] = $this->build_queue( $files_root );
            $state['total'] = count( $state['queue'] );
            $state['done']  = 0;
            $state['errors']= 0;
            $state['bytes'] = 0;
            $state['restored'] = 0;
            $state['failed_writes'] = 0;
            $state['hash_mismatch'] = 0;
            $state['failed_samples'] = array();
            $state['hash_mismatch_samples'] = array();
            $state['retry_map'] = array();

            if ( 0 === (int) $state['total'] ) {
                return array( 'done' => true, 'progress' => 100, 'log' => 'Files payload empty. Skipping.', 'restored' => 0, 'bytes' => 0 );
            }
        }

        $processed = 0;
        while ( ! empty( $state['queue'] ) ) {
            $rel = array_shift( $state['queue'] );
            $src = $files_root . '/' . $rel;
            $state['current_file'] = $rel;
            $state['current_action'] = 'copy';
            $state['last_error_note'] = '';
            $dst = rtrim( ABSPATH, '/\\' ) . '/' . $rel;
            $dst_dir = dirname( $dst );
            if ( ! is_dir( $dst_dir ) ) {
                wp_mkdir_p( $dst_dir );
            }

            if ( is_dir( $src ) ) {

            } else {
                $ok = false;
                $err_note = '';

                if ( ! is_file( $src ) ) {
                    $err_note = 'Source file missing inside package.';
                    $ok = false;
                } else {
                    $ok = @copy( $src, $dst );
                    if ( ! $ok ) {
                        $err_note = 'copy() failed (permissions/disk/timeout).';
                    }
                }

                if ( $ok ) {
                    $src_hash = @hash_file( 'sha256', $src );
                    $dst_hash = @hash_file( 'sha256', $dst );
                    if ( empty( $src_hash ) || empty( $dst_hash ) || $src_hash !== $dst_hash ) {
                        $ok = false;
                        $state['hash_mismatch'] = isset( $state['hash_mismatch'] ) ? ( (int) $state['hash_mismatch'] + 1 ) : 1;
                        $err_note = 'Hash mismatch after copy (file may be corrupted).';
                        if ( isset( $state['hash_mismatch_samples'] ) && is_array( $state['hash_mismatch_samples'] ) && count( $state['hash_mismatch_samples'] ) < self::MAX_SAMPLES ) {
                            $state['hash_mismatch_samples'][] = array(
                                'rel' => $rel,
                                'src' => $src,
                                'dst' => $dst,
                                'src_hash' => $src_hash,
                                'dst_hash' => $dst_hash,
                            );
                        }

                        @unlink( $dst );
                    }
                }

                if ( ! $ok ) {
                    $state['errors'] = (int) $state['errors'] + 1;
                    $state['last_error_note'] = $rel . ' - ' . $err_note;
                    $state['last_error_note'] = (string) $err_note;

                    if ( ! isset( $state['retry_map'] ) || ! is_array( $state['retry_map'] ) ) {
                        $state['retry_map'] = array();
                    }
                    $attempts = isset( $state['retry_map'][ $rel ] ) ? (int) $state['retry_map'][ $rel ] : 0;
                    $attempts++;
                    $state['retry_map'][ $rel ] = $attempts;

                    if ( $attempts <= self::MAX_RETRIES ) {
                        $state['queue'][] = $rel;
                    } else {
                        $state['failed_writes'] = isset( $state['failed_writes'] ) ? ( (int) $state['failed_writes'] + 1 ) : 1;
                        if ( isset( $state['failed_samples'] ) && is_array( $state['failed_samples'] ) && count( $state['failed_samples'] ) < self::MAX_SAMPLES ) {
                            $state['failed_samples'][] = array(
                                'rel' => $rel,
                                'dst' => $dst,
                                'note' => $err_note,
                                'attempts' => $attempts,
                            );
                        }
                    }
                } else {
                    @touch( $dst, @filemtime( $src ) );
                    $sz = @filesize( $src );
                    if ( false !== $sz ) { $state['bytes'] = (int) $state['bytes'] + (int) $sz; }
                    $state['restored'] = (int) $state['restored'] + 1;
                }
            }

            $state['done'] = (int) $state['done'] + 1;
            $processed++;

            if ( $processed >= (int) $max_files ) {
                break;
            }
            if ( ( microtime( true ) - $start ) >= (float) $max_seconds ) {
                break;
            }
        }

        $pct = 0;
        if ( ! empty( $state['total'] ) ) {
            $pct = (int) floor( ( (int) $state['done'] / (int) $state['total'] ) * 100 );
            $pct = max( 0, min( 100, $pct ) );
        }

        $done = empty( $state['queue'] );
        $log = $done
            ? sprintf( 'File restore complete. Processed %d items with %d errors. FailedWrites=%d, HashMismatch=%d.', (int) $state['done'], (int) $state['errors'], (int) ( $state['failed_writes'] ?? 0 ), (int) ( $state['hash_mismatch'] ?? 0 ) )
            : sprintf( 'Restoring files… %d/%d (%d%%) Current: %s', (int) $state['done'], (int) $state['total'], (int) $pct, isset( $state['current_file'] ) ? (string) $state['current_file'] : '' );

        if ( ! $done && ! empty( $state['last_error_note'] ) ) {
            $log .= ' | Last file error: ' . (string) $state['last_error_note'];
        }

        return array(
            'done' => $done,
            'progress' => $pct,
            'log' => $log,
            'restored' => isset( $state['restored'] ) ? (int) $state['restored'] : 0,
            'bytes' => isset( $state['bytes'] ) ? (int) $state['bytes'] : 0,
            'failed_writes' => isset( $state['failed_writes'] ) ? (int) $state['failed_writes'] : 0,
            'hash_mismatch' => isset( $state['hash_mismatch'] ) ? (int) $state['hash_mismatch'] : 0,
            'failed_samples' => ( isset( $state['failed_samples'] ) && is_array( $state['failed_samples'] ) ) ? $state['failed_samples'] : array(),
            'hash_mismatch_samples' => ( isset( $state['hash_mismatch_samples'] ) && is_array( $state['hash_mismatch_samples'] ) ) ? $state['hash_mismatch_samples'] : array(),
            'current_file' => isset( $state['current_file'] ) ? (string) $state['current_file'] : '',
            'current_action' => isset( $state['current_action'] ) ? (string) $state['current_action'] : '',
            'last_error_note' => isset( $state['last_error_note'] ) ? (string) $state['last_error_note'] : '',
        );
    }

    private function build_queue( $root ) {
        $root = rtrim( (string) $root, '/\\' );
        $queue = array();

        $it = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator( $root, \FilesystemIterator::SKIP_DOTS ),
            \RecursiveIteratorIterator::SELF_FIRST
        );

                foreach ( $it as $path => $info ) {
            if ( $info->isDir() ) {
                continue;
            }
            $rel = ltrim( str_replace( array( $root, '\\' ), array( '', '/' ), (string) $path ), '/' );
            $queue[] = $rel;
        }

        return $queue;

    }
}
