<?php
namespace AegisBackup\Restore;

defined( 'ABSPATH' ) || exit;

class AB_Preflight {
    public function run_checks( array $args = array() ) {
        global $wpdb;

        $results = array();
        $results[] = $this->result( 'site_https', 'Site uses HTTPS', is_ssl(), is_ssl() ? 'OK' : 'Not using HTTPS (recommended).' , ! is_ssl() ? 'warn' : 'ok' );
        $upload_dir = wp_upload_dir();
        $up = isset( $upload_dir['basedir'] ) ? $upload_dir['basedir'] : '';
        $results[] = $this->result( 'uploads_writable', 'Uploads directory writable', ( $up && is_dir( $up ) && wp_is_writable( $up ) ), $up ? $up : '(unknown)' );
        $abspath = defined( 'ABSPATH' ) ? ABSPATH : '';
        $results[] = $this->result( 'abspath_writable', 'ABSPATH writable (for file restore)', ( $abspath && is_dir( $abspath ) && wp_is_writable( $abspath ) ), $abspath ? $abspath : '(unknown)', ( $abspath && wp_is_writable( $abspath ) ) ? 'ok' : 'warn' );
        $max_exec = (int) ini_get( 'max_execution_time' );
        $mem = ini_get( 'memory_limit' );
        $results[] = $this->result( 'php_max_execution_time', 'PHP max_execution_time', true, (string) $max_exec . ' seconds', ( $max_exec > 0 && $max_exec < 30 ) ? 'warn' : 'ok' );
        $results[] = $this->result( 'php_memory_limit', 'PHP memory_limit', true, (string) $mem, $this->memory_is_low( $mem ) ? 'warn' : 'ok' );
        $results[] = $this->result( 'ziparchive', 'ZipArchive available', class_exists( 'ZipArchive' ), class_exists( 'ZipArchive' ) ? 'OK' : 'ZipArchive missing (ZIP operations may fail).', class_exists( 'ZipArchive' ) ? 'ok' : 'fail' );
        $need_bytes = isset( $args['expected_bytes'] ) ? (int) $args['expected_bytes'] : 0;
        $disk_ok = true;
        $disk_msg = 'Unable to determine disk space.';
        if ( function_exists( 'disk_free_space' ) && $up ) {
            $free = @disk_free_space( $up );
            if ( false !== $free ) {
                $disk_msg = 'Free: ' . size_format( (int) $free );
                if ( $need_bytes > 0 ) {
                    $disk_msg .= ' | Needed (est.): ' . size_format( (int) $need_bytes );
                    $disk_ok = ( (int) $free > (int) ( $need_bytes * 1.2 ) ); // 20% buffer
                }
            }
        }
        $results[] = $this->result( 'disk_space', 'Disk space', $disk_ok, $disk_msg, $disk_ok ? 'ok' : 'warn' );
        $db_priv = $this->check_db_privileges( $wpdb );
        $results[] = $db_priv;

        return array(
            'success' => true,
            'results' => $results,
        );
    }

    protected function check_db_privileges( $wpdb ) {
        $prefix = $wpdb->prefix;
        $tbl_a = $prefix . 'ab_priv_a_' . wp_generate_password( 6, false, false );
        $tbl_b = $prefix . 'ab_priv_b_' . wp_generate_password( 6, false, false );

        $ok_create = false;
        $ok_rename = false;
        $ok_drop   = false;

        $r = $wpdb->query( 'CREATE TABLE `' . esc_sql( $tbl_a ) . '` (id INT(11) NOT NULL) ENGINE=InnoDB' );
        $ok_create = ( false !== $r );

        if ( $ok_create ) {
            $wpdb->query( 'CREATE TABLE `' . esc_sql( $tbl_b ) . '` (id INT(11) NOT NULL) ENGINE=InnoDB' );
            $r2 = $wpdb->query( 'RENAME TABLE `' . esc_sql( $tbl_a ) . '` TO `' . esc_sql( $tbl_a ) . '_x`' );
            $ok_rename = ( false !== $r2 );
            $r3 = $wpdb->query( 'DROP TABLE IF EXISTS `' . esc_sql( $tbl_a ) . '_x`' );
            $r4 = $wpdb->query( 'DROP TABLE IF EXISTS `' . esc_sql( $tbl_b ) . '`' );
            $ok_drop = ( false !== $r3 && false !== $r4 );
        }

        $ok = $ok_create && $ok_drop;
        $level = $ok ? 'ok' : 'warn';
        $msg = 'CREATE/DROP ' . ( $ok ? 'OK' : 'may be restricted' ) . '; RENAME ' . ( $ok_rename ? 'OK' : 'may be restricted' ) . '.';

        return $this->result( 'db_privileges', 'Database privileges (best-effort)', $ok, $msg, $level );
    }

    protected function memory_is_low( $mem ) {
        $mem = (string) $mem;
        if ( '' === $mem || '-1' === $mem ) return false;
        $bytes = wp_convert_hr_to_bytes( $mem );
        return ( $bytes > 0 && $bytes < 268435456 ); // < 256MB
    }

    protected function result( $key, $label, $pass, $detail = '', $level = '' ) {
        if ( '' === $level ) {
            $level = $pass ? 'ok' : 'fail';
        }
        return array(
            'key' => (string) $key,
            'label' => (string) $label,
            'pass' => (bool) $pass,
            'level' => (string) $level, // ok|warn|fail
            'detail' => (string) $detail,
        );
    }
}
