<?php
namespace AegisBackup\Utils;

defined( 'ABSPATH' ) || exit;

class License {
	const OPTION_LICENSE_KEY   = 'aegisbackup_license_key';
	const OPTION_LICENSE_EMAIL = 'aegisbackup_license_email';
	const OPTION_STATUS_CACHE  = 'aegisbackup_license_status';
	const OPTION_LAST_CHECK    = 'aegisbackup_license_last_check';
	const OPTION_FREE_OPTIN    = 'aegisbackup_free_reg_optin';
	const OPTION_FREE_STATUS   = 'aegisbackup_free_reg_status';

	public function get_key() {
		return (string) get_option( self::OPTION_LICENSE_KEY, '' );
	}

	public function set_key( $key ) {
		update_option( self::OPTION_LICENSE_KEY, (string) $key, false );
	}

	public function get_email() {
		$e = (string) get_option( self::OPTION_LICENSE_EMAIL, '' );
		if ( '' === $e ) {
			$e = (string) get_option( 'admin_email', '' );
		}
		return $e;
	}

	public function get_cached_status() {
		$st = get_option( self::OPTION_STATUS_CACHE, array() );
		return is_array( $st ) ? $st : array();
	}

	public function get_status() {
		return $this->get_cached_status();
	}

	public function is_pro_active( $version = '' ) {
		$st = $this->get_cached_status();
		$this->maybe_enforce_clm_expiration( $st, $version );
		$st = $this->get_cached_status();

		return (bool) ( ! empty( $st['is_pro'] ) );
	}


	public function activate_now( $version ) {
		return $this->post_to_clm( 'activate', $version );
	}

	public function deactivate_now( $version ) {
		return $this->post_to_clm( 'deactivate', $version );
	}

	public function check_now( $version ) {
		return $this->post_to_clm( 'status', $version );
	}

	public function maybe_daily_check( $version ) {
		$this->check_now( $version );
	}

	public function maybe_send_free_register( $version ) {
		$optin  = (int) get_option( self::OPTION_FREE_OPTIN, 0 );
		$status = (string) get_option( self::OPTION_FREE_STATUS, 'unregistered' );
		if ( 1 !== $optin || 'registered' === $status ) {
			return;
		}

		$sent = (int) get_transient( 'aegisbackup_clm_free_reg_sent' );
		if ( $sent ) {
			return;
		}

		$resp = $this->post_to_clm( 'free-register', $version );
		if ( ! empty( $resp['success'] ) ) {
			update_option( self::OPTION_FREE_STATUS, 'registered', false );
		} else {
			update_option( self::OPTION_FREE_STATUS, 'unregistered', false );
		}
		set_transient( 'aegisbackup_clm_free_reg_sent', 1, DAY_IN_SECONDS );
	}

	protected function post_to_clm( $action, $version ) {
		$action = (string) $action;

		$endpoints = array(
			'activate'      => defined( 'AEGISBACKUP_CLM_ACTIVATE_ENDPOINT' ) ? AEGISBACKUP_CLM_ACTIVATE_ENDPOINT : '',
			'deactivate'    => defined( 'AEGISBACKUP_CLM_DEACTIVATE_ENDPOINT' ) ? AEGISBACKUP_CLM_DEACTIVATE_ENDPOINT : '',
			'status'        => defined( 'AEGISBACKUP_CLM_STATUS_ENDPOINT' ) ? AEGISBACKUP_CLM_STATUS_ENDPOINT : '',
			'free-register' => defined( 'AEGISBACKUP_CLM_FREE_REG_ENDPOINT' ) ? AEGISBACKUP_CLM_FREE_REG_ENDPOINT : '',
		);

		$url = isset( $endpoints[ $action ] ) ? (string) $endpoints[ $action ] : '';
		if ( '' === $url ) {
			return array( 'success' => false, 'message' => 'CLM endpoint missing.' );
		}

		$payload = array(
			'product'     => defined( 'AEGISBACKUP_SLUG' ) ? AEGISBACKUP_SLUG : 'aegisbackup',
			'slug'        => defined( 'AEGISBACKUP_SLUG' ) ? AEGISBACKUP_SLUG : 'aegisbackup',
			'version'     => (string) $version,
			'site_url'    => home_url(),
			'license_key' => (string) $this->get_key(),
			'email'       => (string) $this->get_email(),
		);

		$resp = wp_remote_post( $url, array(
			'timeout' => 20,
			'headers' => array( 'Content-Type' => 'application/json' ),
			'body'    => wp_json_encode( $payload ),
		) );

		if ( is_wp_error( $resp ) ) {
			return array( 'success' => false, 'message' => $resp->get_error_message() );
		}

		$code = (int) wp_remote_retrieve_response_code( $resp );
		$body = (string) wp_remote_retrieve_body( $resp );

		$data = json_decode( $body, true );
		if ( ! is_array( $data ) ) {
			$data = array( 'success' => ( $code >= 200 && $code < 300 ) );
		}

		$d = $data;
		if ( ! empty( $data['data'] ) && is_array( $data['data'] ) ) {
			$d = $data['data'];
		}

		if ( 'status' === $action || 'activate' === $action || 'deactivate' === $action ) {
			if ( is_array( $d ) ) {
				// ✅ Normalize fields (expires_at -> expires) so UI can display expiration.
				$d = $this->normalize_clm_status_fields( $d );

				update_option( self::OPTION_STATUS_CACHE, $d, false );
				update_option( self::OPTION_LAST_CHECK, time(), false );
			}
		}

		return is_array( $d ) ? $d : array( 'success' => false );
	}

	protected function normalize_clm_status_fields( $d ) {
		if ( ! is_array( $d ) ) {
			return $d;
		}

		if ( isset( $d['expires_at'] ) && '' !== (string) $d['expires_at'] ) {
			$d['expires'] = (string) $d['expires_at'];
		}

		if ( ( ! isset( $d['expires_at'] ) || '' === (string) $d['expires_at'] ) && isset( $d['expires'] ) && '' !== (string) $d['expires'] ) {
			$d['expires_at'] = (string) $d['expires'];
		}

		return $d;
	}

	protected function maybe_enforce_clm_expiration( $cached_status, $version = '' ) {
		$cached_status = is_array( $cached_status ) ? $cached_status : array();

		if ( '' === (string) $version ) {
			$version = defined( 'AEGISBACKUP_VERSION' ) ? (string) AEGISBACKUP_VERSION : '';
		}

		$last_check = (int) get_option( self::OPTION_LAST_CHECK, 0 );

		if ( get_transient( 'aegisbackup_clm_status_throttle' ) ) {
			return;
		}

		$needs_refresh = false;

		if ( $last_check <= 0 || ( time() - $last_check ) > ( 12 * HOUR_IN_SECONDS ) ) {
			$needs_refresh = true;
		}

		$cached_is_pro = ! empty( $cached_status['is_pro'] );
		$expires_at    = isset( $cached_status['expires_at'] ) ? (string) $cached_status['expires_at'] : '';

		if ( $cached_is_pro && '' !== $expires_at ) {
			$ts = strtotime( $expires_at );
			if ( $ts && $ts < time() ) {
				$needs_refresh = true;
			}
		}

		$status_str = isset( $cached_status['status'] ) ? strtolower( (string) $cached_status['status'] ) : '';
		if ( ( $cached_is_pro || 'active' === $status_str ) && '' === $expires_at ) {
			$needs_refresh = true;
		}

		if ( ! $needs_refresh ) {
			return;
		}

		set_transient( 'aegisbackup_clm_status_throttle', 1, 10 * MINUTE_IN_SECONDS );

		$this->check_now( $version );
	}
}
