<?php
/**
 * Plugin Name: AegisLink
 * Plugin URI:  https://aegisify.com/aegislink/
 * Description: AegisLink helps WordPress admins build smarter internal linking with keyword links, ShortURL tracking, and a dynamic word cloud—improving SEO, navigation, and content performance.
 * Version: 1.1.18
 * Requires at least: 6.8
 * Requires PHP: 8.2
 * Author:      Aegisify, LLC
 * Author URI:  https://aegisify.com/
 * Text Domain: aegislink
 * License: GPLv2 or later
 */

if (!defined('ABSPATH')) { exit; }

define('AEGISLINK_VERSION', '1.1.18');
define('AEGISLINK_FILE', __FILE__);
define('AEGISLINK_DIR', plugin_dir_path(__FILE__));
define('AEGISLINK_URL', plugin_dir_url(__FILE__));
define('AEGISLINK_BASENAME', plugin_basename(__FILE__));
define('AEGISLINK_SLUG', 'aegislink');
define('AEGISLINK_CLM_BASE', 'https://aegisify.com');
define('AEGISLINK_CLM_FREE_REG_ENDPOINT',  AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/free-register');
define('AEGISLINK_CLM_UPDATE_ENDPOINT',    AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/update-check');
define('AEGISLINK_CLM_ACTIVATE_ENDPOINT',  AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/activate');
define('AEGISLINK_CLM_DEACTIVATE_ENDPOINT',AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/deactivate');
define('AEGISLINK_CLM_STATUS_ENDPOINT',    AEGISLINK_CLM_BASE . '/wp-json/chue-license/v1/aegislink/status');

require_once AEGISLINK_DIR . 'includes/class-aegislink-plugin.php';
require_once AEGISLINK_DIR . 'includes/log/class-aegislink-logger.php';

function aegislink_update_log(string $msg, array $ctx = array(), string $level = 'INFO') : void {
    if (class_exists('\\AegisLink\\Log\\Logger')) {
        $lvl = strtoupper($level);
        if ($lvl === 'ERROR') { \AegisLink\Log\Logger::error($msg, $ctx); return; }
        if ($lvl === 'WARN' || $lvl === 'WARNING') { \AegisLink\Log\Logger::warn($msg, $ctx); return; }
        if ($lvl === 'DEBUG') { \AegisLink\Log\Logger::debug($msg, $ctx); return; }
        \AegisLink\Log\Logger::info($msg, $ctx);
        return;
    }
}

function aegislink() : \AegisLink\Plugin {
    return \AegisLink\Plugin::instance();
}

register_activation_hook(__FILE__, function () {
    require_once AEGISLINK_DIR . 'includes/utils/class-aegislink-license.php';

    $lic = new \AegisLink\Utils\License();
    $lic->maybe_send_free_register(AEGISLINK_VERSION);
});

// NOTE: CLM-based update checker removed per policy/requirements. Licensing remains unchanged.
register_activation_hook(__FILE__, function() {

    if (!defined('AEGISLINK_DIR')) {
        define('AEGISLINK_DIR', plugin_dir_path(__FILE__));
    }
    require_once AEGISLINK_DIR . 'includes/shortlinks/class-aegislink-shortlinks.php';
    \AegisLink\Shortlinks\Shortlinks::activate();
});

register_deactivation_hook(__FILE__, function() {
    if (!defined('AEGISLINK_DIR')) {
        define('AEGISLINK_DIR', plugin_dir_path(__FILE__));
    }
    require_once AEGISLINK_DIR . 'includes/shortlinks/class-aegislink-shortlinks.php';
    \AegisLink\Shortlinks\Shortlinks::deactivate();
});


// Daily CLM check (license status + update eligibility)
add_action('aegislink_clm_daily_check', function () {
    require_once AEGISLINK_DIR . 'includes/utils/class-aegislink-license.php';
    $lic = new \AegisLink\Utils\License();
    $lic->maybe_daily_check(AEGISLINK_VERSION);

    // Force a fresh CLM update check next time WP asks.
    delete_site_transient('aegislink_clm_update_cache');
    delete_transient('aegislink_update_check');
});

register_activation_hook(__FILE__, function () {
    if (!function_exists('wp_next_scheduled')) { return; }
    if (!wp_next_scheduled('aegislink_clm_daily_check')) {
        wp_schedule_event(time() + HOUR_IN_SECONDS, 'daily', 'aegislink_clm_daily_check');
    }
});

register_deactivation_hook(__FILE__, function () {
    if (!function_exists('wp_next_scheduled') || !function_exists('wp_unschedule_event')) { return; }
    $ts = wp_next_scheduled('aegislink_clm_daily_check');
    if ($ts) {
        wp_unschedule_event($ts, 'aegislink_clm_daily_check');
    }
});

add_action('plugins_loaded', function() {
    add_action('init', function() {
    });

    aegislink()->boot();
});
