<?php
use AegisLink\Shortlinks\Shortlinks;

if (!defined('ABSPATH')) { exit; }


(function () {

    // --- SQL identifier helpers (PluginCheck-safe) ---
    $aegislink_sanitize_db_prefix = static function ($prefix) {
        $prefix = (string) $prefix;
        // allow only identifier-safe characters
        $prefix = preg_replace('/[^A-Za-z0-9_]/', '', $prefix);
        return $prefix;
    };

    $aegislink_escape_sql_identifier = static function ($identifier) {
        $identifier = (string) $identifier;
        if ($identifier === '' || !preg_match('/^[A-Za-z0-9_]+$/', $identifier)) {
            wp_die(esc_html__('Invalid database identifier.', 'aegislink'));
        }
        return '`' . $identifier . '`';
    }; 


if (!class_exists('\\AegisLink\\Shortlinks\\Shortlinks')) {
    $p = defined('AEGISLINK_DIR') ? AEGISLINK_DIR . 'includes/shortlinks/class-aegislink-shortlinks.php' : '';
    if ($p && file_exists($p)) { require_once $p; }
}
if (!class_exists('\\AegisLink\\Shortlinks\\Bulk_Terms')) {
    $p = defined('AEGISLINK_DIR') ? AEGISLINK_DIR . 'includes/shortlinks/class-aegislink-shortlinks-bulk-terms.php' : '';
    if ($p && file_exists($p)) { require_once $p; }
}

if (!current_user_can('edit_posts')) {
    echo '<p>' . esc_html__('Forbidden', 'aegislink') . '</p>';
    return;
}

$license_obj = (function_exists('aegislink') && isset(aegislink()->license)) ? aegislink()->license : null;
$is_pro = (is_object($license_obj) && method_exists($license_obj, 'is_pro') && $license_obj->is_pro());
$pro_locked = !$is_pro;

$errors = array();
$notices = array();
// When a manual health check is performed while headers are already sent,
// we cannot rely on a PHP redirect. We'll use History API to clean the URL.
$healthcheck_replace_url = '';

// One-off notice after manual health checks (the check itself redirects back here).
if (isset($_GET['aegislink_health'])) {
    $hv = sanitize_key((string) wp_unslash($_GET['aegislink_health']));
    if ($hv === 'ok') {
        $notices[] = __('Health check completed.', 'aegislink');
    } elseif ($hv === 'error') {
        $errors[] = __('Health check failed. Please confirm the link still exists and try again.', 'aegislink');
    }
}

// PRO lock: allow viewing, but block all mutations (save/delete/import/export).
if ($pro_locked) {
    if (strtoupper(sanitize_text_field(wp_unslash($_SERVER['REQUEST_METHOD'] ?? ''))) === 'POST') {
        $errors[] = __('PRO feature: ShortURL Links is view-only in the free version. Activate a license to unlock editing and settings.', 'aegislink');
        $_POST = array();
    }
    if (isset($_GET['export'])) {
        $errors[] = __('PRO feature: Export is available in PRO.', 'aegislink');
        unset($_GET['export']);
    }
    if (isset($_GET['delete'])) {
        $errors[] = __('PRO feature: Deleting Short Links is available in PRO.', 'aegislink');
        unset($_GET['delete']);
    }
    if (isset($_GET['import'])) {
        $errors[] = __('PRO feature: Import is available in PRO.', 'aegislink');
        unset($_GET['import']);
    }
}

$short = new Shortlinks();
$base_short_url = $short->get_base_url();

$view = isset($_GET['view']) ? sanitize_key((string) wp_unslash($_GET['view'])) : 'manage';
if (!in_array($view, array('manage','bulk','analytics','importexport'), true)) { $view = 'manage'; }

// Manual (on-demand) health check for a single short link.
// Lets admins flip "Pending" to OK/Broken immediately without waiting for cron.
if ($view === 'manage' && isset($_GET['healthcheck'])) {
    $health_id = absint(wp_unslash($_GET['healthcheck']));
    if ($health_id > 0) {
        check_admin_referer('aegislink_shortlink_healthcheck_' . $health_id);

        $hp = get_post($health_id);
        if (!$hp || $hp->post_type !== Shortlinks::CPT) {
            $errors[] = __('Invalid short link.', 'aegislink');
        } else {
            $target = (string) get_post_meta($health_id, '_aegislink_target', true);
            $target = trim($target);

            if ($target === '' || !preg_match('#^https?://#i', $target)) {
                update_post_meta($health_id, '_aegislink_health_ok', 0);
                update_post_meta($health_id, '_aegislink_health_code', 0);
                update_post_meta($health_id, '_aegislink_health_checked', gmdate('Y-m-d H:i:s'));
                $notices[] = __('Health check updated: target URL is missing or invalid.', 'aegislink');
            } else {
                // Apply UTM at check-time (same behavior as redirect-time) so stored Target URL remains clean.
                $final = $target;
                $utm_enable = (int) get_post_meta($health_id, '_aegislink_utm_enable', true);
                if ($utm_enable) {
                    $params = array();
                    $fields = array(
                        'utm_source'   => '_aegislink_utm_source',
                        'utm_medium'   => '_aegislink_utm_medium',
                        'utm_campaign' => '_aegislink_utm_campaign',
                        'utm_term'     => '_aegislink_utm_term',
                        'utm_content'  => '_aegislink_utm_content',
                    );
                    foreach ($fields as $k => $meta_key) {
                        $v = (string) get_post_meta($health_id, $meta_key, true);
                        $v = trim($v);
                        if ($v !== '') { $params[$k] = $v; }
                    }
                    if (!empty($params)) {
                        $final = add_query_arg($params, $final);
                    }
                }

                $args = array(
                    'timeout' => 8,
                    'redirection' => 3,
                    'user-agent' => 'AegisLink/' . (defined('AEGISLINK_VERSION') ? AEGISLINK_VERSION : 'unknown') . ' (+WordPress)',
                );
                $res = wp_remote_head($final, $args);
                if (is_wp_error($res)) {
                    $res = wp_remote_get($final, $args);
                }

                $code = 0;
                $ok = 0;
                if (!is_wp_error($res)) {
                    $code = (int) wp_remote_retrieve_response_code($res);
                    if ($code >= 200 && $code < 400) { $ok = 1; }
                }

                update_post_meta($health_id, '_aegislink_health_ok', $ok);
                update_post_meta($health_id, '_aegislink_health_code', $code);
                update_post_meta($health_id, '_aegislink_health_checked', gmdate('Y-m-d H:i:s'));

                if ($ok) {
    /* translators: %d is the HTTP status code returned by the target URL. */
    $notices[] = sprintf(__('Health check updated: OK (HTTP %d).', 'aegislink'), $code);
} else {
    /* translators: %d is the HTTP status code returned by the target URL. */
    $notices[] = sprintf(__('Health check updated: Broken (HTTP %d).', 'aegislink'), $code);
}
            }
        }

        // IMPORTANT: This page file is rendered inside wp-admin after headers may already be sent.
        // If we attempt a PHP header redirect too late, it can result in a blank page/tab.
        // We therefore:
        //  1) Try a safe PHP redirect if headers are still available.
        //  2) Otherwise, keep rendering this page and remove the healthcheck param via History API.
        $redirect = admin_url('admin.php?page=aegislink&tab=shorturl&view=manage');
        $redirect = add_query_arg('aegislink_health', (!empty($errors) ? 'error' : 'ok'), $redirect);

        if (!headers_sent()) {
            wp_safe_redirect($redirect);
            exit;
        }

        // Prevent re-running the healthcheck on refresh by removing the param without navigation.
        // We'll keep rendering the page with the updated meta + notices.
        unset($_GET['healthcheck']);
        $healthcheck_replace_url = $redirect;
    }
}


$sanitize_csv_list = function($s) {
    $s = (string) $s;
    $parts = preg_split('/\s*,\s*/', $s, -1, PREG_SPLIT_NO_EMPTY);
    $out = array();
    foreach ($parts as $p) {
        $p = trim($p);
        if ($p === '') { continue; }
        $out[] = sanitize_text_field($p);
    }
    return array_values(array_unique($out));
};

if ($view === 'importexport' && isset($_GET['export']) && (int) $_GET['export'] === 1) {
    check_admin_referer('aegislink_shortlinks_export');

    $q = new WP_Query(array(
        'post_type' => Shortlinks::CPT,
        'post_status' => 'publish',
        'posts_per_page' => -1,
        'orderby' => 'title',
        'order' => 'ASC',
        'no_found_rows' => true,
    ));

    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=aegislink-shortlinks-export-' . gmdate('Ymd-His') . '.csv');

    // Lightweight CSV writer (avoids direct fopen()/fclose() usage to satisfy Plugin Check).
    $aegislink_csv_line = static function(array $fields) {
        $out = array();
        foreach ($fields as $v) {
            $v = (string) $v;
            $v = str_replace('"', '""', $v);
            if (preg_match('/[",
]/', $v)) {
                $v = '"' . $v . '"';
            }
            $out[] = $v;
        }
        return implode(',', $out) . "
";
    };

    // phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV export output is a file download (not HTML).
    echo $aegislink_csv_line(array(
        'slug','title','target_url','redirect','track_clicks','rel_nofollow','rel_sponsored','notes',
        'groups','tags',
        'utm_enable','utm_source','utm_medium','utm_campaign','utm_term','utm_content'
    ));

    foreach ($q->posts as $p) {
        $groups = wp_get_post_terms($p->ID, 'aegislink_link_group', array('fields' => 'names'));
        $tags   = wp_get_post_terms($p->ID, 'aegislink_link_tag', array('fields' => 'names'));

        echo $aegislink_csv_line(array(
            $p->post_name,
            $p->post_title,
            (string) get_post_meta($p->ID, '_aegislink_target', true),
            (string) get_post_meta($p->ID, '_aegislink_redirect', true),
            (string) get_post_meta($p->ID, '_aegislink_track', true),
            (string) get_post_meta($p->ID, '_aegislink_rel_nofollow', true),
            (string) get_post_meta($p->ID, '_aegislink_rel_sponsored', true),
            (string) get_post_meta($p->ID, '_aegislink_notes', true),
            implode(',', (array) $groups),
            implode(',', (array) $tags),
            (string) get_post_meta($p->ID, '_aegislink_utm_enable', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_source', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_medium', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_campaign', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_term', true),
            (string) get_post_meta($p->ID, '_aegislink_utm_content', true),
        ));
    }

    // phpcs:enable WordPress.Security.EscapeOutput.OutputNotEscaped
    exit;
}

if (isset($_POST['aegislink_shortlinks_settings_save'])) {
    check_admin_referer('aegislink_shortlinks_settings_save');

    $prefix = isset($_POST['prefix']) ? sanitize_title((string) wp_unslash($_POST['prefix'])) : 'go';
    if ($prefix === '') { $prefix = 'go'; }

    $default_redirect = isset($_POST['default_redirect']) ? (int) $_POST['default_redirect'] : 302;
    if (!in_array($default_redirect, array(301,302,307), true)) { $default_redirect = 302; }

    $track_clicks = isset($_POST['track_clicks']) ? 1 : 0;
    $rel_nf = isset($_POST['default_rel_nofollow']) ? 1 : 0;
    $rel_sp = isset($_POST['default_rel_sponsored']) ? 1 : 0;

    $qr_provider = isset($_POST['qr_provider']) ? sanitize_text_field((string) wp_unslash($_POST['qr_provider'])) : 'qrserver';
    if (!in_array($qr_provider, array('qrserver','disabled'), true)) { $qr_provider = 'qrserver'; }

    $woo_share_enabled = isset($_POST['woo_share_enabled']) ? 1 : 0;

    $settings = array(
        'prefix' => $prefix,
        'default_redirect' => $default_redirect,
        'track_clicks' => $track_clicks,
        'default_rel_nofollow' => $rel_nf,
        'default_rel_sponsored' => $rel_sp,
        'qr_provider' => $qr_provider,
        'woo_share_enabled' => $woo_share_enabled,
    );

    update_option(Shortlinks::OPT_SETTINGS, $settings, false);
    delete_option('aegislink_shortlinks_rewrite_flushed'); // ensure flush happens once
    $notices[] = __('Settings saved.', 'aegislink');
}

$edit_id = isset($_GET['edit']) ? absint($_GET['edit']) : 0;
$editing = null;
if ($edit_id > 0) {
    $p = get_post($edit_id);
    if ($p && $p->post_type === Shortlinks::CPT) { $editing = $p; }
}

if (isset($_POST['aegislink_shortlink_save'])) {
    check_admin_referer('aegislink_shortlink_save');

    $title  = isset($_POST['title']) ? sanitize_text_field((string) wp_unslash($_POST['title'])) : '';
    $target = isset($_POST['target_url']) ? esc_url_raw((string) wp_unslash($_POST['target_url'])) : '';
    $slug   = isset($_POST['slug']) ? sanitize_title((string) wp_unslash($_POST['slug'])) : '';
    $redir  = isset($_POST['redirect']) ? (int) $_POST['redirect'] : 302;
    $track  = isset($_POST['track_clicks']) ? 1 : 0;
    $rel_nf = isset($_POST['rel_nofollow']) ? 1 : 0;
    $rel_sp = isset($_POST['rel_sponsored']) ? 1 : 0;
    $notes  = isset($_POST['notes']) ? sanitize_textarea_field((string) wp_unslash($_POST['notes'])) : '';


    $post_data = wp_unslash( $_POST );
    $groups_raw = isset( $post_data['groups'] ) ? (string) $post_data['groups'] : '';
    $tags_raw   = isset( $post_data['tags'] ) ? (string) $post_data['tags'] : '';
    $groups = ($groups_raw !== '') ? $sanitize_csv_list($groups_raw) : array();
    $tags   = ($tags_raw !== '') ? $sanitize_csv_list($tags_raw) : array();

    $utm_enable  = isset($_POST['utm_enable']) ? 1 : 0;
    $utm_source  = isset($_POST['utm_source']) ? sanitize_text_field((string) wp_unslash($_POST['utm_source'])) : '';
    $utm_medium  = isset($_POST['utm_medium']) ? sanitize_text_field((string) wp_unslash($_POST['utm_medium'])) : '';
    $utm_campaign= isset($_POST['utm_campaign']) ? sanitize_text_field((string) wp_unslash($_POST['utm_campaign'])) : '';
    $utm_term    = isset($_POST['utm_term']) ? sanitize_text_field((string) wp_unslash($_POST['utm_term'])) : '';
    $utm_content = isset($_POST['utm_content']) ? sanitize_text_field((string) wp_unslash($_POST['utm_content'])) : '';

    $incoming_edit_id = isset($_POST['edit_id']) ? absint($_POST['edit_id']) : 0;

    if ($title === '') { $errors[] = __('Title is required.', 'aegislink'); }
    if ($target === '' || !preg_match('#^https?://#i', $target)) { $errors[] = __('Target URL must be a valid http(s) URL.', 'aegislink'); }
    if (!in_array($redir, array(301,302,307), true)) { $redir = 302; }

    if ($slug === '') { $slug = $short->generate_slug($title); }
    if ($short->slug_exists($slug, $incoming_edit_id)) {
        $errors[] = __('That Slug is already in use. Please choose another.', 'aegislink');
    }

    if (empty($errors)) {
        $postarr = array(
            'post_type' => Shortlinks::CPT,
            'post_title' => $title,
            'post_status' => 'publish',
            'post_name' => $slug,
        );

        if ($incoming_edit_id > 0) {
            $postarr['ID'] = $incoming_edit_id;
            $saved_id = wp_update_post($postarr, true);
        } else {
            $saved_id = wp_insert_post($postarr, true);
        }

        if (is_wp_error($saved_id)) {
            $errors[] = $saved_id->get_error_message();
        } else {
            update_post_meta($saved_id, '_aegislink_target', $target);
            update_post_meta($saved_id, '_aegislink_redirect', $redir);
            update_post_meta($saved_id, '_aegislink_track', $track);
            update_post_meta($saved_id, '_aegislink_rel_nofollow', $rel_nf);
            update_post_meta($saved_id, '_aegislink_rel_sponsored', $rel_sp);
            update_post_meta($saved_id, '_aegislink_notes', $notes);

            update_post_meta($saved_id, '_aegislink_utm_enable', $utm_enable);
            update_post_meta($saved_id, '_aegislink_utm_source', $utm_source);
            update_post_meta($saved_id, '_aegislink_utm_medium', $utm_medium);
            update_post_meta($saved_id, '_aegislink_utm_campaign', $utm_campaign);
            update_post_meta($saved_id, '_aegislink_utm_term', $utm_term);
            update_post_meta($saved_id, '_aegislink_utm_content', $utm_content);
            wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false);
            wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false);
            update_post_meta($p->ID, '_aegislink_shortlink_id', $saved_id);
            $notices[] = __('Short Link saved.', 'aegislink');

            wp_safe_redirect(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage'));
            exit;
        }
    }
}

if (isset($_GET['delete']) && (int) $_GET['delete'] > 0 && isset($_GET['_wpnonce'])) {
    $del_id = absint($_GET['delete']);
    if (wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'aegislink_shortlink_delete_' . $del_id)) {
        $p = get_post($del_id);
        if ($p && $p->post_type === Shortlinks::CPT) {
            wp_trash_post($del_id);
            $notices[] = __('Short Link deleted.', 'aegislink');
            $view = 'manage';
        }
    }
}

// Bulk tab actions (approve/delete/regenerate) and per-row inline edits.
// NOTE: This page renders a single bulk form (no nested forms) so nonces and checked IDs post reliably.
if ($view === 'bulk' && isset($_POST['aegislink_bulk_action']) && isset($_POST['aegislink_bulk_apply'])) {
    if (!isset($_POST['aegislink_bulk_actions_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['aegislink_bulk_actions_nonce'])), 'aegislink_bulk_actions')) {
        wp_die(esc_html__('The link you followed has expired.', 'aegislink'));
    }

    $action = isset($_POST['bulk_action']) ? sanitize_key((string) wp_unslash($_POST['bulk_action'])) : '';
    $post_data = wp_unslash( $_POST );
    $ids = ( isset( $post_data['bulk_ids'] ) && is_array( $post_data['bulk_ids'] ) ) ? array_map( 'absint', (array) $post_data['bulk_ids'] ) : array();
    $ids = array_values(array_filter($ids));

    $done = 0;
    if (!empty($ids) && in_array($action, array('approve','delete','regenerate'), true)) {
        foreach ($ids as $id) {
            $p = get_post($id);
            if (!$p || $p->post_type !== Shortlinks::CPT) { continue; }

            if ($action === 'approve') {
                if ($p->post_status !== 'publish') {
                    wp_update_post(array('ID' => $id, 'post_status' => 'publish'));
                }
                $done++;
                continue;
            }

            if ($action === 'delete') {
                wp_trash_post($id);
                $done++;
                continue;
            }

            if ($action === 'regenerate') {
                $base = $p->post_title ? (string) $p->post_title : $p->post_name;
                $new_slug = $short->generate_slug($base);
                if ($new_slug === '') { $new_slug = 'link'; }
                $try = $new_slug;
                $i = 0;
                while ($short->slug_exists($try, $id) && $i < 25) {
                    $try = $new_slug . '-' . strtolower(wp_generate_password(4, false, false));
                    $i++;
                }
                if (!$short->slug_exists($try, $id)) {
                    wp_update_post(array('ID' => $id, 'post_name' => $try));
                    $done++;
                }
                continue;
            }
        }
    }

    if ($action === 'approve') {
        // translators: %d: Number of short links approved.
        $notices[] = sprintf(__('Approved: %d', 'aegislink'), (int) $done);
    } elseif ($action === 'delete') {
        // translators: %d: Number of short links deleted.
        $notices[] = sprintf(__('Deleted: %d', 'aegislink'), (int) $done);
    } elseif ($action === 'regenerate') {
        // translators: %d: Number of short links regenerated.
        $notices[] = sprintf(__('Regenerated: %d', 'aegislink'), (int) $done);
    }
}

if ($view === 'bulk' && isset($_POST['aegislink_bulk_inline_save'])) {
    if (!isset($_POST['aegislink_bulk_inline_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['aegislink_bulk_inline_nonce'])), 'aegislink_bulk_inline_save')) {
        wp_die(esc_html__('The link you followed has expired.', 'aegislink'));
    }

    $id = absint($_POST['aegislink_bulk_inline_save']);
    $p = $id ? get_post($id) : null;
    if ($p && $p->post_type === Shortlinks::CPT) {
        $new_slug = (isset($_POST['row_slug'][$id])) ? sanitize_title((string) wp_unslash($_POST['row_slug'][$id])) : '';
        $new_target = (isset($_POST['row_target'][$id])) ? esc_url_raw((string) wp_unslash($_POST['row_target'][$id])) : '';
        $new_redir = (isset($_POST['row_redirect'][$id])) ? (int) $_POST['row_redirect'][$id] : 302;
        if (!in_array($new_redir, array(301,302,307), true)) { $new_redir = 302; }

        $errs = array();
        if ($new_target === '' || !preg_match('#^https?://#i', $new_target)) {
            $errs[] = __('Target URL must be a valid http(s) URL.', 'aegislink');
        }

        // Slug is optional (empty => keep existing), but if provided it must be unique.
        if ($new_slug !== '' && $short->slug_exists($new_slug, $id)) {
            $errs[] = __('That slug is already in use.', 'aegislink');
        }

        if (!empty($errs)) {
            foreach ($errs as $e) { $errors[] = $e; }
        } else {
            // Update slug (post_name) only if provided and changed.
            if ($new_slug !== '' && $new_slug !== $p->post_name) {
                wp_update_post(array('ID' => $id, 'post_name' => $new_slug));
            }

            update_post_meta($id, '_aegislink_target', $new_target);
            update_post_meta($id, '_aegislink_redirect', $new_redir);

            $notices[] = __('Row updated.', 'aegislink');
        }
    }
}

if ($view === 'bulk' && isset($_POST['aegislink_bulk_create'])) {
    check_admin_referer('aegislink_bulk_create');

    $mode = isset($_POST['bulk_mode']) ? sanitize_key((string) wp_unslash($_POST['bulk_mode'])) : 'urls';

    $redir  = isset($_POST['bulk_redirect']) ? (int) $_POST['bulk_redirect'] : 302;
    if (!in_array($redir, array(301,302,307), true)) { $redir = 302; }

    $track  = isset($_POST['bulk_track']) ? 1 : 0;
    $rel_nf = isset($_POST['bulk_rel_nofollow']) ? 1 : 0;
    $rel_sp = isset($_POST['bulk_rel_sponsored']) ? 1 : 0;

        $post_data = wp_unslash( $_POST );
    $bulk_groups_raw = isset( $post_data['bulk_groups'] ) ? (string) $post_data['bulk_groups'] : '';
    $bulk_tags_raw   = isset( $post_data['bulk_tags'] ) ? (string) $post_data['bulk_tags'] : '';
    $groups = ($bulk_groups_raw !== '') ? $sanitize_csv_list($bulk_groups_raw) : array();
    $tags   = ($bulk_tags_raw !== '') ? $sanitize_csv_list($bulk_tags_raw) : array();

    // Approval workflow (Pretty Links-style): bulk-created links can be created as Draft,
    // then approved/published from the Bulk tab list below.
    $require_approval = isset($_POST['bulk_require_approval']) ? 1 : 0;
    $bulk_post_status = $require_approval ? 'draft' : 'publish';

    $created = 0;
    $skipped = 0;

    if ($mode === 'posts') {
        $post_type = isset($_POST['bulk_post_type']) ? sanitize_key((string) wp_unslash($_POST['bulk_post_type'])) : 'post';
        $limit = isset($_POST['bulk_limit']) ? absint($_POST['bulk_limit']) : 100;

        $pq = new WP_Query(array(
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => $limit,
            'orderby' => 'date',
            'order' => 'DESC',
            'no_found_rows' => true,
        ));

        foreach ($pq->posts as $p) {
            $target = get_permalink($p);
            if (!$target) { $skipped++; continue; }
            $slug = sanitize_title($p->post_name);
            if ($slug === '') { $slug = $short->generate_slug($p->post_title); }
            if ($short->slug_exists($slug, 0)) { $skipped++; continue; }

            $saved_id = wp_insert_post(array(
                'post_type' => Shortlinks::CPT,
                'post_title' => $p->post_title,
                'post_status' => $bulk_post_status,
                'post_name' => $slug,
            ), true);

            if (is_wp_error($saved_id)) { $skipped++; continue; }

            update_post_meta($saved_id, '_aegislink_target', esc_url_raw($target));
            update_post_meta($saved_id, '_aegislink_redirect', $redir);
            update_post_meta($saved_id, '_aegislink_track', $track);
            update_post_meta($saved_id, '_aegislink_rel_nofollow', $rel_nf);
            update_post_meta($saved_id, '_aegislink_rel_sponsored', $rel_sp);

            wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false);
            wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false);

            $created++;
        }
    } elseif ($mode === 'terms') {
        $taxonomy = isset($_POST['bulk_taxonomy']) ? sanitize_key((string) wp_unslash($_POST['bulk_taxonomy'])) : '';
        if ($taxonomy === '' || !taxonomy_exists($taxonomy)) {
            $skipped++;
        } else {
            $limit = isset($_POST['bulk_term_limit']) ? absint($_POST['bulk_term_limit']) : 100;
            if ($limit <= 0) { $limit = 100; }

            $strategy = isset($_POST['bulk_term_slug_strategy']) ? sanitize_key((string) wp_unslash($_POST['bulk_term_slug_strategy'])) : 'term';
            if (!in_array($strategy, array('term','cat','tag'), true)) { $strategy = 'term'; }

            $tq = get_terms(array(
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
                'number' => $limit,
                'orderby' => 'count',
                'order' => 'DESC',
            ));

            if (is_wp_error($tq) || !is_array($tq)) {
                $skipped++;
            } else {
                foreach ($tq as $term) {
                    if (!is_object($term) || !isset($term->term_id)) { $skipped++; continue; }

                    $target = get_term_link($term);
                    if (is_wp_error($target) || !$target) { $skipped++; continue; }

                    $slug = Bulk_Terms::build_slug_for_term($term, $strategy);
                    if ($slug === '' || $short->slug_exists($slug, 0)) { $skipped++; continue; }

                    $title = isset($term->name) ? (string) $term->name : $slug;

                    $saved_id = wp_insert_post(array(
                        'post_type' => Shortlinks::CPT,
                        'post_title' => $title,
                        'post_status' => $bulk_post_status,
                        'post_name' => $slug,
                    ), true);

                    if (is_wp_error($saved_id)) { $skipped++; continue; }

                    update_post_meta($saved_id, '_aegislink_target', esc_url_raw((string) $target));
                    update_post_meta($saved_id, '_aegislink_redirect', $redir);
                    update_post_meta($saved_id, '_aegislink_track', $track);
                    update_post_meta($saved_id, '_aegislink_rel_nofollow', $rel_nf);
                    update_post_meta($saved_id, '_aegislink_rel_sponsored', $rel_sp);

                    wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false);
                    wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false);

                    update_term_meta((int) $term->term_id, '_aegislink_shortlink_id', $saved_id);

                    $created++;
                }
            }
        }

    } else {
        $raw = isset($_POST['bulk_urls']) ? sanitize_textarea_field((string) wp_unslash($_POST['bulk_urls'])) : '';
        $lines = preg_split('/\r\n|\r|\n/', $raw);
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '') { continue; }

            $parts = str_getcsv($line);
            $target = isset($parts[0]) ? esc_url_raw(trim((string) $parts[0])) : '';
            $slug = isset($parts[1]) ? sanitize_title(trim((string) $parts[1])) : '';
            $title = isset($parts[2]) ? sanitize_text_field(trim((string) $parts[2])) : '';

            if ($target === '' || !preg_match('#^https?://#i', $target)) { $skipped++; continue; }

            if ($title === '') { $title = $target; }
            if ($slug === '') { $slug = $short->generate_slug($title); }

            if ($short->slug_exists($slug, 0)) { $skipped++; continue; }

            $saved_id = wp_insert_post(array(
                'post_type' => Shortlinks::CPT,
                'post_title' => $title,
                'post_status' => $bulk_post_status,
                'post_name' => $slug,
            ), true);

            if (is_wp_error($saved_id)) { $skipped++; continue; }

            update_post_meta($saved_id, '_aegislink_target', $target);
            update_post_meta($saved_id, '_aegislink_redirect', $redir);
            update_post_meta($saved_id, '_aegislink_track', $track);
            update_post_meta($saved_id, '_aegislink_rel_nofollow', $rel_nf);
            update_post_meta($saved_id, '_aegislink_rel_sponsored', $rel_sp);

            wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false);
            wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false);

            $created++;
        }
    }

    /* translators: 1: number created, 2: number skipped. */
            $notices[] = sprintf(__('Bulk create finished. Created: %1$d, Skipped: %2$d', 'aegislink'), (int) $created, (int) $skipped);
}

// Bulk tab: checkbox actions (Approve / Delete / Regenerate)
if ($view === 'bulk' && isset($_POST['aegislink_bulk_action'])) {
    check_admin_referer('aegislink_bulk_action');

    $action = isset($_POST['bulk_action']) ? sanitize_key((string) wp_unslash($_POST['bulk_action'])) : '';
        $post_data = wp_unslash( $_POST );
        $ids_raw = ( isset( $post_data['bulk_ids'] ) && is_array( $post_data['bulk_ids'] ) ) ? (array) $post_data['bulk_ids'] : array();
    $ids = array();
    foreach ($ids_raw as $idv) {
        $id = absint( (string) $idv );
        if ($id > 0) { $ids[] = $id; }
    }
    $ids = array_values(array_unique($ids));

    if (empty($ids)) {
        $errors[] = __('Please select at least one short link.', 'aegislink');
    } else {
        $done = 0;
        foreach ($ids as $id) {
            $p = get_post($id);
            if (!$p || $p->post_type !== Shortlinks::CPT) { continue; }

            if ($action === 'approve') {
                $r = wp_update_post(array('ID' => $id, 'post_status' => 'publish'), true);
                if (!is_wp_error($r)) { $done++; }
            } elseif ($action === 'delete') {
                $r = wp_delete_post($id, true);
                if ($r) { $done++; }
            } elseif ($action === 'regenerate') {
                $title = $p->post_title ? $p->post_title : (string) get_post_meta($id, '_aegislink_target', true);
                $new_slug = $short->generate_slug((string) $title);
                // Ensure unique.
                $try = 0;
                while ($short->slug_exists($new_slug, $id) && $try < 50) {
                    $new_slug = sanitize_title($new_slug . '-' . strtolower(wp_generate_password(4, false, false)));
                    $try++;
                }
                if (!$short->slug_exists($new_slug, $id)) {
                    $r = wp_update_post(array('ID' => $id, 'post_name' => $new_slug), true);
                    if (!is_wp_error($r)) { $done++; }
                }
            }
        }

        if ($done > 0) {
            if ($action === 'approve') {
                // translators: %d: Number of short links approved.
                $notices[] = sprintf(__('Approved %d short link(s).', 'aegislink'), (int) $done);
            } elseif ($action === 'delete') {
                // translators: %d: Number of short links deleted.
                $notices[] = sprintf(__('Deleted %d short link(s).', 'aegislink'), (int) $done);
            } elseif ($action === 'regenerate') {
                // translators: %d: Number of short links regenerated.
                $notices[] = sprintf(__('Regenerated %d short link slug(s).', 'aegislink'), (int) $done);
            }
        }
    }
}

// Bulk tab: per-row inline edit (slug / target / redirect)
if ($view === 'bulk' && isset($_POST['aegislink_bulk_inline_save'])) {
    check_admin_referer('aegislink_bulk_inline_save');

    $link_id = isset($_POST['link_id']) ? absint($_POST['link_id']) : 0;
    $p = $link_id ? get_post($link_id) : null;

    if (!$p || $p->post_type !== Shortlinks::CPT) {
        $errors[] = __('Invalid short link.', 'aegislink');
    } else {
        $new_slug = isset($_POST['inline_slug']) ? sanitize_title((string) wp_unslash($_POST['inline_slug'])) : '';
        $new_target = isset($_POST['inline_target']) ? esc_url_raw((string) wp_unslash($_POST['inline_target'])) : '';
        $new_redir = isset($_POST['inline_redirect']) ? (int) $_POST['inline_redirect'] : 302;
        if (!in_array($new_redir, array(301,302,307), true)) { $new_redir = 302; }

        if ($new_target === '' || !preg_match('#^https?://#i', $new_target)) {
            $errors[] = __('Target URL is required (must start with http:// or https://).', 'aegislink');
        } else {
            $post_update = array('ID' => $link_id);
            $needs_update = false;

            if ($new_slug !== '' && $new_slug !== $p->post_name) {
                if ($short->slug_exists($new_slug, $link_id)) {
                    $errors[] = __('That slug is already in use.', 'aegislink');
                } else {
                    $post_update['post_name'] = $new_slug;
                    $needs_update = true;
                }
            }

            if (empty($errors)) {
                if ($needs_update) {
                    $r = wp_update_post($post_update, true);
                    if (is_wp_error($r)) {
                        $errors[] = $r->get_error_message();
                    }
                }

                if (empty($errors)) {
                    update_post_meta($link_id, '_aegislink_target', $new_target);
                    update_post_meta($link_id, '_aegislink_redirect', $new_redir);
                    $notices[] = __('Short link updated.', 'aegislink');
                }
            }
        }
    }
}

if ($view === 'importexport' && isset($_POST['aegislink_import_csv'])) {
    check_admin_referer('aegislink_shortlinks_import');

    if (!isset($_FILES['csv_file']) || empty($_FILES['csv_file']['tmp_name'])) {
        $errors[] = __('Please choose a CSV file.', 'aegislink');
    } else {
        $tmp = sanitize_text_field((string) ($_FILES['csv_file']['tmp_name'] ?? ''));
        if ($tmp === '' || !is_uploaded_file($tmp)) {
            $errors[] = __('Invalid upload.', 'aegislink');
        }
        global $wp_filesystem;
if (empty($wp_filesystem)) {
    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem();
}

$csv = ($wp_filesystem && isset($tmp) && $tmp !== '') ? $wp_filesystem->get_contents($tmp) : false;
if ($csv === false || $csv === '') {
    $notices[] = __('Could not read the uploaded CSV file.', 'aegislink');
} else {
    $lines = preg_split("/\r\n|\n|\r/", (string) $csv);
    $rows  = array();

    foreach ($lines as $line) {
        $line = trim((string) $line);
        if ($line === '') {
            continue;
        }
        $rows[] = str_getcsv($line);
    }

    if (empty($rows)) {
        $notices[] = __('CSV file appears to be empty.', 'aegislink');
    } else {
        $header = array_shift($rows);
        $map = array();

        foreach ((array) $header as $i => $col) {
            $key = strtolower(trim((string) $col));
            if ($key !== '') {
                $map[$key] = (int) $i;
            }
        }

        foreach ($rows as $row) {

                $slug  = isset($map['slug']) ? sanitize_title((string) ($row[$map['slug']] ?? '')) : '';
                $title = isset($map['title']) ? sanitize_text_field((string) ($row[$map['title']] ?? '')) : '';
                $target= isset($map['target_url']) ? esc_url_raw((string) ($row[$map['target_url']] ?? '')) : '';

                if ($target === '' || !preg_match('#^https?://#i', $target)) { $skipped++; continue; }
                if ($title === '') { $title = $target; }
                if ($slug === '') { $slug = $short->generate_slug($title); }

                $existing = get_page_by_path($slug, OBJECT, Shortlinks::CPT);

                $postarr = array(
                    'post_type' => Shortlinks::CPT,
                    'post_title' => $title,
                    'post_status' => 'publish',
                    'post_name' => $slug,
                );

                if ($existing && $existing->ID) {
                    $postarr['ID'] = (int) $existing->ID;
                    $saved_id = wp_update_post($postarr, true);
                } else {
                    $saved_id = wp_insert_post($postarr, true);
                }

                if (is_wp_error($saved_id)) { $skipped++; continue; }

                $redir = isset($map['redirect']) ? (int) ($row[$map['redirect']] ?? 302) : 302;
                if (!in_array($redir, array(301,302,307), true)) { $redir = 302; }

                update_post_meta($saved_id, '_aegislink_target', $target);
                update_post_meta($saved_id, '_aegislink_redirect', $redir);
                update_post_meta($saved_id, '_aegislink_track', isset($map['track_clicks']) ? (int) ($row[$map['track_clicks']] ?? 1) : 1);
                update_post_meta($saved_id, '_aegislink_rel_nofollow', isset($map['rel_nofollow']) ? (int) ($row[$map['rel_nofollow']] ?? 1) : 1);
                update_post_meta($saved_id, '_aegislink_rel_sponsored', isset($map['rel_sponsored']) ? (int) ($row[$map['rel_sponsored']] ?? 0) : 0);
                update_post_meta($saved_id, '_aegislink_notes', isset($map['notes']) ? sanitize_textarea_field((string) ($row[$map['notes']] ?? '')) : '');

                update_post_meta($saved_id, '_aegislink_utm_enable', isset($map['utm_enable']) ? (int) ($row[$map['utm_enable']] ?? 0) : 0);
                update_post_meta($saved_id, '_aegislink_utm_source', isset($map['utm_source']) ? sanitize_text_field((string) ($row[$map['utm_source']] ?? '')) : '');
                update_post_meta($saved_id, '_aegislink_utm_medium', isset($map['utm_medium']) ? sanitize_text_field((string) ($row[$map['utm_medium']] ?? '')) : '');
                update_post_meta($saved_id, '_aegislink_utm_campaign', isset($map['utm_campaign']) ? sanitize_text_field((string) ($row[$map['utm_campaign']] ?? '')) : '');
                update_post_meta($saved_id, '_aegislink_utm_term', isset($map['utm_term']) ? sanitize_text_field((string) ($row[$map['utm_term']] ?? '')) : '');
                update_post_meta($saved_id, '_aegislink_utm_content', isset($map['utm_content']) ? sanitize_text_field((string) ($row[$map['utm_content']] ?? '')) : '');

                $groups = isset($map['groups']) ? $sanitize_csv_list((string) ($row[$map['groups']] ?? '')) : array();
                $tags   = isset($map['tags']) ? $sanitize_csv_list((string) ($row[$map['tags']] ?? '')) : array();
                if (!empty($groups)) { wp_set_post_terms($saved_id, $groups, 'aegislink_link_group', false); }
                if (!empty($tags)) { wp_set_post_terms($saved_id, $tags, 'aegislink_link_tag', false); }

                if ($existing && $existing->ID) { $updated++; } else { $created++; }
        }
    }
}

            /* translators: 1: number created, 2: number updated, 3: number skipped. */
            $notices[] = sprintf(__('Import finished. Created: %1$d, Updated: %2$d, Skipped: %3$d', 'aegislink'), (int) $created, (int) $updated, (int) $skipped);
        }
    }


$settings = get_option(Shortlinks::OPT_SETTINGS, array());
$settings = wp_parse_args($settings, array(
    'prefix' => 'go',
    'default_redirect' => 302,
    'track_clicks' => 1,
    'default_rel_nofollow' => 1,
    'default_rel_sponsored' => 1,
    'qr_provider' => 'qrserver',
));

?>
<div class="wrap aegislink-wrap">
    <h1><?php echo esc_html__('External Linking', 'aegislink'); ?></h1>

    <div class="nav-tab-wrapper">
        <a class="nav-tab <?php echo $view==='manage'?'nav-tab-active':''; ?>" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage')); ?>"><?php echo esc_html__('Manage', 'aegislink'); ?></a>
        <a class="nav-tab <?php echo $view==='bulk'?'nav-tab-active':''; ?>" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=bulk')); ?>"><?php echo esc_html__('Bulk', 'aegislink'); ?></a>
        <a class="nav-tab <?php echo $view==='analytics'?'nav-tab-active':''; ?>" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=analytics')); ?>"><?php echo esc_html__('Analytics', 'aegislink'); ?></a>
        <a class="nav-tab <?php echo $view==='importexport'?'nav-tab-active':''; ?>" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=importexport')); ?>"><?php echo esc_html__('Import/Export', 'aegislink'); ?></a>
    </div>

    <?php if (!empty($errors)) : ?>
        <div class="notice notice-error"><p><?php echo esc_html(implode(' ', $errors)); ?></p></div>
    <?php endif; ?>
	<?php if (!empty($notices)) : ?>
		<div class="notice notice-success is-dismissible"><p><?php echo esc_html(implode(' ', $notices)); ?></p></div>
	<?php endif; ?>

	<?php if (!empty($healthcheck_replace_url)) : ?>
		<script>
			// Remove the healthcheck param from the URL so refresh won't re-run the check.
			try {
				if (window.history && window.history.replaceState) {
					window.history.replaceState(null, document.title, <?php echo wp_json_encode($healthcheck_replace_url); ?>);
				}
			} catch (e) {}
		</script>
	<?php endif; ?>

    <?php if ($pro_locked) : ?>
        <div class="notice notice-warning" style="margin-top:14px;">
            <p><strong><?php echo esc_html__('PRO Feature:', 'aegislink'); ?></strong> <?php echo esc_html__('ShortURL Links is view-only in the free version. Activate a license to unlock editing, import/export, and settings.', 'aegislink'); ?></p>
            <p style="margin:8px 0 0 0;"><a class="button button-primary" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=tools')); ?>"><?php echo esc_html__('Activate License', 'aegislink'); ?></a></p>
        </div>
        <div class="aegislink-pro-locked" aria-disabled="true">
    <?php endif; ?>

    <?php if ($view === 'manage') : ?>

        <h2><?php echo esc_html__('Settings', 'aegislink'); ?></h2>
        <form method="post">
            <?php wp_nonce_field('aegislink_shortlinks_settings_save'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="prefix"><?php echo esc_html__('Short Link Prefix', 'aegislink'); ?></label></th>
                    <td>
                        <input type="text" class="regular-text" id="prefix" name="prefix" value="<?php echo esc_attr($settings['prefix']); ?>" />
                        <p class="description"><?php echo esc_html__('Example: "go" creates URLs like /go/your-slug', 'aegislink'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Default Redirect', 'aegislink'); ?></th>
                    <td>
                        <select name="default_redirect">
                            <?php foreach (array(301,302,307) as $c) : ?>
                                <option value="<?php echo (int) $c; ?>" <?php selected((int) $settings['default_redirect'], (int) $c); ?>><?php echo (int) $c; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Default Click Tracking', 'aegislink'); ?></th>
                    <td><label><input type="checkbox" name="track_clicks" value="1" <?php checked((int) $settings['track_clicks'], 1); ?> /> <?php echo esc_html__('Enabled', 'aegislink'); ?></label></td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Default rel attributes', 'aegislink'); ?></th>
                    <td>
                        <label style="margin-right:12px;"><input type="checkbox" name="default_rel_nofollow" value="1" <?php checked((int) $settings['default_rel_nofollow'], 1); ?> /> <?php echo esc_html__('nofollow', 'aegislink'); ?></label>
                        <label><input type="checkbox" name="default_rel_sponsored" value="1" <?php checked((int) $settings['default_rel_sponsored'], 1); ?> /> <?php echo esc_html__('sponsored', 'aegislink'); ?></label>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('QR Provider', 'aegislink'); ?></th>
                    <td>
                        <select name="qr_provider">
                            <option value="qrserver" <?php selected($settings['qr_provider'], 'qrserver'); ?>><?php echo esc_html__('QRServer (external image)', 'aegislink'); ?></option>
                            <option value="disabled" <?php selected($settings['qr_provider'], 'disabled'); ?>><?php echo esc_html__('Disabled', 'aegislink'); ?></option>
                        </select>
                        <p class="description"><?php echo esc_html__('QR code images are shown in Share modal. Disable if you prefer not to call external QR image endpoints.', 'aegislink'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('WooCommerce Share Button', 'aegislink'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="woo_share_enabled" value="1" <?php checked(!empty($settings['woo_share_enabled'])); ?> />
                            <?php echo esc_html__('Enable Share button on WooCommerce single product pages when a product short link exists.', 'aegislink'); ?>
                        </label>
                        <p class="description"><?php echo esc_html__('Bulk-generate product short links to auto-enable sharing. The button will only appear if the product has a mapped short link.', 'aegislink'); ?></p>
                    </td>
                </tr>

            </table>
            <p><button type="submit" class="button button-primary" name="aegislink_shortlinks_settings_save" value="1"><?php echo esc_html__('Save Settings', 'aegislink'); ?></button></p>
	        </form>

	        <?php
	        // Bulk pagination links.
	        if ((int) $bulk_q->max_num_pages > 1) {
	            $base = add_query_arg(array(
	                'page' => 'aegislink',
	                'tab' => 'shorturl',
	                'view' => 'bulk',
	                'bulk_pp' => (int) $bulk_pp,
	                'bulk_paged' => '%#%',
	            ), admin_url('admin.php'));

			echo '<div class="tablenav" style="margin-top:12px;"><div class="tablenav-pages">';
			echo wp_kses_post(paginate_links(array(
				'base' => $base,
				'format' => '',
				'current' => (int) $bulk_paged,
				'total' => (int) $bulk_q->max_num_pages,
				'prev_text' => __('&laquo; Prev', 'aegislink'),
				'next_text' => __('Next &raquo;', 'aegislink'),
				'type' => 'plain',
			)));
			echo '</div></div>';

	        }
	        ?>

	        <?php
	        // Bulk pagination links.
	        if ((int) $bulk_q->max_num_pages > 1) {
	            $base = add_query_arg(array(
	                'page' => 'aegislink',
	                'tab' => 'shorturl',
	                'view' => 'bulk',
	                'bulk_pp' => $bulk_pp,
	                'bulk_paged' => '%#%',
	            ), admin_url('admin.php'));
	            echo '<div class="tablenav"><div class="tablenav-pages" style="margin:10px 0;">' . wp_kses_post(paginate_links(array(
	                'base' => $base,
	                'format' => '',
	                'current' => (int) $bulk_paged,
	                'total' => (int) $bulk_q->max_num_pages,
	                'prev_text' => __('&laquo;', 'aegislink'),
	                'next_text' => __('&raquo;', 'aegislink'),
	            ))) . '</div></div>';
	        }
	        ?>

	        <script>
	        // Robust "select all" handler (works even if cached admin.js is stale or another script errors).
	        (function(){
	            var master = document.getElementById('aegislink-bulk-checkall');
	            if (!master) { return; }
	            function boxes(){ return document.querySelectorAll('input.aegislink-bulk-id[type="checkbox"]'); }
	            function setAll(on){
	                var list = boxes();
	                Array.prototype.forEach.call(list, function(cb){ cb.checked = !!on; });
	                master.indeterminate = false;
	            }
	            function syncMaster(){
	                var list = boxes();
	                var total = list.length;
	                var checked = 0;
	                Array.prototype.forEach.call(list, function(cb){ if (cb.checked) { checked++; } });
	                if (total === 0) { master.checked = false; master.indeterminate = false; return; }
	                if (checked === 0) { master.checked = false; master.indeterminate = false; return; }
	                if (checked === total) { master.checked = true; master.indeterminate = false; return; }
	                master.checked = false;
	                master.indeterminate = true;
	            }
	            master.addEventListener('change', function(){ setAll(master.checked); });
	            document.addEventListener('change', function(e){
	                if (e && e.target && e.target.classList && e.target.classList.contains('aegislink-bulk-id')) {
	                    syncMaster();
	                }
	            });
	        })();
	        </script>

        <hr />

        <h2><?php echo $editing ? esc_html__('Edit Short Link', 'aegislink') : esc_html__('Add Short Link', 'aegislink'); ?></h2>

        <?php
        $ed_target = $editing ? (string) get_post_meta($editing->ID, '_aegislink_target', true) : '';
        $ed_redir  = $editing ? (int) get_post_meta($editing->ID, '_aegislink_redirect', true) : (int) $settings['default_redirect'];
        $ed_track  = $editing ? (int) get_post_meta($editing->ID, '_aegislink_track', true) : (int) $settings['track_clicks'];
        $ed_nf     = $editing ? (int) get_post_meta($editing->ID, '_aegislink_rel_nofollow', true) : (int) $settings['default_rel_nofollow'];
        $ed_sp     = $editing ? (int) get_post_meta($editing->ID, '_aegislink_rel_sponsored', true) : (int) $settings['default_rel_sponsored'];
        $ed_notes  = $editing ? (string) get_post_meta($editing->ID, '_aegislink_notes', true) : '';

        $ed_groups = $editing ? wp_get_post_terms($editing->ID, 'aegislink_link_group', array('fields' => 'names')) : array();
        $ed_tags   = $editing ? wp_get_post_terms($editing->ID, 'aegislink_link_tag', array('fields' => 'names')) : array();

        $ed_utm_enable = $editing ? (int) get_post_meta($editing->ID, '_aegislink_utm_enable', true) : 0;
        $ed_utm_source = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_source', true) : '';
        $ed_utm_medium = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_medium', true) : '';
        $ed_utm_campaign = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_campaign', true) : '';
        $ed_utm_term = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_term', true) : '';
        $ed_utm_content = $editing ? (string) get_post_meta($editing->ID, '_aegislink_utm_content', true) : '';
        ?>

        <form method="post">
            <?php wp_nonce_field('aegislink_shortlink_save'); ?>
            <input type="hidden" name="edit_id" value="<?php echo $editing ? (int) $editing->ID : 0; ?>" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="title"><?php echo esc_html__('Title', 'aegislink'); ?></label></th>
                    <td><input type="text" class="regular-text" id="title" name="title" value="<?php echo esc_attr($editing ? $editing->post_title : ''); ?>" required /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="target_url"><?php echo esc_html__('Target URL', 'aegislink'); ?></label></th>
                    <td><input type="url" class="large-text" id="target_url" name="target_url" value="<?php echo esc_attr($ed_target); ?>" placeholder="https://example.com/..." required /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="slug"><?php echo esc_html__('Short Slug', 'aegislink'); ?></label></th>
                    <td>
                        <input type="text" class="regular-text" id="slug" name="slug" value="<?php echo esc_attr($editing ? $editing->post_name : ''); ?>" />
                        <p class="description"><?php echo esc_html__('Leave blank to auto-generate from title.', 'aegislink'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php echo esc_html__('Redirect', 'aegislink'); ?></th>
                    <td>
                        <select name="redirect">
                            <?php foreach (array(301,302,307) as $c) : ?>
                                <option value="<?php echo (int) $c; ?>" <?php selected((int) $ed_redir, (int) $c); ?>><?php echo (int) $c; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('Tracking', 'aegislink'); ?></th>
                    <td><label><input type="checkbox" name="track_clicks" value="1" <?php checked((int) $ed_track, 1); ?> /> <?php echo esc_html__('Track clicks for this link', 'aegislink'); ?></label></td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('rel attributes', 'aegislink'); ?></th>
                    <td>
                        <label style="margin-right:12px;"><input type="checkbox" name="rel_nofollow" value="1" <?php checked((int) $ed_nf, 1); ?> /> <?php echo esc_html__('nofollow', 'aegislink'); ?></label>
                        <label><input type="checkbox" name="rel_sponsored" value="1" <?php checked((int) $ed_sp, 1); ?> /> <?php echo esc_html__('sponsored', 'aegislink'); ?></label>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="groups"><?php echo esc_html__('Groups (comma-separated)', 'aegislink'); ?></label></th>
                    <td><input type="text" class="regular-text" id="groups" name="groups" value="<?php echo esc_attr(implode(', ', (array) $ed_groups)); ?>" placeholder="Affiliates, Partners, Products" /></td>
                </tr>
                <tr>
                    <th scope="row"><label for="tags"><?php echo esc_html__('Tags (comma-separated)', 'aegislink'); ?></label></th>
                    <td><input type="text" class="regular-text" id="tags" name="tags" value="<?php echo esc_attr(implode(', ', (array) $ed_tags)); ?>" placeholder="black-friday, promo, instagram" /></td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('UTM Builder', 'aegislink'); ?></th>
                    <td>
                        <label><input type="checkbox" name="utm_enable" value="1" <?php checked((int) $ed_utm_enable, 1); ?> /> <?php echo esc_html__('Append UTM params on redirect', 'aegislink'); ?></label>
                        <div style="margin-top:10px; display:grid; grid-template-columns: 1fr 1fr; gap:10px; max-width: 760px;">
                            <input type="text" name="utm_source" class="regular-text" placeholder="utm_source" value="<?php echo esc_attr($ed_utm_source); ?>" />
                            <input type="text" name="utm_medium" class="regular-text" placeholder="utm_medium" value="<?php echo esc_attr($ed_utm_medium); ?>" />
                            <input type="text" name="utm_campaign" class="regular-text" placeholder="utm_campaign" value="<?php echo esc_attr($ed_utm_campaign); ?>" />
                            <input type="text" name="utm_term" class="regular-text" placeholder="utm_term" value="<?php echo esc_attr($ed_utm_term); ?>" />
                            <input type="text" name="utm_content" class="regular-text" placeholder="utm_content" value="<?php echo esc_attr($ed_utm_content); ?>" />
                        </div>
                        <p class="description"><?php echo esc_html__('These are added at redirect-time so your stored Target URL remains clean.', 'aegislink'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="notes"><?php echo esc_html__('Notes', 'aegislink'); ?></label></th>
                    <td><textarea class="large-text" rows="3" id="notes" name="notes"><?php echo esc_textarea($ed_notes); ?></textarea></td>
                </tr>
            </table>

            <p>
                <button type="submit" class="button button-primary" name="aegislink_shortlink_save" value="1"><?php echo esc_html__('Save Short Link', 'aegislink'); ?></button>
                <?php if ($editing) : ?>
                    <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage')); ?>"><?php echo esc_html__('Cancel', 'aegislink'); ?></a>
                <?php endif; ?>
            </p>
        </form>

        <hr />

        <h2><?php echo esc_html__('Short Links', 'aegislink'); ?></h2>

        <?php
	        $manage_pp_allowed = array(25, 50, 100);
	        $manage_pp = isset($_GET['manage_pp']) ? (int) $_GET['manage_pp'] : 25;
	        if (!in_array($manage_pp, $manage_pp_allowed, true)) { $manage_pp = 25; }
	        $manage_paged = isset($_GET['manage_paged']) ? max(1, (int) $_GET['manage_paged']) : 1;

	        $q = new WP_Query(array(
	            'post_type' => Shortlinks::CPT,
	            // Show both approved and pending-approval (draft) links.
	            'post_status' => array('publish','draft'),
	            'posts_per_page' => $manage_pp,
	            'paged' => $manage_paged,
	            'orderby' => 'modified',
	            'order' => 'DESC',
	            'no_found_rows' => false,
	        ));
        ?>

	        <div style="display:flex; align-items:center; justify-content:flex-end; gap:12px; margin: 8px 0 10px 0;">
	            <form method="get" style="margin:0;">
	                <input type="hidden" name="page" value="aegislink" />
	                <input type="hidden" name="tab" value="shorturl" />
	                <input type="hidden" name="view" value="manage" />
	                <label for="aegislink-manage-pp" style="font-weight:600;"><?php echo esc_html__('Show', 'aegislink'); ?></label>
	                <select id="aegislink-manage-pp" name="manage_pp" onchange="this.form.submit();">
	                    <?php foreach ($manage_pp_allowed as $pp) : ?>
	                        <option value="<?php echo (int) $pp; ?>" <?php selected((int) $manage_pp, (int) $pp); ?>><?php echo (int) $pp; ?></option>
	                    <?php endforeach; ?>
	                </select>
	                <span style="opacity:.75;"><?php echo esc_html__('per page', 'aegislink'); ?></span>
	            </form>
	        </div>

        <table class="widefat striped">
            <thead>
                <tr>
                    <th><?php echo esc_html__('Short URL', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Target URL', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Groups / Tags', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Status', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Health', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Clicks', 'aegislink'); ?></th>
                    <th><?php echo esc_html__('Actions', 'aegislink'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($q->posts)) : ?>
                    <tr><td colspan="7"><?php echo esc_html__('No short links yet.', 'aegislink'); ?></td></tr>
                <?php else : foreach ($q->posts as $p) : 
                    $short_url = $base_short_url . $p->post_name;
                    $target = (string) get_post_meta($p->ID, '_aegislink_target', true);
                    $clicks = $short->get_click_count($p->ID);

                    $groups = wp_get_post_terms($p->ID, 'aegislink_link_group', array('fields' => 'names'));
                    $tags   = wp_get_post_terms($p->ID, 'aegislink_link_tag', array('fields' => 'names'));

                    $health_ok = (int) get_post_meta($p->ID, '_aegislink_health_ok', true);
                    $health_code = (int) get_post_meta($p->ID, '_aegislink_health_code', true);
                    $health_checked = (string) get_post_meta($p->ID, '_aegislink_health_checked', true);
                    $health_label = $health_checked ? ($health_ok ? __('OK', 'aegislink') : __('Broken', 'aegislink')) : __('Pending', 'aegislink');
                    if ($health_checked) {
                        /* translators: 1: HTTP status code, 2: date/time string. */
                        $health_tip = sprintf(__('HTTP %1$d last checked %2$s', 'aegislink'), $health_code, $health_checked);
                    } else {
                        $health_tip = __('Not checked yet.', 'aegislink');
                    }

                    $status_label = ($p->post_status === 'publish') ? __('Approved', 'aegislink') : __('Needs Approval', 'aegislink');

                    $health_run_url = wp_nonce_url(
                        admin_url('admin.php?page=aegislink&tab=shorturl&view=manage&healthcheck=' . (int) $p->ID),
                        'aegislink_shortlink_healthcheck_' . (int) $p->ID
                    );
                ?>
                    <tr>
                        <td><code><?php echo esc_html($short_url); ?></code></td>
                        <td style="max-width:520px; overflow:hidden; text-overflow:ellipsis;">
                            <a href="<?php echo esc_url($target); ?>" target="_blank" rel="noopener"><?php echo esc_html($target); ?></a>
                        </td>
                        <td>
                            <?php if (!empty($groups)) : ?>
                                <div><strong><?php echo esc_html__('Groups:', 'aegislink'); ?></strong> <?php echo esc_html(implode(', ', $groups)); ?></div>
                            <?php endif; ?>
                            <?php if (!empty($tags)) : ?>
                                <div><strong><?php echo esc_html__('Tags:', 'aegislink'); ?></strong> <?php echo esc_html(implode(', ', $tags)); ?></div>
                            <?php endif; ?>
                            <?php if (empty($groups) && empty($tags)) : ?>
                                <span style="opacity:.7;"><?php echo esc_html__('—', 'aegislink'); ?></span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ($p->post_status === 'publish') : ?>
                                <span class="aegislink-pill aegislink-pill-ok"><?php echo esc_html($status_label); ?></span>
                            <?php else : ?>
                                <span class="aegislink-pill aegislink-pill-pending"><?php echo esc_html($status_label); ?></span>
                            <?php endif; ?>
                        </td>
                        <td title="<?php echo esc_attr($health_tip); ?>">
                            <?php if ($health_label === 'OK') : ?>
                                <span class="aegislink-pill aegislink-pill-ok"><?php echo esc_html($health_label); ?></span>
                            <?php elseif ($health_label === 'Broken') : ?>
                                <span class="aegislink-pill aegislink-pill-bad"><?php echo esc_html($health_label); ?></span>
                            <?php else : ?>
                                <span class="aegislink-pill aegislink-pill-pending"><?php echo esc_html($health_label); ?></span>
                            <?php endif; ?>

						<a class="button button-small" style="margin-left:8px;" href="<?php echo esc_url($health_run_url); ?>"><?php echo esc_html__('Check Health', 'aegislink'); ?></a>
                        </td>
                        <td><?php echo (int) $clicks; ?></td>
                        <td>
                            <a class="button-link" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage&edit=' . (int) $p->ID)); ?>"><?php echo esc_html__('Edit', 'aegislink'); ?></a>
                            <span class="sep">|</span>
                            <a class="button-link" href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage&delete=' . (int) $p->ID), 'aegislink_shortlink_delete_' . (int) $p->ID)); ?>" onclick="return confirm('<?php echo esc_js(__('Delete this short link?', 'aegislink')); ?>');"><?php echo esc_html__('Delete', 'aegislink'); ?></a>
                            <span class="sep">|</span>
                            <a href="#" class="button-link aegislink-share" data-short="<?php echo esc_attr($short_url); ?>" data-title="<?php echo esc_attr($p->post_title); ?>" data-qr="<?php echo esc_attr($settings['qr_provider']); ?>"><?php echo esc_html__('Share', 'aegislink'); ?></a>
                        </td>
                    </tr>
                <?php endforeach; endif; ?>
            </tbody>
        </table>

	        <?php
	        if ((int) $q->max_num_pages > 1) {
	            $base = add_query_arg(array(
	                'page' => 'aegislink',
	                'tab' => 'shorturl',
	                'view' => 'manage',
	                'manage_pp' => (int) $manage_pp,
	            ), admin_url('admin.php'));

	            echo '<div class="tablenav"><div class="tablenav-pages">';
	             $aegislink_paginate = paginate_links(array(
	                'base' => add_query_arg('manage_paged', '%#%', $base),
	                'format' => '',
	                'prev_text' => '&laquo;',
	                'next_text' => '&raquo;',
	                'total' => (int) $q->max_num_pages,
	                'current' => (int) $manage_paged,
	                'type' => 'plain',
	            ));
	            echo '</div></div>';
	        }
	        ?>

        <p class="description" style="margin-top:10px;">
            <?php echo esc_html__('Tip: Health checks run daily in small batches. Broken targets show a warning.', 'aegislink');
                    echo wp_kses_post($aegislink_paginate);
                    ?>
        </p>

        <div id="aegislink-share-modal" class="aegislink-modal" style="display:none;">
            <div class="aegislink-modal-inner">
                <button type="button" class="button-link aegislink-modal-close" aria-label="<?php echo esc_attr__('Close', 'aegislink'); ?>">×</button>
                <h2 id="aegislink-share-title"><?php echo esc_html__('Share Short Link', 'aegislink'); ?></h2>
                <p><input type="text" class="large-text" id="aegislink-share-url" readonly /></p>
                <p>
                    <button type="button" class="button button-primary" id="aegislink-copy-btn"><?php echo esc_html__('Copy', 'aegislink'); ?></button>
                </p>
                <div id="aegislink-qr-wrap" style="margin-top:12px;"></div>
                <div style="margin-top:14px;">
                    <a href="#" target="_blank" rel="noopener" id="aegislink-share-twitter">X</a>
                    <span class="sep">|</span>
                    <a href="#" target="_blank" rel="noopener" id="aegislink-share-facebook">Facebook</a>
                    <span class="sep">|</span>
                    <a href="#" target="_blank" rel="noopener" id="aegislink-share-linkedin">LinkedIn</a>
                </div>
            </div>
        </div>

    <?php elseif ($view === 'bulk') : ?>

        <h2><?php echo esc_html__('Bulk Short Links', 'aegislink'); ?></h2>
        <p class="description"><?php echo esc_html__('Create many short links at once (perfect for WooCommerce product URLs and category-heavy sites).', 'aegislink'); ?></p>

        <form method="post">
            <?php wp_nonce_field('aegislink_bulk_create'); ?>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><?php echo esc_html__('Source Type', 'aegislink'); ?></th>
                    <td>
                        <label><input type="radio" name="bulk_mode" value="urls" checked /> <?php echo esc_html__('Paste URLs', 'aegislink'); ?></label>
                        <br />
                        <label><input type="radio" name="bulk_mode" value="posts" /> <?php echo esc_html__('Generate from Posts (Products, Pages, etc.)', 'aegislink'); ?></label>
                        <br />
                        <label><input type="radio" name="bulk_mode" value="terms" <?php checked($bulk_mode, 'terms'); ?> /> <?php echo esc_html__('Generate from Terms (Categories/Tags/Taxonomies)', 'aegislink'); ?></label>
                    </td>
                </tr>

                <tr class="aegislink-bulk-urls">
                    <th scope="row"><?php echo esc_html__('URLs List', 'aegislink'); ?></th>
                    <td>
                        <textarea class="large-text" rows="8" name="bulk_urls" placeholder="https://example.com/product/blue-shoes/,blue-shoes,Blue Shoes&#10;https://example.com/category/mens/"> </textarea>
                        <p class="description"><?php echo esc_html__('One per line. Optional CSV columns: URL,slug,title', 'aegislink'); ?></p>
                    </td>
                </tr>

                <tr class="aegislink-bulk-posts" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Post Type', 'aegislink'); ?></th>
                    <td>
                        <?php
                        $pts = get_post_types(array('public' => true), 'objects');
                        ?>
                        <select name="bulk_post_type">
                            <?php foreach ($pts as $pt => $obj) : ?>
                                <option value="<?php echo esc_attr($pt); ?>"><?php echo esc_html($obj->labels->singular_name); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php echo esc_html__('If WooCommerce is active, choose Products.', 'aegislink'); ?></p>
                    </td>
                </tr>
                <tr class="aegislink-bulk-terms" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Taxonomy', 'aegislink'); ?></th>
                    <td>
                        <?php
                        $taxes = Bulk_Terms::get_taxonomies();
                        $sel_tax = isset($_POST['bulk_taxonomy']) ? sanitize_key((string) wp_unslash($_POST['bulk_taxonomy'])) : '';
                        if ($sel_tax === '') { $sel_tax = taxonomy_exists('product_cat') ? 'product_cat' : 'category'; }
                        $sel_strategy = isset($_POST['bulk_term_slug_strategy']) ? sanitize_key((string) wp_unslash($_POST['bulk_term_slug_strategy'])) : 'term';
                        if (!in_array($sel_strategy, array('term','cat','tag'), true)) { $sel_strategy = 'term'; }
                        $sel_limit = isset($_POST['bulk_term_limit']) ? absint($_POST['bulk_term_limit']) : 100;
                        if ($sel_limit <= 0) { $sel_limit = 100; }
                        ?>
                        <select name="bulk_taxonomy">
                            <?php foreach ($taxes as $tx) : ?>
                                <option value="<?php echo esc_attr($tx['name']); ?>" <?php selected($sel_tax, $tx['name']); ?>>
                                    <?php echo esc_html($tx['label'] . ' (' . $tx['name'] . ')'); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php echo esc_html__('Select which taxonomy terms to generate short links for (WooCommerce product categories/tags are supported when WooCommerce is active).', 'aegislink'); ?></p>
                    </td>
                </tr>

                <tr class="aegislink-bulk-terms" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Slug Strategy', 'aegislink'); ?></th>
                    <td>
                        <select name="bulk_term_slug_strategy">
                            <option value="term" <?php selected($sel_strategy, 'term'); ?>><?php echo esc_html__('Use term slug', 'aegislink'); ?></option>
                            <option value="cat" <?php selected($sel_strategy, 'cat'); ?>><?php echo esc_html__('Prefix cat-{slug}', 'aegislink'); ?></option>
                            <option value="tag" <?php selected($sel_strategy, 'tag'); ?>><?php echo esc_html__('Prefix tag-{slug}', 'aegislink'); ?></option>
                        </select>
                        <p class="description"><?php echo esc_html__('Default uses the term slug. Optional prefixes help avoid collisions between similar term slugs.', 'aegislink'); ?></p>
                    </td>
                </tr>

                <tr class="aegislink-bulk-terms" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Limit', 'aegislink'); ?></th>
                    <td>
                        <input type="number" min="1" max="2000" name="bulk_term_limit" value="<?php echo esc_attr($sel_limit); ?>" />
                        <p class="description"><?php echo esc_html__('How many terms to process (starts with most-used terms).', 'aegislink'); ?></p>
                    </td>
                </tr>


                <tr class="aegislink-bulk-posts" style="display:none;">
                    <th scope="row"><?php echo esc_html__('Limit', 'aegislink'); ?></th>
                    <td><input type="number" min="1" max="5000" name="bulk_limit" value="100" /></td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('Defaults', 'aegislink'); ?></th>
                    <td>
                        <label><?php echo esc_html__('Redirect:', 'aegislink'); ?>
                            <select name="bulk_redirect">
                                <?php foreach (array(301,302,307) as $c) : ?>
                                    <option value="<?php echo (int) $c; ?>" <?php selected((int) $settings['default_redirect'], (int) $c); ?>><?php echo (int) $c; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </label>
                        <div style="margin-top:8px;">
                            <label style="margin-right:12px;"><input type="checkbox" name="bulk_track" value="1" <?php checked((int) $settings['track_clicks'], 1); ?> /> <?php echo esc_html__('Track clicks', 'aegislink'); ?></label>
                            <label style="margin-right:12px;"><input type="checkbox" name="bulk_rel_nofollow" value="1" <?php checked((int) $settings['default_rel_nofollow'], 1); ?> /> <?php echo esc_html__('nofollow', 'aegislink'); ?></label>
                            <label><input type="checkbox" name="bulk_rel_sponsored" value="1" <?php checked((int) $settings['default_rel_sponsored'], 1); ?> /> <?php echo esc_html__('sponsored', 'aegislink'); ?></label>
                        </div>
                        <div style="margin-top:8px;">
                            <label><input type="checkbox" name="bulk_require_approval" value="1" <?php checked(!isset($_POST['aegislink_bulk_create']) || isset($_POST['bulk_require_approval']), true); ?> /> <?php echo esc_html__('Require approval (create as Draft)', 'aegislink'); ?></label>
                            <p class="description" style="margin:6px 0 0 0;"><?php echo esc_html__('When enabled, bulk-created links appear in the list below as “Needs Approval” until you approve/publish them.', 'aegislink'); ?></p>
                        </div>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php echo esc_html__('Assign Groups / Tags', 'aegislink'); ?></th>
                    <td>
                        <input type="text" class="regular-text" name="bulk_groups" placeholder="Groups (comma-separated)" />
                        <input type="text" class="regular-text" name="bulk_tags" placeholder="Tags (comma-separated)" />
                    </td>
                </tr>
            </table>

            <p><button type="submit" class="button button-primary" name="aegislink_bulk_create" value="1"><?php echo esc_html__('Create Short Links', 'aegislink'); ?></button></p>
        </form>

        <?php
        // Bulk Review Queue: latest links (draft + publish) for quick approval/actions and inline edits.
        // Bulk Review pagination controls (default 25; allow 50/100).
        $bulk_pp_allowed = array(25, 50, 100);
        $bulk_pp = isset($_GET['bulk_pp']) ? absint($_GET['bulk_pp']) : 25;
        if (!in_array($bulk_pp, $bulk_pp_allowed, true)) { $bulk_pp = 25; }
        $bulk_paged = isset($_GET['bulk_paged']) ? max(1, absint($_GET['bulk_paged'])) : 1;

        $bulk_q = new WP_Query(array(
            'post_type' => Shortlinks::CPT,
            'post_status' => array('publish','draft'),
            'posts_per_page' => $bulk_pp,
            'paged' => $bulk_paged,
            'orderby' => 'date',
            'order' => 'DESC',
            'no_found_rows' => false,
        ));
        ?>

        <hr />
        <h2><?php echo esc_html__('Bulk Review & Edit', 'aegislink'); ?></h2>
        <p class="description"><?php echo esc_html__('Review recently created short links. Select rows to Approve / Delete / Regenerate, or edit inline (slug, target, redirect) like Pretty Links.', 'aegislink'); ?></p>

	        <div style="display:flex; align-items:center; justify-content:space-between; gap:12px; margin-top:12px;">
	            <div></div>
	            <form method="get" style="margin:0;">
	                <input type="hidden" name="page" value="aegislink" />
	                <input type="hidden" name="tab" value="shorturl" />
	                <input type="hidden" name="view" value="bulk" />
	                <label for="aegislink-bulk-pp" style="font-weight:600;"><?php echo esc_html__('Show', 'aegislink'); ?></label>
	                <select id="aegislink-bulk-pp" name="bulk_pp" onchange="this.form.submit();">
	                    <?php foreach ($bulk_pp_allowed as $pp) : ?>
	                        <option value="<?php echo (int) $pp; ?>" <?php selected((int) $bulk_pp, (int) $pp); ?>><?php echo (int) $pp; ?></option>
	                    <?php endforeach; ?>
	                </select>
	                <span style="opacity:.75;"><?php echo esc_html__('per page', 'aegislink'); ?></span>
	            </form>
	        </div>

	        <form method="post" style="margin:12px 0 0 0;">
	            <?php wp_nonce_field('aegislink_bulk_actions', 'aegislink_bulk_actions_nonce'); ?>
	            <?php wp_nonce_field('aegislink_bulk_inline_save', 'aegislink_bulk_inline_nonce'); ?>
	            <input type="hidden" name="aegislink_bulk_action" value="1" />

	            <div style="display:flex; gap:10px; align-items:center; margin: 0 0 10px 0;">
                <select name="bulk_action">
                    <option value=""><?php echo esc_html__('Bulk actions…', 'aegislink'); ?></option>
                    <option value="approve"><?php echo esc_html__('Approve (Publish)', 'aegislink'); ?></option>
                    <option value="delete"><?php echo esc_html__('Delete', 'aegislink'); ?></option>
                    <option value="regenerate"><?php echo esc_html__('Regenerate Slug', 'aegislink'); ?></option>
                </select>
                <button type="submit" class="button" name="aegislink_bulk_apply" value="1"><?php echo esc_html__('Apply', 'aegislink'); ?></button>
            </div>

            <div class="aegislink-bulk-review-wrap" style="width:100%; overflow-x:auto;">
            <table class="widefat striped" style="width:100%; max-width:none;">
                <thead>
                    <tr>
                        <th style="width:34px;"><input type="checkbox" id="aegislink-bulk-checkall" /></th>
                        <th style="width:180px;"><?php echo esc_html__('Short Slug', 'aegislink'); ?></th>
                        <th><?php echo esc_html__('Target URL', 'aegislink'); ?></th>
                        <th style="width:90px;"><?php echo esc_html__('Redirect', 'aegislink'); ?></th>
                        <th style="width:140px;"><?php echo esc_html__('Status', 'aegislink'); ?></th>
                        <th style="width:170px;"><?php echo esc_html__('Inline Edit', 'aegislink'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($bulk_q->posts)) : ?>
                        <tr><td colspan="6"><?php echo esc_html__('No short links found yet.', 'aegislink'); ?></td></tr>
                    <?php else : foreach ($bulk_q->posts as $bp) :
                        $sid = (int) $bp->ID;
                        $s_short = $base_short_url . $bp->post_name;
                        $s_target = (string) get_post_meta($sid, '_aegislink_target', true);
                        $s_redir = (int) get_post_meta($sid, '_aegislink_redirect', true);
                        if (!in_array($s_redir, array(301,302,307), true)) { $s_redir = (int) $settings['default_redirect']; }
                        $s_status = ($bp->post_status === 'publish') ? __('Approved', 'aegislink') : __('Needs Approval', 'aegislink');
                    ?>
                        <tr>
                            <td><input type="checkbox" class="aegislink-bulk-id" name="bulk_ids[]" value="<?php echo (int) $sid; ?>" /></td>
	                            <td>
	                                <div><code><?php echo esc_html($s_short); ?></code></div>
	                                <input type="text" class="regular-text" value="<?php echo esc_attr($bp->post_name); ?>" name="row_slug[<?php echo (int) $sid; ?>]" />
	                            </td>
	                            <td>
	                                <input type="url" class="large-text" value="<?php echo esc_attr($s_target); ?>" name="row_target[<?php echo (int) $sid; ?>]" />
	                            </td>
	                            <td>
	                                <select name="row_redirect[<?php echo (int) $sid; ?>]">
                                    <?php foreach (array(301,302,307) as $c) : ?>
                                        <option value="<?php echo (int) $c; ?>" <?php selected((int) $s_redir, (int) $c); ?>><?php echo (int) $c; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                            <td>
                                <?php if ($bp->post_status === 'publish') : ?>
                                    <span class="aegislink-pill aegislink-pill-ok"><?php echo esc_html($s_status); ?></span>
                                <?php else : ?>
                                    <span class="aegislink-pill aegislink-pill-pending"><?php echo esc_html($s_status); ?></span>
                                <?php endif; ?>
                            </td>
	                            <td>
	                                <button type="submit" class="button" name="aegislink_bulk_inline_save" value="<?php echo (int) $sid; ?>"><?php echo esc_html__('Save', 'aegislink'); ?></button>
	                                <a class="button-link" style="margin-left:8px;" href="<?php echo esc_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=manage&edit=' . (int) $sid)); ?>"><?php echo esc_html__('Open', 'aegislink'); ?></a>
	                            </td>
                        </tr>
                    <?php endforeach; endif; ?>
                </tbody>
            </table>
            </div>
	        </form>

	        <?php
	        if ((int) $bulk_q->max_num_pages > 1) {
	            $base = add_query_arg(array(
	                'page' => 'aegislink',
	                'tab' => 'shorturl',
	                'view' => 'bulk',
	                'bulk_pp' => (int) $bulk_pp,
	            ), admin_url('admin.php'));

	            echo '<div class="tablenav"><div class="tablenav-pages">';
	             $aegislink_paginate = paginate_links(array(
	                'base' => add_query_arg('bulk_paged', '%#%', $base),
	                'format' => '',
	                'prev_text' => '&laquo;',
	                'next_text' => '&raquo;',
	                'total' => (int) $bulk_q->max_num_pages,
	                'current' => (int) $bulk_paged,
	                'type' => 'plain',
	            ));
	            echo '</div></div>';
	        }
	        ?>

	        <script>
	        (function(){
	            var master = document.getElementById('aegislink-bulk-checkall');
	            if (!master) { return; }
	            function boxes(){
	                return document.querySelectorAll('input.aegislink-bulk-id[type="checkbox"]');
	            }
	            function setAll(on){
	                var b = boxes();
	                for (var i=0;i<b.length;i++){ b[i].checked = !!on; }
	                master.indeterminate = false;
	            }
	            function sync(){
	                var b = boxes();
	                var total = b.length;
	                var checked = 0;
	                for (var i=0;i<total;i++){ if (b[i].checked) checked++; }
	                if (checked === 0) { master.checked = false; master.indeterminate = false; }
	                else if (checked === total) { master.checked = true; master.indeterminate = false; }
	                else { master.checked = false; master.indeterminate = true; }
	            }
	            master.addEventListener('click', function(){ setAll(master.checked); });
	            document.addEventListener('change', function(e){
	                var t = e.target;
	                if (t && t.classList && t.classList.contains('aegislink-bulk-id')) { sync(); }
	            });
	        })();
	        </script>

	    <?php elseif ($view === 'analytics') : ?>

        <h2><?php echo esc_html__('Advanced Analytics', 'aegislink');
                    echo wp_kses_post($aegislink_paginate);
                    ?></h2>
        <?php
        
        global $wpdb;
        $aegislink_table_raw = $wpdb->prefix . 'aegislink_clicks';

        $aegislink_total_clicks = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                'SELECT COUNT(*) FROM %i WHERE 1=%d',
                $aegislink_table_raw,
                1
            )
        ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        $aegislink_since = gmdate('Y-m-d H:i:s', time() - 30 * DAY_IN_SECONDS);

        $aegislink_daily = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT DATE(created_at) AS d, COUNT(*) AS c
                    FROM %i
                    WHERE created_at >= %s
                    GROUP BY DATE(created_at)
                    ORDER BY d DESC
                    LIMIT 30",
                $aegislink_table_raw,
                $aegislink_since
            )
        ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        $aegislink_top_ref = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT referer, COUNT(*) AS c
                    FROM %i
                    WHERE created_at >= %s AND referer <> '' AND referer IS NOT NULL
                    GROUP BY referer
                    ORDER BY c DESC
                    LIMIT 20",
                $aegislink_table_raw,
                $aegislink_since
            )
        ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        $aegislink_top_ua = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT ua, COUNT(*) AS c
                    FROM %i
                    WHERE created_at >= %s AND ua <> '' AND ua IS NOT NULL
                    GROUP BY ua
                    ORDER BY c DESC
                    LIMIT 10",
                $aegislink_table_raw,
                $aegislink_since
            )
        ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        ?>

        <p><strong><?php echo esc_html__('Total clicks recorded:', 'aegislink'); ?></strong> <?php echo (int) $aegislink_total_clicks; ?></p>

        <h3><?php echo esc_html__('Daily Clicks (Last 30 Days)', 'aegislink'); ?></h3>
        <table class="widefat striped" style="max-width:420px;">
            <thead><tr><th><?php echo esc_html__('Date', 'aegislink'); ?></th><th><?php echo esc_html__('Clicks', 'aegislink'); ?></th></tr></thead>
            <tbody>
            <?php if (empty($aegislink_daily)) : ?>
                <tr><td colspan="2"><?php echo esc_html__('No clicks yet.', 'aegislink'); ?></td></tr>
            <?php else : foreach ($aegislink_daily as $r) : ?>
                <tr><td><?php echo esc_html($r->d); ?></td><td><?php echo (int) $r->c; ?></td></tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <h3 style="margin-top:16px;"><?php echo esc_html__('Top Referrers (Last 30 Days)', 'aegislink'); ?></h3>
        <table class="widefat striped">
            <thead><tr><th><?php echo esc_html__('Referrer', 'aegislink'); ?></th><th><?php echo esc_html__('Clicks', 'aegislink'); ?></th></tr></thead>
            <tbody>
            <?php if (empty($aegislink_top_ref)) : ?>
                <tr><td colspan="2"><?php echo esc_html__('No referrers recorded.', 'aegislink'); ?></td></tr>
            <?php else : foreach ($aegislink_top_ref as $r) : ?>
                <tr><td style="max-width:720px; overflow:hidden; text-overflow:ellipsis;"><?php echo esc_html((string) $r->referer); ?></td><td><?php echo (int) $r->c; ?></td></tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <h3 style="margin-top:16px;"><?php echo esc_html__('Top User Agents (Last 30 Days)', 'aegislink'); ?></h3>
        <table class="widefat striped">
            <thead><tr><th><?php echo esc_html__('User Agent', 'aegislink'); ?></th><th><?php echo esc_html__('Clicks', 'aegislink'); ?></th></tr></thead>
            <tbody>
            <?php if (empty($aegislink_top_ua)) : ?>
                <tr><td colspan="2"><?php echo esc_html__('No user agents recorded.', 'aegislink'); ?></td></tr>
            <?php else : foreach ($aegislink_top_ua as $r) : ?>
                <tr><td style="max-width:720px; overflow:hidden; text-overflow:ellipsis;"><?php echo esc_html((string) $r->ua); ?></td><td><?php echo (int) $r->c; ?></td></tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <p class="description" style="margin-top:12px;">
            <?php echo esc_html__('Privacy note: clicks store referrer + user-agent + an IP hash (not the raw IP).', 'aegislink'); ?>
        </p>

    <?php else : // importexport ?>

        <h2><?php echo esc_html__('Import / Export', 'aegislink'); ?></h2>

        <h3><?php echo esc_html__('Export', 'aegislink'); ?></h3>
        <p>
            <a class="button button-secondary" href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=aegislink&tab=shorturl&view=importexport&export=1'), 'aegislink_shortlinks_export')); ?>">
                <?php echo esc_html__('Download CSV Export', 'aegislink'); ?>
            </a>
        </p>

        <hr />

        <h3><?php echo esc_html__('Import', 'aegislink'); ?></h3>
        <form method="post" enctype="multipart/form-data">
            <?php wp_nonce_field('aegislink_shortlinks_import'); ?>
            <input type="file" name="csv_file" accept=".csv,text/csv" required />
            <p class="description"><?php echo esc_html__('Columns supported: slug,title,target_url,redirect,track_clicks,rel_nofollow,rel_sponsored,notes,groups,tags,utm_*', 'aegislink'); ?></p>
            <p><button type="submit" class="button button-primary" name="aegislink_import_csv" value="1"><?php echo esc_html__('Import CSV', 'aegislink'); ?></button></p>
        </form>

    <?php endif; ?>

    <?php if ($pro_locked) : ?>
        </div>
        <script>
        (function(){
            try {
                var wrap = document.querySelector('.aegislink-pro-locked');
                if(!wrap) return;
                // ensure all form controls are disabled (defense in depth)
                wrap.querySelectorAll('input,select,textarea,button').forEach(function(el){ el.setAttribute('disabled','disabled'); });
            } catch(e) {}
        })();
        </script>
    <?php endif; ?>

</div>

<?php
})();
