<?php
/**
 * Plugin Name: AegisShield Security
 * Plugin URI:  https://aegisify.com/wordpress-aegis-shield-security-product/
 * Description: AegisShield Security safeguards WordPress sites with enterprise-grade malware detection, intrusion prevention, system hardening, database security, and continuous monitoring.
 * Version:     7.1.6
 * Author:      Aegisify | AegisShield
 * Author URI:  https://aegisify.com/
 * Text Domain: aegisshield-security
 * Domain Path: /languages
 */

defined( 'ABSPATH' ) || exit;

define( 'AEGISSHIELD_VERSION', '7.1.6' );
define( 'AEGISSHIELD_PLUGIN_FILE', __FILE__ );
define( 'AEGISSHIELD_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'AEGISSHIELD_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

if ( ! defined( 'AEGISSHIELD_UPDATE_DEBUG' ) ) {
    define( 'AEGISSHIELD_UPDATE_DEBUG', true );
}

if ( ! defined( 'AEGISSHIELD_UPDATE_DEBUG_TRANSIENT_KEY' ) ) {
    define( 'AEGISSHIELD_UPDATE_DEBUG_TRANSIENT_KEY', 'aegisshield_update_check_last_error' );
}

if ( ! function_exists( 'aegisshield_update_log' ) ) {
    function aegisshield_update_log( $message, $context = array() ) {
        if ( ! defined( 'AEGISSHIELD_UPDATE_DEBUG' ) || ! AEGISSHIELD_UPDATE_DEBUG ) {
            return;
        }

        $prefix = '[AegisShield Update] ';
        if ( ! empty( $context ) && is_array( $context ) ) {
            $message .= ' | ' . wp_json_encode( $context );
        }

        error_log( $prefix . $message );
    }
}

if ( ! function_exists( 'aegisshield_update_set_error' ) ) {
    function aegisshield_update_set_error( $reason, $context = array() ) {
        set_transient(
            AEGISSHIELD_UPDATE_DEBUG_TRANSIENT_KEY,
            array(
                'time'    => gmdate( 'c' ),
                'reason'  => (string) $reason,
                'context' => is_array( $context ) ? $context : array(),
            ),
            15 * MINUTE_IN_SECONDS
        );

        aegisshield_update_log( 'ERROR: ' . $reason, $context );
    }
}

function aegisshield_security_load_textdomain() {
    load_plugin_textdomain(
        'aegisshield-security',
        false,
        dirname( plugin_basename( __FILE__ ) ) . '/languages'
    );
}
add_action( 'init', 'aegisshield_security_load_textdomain' );
add_action( 'plugins_loaded', 'aegisshield_clm_register_free_install', 20 );

require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-autoloader.php';
AegisShield\AS_Autoloader::init();

function aegisshield_security_bootstrap() {
    return AegisShield\AS_Plugin::instance();
}
add_action( 'plugins_loaded', 'aegisshield_security_bootstrap' );
function aegisshield_security_activate() {
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-plugin.php';
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-cron.php';

    // Force FREE register ping after activation (next request)
    delete_transient( 'aegisshield_clm_free_reg_sent' );

    AegisShield\AS_Plugin::activate_static();
}
register_activation_hook( __FILE__, 'aegisshield_security_activate' );

function aegisshield_security_deactivate() {
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-cron.php';
    $cron = new AegisShield\AS_Cron();
    $cron->clear_events();
}
register_deactivation_hook( __FILE__, 'aegisshield_security_deactivate' );

if ( ! function_exists( 'aegisshield_is_pro_active' ) ) {
    function aegisshield_is_pro_active() {
        if ( ! class_exists( '\\AegisShield\\AS_Plugin' ) ) {
            return false;
        }

        $plugin = \AegisShield\AS_Plugin::instance();

        if ( method_exists( $plugin, 'get_license_manager' ) && $plugin->get_license_manager() ) {
            return (bool) apply_filters( 'aegisshield_is_pro_active', $plugin->get_license_manager()->is_pro_active() );
        }

        return false;
    }
}

if ( ! function_exists( 'aegisshield_show_pro_upgrade_notice' ) ) {
    add_action(
        'after_plugin_row_aegisshield-security/aegisshield-security.php',
        'aegisshield_show_pro_upgrade_notice',
        10,
        2
    );

    function aegisshield_show_pro_upgrade_notice( $plugin_file, $plugin_data ) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if ( function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active() ) {
            return;
        }

        echo '<tr class="plugin-update-tr"><td colspan="3" class="plugin-update colspanchange"><div class="update-message notice inline notice-warning notice-alt"><p>';

        $message = sprintf(
            __( '%1$sAegisShield Pro gives you our strongest security shield.%2$s Unlock advanced malware detection, automated deep scanning, database hardening automation, and extended activity logs to keep your site locked down on shared hosting. <a href="%3$s">Upgrade now to enable full protection.</a>', 'aegisshield', 'aegisshield-security' ),
            '<strong>',
            '</strong>',
            esc_url( admin_url( 'admin.php?page=aegisshield-license' ) )
        );

        echo wp_kses_post( $message );
        echo '</p></div></td></tr>';
    }
}

add_action( 'admin_notices', function () {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $last = get_transient( AEGISSHIELD_UPDATE_DEBUG_TRANSIENT_KEY );
    if ( empty( $last ) || ! is_array( $last ) ) {
        return;
    }

    $time   = ! empty( $last['time'] ) ? $last['time'] : '';
    $reason = ! empty( $last['reason'] ) ? $last['reason'] : 'Unknown';

    echo '<div class="notice notice-error"><p>';
    echo '<strong>AegisShield update check failed.</strong> ';
    echo esc_html( $reason );
    if ( $time ) {
        echo ' (UTC: ' . esc_html( $time ) . ')';
    }
    echo ' — check <code>wp-content/debug.log</code> for details.';
    echo '</p></div>';
} );

function aegisshield_clm_direct_update_check( $current_version ) {
    $endpoint = 'https://aegisify.com/wp-json/chue-license/v1/aegisshield/update-check';

    $payload = array(
        'product'  => 'aegisshield',
        'version'  => (string) $current_version,
        'site_url' => home_url(),
    );

    $args = array(
        'timeout' => 20,
        'headers' => array(
            'Content-Type' => 'application/json',
            'Accept'       => 'application/json',
            'User-Agent'   => 'AegisShield/' . ( defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : 'unknown' ),
        ),
        'body' => wp_json_encode( $payload ),
    );

    $resp = wp_remote_post( $endpoint, $args );

    if ( is_wp_error( $resp ) ) {
        return array(
            'success' => false,
            'message' => 'CLM request failed: ' . $resp->get_error_message(),
            'status'  => 'http_error',
        );
    }

    $code = (int) wp_remote_retrieve_response_code( $resp );
    $body = (string) wp_remote_retrieve_body( $resp );

    if ( $code < 200 || $code >= 300 ) {
        return array(
            'success' => false,
            'message' => 'CLM HTTP ' . $code,
            'status'  => 'http_' . $code,
            'raw'     => $body,
        );
    }

	$data = json_decode( $body, true );
	if ( ! is_array( $data ) ) {
		return array(
			'success' => false,
			'message' => 'CLM returned non-JSON or invalid JSON.',
			'status'  => 'bad_json',
			'raw'     => $body,
		);
	}

	/**
	 * Normalize CLM response keys:
	 * Some CLM builds return latest_version instead of version.
	 */
	if ( empty( $data['version'] ) && ! empty( $data['latest_version'] ) ) {
		$data['version'] = (string) $data['latest_version'];
	}

	return $data;
}

	function aegisshield_clm_register_free_install() {

		// Hard stop: if we've ever successfully registered, do not re-register automatically.
		if ( get_option( 'aegisshield_clm_free_reg_done' ) === '1' ) {
			return;
		}

		// Backoff (durable): if we recently got a 429 or failure, don't hammer CLM.
		$next_allowed = (int) get_option( 'aegisshield_clm_free_reg_next_allowed', 0 );
		if ( $next_allowed > 0 && time() < $next_allowed ) {
			return;
		}

		// Short lock (durable): prevents repeated calls during storms / concurrent hits.
		$lock_ts = (int) get_option( 'aegisshield_clm_free_reg_lock_ts', 0 );
		if ( $lock_ts > 0 && ( time() - $lock_ts ) < 60 ) {
			return;
		}
		update_option( 'aegisshield_clm_free_reg_lock_ts', time(), false );

		// Your existing transient throttle (keep it as an extra layer)
		if ( get_transient( 'aegisshield_clm_free_reg_sent' ) ) {
			return;
		}

		$endpoint = 'https://aegisify.com/wp-json/chue-license/v1/aegisshield/free-register';

    $site_url = home_url();
    $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

    $payload = array(
        'product'        => 'aegisshield',
        'email'          => get_option( 'admin_email' ),
        'site_url'       => $site_url,
        'domain'         => $domain ? $domain : '',
        'plugin_version' => defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : '',
        'wp_version'     => get_bloginfo( 'version' ),
        'php_version'    => PHP_VERSION,
    );

    $args = array(
        'timeout' => 20,
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body'    => wp_json_encode( $payload ),
    );

	$resp = wp_remote_post( $endpoint, $args );

	if ( is_wp_error( $resp ) ) {
		// Retry soon on failure (don’t lock out for 24h)
		set_transient( 'aegisshield_clm_free_reg_sent', 1, 15 * MINUTE_IN_SECONDS );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( '[AegisShield] CLM free-register failed: ' . $resp->get_error_message() );
		}
		return;
	}

	$code = (int) wp_remote_retrieve_response_code( $resp );
	$body = (string) wp_remote_retrieve_body( $resp );

	// Non-2xx should retry soon
	if ( $code < 200 || $code >= 300 ) {
		set_transient( 'aegisshield_clm_free_reg_sent', 1, 15 * MINUTE_IN_SECONDS );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( '[AegisShield] CLM free-register HTTP ' . $code . ' body=' . $body );
		}
		return;
	}

	// Parse JSON; if invalid, retry soon
	$parsed = json_decode( $body, true );
	if ( ! is_array( $parsed ) ) {
		set_transient( 'aegisshield_clm_free_reg_sent', 1, 15 * MINUTE_IN_SECONDS );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( '[AegisShield] CLM free-register invalid JSON body=' . $body );
		}
		return;
	}

	// Success? then lock out for a day. Otherwise retry soon.
	if ( ! empty( $parsed['success'] ) ) {

		// Durable "done" flag (survives caching issues)
		update_option( 'aegisshield_clm_free_reg_done', '1', false );

		// Next allowed check far in the future (optional; "done" already stops it)
		update_option( 'aegisshield_clm_free_reg_next_allowed', time() + DAY_IN_SECONDS, false );

		set_transient( 'aegisshield_clm_free_reg_sent', 1, DAY_IN_SECONDS );

	} else {

		// If rate limited, back off longer (durable)
		if ( $code === 429 ) {
			update_option( 'aegisshield_clm_free_reg_next_allowed', time() + ( 60 * 60 ), false ); // 1 hour
		} else {
			update_option( 'aegisshield_clm_free_reg_next_allowed', time() + ( 15 * 60 ), false ); // 15 minutes
		}

		set_transient( 'aegisshield_clm_free_reg_sent', 1, 15 * MINUTE_IN_SECONDS );
	}

	// Always release short lock
	update_option( 'aegisshield_clm_free_reg_lock_ts', 0, false );

}

function aegisshield_security_check_for_updates( $transient ) {
    aegisshield_update_log( 'Hook fired: pre_set_site_transient_update_plugins', array(
        'is_object'     => is_object( $transient ),
        'checked_count' => ( is_object( $transient ) && ! empty( $transient->checked ) && is_array( $transient->checked ) ) ? count( $transient->checked ) : 0,
    ) );

    if ( ! is_object( $transient ) || empty( $transient->checked ) ) {
        // This is normal sometimes (WP calls multiple times), but we still log it.
        aegisshield_update_log( 'Transient missing or empty checked list; skipping.' );
        return $transient;
    }

	if ( ! class_exists( '\\AegisShield\\AS_Plugin' ) ) {
		aegisshield_update_set_error( 'AS_Plugin class not loaded (plugin bootstrap may not have run yet).' );
		return $transient;
	}

	$plugin = \AegisShield\AS_Plugin::instance();
	$plugin_file     = 'aegisshield-security/aegisshield-security.php';
	$current_version = isset( $transient->checked[ $plugin_file ] )
		? $transient->checked[ $plugin_file ]
		: ( defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : '' );

	$last = (int) get_site_transient( $last_check_key );
	if ( $last > 0 && ( time() - $last ) < DAY_IN_SECONDS ) {
		// Re-inject cached update payload (if any) so WP UI stays consistent
		$cached = get_site_transient( $cache_key );
		if ( ! empty( $cached ) && is_array( $cached ) ) {
			$transient->response[ $plugin_file ] = $cached;
		}
		return $transient;
	}

	$data = aegisshield_clm_direct_update_check( $current_version );
	aegisshield_update_log( 'CLM response received', array(
		'type' => gettype( $data ),
		'data' => $data,
	) );
	if ( ! is_array( $data ) ) {
		aegisshield_update_set_error( 'CLM returned non-array response.', array( 'data_type' => gettype( $data ) ) );
		return $transient;
	}
	if ( empty( $data['version'] ) && ! empty( $data['latest_version'] ) ) {
		$data['version'] = (string) $data['latest_version'];
	}
	if ( empty( $data['success'] ) ) {
		aegisshield_update_set_error( 'CLM success=false.', array(
			'status'  => isset( $data['status'] ) ? $data['status'] : null,
			'message' => isset( $data['message'] ) ? $data['message'] : null,
			'http'    => isset( $data['status'] ) && strpos( (string) $data['status'], 'http_' ) === 0 ? $data['status'] : null,
			'raw'     => isset( $data['raw'] ) ? $data['raw'] : null,
			'data'    => $data,
		) );
		return $transient;
	}

    $cache_key      = 'aegisshield_update_cached_response_v1';

    $last = (int) get_site_transient( $last_check_key );
    if ( $last > 0 && ( time() - $last ) < DAY_IN_SECONDS ) {
        // If we have a cached update response, re-inject it so WP UI stays consistent.
        $cached = get_site_transient( $cache_key );
        if ( ! empty( $cached ) && is_array( $cached ) ) {
            $plugin_file = defined('AEGISSHIELD_PLUGIN_BASENAME') ? AEGISSHIELD_PLUGIN_BASENAME : plugin_basename( __FILE__ );
            $transient->response[ $plugin_file ] = $cached;
        }
        return $transient;
    }

	if ( empty( $data['update_available'] ) ) {
		// Not an error; still helpful to know it ran.
		aegisshield_update_log( 'CLM reports no update available.', array(
			'remote_version' => ! empty( $data['version'] ) ? (string) $data['version'] : '',
		) );

		return $transient;
	}

	if ( empty( $data['version'] ) ) {
		aegisshield_update_set_error( 'CLM update_available=true but version is missing.', array( 'data' => $data ) );
		return $transient;
	}
	if ( empty( $data['download_url'] ) ) {
		aegisshield_update_set_error( 'CLM update_available=true but download_url is missing.', array( 'data' => $data ) );
		return $transient;
	}
		
	aegisshield_update_log( 'Version compare', array(
		'current_version' => $current_version,
		'remote_version'  => isset( $data['version'] ) ? $data['version'] : null,
		'plugin_file'     => $plugin_file,
	) );

    if ( $current_version && version_compare( $data['version'], $current_version, '<=' ) ) {
        return $transient;
    }

    $update = (object) array(
        'slug'        => 'aegisshield-security',
        'plugin'      => $plugin_file,
        'new_version' => $data['version'],
        'requires'    => isset( $data['requires'] ) ? $data['requires'] : '',
        'package'     => $data['download_url'],
        'url'         => isset( $data['homepage'] ) ? $data['homepage'] : 'https://aegisify.com',
    );
	
	aegisshield_update_log( 'Injecting update object into transient', array(
		'update' => $update,
	) );

    $transient->response[ $plugin_file ] = $update;

    return $transient;
}
add_filter( 'pre_set_site_transient_update_plugins', 'aegisshield_security_check_for_updates' );