<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\Modules\AS_Module_Activity_Log;

defined( 'ABSPATH' ) || exit;

class AS_Page_Activity_Log {

    protected $plugin;

    protected $views_option_key = 'aegisshield_activity_views';

	function __construct( AS_Plugin $plugin ) {
		$this->plugin = $plugin;

		add_action( 'admin_post_aegisshield_al_download_archive', array( $this, 'handle_download_archive' ) );
	}


	protected function get_distinct_event_types() {
		global $wpdb;

		$logger = $this->plugin->get_logger();
		if ( ! $logger ) {
			return array();
		}

		$table = $logger->get_table_name();
		if ( empty( $table ) ) {
			return array();
		}

		// Fetch distinct event types
		$results = $wpdb->get_col(
			"SELECT DISTINCT event_type FROM {$table} ORDER BY event_type ASC"
		);

		if ( ! is_array( $results ) ) {
			return array();
		}

		// Sanitize + remove empties
		$results = array_map( 'sanitize_key', $results );
		$results = array_filter( $results );

		return array_values( $results );
	}

	protected function render_pro_locked_notice() {
		?>
		<div class="notice notice-warning inline">
			<p><strong><?php esc_html_e( 'This is a Pro feature.', 'aegisshield-security' ); ?></strong></p>
			<p><?php esc_html_e( 'Upgrade to unlock Alerts, Saved Views, Session Management and advanced settings.', 'aegisshield-security' ); ?></p>
			<p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>" class="button button-primary">
					<?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
				</a>
			</p>
		</div>
		<?php
	}

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $settings          = $this->plugin->get_settings();
        $section           = 'activity_log';
        $retention_updated = false;
        $views_updated     = false;
        $alerts_updated    = false;
		$manual_purge_ran  = false;

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        if ( $is_pro ) {
            $this->handle_alerts_post( $alerts_updated );
        }

		if ( ! $is_pro && isset( $_POST['as_alert_action'] ) ) {
			AS_Module_Activity_Log::log_violation(
				$this->plugin,
				'alerts',
				'pro_locked_alerts_action',
				'Alerts action attempted while Pro is not active.',
				array(
					'post_action' => sanitize_text_field( wp_unslash( $_POST['as_alert_action'] ) ),
				),
				true,
				'enforced'
			);
		}

		$this->handle_post_actions( $settings, $section, $is_pro, $retention_updated, $views_updated, $manual_purge_ran );

		$this->handle_sessions_post_actions();

        $retention_days = (int) $settings->get( $section, 'retention_days', 7 );
        if ( $retention_days <= 0 ) {
            $retention_days = 7;
        }

		if ( ! $is_pro && $retention_days > 7 ) {
			$retention_days = 7;
		}

        $current_tab  = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'log'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $allowed_tabs = array( 'log', 'alerts', 'views', 'sessions', 'settings' );
        if ( ! in_array( $current_tab, $allowed_tabs, true ) ) {
            $current_tab = 'log';
        }

		if ( 'log' === $current_tab && ! $is_pro && isset( $_GET['as_al_export'] ) ) {
			AS_Module_Activity_Log::log_violation(
				$this->plugin,
				'activity_log',
				'pro_locked_export_csv',
				'CSV export attempted while Pro is not active.',
				array(
					'tab' => 'log',
				),
				true,
				'enforced'
			);
		}

        if ( 'log' === $current_tab && $is_pro && isset( $_GET['as_al_export'] ) && 'csv' === $_GET['as_al_export'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            $this->handle_export_csv();
            return;
        }

        $logs        = array();
        $paged       = 1;
        $total_pages = 1;

        if ( 'log' === $current_tab ) {
            list( $logs, $paged, $total_pages ) = $this->query_logs_with_filters();
        }

		$saved_views = $this->get_saved_views();

		// Local-first + CDN fallback (best of both worlds)
		$plugin_root = dirname( __FILE__, 3 ); // plugin root folder
		$chartjs_rel = 'assets/js/chart.umd.min.js';
		$chartjs_abs = $plugin_root . '/' . $chartjs_rel;

		if ( ! wp_script_is( 'aegisshield-chartjs', 'enqueued' ) ) {
			if ( file_exists( $chartjs_abs ) ) {
				$chartjs_url = plugins_url( $chartjs_rel, $plugin_root . '/aegisshield-security.php' );
				wp_enqueue_script( 'aegisshield-chartjs', $chartjs_url, array(), '4.4.1', true );
			} else {
				wp_enqueue_script(
					'aegisshield-chartjs',
					'https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js',
					array(),
					'4.4.1',
					true
				);
			}
		}

		if ( ! wp_script_is( 'aegisshield-activity-log-charts', 'enqueued' ) ) {
			wp_register_script( 'aegisshield-activity-log-charts', '', array( 'aegisshield-chartjs' ), '1.0.0', true );
			wp_enqueue_script( 'aegisshield-activity-log-charts' );
		}
        ?>
        <div class="wrap aegisshield-activity-log">
            <h1><?php esc_html_e( 'Activity Log', 'aegisshield-security' ); ?></h1>
            <p class="description">
                <?php esc_html_e( 'The Activity Log module provides a centralized, tamper-resistant audit trail of security-relevant activity across a WordPress site, including authentication events, user actions, and system changes. It is designed to log and enforce behavior consistently recording both successful actions and violations, including blocked or Pro-restricted operations so administrators always have clear visibility into what happened, why it happened, and how the system responded.', 'aegisshield-security' ); ?>
            </p>

			<?php if ( $retention_updated ) : ?>
				<div class="notice notice-success is-dismissible">
					<p><?php esc_html_e( 'Activity Log settings saved.', 'aegisshield-security' ); ?></p>
				</div>
			<?php endif; ?>

			<?php if ( $manual_purge_ran ) : ?>
				<div class="notice notice-success is-dismissible">
					<p><?php esc_html_e( 'Manual purge executed. Older logs were removed based on your retention setting. (Details recorded in Activity Log.)', 'aegisshield-security' ); ?></p>
				</div>
			<?php endif; ?>

            <?php if ( $views_updated ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Saved views updated.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( $alerts_updated ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Alert rules updated.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php $this->render_tabs_nav( $current_tab, $is_pro ); ?>

            <?php
            switch ( $current_tab ) {
                case 'alerts':
                    $this->render_alerts_tab( $is_pro );
                    break;
                case 'views':
                    $this->render_views_tab( $is_pro, $saved_views );
                    break;
                case 'sessions':
                    $this->render_sessions_tab( $is_pro );
                    break;
                case 'settings':
                    $this->render_settings_tab( $is_pro, $retention_days );
                    break;
                case 'log':
                default:
                    $this->render_log_tab( $logs, $paged, $total_pages, $is_pro, $saved_views );
                    break;
            }
            ?>
        </div>
        <?php
    }

    protected function handle_post_actions( $settings, $section, $is_pro, &$retention_updated, &$views_updated ) {

        if (
            isset( $_POST['aegisshield_activity_log_nonce'], $_POST['retention_days'] )
            && check_admin_referer( 'aegisshield_activity_log_save', 'aegisshield_activity_log_nonce' )
        ) {
            $days = absint( $_POST['retention_days'] );

            if ( $days <= 0 ) {
                $days = 1;
            }

		if ( ! $is_pro && $days > 7 ) {
			AS_Module_Activity_Log::log_violation(
				$this->plugin,
				'activity_log',
				'retention_clamped_free_limit',
				'Retention days exceeded Free limit; value clamped to 7.',
				array(
					'requested_days' => (int) $days,
					'enforced_days'  => 7,
				),
				true,
				'enforced'
			);
			$days = 7;
		}

			$settings->set( $section, 'retention_days', $days );
			$settings->save();

			$retention_updated = true;

			if ( isset( $_POST['as_manual_purge'] ) ) {
				AS_Module_Activity_Log::purge_logs_by_retention( $this->plugin, 'manual_button' );
				$manual_purge_ran = true;
			}
        }

		if (
			isset( $_POST['aegisshield_activity_log_nonce'] )
			&& check_admin_referer( 'aegisshield_activity_log_save', 'aegisshield_activity_log_nonce' )
		) {
			if ( $is_pro ) {
				$archive_enabled   = isset( $_POST['archive_enabled'] ) ? 1 : 0;
				$archive_frequency = isset( $_POST['archive_frequency'] ) ? sanitize_key( wp_unslash( $_POST['archive_frequency'] ) ) : 'monthly';
				$archive_format    = isset( $_POST['archive_format'] ) ? sanitize_key( wp_unslash( $_POST['archive_format'] ) ) : 'csv';

				if ( ! in_array( $archive_frequency, array( 'monthly', 'quarterly', '6months' ), true ) ) {
					$archive_frequency = 'monthly';
				}
				if ( ! in_array( $archive_format, array( 'csv' ), true ) ) {
					$archive_format = 'csv';
				}

				$settings->set( $section, 'archive_enabled', (int) $archive_enabled );
				$settings->set( $section, 'archive_frequency', $archive_frequency );
				$settings->set( $section, 'archive_format', $archive_format );

				$save_ok = $settings->save();

				if ( false === $save_ok ) {
					\AegisShield\Modules\AS_Module_Activity_Log::log_violation(
						$this->plugin,
						'activity_log',
						'archive_settings_save_failed',
						'Archiving settings save returned false.',
						array(
							'enabled'   => (int) $archive_enabled,
							'frequency' => (string) $archive_frequency,
							'format'    => (string) $archive_format,
						),
						false,
						'failed'
					);
				} else {
					// Reschedule archive job to apply changes immediately.
					$module = new \AegisShield\Modules\AS_Module_Activity_Log( $this->plugin );
					$module->maybe_schedule_activity_log_archive();
				}
			} else if ( isset( $_POST['archive_enabled'] ) || isset( $_POST['archive_frequency'] ) || isset( $_POST['archive_format'] ) ) {
				\AegisShield\Modules\AS_Module_Activity_Log::log_violation(
					$this->plugin,
					'activity_log',
					'archive_settings_blocked_free',
					'Archiving settings attempted while Pro is not active.',
					array(),
					true,
					'enforced'
				);
			}
		}

        if ( ! $is_pro ) {
            return;
        }

		if ( isset( $_POST['aegisshield_activity_views_nonce'] ) && ! $is_pro ) {
			AS_Module_Activity_Log::log_violation(
				$this->plugin,
				'saved_views',
				'pro_locked_saved_views',
				'Saved Views action attempted while Pro is not active.',
				array(
					'post' => array(
						'has_delete_id' => isset( $_POST['as_view_delete_id'] ) && '' !== $_POST['as_view_delete_id'],
						'has_view_name' => isset( $_POST['as_view_name'] ) && '' !== $_POST['as_view_name'],
					),
				),
				true,
				'enforced'
			);
			return;
		}

        if (
            isset( $_POST['aegisshield_activity_views_nonce'] )
            && check_admin_referer( 'aegisshield_activity_views_save', 'aegisshield_activity_views_nonce' )
        ) {
            $views = $this->get_saved_views();

            if ( isset( $_POST['as_view_delete_id'] ) && '' !== $_POST['as_view_delete_id'] ) {
                $delete_id = sanitize_text_field( wp_unslash( $_POST['as_view_delete_id'] ) );
                $views     = array_values(
                    array_filter(
                        $views,
                        static function( $view ) use ( $delete_id ) {
                            return ! isset( $view['id'] ) || $view['id'] !== $delete_id;
                        }
                    )
                );
                $this->save_saved_views( $views );
                $views_updated = true;
                return;
            }

            if ( isset( $_POST['as_view_name'] ) && '' !== $_POST['as_view_name'] ) {
                $view_name = sanitize_text_field( wp_unslash( $_POST['as_view_name'] ) );

                $filter = array(
                    'search' => isset( $_GET['as_al_search'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_search'] ) ) : '', // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    'user'   => isset( $_GET['as_al_user'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_user'] ) ) : '',     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    'type'   => isset( $_GET['as_al_type'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_type'] ) ) : '',     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    'from'   => isset( $_GET['as_al_from'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_from'] ) ) : '',     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    'to'     => isset( $_GET['as_al_to'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_to'] ) ) : '',         // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                );

                $views[] = array(
                    'id'      => uniqid( 'al_view_', true ),
                    'name'    => $view_name,
                    'filters' => $filter,
                );

                $this->save_saved_views( $views );
                $views_updated = true;
            }
        }
    }

    protected function handle_alerts_post( &$alerts_updated ) {

        if (
            ! isset( $_POST['aegisshield_alert_rules_nonce'], $_POST['as_alert_action'] )
            || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['aegisshield_alert_rules_nonce'] ) ), 'aegisshield_alert_rules_save' )
        ) {
            return;
        }

        $action      = sanitize_text_field( wp_unslash( $_POST['as_alert_action'] ) );
        $alert_rules = $this->get_alert_rules();

        if ( 'add' === $action ) {
            $name       = isset( $_POST['as_alert_name'] ) ? sanitize_text_field( wp_unslash( $_POST['as_alert_name'] ) ) : '';
            $event_type = isset( $_POST['as_alert_event_type'] ) ? sanitize_key( wp_unslash( $_POST['as_alert_event_type'] ) ) : '';
            $emails_raw = isset( $_POST['as_alert_emails'] ) ? sanitize_textarea_field( wp_unslash( $_POST['as_alert_emails'] ) ) : '';
            $enabled    = isset( $_POST['as_alert_enabled'] ) ? (bool) $_POST['as_alert_enabled'] : true;

            if ( ! $event_type ) {
                return;
            }

            $rule = array(
                'id'         => uniqid( 'al_rule_', true ),
                'name'       => $name ? $name : $event_type,
                'event_type' => $event_type,
                'emails'     => $emails_raw,
                'enabled'    => $enabled,
            );

            $alert_rules[] = $rule;
            $this->save_alert_rules( $alert_rules );
            $alerts_updated = true;
        } elseif ( 'delete' === $action && isset( $_POST['as_alert_rule_id'] ) ) {
            $rule_id = sanitize_text_field( wp_unslash( $_POST['as_alert_rule_id'] ) );

            $changed = false;
            foreach ( $alert_rules as $index => $rule ) {
                if ( isset( $rule['id'] ) && $rule['id'] === $rule_id ) {
                    unset( $alert_rules[ $index ] );
                    $changed = true;
                }
            }

            if ( $changed ) {
                $this->save_alert_rules( $alert_rules );
                $alerts_updated = true;
            }
        }
    }

    protected function get_alert_rules() {
        $rules = get_option( 'aegisshield_alert_rules', array() );

        if ( ! is_array( $rules ) ) {
            $rules = array();
        }

        return $rules;
    }

    protected function save_alert_rules( $rules ) {
        if ( ! is_array( $rules ) ) {
            $rules = array();
        }

        update_option( 'aegisshield_alert_rules', array_values( $rules ) );
    }

    protected function render_tabs_nav( $current_tab, $is_pro ) {
        $base_url = admin_url( 'admin.php?page=aegisshield-activity-log' );
        ?>
		<h2 class="nav-tab-wrapper">
			<a href="<?php echo esc_url( $base_url ); ?>" class="nav-tab <?php echo ( 'log' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
				<?php esc_html_e( 'Activity Log', 'aegisshield-security' ); ?>
			</a>
			<a href="<?php echo esc_url( add_query_arg( 'tab', 'alerts', $base_url ) ); ?>" class="nav-tab <?php echo ( 'alerts' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
				<?php esc_html_e( 'Alerts', 'aegisshield-security' ); ?>
				<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
			</a>
			<a href="<?php echo esc_url( add_query_arg( 'tab', 'views', $base_url ) ); ?>" class="nav-tab <?php echo ( 'views' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
				<?php esc_html_e( 'Saved Views', 'aegisshield-security' ); ?>
				<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
			</a>
			<a href="<?php echo esc_url( add_query_arg( 'tab', 'sessions', $base_url ) ); ?>" class="nav-tab <?php echo ( 'sessions' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
				<?php esc_html_e( 'Sessions', 'aegisshield-security' ); ?>
				<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
			</a>
			<a href="<?php echo esc_url( add_query_arg( 'tab', 'settings', $base_url ) ); ?>" class="nav-tab <?php echo ( 'settings' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
				<?php esc_html_e( 'Settings', 'aegisshield-security' ); ?>
				<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
			</a>
		</h2>
        <?php
    }

    protected function query_logs_with_filters() {
        global $wpdb;

        $logger = $this->plugin->get_logger();
        $table  = $logger->get_table_name();

        $per_page = 25;
        $paged    = isset( $_GET['paged'] ) ? max( 1, absint( $_GET['paged'] ) ) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $offset   = ( $paged - 1 ) * $per_page;

        $search = isset( $_GET['as_al_search'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_search'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $user   = isset( $_GET['as_al_user'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_user'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $type   = isset( $_GET['as_al_type'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_type'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $from   = isset( $_GET['as_al_from'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_from'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $to     = isset( $_GET['as_al_to'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_to'] ) ) : '';         // phpcs:ignore WordPress.Security.NonceVerification.Recommended

        $where = ' WHERE 1=1 ';

        if ( $search ) {
            $like   = '%' . $wpdb->esc_like( $search ) . '%';
            $where .= $wpdb->prepare(
                ' AND (event_type LIKE %s OR message LIKE %s OR ip_address LIKE %s)',
                $like,
                $like,
                $like
            );
        }

        if ( '' !== $user ) {
            $where .= $wpdb->prepare( ' AND user_id = %d', absint( $user ) );
        }

        if ( '' !== $type ) {
            $where .= $wpdb->prepare( ' AND event_type = %s', $type );
        }

        if ( $from ) {
            $from_dt = $from . ' 00:00:00';
            $where  .= $wpdb->prepare( ' AND event_time >= %s', $from_dt );
        }

        if ( $to ) {
            $to_dt  = $to . ' 23:59:59';
            $where .= $wpdb->prepare( ' AND event_time <= %s', $to_dt );
        }

		$where = trim( (string) $where );

		// Remove any trailing "AND" (or "AND    ").
		$where = preg_replace( '/\s+AND\s*$/i', '', $where );

		if ( '' === $where ) {
			$where = ' WHERE 1=1 ';
		}
		
        $total_items = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table} {$where}" ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

        $logs = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table} {$where} ORDER BY event_time DESC LIMIT %d OFFSET %d", // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
                $per_page,
                $offset
            )
        );

        $total_pages = $total_items > 0 ? (int) ceil( $total_items / $per_page ) : 1;

        return array( $logs, $paged, $total_pages );
    }

    protected function render_log_tab( $logs, $paged, $total_pages, $is_pro, $saved_views ) {
        $base_url = admin_url( 'admin.php?page=aegisshield-activity-log' );
        ?>
        <div class="aegisshield-activity-log-layout">
            <div class="aegisshield-activity-log-main">
                <?php

                global $wpdb;

                $logger = $this->plugin->get_logger();
                $table  = $logger->get_table_name();

                $search = isset( $_GET['as_al_search'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_search'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $user   = isset( $_GET['as_al_user'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_user'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $type   = isset( $_GET['as_al_type'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_type'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $from   = isset( $_GET['as_al_from'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_from'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $to     = isset( $_GET['as_al_to'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_to'] ) ) : '';         // phpcs:ignore WordPress.Security.NonceVerification.Recommended

                $where = ' WHERE 1=1 ';

                if ( $search ) {
                    $like   = '%' . $wpdb->esc_like( $search ) . '%';
                    $where .= $wpdb->prepare(
                        ' AND (event_type LIKE %s OR message LIKE %s OR ip_address LIKE %s)',
                        $like,
                        $like,
                        $like
                    );
                }

                if ( '' !== $user ) {
                    $where .= $wpdb->prepare( ' AND user_id = %d', absint( $user ) );
                }

                if ( $type ) {
                    $like_type = '%' . $wpdb->esc_like( $type ) . '%';
                    $where    .= $wpdb->prepare( ' AND event_type LIKE %s', $like_type );
                }

                if ( $from ) {
                    $from_dt = $from . ' 00:00:00';
                    $where  .= $wpdb->prepare( ' AND event_time >= %s', $from_dt );
                }
                if ( $to ) {
                    $to_dt  = $to . ' 23:59:59';
                    $where .= $wpdb->prepare( ' AND event_time <= %s', $to_dt );
                }

                // (A) Line/Area: events over time (last 14 days by default if no date filter is provided)
                $time_window_sql = '';
                if ( ! $from && ! $to ) {
                    $time_window_sql = " AND event_time >= DATE_SUB(UTC_TIMESTAMP(), INTERVAL 14 DAY) ";
                }

                $rows_time = $wpdb->get_results(
                    "SELECT DATE(event_time) AS d, COUNT(*) AS c
                     FROM {$table}
                     {$where}
                     {$time_window_sql}
                     GROUP BY DATE(event_time)
                     ORDER BY d ASC",
                    ARRAY_A
                );

                $labels_time = array();
                $counts_time = array();
                foreach ( (array) $rows_time as $r ) {
                    $labels_time[] = $r['d'];
                    $counts_time[] = (int) $r['c'];
                }

                // (B) Bar: Top event types (Top 10)
                $rows_types = $wpdb->get_results(
                    "SELECT event_type AS t, COUNT(*) AS c
                     FROM {$table}
                     {$where}
                     GROUP BY event_type
                     ORDER BY c DESC
                     LIMIT 10",
                    ARRAY_A
                );

                $labels_types = array();
                $counts_types = array();
                foreach ( (array) $rows_types as $r ) {
                    $labels_types[] = (string) $r['t'];
                    $counts_types[] = (int) $r['c'];
                }

                // (C) Donut: Top IPs (Top 8 + “Other”)
                $rows_ips = $wpdb->get_results(
                    "SELECT ip_address AS ip, COUNT(*) AS c
                     FROM {$table}
                     {$where}
                     AND ip_address <> ''
                     GROUP BY ip_address
                     ORDER BY c DESC
                     LIMIT 8",
                    ARRAY_A
                );

                $labels_ips = array();
                $counts_ips = array();
                $sum_ips    = 0;
                foreach ( (array) $rows_ips as $r ) {
                    $labels_ips[] = (string) $r['ip'];
                    $counts_ips[] = (int) $r['c'];
                    $sum_ips     += (int) $r['c'];
                }

                // Compute “Other” for donut.
                $total_all = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table} {$where}" ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
                $other     = max( 0, $total_all - $sum_ips );
                if ( $other > 0 ) {
                    $labels_ips[] = 'Other';
                    $counts_ips[] = $other;
                }

                // (D) Radar: bucket event types into categories for instant “what’s happening”
                $buckets = array(
                    'Auth'     => 0,
                    'Security' => 0,
                    'System'   => 0,
                    'Content'  => 0,
                    'Other'    => 0,
                );

                // Use top 2000 recent events for bucketing (fast + relevant).
                $rows_recent = $wpdb->get_results(
                    "SELECT event_type
                     FROM {$table}
                     {$where}
                     ORDER BY event_time DESC
                     LIMIT 2000",
                    ARRAY_A
                );

                foreach ( (array) $rows_recent as $r ) {
                    $t = strtolower( (string) $r['event_type'] );

                    if ( false !== strpos( $t, 'login' ) || false !== strpos( $t, 'auth' ) || false !== strpos( $t, 'logout' ) ) {
                        $buckets['Auth']++;
                    } elseif ( false !== strpos( $t, 'blocked' ) || false !== strpos( $t, 'waf' ) || false !== strpos( $t, 'ddos' ) || false !== strpos( $t, 'bot' ) || false !== strpos( $t, 'malware' ) || false !== strpos( $t, 'integrity' ) ) {
                        $buckets['Security']++;
                    } elseif ( false !== strpos( $t, 'plugin' ) || false !== strpos( $t, 'theme' ) || false !== strpos( $t, 'update' ) || false !== strpos( $t, 'db_' ) || false !== strpos( $t, 'dbtools' ) ) {
                        $buckets['System']++;
                    } elseif ( false !== strpos( $t, 'post' ) || false !== strpos( $t, 'page' ) || false !== strpos( $t, 'media' ) || false !== strpos( $t, 'comment' ) ) {
                        $buckets['Content']++;
                    } else {
                        $buckets['Other']++;
                    }
                }

                // (E) Bubble: volume by hour (UTC) to spot spikes
                $rows_hour = $wpdb->get_results(
                    "SELECT HOUR(event_time) AS h, COUNT(*) AS c
                     FROM {$table}
                     {$where}
                     GROUP BY HOUR(event_time)
                     ORDER BY h ASC",
                    ARRAY_A
                );

                $bubble_points = array();
                foreach ( (array) $rows_hour as $r ) {
                    $h = (int) $r['h'];
                    $c = (int) $r['c'];
                    $bubble_points[] = array(
                        'x' => $h,
                        'y' => $c,
                        'r' => min( 18, max( 4, (int) round( sqrt( $c ) ) ) ),
                    );
                }

                $charts_payload = array(
                    'time'   => array( 'labels' => $labels_time,  'counts' => $counts_time ),
                    'types'  => array( 'labels' => $labels_types, 'counts' => $counts_types ),
                    'ips'    => array( 'labels' => $labels_ips,   'counts' => $counts_ips ),
                    'radar'  => $buckets,
                    'bubble' => $bubble_points,
                );

                $charts_json = wp_json_encode( $charts_payload );
                $init_js = "
                (function(){
                    function ready(fn){ if(document.readyState!=='loading'){fn();} else {document.addEventListener('DOMContentLoaded', fn);} }
                    ready(function(){
                        if(typeof Chart==='undefined'){ return; }

                        var data = {$charts_json};

                        // Shared defaults
                        Chart.defaults.responsive = true;
                        Chart.defaults.maintainAspectRatio = false;

                        // 1) Line (events over time)
                        var el1 = document.getElementById('as_al_chart_line');
                        if(el1){
                            new Chart(el1, {
                                type: 'line',
                                data: {
                                    labels: data.time.labels,
                                    datasets: [{ label: 'Events', data: data.time.counts, tension: 0.25, fill: false }]
                                },
                                options: {
                                    plugins: { legend: { display: false } },
                                    scales: { x: { display: true }, y: { beginAtZero: true } }
                                }
                            });
                        }

                        // 2) Bar (top event types)
                        var el2 = document.getElementById('as_al_chart_bar');
                        if(el2){
                            new Chart(el2, {
                                type: 'bar',
                                data: {
                                    labels: data.types.labels,
                                    datasets: [{ label: 'Count', data: data.types.counts }]
                                },
                                options: {
                                    plugins: { legend: { display: false } },
                                    scales: { x: { ticks: { autoSkip: false, maxRotation: 60, minRotation: 0 } }, y: { beginAtZero: true } }
                                }
                            });
                        }

                        // 3) Doughnut (top IPs)
                        var el3 = document.getElementById('as_al_chart_donut');
                        if(el3){
                            new Chart(el3, {
                                type: 'doughnut',
                                data: {
                                    labels: data.ips.labels,
                                    datasets: [{ data: data.ips.counts }]
                                },
                                options: { plugins: { legend: { position: 'bottom' } } }
                            });
                        }

                        // 4) Radar (behavior categories)
                        var el4 = document.getElementById('as_al_chart_radar');
                        if(el4){
                            var labels = Object.keys(data.radar);
                            var vals   = labels.map(function(k){ return data.radar[k]; });
                            new Chart(el4, {
                                type: 'radar',
                                data: {
                                    labels: labels,
                                    datasets: [{ label: 'Recent Mix', data: vals, fill: true }]
                                },
                                options: { plugins: { legend: { display: false } } }
                            });
                        }

                        // 5) Bubble (spikes by hour)
                        var el5 = document.getElementById('as_al_chart_bubble');
                        if(el5){
                            new Chart(el5, {
                                type: 'bubble',
                                data: {
                                    datasets: [{ label: 'Events by Hour (UTC)', data: data.bubble }]
                                },
                                options: {
                                    plugins: { legend: { display: false } },
                                    scales: { x: { title: { display: true, text: 'Hour (UTC)' }, beginAtZero: true, max: 23 }, y: { beginAtZero: true } }
                                }
                            });
                        }
                    });
                })();
                ";

                wp_add_inline_script( 'aegisshield-activity-log-charts', $init_js );
                ?>

                <style>
                    .as-al-intel-wrap{ margin: 10px 0 14px; }
                    .as-al-intel-grid{
                        display:flex;
                        gap:12px;
                        flex-wrap:wrap;
                        align-items:stretch;
                    }
                    .as-al-intel-card{
                        background:#fff;
                        border:1px solid #dcdcde;
                        border-radius:8px;
                        padding:10px 10px 8px;
                        flex: 1 1 260px;
                        min-width: 260px;
                    }
                    .as-al-intel-title{
                        font-weight:600;
                        margin:0 0 8px;
                        display:flex;
                        align-items:center;
                        justify-content:space-between;
                    }
                    .as-al-intel-canvas{
                        height:180px;
                        position:relative;
                    }
                    @media (min-width: 1400px){
                        .as-al-intel-card{ flex: 1 1 calc(20% - 12px); }
                    }
                </style>

                <div class="as-al-intel-wrap">
                    <div class="as-al-intel-grid">
                        <div class="as-al-intel-card">
                            <div class="as-al-intel-title">Events Over Time</div>
                            <div class="as-al-intel-canvas"><canvas id="as_al_chart_line"></canvas></div>
                        </div>

                        <div class="as-al-intel-card">
                            <div class="as-al-intel-title">Top Event Types</div>
                            <div class="as-al-intel-canvas"><canvas id="as_al_chart_bar"></canvas></div>
                        </div>

                        <div class="as-al-intel-card">
                            <div class="as-al-intel-title">Top IP Sources</div>
                            <div class="as-al-intel-canvas"><canvas id="as_al_chart_donut"></canvas></div>
                        </div>

                        <div class="as-al-intel-card">
                            <div class="as-al-intel-title">Behavior Mix</div>
                            <div class="as-al-intel-canvas"><canvas id="as_al_chart_radar"></canvas></div>
                        </div>

                        <div class="as-al-intel-card">
                            <div class="as-al-intel-title">Spikes By Hour</div>
                            <div class="as-al-intel-canvas"><canvas id="as_al_chart_bubble"></canvas></div>
                        </div>
                    </div>
                </div>

                <form method="get">
                    <input type="hidden" name="page" value="aegisshield-activity-log" />
                    <input type="hidden" name="tab" value="log" />

                    <div class="tablenav top">
                        <div class="alignleft actions">
                            <input
                                type="search"
                                name="as_al_search"
                                value="<?php echo isset( $_GET['as_al_search'] ) ? esc_attr( wp_unslash( $_GET['as_al_search'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>"
                                placeholder="<?php esc_attr_e( 'Search message, type, IP…', 'aegisshield-security' ); ?>"
                            />

                            <label for="as_al_user" style="margin-left:10px;">
                                <?php esc_html_e( 'User', 'aegisshield-security' ); ?>
                            </label>
                            <select name="as_al_user" id="as_al_user">
                                <option value=""><?php esc_html_e( 'Any', 'aegisshield-security' ); ?></option>
                                <?php
                                $users = get_users(
                                    array(
                                        'fields' => array( 'ID', 'user_login' ),
                                    )
                                );
                                $current_user_filter = isset( $_GET['as_al_user'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_user'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                                foreach ( $users as $user ) :
                                    ?>
                                    <option value="<?php echo esc_attr( $user->ID ); ?>" <?php selected( (string) $user->ID, $current_user_filter ); ?>>
                                        <?php echo esc_html( $user->user_login . ' (#' . $user->ID . ')' ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>

                            <label for="as_al_type" style="margin-left:10px;">
                                <?php esc_html_e( 'Event Type', 'aegisshield-security' ); ?>
                            </label>
                            <input
                                type="text"
                                name="as_al_type"
                                id="as_al_type"
                                value="<?php echo isset( $_GET['as_al_type'] ) ? esc_attr( wp_unslash( $_GET['as_al_type'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>"
                                placeholder="<?php esc_attr_e( 'e.g. login_success, plugin_activated', 'aegisshield-security' ); ?>"
                            />
                        </div>

                        <div class="alignright actions">
                            <label for="as_al_from">
                                <?php esc_html_e( 'From', 'aegisshield-security' ); ?>
                            </label>
                            <input
                                type="date"
                                name="as_al_from"
                                id="as_al_from"
                                value="<?php echo isset( $_GET['as_al_from'] ) ? esc_attr( wp_unslash( $_GET['as_al_from'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>"
                            />
                            <label for="as_al_to" style="margin-left:5px;">
                                <?php esc_html_e( 'To', 'aegisshield-security' ); ?>
                            </label>
                            <input
                                type="date"
                                name="as_al_to"
                                id="as_al_to"
                                value="<?php echo isset( $_GET['as_al_to'] ) ? esc_attr( wp_unslash( $_GET['as_al_to'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended ?>"
                            />

                            <input type="submit" class="button" value="<?php esc_attr_e( 'Filter', 'aegisshield-security' ); ?>" style="margin-left:5px;" />

                            <?php if ( $is_pro ) : ?>
                                <a href="<?php echo esc_url( add_query_arg( 'as_al_export', 'csv', $base_url ) ); ?>" class="button button-secondary" style="margin-left:5px;">
                                    <?php esc_html_e( 'Export CSV', 'aegisshield-security' ); ?>
                                </a>
                            <?php else : ?>
                                <span class="description" style="margin-left:8px;">
                                    <?php esc_html_e( 'CSV export is available in AegisShield Pro.', 'aegisshield-security' ); ?>
                                </span>
                            <?php endif; ?>
                        </div>

                        <br class="clear" />
                    </div>

                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <?php
                            if ( $total_pages > 1 ) {
                                echo wp_kses_post(
                                    paginate_links(
                                        array(
                                            'base'      => esc_url( add_query_arg( 'paged', '%#%' ) ),
                                            'format'    => '',
                                            'prev_text' => __( '&laquo;', 'aegisshield-security' ),
                                            'next_text' => __( '&raquo;', 'aegisshield-security' ),
                                            'total'     => $total_pages,
                                            'current'   => $paged,
                                        )
                                    )
                                );
                            }
                            ?>
                        </div>
                    </div>
                </form>

                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Time', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Event', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'User', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'IP Address', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Details', 'aegisshield-security' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ( empty( $logs ) ) : ?>
                            <tr>
                                <td colspan="5">
                                    <?php esc_html_e( 'No activity has been recorded with the current filters.', 'aegisshield-security' ); ?>
                                </td>
                            </tr>
                        <?php else : ?>
                            <?php
                            foreach ( $logs as $row ) :
                                $user_label = __( 'System', 'aegisshield-security' );
                                if ( ! empty( $row->user_id ) ) {
                                    $user = get_userdata( (int) $row->user_id );
                                    if ( $user ) {
                                        $user_label = $user->user_login . ' (#' . (int) $row->user_id . ')';
                                    } else {
                                        $user_label = '#' . (int) $row->user_id;
                                    }
                                }

                                $time = mysql2date(
                                    get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
                                    $row->event_time,
                                    true
                                );
                                ?>
                                <tr>
                                    <td><?php echo esc_html( $time ); ?></td>
                                    <td><strong><?php echo esc_html( $row->event_type ); ?></strong></td>
                                    <td><?php echo esc_html( $user_label ); ?></td>
                                    <td><?php echo esc_html( $row->ip_address ); ?></td>
                                    <td><?php echo wp_kses_post( $row->message ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>

                <?php if ( $is_pro ) : ?>
                    <h2><?php esc_html_e( 'Saved Views', 'aegisshield-security' ); ?></h2>
                    <p class="description">
                        <?php esc_html_e( 'Save the current filter combination and reuse it for recurring incident investigations.', 'aegisshield-security' ); ?>
                    </p>
                    <form method="post">
                        <?php wp_nonce_field( 'aegisshield_activity_views_save', 'aegisshield_activity_views_nonce' ); ?>
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="as_view_name"><?php esc_html_e( 'Save current filters as', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <input type="text" name="as_view_name" id="as_view_name" class="regular-text" />
                                    <p class="description">
                                        <?php esc_html_e( 'The view will appear under the Saved Views tab.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>
                        <p>
                            <button type="submit" class="button button-secondary">
                                <?php esc_html_e( 'Save View', 'aegisshield-security' ); ?>
                            </button>
                        </p>
                    </form>
                <?php else : ?>
                    <div class="notice notice-info inline" style="margin-top:15px;">
                        <p>
                            <?php esc_html_e( 'Saved Views and CSV export are available in AegisShield Pro.', 'aegisshield-security' ); ?>
                        </p>
                    </div>
                <?php endif; ?>
            </div>

            <div style="clear:both;"></div>
        </div>
        <?php
    }

    protected function render_alerts_tab( $is_pro ) {
        ?>
        <div class="aegisshield-alerts-tab">
            <?php if ( ! $is_pro ) : ?>
                <div class="notice notice-warning inline">
                    <p><strong><?php esc_html_e( 'Alerts are a Pro feature.', 'aegisshield-security' ); ?></strong></p>
                    <p>
                        <?php esc_html_e( 'In AegisShield Pro, you can turn critical Activity Log events into instant email alerts for early-warning on admin creation, plugin changes, and more.', 'aegisshield-security' ); ?>
                    </p>
                    <p>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>" class="button button-primary">
                            <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                        </a>
                    </p>
                </div>
            <?php else : ?>
                <?php
                $alert_rules = $this->get_alert_rules();
                ?>
                <h2><?php esc_html_e( 'Alert Rules', 'aegisshield-security' ); ?></h2>
                <p class="description">
                    <?php esc_html_e( 'Define simple rules that watch the Activity Log and send email alerts when specific event types occur.', 'aegisshield-security' ); ?>
                </p>

                <h3><?php esc_html_e( 'Existing Rules', 'aegisshield-security' ); ?></h3>

                <?php if ( empty( $alert_rules ) ) : ?>
                    <p><?php esc_html_e( 'No alert rules defined yet. Use the form below to add your first rule.', 'aegisshield-security' ); ?></p>
                <?php else : ?>
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Rule Name', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Event Type', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Email Recipients', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Status', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Actions', 'aegisshield-security' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $alert_rules as $rule ) : ?>
                                <tr>
                                    <td><?php echo isset( $rule['name'] ) ? esc_html( $rule['name'] ) : ''; ?></td>
                                    <td><code><?php echo isset( $rule['event_type'] ) ? esc_html( $rule['event_type'] ) : ''; ?></code></td>
                                    <td><?php echo isset( $rule['emails'] ) ? esc_html( $rule['emails'] ) : ''; ?></td>
                                    <td>
                                        <?php
                                        $enabled = isset( $rule['enabled'] ) ? (bool) $rule['enabled'] : true;
                                        echo $enabled ? esc_html__( 'Enabled', 'aegisshield-security' ) : esc_html__( 'Disabled', 'aegisshield-security' );
                                        ?>
                                    </td>
                                    <td>
                                        <form method="post" style="display:inline;">
                                            <?php wp_nonce_field( 'aegisshield_alert_rules_save', 'aegisshield_alert_rules_nonce' ); ?>
                                            <input type="hidden" name="as_alert_action" value="delete" />
                                            <input type="hidden" name="as_alert_rule_id" value="<?php echo isset( $rule['id'] ) ? esc_attr( $rule['id'] ) : ''; ?>" />
                                            <button type="submit" class="button-link delete-alert-rule"><?php esc_html_e( 'Delete', 'aegisshield-security' ); ?></button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>

                <hr />

                <h3><?php esc_html_e( 'Add New Alert Rule', 'aegisshield-security' ); ?></h3>
                <form method="post">
                    <?php wp_nonce_field( 'aegisshield_alert_rules_save', 'aegisshield_alert_rules_nonce' ); ?>
                    <input type="hidden" name="as_alert_action" value="add" />

                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="as_alert_name"><?php esc_html_e( 'Rule name', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <input type="text" name="as_alert_name" id="as_alert_name" class="regular-text" />
                                <p class="description">
                                    <?php esc_html_e( 'Example: "Alert if a new admin account is created".', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="as_alert_event_type"><?php esc_html_e( 'Event type', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
								<?php
								$event_types = $this->get_distinct_event_types();
								?>

								<select name="as_alert_event_type" id="as_alert_event_type">
									<?php if ( empty( $event_types ) ) : ?>
										<option value="">
											<?php esc_html_e( 'No events recorded yet', 'aegisshield-security' ); ?>
										</option>
									<?php else : ?>
										<?php foreach ( $event_types as $event_type ) : ?>
											<option value="<?php echo esc_attr( $event_type ); ?>">
												<?php echo esc_html( $event_type ); ?>
											</option>
										<?php endforeach; ?>
									<?php endif; ?>
								</select>
                                <p class="description">
                                    <?php esc_html_e( 'Alert fires whenever a log entry with this event type is recorded.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="as_alert_emails"><?php esc_html_e( 'Email recipients', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <textarea name="as_alert_emails" id="as_alert_emails" rows="3" class="large-text"></textarea>
                                <p class="description">
                                    <?php esc_html_e( 'Comma-separated list of email addresses to notify.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="as_alert_enabled"><?php esc_html_e( 'Enabled', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" name="as_alert_enabled" id="as_alert_enabled" value="1" checked="checked" />
                                    <?php esc_html_e( 'Send alerts when this rule matches', 'aegisshield-security' ); ?>
                                </label>
                            </td>
                        </tr>
                    </table>

                    <p>
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e( 'Add Alert Rule', 'aegisshield-security' ); ?>
                        </button>
                    </p>
                </form>
            <?php endif; ?>
        </div>
        <?php
    }

    protected function render_views_tab( $is_pro, $saved_views ) {
        ?>
        <div class="aegisshield-views-tab">
            <?php if ( ! $is_pro ) : ?>
                <div class="notice notice-warning inline">
                    <p><strong><?php esc_html_e( 'Saved Views are a Pro feature.', 'aegisshield-security' ); ?></strong></p>
                    <p>
                        <?php esc_html_e( 'Use Saved Views in Pro to snapshot important filter combinations and revisit them instantly.', 'aegisshield-security' ); ?>
                    </p>
                    <p>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>" class="button button-primary">
                            <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                        </a>
                    </p>
                </div>
            <?php else : ?>
                <h2><?php esc_html_e( 'Saved Views', 'aegisshield-security' ); ?></h2>
                <p class="description">
                    <?php esc_html_e( 'Apply or remove Saved Views created from the Activity Log tab.', 'aegisshield-security' ); ?>
                </p>

                <?php if ( empty( $saved_views ) ) : ?>
                    <p><?php esc_html_e( 'No saved views yet. Create one from the Activity Log tab.', 'aegisshield-security' ); ?></p>
                <?php else : ?>
                    <form method="post">
                        <?php wp_nonce_field( 'aegisshield_activity_views_save', 'aegisshield_activity_views_nonce' ); ?>
                        <table class="widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e( 'Name', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Filters', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'Actions', 'aegisshield-security' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $saved_views as $view ) : ?>
                                    <?php
                                    $filters    = isset( $view['filters'] ) && is_array( $view['filters'] ) ? $view['filters'] : array();
                                    $view_id    = isset( $view['id'] ) ? $view['id'] : '';
                                    $view_name  = isset( $view['name'] ) ? $view['name'] : '';

                                    $args = array(
                                        'page'         => 'aegisshield-activity-log',
                                        'tab'          => 'log',
                                        'as_al_search' => isset( $filters['search'] ) ? $filters['search'] : '',
                                        'as_al_user'   => isset( $filters['user'] ) ? $filters['user'] : '',
                                        'as_al_type'   => isset( $filters['type'] ) ? $filters['type'] : '',
                                        'as_al_from'   => isset( $filters['from'] ) ? $filters['from'] : '',
                                        'as_al_to'     => isset( $filters['to'] ) ? $filters['to'] : '',
                                    );
                                    $apply_url = add_query_arg( array_map( 'rawurlencode', $args ), admin_url( 'admin.php' ) );
                                    ?>
                                    <tr>
                                        <td><strong><?php echo esc_html( $view_name ); ?></strong></td>
                                        <td><code><?php echo esc_html( wp_json_encode( $filters ) ); ?></code></td>
                                        <td>
                                            <a href="<?php echo esc_url( $apply_url ); ?>" class="button button-secondary">
                                                <?php esc_html_e( 'Apply', 'aegisshield-security' ); ?>
                                            </a>
                                            <button type="submit" class="button-link-delete" name="as_view_delete_id" value="<?php echo esc_attr( $view_id ); ?>" style="margin-left:10px;">
                                                <?php esc_html_e( 'Delete', 'aegisshield-security' ); ?>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </form>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }

	protected function render_sessions_tab( $is_pro ) {

        if ( ! $is_pro ) {
            ?>
            <div class="aegisshield-sessions-tab">
                <div class="notice notice-warning inline">
                    <p><strong><?php esc_html_e( 'Sessions are a Pro feature.', 'aegisshield-security' ); ?></strong></p>
                    <p>
                        <?php esc_html_e( 'In AegisShield Pro, you can see active devices, terminate suspicious sessions, and enforce tighter account hygiene.', 'aegisshield-security' ); ?>
                    </p>
                    <p>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>" class="button button-primary">
                            <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                        </a>
                    </p>
                </div>
            </div>
            <?php
            return;
        }

		$users = get_users([
			'meta_key'     => 'session_tokens',
			'meta_compare' => 'EXISTS',
			'fields'       => ['ID', 'user_login', 'display_name']
		]);

		$parse_device = function( $ua ) {
			$ua = strtolower($ua);

			if (strpos($ua, 'iphone') || strpos($ua, 'ipad') || strpos($ua,'ios')) $os = 'iOS';
			elseif (strpos($ua, 'android')) $os = 'Android';
			elseif (strpos($ua,'mac os') || strpos($ua,'macintosh')) $os = 'macOS';
			elseif (strpos($ua,'windows')) $os = 'Windows';
			else $os = 'Unknown OS';

			if (strpos($ua,'edg')) $browser = 'Edge';
			elseif (strpos($ua,'chrome')) $browser = 'Chrome';
			elseif (strpos($ua,'safari') && !strpos($ua,'chrome')) $browser = 'Safari';
			elseif (strpos($ua,'firefox')) $browser = 'Firefox';
			else $browser = 'Unknown Browser';

			return $browser . ' on ' . $os;
		};
		?>
		<div class="aegisshield-sessions-tab">
			<h2><?php esc_html_e('Active Sessions & Devices', 'aegisshield-security'); ?></h2>
			<p class="description">
				<?php esc_html_e('These sessions come from WordPress core (session_tokens). You may terminate any device.', 'aegisshield-security'); ?>
			</p>

			<?php
			if (empty($users)) {
				echo '<p>No active sessions found.</p>';
				return;
			}
			?>

			<table class="widefat striped">
				<thead>
					<tr>
						<th><?php esc_html_e('User'); ?></th>
						<th><?php esc_html_e('Device'); ?></th>
						<th><?php esc_html_e('IP Address'); ?></th>
						<th><?php esc_html_e('Login Time'); ?></th>
						<th><?php esc_html_e('Expires'); ?></th>
						<th><?php esc_html_e('Current'); ?></th>
						<th><?php esc_html_e('Actions'); ?></th>
					</tr>
				</thead>
				<tbody>
				<?php
				$now = time();
				$current_user_id = get_current_user_id();
				$current_token = wp_get_session_token();

				foreach ( $users as $user ) {
					$manager = \WP_Session_Tokens::get_instance( $user->ID );
					$tokens  = $manager->get_all();
					if (empty($tokens)) continue;

					foreach ($tokens as $token => $session) {
						$ua     = isset($session['ua']) ? $session['ua'] : '';
						$device = $parse_device($ua);
						$ip     = isset($session['ip']) ? $session['ip'] : '';
						$login  = isset($session['login']) ? wp_date('Y-m-d H:i:s', $session['login']) : '—';
						$exp    = isset($session['expiration']) ? wp_date('Y-m-d H:i:s', $session['expiration']) : '—';

						$is_current = ($user->ID == $current_user_id && $token == $current_token);
						?>
						<tr>
							<td><?php echo esc_html($user->display_name ?: $user->user_login); ?></td>
							<td><?php echo esc_html($device); ?></td>
							<td><?php echo esc_html($ip); ?></td>
							<td><?php echo esc_html($login); ?></td>
							<td><?php echo esc_html($exp); ?></td>
							<td><?php echo $is_current ? 'Yes' : '—'; ?></td>
							<td>
								<?php if ( ! $is_current ) : ?>
									<form method="post" style="display:inline;">
										<?php wp_nonce_field('aegisshield_terminate_session', 'aegisshield_session_nonce'); ?>
										<input type="hidden" name="aegis_action" value="terminate_session">
										<input type="hidden" name="user_id" value="<?php echo esc_attr($user->ID); ?>">
										<input type="hidden" name="token" value="<?php echo esc_attr($token); ?>">
										<button class="button button-small"><?php esc_html_e('Terminate'); ?></button>
									</form>
								<?php else : ?>
									<span class="description"><?php esc_html_e('Current Session'); ?></span>
								<?php endif; ?>
							</td>
						</tr>
						<?php
					}
				}
				?>
				</tbody>
			</table>

			<h3><?php esc_html_e('Force Logout Options'); ?></h3>

			<form method="post" style="margin-top:10px;">
				<?php wp_nonce_field('aegisshield_terminate_all', 'aegisshield_all_nonce'); ?>
				<input type="hidden" name="aegis_action" value="terminate_all_sessions">
				<button class="button button-secondary">
					<?php esc_html_e('Terminate ALL sessions (site-wide)'); ?>
				</button>
			</form>

		</div>
		<?php
	}

protected function handle_sessions_post_actions() {

    if ( ! isset($_POST['aegis_action']) ) {
        return;
    }

    // Terminate single session
    if (
        $_POST['aegis_action'] === 'terminate_session' &&
        isset($_POST['aegisshield_session_nonce']) &&
        wp_verify_nonce($_POST['aegisshield_session_nonce'], 'aegisshield_terminate_session')
    ) {
        $uid   = absint($_POST['user_id']);
        $token = sanitize_text_field($_POST['token']);

        $manager = \WP_Session_Tokens::get_instance($uid);
        $manager->destroy($token);
        return;
    }

    // Terminate ALL sessions (site-wide)
    if (
        $_POST['aegis_action'] === 'terminate_all_sessions' &&
        isset($_POST['aegisshield_all_nonce']) &&
        wp_verify_nonce($_POST['aegisshield_all_nonce'], 'aegisshield_terminate_all')
    ) {
        $users = get_users([
            'fields'       => 'ID',
            'meta_key'     => 'session_tokens',
            'meta_compare' => 'EXISTS',
        ]);

        foreach ( $users as $uid ) {
            \WP_Session_Tokens::get_instance($uid)->destroy_all();
        }
    }
}

public function handle_download_archive() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Access denied.', 'aegisshield-security' ) );
    }

    check_admin_referer( 'aegisshield_al_download_archive', 'nonce' );

    $file = isset( $_GET['file'] ) ? sanitize_file_name( wp_unslash( $_GET['file'] ) ) : '';
    if ( '' === $file || false === strpos( $file, 'activity-log-' ) ) {
        wp_die( esc_html__( 'Invalid file.', 'aegisshield-security' ) );
    }

    $uploads = wp_upload_dir();
    $dir     = trailingslashit( $uploads['basedir'] ) . 'aegisshield/activity-log-archives/';
    $path    = $dir . $file;

    if ( ! file_exists( $path ) || ! is_readable( $path ) ) {
        \AegisShield\Modules\AS_Module_Activity_Log::log_violation(
            $this->plugin,
            'activity_log',
            'archive_download_missing',
            'Archive download failed: file missing or unreadable.',
            array( 'file' => (string) $file ),
            false,
            'failed'
        );
        wp_die( esc_html__( 'File not found.', 'aegisshield-security' ) );
    }

    nocache_headers();
    header( 'Content-Type: text/csv; charset=utf-8' );
    header( 'Content-Disposition: attachment; filename=' . $file );
    header( 'Content-Length: ' . filesize( $path ) );

    readfile( $path );
    exit;
}

protected function render_settings_tab( $is_pro, $retention_days ) {
		$settings = $this->plugin->get_settings();
		$section  = 'activity_log';

		$archive_enabled   = (int) $settings->get( $section, 'archive_enabled', 0 );
		$archive_frequency = (string) $settings->get( $section, 'archive_frequency', 'monthly' );
		$archive_format    = (string) $settings->get( $section, 'archive_format', 'csv' );

		// Safety clamps / defaults.
		if ( ! in_array( $archive_frequency, array( 'monthly', 'quarterly', '6months' ), true ) ) {
			$archive_frequency = 'monthly';
		}
		if ( ! in_array( $archive_format, array( 'csv' ), true ) ) {
			$archive_format = 'csv';
		}

		if ( ! $is_pro ) {
			$archive_enabled   = 0;
			$archive_frequency = 'monthly';
			$archive_format    = 'csv';
		}
        ?>
        <div class="aegisshield-settings-tab">
            <h2><?php esc_html_e( 'Retention Settings', 'aegisshield-security' ); ?></h2>
            <form method="post">
                <?php wp_nonce_field( 'aegisshield_activity_log_save', 'aegisshield_activity_log_nonce' ); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="retention_days"><?php esc_html_e( 'Retain logs for (days)', 'aegisshield-security' ); ?></label>
                        </th>
                        <td>
							<input type="number" min="1" max="365" id="retention_days" name="retention_days" value="<?php echo esc_attr( $retention_days ); ?>" />
							<p class="description">
								<?php esc_html_e( 'Older entries are automatically cleaned up by AegisShield maintenance.', 'aegisshield-security' ); ?>
								<?php if ( ! $is_pro ) : ?>
									<br />
									<strong><?php esc_html_e( 'Free version limit:', 'aegisshield-security' ); ?></strong>
									<?php esc_html_e( 'Retain logs for up to 7 days. Upgrade to Pro to keep a longer history.', 'aegisshield-security' ); ?>
								<?php else : ?>
									<br />
									<?php esc_html_e( 'As a Pro user, you can safely use longer retention for compliance and forensics.', 'aegisshield-security' ); ?>
								<?php endif; ?>
							</p>

							<p style="margin-top:8px;">
								<button
									type="submit"
									name="as_manual_purge"
									value="1"
									class="button"
									onclick="return confirm('<?php echo esc_js( __( 'Run manual purge now? This will permanently delete logs older than your retention setting.', 'aegisshield-security' ) ); ?>');"
								>
									<?php esc_html_e( 'Manual Purge', 'aegisshield-security' ); ?>
								</button>
								<span class="description" style="margin-left:8px;">
									<?php esc_html_e( 'Immediately removes logs older than your retention setting.', 'aegisshield-security' ); ?>
								</span>
							</p>
                        </td>
                    </tr>
					<tr class="<?php echo $is_pro ? '' : 'aegisshield-pro-locked'; ?>">
						<th scope="row">
							<?php esc_html_e( 'Archiving', 'aegisshield-security' ); ?>
							<?php if ( ! $is_pro ) : ?>
								<span class="aegisshield-pro-badge"><?php esc_html_e( 'Pro', 'aegisshield-security' ); ?></span>
							<?php endif; ?>
						</th>
						<td>
							<fieldset <?php echo $is_pro ? '' : 'disabled="disabled"'; ?>>
								<label>
									<input type="checkbox" name="archive_enabled" value="1" <?php checked( $archive_enabled ); ?> <?php disabled( ! $is_pro ); ?> />
									<?php esc_html_e( 'Enable scheduled log archiving', 'aegisshield-security' ); ?>
								</label>
								<br /><br />

								<label>
									<?php esc_html_e( 'Archive frequency', 'aegisshield-security' ); ?><br />
									<select name="archive_frequency" <?php disabled( ! $is_pro ); ?>>
										<option value="monthly" <?php selected( $archive_frequency, 'monthly' ); ?>><?php esc_html_e( 'Monthly', 'aegisshield-security' ); ?></option>
										<option value="quarterly" <?php selected( $archive_frequency, 'quarterly' ); ?>><?php esc_html_e( 'Quarterly', 'aegisshield-security' ); ?></option>
										<option value="6months" <?php selected( $archive_frequency, '6months' ); ?>><?php esc_html_e( 'Every 6 months', 'aegisshield-security' ); ?></option>
									</select>
								</label>

								<br /><br />

								<label>
									<?php esc_html_e( 'Archive format', 'aegisshield-security' ); ?><br />
									<select name="archive_format" <?php disabled( ! $is_pro ); ?>>
										<option value="csv" <?php selected( $archive_format, 'csv' ); ?>><?php esc_html_e( 'CSV', 'aegisshield-security' ); ?></option>
									</select>
								</label>

								<p class="description">
									<?php esc_html_e( 'Archives are written to uploads/aegisshield/activity-log-archives and purged according to retention.', 'aegisshield-security' ); ?>
								</p>
							</fieldset>

							<?php if ( $is_pro ) : ?>
								<hr />
								<h3 style="margin:10px 0 6px;"><?php esc_html_e( 'Available Archives', 'aegisshield-security' ); ?></h3>
								<?php
								$uploads = wp_upload_dir();
								$dir     = trailingslashit( $uploads['basedir'] ) . 'aegisshield/activity-log-archives/';
								$files   = is_dir( $dir ) ? glob( $dir . 'activity-log-*.csv' ) : array();

								if ( empty( $files ) ) :
									?>
									<p class="description"><?php esc_html_e( 'No archives yet. Archives will appear here after the scheduled job runs.', 'aegisshield-security' ); ?></p>
									<?php
								else :
									rsort( $files );
									echo '<ul style="margin:0; padding-left:18px;">';
									foreach ( $files as $path ) {
										$base = basename( $path );
										$url  = wp_nonce_url(
											admin_url( 'admin-post.php?action=aegisshield_al_download_archive&file=' . rawurlencode( $base ) ),
											'aegisshield_al_download_archive',
											'nonce'
										);
										echo '<li style="margin:0 0 6px;">';
										echo esc_html( $base ) . ' — ';
										echo '<a class="button button-small" href="' . esc_url( $url ) . '">' . esc_html__( 'Download', 'aegisshield-security' ) . '</a>';
										echo '</li>';
									}
									echo '</ul>';
								endif;
								?>
							<?php endif; ?>
						</td>
					</tr>
                </table>
                <p>
                    <button type="submit" class="button button-primary">
                        <?php esc_html_e( 'Save Settings', 'aegisshield-security' ); ?>
                    </button>
                </p>
            </form>
        </div>
        <?php
    }

    protected function get_saved_views() {
        $views = get_option( $this->views_option_key, array() );
        return is_array( $views ) ? $views : array();
    }

    protected function save_saved_views( array $views ) {
        update_option( $this->views_option_key, $views );
    }

    protected function handle_export_csv() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        global $wpdb;

        $logger = $this->plugin->get_logger();
        $table  = $logger->get_table_name();

        $search = isset( $_GET['as_al_search'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_search'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $user   = isset( $_GET['as_al_user'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_user'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $type   = isset( $_GET['as_al_type'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_type'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $from   = isset( $_GET['as_al_from'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_from'] ) ) : '';     // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $to     = isset( $_GET['as_al_to'] ) ? sanitize_text_field( wp_unslash( $_GET['as_al_to'] ) ) : '';         // phpcs:ignore WordPress.Security.NonceVerification.Recommended

        $where = ' WHERE 1=1 ';

        if ( $search ) {
            $like   = '%' . $wpdb->esc_like( $search ) . '%';
            $where .= $wpdb->prepare(
                ' AND (event_type LIKE %s OR message LIKE %s OR ip_address LIKE %s)',
                $like,
                $like,
                $like
            );
        }

        if ( '' !== $user ) {
            $where .= $wpdb->prepare( ' AND user_id = %d', absint( $user ) );
        }

        if ( '' !== $type ) {
            $where .= $wpdb->prepare( ' AND event_type = %s', $type );
        }

        if ( $from ) {
            $from_dt = $from . ' 00:00:00';
            $where  .= $wpdb->prepare( ' AND event_time >= %s', $from_dt );
        }

        if ( $to ) {
            $to_dt  = $to . ' 23:59:59';
            $where .= $wpdb->prepare( ' AND event_time <= %s', $to_dt );
        }

        $rows = $wpdb->get_results( "SELECT * FROM {$table} {$where} ORDER BY event_time DESC", ARRAY_A ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

        nocache_headers();
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=aegisshield-activity-log.csv' );

        $out = fopen( 'php://output', 'w' );

        fputcsv(
            $out,
            array(
                'event_time',
                'event_type',
                'user_id',
                'ip_address',
                'message',
            )
        );

        if ( ! empty( $rows ) ) {
            foreach ( $rows as $row ) {
                fputcsv(
                    $out,
                    array(
                        isset( $row['event_time'] ) ? $row['event_time'] : '',
                        isset( $row['event_type'] ) ? $row['event_type'] : '',
                        isset( $row['user_id'] ) ? $row['user_id'] : '',
                        isset( $row['ip_address'] ) ? $row['ip_address'] : '',
                        isset( $row['message'] ) ? wp_strip_all_tags( $row['message'] ) : '',
                    )
                );
            }
        }

        fclose( $out );
        exit;
    }
}
