<?php
namespace AegisShield;

defined( 'ABSPATH' ) || exit;

class AS_License_Manager {

    const OPTION_KEY       = 'aegisshield_license';
    const TRANSIENT_STATUS = 'aegisshield_license_status_checked';
    const CHECK_INTERVAL   = 4 * HOUR_IN_SECONDS;

    protected $api_base_url = 'https://aegisify.com/wp-json/chue-license/v1';

    protected $product_slug = 'aegisshield';

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function init() {
        add_action( 'admin_init', array( $this, 'maybe_check_license_status' ) );
        add_filter( 'aegisshield_is_pro_active', array( $this, 'filter_is_pro_active' ) );
    }

    public function get_license_data() {
        $data = get_option( self::OPTION_KEY, array() );

        if ( ! is_array( $data ) ) {
            $data = array();
        }

        return wp_parse_args( $data, array(
            'license_key'   => '',
            'license_email' => '',
            'status'        => 'inactive',
            'plan'          => '',
            'expires_at'    => '',
            'last_checked'  => 0,
            'message'       => '',
        ) );
    }

    public function update_license_data( array $data ) {
        update_option( self::OPTION_KEY, array_merge( $this->get_license_data(), $data ) );
    }

    public function is_pro_active() {
        $data = $this->get_license_data();

        if ( $data['status'] !== 'active' ) {
            return false;
        }

        if ( ! empty( $data['expires_at'] ) ) {
            $ts = strtotime( $data['expires_at'] );
            if ( $ts && $ts < time() ) {
                return false;
            }
        }

        return true;
    }

    public function filter_is_pro_active( $active ) {
        return $this->is_pro_active();
    }

    public function activate_license( $license_key, $email ) {
        $license_key = sanitize_text_field( $license_key );
        $email       = sanitize_email( $email );

        if ( empty( $license_key ) || empty( $email ) ) {
            return array( 'success' => false, 'message' => 'License key and email are required.' );
        }

        $body = array(
            'license_key' => $license_key,
            'email'       => $email,
            'site_url'    => home_url(),
            'plugin_slug' => 'aegisshield-security',
            'version'     => defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : '',
        );

        $response = $this->remote_post( '/' . $this->product_slug . '/activate', $body );

        if ( is_wp_error( $response ) ) {
            return array( 'success' => false, 'message' => $response->get_error_message() );
        }

        $data = $this->parse_api_response( $response );

        $this->update_license_data( array(
            'license_key'   => $license_key,
            'license_email' => $email,
            'status'        => $data['status'] ?? 'inactive',
            'plan'          => $data['plan'] ?? '',
            'expires_at'    => $data['expires_at'] ?? '',
            'message'       => $data['message'] ?? '',
            'last_checked'  => time(),
        ) );

        return array(
            'success' => ( $data['status'] ?? '' ) === 'active',
            'message' => $data['message'] ?? 'License activation completed.',
        );
    }

    public function deactivate_license() {
        $data = $this->get_license_data();

        if ( empty( $data['license_key'] ) ) {
            return array( 'success' => false, 'message' => 'No license key to deactivate.' );
        }

        $response = $this->remote_post(
            '/' . $this->product_slug . '/deactivate',
            array(
                'license_key' => $data['license_key'],
                'email'       => $data['license_email'],
                'site_url'    => home_url(),
            )
        );

        if ( is_wp_error( $response ) ) {
            return array( 'success' => false, 'message' => $response->get_error_message() );
        }

        $this->update_license_data( array(
            'status'       => 'inactive',
            'plan'         => '',
            'expires_at'   => '',
            'last_checked' => time(),
        ) );

        return array( 'success' => true, 'message' => 'License deactivated.' );
    }

    /**
     * CLM v2+ compatible update check with legacy fallback.
     */
    public function check_for_update() {
        $license = $this->get_license_data();

        if ( empty( $license['license_key'] ) || empty( $license['license_email'] ) ) {
            return null;
        }

        $query = array(
            'license_key' => $license['license_key'],
            'email'       => $license['license_email'],
            'site_url'    => home_url(),
            'version'     => defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : '',
        );

        $response = $this->remote_get( '/' . $this->product_slug . '/check', $query );

        if ( is_wp_error( $response ) ) {
            $response = $this->remote_post( '/' . $this->product_slug . '/update-check', $query );
        }

        if ( is_wp_error( $response ) ) {
            return null;
        }

        $data = $this->parse_api_response( $response );

        if ( ! is_array( $data ) ) {
            return null;
        }

        // Normalize fields
        if ( isset( $data['version'] ) && ! isset( $data['latest_version'] ) ) {
            $data['latest_version'] = $data['version'];
        }

        return $data;
    }

    public function maybe_check_license_status() {
        if ( get_transient( self::TRANSIENT_STATUS ) ) {
            return;
        }

        $data = $this->get_license_data();

        if ( empty( $data['license_key'] ) || empty( $data['license_email'] ) ) {
            return;
        }

        $response = $this->remote_post(
            '/' . $this->product_slug . '/status',
            array(
                'license_key' => $data['license_key'],
                'email'       => $data['license_email'],
                'site_url'    => home_url(),
            )
        );

        if ( ! is_wp_error( $response ) ) {
            $api = $this->parse_api_response( $response );

            $this->update_license_data( array(
                'status'       => $api['status'] ?? $data['status'],
                'plan'         => $api['plan'] ?? $data['plan'],
                'expires_at'   => $api['expires_at'] ?? $data['expires_at'],
                'message'      => $api['message'] ?? '',
                'last_checked' => time(),
            ) );
        }

        set_transient( self::TRANSIENT_STATUS, 1, self::CHECK_INTERVAL );
    }

    protected function remote_post( $path, array $body ) {
        return wp_remote_post(
            trailingslashit( $this->api_base_url ) . ltrim( $path, '/' ),
            array( 'timeout' => 15, 'body' => $body )
        );
    }

    protected function remote_get( $path, array $query = array() ) {
        $url = trailingslashit( $this->api_base_url ) . ltrim( $path, '/' );
        if ( ! empty( $query ) ) {
            $url = add_query_arg( $query, $url );
        }
        return wp_remote_get( $url, array( 'timeout' => 15 ) );
    }

    protected function parse_api_response( array $response ) {
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( 200 !== $code || empty( $body ) ) {
            return array( 'status' => 'error', 'message' => 'Invalid license server response.' );
        }

        $data = json_decode( $body, true );
        return is_array( $data ) ? $data : array( 'status' => 'error', 'message' => 'Malformed response.' );
    }
}
