<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Module_Activity_Log implements AS_Module_Interface {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'activity_log';
    }

    public function register_settings() {
        $settings = $this->plugin->get_settings();
        if ( null === $settings->get( 'activity_log', 'retention_days', null ) ) {
            $settings->set( 'activity_log', 'retention_days', 30 );
            $settings->save();
        }
    }

    public function init() {
        add_action( 'wp_login', array( $this, 'log_login_success' ), 10, 2 );
        add_action( 'wp_logout', array( $this, 'log_logout' ) );
        add_action( 'wp_login_failed', array( $this, 'log_login_failed' ) );
        add_action( 'activated_plugin', array( $this, 'log_plugin_activated' ) );
        add_action( 'deactivated_plugin', array( $this, 'log_plugin_deactivated' ) );
        add_action( 'upgrader_process_complete', array( $this, 'log_plugin_install_update' ), 10, 2 );
        add_action( 'user_register', array( $this, 'log_user_created' ) );
        add_action( 'set_user_role', array( $this, 'log_user_role_changed' ), 10, 3 );
		add_filter( 'cron_schedules', array( $this, 'add_cron_schedules' ) );
		add_action( 'aegisshield_activity_log_maintenance', array( $this, 'run_activity_log_maintenance' ) );
		add_action( 'aegisshield_activity_log_archive', array( $this, 'run_activity_log_archive' ) );

		$this->maybe_schedule_activity_log_maintenance();
		$this->maybe_schedule_activity_log_archive();
    }

    public function log_login_success( $user_login, $user ) {
        $logger = $this->plugin->get_logger();

        if ( ! $logger || ! is_object( $logger ) || ! method_exists( $logger, 'log' ) ) {
            self::log_violation(
                $this->plugin,
                'activity_log',
                'logger_unavailable',
                'Logger unavailable; login_success could not be recorded.',
                array(
                    'method'     => __FUNCTION__,
                    'user_login' => (string) $user_login,
                    'user_id'    => is_object( $user ) && isset( $user->ID ) ? (int) $user->ID : 0,
                ),
                false,
                'failed'
            );
            return;
        }

        if ( ! is_object( $user ) || ! isset( $user->ID ) ) {
            self::log_violation(
                $this->plugin,
                'activity_log',
                'invalid_user_object',
                'Invalid user object; login_success skipped.',
                array(
                    'method'     => __FUNCTION__,
                    'user_login' => (string) $user_login,
                ),
                false,
                'failed'
            );
            return;
        }

        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';

        $ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $prev_ip = (string) get_user_meta( $user->ID, 'aegisshield_last_login_ip', true );
        if ( $ip && $prev_ip && $prev_ip !== $ip ) {
            $logger->log(
                'login_new_ip',
                sprintf(
                    __( 'User %1$s logged in from a new IP (%2$s → %3$s).', 'aegisshield-security' ),
                    $user_login,
                    $prev_ip,
                    $ip
                ),
                array( 'user_id' => $user->ID, 'old_ip' => $prev_ip, 'new_ip' => $ip )
            );
        }
        if ( $ip ) {
            update_user_meta( $user->ID, 'aegisshield_last_login_ip', $ip );
        }
        $logger->log(
            'login_success',
            sprintf(
                /* translators: %s: username */
                __( 'User %s logged in successfully.', 'aegisshield-security' ),
                $user_login
            ),
            array( 'user_id' => $user->ID )
        );
    }

    public function log_logout() {
        $logger  = $this->plugin->get_logger();
        $user_id = get_current_user_id();
        if ( $user_id ) {
            $user = get_userdata( $user_id );
            $logger->log(
                'logout',
                sprintf(
                    /* translators: %s: username */
                    __( 'User %s logged out.', 'aegisshield-security' ),
                    $user ? $user->user_login : '#' . $user_id
                ),
                array( 'user_id' => $user_id )
            );
        }
    }

    public function log_login_failed( $username ) {
        $logger = $this->plugin->get_logger();
        $logger->log(
            'login_failed',
            sprintf(
                /* translators: %s: username */
                __( 'Failed login attempt for username %s.', 'aegisshield-security' ),
                $username
            ),
            array( 'username' => $username )
        );
    }

    public function log_plugin_activated( $plugin ) {
        $logger = $this->plugin->get_logger();
        $logger->log(
            'plugin_activated',
            sprintf(
                /* translators: %s: plugin path */
                __( 'Plugin activated: %s', 'aegisshield-security' ),
                $plugin
            )
        );
    }

    public function log_plugin_deactivated( $plugin ) {
        $logger = $this->plugin->get_logger();
        $logger->log(
            'plugin_deactivated',
            sprintf(
                /* translators: %s: plugin path */
                __( 'Plugin deactivated: %s', 'aegisshield-security' ),
                $plugin
            )
        );
    }

    public function log_user_created( $user_id ) {
        $user = get_userdata( $user_id );
        if ( ! $user ) {
            return;
        }

        $logger = $this->plugin->get_logger();
        $roles  = is_array( $user->roles ) ? $user->roles : array();

        $logger->log(
            'user_created',
            sprintf(
                /* translators: %s: username */
                __( 'New user account created: %s.', 'aegisshield-security' ),
                $user->user_login
            ),
            array(
                'user_id'   => $user_id,
                'username'  => $user->user_login,
                'email'     => $user->user_email,
                'roles'     => $roles,
                'is_admin'  => in_array( 'administrator', $roles, true ),
            )
        );
    }

    public function log_user_role_changed( $user_id, $role, $old_roles ) {
        $user = get_userdata( $user_id );
        if ( ! $user ) {
            return;
        }

        $logger    = $this->plugin->get_logger();
        $old_roles = is_array( $old_roles ) ? $old_roles : array();
        $new_roles = is_array( $user->roles ) ? $user->roles : array();

        $logger->log(
            'user_role_changed',
            sprintf(
                /* translators: 1: username, 2: role */
                __( 'User %1$s role updated to %2$s.', 'aegisshield-security' ),
                $user->user_login,
                $role
            ),
            array(
                'user_id'     => $user_id,
                'username'    => $user->user_login,
                'old_roles'   => $old_roles,
                'new_roles'   => $new_roles,
                'changed_to'  => $role,
                'is_admin_now'=> in_array( 'administrator', $new_roles, true ),
            )
        );
    }

    public function log_plugin_install_update( $upgrader, $hook_extra ) {
        if ( empty( $hook_extra['type'] ) || 'plugin' !== $hook_extra['type'] ) {
            return;
        }

        $action = isset( $hook_extra['action'] ) ? (string) $hook_extra['action'] : '';
        if ( 'install' !== $action && 'update' !== $action ) {
            return;
        }

        $logger = $this->plugin->get_logger();

        $plugins = array();
        if ( ! empty( $hook_extra['plugins'] ) && is_array( $hook_extra['plugins'] ) ) {
            $plugins = $hook_extra['plugins'];
        } elseif ( ! empty( $hook_extra['plugin'] ) && is_string( $hook_extra['plugin'] ) ) {
            $plugins = array( $hook_extra['plugin'] );
        }

        foreach ( $plugins as $plugin_file ) {
            $plugin_data = function_exists( 'get_plugin_data' ) && file_exists( WP_PLUGIN_DIR . '/' . $plugin_file )
                ? get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_file, false, false )
                : array();

            $name = isset( $plugin_data['Name'] ) && $plugin_data['Name'] ? $plugin_data['Name'] : $plugin_file;

            if ( 'install' === $action ) {
                $logger->log(
                    'plugin_installed',
                    sprintf(
                        /* translators: %s: plugin name */
                        __( 'Plugin installed: %s.', 'aegisshield-security' ),
                        $name
                    ),
                    array( 'plugin' => $plugin_file, 'plugin_name' => $name )
                );
            } elseif ( 'update' === $action ) {
                $logger->log(
                    'plugin_updated',
                    sprintf(
                        /* translators: %s: plugin name */
                        __( 'Plugin updated: %s.', 'aegisshield-security' ),
                        $name
                    ),
                    array( 'plugin' => $plugin_file, 'plugin_name' => $name )
                );
            }
        }
    }

	public function add_cron_schedules( $schedules ) {
		if ( ! isset( $schedules['aegis_monthly'] ) ) {
			$schedules['aegis_monthly'] = array(
				'interval' => 30 * DAY_IN_SECONDS,
				'display'  => __( 'Aegis Monthly', 'aegisshield-security' ),
			);
		}
		if ( ! isset( $schedules['aegis_quarterly'] ) ) {
			$schedules['aegis_quarterly'] = array(
				'interval' => 90 * DAY_IN_SECONDS,
				'display'  => __( 'Aegis Quarterly', 'aegisshield-security' ),
			);
		}
		if ( ! isset( $schedules['aegis_6months'] ) ) {
			$schedules['aegis_6months'] = array(
				'interval' => 180 * DAY_IN_SECONDS,
				'display'  => __( 'Aegis Every 6 Months', 'aegisshield-security' ),
			);
		}
		return $schedules;
	}

	private function maybe_schedule_activity_log_maintenance() {
		if ( ! wp_next_scheduled( 'aegisshield_activity_log_maintenance' ) ) {
			wp_schedule_event( time() + 300, 'daily', 'aegisshield_activity_log_maintenance' );
		}
	}

	public function maybe_schedule_activity_log_archive() {
		$settings = $this->plugin->get_settings();
		$is_pro   = method_exists( $this->plugin, 'is_pro_active' ) ? (bool) $this->plugin->is_pro_active() : false;

		wp_clear_scheduled_hook( 'aegisshield_activity_log_archive' );

		if ( ! $is_pro ) {
			return;
		}

		$enabled = (bool) $settings->get( 'activity_log', 'archive_enabled', false );
		if ( ! $enabled ) {
			return;
		}

		$freq = (string) $settings->get( 'activity_log', 'archive_frequency', 'monthly' );
		$recurrence = 'aegis_monthly';
		if ( 'quarterly' === $freq ) {
			$recurrence = 'aegis_quarterly';
		} elseif ( '6months' === $freq ) {
			$recurrence = 'aegis_6months';
		}

		wp_schedule_event( time() + 600, $recurrence, 'aegisshield_activity_log_archive' );
	}

	public function run_activity_log_maintenance() {
		self::purge_logs_by_retention( $this->plugin, 'cron' );
	}

	public function run_activity_log_archive() {
		self::archive_and_purge_by_retention( $this->plugin, 'cron_archive' );
	}

	public static function purge_logs_by_retention( AS_Plugin $plugin, $trigger = 'manual' ) {
		global $wpdb;

		$logger = $plugin->get_logger();
		if ( ! $logger || ! is_object( $logger ) || ! method_exists( $logger, 'get_table_name' ) ) {
			self::log_violation(
				$plugin,
				'activity_log',
				'retention_purge_logger_missing',
				'Retention purge failed: logger unavailable.',
				array( 'trigger' => (string) $trigger ),
				false,
				'failed'
			);
			return;
		}

		$settings = $plugin->get_settings();
		$is_pro   = method_exists( $plugin, 'is_pro_active' ) ? (bool) $plugin->is_pro_active() : false;

		$days = (int) $settings->get( 'activity_log', 'retention_days', 30 );
		if ( $days <= 0 ) {
			$days = 30;
		}

		if ( ! $is_pro && $days > 7 ) {
			$days = 7;
		}

		$cutoff_ts = time() - ( $days * DAY_IN_SECONDS );
		$cutoff_dt = gmdate( 'Y-m-d H:i:s', $cutoff_ts );

		$table = $logger->get_table_name();

		$deleted = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$table} WHERE event_time < %s",
				$cutoff_dt
			)
		);

		if ( false === $deleted ) {
			self::log_violation(
				$plugin,
				'activity_log',
				'retention_purge_db_failed',
				'Retention purge failed: database delete returned false.',
				array(
					'trigger'   => (string) $trigger,
					'cutoff_dt' => (string) $cutoff_dt,
					'wpdb_err'  => (string) $wpdb->last_error,
				),
				false,
				'failed'
			);
			return;
		}

		self::log_violation(
			$plugin,
			'activity_log',
			'retention_purge_ok',
			'Retention purge completed.',
			array(
				'trigger'   => (string) $trigger,
				'days'      => (int) $days,
				'cutoff_dt' => (string) $cutoff_dt,
				'deleted'   => (int) $deleted,
			),
			true,
			'allowed'
		);
	}

	public static function archive_and_purge_by_retention( AS_Plugin $plugin, $trigger = 'manual' ) {
		global $wpdb;

		$settings = $plugin->get_settings();
		$is_pro   = method_exists( $plugin, 'is_pro_active' ) ? (bool) $plugin->is_pro_active() : false;

		if ( ! $is_pro ) {
			self::log_violation(
				$plugin,
				'activity_log',
				'archive_denied_free',
				'Archive attempt blocked: Pro not active.',
				array( 'trigger' => (string) $trigger ),
				true,
				'enforced'
			);
			return;
		}

		$enabled = (bool) $settings->get( 'activity_log', 'archive_enabled', false );
		if ( ! $enabled ) {
			return;
		}

		$logger = $plugin->get_logger();
		if ( ! $logger || ! is_object( $logger ) || ! method_exists( $logger, 'get_table_name' ) ) {
			self::log_violation(
				$plugin,
				'activity_log',
				'archive_logger_missing',
				'Archiving failed: logger unavailable.',
				array( 'trigger' => (string) $trigger ),
				false,
				'failed'
			);
			return;
		}

		$days = (int) $settings->get( 'activity_log', 'retention_days', 30 );
		if ( $days <= 0 ) {
			$days = 30;
		}

		$cutoff_ts = time() - ( $days * DAY_IN_SECONDS );
		$cutoff_dt = gmdate( 'Y-m-d H:i:s', $cutoff_ts );

		$freq   = (string) $settings->get( 'activity_log', 'archive_frequency', 'monthly' );
		$format = (string) $settings->get( 'activity_log', 'archive_format', 'csv' );
		if ( '' === $format ) {
			$format = 'csv';
		}

		$uploads = wp_upload_dir();
		$base    = trailingslashit( $uploads['basedir'] ) . 'aegisshield/activity-log-archives/';
		if ( ! wp_mkdir_p( $base ) ) {
			self::log_violation(
				$plugin,
				'activity_log',
				'archive_dir_create_failed',
				'Archiving failed: could not create archive directory.',
				array( 'dir' => (string) $base ),
				false,
				'failed'
			);
			return;
		}

		$now = time();
		$key = gmdate( 'Y-m', $now );
		if ( 'quarterly' === $freq ) {
			$q   = (int) ceil( (int) gmdate( 'n', $now ) / 3 );
			$key = gmdate( 'Y', $now ) . '-Q' . $q;
		} elseif ( '6months' === $freq ) {
			$half = ( (int) gmdate( 'n', $now ) <= 6 ) ? 'H1' : 'H2';
			$key  = gmdate( 'Y', $now ) . '-' . $half;
		}

		$filename = 'activity-log-' . $key . '.csv';
		$path     = $base . $filename;

		$table = $logger->get_table_name();

		$rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT event_time, event_type, user_id, ip_address, message
				 FROM {$table}
				 WHERE event_time < %s
				 ORDER BY event_time ASC",
				$cutoff_dt
			),
			ARRAY_A
		);

		if ( null === $rows ) {
			self::log_violation(
				$plugin,
				'activity_log',
				'archive_select_failed',
				'Archiving failed: select returned null.',
				array(
					'cutoff_dt' => (string) $cutoff_dt,
					'wpdb_err'  => (string) $wpdb->last_error,
				),
				false,
				'failed'
			);
			return;
		}

		if ( empty( $rows ) ) {
			return;
		}

		$is_new = ! file_exists( $path );
		$fh     = fopen( $path, 'a' );
		if ( ! $fh ) {
			self::log_violation(
				$plugin,
				'activity_log',
				'archive_file_open_failed',
				'Archiving failed: could not open archive file for writing.',
				array( 'path' => (string) $path ),
				false,
				'failed'
			);
			return;
		}

		if ( $is_new ) {
			fputcsv( $fh, array( 'event_time', 'event_type', 'user_id', 'ip_address', 'message' ) );
		}

		foreach ( $rows as $r ) {
			fputcsv(
				$fh,
				array(
					isset( $r['event_time'] ) ? (string) $r['event_time'] : '',
					isset( $r['event_type'] ) ? (string) $r['event_type'] : '',
					isset( $r['user_id'] ) ? (string) $r['user_id'] : '',
					isset( $r['ip_address'] ) ? (string) $r['ip_address'] : '',
					isset( $r['message'] ) ? (string) $r['message'] : '',
				)
			);
		}
		fclose( $fh );

		$deleted = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$table} WHERE event_time < %s",
				$cutoff_dt
			)
		);

		if ( false === $deleted ) {
			self::log_violation(
				$plugin,
				'activity_log',
				'archive_purge_failed',
				'Archiving wrote file but purge failed.',
				array(
					'path'     => (string) $path,
					'cutoff_dt'=> (string) $cutoff_dt,
					'wpdb_err' => (string) $wpdb->last_error,
				),
				false,
				'failed'
			);
			return;
		}

		self::log_violation(
			$plugin,
			'activity_log',
			'archive_ok',
			'Archiving completed and old rows purged.',
			array(
				'file'      => (string) $filename,
				'cutoff_dt' => (string) $cutoff_dt,
				'rows'      => (int) count( $rows ),
				'deleted'   => (int) $deleted,
				'freq'      => (string) $freq,
				'format'    => (string) $format,
				'trigger'   => (string) $trigger,
			),
			true,
			'allowed'
		);
	}
	 
    public static function log_violation( AS_Plugin $plugin, $feature, $rule, $details, array $context = array(), $enforced = false, $status = 'violation' ) {
	
        $ip        = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
        $request   = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        $user_id   = function_exists( 'get_current_user_id' ) ? (int) get_current_user_id() : 0;

        $payload = array_merge(
            array(
                'feature'   => (string) $feature,
                'rule'      => (string) $rule,
                'status'    => (string) $status,
                'enforced'  => (bool) $enforced,
                'user_id'   => $user_id,
                'ip'        => $ip,
                'request'   => $request,
            ),
            $context
        );

        $logger = is_object( $plugin ) ? $plugin->get_logger() : null;

        if ( $logger && is_object( $logger ) && method_exists( $logger, 'log' ) ) {
            $logger->log(
                'security_violation',
                sprintf(
                    /* translators: 1: feature, 2: details */
                    __( '[%1$s] %2$s', 'aegisshield-security' ),
                    (string) $feature,
                    (string) $details
                ),
                $payload
            );
            return;
        }

        error_log( '[AegisShield] security_violation: ' . wp_json_encode( $payload ) );
    }


}