<?php
namespace AegisShield\Admin_Pages;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

use AegisShield\AS_Plugin;
use AegisShield\Modules\AS_Module_DB_Tools;

defined( 'ABSPATH' ) || exit;

class AS_Page_DB_Tools {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

	protected function is_pro_active() {
		return function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
	}

	protected function render_pro_locked_notice() {
		?>
		<div class="notice notice-warning inline">
			<p><strong><?php esc_html_e( 'This feature is available in AegisShield Pro.', 'aegisshield-security' ); ?></strong></p>
			<p><?php esc_html_e( 'Upgrade to unlock advanced optimization scheduling and full DB prefix manager with guided renaming.', 'aegisshield-security' ); ?></p>
			<p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>"
				   class="button button-primary">
				   <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
				</a>
			</p>
		</div>
		<?php
	}

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegisshield-security' ) );
        }

        global $wpdb;

        $settings = $this->plugin->get_settings();
        $logger   = $this->plugin->get_logger();
        $section  = 'db_tools';

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        $current_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'overview';
        if ( ! in_array( $current_tab, array( 'overview', 'optimize', 'prefix' ), true ) ) {
            $current_tab = 'overview';
        }

        $weekly_optimize        = (string) $settings->get( $section, 'weekly_optimize', '' );
        $growth_monitoring      = (string) $settings->get( $section, 'growth_monitoring', '' );
        $growth_email           = (string) $settings->get( $section, 'growth_email', '' );
        $growth_email_addresses = (string) $settings->get( $section, 'growth_email_addresses', '' );
		
		$growth_threshold_mb    = (int) $settings->get( $section, 'growth_threshold_mb', 100 );
		$growth_threshold_mb    = max( 10, $growth_threshold_mb );

		$global_primary    = get_option( 'admin_email' );
		$global_additional = '';

		if ( isset( $this->plugin ) && method_exists( $this->plugin, 'get_notifier' ) ) {
			$notifier = $this->plugin->get_notifier();

			if ( $notifier && method_exists( $notifier, 'get_global_recipients' ) ) {
				$global = $notifier->get_global_recipients(); // expected: ['primary' => '', 'additional' => 'a@b.com,c@d.com']
				if ( is_array( $global ) ) {
					$global_primary    = ! empty( $global['primary'] ) ? $global['primary'] : $global_primary;
					$global_additional = ! empty( $global['additional'] ) ? $global['additional'] : '';
				}
			}
		}

		if ( '' === trim( $growth_email ) ) {
			$growth_email = (string) $global_primary;
		}
		if ( '' === trim( $growth_email_addresses ) ) {
			$growth_email_addresses = (string) $global_additional;
		}

        $updated   = false;
        $optimized = false;

        $module = new AS_Module_DB_Tools( $this->plugin );

        if (
            'overview' === $current_tab &&
            isset( $_POST['aegisshield_db_tools_export_csv_nonce'] ) &&
            wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_db_tools_export_csv_nonce'] ) ),
                'aegisshield_db_tools_export_csv'
            )
        ) {
			$module->log_violation(
				'db_tools_export_csv',
				__( 'Database table summary exported as CSV.', 'aegisshield-security' ),
				array( 'tab' => 'overview' ),
				'info'
			);           
		    
			$summary = $module->get_table_summary();
            $tables  = isset( $summary['tables'] ) ? $summary['tables'] : array();

            nocache_headers();
            header( 'Content-Type: text/csv; charset=utf-8' );
            header( 'Content-Disposition: attachment; filename=aegisshield-db-summary.csv' );

            $output = fopen( 'php://output', 'w' );
            if ( $output ) {
                fputcsv(
                    $output,
                    array(
                        'Table',
                        'Engine',
                        'Rows',
                        'Data Size (bytes)',
                        'Index Size (bytes)',
                        'Total Size (bytes)',
                        'Change (bytes)',
                        'Change (%)',
                    )
                );

                foreach ( $tables as $table ) {
                    $name        = isset( $table['name'] ) ? $table['name'] : '';
                    $engine      = isset( $table['engine'] ) ? $table['engine'] : '';
                    $rows        = isset( $table['rows'] ) ? (int) $table['rows'] : 0;
                    $data_bytes  = isset( $table['data_bytes'] ) ? (int) $table['data_bytes'] : 0;
                    $index_bytes = isset( $table['index_bytes'] ) ? (int) $table['index_bytes'] : 0;
                    $total_bytes = isset( $table['total_bytes'] ) ? (int) $table['total_bytes'] : 0;
                    $delta_bytes = isset( $table['delta_bytes'] ) ? (int) $table['delta_bytes'] : 0;
                    $delta_pct   = isset( $table['delta_pct'] ) ? (float) $table['delta_pct'] : 0.0;

                    fputcsv(
                        $output,
                        array(
                            $name,
                            $engine,
                            $rows,
                            $data_bytes,
                            $index_bytes,
                            $total_bytes,
                            $delta_bytes,
                            $delta_pct,
                        )
                    );
                }

                fclose( $output );
            }

            exit;
        }

        if (
            'overview' === $current_tab &&
            isset( $_POST['aegisshield_db_tools_overview_settings_nonce'] ) &&
            wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_db_tools_overview_settings_nonce'] ) ),
                'aegisshield_db_tools_overview_settings'
            )
        ) {
            $growth_monitoring_post      = isset( $_POST['growth_monitoring'] ) ? 'on' : '';
            $growth_threshold_mb_post    = isset( $_POST['growth_threshold_mb'] ) ? (int) $_POST['growth_threshold_mb'] : 100;
			$growth_email_post_raw           = isset( $_POST['growth_email'] ) ? sanitize_email( wp_unslash( $_POST['growth_email'] ) ) : '';
			$growth_email_addresses_post_raw = isset( $_POST['growth_email_addresses'] ) ? sanitize_text_field( wp_unslash( $_POST['growth_email_addresses'] ) ) : '';

			$growth_email_post           = trim( $growth_email_post_raw );
			$growth_email_addresses_post = trim( $growth_email_addresses_post_raw );

			$global_primary_for_compare    = get_option( 'admin_email' );
			$global_additional_for_compare = '';

			if ( isset( $notifier ) && $notifier && method_exists( $notifier, 'get_global_recipients' ) ) {
				$global_cmp = $notifier->get_global_recipients();
				if ( is_array( $global_cmp ) ) {
					$global_primary_for_compare    = ! empty( $global_cmp['primary'] ) ? $global_cmp['primary'] : $global_primary_for_compare;
					$global_additional_for_compare = ! empty( $global_cmp['additional'] ) ? $global_cmp['additional'] : '';
				}
			}

			if ( $growth_email_post === $global_primary_for_compare ) {
				$growth_email_post = '';
			}
			if ( $growth_email_addresses_post === $global_additional_for_compare ) {
				$growth_email_addresses_post = '';
			}


            $settings->set( $section, 'growth_monitoring', $growth_monitoring_post );
            $settings->set( $section, 'growth_threshold_mb', $growth_threshold_mb_post );
            $settings->set( $section, 'growth_email', $growth_email_post );
            $settings->set( $section, 'growth_email_addresses', $growth_email_addresses_post );
            $settings->save();

            $growth_monitoring      = $growth_monitoring_post;
            $growth_threshold_mb    = $growth_threshold_mb_post;
            $growth_email           = $growth_email_post;
            $growth_email_addresses = $growth_email_addresses_post;

            $updated = true;

			$module->log_violation(
				'db_tools_overview_settings_updated',
				__( 'DB Tools overview monitoring settings updated.', 'aegisshield-security' ),
				array(
					'growth_monitoring'   => $growth_monitoring_post,
					'growth_threshold_mb' => $growth_threshold_mb_post,
					'growth_email'        => $growth_email_post,
					'tab'                 => 'overview',
				),
				'info'
			);
        }

        if ( 'optimize' === $current_tab ) {
            if (
                isset( $_POST['aegisshield_db_tools_settings_nonce'] ) &&
                wp_verify_nonce(
                    sanitize_text_field( wp_unslash( $_POST['aegisshield_db_tools_settings_nonce'] ) ),
                    'aegisshield_db_tools_settings'
                )
            ) {
                $weekly_optimize_post = isset( $_POST['weekly_optimize'] ) ? 'on' : '';

                $settings->set( $section, 'weekly_optimize', $weekly_optimize_post );
                $settings->save();

                $weekly_optimize = $weekly_optimize_post;
                $updated         = true;

                if ( $logger ) {
                    $logger->log(
                        'db_tools_optimize_settings_updated',
                        array(
                            'weekly_optimize' => $weekly_optimize_post,
                            'user_id'         => get_current_user_id(),
                        )
                    );
                }
            }

            if (
                isset( $_POST['aegisshield_db_tools_optimize_now_nonce'] ) &&
                wp_verify_nonce(
                    sanitize_text_field( wp_unslash( $_POST['aegisshield_db_tools_optimize_now_nonce'] ) ),
                    'aegisshield_db_tools_optimize_now'
                )
            ) {
				$module->run_manual_optimization();
				$module->log_violation(
					'db_tools_manual_optimize_clicked',
					__( 'Optimize Now clicked in DB Tools.', 'aegisshield-security' ),
					array( 'tab' => 'optimize' ),
					'info'
				);

                $optimized = true;

                if ( $logger ) {
                    $logger->log(
                        'db_tools_manual_optimize_run',
                        array(
                            'user_id' => get_current_user_id(),
                        )
                    );
                }
            }
			$purged_noise_logs = false;
			$purged_count      = 0;

			if (
				isset( $_POST['aegisshield_db_tools_purge_noise_nonce'] ) &&
				wp_verify_nonce(
					sanitize_text_field( wp_unslash( $_POST['aegisshield_db_tools_purge_noise_nonce'] ) ),
					'aegisshield_db_tools_purge_noise'
				)
			) {
				global $wpdb;

				// Use logger table name if available; fallback to default.
				$log_table = ( method_exists( $this->plugin, 'get_logger' ) && $this->plugin->get_logger() )
					? $this->plugin->get_logger()->get_table_name()
					: $wpdb->prefix . 'aegisshield_activity_log';

				$purged_count = (int) $wpdb->query(
					$wpdb->prepare(
						"DELETE FROM {$log_table} WHERE event_type IN (%s,%s)",
						'db_tools_growth_monitoring_running',
						'db_tools_weekly_optimize_enabled'
					)
				);

				$purged_noise_logs = true;

				// Optional: record one clean info event about the purge.
				$module->log_violation(
					'db_tools_noise_logs_purged',
					__( 'Purged noisy DB Tools heartbeat logs.', 'aegisshield-security' ),
					array( 'deleted_rows' => $purged_count ),
					'info'
				);
			}

        }

        $summary  = $module->get_table_summary();
        $tables   = isset( $summary['tables'] ) ? $summary['tables'] : array();
        $warnings = isset( $summary['warnings'] ) ? $summary['warnings'] : array();

        $filter_engine = isset( $_GET['engine'] ) ? sanitize_text_field( wp_unslash( $_GET['engine'] ) ) : '';
        $filter_size   = isset( $_GET['size'] ) ? sanitize_text_field( wp_unslash( $_GET['size'] ) ) : '';
        $filter_growth = isset( $_GET['growth'] ) ? sanitize_text_field( wp_unslash( $_GET['growth'] ) ) : '';
        $search_name   = isset( $_GET['search_table'] ) ? sanitize_text_field( wp_unslash( $_GET['search_table'] ) ) : '';

        $filtered_tables = array();

        foreach ( $tables as $table ) {
            $name        = isset( $table['name'] ) ? $table['name'] : '';
            $engine      = isset( $table['engine'] ) ? $table['engine'] : '';
            $total_bytes = isset( $table['total_bytes'] ) ? (int) $table['total_bytes'] : 0;
            $delta_bytes = isset( $table['delta_bytes'] ) ? (int) $table['delta_bytes'] : 0;

            if ( $filter_engine ) {
                if ( 'innodb' === strtolower( $filter_engine ) && 'InnoDB' !== $engine ) {
                    continue;
                }
                if ( 'myisam' === strtolower( $filter_engine ) && 'MyISAM' !== $engine ) {
                    continue;
                }
                if ( 'memory' === strtolower( $filter_engine ) && 'MEMORY' !== $engine ) {
                    continue;
                }
            }

            if ( $filter_size ) {
                if ( 'gt_100' === $filter_size && $total_bytes <= 100 * 1024 * 1024 ) {
                    continue;
                }
                if ( 'gt_50' === $filter_size && $total_bytes <= 50 * 1024 * 1024 ) {
                    continue;
                }
                if ( 'gt_10' === $filter_size && $total_bytes <= 10 * 1024 * 1024 ) {
                    continue;
                }
            }

            if ( $filter_growth && 'spikes' === $filter_growth && $delta_bytes <= 0 ) {
                continue;
            }

            if ( $search_name && false === stripos( $name, $search_name ) ) {
                continue;
            }

            $filtered_tables[] = $table;
        }

        $sort = isset( $_GET['sort'] ) ? sanitize_text_field( wp_unslash( $_GET['sort'] ) ) : 'total_desc';

        usort(
            $filtered_tables,
            function ( $a, $b ) use ( $sort ) {
                $fields = array(
                    'name'        => 'name',
                    'engine'      => 'engine',
                    'rows'        => 'rows',
                    'total_bytes' => 'total_bytes',
                    'delta_bytes' => 'delta_bytes',
                );

                $field = 'total_bytes';
                $dir   = 'desc';

                if ( 'name_asc' === $sort ) {
                    $field = 'name';
                    $dir   = 'asc';
                } elseif ( 'name_desc' === $sort ) {
                    $field = 'name';
                    $dir   = 'desc';
                } elseif ( 'rows_asc' === $sort ) {
                    $field = 'rows';
                    $dir   = 'asc';
                } elseif ( 'rows_desc' === $sort ) {
                    $field = 'rows';
                    $dir   = 'desc';
                } elseif ( 'delta_desc' === $sort ) {
                    $field = 'delta_bytes';
                    $dir   = 'desc';
                } elseif ( 'delta_asc' === $sort ) {
                    $field = 'delta_bytes';
                    $dir   = 'asc';
                } elseif ( 'total_asc' === $sort ) {
                    $field = 'total_bytes';
                    $dir   = 'asc';
                }

                $av = isset( $a[ $field ] ) ? $a[ $field ] : '';
                $bv = isset( $b[ $field ] ) ? $b[ $field ] : '';

                if ( is_numeric( $av ) && is_numeric( $bv ) ) {
                    $av = (float) $av;
                    $bv = (float) $bv;
                } else {
                    $av = (string) $av;
                    $bv = (string) $bv;
                }

                if ( $av === $bv ) {
                    return 0;
                }

                if ( 'asc' === $dir ) {
                    return ( $av < $bv ) ? -1 : 1;
                }

                return ( $av > $bv ) ? -1 : 1;
            }
        );

        $allowed_per_page = array( 25, 50, 100 );
        $per_page         = isset( $_GET['as_db_per_page'] ) ? (int) $_GET['as_db_per_page'] : 25;
        if ( ! in_array( $per_page, $allowed_per_page, true ) ) {
            $per_page = 25;
        }

        $total_items = count( $filtered_tables );
        $total_pages = $total_items > 0 ? (int) max( 1, ceil( $total_items / $per_page ) ) : 1;

        $current_page = isset( $_GET['as_db_paged'] ) ? (int) $_GET['as_db_paged'] : 1;
        if ( $current_page < 1 ) {
            $current_page = 1;
        }
        if ( $current_page > $total_pages ) {
            $current_page = $total_pages;
        }

        $offset       = ( $current_page - 1 ) * $per_page;
        $paged_tables = array_slice( $filtered_tables, $offset, $per_page );

        $prefix_current        = $wpdb->prefix;
        $prefix_new            = '';
        $prefix_mode           = 'core';
        $exec_mode             = 'standard';
        $prefix_preview        = array();
        $prefix_preview_options = array();
        $prefix_preview_usermeta = array();
        $wp_config_info        = array();
        $prefix_notice         = '';
        $prefix_verify_result = array();
        $prefix_verify_notice = '';

        $backup_notice = '';
        $backups       = array();

        if ( 'prefix' === $current_tab && $is_pro ) {

            if ( method_exists( $module, 'list_db_backups' ) ) {
                $backups = $module->list_db_backups();
            }

            $backup_action = isset( $_GET['as_db_backup_action'] ) ? sanitize_key( wp_unslash( $_GET['as_db_backup_action'] ) ) : '';
            $backup_file   = isset( $_GET['backup_file'] ) ? sanitize_file_name( wp_unslash( $_GET['backup_file'] ) ) : '';

            if ( $backup_action && $backup_file ) {

                $nonce_action = 'aegisshield_db_backup_' . $backup_action . '_' . $backup_file;
                $nonce_value  = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';

                if ( empty( $nonce_value ) || ! wp_verify_nonce( $nonce_value, $nonce_action ) ) {
                    $backup_notice = __( 'Security check failed for backup action.', 'aegisshield-security' );

                    if ( $logger ) {
                        $logger->log(
                            'db_tools_backup_action_nonce_failed',
                            array(
                                'action' => $backup_action,
                                'file'   => $backup_file,
                                'user_id'=> get_current_user_id(),
                            )
                        );
                    }

                } else {

                    $selected = array();
                    foreach ( $backups as $b ) {
                        $bf = isset( $b['file'] ) ? (string) $b['file'] : '';
                        if ( $bf === $backup_file ) {
                            $selected = $b;
                            break;
                        }
                    }

                    $backup_path = isset( $selected['path'] ) ? (string) $selected['path'] : '';

                    if ( empty( $backup_path ) || ! file_exists( $backup_path ) ) {
                        $backup_notice = __( 'Backup snapshot was not found on disk.', 'aegisshield-security' );

                        if ( $logger ) {
                            $logger->log(
                                'db_tools_backup_action_missing_file',
                                array(
                                    'action' => $backup_action,
                                    'file'   => $backup_file,
                                    'path'   => $backup_path,
                                    'user_id'=> get_current_user_id(),
                                )
                            );
                        }

                    } else {

                        if ( 'download' === $backup_action ) {

                            if ( $logger ) {
                                $logger->log(
                                    'db_tools_backup_download',
                                    array(
                                        'file'    => $backup_file,
                                        'path'    => $backup_path,
                                        'user_id' => get_current_user_id(),
                                    )
                                );
                            }

                            nocache_headers();
                            header( 'Content-Type: application/octet-stream' );
                            header( 'Content-Disposition: attachment; filename="' . basename( $backup_path ) . '"' );
                            header( 'Content-Length: ' . (string) filesize( $backup_path ) );
                            header( 'X-Content-Type-Options: nosniff' );
                            readfile( $backup_path );
                            exit;

                        } elseif ( 'delete' === $backup_action ) {

                            $deleted = @unlink( $backup_path );

                            if ( $deleted ) {
                                $backup_notice = __( 'Backup deleted successfully.', 'aegisshield-security' );

                                if ( $logger ) {
                                    $logger->log(
                                        'db_tools_backup_deleted',
                                        array(
                                            'file'    => $backup_file,
                                            'path'    => $backup_path,
                                            'user_id' => get_current_user_id(),
                                        )
                                    );
                                }

                                wp_safe_redirect( remove_query_arg( array( 'as_db_backup_action', 'backup_file', '_wpnonce' ) ) );
                                exit;

                            } else {
                                $backup_notice = __( 'Failed to delete the backup file. Check file permissions.', 'aegisshield-security' );

                                if ( $logger ) {
                                    $logger->log(
                                        'db_tools_backup_delete_failed',
                                        array(
                                            'file'    => $backup_file,
                                            'path'    => $backup_path,
                                            'user_id' => get_current_user_id(),
                                        )
                                    );
                                }
                            }

                        } elseif ( 'restore' === $backup_action ) {

                            $restore_result = array(
                                'success' => false,
                                'message' => __( 'Restore engine is not available in this build.', 'aegisshield-security' ),
                            );

                            if ( method_exists( $module, 'restore_db_backup_snapshot' ) ) {
                                $restore_result = $module->restore_db_backup_snapshot( $backup_file );
                            } elseif ( method_exists( $module, 'restore_db_backup' ) ) {
                                $restore_result = $module->restore_db_backup( $backup_file );
                            } elseif ( method_exists( $module, 'restore_db_backup_from_file' ) ) {
                                $restore_result = $module->restore_db_backup_from_file( $backup_path );
                            }

                            if ( ! empty( $restore_result['success'] ) ) {
                                $backup_notice = ! empty( $restore_result['message'] )
                                    ? (string) $restore_result['message']
                                    : __( 'Restore completed successfully.', 'aegisshield-security' );

                                if ( $logger ) {
                                    $logger->log(
                                        'db_tools_backup_restored',
                                        array(
                                            'file'    => $backup_file,
                                            'path'    => $backup_path,
                                            'user_id' => get_current_user_id(),
                                        )
                                    );
                                }

                                wp_safe_redirect( remove_query_arg( array( 'as_db_backup_action', 'backup_file', '_wpnonce' ) ) );
                                exit;

                            } else {
                                $backup_notice = ! empty( $restore_result['message'] )
                                    ? (string) $restore_result['message']
                                    : __( 'Restore failed.', 'aegisshield-security' );

                                if ( $logger ) {
                                    $logger->log(
                                        'db_tools_backup_restore_failed',
                                        array(
                                            'file'    => $backup_file,
                                            'path'    => $backup_path,
                                            'user_id' => get_current_user_id(),
                                            'message' => $backup_notice,
                                        )
                                    );
                                }
                            }
                        }
                    }
                }
            }


            if (
                isset( $_POST['aegisshield_db_backup_create_nonce'] ) &&
                wp_verify_nonce(
                    sanitize_text_field( wp_unslash( $_POST['aegisshield_db_backup_create_nonce'] ) ),
                    'aegisshield_db_backup_create'
                )
            ) {
                if ( method_exists( $module, 'create_db_backup_snapshot' ) ) {
                    $result = $module->create_db_backup_snapshot();
                    if ( method_exists( $module, 'list_db_backups' ) ) {
                        $backups = $module->list_db_backups();
                    }
                    if ( ! empty( $result['success'] ) ) {
                        $backup_notice = isset( $result['message'] ) ? (string) $result['message'] : __( 'Backup created.', 'aegisshield-security' );
                    } else {
                        $backup_notice = isset( $result['message'] ) ? (string) $result['message'] : __( 'Backup failed.', 'aegisshield-security' );
                    }
                } else {
                    $backup_notice = __( 'Backup engine is not available in this build.', 'aegisshield-security' );
                }

            } elseif (

                isset( $_POST['aegisshield_db_prefix_verify_nonce'] ) &&
                wp_verify_nonce(
                    sanitize_text_field( wp_unslash( $_POST['aegisshield_db_prefix_verify_nonce'] ) ),
                    'aegisshield_db_prefix_verify'
                )
            ) {
                $migration = get_option( 'aegisshield_dbtools_last_prefix_migration', array() );
                $old_pref  = '';
                if ( is_array( $migration ) && ! empty( $migration['old_prefix'] ) ) {
                    $old_pref = (string) $migration['old_prefix'];
                }

                $prefix_verify_result = $module->run_prefix_verification_scan( $wpdb->prefix, $old_pref );

                if ( empty( $prefix_verify_result['success'] ) ) {
                    $prefix_verify_notice = __( 'Verification found blocking issues. Review the results before proceeding.', 'aegisshield-security' );
                } else {
                    $prefix_verify_notice = __( 'Verification scan completed. No blocking issues detected.', 'aegisshield-security' );
                }
} elseif (

                isset( $_POST['aegisshield_db_prefix_preview_nonce'] ) &&
                wp_verify_nonce(
                    sanitize_text_field( wp_unslash( $_POST['aegisshield_db_prefix_preview_nonce'] ) ),
                    'aegisshield_db_prefix_preview'
                )
            ) {
                $prefix_new  = isset( $_POST['new_prefix'] ) ? sanitize_text_field( wp_unslash( $_POST['new_prefix'] ) ) : '';
                $prefix_mode = isset( $_POST['prefix_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['prefix_mode'] ) ) : 'core';
				if ( $prefix_mode === 'core' ) {
					$prefix_mode = 'preview';
				}
                $exec_mode   = isset( $_POST['exec_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['exec_mode'] ) ) : 'standard';

                

                $validation_error = $this->validate_new_prefix( $prefix_new, $prefix_current );
                if ( $validation_error ) {
                    $prefix_notice = $validation_error;
                } else {
                    $prefix_preview = $this->build_prefix_preview( $tables, $prefix_current, $prefix_new, $prefix_mode );
                    $prefix_preview_options  = $this->build_prefix_option_keys_preview( $prefix_current, $prefix_new );
                    $prefix_preview_usermeta = $this->build_prefix_usermeta_keys_preview( $prefix_current, $prefix_new );
                    $wp_config_info          = $this->get_wp_config_prefix_info( $prefix_new, $prefix_current );

                    if ( empty( $prefix_preview ) ) {
                        $prefix_notice = __( 'No tables would be affected by this prefix change.', 'aegisshield-security' );
                    }
                }
            } elseif (
                isset( $_POST['aegisshield_db_prefix_apply_nonce'] ) &&
                wp_verify_nonce(
                    sanitize_text_field( wp_unslash( $_POST['aegisshield_db_prefix_apply_nonce'] ) ),
                    'aegisshield_db_prefix_apply'
                )
            ) {
                $prefix_new  = isset( $_POST['new_prefix'] ) ? sanitize_text_field( wp_unslash( $_POST['new_prefix'] ) ) : '';
                $prefix_mode = isset( $_POST['prefix_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['prefix_mode'] ) ) : 'core';
                $exec_mode   = isset( $_POST['exec_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['exec_mode'] ) ) : 'standard';

                $backup_file = isset( $_POST['backup_file'] ) ? sanitize_text_field( wp_unslash( $_POST['backup_file'] ) ) : '';
                $confirm     = isset( $_POST['confirm_apply'] ) ? sanitize_text_field( wp_unslash( $_POST['confirm_apply'] ) ) : '';

                if ( '1' !== $confirm ) {
                    $prefix_notice = __( 'You must confirm you have a backup and understand the risk before applying.', 'aegisshield-security' );
                } elseif ( empty( $backup_file ) ) {
                    $prefix_notice = __( 'Please select a backup snapshot before applying.', 'aegisshield-security' );
                } else {


                $validation_error = $this->validate_new_prefix( $prefix_new, $prefix_current );
                if ( $validation_error ) {
                    $prefix_notice = $validation_error;
                } else {
                    if ( method_exists( $module, 'apply_prefix_change' ) ) {
                        $apply_result = $module->apply_prefix_change( $tables, $prefix_current, $prefix_new, $prefix_mode, $exec_mode, $backup_file );
                    } else {
                        $apply_result = array(
                            'success' => false,
                            'message' => __( 'Apply is not available in this build. Phase 1 is preview/backup only.', 'aegisshield-security' ),
                        );
                    }

                    if ( $apply_result['success'] ) {
                        $prefix_notice  = __( 'Database prefix change completed successfully.', 'aegisshield-security' );
                        $prefix_current = $prefix_new;
                    } else {
                        $prefix_notice = isset( $apply_result['message'] ) ? $apply_result['message'] : __( 'An error occurred while changing the prefix.', 'aegisshield-security' );
                    }
                }
                }

            }
            if ( method_exists( $module, 'list_db_backups' ) ) {
                $backups = $module->list_db_backups();
            }
        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'DB Tools', 'aegisshield-security' ); ?></h1>
			<p><?php esc_html_e( 'AegisShield DB Tools provides WordPress administrators with safe, practical control over database health, growth, and structure without introducing unnecessary risk. It combines visibility, automation, and safeguards—monitoring unusual table growth, performing non-aggressive optimizations, and managing database prefixes with backups, previews, and verification—so admins can prevent performance issues, detect early warning signs of misuse or abuse, and make informed database changes with confidence.' ); ?></p>
            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'overview', admin_url( 'admin.php?page=aegisshield-db-tools' ) ) ); ?>" class="nav-tab <?php echo ( 'overview' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Overview', 'aegisshield-security' ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'optimize', admin_url( 'admin.php?page=aegisshield-db-tools' ) ) ); ?>" class="nav-tab <?php echo ( 'optimize' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Optimize', 'aegisshield-security' ); ?>
					<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'prefix', admin_url( 'admin.php?page=aegisshield-db-tools' ) ) ); ?>" class="nav-tab <?php echo ( 'prefix' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'DB Prefix Manager', 'aegisshield-security' ); ?>
					<?php if ( ! $is_pro ) : ?>
						<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
							<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
						</span>
					<?php endif; ?>
                </a>
            </h2>

            <?php if ( 'prefix' === $current_tab && ! $is_pro ) : ?>
                <div class="notice notice-info">
                    <p><strong><?php esc_html_e( 'DB Prefix Manager is a Pro feature.', 'aegisshield-security' ); ?></strong>
                    <?php esc_html_e( 'Upgrade to AegisShield Pro to unlock advanced prefix hardening, prefix randomization, and guided rename tooling.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( $updated ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( $optimized && 'optimize' === $current_tab ) : ?>
                <div class="notice notice-info is-dismissible">
                    <p><?php esc_html_e( 'Database optimization completed.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( 'overview' === $current_tab ) : ?>

                <?php if ( ! empty( $warnings ) ) : ?>
                    <?php foreach ( $warnings as $warning ) : ?>
                        <div class="notice notice-warning is-dismissible">
                            <p><?php echo esc_html( $warning ); ?></p>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>

                <p>
                    <?php esc_html_e( 'Lightweight overview of your WordPress database tables, sizes, engines, and recent growth.', 'aegisshield-security' ); ?>
                </p>
				<?php $this->render_db_tools_dashboard_visuals(); ?>
                <h2><?php esc_html_e( 'Monitoring & Notifications', 'aegisshield-security' ); ?></h2>
                <form method="post">
                    <?php wp_nonce_field( 'aegisshield_db_tools_overview_settings', 'aegisshield_db_tools_overview_settings_nonce' ); ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="growth_monitoring"><?php esc_html_e( 'Monitor for unusual growth', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="growth_monitoring" name="growth_monitoring" <?php disabled( ! $is_pro ); ?> <?php checked( $growth_monitoring, 'on' ); ?> />
                                    <?php esc_html_e( 'Enable growth monitoring across key WordPress tables (posts, postmeta, logs, sessions, etc.).', 'aegisshield-security' ); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e( 'AegisShield tracks total size and row counts over time and flags sudden spikes that might indicate log floods, spam, or malware.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="growth_threshold_mb"><?php esc_html_e( 'Alert threshold (MB)', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <input type="number" id="growth_threshold_mb" name="growth_threshold_mb" value="<?php echo esc_attr( $growth_threshold_mb ); ?>" min="10" step="10" <?php disabled( ! $is_pro ); ?> />
                                <p class="description">
                                    <?php esc_html_e( 'If the combined size of monitored tables grows by more than this amount between snapshots, AegisShield will trigger an alert.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="growth_email"><?php esc_html_e( 'Primary alert email', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <input type="email" id="growth_email" name="growth_email" value="<?php echo esc_attr( $growth_email ); ?>" class="regular-text" <?php disabled( ! $is_pro ); ?> />
                                <p class="description">
                                    <?php esc_html_e( 'Alerts will be sent to this address when unusual growth is detected.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="growth_email_addresses"><?php esc_html_e( 'Additional recipients', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <p>
                                    <label for="growth_email_addresses">
                                        <?php esc_html_e( 'Additional email addresses (comma-separated)', 'aegisshield-security' ); ?>
                                    </label><br />
                                    <input type="text" id="growth_email_addresses" name="growth_email_addresses" class="regular-text" value="<?php echo esc_attr( $growth_email_addresses ); ?>" <?php disabled( ! $is_pro ); ?> />
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <td colspan="2">
                                <p class="description aegisshield-pro-notice">
                                    <?php esc_html_e( 'These database growth alerts require an active AegisShield Pro license. Upgrade to Pro to enable these options.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button( __( 'Save Monitoring Settings', 'aegisshield-security' ) ); ?>
                </form>

                <h2><?php esc_html_e( 'Table Summary', 'aegisshield-security' ); ?></h2>

                <form method="get" style="margin-bottom:1em;">
                    <input type="hidden" name="page" value="aegisshield-db-tools" />
                    <input type="hidden" name="tab" value="overview" />
                    <label>
                        <?php esc_html_e( 'Engine:', 'aegisshield-security' ); ?>
                        <select name="engine">
                            <option value=""><?php esc_html_e( 'All', 'aegisshield-security' ); ?></option>
                            <option value="innodb" <?php selected( $filter_engine, 'innodb' ); ?>>InnoDB</option>
                            <option value="myisam" <?php selected( $filter_engine, 'myisam' ); ?>>MyISAM</option>
                            <option value="memory" <?php selected( $filter_engine, 'memory' ); ?>>MEMORY</option>
                        </select>
                    </label>
                    &nbsp;
                    <label>
                        <?php esc_html_e( 'Size filter:', 'aegisshield-security' ); ?>
                        <select name="size">
                            <option value=""><?php esc_html_e( 'All', 'aegisshield-security' ); ?></option>
                            <option value="gt_10" <?php selected( $filter_size, 'gt_10' ); ?>>&gt; 10MB</option>
                            <option value="gt_50" <?php selected( $filter_size, 'gt_50' ); ?>>&gt; 50MB</option>
                            <option value="gt_100" <?php selected( $filter_size, 'gt_100' ); ?>>&gt; 100MB</option>
                        </select>
                    </label>
                    &nbsp;
                    <label>
                        <?php esc_html_e( 'Growth:', 'aegisshield-security' ); ?>
                        <select name="growth">
                            <option value=""><?php esc_html_e( 'All', 'aegisshield-security' ); ?></option>
                            <option value="spikes" <?php selected( $filter_growth, 'spikes' ); ?>><?php esc_html_e( 'Spikes only', 'aegisshield-security' ); ?></option>
                        </select>
                    </label>
                    &nbsp;
                    <label>
                        <?php esc_html_e( 'Search table:', 'aegisshield-security' ); ?>
                        <input type="search" name="search_table" value="<?php echo esc_attr( $search_name ); ?>" />
                    </label>
                    &nbsp;
                    <label>
                        <?php esc_html_e( 'Show:', 'aegisshield-security' ); ?>
                        <select name="as_db_per_page">
                            <?php foreach ( $allowed_per_page as $size ) : ?>
                                <option value="<?php echo esc_attr( $size ); ?>" <?php selected( $per_page, $size ); ?>>
                                    <?php echo esc_html( $size ); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </label>
                    &nbsp;
                    <?php submit_button( __( 'Filter', 'aegisshield-security' ), 'secondary', '', false ); ?>
                </form>

                <form method="post" style="margin-bottom:1em;">
                    <?php wp_nonce_field( 'aegisshield_db_tools_export_csv', 'aegisshield_db_tools_export_csv_nonce' ); ?>
                    <button type="submit" class="button"><?php esc_html_e( 'Export table summary as CSV', 'aegisshield-security' ); ?></button>
                </form>

                <?php if ( empty( $filtered_tables ) ) : ?>
                    <p><?php esc_html_e( 'No tables found for the current filters.', 'aegisshield-security' ); ?></p>
                <?php else : ?>
                    <?php
                    if ( isset( $total_pages ) && $total_pages > 1 ) :
                        $page_url   = admin_url( 'admin.php?page=aegisshield-db-tools' );
                        $first_page = 1;
                        $last_page  = $total_pages;
                        $prev_page  = max( $first_page, $current_page - 1 );
                        $next_page  = min( $last_page, $current_page + 1 );

                        $base_args = array(
                            'tab'            => 'overview',
                            'engine'         => $filter_engine,
                            'size'           => $filter_size,
                            'growth'         => $filter_growth,
                            'search_table'   => $search_name,
                            'as_db_per_page' => $per_page,
                        );
                    ?>
                    <div class="tablenav">
                        <div class="tablenav-pages">
                            <?php
                            printf(
                                '<span class="displaying-num">' . esc_html(
                                    _n(
                                        '%s table',
                                        '%s tables',
                                        $total_items,
                                        'aegisshield-security'
                                    )
                                ) . '</span>',
                                number_format_i18n( $total_items )
                            );
                            ?>
                            <span class="pagination-links">
                                <a class="first-page<?php echo ( $current_page === $first_page ) ? ' disabled' : ''; ?>"
                                   href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_db_paged' => $first_page ) ), $page_url ) ); ?>">
                                    <span class="screen-reader-text"><?php esc_html_e( 'First page', 'aegisshield-security' ); ?></span>
                                    <span aria-hidden="true">&laquo;</span>
                                </a>

                                <a class="prev-page<?php echo ( $current_page === $first_page ) ? ' disabled' : ''; ?>"
                                   href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_db_paged' => $prev_page ) ), $page_url ) ); ?>">
                                    <span class="screen-reader-text"><?php esc_html_e( 'Previous page', 'aegisshield-security' ); ?></span>
                                    <span aria-hidden="true">&lsaquo;</span>
                                </a>

                                <span class="paging-input">
                                    <?php
                                    printf(
                                        esc_html__( '%1$s of %2$s', 'aegisshield-security' ),
                                        '<span class="current-page">' . esc_html( $current_page ) . '</span>',
                                        '<span class="total-pages">' . esc_html( $total_pages ) . '</span>'
                                    );
                                    ?>
                                </span>

                                <a class="next-page<?php echo ( $current_page === $last_page ) ? ' disabled' : ''; ?>"
                                   href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_db_paged' => $next_page ) ), $page_url ) ); ?>">
                                    <span class="screen-reader-text"><?php esc_html_e( 'Next page', 'aegisshield-security' ); ?></span>
                                    <span aria-hidden="true">&rsaquo;</span>
                                </a>

                                <a class="last-page<?php echo ( $current_page === $last_page ) ? ' disabled' : ''; ?>"
                                   href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_db_paged' => $last_page ) ), $page_url ) ); ?>">
                                    <span class="screen-reader-text"><?php esc_html_e( 'Last page', 'aegisshield-security' ); ?></span>
                                    <span aria-hidden="true">&raquo;</span>
                                </a>
                            </span>
                        </div>
                    </div>
                    <?php endif; ?>

                    <table class="widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Table', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Engine', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Rows', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Data Size', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Index Size', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Total Size', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Change Since Last Snapshot', 'aegisshield-security' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $paged_tables as $table ) : ?>
                                <?php
                                $name        = isset( $table['name'] ) ? $table['name'] : '';
                                $engine      = isset( $table['engine'] ) ? $table['engine'] : '';
                                $rows        = isset( $table['rows'] ) ? (int) $table['rows'] : 0;
                                $data_bytes  = isset( $table['data_bytes'] ) ? (int) $table['data_bytes'] : 0;
                                $index_bytes = isset( $table['index_bytes'] ) ? (int) $table['index_bytes'] : 0;
                                $total_bytes = isset( $table['total_bytes'] ) ? (int) $table['total_bytes'] : 0;
                                $delta_bytes = isset( $table['delta_bytes'] ) ? (int) $table['delta_bytes'] : 0;
                                $delta_pct   = isset( $table['delta_pct'] ) ? (float) $table['delta_pct'] : 0.0;

                                $data_size  = size_format( $data_bytes );
                                $index_size = size_format( $index_bytes );
                                $total_size = size_format( $total_bytes );

                                if ( $delta_bytes > 0 ) {
                                    $delta_label = '+' . size_format( $delta_bytes );
                                } elseif ( $delta_bytes < 0 ) {
                                    $delta_label = '-' . size_format( abs( $delta_bytes ) );
                                } else {
                                    $delta_label = size_format( 0 );
                                }

                                if ( $delta_pct !== 0.0 ) {
                                    $delta_label .= sprintf( ' (%.1f%%)', $delta_pct * 100 );
                                }
                                ?>
                                <tr>
                                    <td><code><?php echo esc_html( $name ); ?></code></td>
                                    <td><?php echo esc_html( $engine ); ?></td>
                                    <td><?php echo esc_html( number_format_i18n( $rows ) ); ?></td>
                                    <td><?php echo esc_html( $data_size ); ?></td>
                                    <td><?php echo esc_html( $index_size ); ?></td>
                                    <td><?php echo esc_html( $total_size ); ?></td>
                                    <td><?php echo esc_html( $delta_label ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <?php if ( isset( $total_pages ) && $total_pages > 1 ) : ?>
                    <div class="tablenav">
                        <div class="tablenav-pages">
                            <?php
                            printf(
                                '<span class="displaying-num">' . esc_html(
                                    _n(
                                        '%s table',
                                        '%s tables',
                                        $total_items,
                                        'aegisshield-security'
                                    )
                                ) . '</span>',
                                number_format_i18n( $total_items )
                            );
                            ?>
                            <span class="pagination-links">
                                <a class="first-page<?php echo ( $current_page === $first_page ) ? ' disabled' : ''; ?>"
                                   href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_db_paged' => $first_page ) ), $page_url ) ); ?>">
                                    <span class="screen-reader-text"><?php esc_html_e( 'First page', 'aegisshield-security' ); ?></span>
                                    <span aria-hidden="true">&laquo;</span>
                                </a>

                                <a class="prev-page<?php echo ( $current_page === $first_page ) ? ' disabled' : ''; ?>"
                                   href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_db_paged' => $prev_page ) ), $page_url ) ); ?>">
                                    <span class="screen-reader-text"><?php esc_html_e( 'Previous page', 'aegisshield-security' ); ?></span>
                                    <span aria-hidden="true">&lsaquo;</span>
                                </a>

                                <span class="paging-input">
                                    <?php
                                    printf(
                                        esc_html__( '%1$s of %2$s', 'aegisshield-security' ),
                                        '<span class="current-page">' . esc_html( $current_page ) . '</span>',
                                        '<span class="total-pages">' . esc_html( $total_pages ) . '</span>'
                                    );
                                    ?>
                                </span>

                                <a class="next-page<?php echo ( $current_page === $last_page ) ? ' disabled' : ''; ?>"
                                   href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_db_paged' => $next_page ) ), $page_url ) ); ?>">
                                    <span class="screen-reader-text"><?php esc_html_e( 'Next page', 'aegisshield-security' ); ?></span>
                                    <span aria-hidden="true">&rsaquo;</span>
                                </a>

                                <a class="last-page<?php echo ( $current_page === $last_page ) ? ' disabled' : ''; ?>"
                                   href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_db_paged' => $last_page ) ), $page_url ) ); ?>">
                                    <span class="screen-reader-text"><?php esc_html_e( 'Last page', 'aegisshield-security' ); ?></span>
                                    <span aria-hidden="true">&raquo;</span>
                                </a>
                            </span>
                        </div>
                    </div>
                    <?php endif; ?>

                <?php endif; ?>

			<?php elseif ( 'optimize' === $current_tab ) : ?>

				<?php if ( ! $is_pro ) : ?>

					<?php $this->render_pro_locked_notice(); ?>

				<?php else : ?>

                <p>
                    <?php esc_html_e( 'Configure safe, non-aggressive database optimizations.', 'aegisshield-security' ); ?>
                </p>

                <h2><?php esc_html_e( 'Optimization Settings', 'aegisshield-security' ); ?></h2>
                <form method="post">
                    <?php wp_nonce_field( 'aegisshield_db_tools_settings_save', 'aegisshield_db_tools_settings_nonce' ); ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="weekly_optimize"><?php esc_html_e( 'Weekly optimization', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <label>
                                    <input type="checkbox" id="weekly_optimize" name="weekly_optimize" <?php disabled( ! $is_pro ); ?> <?php checked( $weekly_optimize, 'on' ); ?> />
                                    <?php esc_html_e( 'Run OPTIMIZE TABLE on WordPress tables at most once per week during AegisShield maintenance.', 'aegisshield-security' ); ?>
                                </label>
                                <p class="description">
                                    <?php esc_html_e( 'This uses safe OPTIMIZE TABLE routines on core tables only. It will not perform aggressive or dangerous changes.', 'aegisshield-security' ); ?>
                                </p>
                                <p class="description aegisshield-pro-notice">
                                    <?php esc_html_e( 'Scheduled optimization requires an active AegisShield Pro license. Upgrade to Pro to enable this automation.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button( __( 'Save Optimization Settings', 'aegisshield-security' ) ); ?>
                </form>
				<form method="post" style="margin-top:12px;">
					<?php wp_nonce_field( 'aegisshield_db_tools_purge_noise', 'aegisshield_db_tools_purge_noise_nonce' ); ?>
					<button type="submit" class="button button-secondary">
						<?php esc_html_e( 'Purge Noisy DB Tools Logs', 'aegisshield-security' ); ?>
					</button>
					<p class="description">
						<?php esc_html_e( 'Removes repetitive DB Tools heartbeat entries from the Activity Log.', 'aegisshield-security' ); ?>
					</p>
				</form>

                <h2><?php esc_html_e( 'Optimize Now', 'aegisshield-security' ); ?></h2>
                <form method="post">
                    <?php wp_nonce_field( 'aegisshield_db_tools_optimize_now', 'aegisshield_db_tools_optimize_now_nonce' ); ?>
                    <p>
                        <button type="submit" class="button button-primary"><?php esc_html_e( 'Run Optimization', 'aegisshield-security' ); ?></button>
                    </p>
                    <p class="description">
                        <?php esc_html_e( 'Runs OPTIMIZE TABLE on selected WordPress tables immediately. Best used during low-traffic windows.', 'aegisshield-security' ); ?>
                    </p>
                </form>
				
			<?php endif; ?>  <!-- closes Pro gating for Optimize -->

			<?php elseif ( 'prefix' === $current_tab ) : ?>

				<?php if ( ! $is_pro ) : ?>

					<?php $this->render_pro_locked_notice(); ?>

				<?php else : ?>

                <p>
                    <?php esc_html_e( 'Safely inspect and change your WordPress database table prefix.', 'aegisshield-security' ); ?>
                </p>

                <?php if ( $prefix_notice ) : ?>
                    <div class="notice notice-info is-dismissible">
                        <p><?php echo esc_html( $prefix_notice ); ?></p>
                    </div>
                <?php endif; ?>


                
                <?php if ( ! empty( $apply_result ) && ! empty( $apply_result['success'] ) ) : ?>
                    <div class="notice notice-success" style="margin-top:10px;">
                        <p><strong><?php esc_html_e( 'Prefix change applied.', 'aegisshield-security' ); ?></strong></p>
                        <?php if ( ! empty( $apply_result['manual_wpconfig'] ) ) : ?>
                            <p><?php esc_html_e( 'wp-config.php must contain this line:', 'aegisshield-security' ); ?></p>
                            <p><code><?php echo esc_html( $apply_result['manual_wpconfig'] ); ?></code></p>
                        <?php endif; ?>

                        <?php if ( ! empty( $apply_result['details']['wpconfig'] ) ) : ?>
                            <p>
                                <?php esc_html_e( 'wp-config.php status:', 'aegisshield-security' ); ?>
                                <code><?php echo esc_html( $apply_result['details']['wpconfig']['path'] ); ?></code>
                                — <?php echo ! empty( $apply_result['details']['wpconfig']['updated'] ) ? esc_html__( 'updated automatically', 'aegisshield-security' ) : esc_html__( 'manual update required', 'aegisshield-security' ); ?>
                            </p>
                        <?php endif; ?>

                        <?php if ( ! empty( $apply_result['details'] ) ) : ?>
                            <ul style="margin-left: 1.2em; list-style: disc;">
                                <li><?php esc_html_e( 'Tables renamed:', 'aegisshield-security' ); ?> <strong><?php echo isset( $apply_result['details']['renamed_tables'] ) ? (int) count( $apply_result['details']['renamed_tables'] ) : 0; ?></strong></li>
                                <li><?php esc_html_e( 'Options keys updated:', 'aegisshield-security' ); ?> <strong><?php echo isset( $apply_result['details']['options_rows'] ) ? (int) $apply_result['details']['options_rows'] : 0; ?></strong></li>
                                <li><?php esc_html_e( 'Usermeta keys updated:', 'aegisshield-security' ); ?> <strong><?php echo isset( $apply_result['details']['usermeta_rows'] ) ? (int) $apply_result['details']['usermeta_rows'] : 0; ?></strong></li>
                                <li><?php esc_html_e( 'Backup used:', 'aegisshield-security' ); ?> <strong><?php echo isset( $apply_result['details']['backup_used'] ) ? esc_html( $apply_result['details']['backup_used'] ) : ''; ?></strong></li>
                            </ul>
                        <?php endif; ?>

                        <p><strong><?php esc_html_e( 'Do not leave this page until you can reload wp-admin successfully.', 'aegisshield-security' ); ?></strong></p>
                    </div>
                <?php endif; ?>

<h2><?php esc_html_e( 'Database Backup & Restore', 'aegisshield-security' ); ?></h2>

                <?php if ( $backup_notice ) : ?>
                    <div class="notice notice-info is-dismissible">
                        <p><?php echo esc_html( $backup_notice ); ?></p>
                    </div>
                <?php endif; ?>

                <form method="post" style="margin-bottom: 1em;">
                    <?php wp_nonce_field( 'aegisshield_db_backup_create', 'aegisshield_db_backup_create_nonce' ); ?>
                    <p>
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e( 'Create Backup Snapshot', 'aegisshield-security' ); ?>
                        </button>
                        <span class="description" style="margin-left:10px;">
                            <?php esc_html_e( 'Creates a timestamped SQL export file (no database changes). Recommended before any prefix change.', 'aegisshield-security' ); ?>
                        </span>
                    </p>
                </form>

                <?php if ( ! empty( $backups ) ) : ?>
                    <h3 style="margin-top:0;"><?php esc_html_e( 'Available Backups', 'aegisshield-security' ); ?></h3>
                    <table class="widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Date', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'File', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Size', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Location', 'aegisshield-security' ); ?></th>
                                <<th><?php esc_html_e( 'Actions', 'aegisshield-security' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $backups as $b ) : ?>
                                <tr>
                                    <td><?php echo esc_html( isset( $b['date'] ) ? $b['date'] : '' ); ?></td>
                                    <td><code><?php echo esc_html( isset( $b['file'] ) ? $b['file'] : '' ); ?></code></td>
                                    <td><?php echo esc_html( isset( $b['size'] ) ? $b['size'] : '' ); ?></td>
                                    <td><code><?php echo esc_html( isset( $b['path'] ) ? $b['path'] : '' ); ?></code></td>
									<td>
										<?php
											$file = isset( $b['file'] ) ? (string) $b['file'] : '';

											if ( '' === $file ) {
												echo '&mdash;';
											} else {

												$download_url = wp_nonce_url(
													add_query_arg(
														array(
															'tab'                => 'prefix',
															'as_db_backup_action' => 'download',
															'backup_file'         => $file,
														)
													),
													'aegisshield_db_backup_download_' . $file
												);

												$delete_url = wp_nonce_url(
													add_query_arg(
														array(
															'tab'                => 'prefix',
															'as_db_backup_action' => 'delete',
															'backup_file'         => $file,
														)
													),
													'aegisshield_db_backup_delete_' . $file
												);

												$restore_url = wp_nonce_url(
													add_query_arg(
														array(
															'tab'                => 'prefix',
															'as_db_backup_action' => 'restore',
															'backup_file'         => $file,
														)
													),
													'aegisshield_db_backup_restore_' . $file
												);

												echo '<a href="' . esc_url( $download_url ) . '">' . esc_html__( 'Download', 'aegisshield-security' ) . '</a>';
												echo ' | ';
												echo '<a href="' . esc_url( $delete_url ) . '" onclick="return confirm(\'' . esc_js( __( 'Delete this backup snapshot? This cannot be undone.', 'aegisshield-security' ) ) . '\');">' . esc_html__( 'Delete', 'aegisshield-security' ) . '</a>';
												echo ' | ';
												echo '<a href="' . esc_url( $restore_url ) . '" onclick="return confirm(\'' . esc_js( __( 'Restore this backup into the database now? This will overwrite current database state.', 'aegisshield-security' ) ) . '\');">' . esc_html__( 'Restore', 'aegisshield-security' ) . '</a>';
											}
										?>
									</td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <p class="description" style="margin-top:6px;">
                        <?php esc_html_e( 'Restore is intentionally disabled in Phase 1. We will wire restore with strict safety checks in Phase 3.', 'aegisshield-security' ); ?>
                    </p>
                <?php else : ?>
                    <p class="description">
                        <?php esc_html_e( 'No backups found yet. Create a snapshot before attempting any prefix change.', 'aegisshield-security' ); ?>
                    </p>
                <?php endif; ?>

                <h2><?php esc_html_e( 'Current Prefix & Preview', 'aegisshield-security' ); ?></h2>
                <form method="post">
                    <?php wp_nonce_field( 'aegisshield_db_prefix_preview', 'aegisshield_db_prefix_preview_nonce' ); ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label><?php esc_html_e( 'Current prefix', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <code><?php echo esc_html( $prefix_current ); ?></code>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="new_prefix"><?php esc_html_e( 'New prefix', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <input type="text" id="new_prefix" name="new_prefix" value="<?php echo esc_attr( $prefix_new ); ?>" class="regular-text" />
                                <p class="description">
                                    <?php esc_html_e( 'Use only letters, numbers, and underscores. A trailing underscore is recommended (e.g. wpsecure_).', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="prefix_mode"><?php esc_html_e( 'Scope', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
								<select id="prefix_mode" name="prefix_mode">
									<option value="preview" <?php selected( $prefix_mode, 'preview' ); ?>>
										<?php esc_html_e( 'Preview only (no changes)', 'aegisshield-security' ); ?>
									</option>
									<option value="all" <?php selected( $prefix_mode, 'all' ); ?>>
										<?php esc_html_e( 'All matching tables (recommended)', 'aegisshield-security' ); ?>
									</option>
								</select>
                                <p class="description">
                                    <?php esc_html_e( 'Core mode affects only standard WordPress tables. All matching tables will attempt to rename any table that starts with the current prefix.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="exec_mode"><?php esc_html_e( 'Execution mode', 'aegisshield-security' ); ?></label>
                            </th>
                            <td>
                                <select id="exec_mode" name="exec_mode">
                                    <option value="standard" <?php selected( $exec_mode, 'standard' ); ?>><?php esc_html_e( 'Standard (one pass)', 'aegisshield-security' ); ?></option>
                                    <option value="dry_run" <?php selected( $exec_mode, 'dry_run' ); ?>><?php esc_html_e( 'Dry run only (no changes)', 'aegisshield-security' ); ?></option>
                                </select>
                                <p class="description">
                                    <?php esc_html_e( 'Use dry run first to see exactly which tables would be renamed before applying changes.', 'aegisshield-security' ); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button( __( 'Preview Changes', 'aegisshield-security' ) ); ?>
                </form>

                <?php if ( ! empty( $prefix_preview ) ) : ?>
                    <h2><?php esc_html_e( 'Preview of Affected Tables', 'aegisshield-security' ); ?></h2>
                    <table class="widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Current Name', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'New Name', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Type', 'aegisshield-security' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $prefix_preview as $row ) : ?>
                                <tr>
                                    <td><code><?php echo esc_html( $row['current'] ); ?></code></td>
                                    <td><code><?php echo esc_html( $row['new'] ); ?></code></td>
                                    <td><?php echo esc_html( $row['type'] ); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <h2 style="margin-top:1.2em;"><?php esc_html_e( 'Preview of Prefix-Bound Option Keys', 'aegisshield-security' ); ?></h2>
                    <p class="description"><?php esc_html_e( 'These option_name values must be updated to match the new prefix (includes user roles and plugin options).', 'aegisshield-security' ); ?></p>
                    <?php if ( ! empty( $prefix_preview_options ) ) : ?>
                        <table class="widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e( 'Current option_name', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'New option_name', 'aegisshield-security' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $prefix_preview_options as $row ) : ?>
                                    <tr>
                                        <td><code><?php echo esc_html( $row['current'] ); ?></code></td>
                                        <td><code><?php echo esc_html( $row['new'] ); ?></code></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else : ?>
                        <p class="description"><?php esc_html_e( 'No prefixed option keys were detected (this is unusual). Proceed with caution and verify wp_user_roles / *_user_roles exists.', 'aegisshield-security' ); ?></p>
                    <?php endif; ?>

                    <h2 style="margin-top:1.2em;"><?php esc_html_e( 'Preview of Prefix-Bound Usermeta Keys', 'aegisshield-security' ); ?></h2>
                    <p class="description"><?php esc_html_e( 'These meta_key values MUST be updated (capabilities/user_level). Missing this step can lock you out of wp-admin.', 'aegisshield-security' ); ?></p>
                    <?php if ( ! empty( $prefix_preview_usermeta ) ) : ?>
                        <table class="widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e( 'Current meta_key', 'aegisshield-security' ); ?></th>
                                    <th><?php esc_html_e( 'New meta_key', 'aegisshield-security' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $prefix_preview_usermeta as $row ) : ?>
                                    <tr>
                                        <td><code><?php echo esc_html( $row['current'] ); ?></code></td>
                                        <td><code><?php echo esc_html( $row['new'] ); ?></code></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else : ?>
                        <p class="description"><?php esc_html_e( 'No prefixed usermeta keys were detected (this is unusual). Verify the current prefix is correct and that capabilities/user_level keys exist.', 'aegisshield-security' ); ?></p>
                    <?php endif; ?>

                    <h2 style="margin-top:1.2em;"><?php esc_html_e( 'wp-config.php Readiness', 'aegisshield-security' ); ?></h2>
                    <?php if ( ! empty( $wp_config_info ) ) : ?>
                        <table class="widefat fixed striped">
                            <tbody>
                                <tr>
                                    <th style="width:240px;"><?php esc_html_e( 'Config file', 'aegisshield-security' ); ?></th>
                                    <td><code><?php echo esc_html( isset( $wp_config_info['path'] ) ? $wp_config_info['path'] : '' ); ?></code></td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e( 'Writable', 'aegisshield-security' ); ?></th>
                                    <td><?php echo esc_html( ! empty( $wp_config_info['writable'] ) ? __( 'Yes', 'aegisshield-security' ) : __( 'No', 'aegisshield-security' ) ); ?></td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e( 'Current $table_prefix', 'aegisshield-security' ); ?></th>
                                    <td><code><?php echo esc_html( isset( $wp_config_info['current'] ) ? $wp_config_info['current'] : '' ); ?></code></td>
                                </tr>
                                <tr>
                                    <th><?php esc_html_e( 'Planned $table_prefix', 'aegisshield-security' ); ?></th>
                                    <td><code><?php echo esc_html( isset( $wp_config_info['planned'] ) ? $wp_config_info['planned'] : '' ); ?></code></td>
                                </tr>
                                <?php if ( ! empty( $wp_config_info['manual_line'] ) ) : ?>
                                    <tr>
                                        <th><?php esc_html_e( 'Manual line (if not writable)', 'aegisshield-security' ); ?></th>
                                        <td><code><?php echo esc_html( $wp_config_info['manual_line'] ); ?></code></td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    <?php else : ?>
                        <p class="description"><?php esc_html_e( 'wp-config.php info was not detected.', 'aegisshield-security' ); ?></p>
                    <?php endif; ?>

                    <form method="post" style="margin-top:1em;">
                        <?php wp_nonce_field( 'aegisshield_db_prefix_apply', 'aegisshield_db_prefix_apply_nonce' ); ?>
                        <input type="hidden" name="new_prefix" value="<?php echo esc_attr( $prefix_new ); ?>" />
                        <input type="hidden" name="prefix_mode" value="<?php echo esc_attr( $prefix_mode ); ?>" />
                        <input type="hidden" name="exec_mode" value="<?php echo esc_attr( $exec_mode ); ?>" />

                        <table class="form-table" role="presentation">
                            <tr>
                                <th scope="row">
                                    <label for="as_dbtools_backup_file"><?php esc_html_e( 'Backup to use (required)', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <select id="as_dbtools_backup_file" name="backup_file">
                                        <option value=""><?php esc_html_e( '-- Select a backup snapshot --', 'aegisshield-security' ); ?></option>
                                        <?php if ( ! empty( $backups ) ) : ?>
                                            <?php foreach ( $backups as $b ) : ?>
                                                <option value="<?php echo esc_attr( $b['file'] ); ?>">
                                                    <?php echo esc_html( $b['file'] . ' (' . $b['date_utc'] . ', ' . $b['size_human'] . ')' ); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                    <p class="description">
                                        <?php esc_html_e( 'Select a backup snapshot created before applying. If something goes wrong, this is your fastest recovery path.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row"><?php esc_html_e( 'Confirmation', 'aegisshield-security' ); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="confirm_apply" value="1" />
                                        <?php esc_html_e( 'I understand this can break my site if wp-config.php is not updated. I have selected a backup and I am ready to apply.', 'aegisshield-security' ); ?>
                                    </label>
                                    <p class="description">
                                        <?php esc_html_e( 'After apply, WordPress will only work when wp-config.php $table_prefix matches the new prefix.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>

                        <p>
                            <button type="submit" class="button button-primary">
                                <?php esc_html_e( 'Apply Prefix Change (Phase 2)', 'aegisshield-security' ); ?>
                            </button>
                        </p>
                        <hr style="margin:20px 0;" />

                        <h3><?php esc_html_e( 'Phase 3A — Verification Scan (no writes)', 'aegisshield-security' ); ?></h3>
                        <p class="description">
                            <?php esc_html_e( 'Runs a read-only scan to confirm: required core tables exist under the expected prefix, no critical tables remain on the old prefix, and no old prefix-bound option/usermeta keys remain.', 'aegisshield-security' ); ?>
                        </p>

                        <form method="post" style="margin-top:10px;">
                            <?php wp_nonce_field( 'aegisshield_db_prefix_verify', 'aegisshield_db_prefix_verify_nonce' ); ?>
                            <p>
                                <button type="submit" class="button">
                                    <?php esc_html_e( 'Run Verification Scan (Phase 3A)', 'aegisshield-security' ); ?>
                                </button>
                            </p>
                        </form>

                        <?php if ( ! empty( $prefix_verify_notice ) ) : ?>
                            <div class="notice notice-info inline" style="margin-top:10px;">
                                <p><?php echo esc_html( $prefix_verify_notice ); ?></p>
                            </div>
                        <?php endif; ?>

                        <?php if ( ! empty( $prefix_verify_result ) && is_array( $prefix_verify_result ) ) : ?>
                            <div style="margin-top:10px;">
                                <?php if ( ! empty( $prefix_verify_result['issues'] ) ) : ?>
                                    <div class="notice notice-error inline">
                                        <p><strong><?php esc_html_e( 'Blocking issues', 'aegisshield-security' ); ?></strong></p>
                                        <ul>
                                            <?php foreach ( $prefix_verify_result['issues'] as $msg ) : ?>
                                                <li><?php echo esc_html( $msg ); ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>

                                <?php if ( ! empty( $prefix_verify_result['warnings'] ) ) : ?>
                                    <div class="notice notice-warning inline">
                                        <p><strong><?php esc_html_e( 'Warnings', 'aegisshield-security' ); ?></strong></p>
                                        <ul>
                                            <?php foreach ( $prefix_verify_result['warnings'] as $msg ) : ?>
                                                <li><?php echo esc_html( $msg ); ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>

                                <?php if ( ! empty( $prefix_verify_result['samples'] ) ) : ?>
                                    <details style="margin-top:10px;">
                                        <summary><?php esc_html_e( 'View samples', 'aegisshield-security' ); ?></summary>
                                        <pre style="background:#fff;border:1px solid #ccd0d4;padding:10px;overflow:auto;"><?php echo esc_html( wp_json_encode( $prefix_verify_result['samples'], JSON_PRETTY_PRINT ) ); ?></pre>
                                    </details>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>


                        <div class="notice notice-warning inline" style="margin-top:10px;">
                            <p><strong><?php esc_html_e( 'High Risk Operation', 'aegisshield-security' ); ?></strong> — <?php esc_html_e( 'Do not navigate away until you verify wp-config.php and can load wp-admin again.', 'aegisshield-security' ); ?></p>
                        </div>
                    </form>
                <?php endif; ?>
	
			<?php endif; ?> 
			<?php endif; ?>
        </div>
        <?php
    }

    protected function validate_new_prefix( $new_prefix, $current_prefix ) {
        $new_prefix = (string) $new_prefix;

        if ( $new_prefix === $current_prefix ) {
            return __( 'The new prefix is the same as the current prefix.', 'aegisshield-security' );
        }

        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $new_prefix ) ) {
            return __( 'The new prefix must contain only letters, numbers, and underscores.', 'aegisshield-security' );
        }

        if ( '_' !== substr( $new_prefix, -1 ) ) {
            return __( 'It is recommended that the prefix end with an underscore (e.g. wpsecure_).', 'aegisshield-security' );
        }

        return '';
    }

    protected function build_prefix_preview( $tables, $current_prefix, $new_prefix, $mode ) {
        $preview = array();

        foreach ( $tables as $table ) {
            $name = isset( $table['name'] ) ? $table['name'] : '';

            if ( strpos( $name, $current_prefix ) !== 0 ) {
                continue;
            }

            if ( 'core' === $mode ) {
                $core_patterns = array(
                    'options',
                    'posts',
                    'postmeta',
                    'users',
                    'usermeta',
                    'term',
                    'comment',
                    'links',
                    'blogs',
                    'blogmeta',
                    'sitemeta',
                );

                $suffix       = substr( $name, strlen( $current_prefix ) );
                $matched_core = false;
                foreach ( $core_patterns as $pattern ) {
                    if ( 0 === strpos( $suffix, $pattern ) ) {
                        $matched_core = true;
                        break;
                    }
                }

                if ( ! $matched_core ) {
                    continue;
                }
            }

            $new_name = $new_prefix . substr( $name, strlen( $current_prefix ) );

            $preview[] = array(
                'current' => $name,
                'new'     => $new_name,
                'type'    => ( 'core' === $mode ) ? __( 'Core', 'aegisshield-security' ) : __( 'All', 'aegisshield-security' ),
            );
        }

        return $preview;
    }

    protected function build_prefix_option_keys_preview( $current_prefix, $new_prefix ) {
        global $wpdb;

        $rows = array();

        $options_table = $current_prefix . 'options';
        $exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $options_table ) );
        if ( $exists !== $options_table ) {
            return $rows;
        }

        $like = $wpdb->esc_like( $current_prefix ) . '%';

        $names = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM {$options_table} WHERE option_name LIKE %s ORDER BY option_name ASC",
                $like
            )
        );

        if ( empty( $names ) ) {
            return $rows;
        }

        foreach ( $names as $name ) {
            $name = (string) $name;
            if ( strpos( $name, $current_prefix ) !== 0 ) {
                continue;
            }
            $rows[] = array(
                'current' => $name,
                'new'     => $new_prefix . substr( $name, strlen( $current_prefix ) ),
            );
        }

        $user_roles_key = $current_prefix . 'user_roles';
        if ( in_array( $user_roles_key, $names, true ) ) {
        }

        return $rows;
    }

    protected function build_prefix_usermeta_keys_preview( $current_prefix, $new_prefix ) {
        global $wpdb;

        $rows = array();

        $usermeta_table = $current_prefix . 'usermeta';
        $exists         = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $usermeta_table ) );
        if ( $exists !== $usermeta_table ) {
            return $rows;
        }

        $like = $wpdb->esc_like( $current_prefix ) . '%';

        $keys = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT DISTINCT meta_key FROM {$usermeta_table} WHERE meta_key LIKE %s ORDER BY meta_key ASC",
                $like
            )
        );

        if ( empty( $keys ) ) {
            return $rows;
        }

        foreach ( $keys as $key ) {
            $key = (string) $key;
            if ( strpos( $key, $current_prefix ) !== 0 ) {
                continue;
            }
            $rows[] = array(
                'current' => $key,
                'new'     => $new_prefix . substr( $key, strlen( $current_prefix ) ),
            );
        }

        return $rows;
    }

    protected function get_wp_config_prefix_info( $planned_prefix, $current_prefix ) {
        $info = array(
            'path'        => '',
            'writable'    => false,
            'current'     => (string) $current_prefix,
            'planned'     => (string) $planned_prefix,
            'manual_line' => '',
        );

        if ( defined( 'ABSPATH' ) ) {
            $path = ABSPATH . 'wp-config.php';
            if ( ! file_exists( $path ) ) {
                // One-level up fallback (common on some hosts).
                $alt = dirname( ABSPATH ) . '/wp-config.php';
                if ( file_exists( $alt ) ) {
                    $path = $alt;
                }
            }

            if ( file_exists( $path ) ) {
                $info['path']     = $path;
                $info['writable'] = is_writable( $path );
                $info['manual_line'] = '$table_prefix = \''
                    . str_replace( "'", "\\'", (string) $planned_prefix )
                    . "\\';";
            }
        }

        return $info;
    }

	protected function render_db_tools_dashboard_visuals() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$data = $this->get_db_tools_dashboard_data();

		$plugin_file = defined( 'AEGISSHIELD_PLUGIN_FILE' )
			? AEGISSHIELD_PLUGIN_FILE
			: dirname( __FILE__, 3 ) . '/aegisshield-security.php';

		$chart_src = plugins_url( 'assets/js/chart.umd.min.js', $plugin_file );
		?>
		<style>
			.aegis-db-dash{
				display:grid;
				grid-template-columns:repeat(4,minmax(0,1fr));
				gap:12px;
				margin:12px 0 14px;
			}
			.aegis-db-card{
				background:#fff;
				border:1px solid #dcdcde;
				border-radius:10px;
				padding:12px 12px 10px;
				box-shadow:0 1px 1px rgba(0,0,0,.04);
				min-height:235px;
			}
			.aegis-db-card h3{
				margin:0 0 8px;
				font-size:13px;
				line-height:1.3;
				display:flex;
				align-items:center;
				gap:8px;
			}
			.aegis-db-pill{
				font-size:12px;
				padding:2px 8px;
				border-radius:999px;
				border:1px solid #dcdcde;
				background:#f6f7f7;
			}
			.aegis-db-canvas{ position:relative; height:175px; }
			.aegis-db-sub{
				margin-top:8px;
				font-size:12px;
				color:#646970;
				display:flex;
				justify-content:space-between;
				gap:10px;
			}
			@media (max-width:1200px){ .aegis-db-dash{ grid-template-columns:repeat(2,minmax(0,1fr)); } }
			@media (max-width:782px){ .aegis-db-dash{ grid-template-columns:1fr; } }
		</style>

		<div class="aegis-db-dash" id="aegis-db-dash"
			 data-db="<?php echo esc_attr( wp_json_encode( $data ) ); ?>">

			<div class="aegis-db-card">
				<h3>
					<span class="dashicons dashicons-database"></span>
					<?php esc_html_e( 'DB Size Breakdown', 'aegisshield-security' ); ?>
					<span class="aegis-db-pill"><?php echo esc_html( $data['totals']['total_mb'] ); ?> MB</span>
				</h3>
				<div class="aegis-db-canvas"><canvas id="aegisDbChartBreakdown"></canvas></div>
				<div class="aegis-db-sub">
					<span><?php esc_html_e( 'Data vs Index', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( $data['totals']['tables_count'] ); ?> <?php esc_html_e( 'tables', 'aegisshield-security' ); ?></span>
				</div>
			</div>

			<div class="aegis-db-card">
				<h3>
					<span class="dashicons dashicons-chart-bar"></span>
					<?php esc_html_e( 'Largest Tables', 'aegisshield-security' ); ?>
					<span class="aegis-db-pill"><?php echo esc_html( $data['largest']['count'] ); ?></span>
				</h3>
				<div class="aegis-db-canvas"><canvas id="aegisDbChartLargest"></canvas></div>
				<div class="aegis-db-sub">
					<span><?php esc_html_e( 'Top by total size (MB)', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( $data['largest']['top_mb'] ); ?> MB</span>
				</div>
			</div>

			<div class="aegis-db-card">
				<h3>
					<span class="dashicons dashicons-controls-repeat"></span>
					<?php esc_html_e( 'Fastest Growth', 'aegisshield-security' ); ?>
					<span class="aegis-db-pill"><?php echo esc_html( $data['growth']['count'] ); ?></span>
				</h3>
				<div class="aegis-db-canvas"><canvas id="aegisDbChartGrowth"></canvas></div>
				<div class="aegis-db-sub">
					<span><?php esc_html_e( 'Since last snapshot (MB)', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( $data['growth']['total_delta_mb'] ); ?> MB</span>
				</div>
			</div>

			<div class="aegis-db-card">
				<h3>
					<span class="dashicons dashicons-admin-tools"></span>
					<?php esc_html_e( 'Storage Engine by Size', 'aegisshield-security' ); ?>
					<span class="aegis-db-pill"><?php echo esc_html( $data['engines']['top'] ); ?></span>
				</h3>
				<div class="aegis-db-canvas"><canvas id="aegisDbChartEngines"></canvas></div>
				<div class="aegis-db-sub">
					<span><?php esc_html_e( 'InnoDB / MyISAM / Other', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( $data['engines']['engine_count'] ); ?> <?php esc_html_e( 'types', 'aegisshield-security' ); ?></span>
				</div>
			</div>

		</div>

		<script>
		(function(){
			// Load Chart.js once (shared across admin pages).
			if (typeof window.__aegisChartLoaded === 'undefined') {
				window.__aegisChartLoaded = true;
				var s=document.createElement('script');
				s.src=<?php echo wp_json_encode( $chart_src ); ?>;
				s.onload=initDbCharts;
				document.head.appendChild(s);
			} else {
				initDbCharts();
			}

			function initDbCharts() {
				var wrap=document.getElementById('aegis-db-dash');
				if(!wrap) return;
				var raw=wrap.getAttribute('data-db');
				if(!raw) return;

				var data;
				try { data=JSON.parse(raw); } catch(e){ return; }
				if(typeof Chart==='undefined') return;

				function ctx(id){ var el=document.getElementById(id); return el?el.getContext('2d'):null; }

				// 1) Breakdown (doughnut): Data vs Index
				var c1=ctx('aegisDbChartBreakdown');
				if(c1 && data.totals){
					new Chart(c1,{
						type:'doughnut',
						data:{
							labels:['Data','Index'],
							datasets:[{ data:[data.totals.data_mb, data.totals.index_mb] }]
						},
						options:{ responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:'bottom' } }, cutout:'65%' }
					});
				}

				// 2) Largest tables (bar)
				var c2=ctx('aegisDbChartLargest');
				if(c2 && data.largest && data.largest.labels){
					new Chart(c2,{
						type:'bar',
						data:{ labels:data.largest.labels, datasets:[{ label:'MB', data:data.largest.values }] },
						options:{
							responsive:true, maintainAspectRatio:false,
							plugins:{ legend:{ display:false } },
							scales:{ x:{ grid:{ display:false } }, y:{ beginAtZero:true, ticks:{ precision:0 } } }
						}
					});
				}

				// 3) Growth (horizontal bar): delta MB since last snapshot
				var c3=ctx('aegisDbChartGrowth');
				if(c3 && data.growth && data.growth.labels){
					new Chart(c3,{
						type:'bar',
						data:{ labels:data.growth.labels, datasets:[{ label:'MB delta', data:data.growth.values }] },
						options:{
							indexAxis:'y',
							responsive:true, maintainAspectRatio:false,
							plugins:{ legend:{ display:false } },
							scales:{ x:{ beginAtZero:true, ticks:{ precision:0 } }, y:{ grid:{ display:false } } }
						}
					});
				}

				// 4) Engines by size (doughnut)
				var c4=ctx('aegisDbChartEngines');
				if(c4 && data.engines && data.engines.labels){
					new Chart(c4,{
						type:'doughnut',
						data:{ labels:data.engines.labels, datasets:[{ data:data.engines.values }] },
						options:{ responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:'bottom' } }, cutout:'65%' }
					});
				}
			}
		})();
		</script>
		<?php
	}

	protected function get_db_tools_dashboard_data() {
		$module  = new \AegisShield\Modules\AS_Module_DB_Tools( $this->plugin );
		$summary = $module->get_table_summary();

		$tables = isset( $summary['tables'] ) && is_array( $summary['tables'] ) ? $summary['tables'] : array();

		$data_bytes  = 0;
		$index_bytes = 0;
		$total_bytes = 0;

		$engine_bytes = array(); // engine => bytes
		foreach ( $tables as $t ) {
			if ( ! is_array( $t ) ) { continue; }
			$data_bytes  += (int) ( $t['data_bytes']  ?? 0 );
			$index_bytes += (int) ( $t['index_bytes'] ?? 0 );
			$total_bytes += (int) ( $t['total_bytes'] ?? 0 );

			$eng = (string) ( $t['engine'] ?? 'Unknown' );
			$eng = $eng !== '' ? $eng : 'Unknown';
			if ( ! isset( $engine_bytes[ $eng ] ) ) { $engine_bytes[ $eng ] = 0; }
			$engine_bytes[ $eng ] += (int) ( $t['total_bytes'] ?? 0 );
		}

		$to_mb = function( $bytes ) {
			return (int) round( ( (int) $bytes ) / 1024 / 1024 );
		};

		// Largest tables by total_bytes
		$largest = $tables;
		usort( $largest, function( $a, $b ) {
			return ( (int) ( $b['total_bytes'] ?? 0 ) ) <=> ( (int) ( $a['total_bytes'] ?? 0 ) );
		} );
		$largest = array_slice( $largest, 0, 8 );

		$largest_labels = array();
		$largest_values = array();
		$largest_top_mb = 0;
		foreach ( $largest as $t ) {
			$name = (string) ( $t['name'] ?? '' );
			$mb   = $to_mb( (int) ( $t['total_bytes'] ?? 0 ) );
			$largest_labels[] = $name;
			$largest_values[] = $mb;
			if ( $mb > $largest_top_mb ) { $largest_top_mb = $mb; }
		}

		// Fastest growing tables since last snapshot (delta_bytes)
		$growth = $tables;
		usort( $growth, function( $a, $b ) {
			return ( (int) ( $b['delta_bytes'] ?? 0 ) ) <=> ( (int) ( $a['delta_bytes'] ?? 0 ) );
		} );

		// Only keep positive deltas
		$growth = array_values( array_filter( $growth, function( $t ) {
			return (int) ( $t['delta_bytes'] ?? 0 ) > 0;
		} ) );
		$growth = array_slice( $growth, 0, 8 );

		$growth_labels = array();
		$growth_values = array();
		$growth_total_delta_mb = 0;

		foreach ( $growth as $t ) {
			$name = (string) ( $t['name'] ?? '' );
			$mb   = $to_mb( (int) ( $t['delta_bytes'] ?? 0 ) );
			$growth_labels[] = $name;
			$growth_values[] = $mb;
			$growth_total_delta_mb += $mb;
		}

		// Engines (group small ones into "Other" for cleaner donut)
		arsort( $engine_bytes );
		$engine_labels = array();
		$engine_values = array();

		$top_engines = array_slice( $engine_bytes, 0, 4, true );
		$other_bytes = 0;
		foreach ( $engine_bytes as $eng => $b ) {
			if ( isset( $top_engines[ $eng ] ) ) { continue; }
			$other_bytes += (int) $b;
		}

		foreach ( $top_engines as $eng => $b ) {
			$engine_labels[] = $eng;
			$engine_values[] = $to_mb( $b );
		}
		if ( $other_bytes > 0 ) {
			$engine_labels[] = 'Other';
			$engine_values[] = $to_mb( $other_bytes );
		}

		$top_engine_label = '';
		if ( ! empty( $engine_labels ) ) {
			$top_engine_label = (string) $engine_labels[0];
		}

		return array(
			'totals' => array(
				'tables_count' => (int) count( $tables ),
				'data_mb'      => $to_mb( $data_bytes ),
				'index_mb'     => $to_mb( $index_bytes ),
				'total_mb'     => $to_mb( $total_bytes ),
			),
			'largest' => array(
				'labels' => $largest_labels,
				'values' => $largest_values,
				'count'  => (int) count( $largest_labels ),
				'top_mb' => (string) $largest_top_mb,
			),
			'growth' => array(
				'labels'         => $growth_labels,
				'values'         => $growth_values,
				'count'          => (int) count( $growth_labels ),
				'total_delta_mb' => (string) $growth_total_delta_mb,
			),
			'engines' => array(
				'labels'       => $engine_labels,
				'values'       => $engine_values,
				'engine_count' => (int) count( $engine_bytes ),
				'top'          => $top_engine_label,
			),
		);
	}

}
