<?php
namespace AegisShield\Admin_Pages\Hardening;

use AegisShield\AS_Plugin;
use AegisShield\Modules\AS_Module_Hardening;

defined( 'ABSPATH' ) || exit;

class AS_Page_Hardening_Tab_Vuln {

    public static function render_tab( AS_Plugin $plugin ) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $settings = $plugin->get_settings();
        $section  = 'hardening';

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        $vuln_source   = (string) $settings->get( $section, 'hardening_vuln_source', 'local' );
        $patchstack_key = (string) $settings->get( $section, 'hardening_vuln_patchstack_key', '' );
        $last_run      = (int) $settings->get( $section, 'hardening_vuln_last_run', 0 );
        $results       = $settings->get( $section, 'hardening_vuln_results', array() );

        if ( ! is_array( $results ) ) {
            $results = array();
        }

        $scan_ran = false;

        if (
            isset( $_POST['aegisshield_hardening_vuln_nonce'] )
            && wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_hardening_vuln_nonce'] ) ),
                'aegisshield_hardening_vuln_scan'
            )
        ) {
            if ( isset( $_POST['hardening_vuln_source'] ) ) {
                $new_source = sanitize_text_field( wp_unslash( $_POST['hardening_vuln_source'] ) );
                if ( in_array( $new_source, array( 'local', 'patchstack', 'wpvulndb' ), true ) ) {
                    $vuln_source = $new_source;
                    $settings->set( $section, 'hardening_vuln_source', $vuln_source );
                }
            }

            if ( isset( $_POST['hardening_vuln_patchstack_key'] ) ) {
                $patchstack_key = sanitize_text_field( wp_unslash( $_POST['hardening_vuln_patchstack_key'] ) );
                $settings->set( $section, 'hardening_vuln_patchstack_key', $patchstack_key );
            }

            $settings->save();

            if ( $is_pro && isset( $_POST['run_vuln_scan_now'] ) ) {
                $module  = new AS_Module_Hardening( $plugin );
                $results = $module->run_vulnerability_scan();
                $last_run = time();
                $scan_ran = true;
            }

            $vuln_source   = (string) $settings->get( $section, 'hardening_vuln_source', $vuln_source );
            $patchstack_key = (string) $settings->get( $section, 'hardening_vuln_patchstack_key', $patchstack_key );
            $last_run      = (int) $settings->get( $section, 'hardening_vuln_last_run', $last_run );
            $results       = $settings->get( $section, 'hardening_vuln_results', $results );
            if ( ! is_array( $results ) ) {
                $results = array();
            }
        }

        ?>
        <div class="as-hardening-tab as-hardening-tab-vuln">
            <h2><?php esc_html_e( 'Vulnerability Scan', 'aegisshield-security' ); ?></h2>

            <p class="description">
                <?php esc_html_e(
                    'Check installed plugins and themes against a vulnerability feed. You decide which source to use.',
                    'aegisshield-security'
                ); ?>
            </p>

            <?php if ( ! $is_pro ) : ?>
                <div class="notice notice-info">
                    <p>
                        <?php esc_html_e(
                            'Live vulnerability awareness is a Pro feature. You can still configure options here, but scans require an active Pro license.',
                            'aegisshield-security'
                        ); ?>
                    </p>
                </div>
            <?php endif; ?>

            <?php if ( $scan_ran ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Vulnerability scan completed. See the report below.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <form method="post">
                <?php
                wp_nonce_field(
                    'aegisshield_hardening_vuln_scan',
                    'aegisshield_hardening_vuln_nonce'
                );
                ?>

                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row">
                                <label for="hardening_vuln_source">
                                    <?php esc_html_e( 'Vulnerability source', 'aegisshield-security' ); ?>
                                </label>
                            </th>
                            <td>
                                <select id="hardening_vuln_source"
                                        name="hardening_vuln_source"
                                    <?php disabled( ! $is_pro ); ?>>
                                    <option value="local" <?php selected( $vuln_source, 'local' ); ?>>
                                        <?php esc_html_e( 'Local-only (no external calls)', 'aegisshield-security' ); ?>
                                    </option>
                                    <option value="patchstack" <?php selected( $vuln_source, 'patchstack' ); ?>>
                                        <?php esc_html_e( 'Patchstack API (Pro)', 'aegisshield-security' ); ?>
                                    </option>
                                    <option value="wpvulndb" <?php selected( $vuln_source, 'wpvulndb' ); ?>>
                                        <?php esc_html_e( 'WPVulnDB API (Pro)', 'aegisshield-security' ); ?>
                                    </option>
                                </select>
                                <p class="description">
                                    <?php esc_html_e(
                                        'Choose where AegisShield should look up known vulnerabilities. Local-only keeps everything on your server.',
                                        'aegisshield-security'
                                    ); ?>
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="hardening_vuln_patchstack_key">
                                    <?php esc_html_e( 'Patchstack API key', 'aegisshield-security' ); ?>
                                </label>
                            </th>
                            <td>
                                <input type="text"
                                       class="regular-text"
                                       id="hardening_vuln_patchstack_key"
                                       name="hardening_vuln_patchstack_key"
                                       value="<?php echo esc_attr( $patchstack_key ); ?>"
                                    <?php disabled( ! $is_pro ); ?> />
                                <p class="description">
                                    <?php esc_html_e(
                                        'If you select Patchstack as the source, provide your API key here. AegisShield will only call the API when you run a scan.',
                                        'aegisshield-security'
                                    ); ?>
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <?php esc_html_e( 'Last scan', 'aegisshield-security' ); ?>
                            </th>
                            <td>
                                <?php
                                if ( $last_run > 0 ) {
                                    /* translators: %s: date of last scan. */
                                    printf(
                                        esc_html__( 'Last run: %s', 'aegisshield-security' ),
                                        esc_html( wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $last_run ) )
                                    );
                                } else {
                                    esc_html_e( 'No vulnerability scans have been run yet.', 'aegisshield-security' );
                                }
                                ?>
                            </td>
                        </tr>
                    </tbody>
                </table>

                <p>
                    <?php submit_button(
                        __( 'Save Settings', 'aegisshield-security' ),
                        'secondary',
                        'submit',
                        false
                    ); ?>

                    <button type="submit"
                            name="run_vuln_scan_now"
                            class="button button-primary"
                        <?php disabled( ! $is_pro ); ?>>
                        <?php esc_html_e( 'Run Vulnerability Scan Now', 'aegisshield-security' ); ?>
                    </button>
                </p>
            </form>

            <hr />

            <h3><?php esc_html_e( 'Vulnerability Scan Report', 'aegisshield-security' ); ?></h3>

            <?php
            $plugin_results = isset( $results['plugins'] ) && is_array( $results['plugins'] )
                ? $results['plugins']
                : array();
            $theme_results  = isset( $results['themes'] ) && is_array( $results['themes'] )
                ? $results['themes']
                : array();

            if ( empty( $plugin_results ) && empty( $theme_results ) ) :
                ?>
                <p class="description">
                    <?php esc_html_e(
                        'No vulnerabilities have been reported by the selected source. Run a scan to refresh this report.',
                        'aegisshield-security'
                    ); ?>
                </p>
                <?php
                return;
            endif;
            ?>

            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Item', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Type', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Risk', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Details', 'aegisshield-security' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ( $plugin_results as $slug => $info ) : ?>
                        <tr>
                            <td>
                                <code><?php echo esc_html( $slug ); ?></code>
                            </td>
                            <td><?php esc_html_e( 'Plugin', 'aegisshield-security' ); ?></td>
                            <td>
                                <?php
                                $risk = isset( $info['risk'] ) ? $info['risk'] : 'unknown';
                                echo esc_html( ucfirst( $risk ) );
                                ?>
                            </td>
                            <td>
                                <?php
                                $reason = isset( $info['reason'] ) ? $info['reason'] : '';
                                echo esc_html( $reason );
                                ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>

                    <?php foreach ( $theme_results as $slug => $info ) : ?>
                        <tr>
                            <td>
                                <code><?php echo esc_html( $slug ); ?></code>
                            </td>
                            <td><?php esc_html_e( 'Theme', 'aegisshield-security' ); ?></td>
                            <td>
                                <?php
                                $risk = isset( $info['risk'] ) ? $info['risk'] : 'unknown';
                                echo esc_html( ucfirst( $risk ) );
                                ?>
                            </td>
                            <td>
                                <?php
                                $reason = isset( $info['reason'] ) ? $info['reason'] : '';
                                echo esc_html( $reason );
                                ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
    }
}

