<?php
namespace AegisShield\Admin_Pages\Malware;

use AegisShield\AS_Plugin;
use AegisShield\Modules\Malware\AS_Malware_Incidents;

defined( 'ABSPATH' ) || exit;

class AS_Page_Malware_Tab_Attack_Story {

    public static function render_tab( AS_Plugin $plugin ) {
        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        if ( ! $is_pro ) {
            ?>
            <div class="wrap">
                <h1><?php esc_html_e( 'Attack Story (Pro)', 'aegisshield-security' ); ?></h1>
                <p>
                    <?php esc_html_e( 'Attack Story correlates malware detections with file changes, admin actions and login attempts into a single investigation timeline. This feature is available in AegisShield Pro.', 'aegisshield-security' ); ?>
                </p>
                <p>
                    <strong><?php esc_html_e( 'Upgrade to AegisShield Pro to unlock detailed incident timelines and reporting.', 'aegisshield-security' ); ?></strong>
                </p>
            </div>
            <?php
            return;
        }

        $incidents_helper = new AS_Malware_Incidents( $plugin );
        $incidents        = $incidents_helper->get_recent_incidents( 10 );

        ?>
        <div class="wrap aegisshield-attack-story">
            <h1><?php esc_html_e( 'Attack Story', 'aegisshield-security' ); ?></h1>
            <p>
                <?php esc_html_e( 'Review recent malware incidents and see how file changes, admin actions and login activity lined up around each suspicious file.', 'aegisshield-security' ); ?>
            </p>
            <?php if ( empty( $incidents ) ) : ?>
                <p>
                    <?php esc_html_e( 'No malware incidents have been recorded yet. Run a malware scan first, then return to see the correlated attack story.', 'aegisshield-security' ); ?>
                </p>
                <?php
                return;
            endif;
 
            $selected_id = '';
            if ( isset( $_GET['as_ms_incident'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $selected_id = sanitize_text_field( wp_unslash( $_GET['as_ms_incident'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            }

            $selected_incident = null;
            if ( $selected_id ) {
                foreach ( $incidents as $incident ) {
                    if ( (string) $incident['incident_id'] === $selected_id ) {
                        $selected_incident = $incident;
                        break;
                    }
                }
            }

            if ( null === $selected_incident ) {
                $selected_incident = $incidents[0];
            }

            $timeline = $incidents_helper->build_timeline_for_incident( $selected_incident );

            $page_url = admin_url( 'admin.php?page=aegisshield-malware' );
            $report_base_url = admin_url( 'admin.php?page=aegisshield-incident-report' );
            ?>
            <div class="aegisshield-attack-story-layout" style="display:flex; gap:24px; align-items:flex-start;">
                <div class="aegisshield-attack-story-incidents" style="flex:0 0 35%; max-width:35%;">
                    <h2><?php esc_html_e( 'Recent Incidents', 'aegisshield-security' ); ?></h2>
                    <ul class="aegisshield-attack-story-incident-list">
                        <?php foreach ( $incidents as $incident ) : ?>
                            <?php
                            $is_active   = ( (string) $incident['incident_id'] === (string) $selected_incident['incident_id'] );
                            $scan_time   = isset( $incident['scan_time'] ) ? (int) $incident['scan_time'] : time();
                            $scan_label  = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $scan_time );
                            $score       = isset( $incident['score'] ) ? (int) $incident['score'] : 0;
                            $level       = isset( $incident['level'] ) ? (string) $incident['level'] : '';
                            $file        = isset( $incident['file'] ) ? (string) $incident['file'] : '';
                            $item_url    = add_query_arg(
                                array(
                                    'tab'            => 'attack_story',
                                    'as_ms_incident' => $incident['incident_id'],
                                ),
                                $page_url
                            );
                            ?>
                            <li class="<?php echo $is_active ? 'active' : ''; ?>" style="margin-bottom:12px;">
                                <a href="<?php echo esc_url( $item_url ); ?>" style="text-decoration:none;">
                                    <div style="padding:8px 10px; border:1px solid #ccd0d4; border-radius:4px; <?php echo $is_active ? 'background:#f0f6fc;' : 'background:#fff;'; ?>">
                                        <div style="font-weight:600; word-break:break-all;">
                                            <?php echo $file ? esc_html( $file ) : esc_html__( '(Unknown file)', 'aegisshield-security' ); ?>
                                        </div>
                                        <div style="font-size:12px; color:#555;">
                                            <?php
                                            /* translators: 1: score, 2: level */
                                            printf(
                                                esc_html__( 'Score %1$d – %2$s', 'aegisshield-security' ),
                                                $score,
                                                $level ? esc_html( ucfirst( $level ) ) : esc_html__( 'Unknown', 'aegisshield-security' )
                                            );
                                            ?>
                                        </div>
                                        <div style="font-size:12px; color:#777;">
                                            <?php echo esc_html( $scan_label ); ?>
                                        </div>
                                    </div>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>

                <div class="aegisshield-attack-story-timeline" style="flex:1; min-width:0;">
                    <?php
                    $report_url = add_query_arg(
                        array(
                            'incident' => $selected_incident['incident_id'],
                        ),
                        $report_base_url
                    );
                    ?>
                    <div style="text-align:right; margin-bottom:8px;">
                        <a href="<?php echo esc_url( $report_url ); ?>" target="_blank" class="button button-secondary">
                            <?php esc_html_e( 'Export Incident Report (HTML/PDF)', 'aegisshield-security' ); ?>
                        </a>
                    </div>
                    <h2><?php esc_html_e( 'Incident Timeline', 'aegisshield-security' ); ?></h2>
                    <p>
                        <?php
                        /* translators: %s: file path */
                        printf(
                            esc_html__( 'Timeline for file: %s', 'aegisshield-security' ),
                            isset( $selected_incident['file'] ) ? esc_html( $selected_incident['file'] ) : esc_html__( '(Unknown file)', 'aegisshield-security' )
                        );
                        ?>
                    </p>

                    <?php if ( empty( $timeline ) ) : ?>
                        <p>
                            <?php esc_html_e( 'No related events were found around this incident. This may occur if logging was recently enabled or retention is very short.', 'aegisshield-security' ); ?>
                        </p>
                    <?php else : ?>
                        <ul class="aegisshield-attack-story-timeline-list" style="list-style:none; margin:0; padding:0; border-left:2px solid #ccd0d4;">
                            <?php foreach ( $timeline as $entry ) : ?>
                                <?php
                                $event_ts   = isset( $entry['event_time'] ) ? strtotime( $entry['event_time'] ) : false;
                                $time_label = $event_ts ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $event_ts ) : esc_html__( '(Unknown time)', 'aegisshield-security' );
                                $category   = isset( $entry['category'] ) ? $entry['category'] : 'activity';
                                $etype      = isset( $entry['event_type'] ) ? $entry['event_type'] : '';
                                $message    = isset( $entry['message'] ) ? $entry['message'] : '';
                                $primary    = ! empty( $entry['primary'] );
                                ?>
                                <li style="margin:0 0 16px 0; padding:0 0 0 16px; position:relative;">
                                    <span style="position:absolute; left:-7px; top:4px; width:10px; height:10px; border-radius:50%; background:
                                        <?php
                                        switch ( $category ) {
                                            case 'malware':
                                                echo '#dc3232'; // red.
                                                break;
                                            case 'file_monitor':
                                                echo '#ffb900'; // yellow.
                                                break;
                                            case 'login':
                                                echo '#00a0d2'; // blue.
                                                break;
                                            default:
                                                echo '#82878c'; // gray.
                                                break;
                                        }
                                        ?>
                                    "></span>
                                    <div style="font-size:12px; color:#777;">
                                        <?php echo esc_html( $time_label ); ?>
                                    </div>
                                    <div style="font-weight:600; margin-top:2px;">
                                        <?php
                                        switch ( $category ) {
                                            case 'malware':
                                                esc_html_e( 'Malware', 'aegisshield-security' );
                                                break;
                                            case 'file_monitor':
                                                esc_html_e( 'File Monitor', 'aegisshield-security' );
                                                break;
                                            case 'login':
                                                esc_html_e( 'Login Guard', 'aegisshield-security' );
                                                break;
                                            default:
                                                esc_html_e( 'Activity', 'aegisshield-security' );
                                                break;
                                        }
                                        ?>
                                        <?php if ( $etype ) : ?>
                                            <span style="font-weight:400; color:#555;">
                                                (<?php echo esc_html( $etype ); ?>)
                                            </span>
                                        <?php endif; ?>
                                        <?php if ( $primary ) : ?>
                                            <span style="display:inline-block; margin-left:6px; padding:1px 5px; font-size:10px; border-radius:3px; background:#f0f6fc; border:1px solid #ccd0d4;">
                                                <?php esc_html_e( 'This file', 'aegisshield-security' ); ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ( $message ) : ?>
                                        <div style="margin-top:2px;">
                                            <?php echo esc_html( $message ); ?>
                                        </div>
                                    <?php endif; ?>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
}
