<?php
namespace AegisShield\Alerts;

defined( 'ABSPATH' ) || exit;

class AS_Alert_Engine {

    public static function check_rules( $event ) {

        // Only run alerts for Pro installations.
        if ( ! function_exists( 'aegisshield_is_pro_active' ) || ! aegisshield_is_pro_active() ) {
            return;
        }

        if ( empty( $event ) || ! is_array( $event ) ) {
            return;
        }

        $event_type = isset( $event['event_type'] ) ? sanitize_key( $event['event_type'] ) : '';
        if ( ! $event_type ) {
            return;
        }

        $rules = get_option( 'aegisshield_alert_rules', array() );
        if ( empty( $rules ) || ! is_array( $rules ) ) {
            return;
        }

        foreach ( $rules as $rule ) {
            $rule_event = isset( $rule['event_type'] ) ? sanitize_key( $rule['event_type'] ) : '';

            if ( ! $rule_event || $rule_event !== $event_type ) {
                continue;
            }

            $enabled = isset( $rule['enabled'] ) ? (bool) $rule['enabled'] : true;
            if ( ! $enabled ) {
                continue;
            }

            $emails_raw = isset( $rule['emails'] ) ? $rule['emails'] : '';
            $emails     = array_filter(
                array_map(
                    'sanitize_email',
                    array_map( 'trim', explode( ',', (string) $emails_raw ) )
                )
            );

            if ( empty( $emails ) ) {
                continue;
            }

            self::send_alert_email( $rule, $event, $emails );
        }
    }

    protected static function send_alert_email( $rule, $event, $emails ) {

        $event_type = isset( $event['event_type'] ) ? sanitize_key( $event['event_type'] ) : '';
        $username   = isset( $event['username'] ) && $event['username'] ? $event['username'] : 'System/unknown';
        $ip         = isset( $event['ip'] ) ? $event['ip'] : '';
        $timestamp  = isset( $event['timestamp'] ) ? $event['timestamp'] : current_time( 'mysql' );
        $details    = isset( $event['details'] ) ? $event['details'] : null;

        $rule_name = isset( $rule['name'] ) && $rule['name']
            ? $rule['name']
            : $event_type;

        $subject = sprintf(
            /* translators: %s: alert rule name */
            __( 'AegisShield Alert: %s', 'aegisshield-security' ),
            $rule_name
        );

        $body_lines   = array();
        $body_lines[] = __( 'AegisShield Alert Triggered', 'aegisshield-security' );
        $body_lines[] = sprintf(
            /* translators: %s: alert rule name */
            __( 'Rule: %s', 'aegisshield-security' ),
            $rule_name
        );
        $body_lines[] = sprintf(
            /* translators: %s: event type slug */
            __( 'Event: %s', 'aegisshield-security' ),
            $event_type
        );
        $body_lines[] = sprintf(
            /* translators: %s: username or System/unknown */
            __( 'User: %s', 'aegisshield-security' ),
            $username
        );
        if ( $ip ) {
            $body_lines[] = sprintf(
                /* translators: %s: IP address */
                __( 'IP Address: %s', 'aegisshield-security' ),
                $ip
            );
        }
        $body_lines[] = sprintf(
            /* translators: %s: event time */
            __( 'Time: %s', 'aegisshield-security' ),
            $timestamp
        );

        // Details / meta.
        if ( null !== $details ) {
            $body_lines[] = '';
            $body_lines[] = __( 'Details:', 'aegisshield-security' );
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r
            $body_lines[] = print_r( $details, true );
        }

        $body = implode( "\n", $body_lines );

        $plugin   = class_exists( '\\AegisShield\\AS_Plugin' ) ? \AegisShield\AS_Plugin::instance() : null;
        $notifier = ( $plugin && method_exists( $plugin, 'get_notifier' ) ) ? $plugin->get_notifier() : null;

        if ( $notifier && method_exists( $notifier, 'send_email_to' ) ) {
            $html_body = $notifier->format_email( $subject, nl2br( esc_html( $body ) ) );
            $notifier->send_email_to( $emails, $subject, $html_body );
        } else {
            // Fallback: direct mail (should be rare).
            foreach ( $emails as $email ) {
                if ( is_email( $email ) ) {
                    wp_mail( $email, $subject, $body );
                }
            }
        }
}
}

