<?php
namespace AegisShield\Helpers;

use AegisShield\MFA\AS_MFA_Storage;
use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Register_MFA_Shortcode {

    public static function register() {
        add_shortcode( 'aegis_register_mfa', array( __CLASS__, 'render' ) );
    }

    public static function render() {

        $step    = isset( $_GET['aegis_mfa_enroll'] ) ? 'enroll' : 'register';
        $error   = '';
        $success = '';

        /* =========================================================
         * STEP 1 — REGISTRATION
         * ======================================================= */
        if ( 'register' === $step ) {

            if ( isset( $_POST['aegis_register_submit'] ) ) {

                $first = sanitize_text_field( $_POST['first_name'] ?? '' );
                $last  = sanitize_text_field( $_POST['last_name'] ?? '' );
                $user  = sanitize_user( $_POST['username'] ?? '' );
                $email = sanitize_email( $_POST['email'] ?? '' );
                $p1    = (string) ( $_POST['password'] ?? '' );
                $p2    = (string) ( $_POST['confirm_password'] ?? '' );

                if ( ! $first || ! $last || ! $user || ! $email || ! $p1 || ! $p2 ) {
                    $error = 'All fields are required.';
                } elseif ( $p1 !== $p2 ) {
                    $error = 'Passwords do not match.';
                } elseif ( ! self::passes_aegisshield_password_policy( $p1 ) ) {
                    $error = 'Password does not meet security requirements.';
                } else {

                    $user_id = wp_create_user( $user, $p1, $email );

                    if ( is_wp_error( $user_id ) ) {
                        $error = $user_id->get_error_message();
                    } else {

                        update_user_meta( $user_id, 'first_name', $first );
                        update_user_meta( $user_id, 'last_name', $last );

                        wp_set_current_user( $user_id );
                        wp_set_auth_cookie( $user_id );

                        wp_safe_redirect(
                            add_query_arg( 'aegis_mfa_enroll', '1', get_permalink() )
                        );
                        exit;
                    }
                }
            }

            return self::render_registration_form( $error );
        }

        /* =========================================================
         * STEP 2 — MFA ENROLLMENT
         * ======================================================= */
        if ( ! is_user_logged_in() ) {
            return '<p>Please log in to enroll MFA.</p>';
        }

        $user_id = get_current_user_id();
        $user    = wp_get_current_user();

        $secret = get_user_meta( $user_id, 'aegisshield_2fa_secret', true );
        if ( ! $secret ) {
            $secret = self::generate_totp_secret_base32();
            update_user_meta( $user_id, 'aegisshield_2fa_secret', $secret );
        }

        if ( isset( $_POST['aegis_confirm_mfa'] ) ) {

            $code = preg_replace( '/[^0-9]/', '', $_POST['aegisshield_2fa_code'] ?? '' );

            if ( strlen( $code ) !== 6 ) {
                $error = 'Invalid authentication code.';
            } elseif ( AS_MFA_Storage::verify_totp( $user, $code ) ) {
                update_user_meta( $user_id, 'aegisshield_2fa_enabled', '1' );
                // Phase 2: Always log user-level MFA enrollment completion.
                if ( class_exists( '\\AegisShield\\AS_Plugin' ) ) {
                    $plugin = AS_Plugin::instance();
                    if ( $plugin && method_exists( $plugin, 'get_logger' ) ) {
                        $plugin->get_logger()->log(
                            'mfa_enrollment_completed',
                            __( 'User completed MFA enrollment.', 'aegisshield-security' ),
                            'low',
                            array(
                                'user_id' => (int) $user_id,
                                'method'  => 'totp',
                            )
                        );
                    }
                }
                $success = 'MFA successfully enabled.';
            } else {
                $error = 'Code verification failed. Try again.';
            }
        }

        return self::render_enrollment( $user, $secret, $error, $success );
    }

    /* =========================================================
     * UI — REGISTRATION
     * ======================================================= */
    private static function render_registration_form( $error ) {
        ob_start(); ?>
        <form method="post" style="max-width:520px;margin:auto;border:1px solid #ddd;padding:20px;border-radius:8px">
            <h3>Create Account</h3>

            <?php if ( $error ) : ?>
                <p style="color:#b00"><?php echo esc_html( $error ); ?></p>
            <?php endif; ?>

            <div style="display:flex;gap:10px">
                <input name="first_name" placeholder="First Name" required style="flex:1">
                <input name="last_name" placeholder="Last Name" required style="flex:1">
            </div>

            <input name="username" placeholder="Username" required style="width:100%;margin-top:10px">
            <input name="email" type="email" placeholder="Email" required style="width:100%;margin-top:10px">

            <div style="display:flex;gap:10px;margin-top:10px">
                <div style="flex:1;position:relative">
                    <input type="password" id="aegis_pass_1" name="password" placeholder="Password" required style="width:100%">
                    <span onclick="aegisTogglePass('aegis_pass_1')" style="position:absolute;right:10px;top:50%;cursor:pointer">👁</span>
                </div>

                <div style="flex:1;position:relative">
                    <input type="password" id="aegis_pass_2" name="confirm_password" placeholder="Confirm Password" required style="width:100%">
                    <span onclick="aegisTogglePass('aegis_pass_2')" style="position:absolute;right:10px;top:50%;cursor:pointer">👁</span>
                </div>
            </div>

            <button type="submit" name="aegis_register_submit" style="margin-top:15px;width:100%">
                Create Account
            </button>
        </form>

        <script>
        function aegisTogglePass(id){
            const el = document.getElementById(id);
            if(!el) return;
            el.type = el.type === 'password' ? 'text' : 'password';
        }
        </script>
        <?php
        return ob_get_clean();
    }

    /* =========================================================
     * UI — MFA ENROLLMENT
     * ======================================================= */
    private static function render_enrollment( $user, $secret, $error, $success ) {

        $otpauth = sprintf(
            'otpauth://totp/%s:%s?secret=%s&issuer=%s',
            rawurlencode( get_bloginfo( 'name' ) ),
            rawurlencode( $user->user_login ),
            $secret,
            rawurlencode( get_bloginfo( 'name' ) )
        );

        $qr_html = self::generate_qr_html( $otpauth );

        ob_start(); ?>
        <div style="max-width:520px;margin:auto;border:1px solid #ddd;padding:20px;border-radius:8px">
            <h3>Secure Your Account</h3>

            <?php if ( $error ) : ?><p style="color:#b00"><?php echo esc_html( $error ); ?></p><?php endif; ?>
            <?php if ( $success ) : ?><p style="color:green"><?php echo esc_html( $success ); ?></p><?php endif; ?>

            <?php if ( $qr_html ) : ?>
                <div class="aegis-qr-wrap"><?php echo $qr_html; ?></div>
            <?php else : ?>
                <p><strong>QR unavailable.</strong> Use manual key below.</p>
            <?php endif; ?>

            <p><strong>Manual setup key:</strong><br><code><?php echo esc_html( $secret ); ?></code></p>

            <?php if ( ! $success ) : ?>
                <form method="post">
                    <input name="aegisshield_2fa_code" placeholder="6-digit code" required style="width:100%">
                    <button name="aegis_confirm_mfa" style="margin-top:10px;width:100%">
                        Confirm & Enable MFA
                    </button>
                </form>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /* =========================================================
     * HELPERS
     * ======================================================= */

    private static function passes_aegisshield_password_policy( $password ) {

        if (
            class_exists( '\AegisShield\Modules\AS_Module_Hardening' ) &&
            method_exists( '\AegisShield\Modules\AS_Module_Hardening', 'validate_password' )
        ) {
            return \AegisShield\Modules\AS_Module_Hardening::validate_password( $password );
        }

        // Fallback safety
        return strlen( $password ) >= 12;
    }

    private static function generate_qr_html( $text ) {

        if ( ! class_exists( '\Qrc\QrCode' ) ) {
            return '';
        }

        try {
            $renderer = new \Qrc\Renderer\SvgRenderer();

            return \Qrc\QrCode::newInstance(
                $renderer,
                \Qrc\ErrorCorrection::Medium
            )
            ->setData( $text )
            ->render()
            ->toHtml( true );
        } catch ( \Throwable $e ) {
            return '';
        }
    }

    private static function generate_totp_secret_base32() {

        $bytes = random_bytes( 16 );
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $bits = '';
        $out = '';

        foreach ( str_split( $bytes ) as $c ) {
            $bits .= str_pad( decbin( ord( $c ) ), 8, '0', STR_PAD_LEFT );
        }

        foreach ( str_split( $bits, 5 ) as $chunk ) {
            $out .= $alphabet[ bindec( str_pad( $chunk, 5, '0', STR_PAD_RIGHT ) ) ];
        }

        return $out;
    }
}