<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Module_File_Monitor implements AS_Module_Interface {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'file_monitor';
    }

    public function register_settings() {
        $settings = $this->plugin->get_settings();
        $section  = 'file_monitor';

        if ( null === $settings->get( $section, 'enabled', null ) ) {
            $settings->set( $section, 'enabled', 'yes' );
        }

        if ( null === $settings->get( $section, 'interval', null ) ) {
            $settings->set( $section, 'interval', '15' );
        }

        $settings->save();
    }

    public function init() {
        add_filter( 'cron_schedules', array( $this, 'register_cron_schedules' ) );

        $hook = 'aegisshield_file_monitor_scan';

        $schedule = $this->get_cron_schedule_slug();

		$next = wp_next_scheduled( $hook );

		if ( $next ) {
			wp_unschedule_event( $next, $hook );
			$next = false; // IMPORTANT: force reschedule below
		}

		if ( ! $next ) {
			wp_schedule_event( time() + MINUTE_IN_SECONDS, $schedule, $hook );
		}

        add_action( $hook, array( $this, 'run_monitor_scan' ) );
    }

	protected function log_fim_violation( $rule, $message, $severity = 'medium', $meta = array(), $event = 'fim_violation' ) {
		$logger = $this->plugin ? $this->plugin->get_logger() : null;
		if ( ! $logger ) {
			return;
		}

		if ( ! is_array( $meta ) ) {
			$meta = array();
		}

		if ( isset( $meta['sample_paths'] ) && is_array( $meta['sample_paths'] ) ) {
			$meta['sample_paths'] = array_slice( $meta['sample_paths'], 0, 20 );
		}

		$meta = array_merge(
			array(
				'module' => 'file_monitor',
				'rule'   => (string) $rule,
			),
			$meta
		);

		$logger->log( (string) $event, (string) $message, (string) $severity, $meta );
	}

	protected function fim_rule_enabled( $rule ) {
		$settings     = $this->plugin->get_settings();
		$email_events = (array) $settings->get( 'file_monitor', 'email_events', array() );

		return in_array( (string) $rule, $email_events, true );
	}

	protected function is_high_risk_path( $rel_path ) {
		$rel_path = ltrim( (string) $rel_path, '/\\' );
		return ( 0 === strpos( $rel_path, 'wp-admin/' ) || 0 === strpos( $rel_path, 'wp-includes/' ) );
	}

	protected function is_plugin_or_theme_path( $abs_path ) {
		$abs_path = (string) $abs_path;

		if ( defined( 'WP_PLUGIN_DIR' ) && 0 === strpos( $abs_path, wp_normalize_path( WP_PLUGIN_DIR ) ) ) {
			return true;
		}

		if ( function_exists( 'get_theme_root' ) ) {
			$theme_root = wp_normalize_path( get_theme_root() );
			if ( $theme_root && 0 === strpos( $abs_path, $theme_root ) ) {
				return true;
			}
		}
		return false;
		function is_log_file_path( $rel_path ) {
			$rel_path = ltrim( (string) $rel_path, '/\\' );
			$rel_lc   = strtolower( $rel_path );

			$base = strtolower( basename( $rel_path ) );
			$ext  = strtolower( pathinfo( $rel_path, PATHINFO_EXTENSION ) );

			// Common WordPress / PHP logs.
			if ( 'log' === $ext ) {
				return true;
			}

			if ( 'debug.log' === $base || 'error_log' === $base ) {
				return true;
			}

			// Any directory explicitly named "logs".
			// (Catches wp-content/uploads/.../logs/... etc.)
			if ( false !== strpos( $rel_lc, '/logs/' ) || 0 === strpos( $rel_lc, 'logs/' ) ) {
				return true;
			}

			return false;
		}

	}

    public function register_cron_schedules( $schedules ) {
        if ( ! isset( $schedules['aegisshield_file_monitor_5min'] ) ) {
            $schedules['aegisshield_file_monitor_5min'] = array(
                'interval' => 5 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 5 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        // 10 minutes.
        if ( ! isset( $schedules['aegisshield_file_monitor_10min'] ) ) {
            $schedules['aegisshield_file_monitor_10min'] = array(
                'interval' => 10 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 10 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        // 15 minutes (default).
        if ( ! isset( $schedules['aegisshield_file_monitor_15min'] ) ) {
            $schedules['aegisshield_file_monitor_15min'] = array(
                'interval' => 15 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 15 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        // 30 minutes.
        if ( ! isset( $schedules['aegisshield_file_monitor_30min'] ) ) {
            $schedules['aegisshield_file_monitor_30min'] = array(
                'interval' => 30 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 30 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        // 60 minutes.
        if ( ! isset( $schedules['aegisshield_file_monitor_60min'] ) ) {
            $schedules['aegisshield_file_monitor_60min'] = array(
                'interval' => 60 * MINUTE_IN_SECONDS,
                'display'  => __( 'Every 60 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
            );
        }

        return $schedules;
    }

    protected function get_cron_schedule_slug() {
        $settings = $this->plugin->get_settings();
        $section  = 'file_monitor';

        $interval = (string) $settings->get( $section, 'interval', '15' );
        if ( ! in_array( $interval, array( '5', '10', '15', '30', '60' ), true ) ) {
            $interval = '15';
        }

        switch ( $interval ) {
            case '5':
                return 'aegisshield_file_monitor_5min';
            case '10':
                return 'aegisshield_file_monitor_10min';
            case '30':
                return 'aegisshield_file_monitor_30min';
            case '60':
                return 'aegisshield_file_monitor_60min';
            case '15':
            default:
                return 'aegisshield_file_monitor_15min';
        }
    }

    protected function get_table_name() {
        global $wpdb;

        return $wpdb->prefix . 'aegisshield_file_monitor';
    }

    protected function maybe_create_table() {
        global $wpdb;

        $table  = $this->get_table_name();
        $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );

        if ( $exists === $table ) {
            return;
        }

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            file_path VARCHAR(255) NOT NULL,
            hash VARCHAR(64) NOT NULL,
            size BIGINT UNSIGNED NOT NULL DEFAULT 0,
            mtime BIGINT UNSIGNED NOT NULL DEFAULT 0,
            first_seen DATETIME NOT NULL,
            last_seen DATETIME NOT NULL,
            status VARCHAR(20) DEFAULT 'unchanged',
            PRIMARY KEY  (id),
            KEY status (status),
            KEY mtime (mtime)
        ) {$charset_collate};";

        dbDelta( $sql );
    }

	public function run_monitor_scan() {
		$settings = $this->plugin->get_settings();

		// ENFORCE: if monitor is disabled, do not scan.
		$enabled = (string) $settings->get( 'file_monitor', 'enabled', 'yes' );
		if ( 'yes' !== $enabled ) {
			$this->log_fim_violation(
				'monitor_disabled',
				__( 'File Change Monitor is disabled but the scheduled scan hook executed.', 'aegisshield-security' ),
				'medium',
				array(),
				'fim_violation'
			);
			return;
		}

		$this->maybe_create_table();
		
		global $wpdb;
		$table = $this->get_table_name();

		// LOG: if table still doesn't exist, the monitor cannot function.
		$exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table ) );
		if ( $exists !== $table ) {
			$this->log_fim_violation(
				'monitor_table_missing',
				__( 'File Change Monitor table is missing; monitoring cannot run.', 'aegisshield-security' ),
				'high',
				array( 'table' => $table ),
				'fim_violation'
			);
			return;
		}

        global $wpdb;

        $table = $this->get_table_name();

        $roots = array();

        if ( defined( 'WP_CONTENT_DIR' ) ) {
            $roots[] = WP_CONTENT_DIR;
        }
        if ( defined( 'WP_PLUGIN_DIR' ) ) {
            $roots[] = WP_PLUGIN_DIR;
        }
        if ( function_exists( 'get_theme_root' ) ) {
            $roots[] = get_theme_root();
        }

        if ( empty( $roots ) ) {
            return;
        }

        $roots = array_values(
            array_unique(
                array_map(
                    'untrailingslashit',
                    array_filter( $roots, 'is_dir' )
                )
            )
        );

		$custom_paths = (array) $settings->get( 'file_monitor', 'custom_paths', array() );
		$exclude_dirs = (array) $settings->get( 'file_monitor', 'exclude_dirs', array() );

		foreach ( $custom_paths as $p ) {
			$p = (string) $p;
			if ( '' === trim( $p ) ) {
				continue;
			}
			$p = untrailingslashit( wp_normalize_path( $p ) );
			if ( @is_dir( $p ) ) { // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
				$roots[] = $p;
			} else {
				$this->log_fim_violation(
					'custom_path_invalid',
					__( 'File Change Monitor custom path is not a directory and was skipped.', 'aegisshield-security' ),
					'medium',
					array( 'path' => $p ),
					'fim_violation'
				);
			}
		}

		$roots = array_values( array_unique( array_map( 'untrailingslashit', $roots ) ) );

		$exclude_dirs = array_values(
			array_unique(
				array_filter(
					array_map(
						function ( $p ) {
							$p = (string) $p;
							if ( '' === trim( $p ) ) {
								return '';
							}
							return untrailingslashit( wp_normalize_path( $p ) );
						},
						$exclude_dirs
					)
				)
			)
		);

        if ( empty( $roots ) ) {
            return;
        }

        $existing_rows = array();
        $rows          = $wpdb->get_results(
            "SELECT id, file_path, hash, size, mtime, status FROM {$table}",
            ARRAY_A
        );

        if ( is_array( $rows ) ) {
            foreach ( $rows as $row ) {
                $existing_rows[ $row['file_path'] ] = $row;
            }
        }

        $now_mysql  = current_time( 'mysql' );
        $seen_paths = array();

        $logger = $this->plugin->get_logger();

        foreach ( $roots as $root ) {
            try {
                $iterator = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator(
                        $root,
                        \FilesystemIterator::SKIP_DOTS | \FilesystemIterator::FOLLOW_SYMLINKS
                    ),
                    \RecursiveIteratorIterator::SELF_FIRST
                );
            } catch ( \Exception $e ) {
                continue;
            }

            foreach ( $iterator as $file ) {
                if ( ! $file->isFile() ) {
                    continue;
                }

                $path = $file->getPathname();

				$norm_path = untrailingslashit( wp_normalize_path( $path ) );

				// ENFORCE exclusions
				$excluded = false;
				foreach ( $exclude_dirs as $ex ) {
					if ( '' !== $ex && 0 === strpos( $norm_path, $ex ) ) {
						$excluded = true;
						break;
					}
				}

				if ( $excluded ) {
					continue;
				}

                $rel_path = ltrim( str_replace( ABSPATH, '', $path ), '/' );
				
				// Ignore log files completely (prevents baseline inserts/updates + future alerts).
				if ( $this->is_log_file_path( $rel_path ) ) {
					continue;
				}


                $seen_paths[ $rel_path ] = true;

                $size  = (int) $file->getSize();
                $mtime = (int) $file->getMTime();

                $ext      = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );
                $hashable = in_array(
                    $ext,
                    array( 'php', 'php5', 'php7', 'phtml', 'js', 'css', 'inc', 'html', 'htm' ),
                    true
                );

                $hash = '';
                if ( $hashable ) {
                    $hash_raw = @hash_file( 'sha256', $path ); // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
                    if ( false !== $hash_raw ) {
                        $hash = $hash_raw;
                    }
                }

                if ( ! isset( $existing_rows[ $rel_path ] ) ) {
                    $wpdb->insert(
                        $table,
                        array(
                            'file_path'  => $rel_path,
                            'hash'       => $hash,
                            'size'       => $size,
                            'mtime'      => $mtime,
                            'first_seen' => $now_mysql,
                            'last_seen'  => $now_mysql,
                            'status'     => 'new',
                        ),
                        array(
                            '%s',
                            '%s',
                            '%d',
                            '%d',
                            '%s',
                            '%s',
                            '%s',
                        )
                    );

					if ( $this->fim_rule_enabled( 'new' ) ) {
						$sev = 'medium';

						if ( in_array( $ext, array( 'php', 'php5', 'php7', 'phtml' ), true ) && $this->fim_rule_enabled( 'php' ) ) {
							$sev = 'high';
							$this->log_fim_violation(
								'php',
								__( 'FIM rule triggered: new PHP file detected.', 'aegisshield-security' ),
								'high',
								array(
									'path' => $rel_path,
									'ext'  => $ext,
								),
								'fim_php_file_violation'
							);
						}

						if ( $this->fim_rule_enabled( 'high_risk' ) && $this->is_high_risk_path( $rel_path ) ) {
							$sev = 'high';
							$this->log_fim_violation(
								'high_risk',
								__( 'FIM rule triggered: new file detected in high-risk WordPress directory.', 'aegisshield-security' ),
								'high',
								array(
									'path' => $rel_path,
									'ext'  => $ext,
								),
								'fim_high_risk_path_violation'
							);
						}

						if ( $this->fim_rule_enabled( 'plugin_theme' ) && $this->is_plugin_or_theme_path( wp_normalize_path( $path ) ) ) {
							$this->log_fim_violation(
								'plugin_theme',
								__( 'FIM rule triggered: new file detected in plugin/theme directory.', 'aegisshield-security' ),
								$sev,
								array(
									'path' => $rel_path,
									'ext'  => $ext,
								),
								'fim_rules_access'
							);
						}

						$this->log_fim_violation(
							'new',
							__( 'FIM rule triggered: new file detected.', 'aegisshield-security' ),
							$sev,
							array(
								'path' => $rel_path,
								'ext'  => $ext,
							),
							'fim_file_created'
						);
					}
                } else {
                    $row           = $existing_rows[ $rel_path ];
                    $previous_hash = (string) $row['hash'];
                    $previous_size = (int) $row['size'];
                    $previous_mtime = (int) $row['mtime'];

                    $status = 'unchanged';

                    if ( $previous_hash !== (string) $hash || $previous_size !== $size || $previous_mtime !== $mtime ) {
                        $status = 'modified';

						if ( $this->fim_rule_enabled( 'modified' ) ) {
							$sev = 'medium';

							if ( in_array( $ext, array( 'php', 'php5', 'php7', 'phtml' ), true ) && $this->fim_rule_enabled( 'php' ) ) {
								$sev = 'high';
								$this->log_fim_violation(
									'php',
									__( 'FIM rule triggered: PHP file modified.', 'aegisshield-security' ),
									'high',
									array(
										'path' => $rel_path,
										'ext'  => $ext,
									),
									'fim_php_file_violation'
								);
							}

							if ( $this->fim_rule_enabled( 'high_risk' ) && $this->is_high_risk_path( $rel_path ) ) {
								$sev = 'high';
								$this->log_fim_violation(
									'high_risk',
									__( 'FIM rule triggered: file modified in high-risk WordPress directory.', 'aegisshield-security' ),
									'high',
									array(
										'path' => $rel_path,
										'ext'  => $ext,
									),
									'fim_high_risk_path_violation'
								);
							}

							if ( $this->fim_rule_enabled( 'plugin_theme' ) && $this->is_plugin_or_theme_path( wp_normalize_path( $path ) ) ) {
								$this->log_fim_violation(
									'plugin_theme',
									__( 'FIM rule triggered: plugin/theme file modified.', 'aegisshield-security' ),
									$sev,
									array(
										'path' => $rel_path,
										'ext'  => $ext,
									),
									'fim_rules_access'
								);
							}

							$this->log_fim_violation(
								'modified',
								__( 'FIM rule triggered: file modified.', 'aegisshield-security' ),
								$sev,
								array(
									'path' => $rel_path,
									'ext'  => $ext,
								),
								'fim_file_modified'
							);
						}
                    }

                    $wpdb->update(
                        $table,
                        array(
                            'hash'      => $hash,
                            'size'      => $size,
                            'mtime'     => $mtime,
                            'last_seen' => $now_mysql,
                            'status'    => $status,
                        ),
                        array(
                            'id' => (int) $row['id'],
                        ),
                        array(
                            '%s',
                            '%d',
                            '%d',
                            '%s',
                            '%s',
                        ),
                        array(
                            '%d',
                        )
                    );

                    unset( $existing_rows[ $rel_path ] );
                }
            }
        }

        if ( ! empty( $existing_rows ) ) {
            foreach ( $existing_rows as $rel_path => $row ) {
                $wpdb->update(
                    $table,
                    array(
                        'last_seen' => $now_mysql,
                        'status'    => 'deleted',
                    ),
                    array(
                        'id' => (int) $row['id'],
                    ),
                    array(
                        '%s',
                        '%s',
                    ),
                    array(
                        '%d',
                    )
                );

				if ( $this->fim_rule_enabled( 'deleted' ) ) {
					$sev = 'high';

					if ( $this->fim_rule_enabled( 'high_risk' ) && $this->is_high_risk_path( $rel_path ) ) {
						$this->log_fim_violation(
							'high_risk',
							__( 'FIM rule triggered: file deleted in high-risk WordPress directory.', 'aegisshield-security' ),
							'high',
							array( 'path' => $rel_path ),
							'fim_high_risk_path_violation'
						);
					}

					$this->log_fim_violation(
						'deleted',
						__( 'FIM rule triggered: file deleted/missing.', 'aegisshield-security' ),
						$sev,
						array( 'path' => $rel_path ),
						'fim_file_deleted'
					);	
				}
				// If a previously tracked item is a log file, remove it quietly to prevent log flooding.
				if ( $this->is_log_file_path( $rel_path ) ) {
					$wpdb->delete(
						$table,
						array( 'id' => (int) $row['id'] ),
						array( '%d' )
					);
					continue;
				}
            }
        }
    }
}
