<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;
use AegisShield\Modules\Malware\AS_Malware_Profiles;
use AegisShield\Modules\Malware\AS_Malware_Incremental;

defined( 'ABSPATH' ) || exit;

class AS_Module_Malware implements AS_Module_Interface {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'malware';
    }

	public function register_settings() {
		$settings = $this->plugin->get_settings();
		$section  = 'malware';

		// Enable module by default so manual scans + UI work on fresh installs.
		if ( null === $settings->get( $section, 'enabled', null ) ) {
			$settings->set( $section, 'enabled', 'yes' );
		}

		// Provide sane default scan locations so full scans return results.
		if ( null === $settings->get( $section, 'scan_dirs', null ) ) {
			$settings->set( $section, 'scan_dirs', array( 'plugins', 'themes' ) );
		}

		// Optional: enable auto-scan for Free legacy behavior (your code supports it).
		if ( null === $settings->get( $section, 'auto_scan', null ) ) {
			$settings->set( $section, 'auto_scan', 'yes' );
		}

		// Default handling mode to read-only (safe on shared hosting).
		if ( null === $settings->get( $section, 'handling_mode', null ) ) {
			$settings->set( $section, 'handling_mode', 'a' );
		}
		
        // Hash allowlist (path => sha256). Used to reduce repeat false positives safely.
        if ( null === $settings->get( $section, 'allowlist_hashes', null ) ) {
            $settings->set( $section, 'allowlist_hashes', array() );
        }

		$settings->save();
	}

    public function init() {
        $settings = $this->plugin->get_settings();
        $section  = 'malware';

        $enabled = (string) $settings->get( $section, 'enabled', '' );
        if ( ! in_array( $enabled, array( 'on', '1', 'yes' ), true ) ) {
            return;
        }

        add_action( 'aegisshield_hourly_maintenance', array( $this, 'maybe_run_auto_scan' ) );
    }

    public function maybe_run_auto_scan() {
        $settings = $this->plugin->get_settings();
        $section  = 'malware';

        $enabled = (string) $settings->get( $section, 'enabled', '' );
        if ( ! in_array( $enabled, array( 'on', '1', 'yes' ), true ) ) {
            return;
        }

        $auto_scan = (string) $settings->get( $section, 'auto_scan', '' );
        if ( ! in_array( $auto_scan, array( 'on', '1', 'yes' ), true ) ) {
            return;
        }

        $now     = time();
        $is_pro  = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
        $did_run = false;

        if ( $is_pro ) {
            // Pro: honour lightweight scheduled quick scans and weekly full scans.
            $quick_enabled  = (string) $settings->get( $section, 'schedule_quick_enabled', '' );
            $weekly_enabled = (string) $settings->get( $section, 'schedule_weekly_enabled', '' );

            $quick_enabled_bool  = in_array( $quick_enabled, array( 'on', '1', 'yes' ), true );
            $weekly_enabled_bool = in_array( $weekly_enabled, array( 'on', '1', 'yes' ), true );

            // Frequency for incremental quick scans (admin-controlled).
            $quick_frequency = (string) $settings->get( $section, 'quick_schedule_frequency', 'daily' );
            if ( ! in_array( $quick_frequency, array( 'daily', 'weekly', 'monthly', 'quarterly' ), true ) ) {
                $quick_frequency = 'daily';
            }

            switch ( $quick_frequency ) {
                case 'weekly':
                    $quick_interval = WEEK_IN_SECONDS;
                    break;
                case 'monthly':
                    // Approximate month as 30 days for cron safety.
                    $quick_interval = 30 * DAY_IN_SECONDS;
                    break;
                case 'quarterly':
                    // Approximate quarter as 90 days.
                    $quick_interval = 90 * DAY_IN_SECONDS;
                    break;
                case 'daily':
                default:
                    $quick_interval = DAY_IN_SECONDS;
                    break;
            }

            $last_quick = (int) $settings->get( $section, 'last_quick_scan', 0 );
            $last_full  = (int) $settings->get( $section, 'last_full_scan', 0 );

            // Incremental quick scan according to chosen frequency.
            if ( $quick_enabled_bool && ( ! $last_quick || ( $now - $last_quick ) >= $quick_interval ) ) {
                $this->run_incremental_scan( 'auto_quick' );
                $settings->set( $section, 'last_quick_scan', $now );
                $did_run = true;
            }

            // Weekly full scan (if enabled).
            if ( $weekly_enabled_bool && ( ! $last_full || ( $now - $last_full ) >= WEEK_IN_SECONDS ) ) {
                $this->run_scan( 'auto' );
                $settings->set( $section, 'last_full_scan', $now );
                $did_run = true;
            }

            // If no specific Pro schedule is enabled, fall back to the legacy daily auto-scan.
            if ( ! $quick_enabled_bool && ! $weekly_enabled_bool ) {
                $last_auto = (int) $settings->get( $section, 'last_auto_scan', 0 );

                if ( ! $last_auto || ( $now - $last_auto ) >= DAY_IN_SECONDS ) {
                    $this->run_scan( 'auto' );
                    $settings->set( $section, 'last_auto_scan', $now );
                    $did_run = true;
                }
            }
        } else {
            // Free: preserve original behaviour – at most one full scan per day.
            $last_auto = (int) $settings->get( $section, 'last_auto_scan', 0 );

            if ( $last_auto && ( $now - $last_auto ) < DAY_IN_SECONDS ) {
                return;
            }

            $this->run_scan( 'auto' );
            $settings->set( $section, 'last_auto_scan', $now );
            $did_run = true;
        }

        if ( $did_run ) {
            $settings->save();
        }
    }

    public function run_incremental_scan( $scan_type ) {
        $settings = $this->plugin->get_settings();
        $logger   = $this->plugin->get_logger();
        $section  = 'malware';

        $started_at    = time();
        $scanned_files = 0;
        $is_partial    = false;

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
        if ( ! $is_pro ) {
            return array();
        }

        $handling_mode = (string) $settings->get( $section, 'handling_mode', '' );
        if ( ! in_array( $handling_mode, array( 'a', 'b', 'c' ), true ) ) {
            $handling_mode = 'a';
        }

        $ignore_list = $settings->get( $section, 'ignore_list', array() );
        if ( ! is_array( $ignore_list ) ) {
            $ignore_list = array();
        }

        $profile_key = AS_Malware_Profiles::get_selected_profile( $settings, $section );
        $threshold   = AS_Malware_Profiles::get_threshold( $profile_key );

        $limit   = 300;
        $targets = AS_Malware_Incremental::get_incremental_targets();

        if ( empty( $targets ) || ! is_array( $targets ) ) {
            if ( $logger ) {
                $logger->log(
                    'malware_quick_scan_skipped',
                    __( 'Incremental quick scan skipped: no changed files were found in the File Monitor table.', 'aegisshield-security' ),
                    array(
                        'scan_type' => $scan_type,
                    )
                );
            }
            return array();
        }

        if ( count( $targets ) >= $limit ) {
            $is_partial = true;
        }

        $results = array();

        foreach ( $targets as $rel_path ) {
            $rel_path = ltrim( (string) $rel_path, '/' );

            if ( in_array( $rel_path, $ignore_list, true ) ) {
                continue;
            }

            $allowlist_hashes = $settings->get( $section, 'allowlist_hashes', array() );
            if ( is_array( $allowlist_hashes ) && isset( $allowlist_hashes[ $rel_path ] ) ) {
                $abs_tmp = trailingslashit( ABSPATH ) . $rel_path;
                $cur = @hash_file( 'sha256', $abs_tmp );
                if ( $cur && hash_equals( (string) $allowlist_hashes[ $rel_path ], (string) $cur ) ) {
                    continue;
                }
            }
			
            $abs_path = trailingslashit( ABSPATH ) . $rel_path;

            if ( ! file_exists( $abs_path ) || ! is_file( $abs_path ) ) {
                continue;
            }

            $ext = strtolower( pathinfo( $abs_path, PATHINFO_EXTENSION ) );
            if ( ! in_array( $ext, array( 'php', 'phtml', 'php5', 'php7' ), true ) ) {
                continue;
            }

            // Skip files inside quarantine directory.
            if ( false !== strpos( $abs_path, DIRECTORY_SEPARATOR . 'aegisshield-quarantine' . DIRECTORY_SEPARATOR ) ) {
                continue;
            }

            $scanned_files++;

            $analysis = $this->analyze_file( $abs_path, $rel_path );
            if ( ! $analysis ) {
                continue;
            }

            // Apply profile threshold.
            if ( isset( $analysis['score'] ) && (int) $analysis['score'] < $threshold ) {
                continue;
            }

            $results[] = $analysis;

            // Auto-quarantine for mode C and high-risk files.
            if ( 'c' === $handling_mode && isset( $analysis['score'] ) && $analysis['score'] >= 8 ) {
                $this->quarantine_file( $abs_path, $rel_path );
                $analysis['quarantined'] = true;
            }

            // Log finding.
            if ( $logger ) {
                $logger->log(
                    'malware_detected',
                    sprintf(
                        /* translators: 1: file path, 2: score */
                        __( 'Suspicious file detected during quick scan: %1$s (score %2$d)', 'aegisshield-security' ),
                        $rel_path,
                        (int) $analysis['score']
                    ),
                    array(
                        'file'      => $rel_path,
                        'score'     => (int) $analysis['score'],
                        'level'     => isset( $analysis['level'] ) ? $analysis['level'] : '',
                        'reasons'   => isset( $analysis['reasons'] ) ? $analysis['reasons'] : array(),
                        'scan_type' => $scan_type,
                    )
                );
            }
        }

        $settings->set( $section, 'last_results', $results );
        $settings->set(
            $section,
            'last_results_meta',
            array(
                'scan_type'     => $scan_type,
                'started_at'    => $started_at,
                'completed_at'  => time(),
                'file_count'    => $scanned_files,
                'suspect_count' => count( $results ),
                'partial'       => $is_partial ? 1 : 0,
            )
        );
        $settings->save();

        if ( $is_partial && $logger ) {
            $logger->log(
                'malware_quick_scan_partial',
                __( 'Quick malware scan cap reached; remaining files will be scanned in a later run.', 'aegisshield-security' ),
                array(
                    'limit'         => $limit,
                    'file_count'    => $scanned_files,
                    'suspect_count' => count( $results ),
                    'scan_type'     => $scan_type,
                )
            );
        }

        return $results;
    }

    public function run_manual_scan() {
        return $this->run_scan( 'manual' );
    }

    public function run_core_checksum_scan() {
        return $this->run_core_checksums_scan();
    }

    protected function run_core_checksums_scan() {
        $settings = $this->plugin->get_settings();
        $logger   = $this->plugin->get_logger();
        $section  = 'malware';

        $started_at = time();
        $results    = array();

        if ( ! function_exists( 'get_core_checksums' ) ) {
            $this->log_violation(
                'malware_core_checksums_unavailable',
                __( 'Core checksums scan unavailable: get_core_checksums() not found.', 'aegisshield-security' ),
                array()
            );
            return array( 'results' => array(), 'meta' => array( 'scan_type' => 'core_checksums', 'started_at' => $started_at, 'completed_at' => time() ) );
        }

        global $wp_version;
        $version = is_string( $wp_version ) ? $wp_version : '';
        $locale  = function_exists( 'get_locale' ) ? (string) get_locale() : 'en_US';

        if ( ! $version ) {
            $this->log_violation(
                'malware_core_checksums_no_version',
                __( 'Core checksums scan failed: WordPress version not detected.', 'aegisshield-security' ),
                array()
            );
            return array( 'results' => array(), 'meta' => array( 'scan_type' => 'core_checksums', 'started_at' => $started_at, 'completed_at' => time() ) );
        }

        $checksums = get_core_checksums( $version, $locale );
        if ( empty( $checksums ) || ! is_array( $checksums ) ) {
            $this->log_violation(
                'malware_core_checksums_fetch_failed',
                __( 'Core checksums scan failed: could not fetch checksums (version/locale).', 'aegisshield-security' ),
                array( 'version' => $version, 'locale' => $locale )
            );
            return array( 'results' => array(), 'meta' => array( 'scan_type' => 'core_checksums', 'started_at' => $started_at, 'completed_at' => time() ) );
        }

        // Expected format is array( 'file/path.php' => 'md5hash', ... )
        foreach ( $checksums as $rel => $expected_md5 ) {
            $rel = ltrim( (string) $rel, '/' );
            if ( '' === $rel ) {
                continue;
            }

            $abs = trailingslashit( ABSPATH ) . $rel;

            if ( ! file_exists( $abs ) || ! is_file( $abs ) ) {
                $results[] = array(
                    'file'        => $rel,
                    'score'       => 10,
                    'level'       => 'high',
                    'reasons'     => array( __( 'WordPress core file missing (checksum list).', 'aegisshield-security' ) ),
                    'expected_md5'=> (string) $expected_md5,
                    'actual_md5'  => '',
                    'modified'    => 0,
                    'scan_time'   => time(),
                    'quarantined' => false,
                    'core_file'   => true,
                );
                continue;
            }

            $actual_md5 = @md5_file( $abs );
            if ( $actual_md5 && strtolower( (string) $actual_md5 ) !== strtolower( (string) $expected_md5 ) ) {
                $results[] = array(
                    'file'        => $rel,
                    'score'       => 10,
                    'level'       => 'high',
                    'reasons'     => array( __( 'WordPress core checksum mismatch (file modified).', 'aegisshield-security' ) ),
                    'expected_md5'=> (string) $expected_md5,
                    'actual_md5'  => (string) $actual_md5,
                    'modified'    => @filemtime( $abs ) ?: 0,
                    'scan_time'   => time(),
                    'quarantined' => false,
                    'core_file'   => true,
                );
            }
        }

        // Persist like your existing scans.
        $settings->set( $section, 'last_results', $results );
        $settings->set(
            $section,
            'last_results_meta',
            array(
                'scan_type'     => 'core_checksums',
                'started_at'    => $started_at,
                'completed_at'  => time(),
                'file_count'    => is_array( $checksums ) ? count( $checksums ) : 0,
                'suspect_count' => count( $results ),
                'version'       => $version,
                'locale'        => $locale,
            )
        );
        $settings->save();

        if ( $logger ) {
            $logger->log(
                'malware_core_checksums_completed',
                array(
                    'suspect_count' => count( $results ),
                    'version'       => $version,
                    'locale'        => $locale,
                )
            );
        }

        return array( 'results' => $results, 'meta' => $settings->get( $section, 'last_results_meta', array() ) );
    }

    /**
     * Shannon entropy helper for suspicious long strings (obfuscation).
     */
    protected function shannon_entropy( $s ) {
        $s = (string) $s;
        $len = strlen( $s );
        if ( $len <= 0 ) {
            return 0.0;
        }
        $counts = count_chars( $s, 1 );
        $entropy = 0.0;
        foreach ( $counts as $c ) {
            $p = $c / $len;
            $entropy -= $p * log( $p, 2 );
        }
        return $entropy;
    }

	public function quarantine_rel( $file_rel ) {
		$file_rel = (string) $file_rel;
		$file_rel = ltrim( $file_rel, "/\\ \t\n\r\0\x0B" );

		if ( '' === $file_rel ) {
			$this->log_violation( 'malware_quarantine_failed_empty', __( 'Quarantine failed: empty file path.', 'aegisshield-security' ) );
			return false;
		}

		$absolute = trailingslashit( ABSPATH ) . $file_rel;

		if ( ! file_exists( $absolute ) ) {
			$this->log_violation(
				'malware_quarantine_failed_missing',
				__( 'Quarantine failed: file does not exist.', 'aegisshield-security' ),
				array( 'file' => $file_rel )
			);
			return false;
		}

		if ( ! is_readable( $absolute ) ) {
			$this->log_violation(
				'malware_quarantine_failed_unreadable',
				__( 'Quarantine failed: file not readable.', 'aegisshield-security' ),
				array( 'file' => $file_rel )
			);
			return false;
		}

		return (bool) $this->quarantine_file( $absolute, $file_rel );
	}

    protected function run_scan( $scan_type ) {
        $settings = $this->plugin->get_settings();
        $logger   = $this->plugin->get_logger();
        $section  = 'malware';

        $started_at    = time();
        $scanned_files = 0;

        $scan_dirs = $settings->get( $section, 'scan_dirs', array() );
        if ( ! is_array( $scan_dirs ) ) {
            $scan_dirs = array();
        }

        $is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        $custom_dirs = $is_pro ? $settings->get( $section, 'custom_dirs', array() ) : array();
        if ( ! is_array( $custom_dirs ) ) {
            $custom_dirs = array();
        }

		if ( empty( $scan_dirs ) ) {
			$fallback_dirs = $settings->get( $section, 'directories', array() );
			if ( is_array( $fallback_dirs ) && ! empty( $fallback_dirs ) ) {
				$scan_dirs = $fallback_dirs;

				$this->log_violation(
					'malware_scan_dirs_fallback_used',
					__( 'Scanner used fallback settings key "directories" because "scan_dirs" was empty.', 'aegisshield-security' ),
					array( 'scan_type' => $scan_type )
				);
			}
		}

		if ( $is_pro && empty( $custom_dirs ) ) {
			$fallback_custom = $settings->get( $section, 'custom_directories', array() );
			if ( is_array( $fallback_custom ) && ! empty( $fallback_custom ) ) {
				$custom_dirs = $fallback_custom;

				$this->log_violation(
					'malware_custom_dirs_fallback_used',
					__( 'Scanner used fallback settings key "custom_directories" because "custom_dirs" was empty.', 'aegisshield-security' ),
					array( 'scan_type' => $scan_type )
				);
			}
		}

		if ( empty( $scan_dirs ) && empty( $custom_dirs ) ) {
			$this->log_violation(
				'malware_scan_skipped_no_dirs',
				__( 'Scan skipped because no directories were selected (scan_dirs/custom_dirs empty).', 'aegisshield-security' ),
				array( 'scan_type' => $scan_type )
			);
			return array();
		}

        $handling_mode = (string) $settings->get( $section, 'handling_mode', '' );
        if ( ! in_array( $handling_mode, array( 'a', 'b', 'c' ), true ) ) {
            $handling_mode = 'a'; // Fallback to read-only if not configured.
        }

        $ignore_list = $settings->get( $section, 'ignore_list', array() );
        if ( ! is_array( $ignore_list ) ) {
            $ignore_list = array();
        }

        $profile_key = null;
        $threshold   = null;

        if ( $is_pro ) {
            $profile_key = AS_Malware_Profiles::get_selected_profile( $settings, $section );
            $threshold   = AS_Malware_Profiles::get_threshold( $profile_key );
        }

        $roots = $this->get_scan_roots( $scan_dirs, $custom_dirs );
        if ( empty( $roots ) ) {
            return array();
        }

        $results = array();
        foreach ( $roots as $root_label => $root_path ) {
            if ( ! $root_path || ! is_dir( $root_path ) ) {
                continue;
            }
            $root_path = rtrim( $root_path, DIRECTORY_SEPARATOR );

            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator(
                    $root_path,
                    \RecursiveDirectoryIterator::SKIP_DOTS
                ),
                \RecursiveIteratorIterator::SELF_FIRST
            );

            foreach ( $iterator as $file ) {
                if ( ! $file->isFile() ) {
                    continue;
                }

                $path = $file->getPathname();

                $ext = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );
                if ( ! in_array( $ext, array( 'php', 'phtml', 'php5', 'php7' ), true ) ) {
                    continue;
                }

                if ( false !== strpos( $path, DIRECTORY_SEPARATOR . 'aegisshield-quarantine' . DIRECTORY_SEPARATOR ) ) {
                    continue;
                }

                $rel_path = $this->relative_path( $path );

                if ( in_array( $rel_path, $ignore_list, true ) ) {
                    continue;
                }
                // Hash allowlist: skip if current file hash matches allowlisted hash for this path.
                $allowlist_hashes = $settings->get( $section, 'allowlist_hashes', array() );
                if ( is_array( $allowlist_hashes ) && isset( $allowlist_hashes[ $rel_path ] ) ) {
                    $current_hash = @hash_file( 'sha256', $path );
                    if ( $current_hash && hash_equals( (string) $allowlist_hashes[ $rel_path ], (string) $current_hash ) ) {
                        continue;
                    }
                }
				
                $scanned_files++;

                $analysis = $this->analyze_file( $path, $rel_path );
                if ( ! $analysis ) {
                    continue;
                }

                if ( $is_pro && null !== $threshold && isset( $analysis['score'] ) && (int) $analysis['score'] < $threshold ) {
                    continue;
                }

                $results[] = $analysis;

                if ( 'c' === $handling_mode && $analysis['score'] >= 8 ) {
                    $this->quarantine_file( $path, $rel_path );
                    $analysis['quarantined'] = true;
                }

                $logger->log(
                    'malware_detected',
                    sprintf(
                        __( 'Suspicious file detected: %1$s (score %2$d)', 'aegisshield-security' ),
                        $rel_path,
                        (int) $analysis['score']
                    ),
                    array(
                        'file'      => $rel_path,
                        'score'     => (int) $analysis['score'],
                        'level'     => $analysis['level'],
                        'reasons'   => $analysis['reasons'],
                        'scan_type' => $scan_type,
                    )
                );
            }
        }

        $settings->set( $section, 'last_results', $results );
        $settings->set(
            $section,
            'last_results_meta',
            array(
                'scan_type'     => $scan_type,
                'started_at'    => $started_at,
                'completed_at'  => time(),
                'file_count'    => $scanned_files,
                'suspect_count' => count( $results ),
            )
        );
        $settings->save();

        return $results;
    }

    protected function get_scan_roots( array $scan_dirs, array $custom_dirs = array() ) {
        $roots = array();

        if ( in_array( 'uploads', $scan_dirs, true ) ) {
            $uploads = wp_get_upload_dir();
            if ( ! empty( $uploads['basedir'] ) && is_dir( $uploads['basedir'] ) ) {
                $roots['uploads'] = $uploads['basedir'];
            }
        }

        if ( in_array( 'plugins', $scan_dirs, true ) && defined( 'WP_PLUGIN_DIR' ) && is_dir( WP_PLUGIN_DIR ) ) {
            $roots['plugins'] = WP_PLUGIN_DIR;
        }

        if ( in_array( 'themes', $scan_dirs, true ) && function_exists( 'get_theme_root' ) ) {
            $themes = get_theme_root();
            if ( $themes && is_dir( $themes ) ) {
                $roots['themes'] = $themes;
            }
        }

        if ( ! empty( $custom_dirs ) ) {
            $abs_root = wp_normalize_path( ABSPATH );

            foreach ( $custom_dirs as $dir ) {
                $dir = trim( (string) $dir );
                if ( '' === $dir ) {
                    continue;
                }

                $relative = ltrim( $dir, '/\\' );
                if ( '' === $relative ) {
                    continue;
                }

                $abs_path = wp_normalize_path( trailingslashit( $abs_root ) . $relative );

                // Ensure the path stays within the WordPress root and exists.
                if ( 0 !== strpos( $abs_path, $abs_root ) || ! is_dir( $abs_path ) ) {
                    continue;
                }

                $key = 'custom_' . md5( $relative );
                $roots[ $key ] = $abs_path;
            }
        }

        return $roots;
    }

    protected function analyze_file( $path, $rel ) {
        $contents = @file_get_contents( $path );
        if ( false === $contents ) {
            return null;
        }

        // Limit analysis to first ~256KB to avoid memory issues.
        if ( strlen( $contents ) > 262144 ) {
            $contents = substr( $contents, 0, 262144 );
        }

        $score   = 0;
        $reasons = array();

        $lower = strtolower( $contents );

        // Suspicious functions - NOTE: These strings are detection signatures only; not executed.  Malware scanners may have false-positive
        $patterns = array(
            'base64_decode' => 3,
            'gzinflate'     => 3,
            'str_rot13'     => 2,
            'eval('         => 5,
            'preg_replace'  => 2,
            'assert('       => 4,
            'shell_exec'    => 4,
            'system('       => 4,
            'passthru'      => 4,
            'exec('         => 4,
        );

        $hit_functions = 0;
        foreach ( $patterns as $needle => $value ) {
            if ( false !== strpos( $lower, $needle ) ) {
                $score    += $value;
                $hit_functions++;
                $reasons[] = sprintf(
                    /* translators: %s is function name. */
                    __( 'Uses suspicious function: %s', 'aegisshield-security' ),
                    $needle
                );
            }
        }

        // preg_replace with /e modifier.
        if ( preg_match( '#preg_replace\s*\([^)]*/e#i', $contents ) ) {
            $score    += 4;
            $reasons[] = __( 'Uses preg_replace with /e modifier', 'aegisshield-security' );
        }

        // Long encoded-looking strings.
        if ( preg_match_all( '#[A-Za-z0-9+/]{100,}={0,2}#', $contents, $matches ) ) {
            if ( ! empty( $matches[0] ) ) {
                $score    += 2;
                $reasons[] = __( 'Contains long encoded-looking strings', 'aegisshield-security' );
            }
        }

        // Suspicious wp-login clones.
        if ( preg_match( '#wp-login\.php$#i', $rel ) && strpos( $rel, 'wp-login.php' ) !== strlen( $rel ) - strlen( 'wp-login.php' ) ) {
            $score    += 5;
            $reasons[] = __( 'wp-login.php clone in non-standard location', 'aegisshield-security' );
        }

        // Randomized-looking filenames.
        $basename = basename( $rel );
        $name     = pathinfo( $basename, PATHINFO_FILENAME );
        if ( strlen( $name ) >= 10 && strlen( $name ) <= 40 && preg_match( '#^[a-z0-9]+$#i', $name ) ) {
            $score    += 3;
            $reasons[] = __( 'Random-looking filename', 'aegisshield-security' );
        }

        // Large file with obfuscated traits.
        if ( $hit_functions >= 3 ) {
            $score    += 5;
            $reasons[] = __( 'Multiple high-risk functions present', 'aegisshield-security' );
        }

        // 1) uploads-PHP rule (very high risk)
        if ( preg_match( '#^wp-content/uploads/#i', $rel ) && preg_match( '#\.(php|phtml|php5|php7|phar)$#i', $rel ) ) {
            $score    += 8;
            $reasons[] = __( 'PHP file located inside uploads directory (high risk).', 'aegisshield-security' );
        }

        // 2) Obfuscation chain (base64 + inflate/uncompress + eval/assert or function creation)
        $has_b64   = ( false !== stripos( $lower, 'base64_decode' ) );
        $has_infl  = ( false !== stripos( $lower, 'gzinflate' ) || false !== stripos( $lower, 'gzuncompress' ) );
        $has_execy = ( false !== stripos( $lower, 'eval(' ) || false !== stripos( $lower, 'assert(' ) || preg_match( '#preg_replace\s*\([^)]*/e#i', $contents ) );

        if ( $has_b64 && $has_infl && $has_execy ) {
            $score    += 10;
            $reasons[] = __( 'Obfuscation chain detected (base64_decode + gzinflate/gzuncompress + eval/assert).', 'aegisshield-security' );
        }

        // 3) Webshell keyword indicators (ONLY strong when paired with execution capability)
        $webshell_kw = false;
        if ( preg_match( '#\b(c99|r57|wso|filesman|files_man|b374k|webshell|php\s*shell)\b#i', $contents ) ) {
            $webshell_kw = true;
        }
        // Common superglobal execution patterns (high confidence)
        $super_exec = false;
        if (
            preg_match( '#\b(eval|assert)\s*\(\s*\$_(post|get|request|cookie)\b#i', $contents )
            || preg_match( '#\b(system|exec|passthru|shell_exec)\s*\(\s*\$_(post|get|request|cookie)\b#i', $contents )
            || preg_match( '#\bcall_user_func\s*\(\s*\$_(post|get|request|cookie)\b#i', $contents )
        ) {
            $super_exec = true;
        }

        if ( $super_exec ) {
            $score    += 12;
            $reasons[] = __( 'High-confidence webshell behavior (exec/eval driven by superglobal input).', 'aegisshield-security' );
        } elseif ( $webshell_kw && $hit_functions >= 1 ) {
            $score    += 6;
            $reasons[] = __( 'Possible webshell keywords present with risky functions.', 'aegisshield-security' );
        }

        // 4) Encoded blob entropy (only meaningful if combined with decode)
        $blob_hit = false;
        if ( preg_match_all( '#[A-Za-z0-9+/]{160,}={0,2}#', $contents, $matches ) && ! empty( $matches[0] ) ) {
            $sample = (string) $matches[0][0];
            $ent = $this->shannon_entropy( substr( $sample, 0, 512 ) );
            // High entropy ~ obfuscated payloads; only escalate if decode/execution indicators exist.
            if ( $ent >= 4.2 && ( $has_b64 || $has_execy ) ) {
                $score    += 4;
                $reasons[] = __( 'High-entropy encoded blob combined with decode/exec indicators.', 'aegisshield-security' );
            } else {
                $blob_hit = true; // used for dampening below
            }
        }

        // 5) False-positive dampening:
        if ( $blob_hit && ! $has_b64 && ! $has_execy && (int) $hit_functions <= 0 && $score <= 2 ) {
            $score = 0;
            $reasons = array();
        }

        if ( $score <= 0 ) {
            return null;
        }

        if ( $score >= 8 ) {
            $level = 'high';
        } elseif ( $score >= 4 ) {
            $level = 'medium';
        } else {
            $level = 'low';
        }

        return array(
            'file'       => $rel,
            'score'      => (int) $score,
            'level'      => $level,
            'reasons'    => $reasons,
            'modified'   => @filemtime( $path ) ?: 0,
            'scan_time'  => time(),
            'quarantined'=> false,
        );
    }

    public function quarantine_file( $path, $rel ) {
        if ( ! file_exists( $path ) ) {
            return false;
        }

        if ( ! defined( 'WP_CONTENT_DIR' ) ) {
            return false;
        }

        $quarantine_root = WP_CONTENT_DIR . '/aegisshield-quarantine';

        if ( ! is_dir( $quarantine_root ) && ! wp_mkdir_p( $quarantine_root ) ) {
            return false;
        }

        // Preserve relative structure inside quarantine.
        $target_path = $quarantine_root . '/' . str_replace( array( '\\', ':' ), array( '/', '' ), $rel );
        $target_dir  = dirname( $target_path );

        if ( ! is_dir( $target_dir ) && ! wp_mkdir_p( $target_dir ) ) {
            return false;
        }

        // Move the original file.
        if ( ! @rename( $path, $target_path ) ) {
            return false;
        }

        // Create harmless placeholder.
        $placeholder  = "<?php\n";
        $placeholder .= "// AegisShield Security placeholder.\n";
        $placeholder .= "// Original file quarantined on " . gmdate( 'c' ) . "\n";
        $placeholder .= "exit;\n";

        @file_put_contents( $path, $placeholder );

        return true;
    }

	public function log_violation( $event, $message, $context = array() ) {
		$logger = $this->plugin ? $this->plugin->get_logger() : null;

		$payload = is_array( $context ) ? $context : array();
		$payload['module'] = 'malware';

		if ( $logger ) {
			$logger->log( (string) $event, (string) $message, $payload );
			return;
		}

		// Fallback for environments where logger isn't available.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( '[AegisShield Malware][' . $event . '] ' . $message . ' ' . wp_json_encode( $payload ) );
		}
	}

    protected function relative_path( $path ) {
        $path     = str_replace( array( '\\', '/' ), DIRECTORY_SEPARATOR, $path );
        $abs_path = str_replace( array( '\\', '/' ), DIRECTORY_SEPARATOR, ABSPATH );

        if ( 0 === strpos( $path, $abs_path ) ) {
            $rel = substr( $path, strlen( $abs_path ) );
        } else {
            $rel = $path;
        }

        $rel = ltrim( str_replace( array( '\\', DIRECTORY_SEPARATOR ), '/', $rel ), '/' );

        return $rel;
    }
}