<?php
namespace AegisShield\Modules\Malware;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Malware_Incidents {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    protected function get_activity_table_name() {
        global $wpdb;

        $logger = $this->plugin->get_logger();
        if ( $logger && method_exists( $logger, 'get_table_name' ) ) {
            $table = $logger->get_table_name();
            if ( ! empty( $table ) ) {
                return $table;
            }
        }


        return $wpdb->prefix . 'aegisshield_activity_log';
    }


    public function get_recent_incidents( $limit = 10 ) {
        global $wpdb;

        $limit = max( 1, absint( $limit ) );

        $table = $this->get_activity_table_name();

        $rows = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, event_time, message, context FROM {$table} WHERE event_type = %s ORDER BY event_time DESC LIMIT %d",
                'malware_detected',
                $limit
            ),
            ARRAY_A
        );

        if ( ! $rows ) {
            return array();
        }

        $incidents = array();

        foreach ( $rows as $row ) {
            $ctx = array();
            if ( ! empty( $row['context'] ) ) {
                $decoded = json_decode( $row['context'], true );
                if ( is_array( $decoded ) ) {
                    $ctx = $decoded;
                }
            }

            $file    = isset( $ctx['file'] ) ? (string) $ctx['file'] : '';
            $score   = isset( $ctx['score'] ) ? (int) $ctx['score'] : 0;
            $level   = isset( $ctx['level'] ) ? (string) $ctx['level'] : '';
            $reasons = isset( $ctx['reasons'] ) && is_array( $ctx['reasons'] ) ? $ctx['reasons'] : array();
            $scan_type = isset( $ctx['scan_type'] ) ? (string) $ctx['scan_type'] : '';

            $scan_ts = strtotime( $row['event_time'] );
            if ( ! $scan_ts ) {
                $scan_ts = time();
            }

            $incidents[] = array(
                'incident_id' => (string) $row['id'],
                'file'        => $file,
                'score'       => $score,
                'level'       => $level,
                'reasons'     => $reasons,
                'scan_type'   => $scan_type,
                'scan_time'   => $scan_ts,
                'scan_time_gmt' => $row['event_time'],
                'raw_message' => isset( $row['message'] ) ? (string) $row['message'] : '',
            );
        }

        return $incidents;
    }

    public function get_incident_by_id( $incident_id ) {
        $incidents = $this->get_recent_incidents( 25 );
        foreach ( $incidents as $incident ) {
            if ( (string) $incident['incident_id'] === (string) $incident_id ) {
                return $incident;
            }
        }

        return null;
    }

    public function build_timeline_for_incident( array $incident ) {
        global $wpdb;

        $table = $this->get_activity_table_name();

        $file      = isset( $incident['file'] ) ? (string) $incident['file'] : '';
        $scan_ts   = isset( $incident['scan_time'] ) ? (int) $incident['scan_time'] : time();
        $window    = 2 * HOUR_IN_SECONDS; // +/- 2 hours window.

        $from_ts = $scan_ts - $window;
        $to_ts   = $scan_ts + $window;

        $from = gmdate( 'Y-m-d H:i:s', $from_ts );
        $to   = gmdate( 'Y-m-d H:i:s', $to_ts );

        $rows = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE event_time BETWEEN %s AND %s ORDER BY event_time ASC",
                $from,
                $to
            ),
            ARRAY_A
        );

        if ( ! $rows ) {
            return array();
        }

        $timeline = array();

        foreach ( $rows as $row ) {
            $event_type = isset( $row['event_type'] ) ? (string) $row['event_type'] : '';
            $message    = isset( $row['message'] ) ? (string) $row['message'] : '';

            $ctx = array();
            if ( ! empty( $row['context'] ) ) {
                $decoded = json_decode( $row['context'], true );
                if ( is_array( $decoded ) ) {
                    $ctx = $decoded;
                }
            }

            $category = 'activity';

            if ( in_array( $event_type, array( 'malware_detected', 'malware_mark_safe', 'malware_quarantined' ), true ) ) {
                $category = 'malware';
            } elseif ( in_array( $event_type, array( 'file_monitor', 'file_new', 'file_modified', 'file_deleted' ), true ) ) {
                $category = 'file_monitor';
            } elseif ( in_array( $event_type, array( 'login_failed', 'login_success', 'login_ip_unblocked' ), true ) ) {
                $category = 'login';
            }

            $primary = false;
            if ( $file && isset( $ctx['file'] ) && (string) $ctx['file'] === $file ) {
                $primary = true;
            }

            $timeline[] = array(
                'event_time' => isset( $row['event_time'] ) ? $row['event_time'] : '',
                'event_type' => $event_type,
                'category'   => $category,
                'message'    => $message,
                'context'    => $ctx,
                'primary'    => $primary,
            );
        }

        return $timeline;
    }
}
