<?php
namespace AegisShield\Reports;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Incident_Report {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function render_html( array $incident, array $timeline ) {
        $site_name = get_bloginfo( 'name' );
        $site_url  = home_url( '/' );

        $file      = isset( $incident['file'] ) ? (string) $incident['file'] : '';
        $score     = isset( $incident['score'] ) ? (int) $incident['score'] : 0;
        $level     = isset( $incident['level'] ) ? (string) $incident['level'] : '';
        $reasons   = isset( $incident['reasons'] ) && is_array( $incident['reasons'] ) ? $incident['reasons'] : array();
        $scan_type = isset( $incident['scan_type'] ) ? (string) $incident['scan_type'] : '';
        $scan_ts   = isset( $incident['scan_time'] ) ? (int) $incident['scan_time'] : time();

        $scan_label = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $scan_ts );

        $level_label = '';
        if ( 'high' === $level ) {
            $level_label = __( 'High', 'aegisshield-security' );
        } elseif ( 'medium' === $level ) {
            $level_label = __( 'Medium', 'aegisshield-security' );
        } elseif ( 'low' === $level ) {
            $level_label = __( 'Low', 'aegisshield-security' );
        }

        $version = defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : '';

        // Collect actions taken from the timeline.
        $actions = array();
        foreach ( $timeline as $entry ) {
            $etype = isset( $entry['event_type'] ) ? $entry['event_type'] : '';
            if ( in_array( $etype, array( 'malware_mark_safe', 'malware_quarantined', 'malware_restored' ), true ) ) {
                $actions[] = $entry;
            }
        }

        ob_start();
        ?>
        <div class="wrap aegisshield-incident-report" style="max-width:960px;margin-top:20px;">
            <h1 style="margin-bottom:4px;"><?php esc_html_e( 'AegisShield Security – Incident Report', 'aegisshield-security' ); ?></h1>
            <p style="margin-top:0;color:#555;">
                <?php
                printf(
                    esc_html__( 'Site: %1$s (%2$s)', 'aegisshield-security' ),
                    esc_html( $site_name ),
                    esc_url( $site_url )
                );
                ?>
                <br />
                <?php
                /* translators: %s: date/time */
                printf(
                    esc_html__( 'Generated: %s', 'aegisshield-security' ),
                    esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) ) )
                );
                ?>
            </p>

            <hr />

            <h2><?php esc_html_e( 'Incident Summary', 'aegisshield-security' ); ?></h2>
            <table class="widefat striped" style="margin-top:8px;">
                <tbody>
                    <tr>
                        <th style="width:180px;"><?php esc_html_e( 'File', 'aegisshield-security' ); ?></th>
                        <td><code><?php echo $file ? esc_html( $file ) : esc_html__( '(Unknown file)', 'aegisshield-security' ); ?></code></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Score / Severity', 'aegisshield-security' ); ?></th>
                        <td>
                            <?php
                            /* translators: 1: score, 2: level label */
                            printf(
                                esc_html__( 'Score %1$d – %2$s', 'aegisshield-security' ),
                                $score,
                                $level_label ? esc_html( $level_label ) : esc_html__( 'Unknown', 'aegisshield-security' )
                            );
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Scan type', 'aegisshield-security' ); ?></th>
                        <td><?php echo $scan_type ? esc_html( ucfirst( $scan_type ) ) : esc_html__( '(Not recorded)', 'aegisshield-security' ); ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Scan time', 'aegisshield-security' ); ?></th>
                        <td><?php echo esc_html( $scan_label ); ?></td>
                    </tr>
                    <?php if ( ! empty( $reasons ) ) : ?>
                        <tr>
                            <th><?php esc_html_e( 'Heuristics / Reasons', 'aegisshield-security' ); ?></th>
                            <td>
                                <ul style="margin:0;padding-left:18px;">
                                    <?php foreach ( $reasons as $reason ) : ?>
                                        <li><?php echo esc_html( $reason ); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>

            <h2 style="margin-top:24px;"><?php esc_html_e( 'Actions Taken', 'aegisshield-security' ); ?></h2>
            <?php if ( empty( $actions ) ) : ?>
                <p><?php esc_html_e( 'No follow-up actions have been recorded for this incident yet.', 'aegisshield-security' ); ?></p>
            <?php else : ?>
                <table class="widefat striped" style="margin-top:8px;">
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Time', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Action', 'aegisshield-security' ); ?></th>
                            <th><?php esc_html_e( 'Details', 'aegisshield-security' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $actions as $entry ) : ?>
                            <?php
                            $etime   = isset( $entry['event_time'] ) ? strtotime( $entry['event_time'] ) : false;
                            $label   = $etime ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $etime ) : esc_html__( '(Unknown time)', 'aegisshield-security' );
                            $etype   = isset( $entry['event_type'] ) ? $entry['event_type'] : '';
                            $message = isset( $entry['message'] ) ? $entry['message'] : '';
                            ?>
                            <tr>
                                <td><?php echo esc_html( $label ); ?></td>
                                <td><?php echo esc_html( $etype ); ?></td>
                                <td><?php echo esc_html( $message ); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>

            <h2 style="margin-top:24px;"><?php esc_html_e( 'Attack Story Timeline', 'aegisshield-security' ); ?></h2>
            <?php if ( empty( $timeline ) ) : ?>
                <p><?php esc_html_e( 'No related events were found around this incident. This may occur if logging was recently enabled or log retention is short.', 'aegisshield-security' ); ?></p>
            <?php else : ?>
                <ul style="list-style:none;margin:0;padding:0;border-left:2px solid #ccd0d4;">
                    <?php foreach ( $timeline as $entry ) : ?>
                        <?php
                        $event_ts   = isset( $entry['event_time'] ) ? strtotime( $entry['event_time'] ) : false;
                        $time_label = $event_ts ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $event_ts ) : esc_html__( '(Unknown time)', 'aegisshield-security' );
                        $category   = isset( $entry['category'] ) ? $entry['category'] : 'activity';
                        $etype      = isset( $entry['event_type'] ) ? $entry['event_type'] : '';
                        $message    = isset( $entry['message'] ) ? $entry['message'] : '';
                        $primary    = ! empty( $entry['primary'] );
                        ?>
                        <li style="margin:0 0 16px 0;padding:0 0 0 16px;position:relative;">
                            <span style="position:absolute;left:-7px;top:4px;width:10px;height:10px;border-radius:50%;background:
                                <?php
                                switch ( $category ) {
                                    case 'malware':
                                        echo '#dc3232'; // red.
                                        break;
                                    case 'file_monitor':
                                        echo '#ffb900'; // yellow.
                                        break;
                                    case 'login':
                                        echo '#00a0d2'; // blue.
                                        break;
                                    default:
                                        echo '#82878c'; // gray.
                                        break;
                                }
                                ?>
                            "></span>
                            <div style="font-size:12px;color:#777;">
                                <?php echo esc_html( $time_label ); ?>
                            </div>
                            <div style="font-weight:600;margin-top:2px;">
                                <?php
                                switch ( $category ) {
                                    case 'malware':
                                        esc_html_e( 'Malware', 'aegisshield-security' );
                                        break;
                                    case 'file_monitor':
                                        esc_html_e( 'File Monitor', 'aegisshield-security' );
                                        break;
                                    case 'login':
                                        esc_html_e( 'Login Guard', 'aegisshield-security' );
                                        break;
                                    default:
                                        esc_html_e( 'Activity', 'aegisshield-security' );
                                        break;
                                }
                                ?>
                                <?php if ( $etype ) : ?>
                                    <span style="font-weight:400;color:#555;">
                                        (<?php echo esc_html( $etype ); ?>)
                                    </span>
                                <?php endif; ?>
                                <?php if ( $primary ) : ?>
                                    <span style="display:inline-block;margin-left:6px;padding:1px 5px;font-size:10px;border-radius:3px;background:#f0f6fc;border:1px solid #ccd0d4;">
                                        <?php esc_html_e( 'This file', 'aegisshield-security' ); ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                            <?php if ( $message ) : ?>
                                <div style="margin-top:2px;">
                                    <?php echo esc_html( $message ); ?>
                                </div>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>

            <p style="margin-top:32px;font-size:11px;color:#777;">
                <?php
                /* translators: %s: plugin version */
                printf(
                    esc_html__( 'Report generated by AegisShield Security %s.', 'aegisshield-security' ),
                    esc_html( $version )
                );
                ?>
            </p>
        </div>
        <?php

        return (string) ob_get_clean();
    }
}
