<?php
/**
 * Plugin Name: AegisSitemap
 * Description: XML sitemap engine by Aegisify — fast, cache-friendly sitemap index, HTML sitemap, images, and optional video/news sitemaps.
 * Version: 1.2.12
 * Requires at least: 6.8
 * Requires PHP: 8.2
 * Author:      Aegisify | AegisWAF | AegisShield | AegisSEO
 * Author URI:  https://aegisify.com/
 * Text Domain: aegissitemap
 * Domain Path: /languages
 */
if (!defined('ABSPATH')) { exit; }

define('AEGISSITEMAP_VERSION', '1.2.12');
define('AEGISSITEMAP_FILE', __FILE__);
define('AEGISSITEMAP_DIR', plugin_dir_path(__FILE__));
define('AEGISSITEMAP_URL', plugin_dir_url(__FILE__));
define('AEGISSITEMAP_BASENAME', plugin_basename(__FILE__));
define('AEGISSITEMAP_SLUG', 'aegissitemap');

define('AEGISSITEMAP_CLM_BASE', 'https://aegisify.com');
define('AEGISSITEMAP_CLM_FREE_REG_ENDPOINT', AEGISSITEMAP_CLM_BASE . '/wp-json/chue-license/v1/aegissitemap/free-register');
define('AEGISSITEMAP_CLM_UPDATE_ENDPOINT',    AEGISSITEMAP_CLM_BASE . '/wp-json/chue-license/v1/aegissitemap/update-check');
define('AEGISSITEMAP_CLM_ACTIVATE_ENDPOINT',   AEGISSITEMAP_CLM_BASE . '/wp-json/chue-license/v1/aegissitemap/activate');
define('AEGISSITEMAP_CLM_DEACTIVATE_ENDPOINT', AEGISSITEMAP_CLM_BASE . '/wp-json/chue-license/v1/aegissitemap/deactivate');
define('AEGISSITEMAP_CLM_STATUS_ENDPOINT',     AEGISSITEMAP_CLM_BASE . '/wp-json/chue-license/v1/aegissitemap/status');

// Daily license + update ping (keeps PRO state fresh even if WP update checks are delayed).
define('AEGISSITEMAP_CLM_DAILY_HOOK', 'aegissitemap_clm_daily_check');

function aegissitemap_update_log(string $message, array $context = array()) : void {
    $line = '[' . gmdate('Y-m-d H:i:s') . ' UTC] ' . $message;
    if (!empty($context)) {
        $line .= ' | ' . wp_json_encode($context);
    }
    $line .= "\n";

    $dir  = trailingslashit(AEGISSITEMAP_DIR) . 'includes/log/';
    $file = $dir . 'aegissitemap.log';

    if (!is_dir($dir)) {
        if (function_exists('wp_mkdir_p')) {
            @wp_mkdir_p($dir);
        }
    }

    if (is_dir($dir) && (is_writable($dir) || (file_exists($file) && is_writable($file)))) {
        @file_put_contents($file, $line, FILE_APPEND);
        return;
    }

    if (function_exists('error_log')) {
        error_log('[AegisSiteMap] ' . trim($line));
    }
}

function aegissitemap_clm_register_free_install(bool $force = false) : array {

    if (!$force && get_transient('aegissitemap_clm_free_reg_sent')) {
        return array(
            'success' => false,
            'status'  => 'backoff',
            'message' => 'Backoff active (already attempted recently).',
        );
    }

    $site_url = home_url();
    $domain   = wp_parse_url($site_url, PHP_URL_HOST);

    $payload = array(
        'product'        => AEGISSITEMAP_SLUG,
        'slug'           => AEGISSITEMAP_SLUG,
        'email'          => (string) get_option('admin_email'),
        'site_url'       => (string) $site_url,
        'domain'         => (string) $domain,
        'plugin_version' => defined('AEGISSITEMAP_VERSION') ? (string) AEGISSITEMAP_VERSION : 'unknown',
        'wp_version'     => (string) get_bloginfo('version'),
        'php_version'    => (string) PHP_VERSION,
    );

    $args = array(
        'timeout' => 15,
        'headers' => array('Content-Type' => 'application/json'),
        'body'    => wp_json_encode($payload),
    );

    $resp = wp_remote_post(AEGISSITEMAP_CLM_FREE_REG_ENDPOINT, $args);

	if (is_wp_error($resp)) {

		$msg = 'CLM free-register WP_Error: ' . $resp->get_error_message();
		update_option('aegissitemap_clm_free_reg_last_error', $msg);

		if (function_exists('aegissitemap_update_log')) {
			aegissitemap_update_log('CLM free-register failed (WP_Error).', array(
				'endpoint' => AEGISSITEMAP_CLM_FREE_REG_ENDPOINT,
				'error'    => $resp->get_error_message(),
				'payload'  => $payload,
			));
		}

		return array(
			'success' => false,
			'status'  => 'wp_error',
			'message' => $resp->get_error_message(),
		);
	}

    $code = (int) wp_remote_retrieve_response_code($resp);
    $body = (string) wp_remote_retrieve_body($resp);

    $backoff = ($code >= 500) ? HOUR_IN_SECONDS : DAY_IN_SECONDS;
    set_transient('aegissitemap_clm_free_reg_sent', 1, $backoff);

    aegissitemap_update_log('CLM free-register response.', array(
        'endpoint'  => AEGISSITEMAP_CLM_FREE_REG_ENDPOINT,
        'http_code' => $code,
        'body'      => $body,
        'backoff_s' => $backoff,
    ));

    if ($code >= 200 && $code < 300) {
        return array(
            'success' => true,
            'status'  => 'ok',
            'message' => 'Registered',
        );
    }

    return array(
        'success' => false,
        'status'  => 'http_' . $code,
        'message' => 'Non-2xx response from CLM',
        'body'    => $body,
    );
}

function aegissitemap_http_json_post(string $url, array $payload, int $timeout = 20): array {
    $resp = wp_remote_post($url, array(
        'timeout' => $timeout,
        'headers' => array('Content-Type' => 'application/json'),
        'body'    => wp_json_encode($payload),
    ));
    if (is_wp_error($resp)) {
        return array('success' => false, 'status' => 'wp_error', 'message' => $resp->get_error_message());
    }
    $code = (int) wp_remote_retrieve_response_code($resp);
    $body = (string) wp_remote_retrieve_body($resp);
    if ($code < 200 || $code >= 300) {
        return array('success' => false, 'status' => 'http_' . $code, 'message' => $body);
    }
    $data = json_decode($body, true);
    if (!is_array($data)) {
        return array('success' => false, 'status' => 'bad_json', 'message' => $body);
    }
    return array('success' => true, 'status' => 'ok', 'data' => $data);
}

	require_once AEGISSITEMAP_DIR . 'includes/class-aegissitemap-plugin.php';

	function aegissitemap_clm_check_for_updates( $transient ) {
		if ( ! is_object( $transient ) || empty( $transient->checked ) || ! is_array( $transient->checked ) ) {
			return $transient;
		}

		if ( ! defined( 'AEGISSITEMAP_BASENAME' ) || empty( $transient->checked[ AEGISSITEMAP_BASENAME ] ) ) {
			return $transient;
		}

		$cache_key = 'aegissitemap_clm_update_cache';
		$last_key  = 'aegissitemap_clm_update_last_check';
		$last_check = (int) get_site_option( $last_key, 0 );
		$within_day = ( $last_check > 0 && ( time() - $last_check ) < DAY_IN_SECONDS );

		$cached = get_site_transient( $cache_key );
		if ( is_object( $cached ) && ! empty( $cached->new_version ) ) {

			$installed_version = (string) $transient->checked[ AEGISSITEMAP_BASENAME ];
			if ( $installed_version !== '' && version_compare( $installed_version, (string) $cached->new_version, '>=' ) ) {
				delete_site_transient( $cache_key );
			} else {
				$transient->response[ AEGISSITEMAP_BASENAME ] = $cached;
				return $transient;
			}
		}

		// If we already checked within the last 24h and there's no cached update object, do not hammer CLM.
		if ( $within_day ) {
			return $transient;
		}

		$current_version = (string) $transient->checked[ AEGISSITEMAP_BASENAME ];

		$payload = array(
			'version'     => $current_version,
			'site_url'    => home_url(),
			'license_key' => (string) get_option( 'aegissitemap_license_key', '' ),
			'email'       => (string) get_option( 'aegissitemap_license_email', get_option( 'admin_email', '' ) ),
		);

		// Mark the attempt time before the remote call to ensure a hard 24h ceiling even on failures.
		update_site_option( $last_key, time() );

		$result = aegissitemap_http_json_post( AEGISSITEMAP_CLM_UPDATE_ENDPOINT, $payload, 20 );

		if ( empty( $result['success'] ) || empty( $result['data'] ) || ! is_array( $result['data'] ) ) {
			return $transient;
		}

		$data = $result['data'];

		if ( empty( $data['success'] ) || empty( $data['update_available'] ) ) {
			return $transient;
		}
		if ( empty( $data['version'] ) || empty( $data['download_url'] ) ) {
			return $transient;
		}

		$update              = new stdClass();
		$update->slug        = AEGISSITEMAP_SLUG;
		$update->plugin      = AEGISSITEMAP_BASENAME;
		$update->new_version = (string) $data['version'];
		$update->package     = (string) $data['download_url'];
		$update->url         = ! empty( $data['homepage'] ) ? (string) $data['homepage'] : 'https://aegisify.com';

		$transient->response[ AEGISSITEMAP_BASENAME ] = $update;

		// Cache the update object for up to 24h (WP itself checks updates ~2x daily).
		set_site_transient( $cache_key, $update, DAY_IN_SECONDS );

		return $transient;
	}

	add_filter( 'pre_set_site_transient_update_plugins', 'aegissitemap_clm_check_for_updates', 25, 1 );
	
	add_action('upgrader_process_complete', function($upgrader, $hook_extra) {
		if (!empty($hook_extra['action']) && $hook_extra['action'] === 'update'
			&& !empty($hook_extra['type']) && $hook_extra['type'] === 'plugin'
			&& !empty($hook_extra['plugins']) && is_array($hook_extra['plugins'])
			&& in_array(AEGISSITEMAP_BASENAME, $hook_extra['plugins'], true)
		) {
			delete_site_transient('aegissitemap_clm_update_cache');
		}
	}, 10, 2);

	function aegissitemap_clm_plugins_api( $result, $action, $args ) {
		if ( $action !== 'plugin_information' ) {
			return $result;
		}
		if ( empty( $args->slug ) || $args->slug !== AEGISSITEMAP_SLUG ) {
			return $result;
		}

		$info = new stdClass();
		$info->name     = 'AegisSitemap';
		$info->slug     = AEGISSITEMAP_SLUG;
		$info->version  = AEGISSITEMAP_VERSION;
		$info->author   = 'Aegisify';
		$info->homepage = 'https://aegisify.com/';
		$info->sections = array(
			'description' => 'XML sitemap engine by Aegisify — fast, cache-friendly sitemap index, HTML sitemap, images, and optional video/news sitemaps.',
		);

		return $info;
	}
	add_filter( 'plugins_api', 'aegissitemap_clm_plugins_api', 10, 3 );

	register_activation_hook(__FILE__, array('AegisSitemap\\Plugin', 'activate_static'));

	register_activation_hook(__FILE__, function () {
		require_once AEGISSITEMAP_DIR . 'includes/utils/class-aegissitemap-license.php';
		$lic = new \AegisSitemap\Utils\License();
		$lic->maybe_send_free_register(AEGISSITEMAP_VERSION);

		if (!wp_next_scheduled('aegissitemap_clm_free_register_retry')) {
			wp_schedule_single_event(time() + 120, 'aegissitemap_clm_free_register_retry');
		}

		// Daily: refresh license status + update cache once per day.
		if ( defined('AEGISSITEMAP_CLM_DAILY_HOOK') && ! wp_next_scheduled( AEGISSITEMAP_CLM_DAILY_HOOK ) ) {
			wp_schedule_event( time() + 300, 'daily', AEGISSITEMAP_CLM_DAILY_HOOK );
		}
	});

	add_action('aegissitemap_clm_free_register_retry', function () {
		require_once AEGISSITEMAP_DIR . 'includes/utils/class-aegissitemap-license.php';
		$lic = new \AegisSitemap\Utils\License();
		$lic->maybe_send_free_register(AEGISSITEMAP_VERSION);
	});

	add_action('init', function () {
		require_once AEGISSITEMAP_DIR . 'includes/utils/class-aegissitemap-license.php';
		$lic = new \AegisSitemap\Utils\License();
		$lic->maybe_send_free_register(AEGISSITEMAP_VERSION);
	});

	register_deactivation_hook(__FILE__, array('AegisSitemap\\Plugin', 'deactivate_static'));

	// Unschedule daily CLM ping.
	register_deactivation_hook(__FILE__, function () {
		if ( ! defined('AEGISSITEMAP_CLM_DAILY_HOOK') ) { return; }
		$ts = wp_next_scheduled( AEGISSITEMAP_CLM_DAILY_HOOK );
		if ( $ts ) {
			wp_unschedule_event( $ts, AEGISSITEMAP_CLM_DAILY_HOOK );
		}
	});

	add_action( defined('AEGISSITEMAP_CLM_DAILY_HOOK') ? AEGISSITEMAP_CLM_DAILY_HOOK : 'aegissitemap_clm_daily_check', function () {
		require_once AEGISSITEMAP_DIR . 'includes/utils/class-aegissitemap-license.php';
		$lic = new \AegisSitemap\Utils\License();
		// Refresh PRO state.
		$lic->check_now( defined('AEGISSITEMAP_VERSION') ? (string) AEGISSITEMAP_VERSION : '' );
		// Refresh update cache (used by the update transient filter).
		delete_site_transient( 'aegissitemap_clm_update_cache' );
		if ( function_exists( 'wp_update_plugins' ) ) {
			wp_update_plugins();
		} else {
			delete_site_transient( 'update_plugins' );
		}
	});

	add_action('plugins_loaded', function() { AegisSitemap\Plugin::instance(); });

	function aegissitemap() { return AegisSitemap\Plugin::instance(); }
