(function($){
    'use strict';

    function getGutenbergContent() {
        try {
            if (window.wp && wp.data && wp.data.select) {
                var ed = wp.data.select('core/editor');
                if (ed) {
                    var content = ed.getEditedPostContent ? ed.getEditedPostContent() : '';
                    var title = ed.getEditedPostAttribute ? ed.getEditedPostAttribute('title') : '';
                    return { content: content || '', title: title || '' };
                }
            }
        } catch(e) {}
        return { content: '', title: '' };
    }

    function getClassicContent() {
        var c = '';
        if (typeof window.tinyMCE !== 'undefined' && window.tinyMCE.activeEditor && !window.tinyMCE.activeEditor.isHidden()) {
            c = window.tinyMCE.activeEditor.getContent() || '';
        } else if ($('#content').length) {
            c = $('#content').val() || '';
        }
        return c;
    }

    function truncate(str, n) {
        str = (str || '').toString().trim();
        if (str.length <= n) return str;
        return str.substring(0, n - 1).trim() + '…';
    }

    function pickFirstNonEmpty() {
        for (var i=0;i<arguments.length;i++) {
            var v = (arguments[i] || '').toString().trim();
            if (v !== '') return v;
        }
        return '';
    }

    function computeDescFallback(html) {
        var text = $('<div>').html(html || '').text().replace(/\s{2,}/g,' ').trim();
        return truncate(text, 155);
    }

    function updatePreviews($box) {
        var postId = parseInt($box.data('post-id') || 0, 10);
        var permalink = ($box.data('permalink') || '').toString();

        var focus = $('input[name="aegissitemap_focus_phrase"]').val() || '';
        var seoTitle = $('input[name="aegissitemap_title"]').val() || '';
        var seoDesc  = $('textarea[name="aegissitemap_description"]').val() || '';

        var ogTitle = $('input[name="aegissitemap_og_title"]').val() || '';
        var ogDesc  = $('textarea[name="aegissitemap_og_description"]').val() || '';
        var ogImg   = $('input[name="aegissitemap_og_image"]').val() || '';

        var twTitle = $('input[name="aegissitemap_twitter_title"]').val() || '';
        var twDesc  = $('textarea[name="aegissitemap_twitter_description"]').val() || '';
        var twImg   = $('input[name="aegissitemap_twitter_image"]').val() || '';

        var gb = getGutenbergContent();
        var content = gb.content || getClassicContent();
        var postTitle = pickFirstNonEmpty(gb.title, $('input#title').val(), document.title);

        var finalTitle = pickFirstNonEmpty(seoTitle, postTitle);
        var finalDesc  = pickFirstNonEmpty(seoDesc, computeDescFallback(content));

        // SERP
        $box.find('.aegissitemap-serp-title').text(truncate(finalTitle, 70));
        $box.find('.aegissitemap-serp-url').text((permalink || '').replace(/^https?:\/\//,''));
        $box.find('.aegissitemap-serp-desc').text(truncate(finalDesc, 160));

        // Social (OG default)
        var site = (permalink || '').replace(/^https?:\/\//,'').split('/')[0] || window.location.host;
        var social = $box.find('.aegissitemap-social-preview').attr('data-social') || 'og';

        var sTitle = (social === 'twitter') ? pickFirstNonEmpty(twTitle, ogTitle, finalTitle) : pickFirstNonEmpty(ogTitle, finalTitle);
        var sDesc  = (social === 'twitter') ? pickFirstNonEmpty(twDesc, ogDesc, finalDesc) : pickFirstNonEmpty(ogDesc, finalDesc);
        var sImg   = (social === 'twitter') ? pickFirstNonEmpty(twImg, ogImg) : ogImg;

        $box.find('.aegissitemap-social-site').text(site);
        $box.find('.aegissitemap-social-title').text(truncate(sTitle, 90));
        $box.find('.aegissitemap-social-desc').text(truncate(sDesc, 140));
        if (sImg) {
            $box.find('.aegissitemap-social-img').html('<img src="'+encodeURI(sImg)+'" alt="" />');
        } else {
            $box.find('.aegissitemap-social-img').html('<div class="aegissitemap-social-img-placeholder">No image set</div>');
        }

        return { post_id: postId, focus: focus, title: finalTitle, description: finalDesc, content: content };
    }

    var debounceTimer = null;
    function requestAnalysis($box) {
        if (!window.AegisSEOAdmin || !AegisSEOAdmin.ajax_url) return;

        var payload = updatePreviews($box);

        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(function(){
            $.post(AegisSEOAdmin.ajax_url, {
                action: 'aegissitemap_analyze_content',
                nonce: AegisSEOAdmin.nonce,
                post_id: payload.post_id,
                focus: payload.focus,
                title: payload.title,
                description: payload.description,
                content: payload.content
            }).done(function(resp){
                if (!resp || !resp.success || !resp.data) return;

                var score = parseInt(resp.data.score || 0, 10);
                var actions = resp.data.actions || [];

                var $badge = $box.find('.aegissitemap-score-badge');
                var $label = $box.find('.aegissitemap-score-label');
                var $list  = $box.find('.aegissitemap-actions-list');

                $badge.text(score);
                $badge.removeClass('is-bad is-ok is-good');
                if (score < 50) $badge.addClass('is-bad');
                else if (score < 80) $badge.addClass('is-ok');
                else $badge.addClass('is-good');

                $label.text(score < 50 ? 'Needs work' : (score < 80 ? 'Good progress' : 'Strong'));

                $list.empty();
                if (!actions.length) {
                    $list.append('<li><strong>Nice.</strong> Nothing urgent detected.</li>');
                    return;
                }
                actions.forEach(function(a){
                    var pri = parseInt(a.priority || 3, 10);
                    var priLabel = (pri === 1 ? 'High' : (pri === 2 ? 'Med' : 'Low'));
                    var hint = (a.hint || '').toString();
                    $list.append(
                        '<li><span class="aegissitemap-pri pri-'+pri+'">'+priLabel+'</span> ' +
                        '<strong>'+escapeHtml(a.label || '')+'</strong> ' +
                        (a.detail ? '<span class="aegissitemap-detail">('+escapeHtml(a.detail)+')</span>' : '') +
                        (hint ? '<div class="aegissitemap-hint">'+escapeHtml(hint)+'</div>' : '') +
                        '</li>'
                    );
                });
            });
        }, 450);
    }

    function escapeHtml(s) {
        return (s || '').toString()
            .replace(/&/g,'&amp;')
            .replace(/</g,'&lt;')
            .replace(/>/g,'&gt;')
            .replace(/"/g,'&quot;')
            .replace(/'/g,'&#039;');
    }

    function initMetabox() {
        var $box = $('.aegissitemap-metabox-phaseA');
        if (!$box.length) return;

        $box.on('click', '.aegissitemap-preview-toggle [data-device]', function(){
            var d = $(this).data('device');
            $box.find('.aegissitemap-preview-toggle [data-device]').removeClass('is-active');
            $(this).addClass('is-active');
            $box.find('.aegissitemap-serp-preview').attr('data-device', d);
        });

        $box.on('click', '.aegissitemap-preview-toggle [data-social]', function(){
            var s = $(this).data('social');
            $box.find('.aegissitemap-preview-toggle [data-social]').removeClass('is-active');
            $(this).addClass('is-active');
            $box.find('.aegissitemap-social-preview').attr('data-social', s);
            requestAnalysis($box);
        });

        $(document).on('input change', 'input[name^="aegissitemap_"], textarea[name^="aegissitemap_"]', function(){
            requestAnalysis($box);
        });

        if (window.wp && wp.data && wp.data.subscribe) {
            var last = '';
            wp.data.subscribe(function(){
                var gb = getGutenbergContent();
                var now = (gb.title || '') + '|' + (gb.content || '').substring(0, 200);
                if (now !== last) {
                    last = now;
                    requestAnalysis($box);
                }
            });
        } else {
            setInterval(function(){
                requestAnalysis($box);
            }, 2500);
        }

        requestAnalysis($box);
    }

    $(function(){
        initMetabox();
    });

    $(document).on('submit', '#aegissitemap-bulk-form', function(e){
        e.preventDefault();
        var $form = $(this);
        var $btn = $('#aegissitemap-bulk-save-btn');
        var $saving = $('#aegissitemap-bulk-saving');
        var $notice = $('#aegissitemap-bulk-notice');

        var nonce = $form.data('aegissitemap-ajax-nonce') || '';
        if (!nonce) {
            this.submit();
            return;
        }

        $btn.prop('disabled', true);
        $saving.show();
        $notice.empty();

        var data = $form.serializeArray();
        data.push({name:'action', value:'aegissitemap_bulk_save_ajax'});
        data.push({name:'nonce', value: nonce});

        $.post((window.AegisSEOAdmin && AegisSEOAdmin.ajax_url) ? AegisSEOAdmin.ajax_url : ajaxurl, data)
            .done(function(resp){
                if (resp && resp.success) {
                    var upd = resp.data && resp.data.updated ? resp.data.updated : 0;
                    var msg = 'Saved. Updated: ' + upd;
                    $notice.html('<div class="notice notice-success is-dismissible"><p>' + msg + '</p></div>');
                } else {
                    var em = (resp && resp.data && resp.data.message) ? resp.data.message : 'Save failed.';
                    $notice.html('<div class="notice notice-error"><p>' + em + '</p></div>');
                }
            })
            .fail(function(xhr){
                var em = 'Save failed.';
                if (xhr && xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    em = xhr.responseJSON.data.message;
                }
                $notice.html('<div class="notice notice-error"><p>' + em + '</p></div>');
            })
            .always(function(){
                $btn.prop('disabled', false);
                $saving.hide();
            });
    });


})(jQuery);
