<?php
namespace AegisSitemap\SEO;

use AegisSitemap\Utils\Options;

if (!defined('ABSPATH')) { exit; }

class Robots {

    private $options;

    public function __construct(Options $options) {
        $this->options = $options;

        add_filter('robots_txt', array($this, 'filter_robots_txt'), 20, 2);
        add_action('update_option_' . Options::OPTION_KEY, array($this, 'on_settings_updated'), 10, 2);
		add_action('admin_init', array($this, 'maybe_write_pending_robots'));
    }

	public function maybe_write_pending_robots() : void {
		if (!get_transient('aegissitemap_robots_pending_write')) {
			return;
		}

		delete_transient('aegissitemap_robots_pending_write');

		if ((int) $this->options->get('robots_enabled', 0) !== 1) { return; }
		if ((int) $this->options->get('robots_write_physical', 1) !== 1) { return; }

		$content = $this->generate();
		if ($content === '') { return; }

		$this->write_physical_file($content);
	}

    public function generate() : string {
        $enabled = (int) $this->options->get('robots_enabled', 0);
        if ($enabled !== 1) {
            return '';
        }

        $lines = array();

        $lines[] = '# robots.txt generated by AegisSitemap';
        $lines[] = '# NOTE: robots.txt is crawl guidance, not a security feature.';
        $lines[] = '';

        $lines[] = 'User-agent: *';

        $excluded = $this->get_excluded_dirs();
        foreach ($excluded as $path) {
            $lines[] = 'Disallow: ' . $path;
        }

        $lines[] = 'Allow: /wp-admin/admin-ajax.php';

        if ((int) $this->options->get('robots_optimization_enabled', 1) === 1) {

            if ((int) $this->options->get('robots_block_internal_search', 1) === 1) {
                $lines[] = 'Disallow: /search/';
                $lines[] = 'Disallow: /*?s=';
                $lines[] = 'Disallow: /*&s=';
            }

            if ((int) $this->options->get('robots_block_feeds', 0) === 1) {
                $lines[] = 'Disallow: /feed/';
                $lines[] = 'Disallow: /*/feed/';
                $lines[] = 'Disallow: /*?feed=';
            }

            if ((int) $this->options->get('robots_block_attachments', 0) === 1) {
                $lines[] = 'Disallow: /?attachment_id=';
            }
        }

        $lines[] = '';

        foreach ($this->get_sitemap_lines() as $s) {
            $lines[] = $s;
        }

        $custom_enabled = (int) $this->options->get('robots_custom_enabled', 0);
        $custom = (string) $this->options->get('robots_custom_rules', '');
        $custom = str_replace(array("\r\n", "\r"), "\n", $custom);
        $custom = trim($custom);

        if ($custom_enabled === 1 && $custom !== '') {
            $lines[] = '';
            $lines[] = '# BEGIN AegisSitemap';
            foreach (explode("\n", $custom) as $line) {
                $line = rtrim($line);
                if ($line === '') { continue; }
                $lines[] = $line;
            }
            $lines[] = '# END AegisSitemap';
        }

        return rtrim(implode("\n", $lines)) . "\n";
    }

    public function filter_robots_txt(string $output, bool $public) : string {
        $generated = $this->generate();
        if ($generated === '') {
            return $output;
        }
        return $generated;
    }

	public function on_settings_updated($old_value, $new_value) : void {
		static $running = false;
		if ($running) { return; }
		$running = true;

		try {
			$watched = array(
				'robots_enabled',
				'robots_write_physical',
				'robots_excluded_dirs',
				'robots_optimization_enabled',
				'robots_block_internal_search',
				'robots_block_feeds',
				'robots_block_attachments',
				'robots_custom_enabled',
				'robots_custom_rules',
				'sitemap_index_enabled',
				'sitemap_video_enabled',
				'sitemap_news_enabled',
			);

			$old = is_array($old_value) ? $old_value : array();
			$new = is_array($new_value) ? $new_value : array();

			$changed = false;
			foreach ($watched as $k) {
				if (($old[$k] ?? null) !== ($new[$k] ?? null)) { $changed = true; break; }
			}
			if (!$changed) { return; }

			set_transient('aegissitemap_robots_pending_write', 1, 5 * MINUTE_IN_SECONDS);

		} finally {
			$running = false;
		}
	}


	public function write_robots_now() : void {
		$content = $this->generate();
		if ($content === '') { return; }

		$write_physical = (int) $this->options->get('robots_write_physical', 1);
		if ($write_physical !== 1) { return; }

		$this->write_physical_file($content);
	}

	public function robots_file_path() : string {
		if (!function_exists('get_home_path')) {
			require_once ABSPATH . 'wp-admin/includes/file.php';
		}
		$home_path = get_home_path();
		return trailingslashit($home_path) . 'robots.txt';
	}

    public function robots_file_status() : array {
        $path = $this->robots_file_path();
        $exists = file_exists($path);
        $writable = is_writable(dirname($path)) && (!$exists || is_writable($path));
        return array(
            'path' => $path,
            'exists' => $exists ? 1 : 0,
            'writable' => $writable ? 1 : 0,
            'modified' => $exists ? (int) @filemtime($path) : 0,
            'size' => $exists ? (int) @filesize($path) : 0,
        );
    }

	public function write_physical_file(string $content) : void {
		$path = $this->robots_file_path();
		$dir  = dirname($path);

		if (!is_dir($dir) || !is_writable($dir)) {

			if (!file_exists($path) || !is_writable($path)) {
				if (function_exists('aegissitemap_update_log')) {
					aegissitemap_update_log('robots.txt write failed (not writable).', array(
						'path' => $path,
						'dir'  => $dir,
					));
				}
				return;
			}
		}

		$written = @file_put_contents($path, $content);
		if ($written === false && function_exists('aegissitemap_update_log')) {
			aegissitemap_update_log('robots.txt write failed (file_put_contents returned false).', array(
				'path' => $path,
			));
		}

	}

    private function get_excluded_dirs() : array {
        $dirs = $this->options->get('robots_excluded_dirs', array());
        if (!is_array($dirs)) { $dirs = array(); }

        $out = array();

        foreach ($dirs as $d) {
            $d = is_string($d) ? trim($d) : '';
            if ($d === '') { continue; }

            if ($d[0] !== '/') { $d = '/' . $d; }

            if (strpos($d, '*') === false && strpos($d, '?') === false && substr($d, -1) !== '/') {
                $d .= '/';
            }

            $out[$d] = 1;
        }

        if ((int) $this->options->get('robots_optimization_enabled', 1) === 1) {
            $out['/wp-admin/'] = 1;
            $out['/wp-includes/'] = 1;
        }

        return array_keys($out);
    }

    private function get_sitemap_lines() : array {
        $lines = array();

        $index_enabled = (int) $this->options->get('sitemap_index_enabled', 1);
        $sitemap_url = home_url('/sitemap.xml');
        $index_url = home_url('/sitemap_index.xml');

        if ($index_enabled === 1) {
            $lines[] = 'Sitemap: ' . $index_url;
        } else {
            $lines[] = 'Sitemap: ' . $sitemap_url;
        }

        if ((int) $this->options->get('sitemap_video_enabled', 0) === 1) {
            $lines[] = 'Sitemap: ' . home_url('/video-sitemap.xml');
        }
        if ((int) $this->options->get('sitemap_news_enabled', 0) === 1) {
            $lines[] = 'Sitemap: ' . home_url('/news-sitemap.xml');
        }

        return $lines;
    }
}
