<?php
namespace AegisSitemap\Utils;

if (!defined('ABSPATH')) { exit; }

class License {

    const OPTION_KEY = 'aegissitemap_license_key';
    const OPTION_EMAIL = 'aegissitemap_license_email';
    const TRANSIENT_STATUS = 'aegissitemap_license_status';

    public function get_key() : string {
        return (string) get_option(self::OPTION_KEY, '');
    }

    public function get_email() : string {
        return (string) get_option(self::OPTION_EMAIL, '');
    }

    public function set_key(string $key) : void {
        update_option(self::OPTION_KEY, trim($key), false);
        delete_transient(self::TRANSIENT_STATUS);
    }

    public function set_email(string $email) : void {
        update_option(self::OPTION_EMAIL, sanitize_email($email), false);
        delete_transient(self::TRANSIENT_STATUS);
    }

    public function get_cached_status() : array {
        $s = get_transient(self::TRANSIENT_STATUS);
        return is_array($s) ? $s : array();
    }

    public function is_pro() : bool {
        $s = $this->get_cached_status();
        if (isset($s['is_pro'])) {
            return (bool) $s['is_pro'];
        }
        return false;
    }

    public function check_now(string $current_version) : array {
        $site_url = home_url();
        $domain   = wp_parse_url($site_url, PHP_URL_HOST);

        $license_key = $this->get_key();
        $license_email = $this->get_email();

        $payload = array(
            'slug'            => defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap',
            'product'         => defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap',
            'version'         => (string) $current_version,
            'domain'          => (string) $domain,
            'site_url'        => (string) $site_url,
            'license_key'     => (string) $license_key,
			'email' 		  => (string) $license_email,
            'wp_version'      => (string) get_bloginfo('version'),
            'php_version'     => (string) PHP_VERSION,
            'plugin_basename' => defined('AEGISSITEMAP_BASENAME') ? (string) AEGISSITEMAP_BASENAME : '',
        );

        $args = array(
            'timeout' => 20,
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body'    => wp_json_encode($payload),
        );

		$endpoint = defined('AEGISSITEMAP_CLM_STATUS_ENDPOINT') ? AEGISSITEMAP_CLM_STATUS_ENDPOINT : '';
		if (!$endpoint && defined('AEGISSITEMAP_CLM_UPDATE_ENDPOINT')) {
			$endpoint = AEGISSITEMAP_CLM_UPDATE_ENDPOINT;
		}

		$resp = wp_remote_post($endpoint, $args);

		if (is_wp_error($resp)) {
			set_transient('aegissitemap_clm_free_reg_backoff', 1, 30 * MINUTE_IN_SECONDS);

			return array(
				'success' => false,
				'status'  => 'wp_error',
				'message' => $resp->get_error_message(),
			);
		}

		$code = (int) wp_remote_retrieve_response_code($resp);
		$body = (string) wp_remote_retrieve_body($resp);

		if ($code === 429) {
			set_transient('aegissitemap_clm_free_reg_backoff', 1, 30 * MINUTE_IN_SECONDS);

			return array(
				'success' => false,
				'status'  => 'rate_limited',
				'message' => 'Rate limited.',
				'raw'     => $body,
			);
		}

        if ($code < 200 || $code >= 300) {
            return array('success'=>false,'status'=>'http','message'=>'CLM HTTP '.$code,'raw'=>$body);
        }

        $data = json_decode($body, true);
        if (!is_array($data)) {
            return array('success'=>false,'status'=>'bad_json','message'=>'CLM returned invalid JSON','raw'=>$body);
        }

        $is_pro = false;
        $candidates_true = array(
            $data['is_pro'] ?? null,
            $data['pro'] ?? null,
            $data['license_active'] ?? null,
            $data['active'] ?? null,
        );
        foreach ($candidates_true as $v) {
            if ($v === true || $v === 1 || $v === '1' || $v === 'true') { $is_pro = true; break; }
        }
        if (!$is_pro) {
            $status = strtolower((string)($data['license_status'] ?? $data['status'] ?? ''));
            $plan   = strtolower((string)($data['plan'] ?? $data['tier'] ?? $data['license_type'] ?? ''));
            if (in_array($status, array('active','valid','licensed'), true) && in_array($plan, array('pro','premium','paid'), true)) {
                $is_pro = true;
            }
            if (in_array($plan, array('pro','premium','paid'), true) && $status === '') {
                $is_pro = true;
            }
        }

        // Hard rule: a PRO activation is only valid if the email matches the license email.
        // If CLM returns an email field, we require an exact (case-insensitive) match.
        $server_email = '';
        foreach (array('email','customer_email','license_email','registered_email','billing_email') as $k) {
            if (!empty($data[$k]) && is_string($data[$k])) { $server_email = (string) $data[$k]; break; }
        }
        if ($server_email !== '' && $license_email !== '') {
            if (strcasecmp(trim($server_email), trim($license_email)) !== 0) {
                $is_pro = false;
            }
        }

        $status_row = array(
            'success'     => !empty($data['success']) ? true : true, 
            'checked_at'  => time(),
            'is_pro'      => (bool) $is_pro,
            'license_key' => (string) $license_key,
            'status'      => (string)($data['license_status'] ?? $data['status'] ?? ''),
            'plan'        => (string)($data['plan'] ?? $data['tier'] ?? $data['license_type'] ?? ''),
            'expires'     => (string)($data['expires'] ?? $data['expires_at'] ?? ''),
            'email'       => (string) ($server_email !== '' ? $server_email : $license_email),
            'raw'         => $data,
        );

        if ($server_email !== '' && $license_email !== '' && strcasecmp(trim($server_email), trim($license_email)) !== 0) {
            $status_row['status'] = 'email_mismatch';
            $status_row['message'] = 'License email mismatch.';
        }

        set_transient(self::TRANSIENT_STATUS, $status_row, HOUR_IN_SECONDS * 6);

        return $status_row;
    }

	public function maybe_send_free_register(string $current_version) : void {
		// IMPORTANT: free-install registration is optional telemetry and is NOT licensing.
		// If the user has a license key (or is already PRO), we do not perform free registration.
		if (trim($this->get_key()) !== '' || $this->is_pro()) {
			return;
		}
		if ((int) get_option('aegissitemap_free_reg_optin', 0) !== 1) {
			return;
		}
		$site_url = home_url();
		$domain   = wp_parse_url($site_url, PHP_URL_HOST);

		$sent_key = 'aegissitemap_clm_free_reg_sent_' . md5((string) $domain);

		$backoff_key = 'aegissitemap_clm_free_reg_backoff_' . md5((string) $domain);

		if (get_transient($sent_key)) { return; }
		if (get_transient($backoff_key)) { return; }

		$email = (string) get_option('aegissitemap_license_email', '');
		if (!$email) {
			$email = (string) get_option('admin_email', '');
		}

		$payload = array(
			'slug'           => defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap',
			'product'        => defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap',
			'plugin_version' => (string) $current_version,
			'email'          => (string) $email,
			'domain'         => (string) $domain,
			'site_url'       => (string) $site_url,
			'wp_version'     => (string) get_bloginfo('version'),
			'php_version'    => (string) PHP_VERSION,
		);
		$endpoint = defined('AEGISSITEMAP_CLM_FREE_REG_ENDPOINT') ? AEGISSITEMAP_CLM_FREE_REG_ENDPOINT : '';
		if (!$endpoint) { return; }

		$resp = wp_remote_post($endpoint, array(
			'timeout' => 15,
			'headers' => array('Content-Type' => 'application/json'),
			'body'    => wp_json_encode($payload),
		));

		if (is_wp_error($resp)) {
			set_transient($backoff_key, 1, 30 * MINUTE_IN_SECONDS);

			$msg = 'CLM free-register WP_Error: ' . $resp->get_error_message();
			update_option('aegissitemap_clm_free_reg_last_error', $msg, false);

			if (function_exists('aegissitemap_update_log')) {
				aegissitemap_update_log('CLM free-register failed (WP_Error).', array(
					'endpoint' => $endpoint,
					'error'    => $resp->get_error_message(),
				));
			}

			return;
		}

		$code = (int) wp_remote_retrieve_response_code($resp);
		$body = (string) wp_remote_retrieve_body($resp);
		
		if ($code < 200 || $code >= 300) {
			$msg = 'CLM free-register HTTP ' . $code . ': ' . $body;
			update_option('aegissitemap_clm_free_reg_last_error', $msg, false);

			if (function_exists('aegissitemap_update_log')) {
				aegissitemap_update_log('CLM free-register non-2xx.', array(
					'endpoint'  => $endpoint,
					'http_code' => $code,
					'body'      => $body,
				));
			}
		}

		if ($code === 429) {
			set_transient($backoff_key, 1, 30 * MINUTE_IN_SECONDS);

			$msg = 'CLM free-register rate limited (HTTP 429): ' . $body;
			update_option('aegissitemap_clm_free_reg_last_error', $msg, false);

			if (function_exists('aegissitemap_update_log')) {
				aegissitemap_update_log('CLM free-register rate limited (HTTP 429).', array(
					'endpoint'  => $endpoint,
					'http_code' => $code,
					'body'      => $body,
				));
			}

			return;
		}

		if ($code >= 200 && $code < 300) {
			set_transient($sent_key, 1, DAY_IN_SECONDS);
		}

		if (function_exists('aegissitemap_update_log')) {
			aegissitemap_update_log('CLM free-register attempted.', array(
				'endpoint'  => $endpoint,
				'http_code' => $code,
				'body'      => $body,
			));
		}
	}



	public function hook_update_check() : void {
		add_filter('pre_set_site_transient_update_plugins', function ($transient) {
			if (!is_object($transient) || empty($transient->checked) || !is_array($transient->checked)) {
				return $transient;
			}

			$basename = defined('AEGISSITEMAP_BASENAME') ? AEGISSITEMAP_BASENAME : '';
			if (!$basename || empty($transient->checked[$basename])) {
				return $transient;
			}

			$current_version = (string) $transient->checked[$basename];

			$payload = array(
				'product'     => defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap',
				'slug'        => defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap',
				'version'     => $current_version,
				'site_url'    => home_url(),
				'license_key' => $this->get_key(),
				'email'       => (string) get_option('aegissitemap_license_email', get_option('admin_email', '')),
			);

			$endpoint = defined('AEGISSITEMAP_CLM_UPDATE_ENDPOINT') ? AEGISSITEMAP_CLM_UPDATE_ENDPOINT : '';
			if (!$endpoint) { return $transient; }

			$resp = wp_remote_post($endpoint, array(
				'timeout' => 20,
				'headers' => array('Content-Type' => 'application/json'),
				'body'    => wp_json_encode($payload),
			));

			if (is_wp_error($resp)) { return $transient; }

			$code = (int) wp_remote_retrieve_response_code($resp);
			if ($code < 200 || $code >= 300) { return $transient; }

			$data = json_decode((string) wp_remote_retrieve_body($resp), true);
			if (!is_array($data) || empty($data['success']) || empty($data['update_available'])) {
				return $transient;
			}

			if (empty($data['version']) || empty($data['download_url'])) {
				return $transient;
			}

			$update              = new \stdClass();
			$update->slug        = defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap';
			$update->plugin      = $basename;
			$update->new_version = (string) $data['version'];
			$update->package     = (string) $data['download_url'];
			$update->url         = !empty($data['homepage']) ? (string) $data['homepage'] : '';

			if (!isset($transient->response) || !is_array($transient->response)) {
				$transient->response = array();
			}
			$transient->response[$basename] = $update;

			return $transient;
		});

		add_filter('plugins_api', function ($result, $action, $args) {
			if ($action !== 'plugin_information') return $result;
			if (empty($args->slug) || $args->slug !== (defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap')) {
				return $result;
			}

			$info = new \stdClass();
			$info->name          = 'AegisSitemap';
			$info->slug          = defined('AEGISSITEMAP_SLUG') ? AEGISSITEMAP_SLUG : 'aegissitemap';
			$info->version       = defined('AEGISSITEMAP_VERSION') ? AEGISSITEMAP_VERSION : '';
			$info->author        = 'Aegisify';
			$info->homepage      = 'https://aegisify.com/';
			$info->sections      = array(
				'description' => 'XML sitemap engine by Aegisify — fast, cache-friendly sitemap index, HTML sitemap, images, and optional video/news sitemaps.',
			);

			return $info;
		}, 10, 3);
	}

    public static function render_pro_banner(string $title = 'Upgrade to PRO', string $message = '') : void {
        if ($message === '') {
            $message = __('PRO feature: This section is available in AegisSitemap PRO. Unlock advanced sitemap controls, exclusions, and automation.', 'aegissitemap');
        }
        ?>
        <div class="aegissitemap-pro-banner">
            <div class="aegissitemap-pro-banner__left">
                <div class="aegissitemap-pro-badge"><?php echo esc_html($title); ?></div>
                <div class="aegissitemap-pro-text"><?php echo esc_html($message); ?></div>
            </div>
            <div class="aegissitemap-pro-banner__right">
                <a class="button button-primary" href="<?php echo esc_url(admin_url('admin.php?page=aegissitemap&tab=license')); ?>">
                    <?php echo esc_html__('Activate License', 'aegissitemap'); ?>
                </a>
            </div>
        </div>
        <?php
    }
}
