<?php
namespace AegisSitemap\Admin;

use AegisSitemap\Utils\Options;
use AegisSitemap\Utils\Sanitize;
use AegisSitemap\Utils\License;

if (!defined('ABSPATH')) { exit; }

class Admin_Settings {

    private $options;
    private $license;

    public function __construct(Options $options, License $license) {
        $this->options = $options;
        $this->license = $license;
        add_action('admin_init', array($this, 'register_settings'));
    }

    public function register_settings() : void {

        register_setting('aegissitemap_settings_group', Options::OPTION_KEY, array($this, 'sanitize'));

        add_settings_section(
            'aegissitemap_section_sitemap',
            __('Sitemap', 'aegissitemap'),
            '__return_false',
            'aegissitemap_sitemap'
        );

        add_settings_field('sitemap_enabled', __('Enable Sitemap', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_enabled', 'label' => __('Enable XML sitemap output', 'aegissitemap')));
        add_settings_field('sitemap_info', __('Sitemap URL', 'aegissitemap'), array($this, 'field_sitemap_info'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap');

        add_settings_field('sitemap_index_enabled', __('Sitemap Index', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_index_enabled', 'label' => __('Enable /sitemap_index.xml', 'aegissitemap')));

        add_settings_field('sitemap_page_size', __('Max URLs per Sitemap', 'aegissitemap'), array($this, 'field_number'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_page_size', 'min' => 500, 'max' => 5000, 'step' => 100));

        add_settings_field('sitemap_include_images', __('Images', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_include_images', 'label' => __('Include featured images in XML', 'aegissitemap')));
        add_settings_field('sitemap_include_attached_images', __('Attached Images', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_include_attached_images', 'label' => __('Include attached images (slower on very large sites)', 'aegissitemap')));

        add_settings_field('sitemap_video_enabled', __('Video Sitemap', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_video_enabled', 'label' => __('Enable /video-sitemap.xml', 'aegissitemap')));
        add_settings_field('sitemap_news_enabled', __('News Sitemap', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_news_enabled', 'label' => __('Enable /news-sitemap.xml (if supported by your content)', 'aegissitemap')));

        add_settings_field('sitemap_html_enabled', __('HTML Sitemap', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_html_enabled', 'label' => __('Enable /sitemap/ HTML listing', 'aegissitemap')));

        add_settings_field('sitemap_robots_add', __('robots.txt', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'sitemap_robots_add', 'label' => __('Add Sitemap: line to robots.txt output', 'aegissitemap')));

        add_settings_field('sitemap_post_types', __('Post Types', 'aegissitemap'), array($this, 'field_post_types'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap');
        add_settings_field('sitemap_taxonomies', __('Taxonomies', 'aegissitemap'), array($this, 'field_taxonomies'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap');

        add_settings_field('sitemap_folder_exclusions', __('Folder Exclusions', 'aegissitemap'), array($this, 'field_folder_exclusions'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap');

        add_settings_field('sitemap_last_ping', __('Last Ping', 'aegissitemap'), array($this, 'field_sitemap_last_ping'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap');
    

        add_settings_field('ping_search_engines', __('Ping Search Engines on Publish', 'aegissitemap'), array($this, 'field_toggle'), 'aegissitemap_sitemap', 'aegissitemap_section_sitemap', array('key' => 'ping_search_engines', 'label' => __('Enabled', 'aegissitemap')));

        // Robots tab
		add_settings_section(
			'aegissitemap_section_robots',
			__('Robots.txt', 'aegissitemap'),
			'__return_false',
			'aegissitemap_robots'
		);

        add_settings_field('robots_enabled', __('Enable Robots', 'aegissitemap'), array($this, 'field_toggle'),
            'aegissitemap_robots', 'aegissitemap_section_robots',
            array('key' => 'robots_enabled', 'label' => __('Enable robots.txt management (recommended)', 'aegissitemap'))
        );

        add_settings_field('robots_write_physical', __('Write robots.txt file', 'aegissitemap'), array($this, 'field_toggle'),
            'aegissitemap_robots', 'aegissitemap_section_robots',
            array('key' => 'robots_write_physical', 'label' => __('Write a physical robots.txt file to the site root (ABSPATH)', 'aegissitemap'))
        );

        add_settings_field('robots_optimization_enabled', __('Optimization', 'aegissitemap'), array($this, 'field_toggle'),
            'aegissitemap_robots', 'aegissitemap_section_robots',
            array('key' => 'robots_optimization_enabled', 'label' => __('Apply safe default optimizations (crawl budget + resource protection)', 'aegissitemap'))
        );

        add_settings_field('robots_excluded_dirs', __('Excluded Directories', 'aegissitemap'),
            array($this, 'field_robots_excluded_dirs'),
            'aegissitemap_robots', 'aegissitemap_section_robots'
        );

        add_settings_field('robots_block_internal_search', __('Block internal search', 'aegissitemap'), array($this, 'field_toggle'),
            'aegissitemap_robots', 'aegissitemap_section_robots',
            array('key' => 'robots_block_internal_search', 'label' => __('Disallow internal search results (Google best practice)', 'aegissitemap'))
        );

        add_settings_field('robots_block_feeds', __('Block feeds', 'aegissitemap'), array($this, 'field_toggle'),
            'aegissitemap_robots', 'aegissitemap_section_robots',
            array('key' => 'robots_block_feeds', 'label' => __('Disallow RSS/Atom feeds (optional)', 'aegissitemap'))
        );

        add_settings_field('robots_block_attachments', __('Block attachment pages', 'aegissitemap'), array($this, 'field_toggle'),
            'aegissitemap_robots', 'aegissitemap_section_robots',
            array('key' => 'robots_block_attachments', 'label' => __('Disallow attachment pages (optional)', 'aegissitemap'))
        );

        add_settings_field('robots_custom_enabled', __('Custom rules', 'aegissitemap'), array($this, 'field_toggle'),
            'aegissitemap_robots', 'aegissitemap_section_robots',
            array('key' => 'robots_custom_enabled', 'label' => __('Append custom robots.txt rules', 'aegissitemap'))
        );

		add_settings_field('robots_custom_rules', __('Custom robots.txt block', 'aegissitemap'),
			array($this, 'field_textarea'),
			'aegissitemap_robots', 'aegissitemap_section_robots',
			array('key' => 'robots_custom_rules', 'rows' => 10, 'placeholder' => "# Example:\nUser-agent: Googlebot\nDisallow: /private/\n")
		);

		} 

		public function sanitize($input) {

        $is_pro = (bool) $this->license->is_pro();
        $current = get_option(Options::OPTION_KEY, $this->options->defaults());
        if (!is_array($current)) { $current = $this->options->defaults(); }

		$out = $this->options->defaults();
		if (!is_array($input)) { return $out; }

		$input = wp_unslash($input);

        foreach (array(
            'sitemap_enabled','sitemap_index_enabled','sitemap_include_images','sitemap_include_attached_images',
            'sitemap_video_enabled','sitemap_news_enabled','sitemap_html_enabled','sitemap_robots_add',
            'ping_search_engines',
            'robots_enabled','robots_write_physical','robots_optimization_enabled',
            'robots_block_internal_search','robots_block_feeds','robots_block_attachments',
            'robots_custom_enabled'
        ) as $k) {
            if (isset($input[$k])) { $out[$k] = Sanitize::bool_int($input[$k]); }
        }

        // Enforce PRO locking server-side (UI is view-only, but POSTs are ignored when not PRO).
        if (!$is_pro) {
            foreach ($this->pro_locked_keys() as $k) {
                if (array_key_exists($k, $current)) {
                    $out[$k] = $current[$k];
                }
            }
        }

        if (isset($input['sitemap_page_size'])) {
            $v = Sanitize::int($input['sitemap_page_size']);
            if ($v < 500) $v = 500;
            if ($v > 5000) $v = 5000;
            $out['sitemap_page_size'] = $v;
        }

        if (!$is_pro) {
            $out['sitemap_page_size'] = isset($current['sitemap_page_size']) ? (int) $current['sitemap_page_size'] : (int) $out['sitemap_page_size'];
        }

        if (isset($input['sitemap_post_types']) && is_array($input['sitemap_post_types'])) {
            $pt = array();
            foreach ($input['sitemap_post_types'] as $k => $v) {
                $k = sanitize_key($k);
                $pt[$k] = (int) (!empty($v));
            }
            $out['sitemap_post_types'] = $pt;
        }

        if (!$is_pro && isset($current['sitemap_post_types'])) {
            $out['sitemap_post_types'] = is_array($current['sitemap_post_types']) ? $current['sitemap_post_types'] : $out['sitemap_post_types'];
        }

        if (isset($input['sitemap_taxonomies']) && is_array($input['sitemap_taxonomies'])) {
            $tx = array();
            foreach ($input['sitemap_taxonomies'] as $k => $v) {
                $k = sanitize_key($k);
                $tx[$k] = (int) (!empty($v));
            }
            $out['sitemap_taxonomies'] = $tx;
        }

        if (!$is_pro && isset($current['sitemap_taxonomies'])) {
            $out['sitemap_taxonomies'] = is_array($current['sitemap_taxonomies']) ? $current['sitemap_taxonomies'] : $out['sitemap_taxonomies'];
        }

        if (isset($input['sitemap_folder_exclusions'])) {
            $raw = (string) $input['sitemap_folder_exclusions'];
            $raw = str_replace("\r", "", $raw);
            $lines = explode("\n", $raw);

            $out_lines = array();
            foreach ($lines as $line) {
                $line = trim($line);
                if ($line === '') continue;

                $line = preg_replace('#^https?://[^/]+#i', '', $line);
                if ($line[0] !== '/') $line = '/' . $line;
                $line = rtrim($line, '/');
                if (strpos($line, '..') !== false) continue;

                $out_lines[] = $line;
            }
            $out_lines = array_values(array_unique($out_lines));
            $out['sitemap_folder_exclusions'] = implode("\n", $out_lines);
        }

        if (!$is_pro && isset($current['sitemap_folder_exclusions'])) {
            $out['sitemap_folder_exclusions'] = (string) $current['sitemap_folder_exclusions'];
        }

        $out['sitemap_cache_v'] = (int) ($this->options->get('sitemap_cache_v', 1));
        $out['sitemap_cache_v']++;

        if (isset($input['robots_excluded_dirs'])) {
            $dirs_in = $input['robots_excluded_dirs'];
            $dirs = array();

            if (is_string($dirs_in)) {
                $raw = str_replace("\r", "", (string) $dirs_in);
                $dirs_in = explode("\n", $raw);
            }

            if (is_array($dirs_in)) {
                foreach ($dirs_in as $d) {
                    $d = is_string($d) ? trim($d) : '';
                    if ($d === '') continue;
                    $d = substr($d, 0, 200);
                    $dirs[] = $d;
                }
            }

            $out['robots_excluded_dirs'] = $dirs;
        }

        if (isset($input['robots_custom_rules'])) {
            $raw = (string) $input['robots_custom_rules'];
            $raw = str_replace("\r", "", $raw);

            if (strlen($raw) > 20000) { $raw = substr($raw, 0, 20000); }
            $out['robots_custom_rules'] = $raw;
        }

		if (isset($input['sitemap_plugins_present'])) {
			$plugins = array();
			if (isset($input['sitemap_plugins']) && is_array($input['sitemap_plugins'])) {
				foreach ($input['sitemap_plugins'] as $p) {
					$p = (string) $p;
					$p = preg_replace('#[^A-Za-z0-9_\-./]#', '', $p);
					$p = trim($p);
					if ($p === '') continue;
					$plugins[] = $p;
				}
			}
			$plugins = array_values(array_unique($plugins));
			sort($plugins);
			$out['sitemap_plugins'] = $plugins; // becomes [] when none submitted
		}

		if (!$is_pro && isset($current['sitemap_plugins'])) {
			$out['sitemap_plugins'] = is_array($current['sitemap_plugins']) ? $current['sitemap_plugins'] : $out['sitemap_plugins'];
		}
return $out;
    }

    private function pro_locked_keys() : array {
        return array(
            'sitemap_page_size',
            'sitemap_include_images',
            'sitemap_include_attached_images',
            'sitemap_video_enabled',
            'sitemap_news_enabled',
            'sitemap_html_enabled',
            'sitemap_robots_add',
            'sitemap_post_types',
            'sitemap_taxonomies',
            'sitemap_folder_exclusions',
            'sitemap_plugins',
            'ping_search_engines',
        );
    }

    private function is_pro_locked_field(string $key) : bool {
        return in_array($key, $this->pro_locked_keys(), true);
    }

    private function field_name(string $key) : string {
        return Options::OPTION_KEY . '[' . $key . ']';
    }

    public function field_toggle($args) : void {
        $key   = (string) ($args['key'] ?? '');
        $label = (string) ($args['label'] ?? '');
        $val   = (int) $this->options->get($key, 0);
        $locked = (!$this->license->is_pro() && $this->is_pro_locked_field($key));
        // Disabled attribute is emitted via the disabled() helper (avoids raw attribute fragments).
        $wrap1 = $locked ? '<span class="aegissitemap-pro-locked-field" style="opacity:0.55;">' : '';
        $wrap2 = $locked ? '</span>' : '';
        echo wp_kses_post($wrap1) . '<label><input type="checkbox" name="' . esc_attr($this->field_name($key)) . '" value="1" ' . checked(1, $val, false) . disabled($locked, true, false) . ' /> ' . esc_html($label) . '</label>' . wp_kses_post($wrap2);
    }

    public function field_number($args) : void {
        $key  = (string) ($args['key'] ?? '');
        $min  = isset($args['min']) ? (int) $args['min'] : 0;
        $max  = isset($args['max']) ? (int) $args['max'] : 999999;
        $step = isset($args['step']) ? (int) $args['step'] : 1;
        $val  = (int) $this->options->get($key, 0);
        $locked = (!$this->license->is_pro() && $this->is_pro_locked_field($key));
        // Disabled attribute is emitted via the disabled() helper (avoids raw attribute fragments).
        $style_val = $locked ? 'opacity:0.55;' : '';
        echo '<input type="number" class="small-text" min="' . esc_attr($min) . '" max="' . esc_attr($max) . '" step="' . esc_attr($step) . '" name="' . esc_attr($this->field_name($key)) . '" value="' . esc_attr($val) . '"' . disabled($locked, true, false) . ($style_val !== '' ? ' style="' . esc_attr($style_val) . '"' : '') . ' />';
    }

    public function field_sitemap_info() : void {
        echo '<p style="margin:0 0 6px 0;"><strong>' . esc_html__('XML sitemap:', 'aegissitemap') . '</strong> <code>' . esc_html(home_url('/sitemap.xml')) . '</code></p>';
        echo '<p style="margin:0 0 6px 0;"><strong>' . esc_html__('Sitemap index:', 'aegissitemap') . '</strong> <code>' . esc_html(home_url('/sitemap_index.xml')) . '</code></p>';
        echo '<p style="margin:0 0 6px 0;"><strong>' . esc_html__('HTML sitemap:', 'aegissitemap') . '</strong> <code>' . esc_html(home_url('/sitemap/')) . '</code></p>';
    }

    public function field_post_types() : void {
        $locked = !$this->license->is_pro();
        $enabled = (array) $this->options->get('sitemap_post_types', array());
        $pts = get_post_types(array('public' => true), 'objects');
        foreach ($pts as $pt) {
            $k = $pt->name;
            $val = !empty($enabled[$k]) ? 1 : 0;
            echo '<label style="display:inline-block; margin:0 14px 8px 0;">';
            // Disabled attribute is emitted via the disabled() helper (avoids raw attribute fragments).
            $wrap1 = $locked ? '<span style="opacity:0.55;">' : '';
            $wrap2 = $locked ? '</span>' : '';
            echo wp_kses_post($wrap1);
            echo '<input type="checkbox" name="' . esc_attr($this->field_name('sitemap_post_types')) . '[' . esc_attr($k) . ']" value="1" ' . checked(1, $val, false) . disabled($locked, true, false) . ' /> ';
            echo esc_html($pt->labels->singular_name);
            echo wp_kses_post($wrap2);
            echo '</label>';
        }
    }

    public function field_taxonomies() : void {
        $locked = !$this->license->is_pro();
        $enabled = (array) $this->options->get('sitemap_taxonomies', array());
        $tax = get_taxonomies(array('public' => true), 'objects');
        foreach ($tax as $t) {
            $k = $t->name;
            $val = !empty($enabled[$k]) ? 1 : 0;
            echo '<label style="display:inline-block; margin:0 14px 8px 0;">';
            // Disabled attribute is emitted via the disabled() helper (avoids raw attribute fragments).
            $wrap1 = $locked ? '<span style="opacity:0.55;">' : '';
            $wrap2 = $locked ? '</span>' : '';
            echo wp_kses_post($wrap1);
            echo '<input type="checkbox" name="' . esc_attr($this->field_name('sitemap_taxonomies')) . '[' . esc_attr($k) . ']" value="1" ' . checked(1, $val, false) . disabled($locked, true, false) . ' /> ';
            echo esc_html($t->labels->singular_name);
            echo wp_kses_post($wrap2);
            echo '</label>';
        }
    }

    public function field_folder_exclusions() : void {
        echo '<p style="margin:0;">' . esc_html__('Use the "Folder Exclusions" panel on the right.', 'aegissitemap') . '</p>';
    }

    

    public function field_textarea($args) : void {
        $key = (string) ($args['key'] ?? '');
        $rows = isset($args['rows']) ? (int) $args['rows'] : 8;
        $placeholder = (string) ($args['placeholder'] ?? '');
        $val = (string) $this->options->get($key, '');
        echo '<textarea name="' . esc_attr($this->field_name($key)) . '" rows="' . esc_attr((string) $rows) . '" style="width:100%;" placeholder="' . esc_attr($placeholder) . '">';
        echo esc_textarea($val);
        echo '</textarea>';
    }

    public function field_robots_excluded_dirs() : void {
        $vals = $this->options->get('robots_excluded_dirs', array());
        if (!is_array($vals)) { $vals = array(); }

        echo '<div class="aegissitemap-sitemap-panel">';
        echo '<p class="aegissitemap-help">' . esc_html__('Add paths or patterns to disallow. Use one entry per item. Examples: /private/, /wp-content/uploads/tmp/, /*?s=', 'aegissitemap') . '</p>';

        echo '<div class="aegissitemap-inline-row" style="margin:10px 0;">';
        echo '<input type="text" id="aegissitemap-robots-exclude-input" placeholder="' . esc_attr__('/private/  (example)', 'aegissitemap') . '" style="flex:1;min-width:260px;" />';
        echo '<button type="button" class="button" id="aegissitemap-robots-exclude-add-btn">' . esc_html__('Add', 'aegissitemap') . '</button>';
        echo '</div>';

        echo '<div id="aegissitemap-robots-excluded-wrap">';
        echo '<ul class="aegissitemap-ul" id="aegissitemap-robots-excluded-list" style="margin-top:0;">';
        foreach ($vals as $v) {
            $v = is_string($v) ? trim($v) : '';
            if ($v === '') continue;
            echo '<li class="aegissitemap-inline-row" style="margin:6px 0;">';
            echo '<span class="aegissitemap-pill aegissitemap-mono">' . esc_html($v) . '</span>';
            echo '<button type="button" class="button-link-delete aegissitemap-robots-exclude-remove" data-val="' . esc_attr($v) . '">' . esc_html__('Remove', 'aegissitemap') . '</button>';
            echo '</li>';
            echo '<input type="hidden" class="aegissitemap-robots-exclude-hidden" name="' . esc_attr($this->field_name('robots_excluded_dirs')) . '[]" value="' . esc_attr($v) . '" />';
        }
        echo '</ul>';
        echo '</div>';

        echo '<p class="aegissitemap-help" style="margin-top:10px;">' . esc_html__('Tip: robots.txt is not a security tool. Use it to guide crawlers and protect crawl budget.', 'aegissitemap') . '</p>';
        echo '</div>';
    }
public function field_sitemap_last_ping() : void {
        $lp = $this->options->get('sitemap_last_ping', array());
        if (!is_array($lp)) { $lp = array(); }
        $sitemap = !empty($lp['sitemap']) ? (string) $lp['sitemap'] : '';
        $when    = !empty($lp['when']) ? (string) $lp['when'] : '';

        if ($sitemap !== '') {
            echo '<p style="margin:0 0 6px 0;">' . esc_html__('Last pinged sitemap:', 'aegissitemap') . ' <code>' . esc_html($sitemap) . '</code></p>';
        } else {
            echo '<p style="margin:0 0 6px 0;">' . esc_html__('No recent ping recorded yet.', 'aegissitemap') . '</p>';
        }
        if ($when !== '') {
            echo '<p style="margin:0;">' . esc_html__('When:', 'aegissitemap') . ' <code>' . esc_html($when) . '</code></p>';
        }
    }
}
