<?php
if (!defined('ABSPATH')) { exit; }

final class ASG_Admin {
    private $engine;
    private $ml;

    public function __construct($engine, $ml) {
        $this->engine = $engine;
        $this->ml = $ml;

        add_action('admin_menu', array($this, 'menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('admin_post_asg_save_settings', array($this, 'save_settings'));
        add_action('admin_post_asg_save_license', array($this, 'save_license'));
        add_action('admin_post_asg_add_list', array($this, 'add_list'));
        add_action('admin_post_asg_delete_list', array($this, 'delete_list'));
        add_action('admin_post_asg_event_action', array($this, 'event_action'));
        add_action('admin_post_asg_bulk_event_action', array($this, 'bulk_event_action'));
        add_action('admin_post_asg_export_settings', array($this, 'export_settings'));
        add_action('admin_post_asg_import_settings', array($this, 'import_settings'));
        add_action('admin_post_asg_cleanup_scan', array($this, 'cleanup_scan'));
		add_action('admin_post_asg_register_free_install', array($this, 'register_free_install'));

        if (is_multisite()) {
            add_action('network_admin_menu', array($this, 'network_menu'));
            add_action('admin_post_asg_save_network_settings', array($this, 'save_network_settings'));
        }

    }
	public function register_free_install() {
		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('Access denied.', 'aegisspam'));
		}

		check_admin_referer('asg_register_free_install');

		$do = isset($_POST['asg_free_install_do']) ? sanitize_text_field(wp_unslash($_POST['asg_free_install_do'])) : 'register';

		$consent = isset($_POST['asg_free_reg_consent']) && '1' === sanitize_text_field( wp_unslash( $_POST['asg_free_reg_consent'] ) );

		if ($do === 'register') {

			if (!$consent) {
				update_option('asg_free_reg_optin', 0);
				update_option('asg_free_reg_status', 'unregistered');
				wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=free_consent_required'));
				exit;
			}

			update_option('asg_free_reg_optin', 1);

			$ok = (class_exists('ASG_Core') && method_exists('ASG_Core', 'clm_register_free_install_manual'))
				? ASG_Core::clm_register_free_install_manual()
				: false;

			update_option('asg_free_reg_status', $ok ? 'registered' : 'error');

			wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=' . ($ok ? 'free_registered' : 'free_register_failed')));
			exit;
		}

		if ($do === 'unregister') {
			update_option('asg_free_reg_optin', 0);
			update_option('asg_free_reg_status', 'unregistered');

			wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=free_unregistered'));
			exit;
		}

		wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license'));
		exit;
	}
    
    public function enqueue_assets($hook) {
        if ( empty( $_GET['page'] ) ) { return; } // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- admin page slug check.
        $page = sanitize_key( wp_unslash( $_GET['page'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- admin page slug check.
        if (($page !== 'aegis-spam-guard') && ($page !== 'aegis-spam-guard-network')) { return; }
        wp_enqueue_style('asg-admin', ASG_PLUGIN_URL . 'assets/admin.css', array(), ASG_VERSION);
        wp_enqueue_script('asg-admin', ASG_PLUGIN_URL . 'assets/admin.js', array('jquery'), ASG_VERSION, true);
    }

	public function menu() {
        add_menu_page(
            'AegisSpamGuard',
            'AegisSpamGuard',
            'manage_options',
            'aegis-spam-guard',
            array($this, 'page'),
            'dashicons-shield',
            58
        );

    }

    public function network_menu() {
        if (!is_multisite() || !is_network_admin()) { return; }
        add_submenu_page(
            'settings.php',
            'AegisSpamGuard (Network)',
            'AegisSpamGuard',
            'manage_network_options',
            'aegis-spam-guard-network',
            array($this, 'network_page')
        );
    }

    
    public function network_page() {
        if (!current_user_can('manage_network_options')) { wp_die('Access denied'); }
        $opt = get_site_option('asg_network_settings', array());
        if (!is_array($opt)) { $opt = array(); }

        echo '<div class="wrap asg-wrap">';
        echo '<div class="asg-header"><div class="asg-header__title"><h1 class="wp-heading-inline">AegisSpamGuard — Network Defaults</h1>';
        echo '<p class="description">Set default protection behavior for all sites in this network. Individual sites can override these values in their own Settings tab.</p>';
        echo '</div></div>';

        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" class="asg-form">';
        echo '<input type="hidden" name="action" value="asg_save_network_settings">';
        wp_nonce_field('asg_save_network_settings');

        $sections = array(
            array(
                'icon' => 'dashicons-shield',
                'title' => 'Policy thresholds',
                'desc'  => 'Default score ranges for Allow → Challenge → Hold → Block.',
                'fields'=> array(
                    array('k'=>'threshold_allow','t'=>'number','label'=>'Allow max score','desc'=>'Scores at or below this are allowed. Example: 0–29.'),
                    array('k'=>'threshold_challenge','t'=>'number','label'=>'Challenge max score','desc'=>'Scores up to this trigger a soft challenge. Example: 30–59.'),
                    array('k'=>'threshold_hold','t'=>'number','label'=>'Hold max score','desc'=>'Scores up to this are held for review. Example: 60–79.'),
                    array('k'=>'fp_hold_instead_of_block','t'=>'bool','label'=>'False positive protection','desc'=>'When enabled, would-be blocks are held for review (safer defaults).'),
                ),
            ),

            array(
                'icon' => 'dashicons-admin-generic',
                'title' => 'Modules & signal groups',
                'desc'  => 'Enable/disable entire groups of signals. Turn a group off to reduce false positives or troubleshoot scoring.',
                'fields'=> array(
                    array('k'=>'mod_fingerprinting','t'=>'check','l'=>'Request fingerprinting','desc'=>'Header/UA anomalies, referer checks, basic proxy hints. Example: enable to catch non-browser bots.'),
                    array('k'=>'mod_behavior','t'=>'check','l'=>'Behavior & velocity','desc'=>'Burst submissions, time-to-submit, repeated payload patterns. Example: disable if you have legitimate fast multi-submit workflows.'),
                    array('k'=>'mod_content','t'=>'check','l'=>'Content intelligence','desc'=>'Links/TLDs/shorteners, obfuscation, n-grams, similarity-to-known-spam. Example: enable for aggressive form spam detection.'),
                    array('k'=>'mod_identity','t'=>'check','l'=>'Identity heuristics','desc'=>'Disposable email, MX checks, name/email mismatch. Example: disable if your users use many aliases.'),
                    array('k'=>'mod_geo_asn','t'=>'check','l'=>'Geo/ASN rules','desc'=>'Country/ASN allow/deny evaluation and geo enrichment. Example: enable if you want geo-based rules.'),
                    array('k'=>'mod_challenges','t'=>'check','l'=>'Challenges & proofs','desc'=>'Honeypot, JS proof, optional CAPTCHA and progressive delay. Example: enable to reduce bot traffic without hard blocks.'),
                    array('k'=>'enable_progressive_delay','t'=>'check','l'=>'Progressive delay (forms)','desc'=>'When an event lands in the Challenge band, slow the response to discourage bots. Example: 150–1800ms depending on score.'),
                    array('k'=>'progressive_delay_min_ms','t'=>'number','l'=>'Progressive delay min (ms)','min'=>0,'max'=>5000,'step'=>50,'desc'=>'Minimum delay in milliseconds for low Challenge scores. Example: 150ms.'),
                    array('k'=>'progressive_delay_max_ms','t'=>'number','l'=>'Progressive delay max (ms)','min'=>0,'max'=>5000,'step'=>50,'desc'=>'Maximum delay in milliseconds for high Challenge scores. Example: 1800ms.'),
                ),
            ),

            array(
                'icon' => 'dashicons-privacy',
                'title' => 'Privacy defaults',
                'desc'  => 'Default data-handling behavior for the network.',
                'fields'=> array(
                    array('k'=>'ip_mode','t'=>'select','label'=>'IP handling mode','desc'=>'Store raw IP, anonymize, hash, or store no IP.',
                        'options'=>array('store'=>'Store (raw IP)','anonymize'=>'Anonymize','hash'=>'Hash (no raw IP)','off'=>'Off (no IP)')),
                    array('k'=>'no_external_calls','t'=>'bool','label'=>'No external calls','desc'=>'When enabled, sites will not fetch external lists or contact third parties.'),
                ),
            ),
            array(
                'icon' => 'dashicons-admin-site-alt3',
                'title' => 'Geo / ASN defaults',
                'desc'  => 'Optional geo/ASN rules (cached).',
                'fields'=> array(
                    array('k'=>'enable_geo_rules','t'=>'bool','label'=>'Enable country rules','desc'=>'Default for country allow/deny behavior.'),
                    array('k'=>'enable_asn_rules','t'=>'bool','label'=>'Enable ASN rules','desc'=>'Default for ASN allow/deny behavior.'),
                    array('k'=>'geo_provider','t'=>'select','label'=>'Geo provider','desc'=>'Headers = edge header detection, MaxMind = local mmdb, None = disable.',
                        'options'=>array('headers'=>'Headers','maxmind'=>'MaxMind (local file)','none'=>'None')),
                ),
            ),
        );

        foreach ($sections as $sec) {

            $is_pro_section = in_array((string) $sec['title'], array(
                'Content intelligence',
                'Challenges',
                'Local learning',
                'Signal weights',
            ), true);

            $locked = ($is_pro_section && !$this->is_pro_active());

            echo '<div class="asg-card asg-section' . ($locked ? ' asg-pro-locked' : '') . '">';
            if ($locked) {
                echo '<div class="asg-pro-lockbar"><span class="asg-pro-pill">PRO</span> This block is available in <strong>AegisSpamGuard PRO</strong>. <a class="button button-primary asg-btn asg-btn--pro" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=license')) . '">Upgrade to PRO</a></div>';
            }
            echo '<div class="asg-section__head">';
            echo '<span class="dashicons ' . esc_attr($sec['icon']) . ' asg-section__icon"></span>';
            echo '<div><h3 class="asg-section__title">' . esc_html($sec['title']) . '</h3>';
            echo '<p class="asg-section__desc">' . wp_kses_post($sec['desc']) . '</p></div>';
            echo '</div>';
            echo '<table class="form-table asg-table"><tbody>';

            foreach ($sec['fields'] as $f) {
                $k = $f['k'];
                $label = $f['label'];
                $desc = isset($f['desc']) ? $f['desc'] : '';
                $val = isset($opt[$k]) ? $opt[$k] : '';
	                $disabled_attr = $locked ? ' disabled' : '';

                echo '<tr><th scope="row"><label for="' . esc_attr($k) . '">' . esc_html($label) . '</label></th><td>';

                if ($f['t'] === 'bool') {
                    $checked = !empty($val) ? ' checked' : '';
                    echo '<label class="asg-checkmark">';
	                    echo '<input type="checkbox" id="' . esc_attr($k) . '" name="' . esc_attr($k) . '" value="1"' . esc_attr($checked) . esc_attr($disabled_attr) . ' />';
                    echo '<span class="asg-checkmark__box" aria-hidden="true"></span>';
                    echo '<span class="asg-checkmark__text">' . (!empty($val) ? esc_html__('Enabled', 'aegisspam') : esc_html__('Disabled', 'aegisspam')) . '</span>';
                    echo '</label>';
                } elseif ($f['t'] === 'select') {
	                    echo '<select id="' . esc_attr($k) . '" name="' . esc_attr($k) . '" ' . esc_attr($disabled_attr) . '>';
                    foreach ($f['options'] as $ok=>$ov) {
                        $sel = ((string) $val === (string) $ok) ? ' selected' : '';
                        echo '<option value="' . esc_attr($ok) . '"' . esc_attr($sel) . '>' . esc_html($ov) . '</option>';
                    }
                    echo '</select>';
                } else {
	                    echo '<input type="number" id="' . esc_attr($k) . '" name="' . esc_attr($k) . '" value="' . esc_attr($val) . '" class="regular-text"' . esc_attr($disabled_attr) . '>';
                }

                if (!empty($desc)) {
                    echo '<p class="description asg-desc">' . wp_kses_post($desc) . '</p>';
                }

                echo '</td></tr>';
            }
            echo '</tbody></table>';
            echo '</div>';
        }

        echo '<div class="asg-actions">';
        submit_button('Save Network Defaults', 'primary', 'submit', false);
        echo '</div>';

        echo '</form>';
        echo '</div>';
    }



    public function page() {
        if (!current_user_can('manage_options')) { wp_die(esc_html__('Access denied.', 'aegisspam')); }

        $tab = isset($_GET['tab']) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'dashboard'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- tab routing.

        echo '<div class="wrap asg-wrap">';
        echo '<div class="asg-header">';
        echo '<div class="asg-header__title">';
        echo '<h1 class="wp-heading-inline">AegisSpamGuard</h1>';
        echo '<p class="description">Local-first, transparent anti-spam protection (score-based) for comments, forms, registrations, Woo checkout, and REST.</p>';
        echo '</div>';
        echo '<div class="asg-header__actions">';
        echo '<a class="button button-secondary" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=settings')) . '">Settings</a> ';
        echo '<a class="button button-primary" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=logs')) . '">View Log</a>';
        echo '</div>';
        echo '</div>';

        echo '<nav class="nav-tab-wrapper asg-tabs">';
        $this->tab_link('dashboard', $tab, 'Dashboard');
        $this->tab_link('logs', $tab, 'Spam Log');
        $this->tab_link('firewall', $tab, 'Basic Firewall');
        $this->tab_link('lists', $tab, 'Allow/Deny');
        $this->tab_link('settings', $tab, 'Settings');
        $this->tab_link('license', $tab, 'License');
        $this->tab_link('cleanup', $tab, 'Cleanup');
        echo '</nav>';

        if ($tab === 'cleanup') {
            if (!$this->is_pro_active()) { $this->render_pro_upsell_tab('Cleanup'); }
            else { $this->render_cleanup(); }
        }
        else if ($tab === 'firewall') {
            if (!$this->is_pro_active()) { $this->render_pro_upsell_tab('Basic Firewall'); }
            else { $this->render_firewall(); }
        }
        else if ($tab === 'dashboard') { $this->render_dashboard(); }
        elseif ($tab === 'lists') {
            if (!$this->is_pro_active()) { $this->render_pro_upsell_tab('Allow/Deny'); }
            else { $this->render_lists(); }
        }
        elseif ($tab === 'settings') { $this->render_settings(); }
        elseif ($tab === 'license') { $this->render_license(); }
        else { $this->render_logs(); }

        echo '</div>';
    }

    private function tab_link($slug, $active, $label) {
        $cls = ($slug === $active) ? 'nav-tab nav-tab-active' : 'nav-tab';
        $url = admin_url('admin.php?page=aegis-spam-guard&tab=' . $slug);
        echo '<a class="' . esc_attr($cls) . '" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
    }


    private function reason_definitions() {
        return array(

            'fp_hold_protection'   => 'False-positive protection is enabled, so a would-be block was held for review.',
            'softened_for_comments'=> 'Comments are softened: high-risk comments are held unless extremely high score.',
            'honeypot_filled'      => 'A hidden honeypot field was filled in. Humans do not see this field; bots often do.',
            'too_fast'             => 'Submitted faster than the “Min seconds to submit” setting. Bots often submit instantly.',
            'js_proof_failed'      => 'JavaScript proof token was missing/invalid. Indicates non-browser automation.',
            'captcha_passed'       => 'A challenge was shown and passed, reducing risk score.',
            'progressive_delay'     => 'A progressive delay was applied (Challenge band, forms only) to slow bots without hard-blocking.',
            'missing_user_agent'    => 'User-Agent header was missing. Legit browsers almost always send one.',
            'suspicious_user_agent' => 'User-Agent matched a common automation signature (curl/wget/python/etc.).',
            'missing_accept_language'=> 'Accept-Language header was missing. Many bots omit it.',
            'missing_referer'       => 'Referer header was missing for a form/registration/checkout event. This can indicate direct POSTs.',
            'external_referer'      => 'Referer did not match this site’s host. This can indicate cross-site posting.',
            'links_many'           => 'Content contained many links, which is a common spam pattern.',
            'suspicious_tld'       => 'A link used a suspicious top-level domain often associated with spam.',
            'url_shortener'        => 'A URL shortener was detected. Shorteners are frequently abused in spam.',
            'entity_obfuscation'   => 'HTML entities were used to obfuscate content (e.g., &#xNN;).',
            'percent_encoding'     => 'Percent-encoded text was detected (e.g., %2F), often used for obfuscation.',
            'zero_width_chars'     => 'Zero-width characters were detected (used to evade keyword filters).',
            'mixed_scripts'        => 'Mixed writing systems detected (Latin/Cyrillic/Greek), often used for look‑alike spam.',
            'high_unicode_ratio'   => 'High amount of non-ASCII characters detected, sometimes used for obfuscation.',
            'spam_phrase_ngrams'   => 'Known spam phrase patterns (bigrams/trigrams) were detected (local model tokens).',
            'similar_to_known_spam'=> 'This submission is similar to spam already blocked on this site.',
            'invalid_email_format' => 'Email format failed validation (not a real email structure).',
            'disposable_email'     => 'Disposable email provider detected (temporary inbox).',
            'email_mx_missing'     => 'Email domain has no MX records (cached check). Often indicates fake domains.',
            'name_email_mismatch'  => 'Name does not resemble the email local-part; random-looking emails are common in spam.',
            'velocity'             => 'Burst behavior detected (too many submissions in a short window).',
            'repeat_payload'       => 'Same or highly similar message was repeated across submissions.',
            'allowlist'            => 'Matched allowlist. Scoring is bypassed and the event is trusted.',
            'denylist'             => 'Matched denylist. Scoring is bypassed and the event is blocked.',
            'ml_spam_signal'       => 'Local on-site learning model detected spam-like text patterns.',
            'ml_ham_signal'        => 'Local model detected non-spam-like patterns, reducing score.',
        );
    }


    public function reason_sentence($code) {
        $code = (string) $code;
        $map = array(
            'disposable_email' => 'Disposable email domain detected.',
            'email_mx_missing' => 'Email domain has no MX records.',
            'invalid_email_format' => 'Email address format looks invalid.',
            'name_email_mismatch' => 'Name does not match email pattern (possible fake identity).',
            'honeypot_filled' => 'Hidden honeypot field was filled.',
            'too_fast' => 'Submitted too quickly for a human.',
            'js_proof_failed' => 'Browser proof token was missing or invalid.',
            'progressive_delay' => 'Progressive delay applied (challenge band).',
            'many_links' => 'Message contains many links.',
            'spam_keywords' => 'Spam keywords detected.',
            'link_markup' => 'Spam-like link markup detected.',
            'shortener' => 'URL shortener detected.',
            'suspicious_tld' => 'Suspicious top-level domain detected.',
            'obfuscation' => 'Obfuscation / Unicode trick patterns detected.',
            'entity_obfuscation' => 'HTML entity obfuscation detected.',
            'percent_encoding' => 'Percent-encoding obfuscation detected.',
            'zero_width_chars' => 'Zero-width Unicode characters detected.',
            'mixed_scripts' => 'Mixed Unicode scripts detected (confusables).',
            'high_unicode_ratio' => 'High ratio of non-ASCII Unicode detected.',
            'multiple_links' => 'Multiple URLs detected in message.',
            'spam_phrase_ngrams' => 'Spam phrase patterns matched (local model).',
            'similar_to_known_spam' => 'Similar to spam already blocked on this site.',
            'velocity_exceeded' => 'Burst submissions from same fingerprint.',
            'repeat_payload' => 'Repeated payload pattern across submissions.',
            'missing_user_agent' => 'User-Agent header missing.',
            'suspicious_user_agent' => 'User-Agent indicates automation.',
            'missing_accept_language' => 'Accept-Language header missing.',
            'missing_referer' => 'Referer header missing for submission.',
            'external_referer' => 'Submission referer did not match this site.',
            'fp_hold_protection' => 'False-positive protection held this event for review.',
            'softened_for_comments' => 'Softened for comments (held instead of blocked).',
            'allowed_by_list' => 'Allowed by allowlist rule.',
            'blocked_by_list' => 'Blocked by denylist rule.',
        );
        if (isset($map[$code])) { return $map[$code]; }
        $code2 = str_replace('_',' ', $code);
        $code2 = ucwords($code2);
        return $code2 . '.';
    }



    private function render_dashboard() {
        global $wpdb;
        $events = self::sanitize_db_prefix($wpdb->prefix) . 'asg_events';
        $fw     = self::sanitize_db_prefix($wpdb->prefix) . 'asg_firewall_events';

        // Cache dashboard aggregates briefly to avoid repeated COUNT/GROUP BY queries.
        $cache_key = 'asg_dash_counts_' . (function_exists('get_current_blog_id') ? (string) get_current_blog_id() : '0');
        $cached = wp_cache_get($cache_key, 'aegisspam');
        if (is_array($cached) && isset($cached['fw_today'], $cached['fw_7d'], $cached['fw_total'], $cached['counts_24h'], $cached['counts_7d'])) {
            $fw_today   = (int) $cached['fw_today'];
            $fw_7d      = (int) $cached['fw_7d'];
            $fw_total   = (int) $cached['fw_total'];
            $counts_24h = (array) $cached['counts_24h'];
            $counts_7d  = (array) $cached['counts_7d'];
        } else {
            $tz = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone('UTC');
            $midnight_local = new DateTime('now', $tz);
            $midnight_local->setTime(0, 0, 0);

            $midnight_utc = clone $midnight_local;
            $midnight_utc->setTimezone(new DateTimeZone('UTC'));
            $since_today = $midnight_utc->format('Y-m-d H:i:s');

            $since_7d_fw = gmdate('Y-m-d H:i:s', time() - 7 * 24 * 3600);

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $fw_today = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM %i WHERE action=%s AND created_at >= %s",
                    $fw,
                    'block',
                    $since_today
                )
            );

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $fw_7d = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM %i WHERE action=%s AND created_at >= %s",
                    $fw,
                    'block',
                    $since_7d_fw
                )
            );

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $fw_total = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM %i WHERE action=%s AND 1=%d",
                    $fw,
                    'block',
                    1
                )
            );

            $since_24h = gmdate('Y-m-d H:i:s', time() - 24 * 3600);
            $since_7d  = gmdate('Y-m-d H:i:s', time() - 7 * 24 * 3600);

            $counts_24h = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    "SELECT action, COUNT(*) as c FROM %i WHERE created_at >= %s GROUP BY action",
                    $events,
                    $since_24h
                ),
                ARRAY_A
            );

            $counts_7d = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    "SELECT action, COUNT(*) as c FROM %i WHERE created_at >= %s GROUP BY action",
                    $events,
                    $since_7d
                ),
                ARRAY_A
            );

            wp_cache_set($cache_key, array(
                'fw_today'   => $fw_today,
                'fw_7d'      => $fw_7d,
                'fw_total'   => $fw_total,
                'counts_24h' => $counts_24h,
                'counts_7d'  => $counts_7d,
            ), 'aegisspam', 60);
        }

        $map = function($rows){
            $out = array('allow'=>0,'challenge'=>0,'hold'=>0,'block'=>0);
            foreach ((array)$rows as $r) {
                $a = isset($r['action']) ? (string)$r['action'] : '';
                if ($a && isset($out[$a])) { $out[$a] = (int)$r['c']; }
            }
            return $out;
        };

        $c24 = $map($counts_24h);
        $c7  = $map($counts_7d);

        $settings = get_option('asg_settings', array());
        $fp_hold = !empty($settings['fp_hold_instead_of_block']);

        echo '<div class="asg-grid">';
        echo '  <div class="asg-card asg-card--wide">';
        echo '    <h2>At a glance</h2>';
        echo '    <div class="asg-metrics">';
        echo '      <div class="asg-metric"><div class="asg-metric__label">Last 24 hours</div><div class="asg-metric__value">' . esc_html(array_sum($c24)) . '</div><div class="asg-metric__sub">events scored</div></div>';
        echo '      <div class="asg-metric"><div class="asg-metric__label">Blocked</div><div class="asg-metric__value">' . esc_html($c24['block']) . '</div><div class="asg-metric__sub">last 24h</div></div>';
        echo '      <div class="asg-metric"><div class="asg-metric__label">Held</div><div class="asg-metric__value">' . esc_html($c24['hold']) . '</div><div class="asg-metric__sub">last 24h</div></div>';
        echo '      <div class="asg-metric"><div class="asg-metric__label">Challenged</div><div class="asg-metric__value">' . esc_html($c24['challenge']) . '</div><div class="asg-metric__sub">last 24h</div></div>';
        echo '    </div>';
        echo '    <div class="asg-mini-note">Tip: Use <strong>Hold</strong> for review when in doubt. Train spam/not spam to improve local learning.</div>';
        echo '  </div>';

        echo '  <div class="asg-card">';
        echo '    <h2>Protection mode</h2>';
        echo '    <p class="asg-kv"><span class="asg-kv__k">False positive protection</span><span class="asg-kv__v">' . ($fp_hold ? '<span class="asg-pill asg-pill--good">ON</span>' : '<span class="asg-pill">OFF</span>') . '</span></p>';
        echo '    <p class="description">When enabled, high-risk events that would be blocked are instead held for review (unless explicitly denied).</p>';
        echo '    <a class="button button-primary" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=settings')) . '">Configure</a>';
        echo '  </div>';

        echo '  <div class="asg-card">';
        echo '    <h2>Last 7 days</h2>';
        echo '    <div class="asg-stack">';
        echo '      <div class="asg-stack__row"><span>Blocked</span><strong>' . esc_html($c7['block']) . '</strong></div>';
        echo '      <div class="asg-stack__row"><span>Held</span><strong>' . esc_html($c7['hold']) . '</strong></div>';
        echo '      <div class="asg-stack__row"><span>Challenged</span><strong>' . esc_html($c7['challenge']) . '</strong></div>';
        echo '      <div class="asg-stack__row"><span>Allowed</span><strong>' . esc_html($c7['allow']) . '</strong></div>';
        echo '    </div>';
        echo '    <a class="button button-secondary" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=logs')) . '">Open Spam Log</a>';
        echo '  </div>';
        echo '  <div class="asg-card">';
        echo '    <h2>Firewall counters</h2>';
        echo '    <div class="asg-stack">';
        echo '      <div class="asg-stack__row"><span>Blocked today</span><strong>' . esc_html($fw_today) . '</strong></div>';
        echo '      <div class="asg-stack__row"><span>Blocked (7 days)</span><strong>' . esc_html($fw_7d) . '</strong></div>';
        echo '      <div class="asg-stack__row"><span>Blocked (total)</span><strong>' . esc_html($fw_total) . '</strong></div>';
        echo '    </div>';
        echo '    <a class="button button-secondary asg-btn" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=firewall')) . '">View Firewall</a>';
        echo '  </div>';

        $last = get_option('asg_cleanup_last', array());
        echo '  <div class="asg-card">';
        echo '    <h2>Cleanup impact</h2>';
        if (is_array($last) && !empty($last['ts'])) {
            $type_lbl = (!empty($last['type']) && $last['type'] === 'users') ? 'users' : 'comments';
            $changed = isset($last['changed']) ? intval($last['changed']) : 0;
            $flagged = isset($last['flagged']) ? intval($last['flagged']) : 0;
            $when = date_i18n(get_option('date_format') . ' ' . get_option('time_format'), intval($last['ts']));
            $headline = ($changed > 0) ? ('Removed ' . number_format_i18n($changed) . ' spam ' . $type_lbl) : ('Flagged ' . number_format_i18n($flagged) . ' suspicious ' . $type_lbl);
            echo '    <div class="asg-kv"><span class="asg-kv__k">Last run</span><span class="asg-kv__v"><strong>' . esc_html($headline) . '</strong></span></div>';
            echo '    <p class="description">Last cleanup: ' . esc_html($when) . '. Run Cleanup to remove legacy spam and keep your database lean.</p>';
        } else {
            echo '    <p class="description">No cleanup runs recorded yet. Use Cleanup to scan existing comments/users and remove legacy spam.</p>';
        }
        echo '    <a class="button button-secondary asg-btn" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=cleanup')) . '">Open Cleanup</a>';
        echo '  </div>';

        echo '  <div class="asg-card asg-card--wide">';
        echo '    <h2>Why we don\'t use CAPTCHA</h2>';
        echo '    <details class="asg-details" open>';
        echo '      <summary><strong>Better security, better UX, and fewer conversions lost</strong></summary>';
        echo '      <div class="asg-details__body">';
        echo '        <p>CAPTCHAs punish real people while sophisticated bots adapt quickly. AegisSpamGuard is built to be <strong>local-first</strong> and <strong>score-based</strong>: we evaluate content signals, identity heuristics, and behavior patterns to stop spam without forcing users to solve puzzles.</p>';
        echo '        <ul class="asg-ul">';
        echo '          <li><strong>Higher conversions:</strong> fewer friction points on comments, forms, and registration.</li>';
        echo '          <li><strong>Accessible by default:</strong> no image/audio challenges that frustrate users.</li>';
        echo '          <li><strong>Less data leakage:</strong> no third-party CAPTCHA scripts watching your visitors.</li>';
        echo '          <li><strong>Cleaner moderation:</strong> scored decisions with reasons you can audit and tune.</li>';
        echo '        </ul>';
        echo '        <p class="asg-mini-note">Tip: Use Challenges for “maybe-spam” flows instead of hard blocks when you want maximum UX.</p>';
        echo '      </div>';
        echo '    </details>';
        echo '  </div>';

        echo '</div>';
    }


    private function render_logs() {
        global $wpdb;
        $events = self::sanitize_db_prefix( $wpdb->prefix ) . 'asg_events';

        $per_page = 25;
        $paged = isset( $_GET['paged'] ) ? max( 1, absint( wp_unslash( $_GET['paged'] ) ) ) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- paging only (read-only).
        $offset = ( $paged - 1 ) * $per_page;

        // Validate the filter nonce only when filters are present (backward compatible for plain paging links).
        $nonce_ok = true;
        if ( isset( $_GET['type'] ) || isset( $_GET['action'] ) || isset( $_GET['q'] ) ) {
            $nonce_ok = ( isset( $_GET['_asgnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_asgnonce'] ) ), 'asg_logs_filters' ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- verified explicitly.
        }

        $type_raw   = ( $nonce_ok && ! empty( $_GET['type'] ) ) ? sanitize_key( wp_unslash( $_GET['type'] ) ) : '';
        $allowed_types = array( 'comment', 'registration', 'rest', 'cf7', 'wpforms', 'gravity', 'woo' );
        $type = in_array( $type_raw, $allowed_types, true ) ? $type_raw : '';

        $action_raw = ( $nonce_ok && ! empty( $_GET['action'] ) ) ? sanitize_key( wp_unslash( $_GET['action'] ) ) : '';
        $allowed_actions = array( 'block', 'hold', 'challenge', 'allow' );
        $action = in_array( $action_raw, $allowed_actions, true ) ? $action_raw : '';

        $q_raw = ( $nonce_ok && ! empty( $_GET['q'] ) ) ? sanitize_text_field( wp_unslash( $_GET['q'] ) ) : '';
        $q = ( $q_raw !== '' ) ? ( '%' . $wpdb->esc_like( $q_raw ) . '%' ) : '';

        $case = 0;
        if ( $type !== '' ) { $case |= 1; }
        if ( $action !== '' ) { $case |= 2; }
        if ( $q !== '' ) { $case |= 4; }

        $since_30d = gmdate( 'Y-m-d H:i:s', time() - 30 * DAY_IN_SECONDS );

        // Fixed SQL strings per filter combination (type/action/search) to satisfy static analysis.
        switch ( $case ) {
            case 0:
                $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i WHERE 1=%d", $events, 1 ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $series_rows = $wpdb->get_results( $wpdb->prepare( "SELECT DATE(created_at) as d, COUNT(*) as c FROM %i WHERE created_at >= %s AND 1=%d GROUP BY DATE(created_at) ORDER BY d ASC", $events, $since_30d, 1 ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE 1=%d ORDER BY id DESC LIMIT %d OFFSET %d", $events, 1, $per_page, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;

            case 1:
                $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i WHERE type=%s", $events, $type ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $series_rows = $wpdb->get_results( $wpdb->prepare( "SELECT DATE(created_at) as d, COUNT(*) as c FROM %i WHERE created_at >= %s AND type=%s GROUP BY DATE(created_at) ORDER BY d ASC", $events, $since_30d, $type ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE type=%s ORDER BY id DESC LIMIT %d OFFSET %d", $events, $type, $per_page, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;

            case 2:
                $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i WHERE action=%s", $events, $action ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $series_rows = $wpdb->get_results( $wpdb->prepare( "SELECT DATE(created_at) as d, COUNT(*) as c FROM %i WHERE created_at >= %s AND action=%s GROUP BY DATE(created_at) ORDER BY d ASC", $events, $since_30d, $action ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE action=%s ORDER BY id DESC LIMIT %d OFFSET %d", $events, $action, $per_page, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;

            case 3:
                $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i WHERE type=%s AND action=%s", $events, $type, $action ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $series_rows = $wpdb->get_results( $wpdb->prepare( "SELECT DATE(created_at) as d, COUNT(*) as c FROM %i WHERE created_at >= %s AND type=%s AND action=%s GROUP BY DATE(created_at) ORDER BY d ASC", $events, $since_30d, $type, $action ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE type=%s AND action=%s ORDER BY id DESC LIMIT %d OFFSET %d", $events, $type, $action, $per_page, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;

            case 4:
                $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i WHERE (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s)", $events, $q, $q, $q, $q ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $series_rows = $wpdb->get_results( $wpdb->prepare( "SELECT DATE(created_at) as d, COUNT(*) as c FROM %i WHERE created_at >= %s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) GROUP BY DATE(created_at) ORDER BY d ASC", $events, $since_30d, $q, $q, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) ORDER BY id DESC LIMIT %d OFFSET %d", $events, $q, $q, $q, $q, $per_page, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;

            case 5:
                $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i WHERE type=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s)", $events, $type, $q, $q, $q, $q ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $series_rows = $wpdb->get_results( $wpdb->prepare( "SELECT DATE(created_at) as d, COUNT(*) as c FROM %i WHERE created_at >= %s AND type=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) GROUP BY DATE(created_at) ORDER BY d ASC", $events, $since_30d, $type, $q, $q, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE type=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) ORDER BY id DESC LIMIT %d OFFSET %d", $events, $type, $q, $q, $q, $q, $per_page, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;

            case 6:
                $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i WHERE action=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s)", $events, $action, $q, $q, $q, $q ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $series_rows = $wpdb->get_results( $wpdb->prepare( "SELECT DATE(created_at) as d, COUNT(*) as c FROM %i WHERE created_at >= %s AND action=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) GROUP BY DATE(created_at) ORDER BY d ASC", $events, $since_30d, $action, $q, $q, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE action=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) ORDER BY id DESC LIMIT %d OFFSET %d", $events, $action, $q, $q, $q, $q, $per_page, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;

            default: // 7
                $total = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM %i WHERE type=%s AND action=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s)", $events, $type, $action, $q, $q, $q, $q ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $series_rows = $wpdb->get_results( $wpdb->prepare( "SELECT DATE(created_at) as d, COUNT(*) as c FROM %i WHERE created_at >= %s AND type=%s AND action=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) GROUP BY DATE(created_at) ORDER BY d ASC", $events, $since_30d, $type, $action, $q, $q, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE type=%s AND action=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) ORDER BY id DESC LIMIT %d OFFSET %d", $events, $type, $action, $q, $q, $q, $q, $per_page, $offset ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
        }

        $defs = $this->reason_definitions();

        echo '<div class="asg-card">';
        echo '<h2>Spam activity (search graph)</h2>';
        echo '<p class="description">Shows how many events match your current filters (type/action/search) over the last 30 days.</p>';

        $days = array();
        for ($i = 29; $i >= 0; $i--) {
            $d = gmdate('Y-m-d', time() - $i * 24 * 3600);
            $days[] = $d;
        }
        $vals = array();
        $maxv = 0;
        $all_zero = true;
        foreach ($days as $d) {
            $v = isset($series[$d]) ? (int)$series[$d] : 0;
            $vals[] = $v;
            if ($v > $maxv) { $maxv = $v; }
            if ($v > 0) { $all_zero = false; }
        }
        if ($maxv < 1) { $maxv = 1; }

        $w = 900; $h = 120; $pad = 8;
        $bar_w = floor(($w - 2*$pad) / count($vals));
        $svg = '<svg class="asg-chart" viewBox="0 0 ' . $w . ' ' . $h . '" role="img" aria-label="Spam activity graph">';
        $svg .= '<line x1="' . (int)$pad . '" y1="' . (int)($h - $pad) . '" x2="' . (int)($w - $pad) . '" y2="' . (int)($h - $pad) . '" class="asg-chart__axis" />';
        $x = $pad;
        for ($i=0; $i<count($vals); $i++) {
            $v = $vals[$i];
            $bh = $all_zero ? 2 : (int) floor((($h - 2*$pad) * $v) / $maxv);
            $y = $h - $pad - $bh;
            $opacity = $all_zero ? ' opacity="0.25"' : '';
            $svg .= '<rect x="' . (int)$x . '" y="' . (int)$y . '" width="' . (int)max(1,$bar_w-1) . '" height="' . (int)$bh . '" rx="2" ry="2"' . $opacity . '></rect>';
            $x += $bar_w;
        }
        if ($all_zero) {
            $svg .= '<text x="' . (int)($w/2) . '" y="' . (int)($h/2) . '" text-anchor="middle" class="asg-chart__empty">No events in the last 30 days</text>';
        }
		$svg .= '</svg>';

		$allowed_svg = array(
			'svg'  => array('class'=>true,'viewBox'=>true,'role'=>true,'aria-label'=>true,'xmlns'=>true),
			'line' => array('x1'=>true,'y1'=>true,'x2'=>true,'y2'=>true,'class'=>true),
			'rect' => array('x'=>true,'y'=>true,'width'=>true,'height'=>true,'rx'=>true,'ry'=>true,'opacity'=>true),
			'text' => array('x'=>true,'y'=>true,'text-anchor'=>true,'class'=>true),
		);

		echo wp_kses($svg, $allowed_svg);

        echo '<div class="asg-mini-note">Tip: Use the search box below to graph only a keyword (e.g., “casino”, “bitcoin”, “viagra”) before you bulk clean.</div>';
        echo '</div>';

echo '<div class="asg-card asg-help">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-list-view"></span> Live Spam Log</h3>';
        echo '<p>This is your site’s audit trail. Each event shows the <strong>score</strong>, the <strong>reasons</strong>, and the recommended next step. Use one-click actions to <strong>Allow</strong>, <strong>Block</strong>, and <strong>Train</strong> the local model.</p>';
        echo '<ul class="asg-bullets">';
        echo '<li><strong>Allow</strong>: Whitelist this fingerprint (fast way to stop false positives).</li>';
        echo '<li><strong>Block</strong>: Deny this fingerprint (fast way to stop repeated spam).</li>';
        echo '<li><strong>Train</strong>: Improves your on-site model over time (no external services).</li>';
        echo '</ul>';
        echo '</div>';
        $nonce = wp_create_nonce( 'asg_logs_filters' );
        echo '<form method="get" class="asg-toolbar">';
        echo '<input type="hidden" name="page" value="aegis-spam-guard">';
        echo '<input type="hidden" name="tab" value="logs">';
        echo '<input type="hidden" name="_asgnonce" value="' . esc_attr( $nonce ) . '">';
        echo '<div class="asg-toolbar__row">';
        echo '<div class="asg-toolbar__group">';
        echo '<select name="type">';
        $types = array('' => 'All Types', 'comment'=>'Comments', 'registration'=>'Registration', 'rest'=>'REST', 'cf7'=>'Contact Form 7', 'wpforms'=>'WPForms', 'gravity'=>'Gravity Forms', 'woo'=>'WooCommerce');
        foreach ($types as $k=>$v) {
        $sel = $type_raw;
            echo '<option value="' . esc_attr($k) . '" ' . selected($sel, $k, false) . '>' . esc_html($v) . '</option>';
        }
        echo '</select> ';
        echo '<select name="action">';
        $acts = array('' => 'All Actions', 'allow'=>'Allow', 'challenge'=>'Challenge', 'hold'=>'Hold', 'block'=>'Block');
        foreach ($acts as $k=>$v) {
        $sel = $action_raw;
            echo '<option value="' . esc_attr($k) . '" ' . selected($sel, $k, false) . '>' . esc_html($v) . '</option>';
        }
        echo '</select>';
        echo '</div>';

        echo '<div class="asg-toolbar__group asg-toolbar__search">';
        echo '<input type="search" name="q" placeholder="Search" value="' . esc_attr( $q_raw ) . '" />';
        echo '</div>';

        echo '<div class="asg-toolbar__group">';
        submit_button('Apply', 'secondary', '', false);
        echo '</div>';
        echo '</div>';
        echo '</form>';
        echo '<div class="asg-drawer" id="asg-drawer" aria-hidden="true">';
        echo '<div class="asg-drawer__overlay" data-asg-close></div>';
        echo '<div class="asg-drawer__panel" role="dialog" aria-modal="true" aria-label="Event details">';
        echo '<button type="button" class="asg-drawer__close" data-asg-close aria-label="Close">&times;</button>';
        echo '<div class="asg-drawer__content">';
        echo '<h3 class="asg-drawer__title">Event details</h3>';
        echo '<div id="asg-drawer-meta" class="asg-drawer__meta"></div>';
        echo '<h4>Reasons</h4><pre class="asg-pre" id="asg-drawer-reasons"></pre>';
        echo '<h4>Payload</h4><pre class="asg-pre" id="asg-drawer-payload"></pre>';
        echo '</div></div></div>';
        echo '<div class="asg-log-layout">';

        echo '<div class="asg-log-main">';

        $actions = array('' => 'All', 'block' => 'Blocked', 'hold' => 'Held', 'challenge' => 'Challenged', 'allow' => 'Allowed');
        $counts = array('allow'=>0,'challenge'=>0,'hold'=>0,'block'=>0,'total'=>0);

        switch ( $case ) {
            case 0:
                $tmp = $wpdb->get_results( $wpdb->prepare( "SELECT action, COUNT(*) c FROM %i WHERE 1=%d GROUP BY action", $events, 1 ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 1:
                $tmp = $wpdb->get_results( $wpdb->prepare( "SELECT action, COUNT(*) c FROM %i WHERE type=%s GROUP BY action", $events, $type ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 2:
                $tmp = $wpdb->get_results( $wpdb->prepare( "SELECT action, COUNT(*) c FROM %i WHERE action=%s GROUP BY action", $events, $action ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 3:
                $tmp = $wpdb->get_results( $wpdb->prepare( "SELECT action, COUNT(*) c FROM %i WHERE type=%s AND action=%s GROUP BY action", $events, $type, $action ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 4:
                $tmp = $wpdb->get_results( $wpdb->prepare( "SELECT action, COUNT(*) c FROM %i WHERE (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) GROUP BY action", $events, $q, $q, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 5:
                $tmp = $wpdb->get_results( $wpdb->prepare( "SELECT action, COUNT(*) c FROM %i WHERE type=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) GROUP BY action", $events, $type, $q, $q, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 6:
                $tmp = $wpdb->get_results( $wpdb->prepare( "SELECT action, COUNT(*) c FROM %i WHERE action=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) GROUP BY action", $events, $action, $q, $q, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            default:
                $tmp = $wpdb->get_results( $wpdb->prepare( "SELECT action, COUNT(*) c FROM %i WHERE type=%s AND action=%s AND (reasons LIKE %s OR payload LIKE %s OR fingerprint LIKE %s OR email_hash LIKE %s) GROUP BY action", $events, $type, $action, $q, $q, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
        }
        foreach ((array)$tmp as $trow) {
            $a = isset($trow['action']) ? (string)$trow['action'] : '';
            $c = isset($trow['c']) ? intval($trow['c']) : 0;
            if (isset($counts[$a])) { $counts[$a] = $c; }
            $counts['total'] += $c;
        }

        $req_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $base = remove_query_arg(array('paged'), $req_uri);
        echo '<div class="asg-pills">';
        foreach ($actions as $akey => $alabel) {
            $is_current = ((string)$akey === (string)$action);
            $url = $base;
            if ($akey === '') { $url = remove_query_arg('action', $url); } else { $url = add_query_arg('action', $akey, $url); }
            $count = ($akey === '') ? $counts['total'] : (isset($counts[$akey]) ? intval($counts[$akey]) : 0);
            echo '<a class="asg-pill ' . ($is_current ? 'asg-pill--active' : '') . '" href="' . esc_url($url) . '">';
            echo esc_html($alabel) . ' <span class="asg-pill__count">' . intval($count) . '</span>';
            echo '</a>';
        }
        echo '</div>';

        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" class="asg-bulk-form">';
        echo '<input type="hidden" name="action" value="asg_bulk_event_action">';
        wp_nonce_field('asg_bulk_event_action');

        echo '<div class="asg-bulkbar" id="asg-bulkbar" style="display:none;">';
        echo '<div class="asg-bulkbar__left"><strong><span id="asg-selected-count">0</span></strong> selected</div>';
        echo '<div class="asg-bulkbar__right">';
        echo '<select name="do" id="asg-bulk-do">';
        echo '<option value="">Bulk action…</option>';
        echo '<option value="allow_fp">Allow (Fingerprint)</option>';
        echo '<option value="deny_fp">Block (Fingerprint)</option>';
        echo '<option value="train_spam">Train as Spam</option>';
        echo '<option value="train_ham">Train as Not Spam</option>';
        echo '</select> ';
        submit_button('Apply', 'primary', 'submit', false);
        echo '</div>';
        echo '</div>';

        echo '<div class="asg-log-cards">';
        if (!empty($rows)) {
            foreach ($rows as $r) {
                $id = intval($r['id']);
                $typev = isset($r['type']) ? (string)$r['type'] : '';
                $act  = isset($r['action']) ? (string)$r['action'] : '';
                $scorev = isset($r['score']) ? intval($r['score']) : 0;
                $time = isset($r['created_at']) ? (string)$r['created_at'] : '';
                $fp = isset($r['fingerprint']) ? (string)$r['fingerprint'] : '';
                $reasons_raw = isset($r['reasons']) ? (string) $r['reasons'] : '';
                $reasons_arr = json_decode($reasons_raw, true);
                if (!is_array($reasons_arr)) { $reasons_arr = array(); }

                $payload_arr = array();
                if (!empty($r['payload'])) {
                    $payload_arr = json_decode((string)$r['payload'], true);
                    if (!is_array($payload_arr)) { $payload_arr = array(); }
                }
                $breakdown = array();
                if (!empty($payload_arr['meta']) && is_array($payload_arr['meta']) && !empty($payload_arr['meta']['score_breakdown']) && is_array($payload_arr['meta']['score_breakdown'])) {
                    $breakdown = $payload_arr['meta']['score_breakdown'];
                }
                $breakdown_json = wp_json_encode($breakdown);
                $content_snip = '';
                if (!empty($payload_arr['content'])) {
                    $content_snip = wp_strip_all_tags((string)$payload_arr['content']);
                    $content_snip = preg_replace('/\s+/', ' ', $content_snip);
                    $content_snip = mb_substr($content_snip, 0, 160);
                }

                echo '<div class="asg-log-card" data-asg-breakdown=\'' . esc_attr($breakdown_json) . '\' data-asg-id="' . intval($id) . '">';
                echo '<div class="asg-log-card__summary">';
                echo '<label class="asg-check"><input type="checkbox" class="asg-select" name="ids[]" value="' . intval($id) . '"><span></span></label>';
                echo '<div class="asg-log-card__meta">';
                echo '<div class="asg-log-card__top">';
                echo '<span class="asg-badge asg-badge--type">' . esc_html($typev) . '</span>';
                echo '<span class="asg-badge asg-badge--action asg-badge--' . esc_attr($act) . '">' . esc_html(strtoupper($act)) . '</span>';
                echo '<span class="asg-badge asg-badge--score">' . intval($scorev) . '</span>';
                echo '<span class="asg-muted">#' . intval($id) . ' · ' . esc_html($time) . '</span>';
                echo '</div>';

                echo '<div class="asg-log-card__title">';
                echo '<strong>' . esc_html($content_snip ? $content_snip : 'Event') . '</strong>';
                echo '</div>';

                echo '<div class="asg-log-card__reasons">';
                $shown = 0;
                foreach ($reasons_arr as $rr) {
                    if ($shown >= 6) { break; }
                    $rrs = (string)$rr; $tip = isset($defs[$rrs]) ? (string)$defs[$rrs] : ''; echo '<span class="asg-pill-mini" title="' . esc_attr($tip) . '">' . esc_html($rrs) . '</span>';
                    $shown++;
                }
                if (count($reasons_arr) > 6) {
                    echo '<span class="asg-pill-mini asg-pill-mini--more">+' . intval(count($reasons_arr)-6) . '</span>';
                }
                echo '</div>';

                echo '</div>'; // meta

                echo '<div class="asg-log-card__actions">';
                $base_action = admin_url('admin-post.php?action=asg_event_action&event_id=' . intval($id));
                echo '<a class="button button-small" href="' . esc_url(wp_nonce_url($base_action . '&do=allow_fp', 'asg_event_action_' . $id)) . '">Allow</a> ';
                echo '<a class="button button-small" href="' . esc_url(wp_nonce_url($base_action . '&do=deny_fp', 'asg_event_action_' . $id)) . '">Block</a> ';
                echo '<button type="button" class="button button-small asg-toggle-details" aria-expanded="false">Details</button>';
                echo '</div>';

                echo '</div>'; // summary

                echo '<div class="asg-log-card__body" style="display:none;">';
                echo '<div class="asg-log-card__bodygrid">';
                echo '<div class="asg-log-card__bodymain">';
                echo '<h4>Reasons (what fired)</h4>';
                $bd_map = array();
                if (is_array($breakdown)) {
                    foreach ($breakdown as $bdi) {
                        if (!empty($bdi['code'])) { $bd_map[(string)$bdi['code']] = $bdi; }
                    }
                }
                echo '<div class="asg-reason-list">';
                if (!empty($reasons_arr)) {
                    foreach ($reasons_arr as $rr2) {
                        $code = (string)$rr2;
                        $label = $this->reason_sentence($code);
                        $tip2 = isset($defs[$code]) ? (string)$defs[$code] : '';
                        $pts = '';
                        if (isset($bd_map[$code]) && isset($bd_map[$code]['points'])) { $pts = (string)$bd_map[$code]['points']; }
                        echo '<div class="asg-reason-item">';
                        echo '<span class="asg-reason-code" title="' . esc_attr($tip2) . '">' . esc_html($code) . '</span>';
                        if ($pts !== '') { echo '<span class="asg-reason-pts">' . esc_html($pts) . '</span>'; }
                        if ($tip2) { echo '<div class="asg-reason-desc">' . esc_html($tip2) . '</div>'; }
                        echo '</div>';
                    }
                } else {
                    echo '<div class="asg-muted">No reasons recorded.</div>';
                }
                echo '</div>';
                echo '<h4>Payload</h4>';
                echo '<pre class="asg-pre">' . esc_html(wp_json_encode($payload_arr, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)) . '</pre>';
                echo '</div>';
                echo '<div class="asg-log-card__bodyside">';
                echo '<h4>Training</h4>';
                echo '<p class="asg-muted">Improve local learning for similar messages.</p>';
                echo '<a class="button button-secondary button-small" href="' . esc_url(wp_nonce_url($base_action . '&do=train_spam', 'asg_event_action_' . $id)) . '">Train Spam</a> ';
                echo '<a class="button button-secondary button-small" href="' . esc_url(wp_nonce_url($base_action . '&do=train_ham', 'asg_event_action_' . $id)) . '">Train Not Spam</a>';
                echo '<hr class="asg-hr">';
                echo '<div class="asg-muted"><strong>Fingerprint</strong><br><code>' . esc_html($fp) . '</code></div>';
                echo '</div>';
                echo '</div>';
                echo '</div>'; 

                echo '</div>'; 
            }
        } else {
            echo '<div class="asg-empty">No events found for these filters.</div>';
        }
        echo '</div>'; 

        echo '</form>'; 
        echo '</div>'; 

        echo '<aside class="asg-log-sidebar">';
        echo '<div class="asg-card asg-sidebar-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-analytics"></span> What happened?</h3>';
        echo '<p class="asg-muted">Select an event or open details to see how its score was computed.</p>';
        echo '<div id="asg-whathappened-empty" class="asg-muted">No event selected.</div>';
        echo '<div id="asg-whathappened" style="display:none;">';
        echo '<div class="asg-wh-row"><span>Event</span> <strong id="asg-wh-id"></strong></div>';
        echo '<div class="asg-wh-row"><span>Score</span> <strong id="asg-wh-score"></strong></div>';
        echo '<h4 class="asg-wh-title">Score breakdown</h4>';
        echo '<div id="asg-wh-breakdown"></div>';
        echo '<p class="asg-muted">Points reflect the signals that fired for this event (after weight multipliers). Tune weights in <a href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=settings')) . '">Settings</a>.</p>';
        echo '</div>';
        echo '</div>';
        echo '</aside>';

        echo '</div>'; 

        $pages = (int) ceil($total / $per_page);
        if ($pages > 1) {
            $current_url = remove_query_arg(array('paged'), isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '');
            echo '<div class="asg-pagination">';
            for ($p=1; $p<=$pages; $p++) {
                $url = add_query_arg('paged', $p, $current_url);
                if ($p === $paged) {
                    echo '<span class="asg-page asg-page--current">' . intval($p) . '</span>';
                } else {
                    echo '<a class="asg-page" href="' . esc_url($url) . '">' . intval($p) . '</a>';
                }
            }
            echo '</div>';
        }
    }

    private function action_btn($base, $event_id, $nonce, $action, $label) {
        $url = add_query_arg(array(
            'action' => 'asg_event_action',
            'event_id' => $event_id,
            'do' => $action,
            '_wpnonce' => $nonce,
        ), $base);
        return '<a class="button button-small" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
    }

    
    private function render_lists() {
        global $wpdb;
        $lists = self::sanitize_db_prefix($wpdb->prefix) . 'asg_lists';

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only filter parameters.
        $type_filter_raw = isset($_GET['lt']) ? sanitize_key(wp_unslash($_GET['lt'])) : '';
        $type_filter = in_array($type_filter_raw, array('allow','deny'), true) ? $type_filter_raw : '';

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only filter parameters.
        $kind_filter_raw = isset($_GET['kind']) ? sanitize_key(wp_unslash($_GET['kind'])) : '';
        $allowed_kinds = array('ip','cidr','email','domain','fingerprint');
        $kind_filter = in_array($kind_filter_raw, $allowed_kinds, true) ? $kind_filter_raw : '';

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only filter parameters.
        $q_raw = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $q = ($q_raw !== '') ? ('%' . $wpdb->esc_like($q_raw) . '%') : '';

        $case = 0;
        if ( $type_filter !== '' ) { $case |= 1; }
        if ( $kind_filter !== '' ) { $case |= 2; }
        if ( $q !== '' ) { $case |= 4; }

        switch ( $case ) {
            case 0:
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i ORDER BY id DESC LIMIT 500", $lists ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 1:
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE list_type=%s ORDER BY id DESC LIMIT 500", $lists, $type_filter ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 2:
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE kind=%s ORDER BY id DESC LIMIT 500", $lists, $kind_filter ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 3:
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE list_type=%s AND kind=%s ORDER BY id DESC LIMIT 500", $lists, $type_filter, $kind_filter ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 4:
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE (value LIKE %s OR note LIKE %s) ORDER BY id DESC LIMIT 500", $lists, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 5:
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE list_type=%s AND (value LIKE %s OR note LIKE %s) ORDER BY id DESC LIMIT 500", $lists, $type_filter, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            case 6:
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE kind=%s AND (value LIKE %s OR note LIKE %s) ORDER BY id DESC LIMIT 500", $lists, $kind_filter, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
            default:
                $rows = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %i WHERE list_type=%s AND kind=%s AND (value LIKE %s OR note LIKE %s) ORDER BY id DESC LIMIT 500", $lists, $type_filter, $kind_filter, $q, $q ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                break;
        }


        echo '<div class="asg-card asg-help">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-lock"></span> Allow / Deny Lists</h3>';
        echo '<p>Lists are your <strong>instant rules</strong>. They bypass scoring: <strong>Allow</strong> trusts immediately, <strong>Deny</strong> blocks immediately.</p>';
        echo '<div class="asg-legend">';
        echo '<span class="asg-pill asg-pill--allow">Allow</span> trusted immediately ';
        echo '<span class="asg-pill asg-pill--block">Deny</span> blocked immediately';
        echo '</div>';
        echo '<div class="asg-mini-note">Example: add a <strong>Fingerprint</strong> deny rule to stop “same spam, different IP”. Add a <strong>CIDR</strong> allow rule to trust an office/network range.</div>';
        echo '</div>';

        $req_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        $base = remove_query_arg(array('paged'), $req_uri);
        $tabs = array('' => 'All', 'deny' => 'Deny', 'allow' => 'Allow');
        echo '<div class="asg-pills">';
        foreach ($tabs as $k=>$lbl) {
            $is = ((string)$k === (string)$type_filter);
            $url = $base;
            if ($k === '') { $url = remove_query_arg('lt', $url); } else { $url = add_query_arg('lt', $k, $url); }
            echo '<a class="asg-pill ' . ($is ? 'asg-pill--active' : '') . '" href="' . esc_url($url) . '">' . esc_html($lbl) . '</a>';
        }
        echo '</div>';

        echo '<div class="asg-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-plus"></span> Add a rule</h3>';
        echo '<p class="asg-muted">Rules apply immediately. Use a note so future you knows why it exists.</p>';
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" class="asg-form-inline asg-form-inline--wrap">';
        echo '<input type="hidden" name="action" value="asg_add_list">';
        wp_nonce_field('asg_add_list');

        echo '<div class="asg-field"><label>Type</label>';
        echo '<select name="list_type"><option value="allow">Allow</option><option value="deny">Deny</option></select></div>';

        echo '<div class="asg-field"><label>Kind</label>';
        echo '<select name="kind">';
        echo '<option value="ip">IP</option>';
        echo '<option value="cidr">CIDR</option>';
        echo '<option value="email">Email Hash</option>';
        echo '<option value="domain">Domain</option>';
        echo '<option value="fingerprint">Fingerprint</option>';
        echo '</select>';
        echo '<div class="asg-field__help">IP: <code>203.0.113.4</code> · CIDR: <code>203.0.113.0/24</code> · Domain: <code>example.com</code></div>';
        echo '</div>';

        echo '<div class="asg-field asg-field--grow"><label>Value</label><input type="text" name="value" class="regular-text" placeholder="203.0.113.4 or example.com or 203.0.113.0/24"></div>';
        echo '<div class="asg-field asg-field--grow"><label>Note</label><input type="text" name="note" class="regular-text" placeholder="Why this rule exists (optional)"></div>';

        submit_button('Add rule', 'primary', 'submit', false);
        echo '</form>';
        echo '</div>';

        echo '<div class="asg-card">';
        echo '<div class="asg-toolbar">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-list-view"></span> Current rules</h3>';
        echo '<form method="get" class="asg-search">';
        echo '<input type="hidden" name="page" value="aegis-spam-guard"><input type="hidden" name="tab" value="lists">';
        if ($type_filter) echo '<input type="hidden" name="lt" value="' . esc_attr($type_filter) . '">';
        if ($kind_filter) echo '<input type="hidden" name="kind" value="' . esc_attr($kind_filter) . '">';
        echo '<input type="search" name="q" value="' . esc_attr($q) . '" placeholder="Search value or note…"> ';
        submit_button('Search', 'secondary', 'submit', false);
        echo '</form>';
        echo '</div>';

        if (!empty($rows)) {
            echo '<div class="asg-rule-cards">';
            foreach ($rows as $r) {
                $id = intval($r['id']);
                $del_url = wp_nonce_url(add_query_arg(array('action'=>'asg_delete_list','id'=>$id), admin_url('admin-post.php')), 'asg_delete_list');
                $pill = ($r['list_type'] === 'allow') ? 'allow' : 'block';
                $kind = (string)$r['kind'];
                $hint = '';
                if ($kind === 'ip') { $hint = 'Exact IP match.'; }
                elseif ($kind === 'cidr') { $hint = 'IP range match.'; }
                elseif ($kind === 'email') { $hint = 'Hashed email match (privacy-safe).'; }
                elseif ($kind === 'domain') { $hint = 'Email domain match.'; }
                elseif ($kind === 'fingerprint') { $hint = 'Strong “same spam” identifier.'; }

                echo '<div class="asg-rule-card">';
                echo '<div class="asg-rule-card__top">';
                echo '<div class="asg-rule-card__badges">';
                echo '<span class="asg-pill asg-pill--' . esc_attr($pill) . '">' . esc_html($r['list_type']) . '</span>';
                echo '<span class="asg-pill asg-pill--type">' . esc_html($kind) . '</span>';
                echo '</div>';
                echo '<div class="asg-rule-card__meta asg-muted">#' . esc_html($id) . ' · ' . esc_html($r['created_at']) . '</div>';
                echo '</div>';
                echo '<div class="asg-rule-card__value"><code>' . esc_html($r['value']) . '</code></div>';
                if (!empty($r['note'])) echo '<div class="asg-rule-card__note">' . esc_html($r['note']) . '</div>';
                if ($hint) echo '<div class="asg-rule-card__hint asg-muted">' . esc_html($hint) . '</div>';
                echo '<div class="asg-rule-card__actions"><a class="button button-small" href="' . esc_url($del_url) . '">Delete</a></div>';
                echo '</div>';
            }
            echo '</div>';
        } else {
            echo '<div class="asg-empty">No rules found.</div>';
        }

        echo '</div>';
    }



    private function render_settings() {
        $opt = get_option('asg_settings', array());
        if (!is_array($opt)) { $opt = array(); }

        echo '<h2 class="asg-h2">Settings</h2>';

        echo '<div class="asg-help asg-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-info-outline"></span> How scoring works</h3>';
        echo '<p>AegisSpamGuard assigns a <strong>risk score (0–100)</strong> from multiple signals, then applies your policy thresholds to decide <em>Allow</em>, <em>Challenge</em>, <em>Hold</em>, or <em>Block</em>. Use the toggles and weights below to tune your site without guessing.</p>';
        echo '</div>';

        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" class="asg-form">';
        echo '<input type="hidden" name="action" value="asg_save_settings">';
        wp_nonce_field('asg_save_settings');

        $sections = array(
            array(
                'icon' => 'dashicons-shield',
                'title' => 'Policy thresholds',
                'desc'  => 'Controls what happens at each score range (Allow → Challenge → Hold → Block).',
                'fields'=> array(
                    array('k'=>'threshold_allow','t'=>'number','label'=>'Allow max score','desc'=>'Scores at or below this are allowed. Example: 0–29 allow.'),
                    array('k'=>'threshold_challenge','t'=>'number','label'=>'Challenge max score','desc'=>'Scores up to this trigger a soft challenge (honeypot/JS token/CAPTCHA). Example: 30–59 challenge.'),
                    array('k'=>'threshold_hold','t'=>'number','label'=>'Hold max score','desc'=>'Scores up to this are held for review (not published). Example: 60–79 hold. Scores above this are blocked.'),
                    array('k'=>'fp_hold_instead_of_block','t'=>'bool','label'=>'False positive protection','desc'=>'When enabled, events that would be blocked are instead <strong>held</strong> for manual review (safer during tuning).'),
                ),
            ),
            array(
                'icon' => 'dashicons-dashboard',
                'title' => 'Behavior & velocity',
                'desc'  => 'Stops bot bursts and “instant submits” that humans rarely do.',
                'fields'=> array(
                    array('k'=>'min_seconds_to_submit','t'=>'number','label'=>'Min seconds to submit','desc'=>'Default 3 seconds. Increase (e.g., 5–8) for longer forms so “too fast” submissions score higher.'),
                    array('k'=>'velocity_window_seconds','t'=>'number','label'=>'Velocity window (seconds)','desc'=>'How far back to count bursts. Example: 600 = 10 minutes.'),
                    array('k'=>'velocity_limit','t'=>'number','label'=>'Velocity limit (submissions per window)','desc'=>'Max submissions allowed per IP/session within the window before scoring increases. Example: 5 per 10 minutes.'),
                ),
            ),
            array(
                'icon' => 'dashicons-admin-users',
                'title' => 'Trust levels',
                'desc'  => 'Reduce false positives by trusting known users.',
                'fields'=> array(
                    array('k'=>'trust_logged_in','t'=>'bool','label'=>'Trust logged-in users','desc'=>'When enabled, logged-in users receive a score reduction to avoid blocking members.'),
                    array('k'=>'trust_customers','t'=>'bool','label'=>'Trust WooCommerce customers','desc'=>'When enabled (and WooCommerce is installed), customers with orders get a score reduction.'),
                    array('k'=>'trust_existing_user_days','t'=>'number','label'=>'Trust existing users if older than (days)','desc'=>'Example: 7 days. Older accounts are treated as more trustworthy. Set 0 to disable.'),
                ),
            ),
            array(
                'icon' => 'dashicons-privacy',
                'title' => 'Privacy & storage',
                'desc'  => 'Control what you store and whether anything ever leaves your server.',
                'fields'=> array(
                    array('k'=>'ip_mode','t'=>'select','label'=>'IP handling mode','desc'=>'Store raw IP, anonymize, hash, or store no IP at all.',
                        'options'=>array('store'=>'Store (raw IP)','anonymize'=>'Anonymize (safer logs)','hash'=>'Hash (no raw IP)','off'=>'Off (no IP storage)')),
                    array('k'=>'no_external_calls','t'=>'bool','label'=>'No external calls (recommended)','desc'=>'When enabled, the plugin will not fetch external lists or call any third-party service. Everything runs locally.'),
                    array('k'=>'log_retention_days','t'=>'number','label'=>'Log retention (days)','desc'=>'Auto-purge older events for privacy and performance. Example: 30 or 90 days.'),
                ),
            ),
            array(
                'icon' => 'dashicons-email',
                'title' => 'Identity checks',
                'desc'  => 'Catches disposable emails, missing MX records, and mismatch patterns.',
                'fields'=> array(
                    array('k'=>'enable_mx_checks','t'=>'bool','label'=>'Enable MX checks','desc'=>'When enabled, checks if the email domain has MX records (cached). Missing MX increases score.'),
                    array('k'=>'enable_external_disposable','t'=>'bool','label'=>'Enable external disposable list','desc'=>'Optional. Fetches a disposable-domain list from the URL below (cached). Disabled automatically when “No external calls” is enabled.'),
                    array('k'=>'external_disposable_url','t'=>'text','label'=>'External disposable list URL','desc'=>'Example: a raw text list of domains (one per line). Leave blank to rely on built-in local list.'),
                ),
            ),
            array(
                'icon' => 'dashicons-admin-site-alt3',
                'title' => 'Geo / ASN rules',
                'desc'  => 'Optional country/ASN filtering (cached). Choose a provider or disable completely.',
                'fields'=> array(
                    array('k'=>'enable_geo_rules','t'=>'bool','label'=>'Enable country rules','desc'=>'When enabled, applies allow/deny logic based on detected country.'),
                    array('k'=>'enable_asn_rules','t'=>'bool','label'=>'Enable ASN rules','desc'=>'When enabled, applies allow/deny logic based on detected ASN.'),
                    array('k'=>'geo_provider','t'=>'select','label'=>'Geo provider','desc'=>'Headers = read from edge headers (fast, no files). MaxMind = use local .mmdb files if available.',
                        'options'=>array('headers'=>'Headers','maxmind'=>'MaxMind (local file)','none'=>'None')),
                    array('k'=>'maxmind_country_mmdb','t'=>'text','label'=>'MaxMind Country .mmdb path','desc'=>'Absolute server path to GeoLite2-Country.mmdb (optional).'),
                    array('k'=>'maxmind_asn_mmdb','t'=>'text','label'=>'MaxMind ASN .mmdb path','desc'=>'Absolute server path to GeoLite2-ASN.mmdb (optional).'),
                ),
            ),
            array(
                'icon' => 'dashicons-admin-network',
                'title' => 'Content intelligence',
                'desc'  => 'Detects spammy links, obfuscation, phrase patterns, and similarity to known spam.',
                'fields'=> array(
                    array('k'=>'enable_similarity','t'=>'bool','label'=>'Enable similarity to known spam','desc'=>'Compares submissions to recently blocked spam on this site (local). High similarity increases score.'),
                    array('k'=>'similarity_lookback_days','t'=>'number','label'=>'Similarity lookback (days)','desc'=>'How far back to compare against blocked events. Example: 30 days.'),
                    array('k'=>'similarity_threshold','t'=>'text','label'=>'Similarity threshold (0–1)','desc'=>'Default 0.72. Higher = stricter match required (fewer false positives).'),
                ),
            ),
            array(
                'icon' => 'dashicons-yes-alt',
                'title' => 'Challenges',
                'desc'  => 'Soft friction for “maybe” spam (better UX than hard blocks).',
                'fields'=> array(
                    array('k'=>'enable_cf7_captcha','t'=>'bool','label'=>'Enable CF7 built-in math challenge','desc'=>'Only shown when a submission lands in the <em>Challenge</em> score range. No third-party CAPTCHA.'),
                ),
            ),
            array(
                'icon' => 'dashicons-chart-area',
                
array(
    'icon' => 'dashicons-admin-site',
    'title' => 'Site-wide protection',
    'desc'  => 'Protects custom forms, searches, and page requests — even when a plugin has no dedicated adapter.',
    'fields'=> array(
        array('k'=>'enable_firewall_precheck','t'=>'bool','label'=>'Spam firewall pre-check','help'=>'Blocks or slows obvious bot traffic before WordPress does heavy work. Uses allow/deny rules + local velocity.'),
        array('k'=>'firewall_blank_page','t'=>'bool','label'=>'Firewall returns blank page','help'=>'When blocked, return an empty 403 response (fast + bot-unfriendly). Uncheck to show a friendly blocked message.'),
        array('k'=>'firewall_hits_10m_threshold','t'=>'number','label'=>'Firewall velocity threshold (10m)','help'=>'Default: 120. If a fingerprint exceeds this many hits in ~10 minutes, AegisSpamGuard will challenge/block early. Lower for small sites, higher for busy sites.'),
        array('k'=>'enable_generic_post_protection','t'=>'bool','label'=>'Generic POST / any form protection','help'=>'Catches custom theme forms and unknown plugins by scoring any front-end POST. Skips known adapters (CF7/WPForms/Gravity) to avoid double-handling.'),
        array('k'=>'generic_post_block_on_challenge','t'=>'bool','label'=>'Generic POST: block on Challenge band','help'=>'If enabled, “Challenge” results block unknown forms after applying a light delay. Disable if you want Challenge to pass through.'),
        array('k'=>'enable_generic_post_include_ajax','t'=>'bool','label'=>'Generic POST: include admin-ajax.php','help'=>'Also protect front-end AJAX POSTs (admin-ajax.php). Safer default is OFF. If enabled, use the allowlist below to avoid breaking legitimate AJAX actions.'),
        array('k'=>'enable_generic_post_include_admin_post','t'=>'bool','label'=>'Generic POST: include admin-post.php','help'=>'Also protect front-end admin-post.php POSTs. Safer default is OFF. If enabled, allowlist the action names your site uses.'),
        array('k'=>'generic_post_allow_ajax_actions','t'=>'text','label'=>'Allowlisted AJAX actions','help'=>'Newline-separated list of admin-ajax.php action names to SKIP from generic protection. Example: wpforms_submit\nmy_custom_ajax_action'),
        array('k'=>'generic_post_allow_admin_post_actions','t'=>'text','label'=>'Allowlisted admin-post actions','help'=>'Newline-separated list of admin-post.php action names to SKIP from generic protection. Example: my_form_submit\nnewsletter_optin'),
        array('k'=>'generic_post_allow_paths','t'=>'text','label'=>'Allowlisted paths','help'=>'Newline-separated path fragments to SKIP from generic protection. Example: /my-safe-endpoint/\n/wc-api/'),
        array('k'=>'enable_search_protection','t'=>'bool','label'=>'Search form protection','help'=>'Stops bots from spamming your site search (SEO spam pages + CPU burn). Challenges rate spikes; blocks high-risk bursts.'),
    ),
),
'title' => 'Local learning',
                'desc'  => 'A tiny on-site model that adapts to your site. Train from the Logs tab.',
                'fields'=> array(
                    array('k'=>'enable_ml','t'=>'bool','label'=>'Enable local ML','desc'=>'When enabled, the Naive Bayes signal contributes to the score. Use “Train Spam / Train Not Spam” in Logs.'),
                    array('k'=>'ml_weight','t'=>'number','label'=>'ML weight (points)','desc'=>'How many points ML can add. Example: 12. Raise slowly after you have training data.'),
                ),
            ),
            array(
                'icon' => 'dashicons-sliders',
                'title' => 'Signal weights',
                'desc'  => 'Low/Med/High multipliers per signal. Start with Med everywhere, then tune based on Logs.',
                'fields'=> array(
                    array('k'=>'w_link_count','t'=>'weight','label'=>'Links in content','desc'=>'More links often means spam. Set High if you get link spam.'),
                    array('k'=>'w_spam_keywords','t'=>'weight','label'=>'Spam keywords','desc'=>'Boosts known spam phrases. Set High if you see repeated sales pitches.'),
                    array('k'=>'w_link_markup','t'=>'weight','label'=>'Link markup','desc'=>'Detects spammy anchor/HTML patterns.'),
                    array('k'=>'w_url_shortener','t'=>'weight','label'=>'URL shorteners','desc'=>'Shorteners are frequently abused.'),
                    array('k'=>'w_honeypot','t'=>'weight','label'=>'Honeypot','desc'=>'Hidden field filled = strong bot signal.'),
                    array('k'=>'w_too_fast','t'=>'weight','label'=>'Too fast submit','desc'=>'Submissions faster than your threshold.'),
                    array('k'=>'w_js_proof','t'=>'weight','label'=>'JS proof token','desc'=>'Missing JS token = likely bot (for supported forms).'),
                    array('k'=>'w_invalid_email','t'=>'weight','label'=>'Invalid email format','desc'=>'Basic validation signal.'),
                    array('k'=>'w_disposable_email','t'=>'weight','label'=>'Disposable email','desc'=>'Disposable domains increase score.'),
                    array('k'=>'w_mx_missing','t'=>'weight','label'=>'Missing MX','desc'=>'Email domain without MX records.'),
                    array('k'=>'w_name_email_mismatch','t'=>'weight','label'=>'Name/email mismatch','desc'=>'Detects random local-parts that don’t resemble the submitted name.'),
                ),
            ),
        );

        foreach ($sections as $sec) {

            $is_pro_section = in_array((string) $sec['title'], array(
                'Content intelligence',
                'Challenges',
                'Local learning',
                'Signal weights',
            ), true);

            $locked = ($is_pro_section && !$this->is_pro_active());

            echo '<div class="asg-card asg-section' . ($locked ? ' asg-pro-locked' : '') . '">';
            if ($locked) {
                echo '<div class="asg-pro-lockbar"><span class="asg-pro-pill">PRO</span> This block is available in <strong>AegisSpamGuard PRO</strong>. <a class="button button-primary asg-btn asg-btn--pro" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=license')) . '">Upgrade to PRO</a></div>';
            }
            echo '<div class="asg-section__head">';
            echo '<span class="dashicons ' . esc_attr($sec['icon']) . ' asg-section__icon"></span>';
            echo '<div><h3 class="asg-section__title">' . esc_html($sec['title']) . '</h3>';
            echo '<p class="asg-section__desc">' . wp_kses_post($sec['desc']) . '</p></div>';
            echo '</div>';
            echo '<table class="form-table asg-table"><tbody>';

            foreach ($sec['fields'] as $f) {
                $k = $f['k'];
                $label = $f['label'];
                $desc = isset($f['desc']) ? $f['desc'] : '';
                $val = isset($opt[$k]) ? $opt[$k] : '';

                echo '<tr><th scope="row"><label for="' . esc_attr($k) . '">' . esc_html($label) . '</label></th><td>';

                $disabled = $locked ? ' disabled' : '';

                if ($f['t'] === 'bool') {
                    $checked = !empty($val) ? ' checked' : '';
                    echo '<label class="asg-checkmark">';
                    echo '<input type="checkbox" id="' . esc_attr($k) . '" name="' . esc_attr($k) . '" value="1"' . esc_attr($checked) . esc_attr($disabled) . ' />';
                    echo '<span class="asg-checkmark__box" aria-hidden="true"></span>';
                    echo '<span class="asg-checkmark__text">' . (!empty($val) ? esc_html__('Enabled', 'aegisspam') : esc_html__('Disabled', 'aegisspam')) . '</span>';
                    echo '</label>';
                } elseif ($f['t'] === 'select') {
                    echo '<select id="' . esc_attr($k) . '" name="' . esc_attr($k) . '" ' . esc_attr($disabled) . '>';
                    foreach ($f['options'] as $ok=>$ov) {
                        $sel = ((string) $val === (string) $ok) ? ' selected' : '';
                        echo '<option value="' . esc_attr($ok) . '"' . esc_attr($sel) . '>' . esc_html($ov) . '</option>';
                    }
                    echo '</select>';
                } elseif ($f['t'] === 'weight') {
                    $levels = array('low'=>'Low','med'=>'Med','high'=>'High');
                    echo '<select id="' . esc_attr($k) . '" name="' . esc_attr($k) . '" ' . esc_attr($disabled) . '>';
                    foreach ($levels as $ok=>$ov) {
                        $sel = ((string) $val === (string) $ok) ? ' selected' : '';
                        echo '<option value="' . esc_attr($ok) . '"' . esc_attr($sel) . '>' . esc_html($ov) . '</option>';
                    }
                    echo '</select>';
                } else {
                    $type = ($f['t'] === 'number') ? 'number' : 'text';
                    $step_val = ($k === 'similarity_threshold') ? '0.01' : '1';
                    echo '<input type="' . esc_attr($type) . '" step="' . esc_attr($step_val) . '" id="' . esc_attr($k) . '" name="' . esc_attr($k) . '" value="' . esc_attr($val) . '" class="regular-text"' . esc_attr($disabled) . '>';
                }

                if (!empty($desc)) {
                    echo '<p class="description asg-desc">' . wp_kses_post($desc) . '</p>';
                }

                echo '</td></tr>';
            }
            echo '</tbody></table>';
            echo '</div>';
        }

        echo '<div class="asg-actions">';
        submit_button('Save settings', 'primary', 'submit', false);
        echo '</div>';

        echo '<div class="asg-card asg-section' . ($locked ? ' asg-pro-locked' : '') . '">';
            if ($locked) {
                echo '<div class="asg-pro-lockbar"><span class="asg-pro-pill">PRO</span> This block is available in <strong>AegisSpamGuard PRO</strong>. <a class="button button-primary asg-btn asg-btn--pro" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=license#asg_license_key')) . '">Upgrade to PRO</a></div>';
            }
        echo '<div class="asg-section__head">';
        echo '<span class="dashicons dashicons-visibility asg-section__icon"></span>';
        echo '<div><h3 class="asg-section__title">Transparency</h3>';
        echo '<p class="asg-section__desc">What is stored, what is hashed, and what leaves your server.</p></div></div>';
        echo '<div class="asg-pad">';
        echo '<p><strong>Email privacy:</strong> emails are hashed using a per-site salt. Raw emails are not stored in logs.</p>';
        echo '<p><strong>External calls:</strong> if <em>No external calls</em> is enabled, the plugin will not fetch external lists or contact any third-party service.</p>';
        echo '</div>';
        echo '</div>';

        echo '</form>';

        echo '<div class="asg-grid2">';
        echo '<div class="asg-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-download"></span> Export rules & settings</h3>';
        echo '<p>Exports allow/deny lists + weights + thresholds to JSON (safe keys only). Use this to migrate settings between sites.</p>';
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
        echo '<input type="hidden" name="action" value="asg_export_settings">';
        wp_nonce_field('asg_export_settings');
        submit_button('Export JSON', 'secondary', 'submit', false);
        echo '</form>';
        echo '</div>';

        echo '<div class="asg-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-upload"></span> Import rules & settings</h3>';
        echo '<p>Import JSON to restore lists, weights, and thresholds. Existing items are preserved; duplicates are skipped.</p>';
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" enctype="multipart/form-data">';
        echo '<input type="hidden" name="action" value="asg_import_settings">';
        wp_nonce_field('asg_import_settings');
        echo '<p><input type="file" name="asg_import_file" accept="application/json" /></p>';
        echo '<p><textarea name="asg_import_json" rows="6" class="large-text" placeholder="…or paste JSON here"></textarea></p>';
        submit_button('Import', 'primary', 'submit', false);
        echo '</form>';
        echo '</div>';
        echo '</div>';

        echo '<div class="asg-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-welcome-learn-more"></span> Local Learning</h3>';
        echo '<p>Use the <strong>Train Spam</strong> / <strong>Train Not Spam</strong> buttons in the <strong>Logs</strong> tab to teach the site-specific model. No external services required.</p>';
        echo '</div>';
    }

    public function export_settings() {
        if (!current_user_can('manage_options')) { wp_die('Access denied'); }
        check_admin_referer('asg_export_settings');
        $data = $this->build_export_payload();
        $json = wp_json_encode($data, JSON_PRETTY_PRINT);
        nocache_headers();
        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="aegisspamguard-export-' . gmdate('Ymd-His') . '.json"');
        echo $json; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- exporting JSON
        exit;
    }

    public function import_settings() {
        if (!current_user_can('manage_options')) { wp_die('Access denied'); }
        check_admin_referer('asg_import_settings');

        $raw = '';
        $raw = '';
        if ( ! empty( $_FILES['asg_import_file']['tmp_name'] ) ) {
            $tmp = sanitize_text_field( wp_unslash( $_FILES['asg_import_file']['tmp_name'] ) );
            if ( $tmp !== '' && is_uploaded_file( $tmp ) ) {
                $raw = (string) file_get_contents( $tmp );
            }
        }
        if ( $raw === '' && ! empty( $_POST['asg_import_json'] ) ) {
            $raw_post = sanitize_textarea_field( wp_unslash( $_POST['asg_import_json'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- validated by check_admin_referer().
            $raw      = trim( $raw_post );
        }
        $data = json_decode($raw, true);
        if (!is_array($data)) {
            wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=settings&import=0'));
            exit;
        }

        $opt = get_option('asg_settings', array());
        if (!is_array($opt)) { $opt = array(); }
		if (!empty($data['settings']) && is_array($data['settings'])) {
			foreach ($data['settings'] as $k=>$v) {

				if (!is_string($k)) { continue; }
				if (!preg_match('/^(threshold_|min_seconds_to_submit|velocity_|enable_|fp_hold_instead_of_block|trust_|ml_weight|w_|log_retention_days|ip_mode|no_external_calls|geo_provider|maxmind_)/', $k)) { continue; }

				if (strpos($k, 'enable_') === 0 || $k === 'fp_hold_instead_of_block' || $k === 'no_external_calls' || strpos($k, 'trust_') === 0) {
					$opt[$k] = !empty($v) ? 1 : 0;
				} elseif ($k === 'external_disposable_url') {
					$opt[$k] = esc_url_raw((string) $v);
				} elseif ($k === 'ip_mode') {
					$mode = sanitize_key((string) $v);
					if (!in_array($mode, array('store','anonymize','hash','off'), true)) { $mode = 'store'; }
					$opt[$k] = $mode;
				} elseif ($k === 'geo_provider') {
					$gp = sanitize_key((string) $v);
					if (!in_array($gp, array('headers','maxmind','none'), true)) { $gp = 'headers'; }
					$opt[$k] = $gp;
				} elseif ($k === 'maxmind_country_mmdb' || $k === 'maxmind_asn_mmdb') {
					$opt[$k] = sanitize_text_field((string) $v);
				} elseif (strpos($k, 'w_') === 0) {
					$w = sanitize_key((string) $v);
					if (!in_array($w, array('low','med','high'), true)) { $w = 'med'; }
					$opt[$k] = $w;
				} elseif ($k === 'ml_weight' || $k === 'similarity_threshold') {
					$opt[$k] = floatval((string) $v);
				} else {
					$opt[$k] = intval($v);
				}
			}

			update_option('asg_settings', $opt, false);
		}

        if (!empty($data['lists']) && is_array($data['lists'])) {
            global $wpdb;
        $lists = self::sanitize_db_prefix($wpdb->prefix) . 'asg_lists';
            foreach ($data['lists'] as $row) {
                if (!is_array($row)) { continue; }
                $list_type = isset($row['list_type']) ? sanitize_key($row['list_type']) : '';
                $kind = isset($row['kind']) ? sanitize_key($row['kind']) : '';
                $value = isset($row['value']) ? sanitize_text_field($row['value']) : '';
                $note = isset($row['note']) ? sanitize_text_field($row['note']) : '';
                if (!in_array($list_type, array('allow','deny'), true)) { continue; }
                if (!in_array($kind, array('ip','cidr','email','domain','fingerprint','country','asn'), true)) { continue; }
                if ($value === '') { continue; }
                $exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM %i WHERE list_type=%s AND kind=%s AND value=%s LIMIT 1", $lists, $list_type, $kind, $value ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                if ($exists) { continue; }
                $wpdb->insert($lists, array( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    'list_type'=>$list_type,
                    'kind'=>$kind,
                    'value'=>$value,
                    'note'=>$note,
                    'created_at'=>gmdate('Y-m-d H:i:s'),
                ), array('%s','%s','%s','%s','%s'));
            }
        }

        wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=settings&import=1'));
        exit;
    }

    private function build_export_payload() {
        $settings = get_option('asg_settings', array());
        if (!is_array($settings)) { $settings = array(); }

        $allow_keys = array(
            'threshold_allow','threshold_challenge','threshold_hold','threshold_block',
            'min_seconds_to_submit','velocity_window_seconds','velocity_limit',
            'enable_ml','ml_weight','enable_similarity','similarity_lookback_days','similarity_threshold',
            'enable_mx_checks','enable_external_disposable','external_disposable_url',
            'fp_hold_instead_of_block','trust_logged_in','trust_customers','trust_existing_user_days',
            'enable_cf7_captcha','log_retention_days',
            'ip_mode','no_external_calls','enable_geo_rules','enable_asn_rules','geo_provider','maxmind_country_mmdb','maxmind_asn_mmdb',
            'w_link_count','w_spam_keywords','w_link_markup','w_url_shortener','w_honeypot','w_too_fast','w_js_proof','w_invalid_email','w_disposable_email','w_mx_missing','w_name_email_mismatch'
        );
        $out_settings = array();
        foreach ($allow_keys as $k) {
            if (isset($settings[$k])) { $out_settings[$k] = $settings[$k]; }
        }
        global $wpdb;
        $lists = self::sanitize_db_prefix($wpdb->prefix) . 'asg_lists';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $rows = $wpdb->get_results($wpdb->prepare("SELECT list_type, kind, value, note FROM %i ORDER BY id DESC", $lists), ARRAY_A);
        return array(
            'exported_at' => gmdate('c'),
            'plugin' => 'AegisSpamGuard',
            'version' => ASG_VERSION,
            'settings' => $out_settings,
            'lists' => is_array($rows) ? $rows : array(),
        );
    }
    public function save_settings() {
        if (!current_user_can('manage_options')) { wp_die('Access denied'); }
        check_admin_referer('asg_save_settings');

	        $is_pro = $this->is_pro_active();
	        // PRO-only settings blocks (must not be modifiable without a valid license)
	        $pro_only_keys = array(
	            'enable_similarity','similarity_lookback_days','similarity_threshold',
	            'enable_cf7_captcha',
	            'enable_ml','ml_weight',
	            'w_link_count','w_spam_keywords','w_link_markup','w_url_shortener',
	            'w_honeypot','w_too_fast','w_js_proof','w_invalid_email','w_disposable_email',
	            'w_mx_missing','w_name_email_mismatch',
	        );

        $opt = get_option('asg_settings', array());
        if (!is_array($opt)) { $opt = array(); }

        $keys = array(
            'threshold_allow','threshold_challenge','threshold_hold',
            'min_seconds_to_submit','velocity_window_seconds','velocity_limit',
            'enable_ml','ml_weight','log_retention_days',
            'enable_similarity','similarity_lookback_days','similarity_threshold',
            'enable_mx_checks','enable_external_disposable','external_disposable_url',
            'ip_mode','no_external_calls',
            'enable_geo_rules','enable_asn_rules','geo_provider','maxmind_country_mmdb','maxmind_asn_mmdb',
            'fp_hold_instead_of_block',
            'trust_logged_in','trust_customers','trust_existing_user_days',
            'enable_cf7_captcha',
            'w_link_count','w_spam_keywords','w_link_markup','w_url_shortener',
            'w_honeypot','w_too_fast','w_js_proof','w_invalid_email','w_disposable_email',
            'w_mx_missing','w_name_email_mismatch'
        );

        $bool_keys = array(
            'enable_ml','enable_similarity','enable_mx_checks','enable_external_disposable',
            'no_external_calls','enable_geo_rules','enable_asn_rules',
            'fp_hold_instead_of_block','trust_logged_in','trust_customers','enable_cf7_captcha'
        );

	        foreach ($bool_keys as $bk) {
	            if (in_array($bk, $pro_only_keys, true) && !$is_pro) { continue; }
            $opt[$bk] = isset($_POST[$bk]) ? 1 : 0;
        }

        foreach ($keys as $k) {
            if (in_array($k, $bool_keys, true)) { continue; }
            if (in_array($k, $pro_only_keys, true) && !$is_pro) { continue; }
            if (!isset($_POST[$k])) { continue; }

            // WP expects: wp_unslash() first, then sanitize/validate
            $raw = wp_unslash($_POST[$k]); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- sanitized/validated per-key below.

            // Defensive: ignore unexpected array input
            if (is_array($raw)) { continue; }

            if ($k === 'external_disposable_url') {
                $opt[$k] = esc_url_raw((string) $raw);

            } elseif ($k === 'ip_mode') {
                $v = sanitize_key((string) $raw);
                if (!in_array($v, array('store','anonymize','hash','off'), true)) { $v = 'store'; }
                $opt[$k] = $v;

            } elseif ($k === 'geo_provider') {
                $v = sanitize_key((string) $raw);
                if (!in_array($v, array('headers','maxmind','none'), true)) { $v = 'headers'; }
                $opt[$k] = $v;

            } elseif ($k === 'maxmind_country_mmdb' || $k === 'maxmind_asn_mmdb') {
                $opt[$k] = sanitize_text_field((string) $raw);

            } elseif (strpos($k, 'w_') === 0) {
                $v = sanitize_key((string) $raw);
                if (!in_array($v, array('low','med','high'), true)) { $v = 'med'; }
                $opt[$k] = $v;

            } elseif ($k === 'similarity_threshold') {
                $opt[$k] = floatval((string) $raw);

            } else {
                $opt[$k] = intval($raw);
            }
        }


        update_option('asg_settings', $opt, false);

        wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=settings&updated=1'));
        exit;
    }

    public function save_network_settings() {
        if (!current_user_can('manage_network_options')) { wp_die('Access denied'); }
        check_admin_referer('asg_save_network_settings');

        $opt = get_site_option('asg_network_settings', array());
        if (!is_array($opt)) { $opt = array(); }

        $keys = array(
            'threshold_allow','threshold_challenge','threshold_hold',
            'min_seconds_to_submit','velocity_window_seconds','velocity_limit',
            'enable_ml','ml_weight','log_retention_days',
            'enable_similarity','similarity_lookback_days','similarity_threshold',
            'enable_mx_checks','enable_external_disposable',
            'ip_mode','no_external_calls',
            'enable_geo_rules','enable_asn_rules','geo_provider',
            'fp_hold_instead_of_block',
            'trust_logged_in','trust_customers','trust_existing_user_days',
            'enable_cf7_captcha'
        );

        $bool_keys = array(
            'enable_ml','enable_similarity','enable_mx_checks','enable_external_disposable',
            'no_external_calls','enable_geo_rules','enable_asn_rules',
            'fp_hold_instead_of_block','trust_logged_in','trust_customers','enable_cf7_captcha'
        );

        foreach ($bool_keys as $bk) {
            $opt[$bk] = isset($_POST[$bk]) ? 1 : 0;
        }

        foreach ($keys as $k) {
            if (in_array($k, $bool_keys, true)) { continue; }
            if (!isset($_POST[$k])) { continue; }

            if ($k === 'ip_mode') {
                $v = sanitize_key((string) $_POST[$k]);
                if (!in_array($v, array('store','anonymize','hash','off'), true)) { $v = 'store'; }
                $opt[$k] = $v;
            } elseif ($k === 'geo_provider') {
                $v = sanitize_key((string) $_POST[$k]);
                if (!in_array($v, array('headers','maxmind','none'), true)) { $v = 'headers'; }
                $opt[$k] = $v;
            } elseif ($k === 'similarity_threshold') {
                $opt[$k] = floatval((string) $_POST[$k]);
            } else {
                $opt[$k] = intval($_POST[$k]);
            }
        }

        update_site_option('asg_network_settings', $opt);

        wp_safe_redirect(network_admin_url('admin.php?page=aegis-spam-guard-network&updated=1'));
        exit;
    }

    public function add_list() {
        if (!current_user_can('manage_options')) { wp_die('Access denied'); }
        check_admin_referer('asg_add_list');

		$list_type = isset($_POST['list_type']) ? sanitize_key(wp_unslash($_POST['list_type'])) : '';
		$kind      = isset($_POST['kind']) ? sanitize_key(wp_unslash($_POST['kind'])) : '';
		$value     = isset($_POST['value']) ? sanitize_text_field(wp_unslash($_POST['value'])) : '';
		$note      = isset($_POST['note']) ? sanitize_text_field(wp_unslash($_POST['note'])) : '';

        if (!in_array($list_type, array('allow','deny'), true)) { $list_type = 'deny'; }
        if (!in_array($kind, array('ip','cidr','email','domain','fingerprint','country','asn'), true)) { $kind = 'ip'; }

        if ($kind === 'ip') { $value = trim($value); }
        if ($kind === 'email') { $value = strtolower(trim($value)); }
        if ($kind === 'domain') { $value = strtolower(trim($value)); }
        if ($kind === 'cidr') { $value = trim($value); }
        if ($kind === 'country') { $value = strtoupper(trim($value)); }
        if ($kind === 'asn') { $value = preg_replace('/[^0-9]/', '', (string)$value); }


        global $wpdb;
        $lists = self::sanitize_db_prefix($wpdb->prefix) . 'asg_lists';
        $wpdb->insert($lists, array( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            'list_type' => $list_type,
            'kind' => $kind,
            'value' => $value,
            'note' => $note,
            'created_at' => gmdate('Y-m-d H:i:s'),
        ), array('%s','%s','%s','%s','%s'));

        wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=lists&added=1'));
        exit;
    }

    public function delete_list() {
        if (!current_user_can('manage_options')) { wp_die('Access denied'); }
        $id = isset($_GET['id']) ? intval( wp_unslash( $_GET['id'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- nonce checked on action handler.
        check_admin_referer('asg_delete_list_' . $id);

        global $wpdb;
        $lists = self::sanitize_db_prefix($wpdb->prefix) . 'asg_lists';
        $wpdb->delete($lists, array('id'=>$id), array('%d')); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=lists&deleted=1'));
        exit;
    }

    public function event_action() {
        if (!current_user_can('manage_options')) { wp_die('Access denied'); }
        $event_id = isset($_GET['event_id']) ? intval( wp_unslash( $_GET['event_id'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- nonce checked on action handler.
        $do_raw = isset($_GET['do']) ? sanitize_key( wp_unslash( $_GET['do'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- nonce checked on action handler.
        $allowed_do = array('train_spam','train_ham','allow_fp','deny_fp','allow_email','deny_email','allow_ip','deny_ip');
        $do = in_array($do_raw, $allowed_do, true) ? $do_raw : '';
        check_admin_referer('asg_event_action_' . $event_id);
        if (!$do) {
            wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=logs&err=badop'));
            exit;
        }

        global $wpdb;
        $events = self::sanitize_db_prefix($wpdb->prefix) . 'asg_events';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $row = $wpdb->get_row($wpdb->prepare("SELECT * FROM %i WHERE id=%d", $events, $event_id), ARRAY_A);
        if (!$row) {
            wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=logs&err=notfound'));
            exit;
        }

        $payload = array();
        if (!empty($row['payload'])) {
            $payload = json_decode($row['payload'], true);
            if (!is_array($payload)) { $payload = array(); }
        }
        $text = '';
        if (!empty($payload['content'])) { $text .= ' ' . (string) $payload['content']; }
        if (!empty($payload['url'])) { $text .= ' ' . (string) $payload['url']; }

        if ($do === 'train_spam') {
            $this->ml->train_text($text, true);
        } elseif ($do === 'train_ham') {
            $this->ml->train_text($text, false);
        } elseif ($do === 'allow_fp') {
            $this->add_fp_to_list('allow', $row['fingerprint'], 'Admin allow fingerprint');
        } elseif ($do === 'deny_fp') {
            $this->add_fp_to_list('deny', $row['fingerprint'], 'Admin deny fingerprint');
        }

        wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=logs&done=1'));
        exit;
    }

    public function bulk_event_action() {
        if (!current_user_can('manage_options')) { wp_die('Access denied'); }
        check_admin_referer('asg_bulk_event_action');

        $do_raw = isset($_POST['do']) ? sanitize_key(wp_unslash($_POST['do'])) : '';
        $allowed_do = array('train_spam','train_ham','allow_fp','deny_fp','allow_email','deny_email','allow_ip','deny_ip');
        $do = in_array($do_raw, $allowed_do, true) ? $do_raw : '';

        $ids = array();
        if (isset($_POST['ids']) && is_array($_POST['ids'])) {
            $ids = array_map('intval', (array) wp_unslash($_POST['ids']));
        }
        $ids = array_values(array_filter($ids, function($v){ return $v > 0; }));
        if (!$do || empty($ids)) {
            wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=logs&err=bulk'));
            exit;
        }

        global $wpdb;
        $events = self::sanitize_db_prefix($wpdb->prefix) . 'asg_events';
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $rows = $wpdb->get_results($wpdb->prepare("SELECT * FROM %i WHERE id IN ($placeholders)", array_merge(array($events), $ids)), ARRAY_A); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        foreach ((array)$rows as $row) {
            $payload = array();
            if (!empty($row['payload'])) {
                $payload = json_decode((string)$row['payload'], true);
                if (!is_array($payload)) { $payload = array(); }
            }
            $text = '';
            if (!empty($payload['content'])) { $text .= ' ' . (string) $payload['content']; }
            if (!empty($payload['url'])) { $text .= ' ' . (string) $payload['url']; }

            if ($do === 'train_spam') {
                $this->ml->train_text($text, true);
            } elseif ($do === 'train_ham') {
                $this->ml->train_text($text, false);
            } elseif ($do === 'allow_fp') {
                if (!empty($row['fingerprint'])) { $this->add_fp_to_list('allow', (string)$row['fingerprint'], 'Admin bulk allow fingerprint'); }
            } elseif ($do === 'deny_fp') {
                if (!empty($row['fingerprint'])) { $this->add_fp_to_list('deny', (string)$row['fingerprint'], 'Admin bulk deny fingerprint'); }
            }
        }

        wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=logs&done=1'));
        exit;
    }


    private function add_fp_to_list($list_type, $fingerprint, $note) {
        global $wpdb;
        $lists = self::sanitize_db_prefix($wpdb->prefix) . 'asg_lists';
        $wpdb->insert($lists, array( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            'list_type' => $list_type,
            'kind' => 'fingerprint',
            'value' => sanitize_text_field($fingerprint),
            'note' => sanitize_text_field($note),
            'created_at' => gmdate('Y-m-d H:i:s'),
        ), array('%s','%s','%s','%s','%s'));
    }

private function render_cleanup() {
    if (!current_user_can('manage_options')) { return; }

    $type = isset($_GET['cleanup_type']) ? sanitize_key( wp_unslash( $_GET['cleanup_type'] ) ) : 'comments'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- UI filtering only.
    $limit = isset($_GET['limit']) ? max( 10, min( 200, intval( wp_unslash( $_GET['limit'] ) ) ) ) : 50; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- UI filtering only.
    $offset = isset($_GET['offset']) ? max( 0, intval( wp_unslash( $_GET['offset'] ) ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- UI filtering only.
    $do = isset($_GET['do']) ? sanitize_key( wp_unslash( $_GET['do'] ) ) : 'scan'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- UI filtering only.
    $apply = isset($_GET['apply']) ? sanitize_key( wp_unslash( $_GET['apply'] ) ) : 'mark_spam'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- UI filtering only.

    echo '<div class="asg-card">';
    echo '<h2>Bulk Cleanup</h2>';
    echo '<p class="description">Scan existing comments and users with your current AegisSpamGuard engine, then apply actions in clean batches (no timeouts). This helps you remove legacy spam that Akismet/CleanTalk missed.</p>';

    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
    wp_nonce_field('asg_cleanup_scan');
    echo '<input type="hidden" name="action" value="asg_cleanup_scan" />';
    echo '<div class="asg-grid asg-grid--2">';
    echo '<div class="asg-field"><label><strong>Target</strong></label><select name="cleanup_type">';
    echo '<option value="comments"' . selected($type,'comments',false) . '>Comments</option>';
    echo '<option value="users"' . selected($type,'users',false) . '>Users</option>';
    echo '</select><div class="asg-help">Choose what to scan. Comments scan will re-score comment content + author identity. Users scan will re-score registrations (email, name mismatch, disposable domains, etc.).</div></div>';

    echo '<div class="asg-field"><label><strong>Batch size</strong></label><input type="number" min="10" max="200" name="limit" value="' . esc_attr($limit) . '" />';
    echo '<div class="asg-help">Default 50. Use smaller batches on shared hosting.</div></div>';

    echo '<div class="asg-field"><label><strong>Offset</strong></label><input type="number" min="0" name="offset" value="' . esc_attr($offset) . '" />';
    echo '<div class="asg-help">Used for paging through large databases.</div></div>';

    echo '<div class="asg-field"><label><strong>When flagged as spam</strong></label><select name="apply">';
    echo '<option value="mark_spam"' . selected($apply,'mark_spam',false) . '>Move to Spam / Mark as spam</option>';
    echo '<option value="delete"' . selected($apply,'delete',false) . '>Delete</option>';
    echo '<option value="report_only"' . selected($apply,'report_only',false) . '>Report only (no changes)</option>';
    echo '</select><div class="asg-help">Safe default is “Move to Spam”. Use “Report only” first to verify accuracy.</div></div>';
    echo '</div>';

    echo '<p><button class="button button-primary">Run Batch</button></p>';
    echo '</form>';

    if ( ! empty( $_GET['ran'] ) && ! empty( $_GET['done'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only notice values.
        $scanned = isset( $_GET['scanned'] ) ? absint( wp_unslash( $_GET['scanned'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only notice values.
        $flagged = isset( $_GET['flagged'] ) ? absint( wp_unslash( $_GET['flagged'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only notice values.
        $changed = isset( $_GET['changed'] ) ? absint( wp_unslash( $_GET['changed'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only notice values.
        echo '<div class="notice notice-success"><p><strong>Batch complete.</strong> Scanned: ' . esc_html( $scanned ) . ' • Flagged: ' . esc_html( $flagged ) . ' • Changed: ' . esc_html( $changed ) . '</p></div>';
    }

    echo '<hr />';
    echo '<h3>How it works</h3>';
    echo '<ul class="ul-disc">';
    echo '<li><strong>Uses your current rules</strong> (lists, weights, thresholds, ML) — no cloud dependency.</li>';
    echo '<li><strong>Conservative defaults</strong>: “Move to Spam” is reversible for comments.</li>';
    echo '<li><strong>Batch processing</strong>: safe on shared hosting.</li>';
    echo '</ul>';
    echo '</div>';
}

public function cleanup_scan() {
    if (!current_user_can('manage_options')) { wp_die('Access denied'); }
    check_admin_referer('asg_cleanup_scan');

    $type = isset($_POST['cleanup_type']) ? sanitize_key( wp_unslash( $_POST['cleanup_type'] ) ) : 'comments';
    $limit = isset($_POST['limit']) ? max(10, min(200, intval($_POST['limit']))) : 50;
    $offset = isset($_POST['offset']) ? max(0, intval($_POST['offset'])) : 0;
    $apply = isset($_POST['apply']) ? sanitize_key($_POST['apply']) : 'mark_spam';

    $scanned = 0; $flagged = 0; $changed = 0;

    if ($type === 'comments') {
        $q = new WP_Comment_Query();
        $comments = $q->query(array(
            'number' => $limit,
            'offset' => $offset,
            'status' => 'all',
            'orderby' => 'comment_ID',
            'order' => 'ASC',
        ));
        if (is_array($comments)) {
            foreach ($comments as $c) {
                if (!is_object($c)) { continue; }
                $scanned++;
                $event = array(
                    'type' => 'comment',
                    'actor' => array(
                        'ip' => (string) $c->comment_author_IP,
                        'user_id' => intval($c->user_id),
                        'session_id' => '',
                    ),
                    'payload' => array(
                        'fields' => array(
                            'author' => $c->comment_author,
                            'email'  => $c->comment_author_email,
                            'url'    => $c->comment_author_url,
                        ),
                        'message' => (string) $c->comment_content,
                        'metadata' => array('adapter' => 'cleanup_scan', 'no_log' => 1),
                    ),
                );
                $res = $this->engine->evaluate($event);
                $action = isset($res['action']) ? (string)$res['action'] : 'allow';
                if ($action === 'block' || $action === 'hold') {
                    $flagged++;
                    if ($apply === 'mark_spam') {
                        if (wp_spam_comment($c->comment_ID)) { $changed++; }
                    } elseif ($apply === 'delete') {
                        if (wp_delete_comment($c->comment_ID, true)) { $changed++; }
                    }
                }
            }
        }
    } elseif ($type === 'users') {
        $users = get_users(array(
            'number' => $limit,
            'offset' => $offset,
            'orderby' => 'ID',
            'order' => 'ASC',
            'fields' => array('ID','user_email','display_name','user_login'),
        ));
        if (is_array($users)) {
            foreach ($users as $u) {
                if (!is_object($u)) { continue; }
                $scanned++;
                $event = array(
                    'type' => 'registration',
                    'actor' => array(
                        'ip' => '',
                        'user_id' => intval($u->ID),
                        'session_id' => '',
                    ),
                    'payload' => array(
                        'fields' => array(
                            'user_login' => $u->user_login,
                            'display_name' => $u->display_name,
                            'user_email' => $u->user_email,
                        ),
                        'message' => (string) $u->user_login . ' ' . (string) $u->display_name,
                        'metadata' => array('adapter' => 'cleanup_scan', 'no_log' => 1),
                    ),
                );
                $res = $this->engine->evaluate($event);
                $action = isset($res['action']) ? (string)$res['action'] : 'allow';
                if ($action === 'block' || $action === 'hold') {
                    $flagged++;
                    if ($apply === 'delete') {
                        require_once ABSPATH . 'wp-admin/includes/user.php';
                        if (wp_delete_user($u->ID)) { $changed++; }
                    }
                }
            }
        }
    }

    update_option('asg_cleanup_last', array(
        'ts' => time(),
        'type' => $type,
        'limit' => $limit,
        'offset' => $offset,
        'apply' => $apply,
        'scanned' => $scanned,
        'flagged' => $flagged,
        'changed' => $changed,
    ), false);

    $redirect = add_query_arg(array(
        'page' => 'aegis-spam-guard',
        'tab' => 'cleanup',
        'cleanup_type' => $type,
        'limit' => $limit,
        'offset' => $offset + $limit,
        'apply' => $apply,
        'ran' => 1,
        'done' => 1,
        'scanned' => $scanned,
        'flagged' => $flagged,
        'changed' => $changed,
    ), admin_url('admin.php'));
    wp_safe_redirect($redirect);
    exit;
}

    private function render_firewall() {
        $s = $this->engine->get_effective_settings();
        echo '<div class="asg-card">';
        echo '<div class="asg-card__head"><h2>Firewall Stream</h2><p class="description">Early-stage basic filtering that blocks or challenges threats before WordPress loads, ensuring baseline firewall protection.</p></div>';

        $action = isset($_GET['fw_action']) ? sanitize_text_field( wp_unslash( (string) $_GET['fw_action'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- UI filtering only.
        $q = isset($_GET['q']) ? sanitize_text_field( wp_unslash( (string) $_GET['q'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- UI filtering only.

        echo '<form method="get" class="asg-toolbar">';
        echo '<input type="hidden" name="page" value="aegis-spam-guard" />';
        echo '<input type="hidden" name="tab" value="firewall" />';
        echo '<select name="fw_action">';
        $opts = array('' => 'All actions', 'block' => 'Blocked', 'challenge' => 'Challenged', 'allow' => 'Allowed');
        foreach ($opts as $k=>$v) {
            echo '<option value="' . esc_attr($k) . '" ' . selected((string) $action, (string) $k, false) . '>' . esc_html($v) . '</option>';
        }
        echo '</select>';
        echo '<input type="search" name="q" value="' . esc_attr($q) . '" placeholder="Search URI, fingerprint, ip_hash" />';
        echo '<button class="button">Filter</button>';
        echo '</form>';

        $rows = $this->engine->get_firewall_events(array('limit' => 200, 'action' => $action, 'q' => $q));

        if (empty($rows)) {
            echo '<p>No firewall events yet.</p>';
            echo '</div>';
            return;
        }

        echo '<div class="asg-fw-list">';
        foreach ($rows as $r) {
            $a = isset($r['action']) ? (string) $r['action'] : '';
            $ts = isset($r['created_at']) ? (string) $r['created_at'] : '';
            $uri = isset($r['uri']) ? (string) $r['uri'] : '';
            $fp = isset($r['fingerprint']) ? (string) $r['fingerprint'] : '';
            $ip_hash = isset($r['ip_hash']) ? (string) $r['ip_hash'] : '';

            $badge = ($a === 'block') ? 'asg-badge asg-badge--danger' : (($a === 'challenge') ? 'asg-badge asg-badge--warn' : 'asg-badge');
            echo '<div class="asg-row asg-row--fw">';
            echo '<div class="asg-row__summary">';
            echo '<span class="' . esc_attr($badge) . '">' . esc_html(strtoupper($a)) . '</span> ';
            echo '<strong>' . esc_html($uri) . '</strong>';
            echo '<span class="asg-muted"> &middot; ' . esc_html($ts) . '</span>';
            echo '</div>';
            echo '<div class="asg-row__meta">';
            if ($fp !== '') { echo '<span class="asg-chip">FP: ' . esc_html(substr($fp,0,10)) . '…</span> '; }
            if ($ip_hash !== '') { echo '<span class="asg-chip">IP hash: ' . esc_html(substr($ip_hash,0,10)) . '…</span>'; }
            echo '</div>';
            echo '</div>';
        }
        echo '</div>';
        echo '</div>';
    }

	function is_pro_active() : bool {

			if (function_exists('aegisify_is_pro') && aegisify_is_pro(true)) {
				return true;
			}

			$key    = (string) get_option('asg_license_key', '');
			$email  = (string) get_option('asg_license_email', '');
			$status = (string) get_option('asg_license_status', '');
			$plan   = (string) get_option('asg_license_plan', '');
			if (trim($key) === '' || trim($email) === '') { return false; }
			if ($status === 'email_mismatch') { return false; }
			if ($status !== 'active') { return false; }
			if (strtolower($plan) !== 'pro') { return false; }
			return true;
		}

    private function render_pro_upsell_tab($tab_label) {
        $tab_label = (string) $tab_label;

        echo '<h2 class="asg-h2">' . esc_html($tab_label) . ' <span class="asg-pro-pill">PRO</span></h2>';
        echo '<div class="asg-card asg-pro-hero">';
        echo '  <h3 class="asg-card__title">This area is available in <strong>AegisSpamGuard PRO</strong></h3>';
        echo '  <p class="description">Unlock advanced firewall pre-checks, allow/deny automation, cleanup tools, and premium controls designed for real WordPress admins.</p>';
        echo '  <div class="asg-pro-actions">';
        echo '    <a class="button button-primary asg-btn asg-btn--pro" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=license#asg_license_key')) . '">Upgrade to PRO</a> ';
        echo '    <a class="button button-secondary asg-btn" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=license#asg-matrix')) . '">View Free vs PRO Matrix</a>';
        echo '  </div>';
        echo '  <hr />';
        echo '  <div class="asg-pro-note"><strong>Tip:</strong> You can keep running the Free version (engine + logs + adapters). PRO adds the locked tabs and locked settings blocks.</div>';
        echo '</div>';
    }

    private function render_license() {
        echo '<h2 class="asg-h2">License</h2>';

        $key    = (string) get_option('asg_license_key', '');
        $email  = (string) get_option('asg_license_email', '');
        $status = (string) get_option('asg_license_status', '');
        $plan   = (string) get_option('asg_license_plan', '');
        $expires = (string) get_option('asg_license_expires', '');
        $reg_email = (string) get_option('asg_license_registered_email', '');
		$suite_pro = (function_exists('aegisify_is_pro') && aegisify_is_pro(true));
        $key_mask = ($key !== '') ? (substr($key, 0, 4) . str_repeat('•', max(0, strlen($key) - 8)) . substr($key, -4)) : '';

		if (isset($_GET['asg_notice'])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only notice display.
			$n = sanitize_text_field(wp_unslash($_GET['asg_notice'])); // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only notice display.
			if ($n === 'activated') { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only notice display.
				echo '<div class="notice notice-success"><p><strong>License activated.</strong> PRO features are now unlocked for this site.</p></div>';
			} elseif ($n === 'email_mismatch') {
                echo '<div class="notice notice-error"><p><strong>Email mismatch.</strong> The email entered does not match the email on record for this license key. PRO remains disabled.</p></div>';
            } elseif ($n === 'invalid') {
                echo '<div class="notice notice-error"><p><strong>Activation failed.</strong> Please verify the email and key and try again.</p></div>';
            } elseif ($n === 'missing') {
                echo '<div class="notice notice-warning"><p><strong>Missing information.</strong> Please enter both the License Email and License Key.</p></div>';
            }
        }
		if ($suite_pro) {
            echo '<div class="notice notice-success"><p><strong>Aegisify Suite active.</strong> PRO features are enabled for AegisSpam via your Aegisify license.</p></div>';
        }
		echo '<div class="asg-card">';
		$free_optin  = (int) get_option('asg_free_reg_optin', 0);
		$free_status = (string) get_option('asg_free_reg_status', 'unregistered');

		echo '<div class="asg-card">';
		echo '<h3 class="asg-card__title"><span class="dashicons dashicons-privacy"></span> ' . esc_html__('Optional: Register this free installation', 'aegisspam') . '</h3>';

		echo '<p class="description" style="margin:0 0 10px 0;">';
		echo esc_html__(
			'This is optional and if you want license-related features through Aegisify, you can register this free installation. This will send your site domain, site URL, and the WordPress admin email to Aegisify. Your information will never leave our database nor be sold to any 3rd party.',
			'aegisspam'
		);
		echo '</p>';

		echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" class="asg-form">';
		echo '<input type="hidden" name="action" value="asg_register_free_install">';
		wp_nonce_field('asg_register_free_install');

		echo '<label style="display:block; margin:0 0 10px 0;">';
		echo '<input type="checkbox" name="asg_free_reg_consent" value="1" ' . ($free_optin === 1 ? 'checked="checked"' : '') . ' />';
		echo ' ' . esc_html__('I consent to send the data described above to Aegisify.', 'aegisspam');
		echo '</label>';

		echo '<p style="margin:0 0 10px 0;"><strong>' . esc_html__('Registration Status:', 'aegisspam') . '</strong> ';
		if ($free_status === 'registered') {
			echo '<span style="color:#0a7b34; font-weight:600;">' . esc_html__('Registered', 'aegisspam') . '</span>';
		} elseif ($free_status === 'error') {
			echo '<span style="color:#b32d2e; font-weight:600;">' . esc_html__('Error', 'aegisspam') . '</span>';
		} else {
			echo '<span style="color:#555; font-weight:600;">' . esc_html__('Unregistered', 'aegisspam') . '</span>';
		}
		echo '</p>';

		if ($free_status === 'registered') {
			echo '<button type="submit" class="button" name="asg_free_install_do" value="unregister">';
			echo esc_html__('Unregister Free Installation', 'aegisspam');
			echo '</button>';
		} else {
			echo '<button type="submit" class="button" name="asg_free_install_do" value="register">';
			echo esc_html__('Register Free Installation', 'aegisspam');
			echo '</button>';
		}

		echo '</form>';
		echo '</div>';

        echo '<div class="asg-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-admin-network"></span> License Activation</h3>';
        echo '<p class="description">Enter the <strong>same email</strong> used when the license was issued. The email must match the license key in CLM or PRO will not activate.</p>';

        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" class="asg-form asg-license-form">';
        echo '<input type="hidden" name="action" value="asg_save_license">';
        wp_nonce_field('asg_save_license');

        echo '<table class="form-table asg-table"><tbody>';

        echo '<tr><th scope="row"><label for="asg_license_email">License Email</label></th><td>';
        echo '<input type="email" id="asg_license_email" name="asg_license_email" value="' . esc_attr($email) . '" class="regular-text" placeholder="you@example.com" autocomplete="email" />';
        echo '<p class="description">This must match the email on record in your license manager for this key.</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row"><label for="asg_license_key">License Key</label></th><td>';
        echo '<input type="text" id="asg_license_key" name="asg_license_key" value="' . esc_attr($key) . '" class="regular-text" autocomplete="off" spellcheck="false" />';
        if ($key_mask !== '') {
            echo '<p class="description">Currently saved: <code>' . esc_html($key_mask) . '</code></p>';
        }
        echo '</td></tr>';

        echo '</tbody></table>';

		echo '<p style="margin:0;">';

		echo '<button type="submit" class="button button-primary" name="asg_license_action" value="activate">';
		echo esc_html__('Activate License', 'aegisspam');
		echo '</button> ';

		if (isset($status) && $status === 'active' && !$suite_pro) {
			$onclick = 'return confirm(' . wp_json_encode( __( 'Deactivate this license on this site?', 'aegisspam' ) ) . ');';
			echo '<button type="submit" class="button" name="asg_license_action" value="deactivate" onclick="' . esc_attr($onclick) . '">';
			echo esc_html__('Deactivate', 'aegisspam');
			echo '</button> ';
		}

		echo '<button type="submit" class="button button-secondary" name="asg_license_action" value="check">';
		echo esc_html__('Check Status', 'aegisspam');
		echo '</button>';

		echo '</p>';
		echo '<div style="margin:12px 0 0 0;">';
		echo '<img src="' . esc_url(ASG_PLUGIN_URL . 'assets/need.png') . '" alt="" style="max-width:100%; height:auto; display:block;" />';
		echo '</div>';
        echo '</form>';

        echo '<hr />';
        echo '<h4 style="margin:0 0 8px 0;">Current Status</h4>';
        echo '<table class="widefat striped asg-table"><tbody>';
        echo '<tr><th style="width:220px;">Status</th><td>' . esc_html($status !== '' ? $status : 'not_activated') . '</td></tr>';
        echo '<tr><th>Plan</th><td>' . esc_html($plan !== '' ? $plan : '—') . '</td></tr>';
        echo '<tr><th>Registered Email (CLM)</th><td>' . esc_html($reg_email !== '' ? $reg_email : '—') . '</td></tr>';
        echo '<tr><th>Expires</th><td>' . esc_html($expires !== '' ? $expires : '—') . '</td></tr>';
        echo '</tbody></table>';

        echo '</div>';

        echo '<div class="asg-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-yes"></span> Free vs PRO Matrix</h3>';
        echo '<p class="description">Everything in Free is local-first and transparent. PRO unlocks advanced controls and premium tabs.</p>';

        echo '<table class="widefat striped asg-matrix"><thead><tr>';
        echo '<th>Capability</th><th style="width:140px">Free</th><th style="width:140px">PRO</th>';
        echo '</tr></thead><tbody>';

        $rows = array(
            array('Engine + scoring framework', '✅', '✅'),
            array('Spam Log (events table + actions)', '✅', '✅'),
            array('WordPress Comments protection', '✅', '✅'),
            array('CF7 / Woo / REST adapters (where available)', '✅', '✅'),
            array('Allow/Deny tab (managed UI + automation)', '—', '✅'),
            array('Firewall tab (site-wide pre-checks)', '—', '✅'),
            array('Cleanup tab (bulk tools + hygiene utilities)', '—', '✅'),
            array('Content Intelligence settings block', '—', '✅'),
            array('Challenges settings block', '—', '✅'),
            array('Local Learning settings block', '—', '✅'),
            array('Signal Weights settings block', '—', '✅'),
            array('PRO update eligibility', '—', '✅'),
        );

        foreach ($rows as $r) {
            echo '<tr><td>' . esc_html($r[0]) . '</td><td class="asg-matrix__cell">' . esc_html($r[1]) . '</td><td class="asg-matrix__cell">' . esc_html($r[2]) . '</td></tr>';
        }

        echo '</tbody></table>';

        echo '<div class="asg-matrix-footer">';
        echo '<a class="button button-primary asg-btn asg-btn--pro" href="' . esc_url(admin_url('admin.php?page=aegis-spam-guard&tab=license#asg_license_key')) . '">Upgrade to PRO</a>';
        echo '</div>';


        // GDPR / Privacy explainer (expandable)
        echo '<div class="asg-card">';
        echo '<h3 class="asg-card__title"><span class="dashicons dashicons-privacy"></span> GDPR & Privacy</h3>';
        echo '<p class="description">AegisSpamGuard is designed to be privacy-forward: it works locally and minimizes what gets stored.</p>';

        echo '<details class="asg-details">';
        echo '<summary><strong>What data is stored?</strong></summary>';
        echo '<div class="asg-details__body">';
        echo '<ul class="asg-ul">';
        echo '<li><strong>Event metadata:</strong> action, score, reasons, time, and the request context used for scoring.</li>';
        echo '<li><strong>Hashed identifiers:</strong> email and IP are stored as hashes for matching repeat abuse without exposing raw values.</li>';
        echo '<li><strong>Optional raw IP mode:</strong> if enabled in settings, IP may be stored to support allow/deny workflows.</li>';
        echo '</ul>';
        echo '</div>';
        echo '</details>';

        echo '<details class="asg-details">';
        echo '<summary><strong>Do you send data to third parties?</strong></summary>';
        echo '<div class="asg-details__body">';
        echo '<p>No. The Free engine runs entirely on your site. AegisSpamGuard does not send visitor submissions to external anti-spam services.</p>';
        echo '<p><em>Exception:</em> license checks and update eligibility for PRO use the Aegisify licensing server (CLM) and include only what\'s required (site URL + license key).</p>';
        echo '</div>';
        echo '</details>';

        echo '<details class="asg-details">';
        echo '<summary><strong>Retention & deletion</strong></summary>';
        echo '<div class="asg-details__body">';
        echo '<p>You control retention by clearing the Spam Log tables and by running Cleanup. If you need stricter retention policies, you can truncate tables or schedule deletion with WP-Cron.</p>';
        echo '</div>';
        echo '</details>';

        echo '</div>';

        echo '</div>';
    }

    public function save_license() {
        if (!current_user_can('manage_options')) { wp_die(esc_html__('Access denied.', 'aegisspam')); }
        check_admin_referer('asg_save_license');
		$action = isset($_POST['asg_license_action']) ? sanitize_text_field(wp_unslash($_POST['asg_license_action'])) : 'activate';
		if (!in_array($action, array('activate', 'deactivate', 'check'), true)) {
			$action = 'activate';
		}
        $key   = isset($_POST['asg_license_key']) ? sanitize_text_field(wp_unslash($_POST['asg_license_key'])) : '';
        $email = isset($_POST['asg_license_email']) ? sanitize_email(wp_unslash($_POST['asg_license_email'])) : '';

        update_option('asg_license_key', $key);
        update_option('asg_license_email', $email);

        if (trim($key) === '' || trim($email) === '') {
            wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=missing'));
            exit;
        }

		if ($action === 'deactivate') {

			if (class_exists('ASG_Core') && method_exists('ASG_Core', 'clm_deactivate_license')) {
				ASG_Core::clm_deactivate_license($key, $email);
			}

			update_option('asg_license_status', 'inactive');
			update_option('asg_license_plan', '');
			update_option('asg_license_expires', '');
			update_option('asg_license_registered_email', '');

			wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=deactivated'));
			exit;
		}

		if ($action === 'check') {

			if (trim($key) === '' || trim($email) === '') {
				wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=missing'));
				exit;
			}

			$data = (class_exists('ASG_Core') && method_exists('ASG_Core', 'clm_status_license'))
				? ASG_Core::clm_status_license($key, $email)
				: array();

			if (!is_array($data) || empty($data)) {
				update_option('asg_license_status', 'inactive');
				update_option('asg_license_plan', '');
				update_option('asg_license_expires', '');
				update_option('asg_license_registered_email', '');
				wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=status_failed'));
				exit;
			}

			$clm_email = method_exists('ASG_Core', 'clm_extract_email') ? ASG_Core::clm_extract_email($data) : '';
			$meta      = method_exists('ASG_Core', 'clm_extract_meta')  ? ASG_Core::clm_extract_meta($data)  : array();

			$status  = (string) ($meta['status'] ?? 'inactive');
			$plan    = (string) ($meta['plan'] ?? '');
			$expires = (string) ($meta['expires'] ?? ($meta['expires_at'] ?? ''));

			update_option('asg_license_status', $status);
			update_option('asg_license_plan', $plan);
			update_option('asg_license_expires', $expires);
			update_option('asg_license_registered_email', $clm_email !== '' ? $clm_email : $email);

			wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=status_ok'));
			exit;
		}

        if (class_exists('ASG_Core')) {
            $data = ASG_Core::clm_activate_license($key, $email);
        } else {
            $data = array();
        }

        if (!is_array($data) || empty($data)) {
            update_option('asg_license_status', 'inactive');
            update_option('asg_license_plan', '');
            update_option('asg_license_expires', '');
            update_option('asg_license_registered_email', '');
            wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=invalid'));
            exit;
        }

        $clm_email = class_exists('ASG_Core') ? ASG_Core::clm_extract_email($data) : '';
        $meta      = class_exists('ASG_Core') ? ASG_Core::clm_extract_meta($data) : array('status'=>'','plan'=>'','expires'=>'');

        if ($clm_email !== '' && strtolower(trim($clm_email)) !== strtolower(trim($email))) {
            update_option('asg_license_status', 'email_mismatch');
            update_option('asg_license_plan', '');
            update_option('asg_license_expires', $meta['expires'] ?? '');
            update_option('asg_license_registered_email', $clm_email);

            wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=email_mismatch'));
            exit;
        }

        $status = (string) ($meta['status'] ?? '');
        $plan   = (string) ($meta['plan'] ?? '');
        $expires = (string) ($meta['expires'] ?? ($meta['expires_at'] ?? ''));

        update_option('asg_license_status', $status !== '' ? $status : 'inactive');
        update_option('asg_license_plan', $plan);
        update_option('asg_license_expires', $expires);
        update_option('asg_license_registered_email', $clm_email !== '' ? $clm_email : $email);

		update_option('asg_free_reg_optin', 0);
		update_option('asg_free_reg_status', 'unregistered');
		delete_transient('asg_clm_free_reg_sent');

        wp_safe_redirect(admin_url('admin.php?page=aegis-spam-guard&tab=license&asg_notice=activated'));
        exit;
    }
    

/**
 * Sanitize DB prefix once (identifier-safe).
 */
private static function sanitize_db_prefix($prefix) {
    $prefix = (string) $prefix;
    return preg_replace('/[^A-Za-z0-9_]/', '', $prefix);
}

/**
 * Strict allowlist for identifiers and backtick-wrap.
 */
private static function escape_sql_identifier($identifier) {
    $identifier = (string) $identifier;
    if ($identifier === '' || !preg_match('/^[A-Za-z0-9_]+$/', $identifier)) {
        return '';
    }
    return '`' . $identifier . '`';
}

/**
 * Replace {{table}} token with an escaped identifier.
 */
private static function sql_with_table($sql, $table) {
    $table = self::escape_sql_identifier($table);
    if ($table === '') {
        return $sql;
    }
    return str_replace('{{table}}', $table, (string) $sql);
}

}
