<?php
if (!defined('ABSPATH')) { exit; }
final class ASG_Engine {

/**
 * Sanitize DB prefix once (identifier-safe).
 */
private static function sanitize_db_prefix($prefix) {
    $prefix = (string) $prefix;
    return preg_replace('/[^A-Za-z0-9_]/', '', $prefix);
}

/**
 * Strict allowlist for identifiers and backtick-wrap.
 */
private static function escape_sql_identifier($identifier) {
    $identifier = (string) $identifier;
    if ($identifier === '' || !preg_match('/^[A-Za-z0-9_]+$/', $identifier)) {
        return '';
    }
    return '`' . $identifier . '`';
}

/**
 * Replace {{table}} token with an escaped identifier.
 */
private static function sql_with_table($sql, $table) {
    $table = self::escape_sql_identifier($table);
    if ($table === '') {
        return $sql;
    }
    return str_replace('{{table}}', $table, (string) $sql);
}

    private $ml;
    private $settings;
    public function __construct($ml) {
        $this->ml = $ml;
        $this->settings = $this->get_effective_settings();
        add_action('asg_daily_cleanup', array($this, 'cleanup'));
        if (!wp_next_scheduled('asg_daily_cleanup')) {
            wp_schedule_event(time() + HOUR_IN_SECONDS, 'daily', 'asg_daily_cleanup');
        }
    }

    public function normalize_event($event) {
        if (!is_array($event)) { $event = array(); }
        if (isset($event['payload']) && is_array($event['payload'])) {
            $p = $event['payload'];
            $actor = isset($event['actor']) && is_array($event['actor']) ? $event['actor'] : array();
            $legacy = array(
                'type' => isset($event['type']) ? sanitize_key($event['type']) : 'unknown',
                'name' => isset($p['name']) ? (string) $p['name'] : '',
                'email' => isset($p['email']) ? (string) $p['email'] : '',
                'content' => isset($p['message']) ? (string) $p['message'] : (isset($p['content']) ? (string) $p['content'] : ''),
                'url' => isset($p['url']) ? (string) $p['url'] : '',
                'meta' => array(),
            );

            if (isset($p['urls']) && is_array($p['urls']) && !empty($p['urls'])) {
                $legacy['meta']['urls'] = array_slice(array_values(array_map('strval', $p['urls'])), 0, 20);
            }
            if (isset($p['fields']) && is_array($p['fields'])) {
                $legacy['meta']['fields'] = $p['fields'];
            }
            if (isset($p['metadata']) && is_array($p['metadata'])) {
                $legacy['meta']['metadata'] = $p['metadata'];
            } elseif (isset($p['meta']) && is_array($p['meta'])) {
                $legacy['meta']['metadata'] = $p['meta'];
            }

            $legacy['meta']['actor'] = array(
                'ip' => isset($actor['ip']) ? (string) $actor['ip'] : '',
                'user_id' => isset($actor['user_id']) ? intval($actor['user_id']) : 0,
                'session_id' => isset($actor['session_id']) ? (string) $actor['session_id'] : '',
            );
            return $legacy;
        }

        return $event;
    }
public function get_effective_settings() {
    $site = get_option('asg_settings', array());
    if (!is_array($site)) { $site = array(); }
    if (is_multisite()) {
        $net = get_site_option('asg_network_settings', array());
        if (is_array($net) && $net) {

            return array_merge($net, $site);
        }
    }
    return $site;
}
private function get_site_salt() {
    $salt = get_option('asg_site_salt', '');
    if (!is_string($salt) || strlen($salt) < 24) {
        $salt = wp_generate_password(48, true, true);
        update_option('asg_site_salt', $salt, true);
    }
    return $salt;
}
private function hash_email($email) {
    $email = strtolower(trim((string) $email));
    if ($email === '') { return null; }
    $salt = $this->get_site_salt();
    return hash('sha256', $salt . '|' . $email);
}
private function hash_ip($ip) {
    $ip = trim((string) $ip);
    if ($ip === '') { return null; }
    $salt = $this->get_site_salt();
    return hash('sha256', $salt . '|' . $ip);
}
private function anonymize_ip($ip) {
    $ip = trim((string) $ip);
    if ($ip === '') { return ''; }
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
        $parts = explode('.', $ip);
        if (count($parts) === 4) {
            $parts[3] = '0';
            return implode('.', $parts);
        }
        return $ip;
    }
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {

        $bin = $this->ip_to_bin($ip);
        if ($bin === '') { return $ip; }
        $prefix = substr($bin, 0, 8); // 64 bits
        $masked = $prefix . str_repeat("\0", 8);
        $packed = $masked;
        $out = @inet_ntop($packed);
        return $out ? $out : $ip;
    }
    return $ip;
}
private function apply_ip_mode($ip) {
    $mode = isset($this->settings['ip_mode']) ? (string) $this->settings['ip_mode'] : 'store';
    $mode = in_array($mode, array('store','anonymize','hash','off'), true) ? $mode : 'store';
    $out = array(
        'mode' => $mode,
        'ip_for_fp' => '',
        'ip_store' => null,
        'ip_hash' => null,
    );
    if ($mode === 'off') {
        return $out;
    }
    if ($mode === 'hash') {
        $out['ip_hash'] = $this->hash_ip($ip);
        $out['ip_for_fp'] = $out['ip_hash'] ? $out['ip_hash'] : '';
        $out['ip_store'] = null;
        return $out;
    }
    if ($mode === 'anonymize') {
        $anon = $this->anonymize_ip($ip);
        $out['ip_store'] = $anon;
        $out['ip_for_fp'] = $anon;
        return $out;
    }
    // store
    $out['ip_store'] = $ip;
    $out['ip_for_fp'] = $ip;
    return $out;
}
private function geo_lookup($ip) {
    $ip = trim((string) $ip);
    if ($ip === '') { return array('country'=>'', 'asn'=>0, 'source'=>''); }
    $provider = isset($this->settings['geo_provider']) ? (string) $this->settings['geo_provider'] : 'headers';
    $provider = in_array($provider, array('none','headers','maxmind'), true) ? $provider : 'headers';
    $cache_key = 'asg_geo_' . substr(hash('sha256', $ip), 0, 20) . '_' . $provider;
    $cached = get_transient($cache_key);
    if (is_array($cached) && isset($cached['country'])) { return $cached; }
    $out = array('country'=>'', 'asn'=>0, 'source'=>$provider);
    if ($provider === 'none') {
        set_transient($cache_key, $out, 6 * HOUR_IN_SECONDS);
        return $out;
    }
    if ($provider === 'headers') {

        $cc = '';
        foreach (array('HTTP_CF_IPCOUNTRY','HTTP_X_APPENGINE_COUNTRY','HTTP_X_COUNTRY_CODE','HTTP_X_COUNTRY') as $hk) {
            if (!empty($_SERVER[$hk]) && is_string($_SERVER[$hk])) { $cc = strtoupper(trim(sanitize_text_field(wp_unslash((string) $_SERVER[$hk])))); break; }
        }
        if (preg_match('/^[A-Z]{2}$/', $cc)) { $out['country'] = $cc; }
        $asn = 0;
        foreach (array('HTTP_CF_IPASN','HTTP_X_ASN','HTTP_X_ORG_ASN') as $ak) {
            if (!empty($_SERVER[$ak])) { $asn = intval(wp_unslash((string) $_SERVER[$ak])); if ($asn > 0) { break; } }
        }
        if ($asn > 0) { $out['asn'] = $asn; }
        set_transient($cache_key, $out, 6 * HOUR_IN_SECONDS);
        return $out;
    }
    if ($provider === 'maxmind') {

        if (!empty($this->settings['no_external_calls'])) {

        }
        $country_db = !empty($this->settings['maxmind_country_mmdb']) ? (string) $this->settings['maxmind_country_mmdb'] : '';
        $asn_db = !empty($this->settings['maxmind_asn_mmdb']) ? (string) $this->settings['maxmind_asn_mmdb'] : '';
        if (class_exists('GeoIp2\\Database\\Reader')) {
            try {
                if ($country_db && file_exists($country_db)) {
                    $r = new GeoIp2\Database\Reader($country_db);
                    $rec = $r->country($ip);
                    if (!empty($rec->country) && !empty($rec->country->isoCode)) {
                        $out['country'] = strtoupper((string)$rec->country->isoCode);
                    }
                }
            } catch (Exception $e) { /* ignore */ }
            try {
                if ($asn_db && file_exists($asn_db)) {
                    $r2 = new GeoIp2\Database\Reader($asn_db);
                    $rec2 = $r2->asn($ip);
                    if (!empty($rec2->autonomousSystemNumber)) {
                        $out['asn'] = intval($rec2->autonomousSystemNumber);
                    }
                }
            } catch (Exception $e) { /* ignore */ }
        }
        set_transient($cache_key, $out, 12 * HOUR_IN_SECONDS);
        return $out;
    }
    set_transient($cache_key, $out, 6 * HOUR_IN_SECONDS);
    return $out;
}
    public static function install_schema() {
        global $wpdb;
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        $charset = $wpdb->get_charset_collate();
        $events  = $wpdb->prefix . 'asg_events';
        $fps     = $wpdb->prefix . 'asg_fingerprints';
        $lists   = $wpdb->prefix . 'asg_lists';
        $tokens  = $wpdb->prefix . 'asg_ml_tokens';
        $fw      = $wpdb->prefix . 'asg_firewall_events';
        $sql1 = "CREATE TABLE $events (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            created_at DATETIME NOT NULL,
            type VARCHAR(32) NOT NULL,
            action VARCHAR(16) NOT NULL,
            score INT(11) NOT NULL,
            reasons LONGTEXT NULL,
            ip VARBINARY(16) NULL,
            ip_hash CHAR(64) NULL,
            ua_hash CHAR(64) NULL,
            email_hash CHAR(64) NULL,
            fingerprint CHAR(64) NOT NULL,
            url TEXT NULL,
            payload LONGTEXT NULL,
            PRIMARY KEY  (id),
            KEY type (type),
            KEY created_at (created_at),
            KEY fingerprint (fingerprint),
            KEY email_hash (email_hash),
            KEY ip_hash (ip_hash)
        ) $charset;";
        $sql2 = "CREATE TABLE $fps (
            fingerprint CHAR(64) NOT NULL,
            first_seen DATETIME NOT NULL,
            last_seen DATETIME NOT NULL,
            hits INT(11) NOT NULL DEFAULT 0,
            hits_10m INT(11) NOT NULL DEFAULT 0,
            window_start INT(11) NOT NULL DEFAULT 0,
            reputation INT(11) NOT NULL DEFAULT 0,
            PRIMARY KEY (fingerprint)
        ) $charset;";
        $sql3 = "CREATE TABLE $lists (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            list_type VARCHAR(8) NOT NULL, /* allow|deny */
            kind VARCHAR(16) NOT NULL, /* ip|cidr|email|domain|fingerprint */
            value VARCHAR(255) NOT NULL,
            note VARCHAR(255) NULL,
            created_at DATETIME NOT NULL,
            PRIMARY KEY (id),
            KEY list_type (list_type),
            KEY kind (kind),
            KEY value (value)
        ) $charset;";
        $sql4 = "CREATE TABLE $tokens (
            token VARCHAR(64) NOT NULL,
            spam_count BIGINT(20) UNSIGNED NOT NULL DEFAULT 0,
            ham_count BIGINT(20) UNSIGNED NOT NULL DEFAULT 0,
            PRIMARY KEY (token)
        ) $charset;";
        $fw = $wpdb->prefix . 'asg_firewall_events';
        $sql5 = "CREATE TABLE $fw (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            created_at DATETIME NOT NULL,
            action VARCHAR(16) NOT NULL,
            reasons LONGTEXT NULL,
            ip VARBINARY(16) NULL,
            ip_hash CHAR(64) NULL,
            fingerprint CHAR(64) NULL,
            uri TEXT NULL,
            ua_hash CHAR(64) NULL,
            meta LONGTEXT NULL,
            PRIMARY KEY (id),
            KEY created_at (created_at),
            KEY action (action),
            KEY ip_hash (ip_hash),
            KEY fingerprint (fingerprint)
        ) $charset;";
        dbDelta($sql1);
        dbDelta($sql2);
        dbDelta($sql3);
        dbDelta($sql4);
        dbDelta($sql5);
    }
    public function cleanup() {
        $days = isset($this->settings['log_retention_days']) ? max(1, intval($this->settings['log_retention_days'])) : 90;
        global $wpdb;
        $events = self::sanitize_db_prefix($wpdb->prefix) . 'asg_events';
        $cut = gmdate('Y-m-d H:i:s', time() - ($days * DAY_IN_SECONDS));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->query( $wpdb->prepare( 'DELETE FROM %i WHERE created_at < %s', $events, $cut ) );
    }
    public function evaluate($event) {
        $this->settings = $this->get_effective_settings();
        $event = $this->normalize_event($event);
        $type = isset($event['type']) ? sanitize_key($event['type']) : 'unknown';
        $raw_ip   = $this->get_ip();
        if (isset($event['meta']['actor']['ip']) && $event['meta']['actor']['ip'] !== '') {
            $raw_ip = (string) $event['meta']['actor']['ip'];
        }
        $ua = '';
        if ( isset( $_SERVER['HTTP_USER_AGENT'] ) && is_string( $_SERVER['HTTP_USER_AGENT'] ) ) {
            $ua = sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
            $ua = trim( $ua );
        }
        $ua_hash = hash('sha256', $ua);
        $email = isset($event['email']) ? strtolower(trim((string) $event['email'])) : '';
        $email_hash = $email ? $this->hash_email($email) : null;
        $ipm = $this->apply_ip_mode($raw_ip);
        $ip = $ipm['ip_store']; // may be null
        $ip_hash = $ipm['ip_hash'];
        $geo = ($raw_ip && (!empty($this->settings['mod_geo_asn'])) && (!empty($this->settings['enable_geo_rules']) || !empty($this->settings['enable_asn_rules']))) ? $this->geo_lookup($raw_ip) : array('country'=>'','asn'=>0,'source'=>'');
        $fingerprint = $this->fingerprint($type, (string)$ipm['ip_for_fp'], $ua, $email);
        $reasons = array();
        $score = 0;
        $breakdown = array();
        $add_bd = function($code, $points) use (&$breakdown) { $breakdown[] = array('code'=> (string)$code, 'points'=> intval($points)); };
        if (!empty($this->settings['mod_fingerprinting'])) {
        $ua = '';
        if ( isset( $_SERVER['HTTP_USER_AGENT'] ) && is_string( $_SERVER['HTTP_USER_AGENT'] ) ) {
            $ua = sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
            $ua = trim( $ua );
        }
        if ($ua === '') {
            $score += 8; $reasons[] = 'missing_user_agent'; $add_bd('missing_user_agent', 8);
        } else {
            $ua_l = strtolower($ua);
            foreach (array('curl','wget','python-requests','httpclient','scrapy','libwww','java','go-http-client') as $sig) {
                if (strpos($ua_l, $sig) !== false) { $score += 6; $reasons[] = 'suspicious_user_agent'; $add_bd('suspicious_user_agent', 6); break; }
            }
        }
        $al = '';
        if ( isset( $_SERVER['HTTP_ACCEPT_LANGUAGE'] ) && is_string( $_SERVER['HTTP_ACCEPT_LANGUAGE'] ) ) {
            $al = sanitize_text_field( wp_unslash( $_SERVER['HTTP_ACCEPT_LANGUAGE'] ) );
            $al = trim( $al );
        }
        if ($al === '' && $type !== 'rest') {
            $score += 4; $reasons[] = 'missing_accept_language'; $add_bd('missing_accept_language', 4);
        }
        $ref = '';
        if ( isset( $_SERVER['HTTP_REFERER'] ) && is_string( $_SERVER['HTTP_REFERER'] ) ) {
            $ref = esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) );
            $ref = trim( $ref );
        }
        if ($ref === '' && in_array($type, array('form','registration','checkout','review','comment'), true)) {
            $score += 3; $reasons[] = 'missing_referer'; $add_bd('missing_referer', 3);
        } elseif ( $ref !== '' ) {
            $host = '';
            if ( isset( $_SERVER['HTTP_HOST'] ) && is_string( $_SERVER['HTTP_HOST'] ) ) {
                $host = sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) );
                $host = trim( $host );
            }
            if ( $host !== '' && strpos( $ref, $host ) === false ) {
                $score += 2; $reasons[] = 'external_referer'; $add_bd('external_referer', 2);
            }
        }
        }
        $list_action = $this->list_decision($raw_ip, $email, $fingerprint, $geo);
        if ($list_action === 'allow') {
            if (!isset($event['meta']) || !is_array($event['meta'])) { $event['meta'] = array(); }
            $event['meta']['score_breakdown'] = array(array('code'=>'allowlist','points'=>0));
            $decision = $this->finalize($event, $type, 'allow', 0, array('allowlist'), $raw_ip, $ip, $ip_hash, $geo, $ua_hash, $email_hash, $fingerprint);
            return $decision;
        }
        if ($list_action === 'deny') {
            if (!isset($event['meta']) || !is_array($event['meta'])) { $event['meta'] = array(); }
            $event['meta']['score_breakdown'] = array(array('code'=>'denylist','points'=>100));
            $decision = $this->finalize($event, $type, 'block', 100, array('denylist'), $raw_ip, $ip, $ip_hash, $geo, $ua_hash, $email_hash, $fingerprint);
            return $decision;
        }

        if (!empty($this->settings['mod_content'])) {
        $content = isset($event['content']) ? (string) $event['content'] : '';
        $url = isset($event['url']) ? (string) $event['url'] : '';
        $link_count = $this->count_links($content . ' ' . $url);
        if ($link_count >= 2) { $p = $this->w(10,'link_count'); $score += $p; $reasons[] = 'multiple_links'; $add_bd('multiple_links',$p); }
        if ($link_count >= 5) { $p = $this->w(20,'link_count'); $score += $p; $reasons[] = 'many_links'; $add_bd('many_links',$p); }
        if (preg_match('/\b(?:viagra|cialis|casino|crypto|loan|payday)\b/i', $content)) { $p = 15; $score += $p; $reasons[] = 'spam_keywords'; $add_bd('spam_keywords',$p); }
        if (preg_match('/\[url\]|\[link\]|<a\s/i', $content)) { $p = $this->w(8,'link_markup'); $score += $p; $reasons[] = 'link_markup'; $add_bd('link_markup',$p); }
        if (preg_match('/(?:https?:\/\/)?(?:bit\.ly|t\.co|tinyurl\.com|goo\.gl)\b/i', $content)) { $p = 12; $score += $p; $reasons[] = 'shortener'; $add_bd('shortener',$p); }

        $tld = $this->suspicious_tld_score($content . ' ' . $url);
        if ($tld['score'] > 0) { $p = intval($tld['score']); $score += $p; $reasons = array_merge($reasons, $tld['reasons']); $add_bd('suspicious_tld',$p); }
        $obf = $this->unicode_obfuscation_score($content);
        if ($obf['score'] > 0) { $p = intval($obf['score']); $score += $p; $reasons = array_merge($reasons, $obf['reasons']); $add_bd('obfuscation',$p); }

        $ng = $this->ngram_phrase_score($content);
        if ($ng['score'] > 0) { $score += $ng['score']; $reasons = array_merge($reasons, $ng['reasons']); }

        $sim = $this->similarity_score($type, $content);
        if ($sim['score'] > 0) { $score += $sim['score']; $reasons = array_merge($reasons, $sim['reasons']); }
        }

        if (!empty($this->settings['mod_behavior'])) {
        $vel = $this->velocity_score($fingerprint);
        if ($vel['score'] > 0) { $p = intval($vel['score']); $score += $p; $reasons = array_merge($reasons, $vel['reasons']); $add_bd('velocity',$p); }
        }

        if (!empty($event['meta']) && is_array($event['meta'])) {
            $meta = $event['meta'];
            if (isset($meta['hp']) && is_string($meta['hp']) && trim($meta['hp']) !== '') {
            if (!empty($this->settings['mod_challenges'])) {
                $p = $this->w(60,'honeypot'); $score += $p; $reasons[] = 'honeypot_filled'; $add_bd('honeypot_filled',$p);
            }
            }
            $min_sec = isset($this->settings['min_seconds_to_submit']) ? max(1, intval($this->settings['min_seconds_to_submit'])) : 3;
            if (!empty($meta['asg_ts']) && is_numeric($meta['asg_ts'])) {
                $delta = time() - intval($meta['asg_ts']);
            if (!empty($this->settings['mod_behavior'])) {
                if ($delta < $min_sec) { $p = $this->w(25,'too_fast'); $score += $p; $reasons[] = 'too_fast'; $add_bd('too_fast',$p); }
            }
            }
            if (isset($meta['asg_nonce']) && isset($meta['asg_js'])) {
                $ok = $this->verify_js_proof((string)$meta['asg_nonce'], (string)$meta['asg_js'], $ua);
            if (!empty($this->settings['mod_challenges'])) {
                if (!$ok) { $p = $this->w(25,'js_proof'); $score += $p; $reasons[] = 'js_proof_failed'; $add_bd('js_proof_failed',$p); }
            }
            }
        }
            if (!empty($meta['asg_captcha_ok'])) {
            if (!empty($this->settings['mod_challenges'])) {
                $p = -20; $score += $p; $reasons[] = 'captcha_passed'; $add_bd('captcha_passed',$p);
            }
            }
        if (!empty($this->settings['mod_identity'])) {

        if ($email) {
            if (!is_email($email)) { $p = $this->w(20,'invalid_email'); $score += $p; $reasons[] = 'invalid_email_format'; $add_bd('invalid_email_format',$p); }
            if ($this->is_disposable_email($email)) { $p = $this->w(18,'disposable_email'); $score += $p; $reasons[] = 'disposable_email'; $add_bd('disposable_email',$p); }

            if (!empty($this->settings['enable_mx_checks'])) {
                if (!$this->email_domain_has_mx($email)) { $p = 10; $score += $p; $reasons[] = 'email_mx_missing'; $add_bd('email_mx_missing',$p); }
            }

            $name = isset($event['name']) ? (string) $event['name'] : '';
            if ($name && $this->name_email_mismatch($name, $email)) { $p = 8; $score += $p; $reasons[] = 'name_email_mismatch'; $add_bd('name_email_mismatch',$p); }
        }
        }

        $enable_ml = !empty($this->settings['enable_ml']);
        if ($enable_ml && $content) {
            $ml_score = $this->ml->score_text($content);
            if ($ml_score !== 0) {
                $weight = isset($this->settings['ml_weight']) ? intval($this->settings['ml_weight']) : 12;
                $adj = (int) round(max(-1, min(1, $ml_score)) * $weight);
                $score += $adj;
                $reasons[] = ($adj >= 0) ? 'ml_spam_signal' : 'ml_ham_signal';
                $add_bd(($adj >= 0) ? 'ml_spam_signal' : 'ml_ham_signal', $adj);
            }
        }

        $score = max(0, min(100, $score));

        $action = $this->score_to_action($score);

        if ($type === 'form' && $action === 'challenge' && !empty($this->settings['enable_progressive_delay'])) {
            $t_low = isset($this->settings['threshold_allow']) ? intval($this->settings['threshold_allow']) : 29;
            $t_ch  = isset($this->settings['threshold_challenge']) ? intval($this->settings['threshold_challenge']) : 59;
            $minms = isset($this->settings['progressive_delay_min_ms']) ? max(0, intval($this->settings['progressive_delay_min_ms'])) : 150;
            $maxms = isset($this->settings['progressive_delay_max_ms']) ? max($minms, intval($this->settings['progressive_delay_max_ms'])) : 1800;
            $band_lo = $t_low + 1;
            $band_hi = $t_ch;
            $ratio = ($band_hi > $band_lo) ? (($score - $band_lo) / ($band_hi - $band_lo)) : 0;
            $ratio = max(0.0, min(1.0, floatval($ratio)));
            $ms = intval(round($minms + ($maxms - $minms) * $ratio));
            if ($ms > 0 && $ms <= 5000) {
                usleep($ms * 1000);
                $reasons[] = 'progressive_delay';
                $add_bd('progressive_delay', 0);
                if (!isset($event['meta']) || !is_array($event['meta'])) { $event['meta'] = array(); }
                $event['meta']['progressive_delay_ms'] = $ms;
            }
        }

        if (!empty($this->settings['fp_hold_instead_of_block']) && $action === 'block') {
            $action = 'hold';
            $reasons[] = 'fp_hold_protection';
        }

        if ($type === 'comment' && $action === 'block' && $score < 95) {
            $action = 'hold';
            $reasons[] = 'softened_for_comments';
        }
        if (!isset($event['meta']) || !is_array($event['meta'])) { $event['meta'] = array(); }
        $event['meta']['score_breakdown'] = $breakdown;
        return $this->finalize($event, $type, $action, $score, $reasons, $raw_ip, $ip, $ip_hash, $geo, $ua_hash, $email_hash, $fingerprint);
    }
    private function finalize($event, $type, $action, $score, $reasons, $raw_ip, $ip_store, $ip_hash, $geo, $ua_hash, $email_hash, $fingerprint) {
        $id = $this->log_event($type, $action, $score, $reasons, $raw_ip, $ip_store, $ip_hash, $geo, $ua_hash, $email_hash, $fingerprint, $event);
        return array(
            'id' => $id,
            'evidence_id' => $id,
            'type' => $type,
            'action' => $action,
            'score' => $score,
            'reasons' => $reasons,
            'fingerprint' => $fingerprint,
        );
    }
    private function score_to_action($score) {
        $a = isset($this->settings['threshold_allow']) ? intval($this->settings['threshold_allow']) : 29;
        $c = isset($this->settings['threshold_challenge']) ? intval($this->settings['threshold_challenge']) : 59;
        $h = isset($this->settings['threshold_hold']) ? intval($this->settings['threshold_hold']) : 79;
        if ($score <= $a) { return 'allow'; }
        if ($score <= $c) { return 'challenge'; }
        if ($score <= $h) { return 'hold'; }
        return 'block';
    }
    private function log_event($type, $action, $score, $reasons, $raw_ip, $ip_store, $ip_hash, $geo, $ua_hash, $email_hash, $fingerprint, $event) {
        
        if (!empty($event['meta']['metadata']['no_log'])) { return 0; }
global $wpdb;
        $events = self::sanitize_db_prefix($wpdb->prefix) . 'asg_events';
		$payload = array(
			'name' => isset($event['name']) ? sanitize_text_field((string) $event['name']) : '',
			'privacy' => array(
				'ip_mode' => isset($this->settings['ip_mode']) ? sanitize_key((string)$this->settings['ip_mode']) : 'store',
				'no_external_calls' => !empty($this->settings['no_external_calls']) ? 1 : 0,
			),
			'geo' => is_array($geo) ? $geo : array('country'=>'','asn'=>0,'source'=>''),
			'ip_hash' => $ip_hash ? sanitize_text_field((string) $ip_hash) : '',
			'raw_ip_present' => ($raw_ip ? 1 : 0),
			'actor' => (isset($event['meta']['actor']) && is_array($event['meta']['actor'])) ? $event['meta']['actor'] : array(),
			'email' => isset($event['email']) ? sanitize_email((string) $event['email']) : '',
			'url' => isset($event['url']) ? esc_url_raw((string) $event['url']) : '',
			// Store a safe text representation (prevents saving HTML/JS payloads)
			'content' => isset($event['content']) ? sanitize_textarea_field(wp_strip_all_tags((string) $event['content'])) : '',
			'meta' => (isset($event['meta']) && is_array($event['meta'])) ? $event['meta'] : array(),
		);
        $ip_bin = $this->ip_to_bin($ip_store);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->insert(
            $events,
            array(
                'created_at' => gmdate('Y-m-d H:i:s'),
                'type' => $type,
                'action' => $action,
                'score' => intval($score),
                'reasons' => wp_json_encode(array_values(array_unique($reasons))),
                'ip' => $ip_bin,
                'ip_hash' => $ip_hash,
                'ua_hash' => $ua_hash,
                'email_hash' => $email_hash,
                'fingerprint' => $fingerprint,
                'url' => isset($event['url']) ? esc_url_raw((string) $event['url']) : '',
                'payload' => wp_json_encode($payload),
            ),
            array('%s','%s','%s','%d','%s','%s','%s','%s','%s','%s','%s','%s')
        );
        return (int) $wpdb->insert_id;
    }
        private function list_decision($raw_ip, $email, $fingerprint, $geo) {
        global $wpdb;
        $lists = self::sanitize_db_prefix($wpdb->prefix) . 'asg_lists';
        $ip_norm = $this->normalize_ip($raw_ip);

        // CIDR rules (optional)
        $cidr_deny = wp_cache_get('asg_cidr_deny', 'aegisspam');
        if (false === $cidr_deny) {
            $cidr_deny = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT value FROM %i WHERE list_type=%s AND kind=%s LIMIT %d',
                    $lists,
                    'deny',
                    'cidr',
                    500
                )
            ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            wp_cache_set('asg_cidr_deny', $cidr_deny, 'aegisspam', 300);
        }
        if ($cidr_deny) {
            foreach ($cidr_deny as $c) {
                if ($this->ip_in_cidr($ip_norm, $c)) { return 'deny'; }
            }
        }

        $cidr_allow = wp_cache_get('asg_cidr_allow', 'aegisspam');
        if (false === $cidr_allow) {
            $cidr_allow = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT value FROM %i WHERE list_type=%s AND kind=%s LIMIT %d',
                    $lists,
                    'allow',
                    'cidr',
                    500
                )
            ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            wp_cache_set('asg_cidr_allow', $cidr_allow, 'aegisspam', 300);
        }
        if ($cidr_allow) {
            foreach ($cidr_allow as $c) {
                if ($this->ip_in_cidr($ip_norm, $c)) { return 'allow'; }
            }
        }

        $domain = '';
        if ($email && strpos($email, '@') !== false) {
            $domain = substr($email, strpos($email, '@') + 1);
            $domain = strtolower(trim($domain));
        }

        // Check deny first
        $deny = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare("SELECT id FROM %i WHERE list_type='deny' AND (
                (kind='ip' AND value=%s) OR
                (kind='email' AND value=%s) OR
                (kind='domain' AND value=%s) OR
                (kind='fingerprint' AND value=%s)
            ) LIMIT 1",
            $lists,
            $ip_norm, $email, $domain, $fingerprint
        )); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        if ($deny) { return 'deny'; }

        $cc = isset($geo['country']) ? strtoupper((string)$geo['country']) : '';
        $asn = isset($geo['asn']) ? intval($geo['asn']) : 0;
        if ($cc !== '') {
            $deny_cc = $wpdb->get_var($wpdb->prepare("SELECT id FROM %i WHERE list_type='deny' AND kind='country' AND value=%s LIMIT 1", $lists, $cc)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            if ($deny_cc) { return 'deny'; }
            $allow_cc = $wpdb->get_var($wpdb->prepare("SELECT id FROM %i WHERE list_type='allow' AND kind='country' AND value=%s LIMIT 1", $lists, $cc)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            if ($allow_cc) { return 'allow'; }
        }
        if ($asn > 0) {
            $deny_asn = $wpdb->get_var($wpdb->prepare("SELECT id FROM %i WHERE list_type='deny' AND kind='asn' AND value=%s LIMIT 1", $lists, (string)$asn)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            if ($deny_asn) { return 'deny'; }
            $allow_asn = $wpdb->get_var($wpdb->prepare("SELECT id FROM %i WHERE list_type='allow' AND kind='asn' AND value=%s LIMIT 1", $lists, (string)$asn)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            if ($allow_asn) { return 'allow'; }
        }

        // Allow list
        $allow = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare("SELECT id FROM %i WHERE list_type='allow' AND (
                (kind='ip' AND value=%s) OR
                (kind='email' AND value=%s) OR
                (kind='domain' AND value=%s) OR
                (kind='fingerprint' AND value=%s)
            ) LIMIT 1",
            $lists,
            $ip_norm, $email, $domain, $fingerprint
        )); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        if ($allow) { return 'allow'; }

        return '';
    }

    private function velocity_score($fingerprint) {
        $window = isset($this->settings['velocity_window_seconds']) ? max(60, intval($this->settings['velocity_window_seconds'])) : 600;
        $limit  = isset($this->settings['velocity_limit']) ? max(1, intval($this->settings['velocity_limit'])) : 5;

        $key = 'asg_vel_' . $fingerprint;
        $state = get_transient($key);
        if (!is_array($state)) {
            $state = array('start'=>time(), 'count'=>0);
        }
        $now = time();
        if (($now - intval($state['start'])) > $window) {
            $state = array('start'=>$now, 'count'=>0);
        }
        $state['count'] = intval($state['count']) + 1;
        set_transient($key, $state, $window + 60);

        $this->touch_fingerprint($fingerprint, $state, $window);
        $reasons = array();
        $score = 0;
        $breakdown = array();
        $add_bd = function($code, $points) use (&$breakdown) { $breakdown[] = array('code'=> (string)$code, 'points'=> intval($points)); };
        if ($state['count'] > $limit) {
            $over = $state['count'] - $limit;
            $score = min(35, 10 + ($over * 5));
            $reasons[] = 'velocity_exceeded';
        }
        return array('score'=>$score,'reasons'=>$reasons,'count'=>$state['count']);
    }
    private function touch_fingerprint($fingerprint, $state, $window) {
        global $wpdb;
        $fps = self::sanitize_db_prefix($wpdb->prefix) . 'asg_fingerprints';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $row = $wpdb->get_row( $wpdb->prepare( 'SELECT fingerprint, hits, hits_10m, window_start FROM %i WHERE fingerprint=%s', $fps, $fingerprint ), ARRAY_A );
        $now = time();
        $ws  = $row ? intval($row['window_start']) : 0;
        $hits10 = $row ? intval($row['hits_10m']) : 0;
        if (!$row) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->insert($fps, array(
                'fingerprint' => $fingerprint,
                'first_seen' => gmdate('Y-m-d H:i:s'),
                'last_seen'  => gmdate('Y-m-d H:i:s'),
                'hits' => 1,
                'hits_10m' => 1,
                'window_start' => $now,
                'reputation' => 0,
            ), array('%s','%s','%s','%d','%d','%d','%d'));
            return;
        }
        if (($now - $ws) > $window) {
            $hits10 = 1;
            $ws = $now;
        } else {
            $hits10 = min(1000000, $hits10 + 1);
        }
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->update($fps, array(
            'last_seen' => gmdate('Y-m-d H:i:s'),
            'hits' => intval($row['hits']) + 1,
            'hits_10m' => $hits10,
            'window_start' => $ws,
        ), array('fingerprint'=>$fingerprint), array('%s','%d','%d','%d'), array('%s'));
    }
    private function verify_js_proof($nonce, $token, $ua) {
        $nonce = trim($nonce);
        $token = trim($token);
        if ($nonce === '' || $token === '') { return false; }
        $key = 'asg_nonce_' . md5($nonce);
        $t = get_transient($key);
        if (!$t) { return false; }

        delete_transient($key);

        $expected = hash('sha256', $nonce . '|' . $ua);
        if (hash_equals($expected, $token)) {
            return true;
        }

        if (preg_match('/^-?\d+$/', $token)) {

            $data = $nonce . '|' . $ua;
            $h = 0;
            $len = strlen($data);
            for ($i=0; $i<$len; $i++) {
                $h = (($h << 5) - $h) + ord($data[$i]);
                $h = $h & 0xFFFFFFFF;
            }
            $h = (int) $h;
            return ((string)$h === (string)$token);
        }
        return false;
    }
    private function is_disposable_email($email) {

        $parts = explode('@', strtolower((string) $email));
        if (count($parts) !== 2) { return false; }
        $domain = trim($parts[1]);
        if ($domain === '') { return false; }
        $domains = $this->get_disposable_domains();
        return in_array($domain, $domains, true);
    }
    private function get_disposable_domains() {
        $cached = get_transient('asg_disposable_domains');
        if (is_array($cached) && $cached) { return $cached; }
        $base = array(
            'mailinator.com','guerrillamail.com','10minutemail.com','tempmail.com','yopmail.com',
            'trashmail.com','dispostable.com','getnada.com','fakeinbox.com','sharklasers.com'
        );

        $base = apply_filters('asg_disposable_domains', $base);

        if (empty($this->settings['no_external_calls']) && !empty($this->settings['enable_external_disposable']) && !empty($this->settings['external_disposable_url'])) {
            $url = esc_url_raw((string) $this->settings['external_disposable_url']);
            if ($url) {
                $res = wp_remote_get($url, array('timeout' => 5, 'redirection' => 2));
                if (!is_wp_error($res)) {
                    $body = wp_remote_retrieve_body($res);
                    if (is_string($body) && $body !== '') {
                        $lines = preg_split("/\r\n|\n|\r/", $body);
                        foreach ($lines as $ln) {
                            $ln = strtolower(trim((string) $ln));
                            if ($ln === '' || $ln[0] === '#') { continue; }
                            if (preg_match('/[^a-z0-9\.\-]/', $ln)) { continue; }
                            $base[] = $ln;
                        }
                    }
                }
            }
        }
        $out = array_values(array_unique(array_filter(array_map('trim', $base))));
        set_transient('asg_disposable_domains', $out, 7 * DAY_IN_SECONDS);
        return $out;
    }
    private function count_links($text) {
        if (!$text) { return 0; }
        preg_match_all('/https?:\/\//i', $text, $m);
        return isset($m[0]) ? count($m[0]) : 0;
    }
    private function extract_domains($text) {
        $text = (string) $text;
        if ($text === '') { return array(); }
        $domains = array();

        if (preg_match_all('#https?://[^\s<>"\']+#i', $text, $m)) {
            foreach ($m[0] as $u) {
                $host = wp_parse_url($u, PHP_URL_HOST);
                if (!$host) { continue; }
                $host = strtolower(preg_replace('/^www\./i', '', $host));
                $domains[] = $host;
            }
        }

        if (preg_match_all('/\b([a-z0-9][a-z0-9\-]{0,62}\.)+[a-z]{2,24}\b/i', $text, $m2)) {
            foreach ($m2[0] as $d) {
                $d = strtolower(preg_replace('/^www\./i', '', $d));
                $domains[] = $d;
            }
        }
        return array_values(array_unique($domains));
    }
    private function suspicious_tld_score($text) {
        $domains = $this->extract_domains($text);
        if (!$domains) { return array('score'=>0,'reasons'=>array()); }
        $suspicious = array(
            'top','xyz','click','info','loan','work','gq','tk','ml','cf','ga','pw','kim','link','shop','icu','cyou',
            'monster','rest','bar','cam','surf','live','biz','buzz','club','online','site','sbs'
        );
        $hits = 0;
        foreach ($domains as $d) {
            $parts = explode('.', $d);
            $tld = end($parts);
            if ($tld && in_array($tld, $suspicious, true)) { $hits++; }
        }
        if ($hits <= 0) { return array('score'=>0,'reasons'=>array()); }
        $score = min(25, $hits * 8);
        return array('score'=>$score, 'reasons'=>array('suspicious_tld'));
    }
    private function unicode_obfuscation_score($text) {
        $text = (string) $text;
        if ($text === '') { return array('score'=>0,'reasons'=>array()); }
        $score = 0;
        $reasons = array();

        if (preg_match('/[\x{200B}-\x{200D}\x{FEFF}]/u', $text)) {
            $score += 12; $reasons[] = 'zero_width_chars';
        }

        if (preg_match('/&#x[0-9a-f]{2,6};|&#\d{2,6};/i', $text)) {
            $score += 8; $reasons[] = 'entity_obfuscation';
        }

        if (preg_match('/%[0-9a-f]{2}/i', $text)) {
            $score += 6; $reasons[] = 'percent_encoding';
        }

        $has_latin = preg_match('/[A-Za-z]/', $text);
        $has_cyr   = preg_match('/[\x{0400}-\x{04FF}]/u', $text);
        $has_grk   = preg_match('/[\x{0370}-\x{03FF}]/u', $text);
        if ($has_latin && ($has_cyr || $has_grk)) {
            $score += 12; $reasons[] = 'mixed_scripts';
        }

        $len = strlen($text);
        if ($len > 0) {
            $ascii = preg_match_all('/[\x00-\x7F]/', $text, $tmp);
            $ratio = ($len - (int)$ascii) / $len;
            if ($ratio >= 0.25) { $score += 6; $reasons[] = 'high_unicode_ratio'; }
        }
        $score = min(25, $score);
        return array('score'=>$score, 'reasons'=>array_values(array_unique($reasons)));
    }
    private function simple_tokens($text) {
        $text = strtolower((string) $text);
        $text = wp_strip_all_tags($text);
        $text = preg_replace('/https?:\/\/[^\s]+/i', ' __url__ ', $text);
        $text = preg_replace('/[a-z0-9._%+-]+@[a-z0-9.-]+\.[a-z]{2,}/i', ' __email__ ', $text);
        $parts = preg_split('/[^a-z0-9_]+/i', $text);
        $out = array();
        foreach ($parts as $p) {
            $p = trim($p);
            if ($p === '' || strlen($p) < 3) { continue; }
            if (preg_match('/^\d+$/', $p)) { $p = '__num__'; }
            $out[] = $p;
        }
        return $out;
    }
    private function ngram_phrase_score($text) {
        $text = (string) $text;
        if ($text === '') { return array('score'=>0,'reasons'=>array()); }
        $tokens = $this->simple_tokens($text);
        if (count($tokens) < 4) { return array('score'=>0,'reasons'=>array()); }
        $max_terms = 24;
        $tokens = array_slice($tokens, 0, $max_terms);
        $ngrams = array();
        $n = count($tokens);
        for ($i=0; $i<$n-1; $i++) {
            $ngrams[] = $tokens[$i] . '_' . $tokens[$i+1];
            if ($i < $n-2) {
                $ngrams[] = $tokens[$i] . '_' . $tokens[$i+1] . '_' . $tokens[$i+2];
            }
        }
        $ngrams = array_values(array_unique($ngrams));
        if (!$ngrams) { return array('score'=>0,'reasons'=>array()); }
        global $wpdb;
        $table = self::sanitize_db_prefix($wpdb->prefix) . 'asg_ml_tokens';
        $table_esc = str_replace('`', '``', (string) $table);

        $ngrams = array_slice($ngrams, 0, 40);

        $n = count($ngrams);
        if ($n <= 10) { $bucket = 10; }
        elseif ($n <= 20) { $bucket = 20; }
        elseif ($n <= 30) { $bucket = 30; }
        else { $bucket = 40; }

        // Pad values so we can use fixed SQL strings per branch (PluginCheck-friendly).
        while (count($ngrams) < $bucket) { $ngrams[] = ''; }

        if ($bucket === 10) {
            $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT token, spam_count, ham_count FROM %i WHERE token IN (%s,%s,%s,%s,%s,%s,%s,%s,%s,%s)',
                    $table,
                    $ngrams[0], $ngrams[1], $ngrams[2], $ngrams[3], $ngrams[4],
                    $ngrams[5], $ngrams[6], $ngrams[7], $ngrams[8], $ngrams[9]
                ),
                ARRAY_A
            );
        } elseif ($bucket === 20) {
            $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT token, spam_count, ham_count FROM %i WHERE token IN (%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s)',
                    $table,
                    $ngrams[0], $ngrams[1], $ngrams[2], $ngrams[3], $ngrams[4],
                    $ngrams[5], $ngrams[6], $ngrams[7], $ngrams[8], $ngrams[9],
                    $ngrams[10], $ngrams[11], $ngrams[12], $ngrams[13], $ngrams[14],
                    $ngrams[15], $ngrams[16], $ngrams[17], $ngrams[18], $ngrams[19]
                ),
                ARRAY_A
            );
        } elseif ($bucket === 30) {
            $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT token, spam_count, ham_count FROM %i WHERE token IN (%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s)',
                    $table,
                    $ngrams[0], $ngrams[1], $ngrams[2], $ngrams[3], $ngrams[4],
                    $ngrams[5], $ngrams[6], $ngrams[7], $ngrams[8], $ngrams[9],
                    $ngrams[10], $ngrams[11], $ngrams[12], $ngrams[13], $ngrams[14],
                    $ngrams[15], $ngrams[16], $ngrams[17], $ngrams[18], $ngrams[19],
                    $ngrams[20], $ngrams[21], $ngrams[22], $ngrams[23], $ngrams[24],
                    $ngrams[25], $ngrams[26], $ngrams[27], $ngrams[28], $ngrams[29]
                ),
                ARRAY_A
            );
        } else {
            $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare(
                    'SELECT token, spam_count, ham_count FROM %i WHERE token IN (%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s)',
                    $table,
                    $ngrams[0], $ngrams[1], $ngrams[2], $ngrams[3], $ngrams[4],
                    $ngrams[5], $ngrams[6], $ngrams[7], $ngrams[8], $ngrams[9],
                    $ngrams[10], $ngrams[11], $ngrams[12], $ngrams[13], $ngrams[14],
                    $ngrams[15], $ngrams[16], $ngrams[17], $ngrams[18], $ngrams[19],
                    $ngrams[20], $ngrams[21], $ngrams[22], $ngrams[23], $ngrams[24],
                    $ngrams[25], $ngrams[26], $ngrams[27], $ngrams[28], $ngrams[29],
                    $ngrams[30], $ngrams[31], $ngrams[32], $ngrams[33], $ngrams[34],
                    $ngrams[35], $ngrams[36], $ngrams[37], $ngrams[38], $ngrams[39]
                ),
                ARRAY_A
            );
        }
        if (!$rows) { return array('score'=>0,'reasons'=>array()); }
        $strong = 0;
        foreach ($rows as $r) {
            $s = (int) $r['spam_count'];
            $h = (int) $r['ham_count'];
            $p = ($s + 1) / ($s + $h + 2);
            if ($p >= 0.85 && ($s + $h) >= 2) { $strong++; }
        }
        if ($strong <= 0) { return array('score'=>0,'reasons'=>array()); }
        $score = min(20, 6 + ($strong * 4));
        return array('score'=>$score, 'reasons'=>array('spam_phrase_ngrams'));
    }
    private function similarity_score($type, $text) {
        $text = (string) $text;
        if ($text === '') { return array('score'=>0,'reasons'=>array()); }
        if (empty($this->settings['enable_similarity'])) { return array('score'=>0,'reasons'=>array()); }
        $lookback = isset($this->settings['similarity_lookback_days']) ? max(3, intval($this->settings['similarity_lookback_days'])) : 30;
        $min_sim  = isset($this->settings['similarity_threshold']) ? floatval($this->settings['similarity_threshold']) : 0.72;
        $cache_key = 'asg_sim_' . substr(hash('sha256', $type . '|' . $text), 0, 16);
        $cached = get_transient($cache_key);
        if (is_array($cached) && isset($cached['score'])) { return $cached; }
        global $wpdb;
        $events = self::sanitize_db_prefix($wpdb->prefix) . 'asg_events';
        $since = gmdate('Y-m-d H:i:s', time() - ($lookback * DAY_IN_SECONDS));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $rows = $wpdb->get_results( $wpdb->prepare(
            'SELECT payload FROM %i WHERE type=%s AND action=%s AND created_at >= %s ORDER BY created_at DESC LIMIT 50',
            $events,
            $type,
            'block',
            $since
        ), ARRAY_A );
        if (!$rows) {
            $out = array('score'=>0,'reasons'=>array());
            set_transient($cache_key, $out, HOUR_IN_SECONDS);
            return $out;
        }
        $a = array_values(array_unique($this->simple_tokens($text)));
        if (count($a) < 5) {
            $out = array('score'=>0,'reasons'=>array());
            set_transient($cache_key, $out, HOUR_IN_SECONDS);
            return $out;
        }
        $a = array_slice($a, 0, 60);
        $aset = array_flip($a);
        $best = 0.0;
        foreach ($rows as $r) {
            $payload = is_string($r['payload']) ? $r['payload'] : '';
            $content = '';
            if ($payload) {
                $j = json_decode($payload, true);
                if (is_array($j) && isset($j['content'])) { $content = (string) $j['content']; }
            }
            if ($content === '') { continue; }
            $b = array_values(array_unique($this->simple_tokens($content)));
            $b = array_slice($b, 0, 60);
            if (count($b) < 5) { continue; }
            $inter = 0;
            foreach ($b as $t) {
                if (isset($aset[$t])) { $inter++; }
            }
            $union = count($a) + count($b) - $inter;
            if ($union <= 0) { continue; }
            $sim = $inter / $union;
            if ($sim > $best) { $best = $sim; }
            if ($best >= 0.90) { break; }
        }
        if ($best < $min_sim) {
            $out = array('score'=>0,'reasons'=>array());
            set_transient($cache_key, $out, HOUR_IN_SECONDS);
            return $out;
        }
        $score = (int) min(25, round(10 + ($best * 15)));
        $out = array('score'=>$score, 'reasons'=>array('similar_to_known_spam'));
        set_transient($cache_key, $out, HOUR_IN_SECONDS);
        return $out;
    }
    private function email_domain_has_mx($email) {
        $parts = explode('@', strtolower((string) $email));
        if (count($parts) !== 2) { return true; }
        $domain = trim($parts[1]);
        if ($domain === '') { return true; }
        $key = 'asg_mx_' . substr(hash('sha256', $domain), 0, 16);
        $cached = get_transient($key);
        if (is_array($cached) && isset($cached['ok'])) { return (bool) $cached['ok']; }

        $ok = true;
        if (function_exists('dns_get_record')) {
            $mx = @dns_get_record($domain, DNS_MX);
            $ok = is_array($mx) && !empty($mx);
        }
        set_transient($key, array('ok'=>$ok), 7 * DAY_IN_SECONDS);
        return $ok;
    }
    private function name_email_mismatch($name, $email) {
        $name = strtolower(trim((string) $name));
        $email = strtolower(trim((string) $email));
        if ($name === '' || $email === '' || strpos($email, '@') === false) { return false; }
        $local = substr($email, 0, strpos($email, '@'));
        $local = preg_replace('/[^a-z0-9]/', ' ', $local);
        $local = trim(preg_replace('/\s+/', ' ', $local));
        if ($local === '') { return false; }
        $parts = preg_split('/\s+/', preg_replace('/[^a-z0-9]+/', ' ', $name));
        $parts = array_values(array_filter($parts, function($p) { return strlen($p) >= 3; }));
        if (!$parts) { return false; }
        $hit = false;
        foreach ($parts as $p) {
            if (strpos($local, $p) !== false) { $hit = true; break; }
        }
        if ($hit) { return false; }

        $randish = (strlen($local) >= 8 && preg_match('/[a-z]/', $local) && preg_match('/\d/', $local));
        return (bool) $randish;
    }
    private function fingerprint($type, $ip, $ua, $email) {
        $ip = $this->normalize_ip($ip);
        $base = $type . '|' . $ip . '|' . substr($ua, 0, 120) . '|' . strtolower(trim((string)$email));
        return hash('sha256', $base);
    }
    private function get_ip() {

        $ip = '';
        if (isset($_SERVER['REMOTE_ADDR']) && is_string($_SERVER['REMOTE_ADDR'])) {
            $ip = sanitize_text_field( wp_unslash( (string) $_SERVER['REMOTE_ADDR'] ) );
            $ip = trim( $ip );
        }
        return $this->normalize_ip($ip);
    }
    private function normalize_ip($ip) {
        $ip = trim((string)$ip);
        if ($ip === '') { return '0.0.0.0'; }
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) { return $ip; }
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) { return $ip; }
        return '0.0.0.0';
    }
    private function ip_to_bin($ip) {
        $packed = @inet_pton($ip);
        if ($packed === false) { return null; }
        return $packed;
    }
    private function w($points, $key) {
        $k = 'w_' . $key;
        $level = isset($this->settings[$k]) ? (string) $this->settings[$k] : 'med';
        $mult = 1.0;
        if ($level === 'low') { $mult = 0.6; }
        elseif ($level === 'high') { $mult = 1.4; }
        $v = (int) round($points * $mult);
        return max(0, $v);
    }
    private function is_trusted_actor($event) {
        if (!empty($this->settings['trust_logged_in']) && function_exists('is_user_logged_in') && is_user_logged_in()) {
            return true;
        }

        if (!empty($this->settings['trust_existing_user_days']) && function_exists('get_user_by')) {
            $email = isset($event['email']) ? (string) $event['email'] : '';
            if ($email && is_email($email)) {
                $u = get_user_by('email', $email);
                if ($u && !empty($u->user_registered)) {
                    $days = max(0, intval($this->settings['trust_existing_user_days']));
                    if ($days > 0) {
                        $age = time() - strtotime($u->user_registered);
                        if ($age >= ($days * DAY_IN_SECONDS)) { return true; }
                    }
                }
            }
        }

        if (!empty($this->settings['trust_customers']) && function_exists('wc_get_customer_total_spent')) {
            $user_id = 0;
            if (function_exists('get_current_user_id')) { $user_id = (int) get_current_user_id(); }
            if ($user_id > 0) {
                $spent = (float) wc_get_customer_total_spent($user_id);
                if ($spent > 0) { return true; }
            }
        }
        return false;
    }
    private function ip_in_cidr($ip, $cidr) {
        $ip = trim((string) $ip);
        $cidr = trim((string) $cidr);
        if ($ip === '' || $cidr === '' || strpos($cidr, '/') === false) { return false; }
        list($net, $bits) = array_pad(explode('/', $cidr, 2), 2, '');
        $net = trim($net);
        $bits = intval($bits);
        $ip_bin = $this->ip_to_bin($ip_store);
        $net_bin = $this->ip_to_bin($net);
        if ($ip_bin === '' || $net_bin === '' || strlen($ip_bin) !== strlen($net_bin)) { return false; }
        $max_bits = strlen($ip_bin) * 8;
        if ($bits < 0) { $bits = 0; }
        if ($bits > $max_bits) { $bits = $max_bits; }
        $bytes = intdiv($bits, 8);
        $rem = $bits % 8;
        if ($bytes > 0 && substr($ip_bin, 0, $bytes) !== substr($net_bin, 0, $bytes)) { return false; }
        if ($rem === 0) { return true; }
        $mask = (0xFF << (8 - $rem)) & 0xFF;
        return (ord($ip_bin[$bytes]) & $mask) === (ord($net_bin[$bytes]) & $mask);
    }
    public static function get_client_ip() {

        $candidates = array();
        foreach (array('HTTP_CF_CONNECTING_IP','HTTP_X_REAL_IP','HTTP_X_FORWARDED_FOR','HTTP_CLIENT_IP') as $k) {
            if (isset($_SERVER[$k]) && is_string($_SERVER[$k]) && $_SERVER[$k] !== '') {
                $candidates[] = trim( sanitize_text_field( wp_unslash( (string) $_SERVER[ $k ] ) ) );
            }
        }
        if (isset($_SERVER['REMOTE_ADDR']) && is_string($_SERVER['REMOTE_ADDR']) && $_SERVER['REMOTE_ADDR'] !== '') {
            $candidates[] = trim( sanitize_text_field( wp_unslash( (string) $_SERVER['REMOTE_ADDR'] ) ) );
        }
        foreach ($candidates as $raw) {
            if ($raw === '') { continue; }

            $parts = preg_split('/\s*,\s*/', $raw);
            foreach ($parts as $p) {
                $ip = trim($p);
                if ($ip !== '' && filter_var($ip, FILTER_VALIDATE_IP)) { return $ip; }
            }
        }
        return '';
    }

public function firewall_check($ctx) {
    $this->settings = $this->get_effective_settings();

    $ip = isset($ctx['ip']) ? (string) $ctx['ip'] : '';
    $ua = isset($ctx['ua']) ? (string) $ctx['ua'] : '';

    $ipm = $this->apply_ip_mode($ip);
    $geo = $this->get_geo($ip);
    $fingerprint = $this->fingerprint('firewall', (string) $ipm['ip_for_fp'], $ua, '');

    $window = isset($this->settings['velocity_window_seconds']) ? intval($this->settings['velocity_window_seconds']) : 600;
    $this->bump_fingerprint($fingerprint, max(60, $window));

    $list = $this->list_decision($ip, '', $fingerprint, $geo);
    if ($list === 'deny') {
        return array('action' => 'block', 'reasons' => array('denylist_match'), 'meta' => array('fingerprint' => $fingerprint));
    }
    if ($list === 'allow') {
        return array('action' => 'allow', 'reasons' => array('allowlist_match'), 'meta' => array('fingerprint' => $fingerprint));
    }

    $threshold = isset($this->settings['firewall_hits_10m_threshold']) ? intval($this->settings['firewall_hits_10m_threshold']) : 120;

    global $wpdb;
    $fps = self::sanitize_db_prefix($wpdb->prefix) . 'asg_fingerprints';
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $row = $wpdb->get_row( $wpdb->prepare( 'SELECT hits_10m FROM %i WHERE fingerprint=%s', $fps, $fingerprint ), ARRAY_A );
    $hits10 = $row ? intval($row['hits_10m']) : 0;

    if ($hits10 >= ($threshold * 2)) {
        return array('action' => 'block', 'reasons' => array('firewall_velocity_hard'), 'meta' => array('fingerprint' => $fingerprint, 'hits_10m' => $hits10));
    }
    if ($hits10 >= $threshold) {
        $delay = min(2000, 250 + (($hits10 - $threshold) * 25));
        return array('action' => 'challenge', 'reasons' => array('firewall_velocity_soft'), 'meta' => array('fingerprint' => $fingerprint, 'hits_10m' => $hits10, 'delay_ms' => $delay));
    }

    return array('action' => 'allow', 'reasons' => array(), 'meta' => array('fingerprint' => $fingerprint, 'hits_10m' => $hits10));
}

    public function log_firewall_event($ctx, $fw) {
        global $wpdb;
        $table = $wpdb->prefix . 'asg_firewall_events';
        if (empty($table)) { return; }

        $ip = isset($ctx['ip']) ? (string) $ctx['ip'] : '';
        $ua = isset($ctx['ua']) ? (string) $ctx['ua'] : '';
        $uri = isset($ctx['uri']) ? (string) $ctx['uri'] : '';

        $ipm = $this->apply_ip_mode($ip);
        $ip_bin = isset($ipm['ip_bin']) ? $ipm['ip_bin'] : null;
        $ip_hash = isset($ipm['ip_hash']) ? (string) $ipm['ip_hash'] : null;

        $fingerprint = isset($fw['meta']['fingerprint']) ? (string) $fw['meta']['fingerprint'] : '';
        $ua_hash = hash('sha256', $ua);

        $reasons = isset($fw['reasons']) && is_array($fw['reasons']) ? wp_json_encode($fw['reasons']) : null;
        $meta = isset($fw['meta']) ? wp_json_encode($fw['meta']) : null;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->insert(
            $table,
            array(
                'created_at' => current_time('mysql'),
                'action' => isset($fw['action']) ? (string) $fw['action'] : 'allow',
                'reasons' => $reasons,
                'ip' => $ip_bin,
                'ip_hash' => $ip_hash,
                'fingerprint' => $fingerprint,
                'uri' => $uri,
                'ua_hash' => $ua_hash,
                'meta' => $meta,
            ),
            array('%s','%s','%s','%s','%s','%s','%s','%s','%s')
        );
    }

    public function get_firewall_events($args = array()) {
        global $wpdb;
        $table = $wpdb->prefix . 'asg_firewall_events';

        $limit  = isset($args['limit']) ? max(1, min(500, intval($args['limit']))) : 200;
        $offset = isset($args['offset']) ? max(0, intval($args['offset'])) : 0;
        $action = isset($args['action']) ? sanitize_text_field( wp_unslash( (string) $args['action'] ) ) : '';
        $q      = isset($args['q']) ? sanitize_text_field( wp_unslash( (string) $args['q'] ) ) : '';

        if ($q !== '') {
            $like = '%' . $wpdb->esc_like($q) . '%';
        } else {
            $like = '';
        }

        if ($action === '' && $q === '') {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    'SELECT * FROM %i ORDER BY id DESC LIMIT %d OFFSET %d',
                    $table,
                    (int) $limit,
                    (int) $offset
                ),
                ARRAY_A
            );
        } elseif ($action !== '' && $q === '') {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    'SELECT * FROM %i WHERE action=%s ORDER BY id DESC LIMIT %d OFFSET %d',
                    $table,
                    $action,
                    (int) $limit,
                    (int) $offset
                ),
                ARRAY_A
            );
        } elseif ($action === '' && $q !== '') {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    'SELECT * FROM %i WHERE (uri LIKE %s OR ip_hash LIKE %s OR fingerprint LIKE %s) ORDER BY id DESC LIMIT %d OFFSET %d',
                    $table,
                    $like,
                    $like,
                    $like,
                    (int) $limit,
                    (int) $offset
                ),
                ARRAY_A
            );
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $rows = $wpdb->get_results(
                $wpdb->prepare(
                    'SELECT * FROM %i WHERE action=%s AND (uri LIKE %s OR ip_hash LIKE %s OR fingerprint LIKE %s) ORDER BY id DESC LIMIT %d OFFSET %d',
                    $table,
                    $action,
                    $like,
                    $like,
                    $like,
                    (int) $limit,
                    (int) $offset
                ),
                ARRAY_A
            );
        }

        return is_array($rows) ? $rows : array();
    }


}