<?php
if (!defined('ABSPATH')) { exit; }

require_once ASG_PLUGIN_DIR . 'includes/class-asg-engine.php';
require_once ASG_PLUGIN_DIR . 'includes/class-asg-admin.php';
require_once ASG_PLUGIN_DIR . 'includes/class-asg-ml.php';

final class ASG_Core {
    private static $instance = null;

    public $engine;

    public $admin;

    public $ml;

    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->ml     = new ASG_ML();
        $this->engine = new ASG_Engine($this->ml);

        if (is_admin()) {
            $this->admin = new ASG_Admin($this->engine, $this->ml);
        }

        self::clm_register_free_install();
        self::clm_maybe_refresh_license();
        add_filter('pre_set_site_transient_update_plugins', array($this, 'clm_update_transient'));

        add_filter('preprocess_comment', array($this, 'filter_comment_preprocess'), 1);
        add_action('wp_insert_comment', array($this, 'action_wp_insert_comment'), 10, 2);
        add_action('user_register', array($this, 'action_user_register'), 10, 1);
        add_action('wpcf7_before_send_mail', array($this, 'cf7_before_send_mail'), 10, 1);
        add_action('woocommerce_after_checkout_validation', array($this, 'woo_after_checkout_validation'), 10, 2);
        add_filter('registration_errors', array($this, 'filter_registration'), 10, 3);
        add_filter('wpcf7_form_elements', array($this, 'cf7_inject_fields'), 10, 1);
        add_filter('wpcf7_validate', array($this, 'cf7_validate'), 5, 2);
        add_filter('rest_pre_dispatch', array($this, 'filter_rest_pre_dispatch'), 5, 3);
        add_filter('wpforms_process_initial_errors', array($this, 'wpforms_initial_errors'), 10, 2);
        add_filter('gform_validation', array($this, 'gravity_validation'), 10, 1);
        add_action('template_redirect', array($this, 'firewall_precheck'), 0);
        add_action('init', array($this, 'generic_post_guard'), 1);
        add_action('pre_get_posts', array($this, 'protect_search_query'), 1);

add_filter('wpforms_frontend_output', array($this, 'wpforms_inject_fields'), 10, 3);
add_filter('gform_get_form_filter', array($this, 'gravity_inject_fields'), 10, 2);

add_action('wp_ajax_nopriv_asg_mint_nonce', array($this, 'ajax_mint_nonce'));
add_action('wp_ajax_asg_mint_nonce', array($this, 'ajax_mint_nonce'));
add_action('wp_footer', array($this, 'aegisform_footer_injector'), 99);

    }

    public static function activate() {
        ASG_Engine::install_schema();
        self::ensure_defaults();
        self::ensure_site_salt();
        if (is_multisite()) { self::ensure_network_defaults(); }

        // Register free install with CLM (Aegisify)
        self::clm_register_free_install();
        // Refresh license status periodically (anti-fraud)
        self::clm_maybe_refresh_license();
    }

    public static function deactivate() {
    }

    private static function ensure_defaults() {
        $defaults = array(
            'threshold_allow' => 29,
            'threshold_challenge' => 59,
            'threshold_hold' => 79,
            'threshold_block' => 100,
            'min_seconds_to_submit' => 3,
            'velocity_window_seconds' => 600,
            'velocity_limit' => 5,
            'enable_ml' => 1,
            'ml_weight' => 12,
            'enable_similarity' => 1,
            'similarity_lookback_days' => 30,
            'similarity_threshold' => 0.72,
            'enable_mx_checks' => 0,
            'enable_external_disposable' => 0,
            'external_disposable_url' => '',
            'fp_hold_instead_of_block' => 1,
            'trust_logged_in' => 1,
            'trust_customers' => 0,
            'trust_existing_user_days' => 7,
            'enable_cf7_captcha' => 0,
            'w_link_count' => 'med',
            'w_spam_keywords' => 'med',
            'w_link_markup' => 'med',
            'w_url_shortener' => 'med',
            'w_honeypot' => 'high',
            'w_too_fast' => 'med',
            'w_js_proof' => 'med',
            'w_invalid_email' => 'med',
            'w_disposable_email' => 'med',
            'w_mx_missing' => 'med',
            'w_name_email_mismatch' => 'med',

            'mod_fingerprinting' => 1,
            'mod_behavior' => 1,
            'mod_content' => 1,
            'mod_identity' => 1,
            'mod_geo_asn' => 1,
            'mod_challenges' => 1,
            'enable_progressive_delay' => 1,
            'progressive_delay_min_ms' => 150,
            'progressive_delay_max_ms' => 1800,
            'log_retention_days' => 90,
            'ip_mode' => 'store',
            'no_external_calls' => 1,
            'geo_provider' => 'headers',
            'maxmind_country_mmdb' => '',
            'maxmind_asn_mmdb' => '',
            'enable_geo_rules' => 0,
            'enable_asn_rules' => 0,
            'enable_generic_post_protection' => 0,
            'generic_post_block_on_challenge' => 0,
            'enable_generic_post_include_ajax' => 0,
            'enable_generic_post_include_admin_post' => 0,
            'generic_post_allow_ajax_actions' => "asg_mint_nonce\n",
            'generic_post_allow_admin_post_actions' => "",
            'generic_post_allow_paths' => "",

            'enable_firewall_log_stream' => 1,
        );
        $opt = get_option('asg_settings', array());
        if (!is_array($opt)) { $opt = array(); }
        $opt = array_merge($defaults, $opt);
        update_option('asg_settings', $opt, false);
        $ml = get_option('asg_ml_totals', array('spam_docs'=>0,'ham_docs'=>0));
        if (!is_array($ml) || !isset($ml['spam_docs']) || !isset($ml['ham_docs'])) {
            update_option('asg_ml_totals', array('spam_docs'=>0,'ham_docs'=>0), false);
        }
    }

    public static function ensure_site_salt() {
        $salt = get_option('asg_site_salt', '');
        if (!is_string($salt) || strlen($salt) < 24) {
            $salt = wp_generate_password(48, true, true);
            update_option('asg_site_salt', $salt, true);
        }
        return $salt;
    }

    public function filter_comment_preprocess($commentdata) {
        $content = isset($commentdata['comment_content']) ? (string) $commentdata['comment_content'] : '';
        $email   = isset($commentdata['comment_author_email']) ? (string) $commentdata['comment_author_email'] : '';
        $author  = isset($commentdata['comment_author']) ? (string) $commentdata['comment_author'] : '';
        $url     = isset($commentdata['comment_author_url']) ? (string) $commentdata['comment_author_url'] : '';

        $event = array(
            'type' => 'comment',
            'email' => $email,
            'name' => $author,
            'content' => $content,
            'url' => $url,
            'meta' => array(
                'post_id' => isset($commentdata['comment_post_ID']) ? intval($commentdata['comment_post_ID']) : 0,
            ),
        );

        $decision = $this->engine->evaluate($event);

        if ($decision['action'] === 'block') {
            wp_die(esc_html__('Spam detected. Please try again later.', 'aegis-spam-guard'), 403);
        }

        if ($decision['action'] === 'hold') {
            $commentdata['comment_approved'] = 0;
        }

        if ($decision['action'] === 'challenge') {
            $commentdata['comment_approved'] = 0;
        }

        return $commentdata;
    }

    public function filter_registration($errors, $sanitized_user_login, $user_email) {
        $event = array(
            'type' => 'registration',
            'email' => (string) $user_email,
            'name' => (string) $sanitized_user_login,
            'content' => '',
            'url' => '',
            'meta' => array(),
        );

        $decision = $this->engine->evaluate($event);

        if ($decision['action'] === 'block') {
            $errors->add('asg_spam_block', esc_html__('Registration blocked as spam. Please contact the site administrator if this is an error.', 'aegis-spam-guard'));
        } elseif ($decision['action'] === 'hold' || $decision['action'] === 'challenge') {
            $errors->add('asg_spam_hold', esc_html__('Registration requires verification due to suspicious activity. Please try again later.', 'aegis-spam-guard'));
        }

        return $errors;
    }

    public function filter_rest_pre_dispatch($result, $server, $request) {

        if ($result !== null) { return $result; }

        if (!($request instanceof WP_REST_Request)) { return $result; }

        $method = strtoupper($request->get_method());
        if ($method === 'GET' || $method === 'HEAD' || $method === 'OPTIONS') { return $result; }

        if (is_user_logged_in()) { return $result; }

        $params = $request->get_params();
        $body = '';
        if (is_array($params)) {

            foreach (array('message','content','comment','body','text','query') as $k) {
                if (isset($params[$k]) && is_string($params[$k]) && $params[$k] !== '') { $body = $params[$k]; break; }
            }
            if ($body === '') {
                $body = wp_json_encode(array_slice($params, 0, 40));
            }
        }

        $path = $request->get_route();

        $event = array(
            'type' => 'rest',
            'actor' => array(
                'ip' => isset($_SERVER['REMOTE_ADDR']) ? (string) $_SERVER['REMOTE_ADDR'] : '',
                'user_id' => 0,
                'email_hash' => null,
                'session_id' => '',
            ),
            'payload' => array(
                'fields' => is_array($params) ? $params : array(),
                'message' => (string) $body,
                'urls' => array(),
                'metadata' => array(
                    'route' => (string) $path,
                    'method' => (string) $method,
                ),
            ),
        );

        $decision = $this->engine->evaluate($event);

        if ($decision['action'] === 'block') {
            return new WP_Error('asg_rest_blocked', esc_html__('Request blocked by AegisSpamGuard.', 'aegis-spam-guard'), array('status' => 403));
        }
        if ($decision['action'] === 'hold' || $decision['action'] === 'challenge') {
            return new WP_Error('asg_rest_challenge', esc_html__('Request requires verification.', 'aegis-spam-guard'), array('status' => 403));
        }

        return $result;
    }

    public function wpforms_initial_errors($errors = array(), $form_data = array()) {
        if (!is_array($errors)) { $errors = array(); }

        $has_existing = !empty($errors);

        $email = isset($_POST['wpforms']['fields']) && is_array($_POST['wpforms']['fields']) ? '' : '';
        $name = '';
        $content = '';

        if (isset($_POST['wpforms']['fields']) && is_array($_POST['wpforms']['fields'])) {
            foreach ($_POST['wpforms']['fields'] as $val) {
                if (!is_array($val)) { continue; }
                $v = isset($val['value']) ? (string) $val['value'] : '';
                $k = isset($val['name']) ? strtolower((string) $val['name']) : '';
                if ($email === '' && (strpos($k, 'email') !== false)) { $email = $v; }
                if ($name === '' && (strpos($k, 'name') !== false)) { $name = $v; }
                if ($content === '' && (strpos($k, 'message') !== false || strpos($k, 'comment') !== false)) { $content = $v; }
            }
        }

$meta = array();
if (isset($_POST['asg_ts'])) { $meta['asg_ts'] = sanitize_text_field(wp_unslash($_POST['asg_ts'])); }
if (isset($_POST['asg_nonce'])) { $meta['asg_nonce'] = sanitize_text_field(wp_unslash($_POST['asg_nonce'])); }
if (isset($_POST['asg_js'])) { $meta['asg_js'] = sanitize_text_field(wp_unslash($_POST['asg_js'])); }
if (isset($_POST['asg_hp'])) { $meta['asg_hp'] = sanitize_text_field(wp_unslash($_POST['asg_hp'])); }
		$event = array(
			'type' => 'form',
			'actor' => array(
				'ip' => isset($_SERVER['REMOTE_ADDR']) ? (string) $_SERVER['REMOTE_ADDR'] : '',
				'user_id' => 0,
				'session_id' => '',
			),
			'payload' => array(
				'fields' => isset($_POST['wpforms']) ? $_POST['wpforms'] : array(),
				'message' => (string) $content,
				'metadata' => array_merge(array('adapter' => 'wpforms'), $meta),
				'email' => (string) $email,
				'name' => (string) $name,
			),
		);

        $decision = $this->engine->evaluate($event);
        if ($decision['action'] === 'block') {
            $errors['asg_block'] = esc_html__('Submission blocked by AegisSpamGuard.', 'aegis-spam-guard');
        } elseif (($decision['action'] === 'hold' || $decision['action'] === 'challenge') && !$has_existing) {
            $errors['asg_hold'] = esc_html__('Submission requires verification.', 'aegis-spam-guard');
        }

        return $errors;
    }

    public function gravity_validation($validation_result) {
        if (!is_array($validation_result) || !isset($validation_result['form'])) { return $validation_result; }
        if (!isset($_POST) || !is_array($_POST)) { return $validation_result; }

        $name = '';
        $email = '';
        $content = '';

        foreach ($_POST as $k=>$v) {
            if (!is_string($v)) { continue; }
            $lk = strtolower((string) $k);
            if ($email === '' && strpos($lk, 'email') !== false) { $email = $v; }
            if ($name === '' && strpos($lk, 'name') !== false) { $name = $v; }
            if ($content === '' && (strpos($lk, 'message') !== false || strpos($lk, 'comment') !== false || strpos($lk, 'text') !== false)) { $content = $v; }
        }

$meta = array();
if (isset($_POST['asg_ts'])) { $meta['asg_ts'] = sanitize_text_field(wp_unslash($_POST['asg_ts'])); }
if (isset($_POST['asg_nonce'])) { $meta['asg_nonce'] = sanitize_text_field(wp_unslash($_POST['asg_nonce'])); }
if (isset($_POST['asg_js'])) { $meta['asg_js'] = sanitize_text_field(wp_unslash($_POST['asg_js'])); }
if (isset($_POST['asg_hp'])) { $meta['asg_hp'] = sanitize_text_field(wp_unslash($_POST['asg_hp'])); }
		$event = array(
			'type' => 'form',
			'actor' => array(
				'ip' => isset($_SERVER['REMOTE_ADDR']) ? (string) $_SERVER['REMOTE_ADDR'] : '',
				'user_id' => 0,
				'session_id' => '',
			),
			'payload' => array(
				'fields' => $_POST,
				'message' => (string) $content,
				'metadata' => array_merge(array('adapter' => 'gravityforms'), $meta),
				'email' => (string) $email,
				'name' => (string) $name,
			),
		);

        $decision = $this->engine->evaluate($event);
        if ($decision['action'] === 'block') {
            $validation_result['is_valid'] = false;
            if (isset($validation_result['form']['fields']) && is_array($validation_result['form']['fields'])) {
                foreach ($validation_result['form']['fields'] as &$field) {
                    if (!isset($field->failed_validation) || $field->failed_validation) { continue; }
                    $field->failed_validation = true;
                    $field->validation_message = esc_html__('Submission blocked by AegisSpamGuard.', 'aegis-spam-guard');
                    break;
                }
            }
        }

        return $validation_result;
    }

function cf7_inject_fields($form) {
        $nonce = wp_generate_password(20, false, false);
        $ts = time();
        $key = 'asg_nonce_' . md5($nonce);
        set_transient($key, array('ts'=>$ts), 20 * MINUTE_IN_SECONDS);

        $hp = '<span class="asg-hp" style="position:absolute;left:-9999px;top:-9999px;height:0;width:0;overflow:hidden;" aria-hidden="true">'
            . '<label>Leave this field empty<input type="text" name="asg_hp" value="" autocomplete="off" tabindex="-1"></label>'
            . '</span>';

        $opt = get_option('asg_settings', array());
        $captcha = '';
        if (!empty($opt['enable_cf7_captcha'])) {
            $a = rand(2, 9);
            $b = rand(1, 9);
            $ans = $a + $b;
            $cap_key = 'asg_captcha_' . md5($nonce);
            set_transient($cap_key, array('ans'=>$ans), 20 * MINUTE_IN_SECONDS);
            $captcha = '<p class="asg-captcha"><label>Anti-spam: What is ' . intval($a) . ' + ' . intval($b) . '? '
                . '<input type="text" name="asg_ca" value="" size="4" inputmode="numeric" autocomplete="off"></label>'
                . '<input type="hidden" name="asg_cn" value="' . esc_attr($nonce) . '"></p>';
        }

        $hidden = sprintf(
            '<input type="hidden" name="asg_ts" value="%d"><input type="hidden" name="asg_nonce" value="%s"><input type="hidden" name="asg_js" value="">',
            intval($ts),
            esc_attr($nonce)
        );

        $js = '<script>(function(){try{var f=document.currentScript&&document.currentScript.parentElement;'
            . 'var forms=document.querySelectorAll("form.wpcf7-form");'
            . 'for(var i=0;i<forms.length;i++){var frm=forms[i];'
            . 'var nonce=frm.querySelector("input[name=asg_nonce]");'
            . 'var out=frm.querySelector("input[name=asg_js]");'
            . 'if(!nonce||!out||out.value){continue;}'
            . 'var ua=navigator.userAgent||"";'
            . 'var data=(nonce.value+"|"+ua);'
            . 'if(window.crypto&&crypto.subtle){'
            . 'var enc=new TextEncoder();'
            . 'crypto.subtle.digest("SHA-256", enc.encode(data)).then(function(buf){'
            . 'var arr=Array.from(new Uint8Array(buf));'
            . 'out.value=arr.map(function(b){return ("00"+b.toString(16)).slice(-2);}).join("");'
            . '}).catch(function(){});'
            . '} else {'
            . 'var h=0; for (var j=0;j<data.length;j++){h=((h<<5)-h)+data.charCodeAt(j); h|=0;} out.value=(""+h);'
            . '}'
            . '} }catch(e){} })();</script>';

        return $form . $hp . $hidden . $captcha . $js;
    }

    public function cf7_validate($result, $tags) {
        $submission = function_exists('WPCF7_Submission::get_instance') ? WPCF7_Submission::get_instance() : null;
        if (!$submission) { return $result; }

        $posted = (array) $submission->get_posted_data();

        $event = array(
            'type' => 'cf7',
            'email' => isset($posted['your-email']) ? (string) $posted['your-email'] : '',
            'name' => isset($posted['your-name']) ? (string) $posted['your-name'] : '',
            'content' => isset($posted['your-message']) ? (string) $posted['your-message'] : '',
            'url' => isset($_SERVER['HTTP_REFERER']) ? (string) $_SERVER['HTTP_REFERER'] : '',
            'meta' => array(
                'posted_keys' => array_keys($posted),
                'hp' => isset($posted['asg_hp']) ? (string) $posted['asg_hp'] : '',
                'asg_ts' => isset($posted['asg_ts']) ? intval($posted['asg_ts']) : 0,
                'asg_nonce' => isset($posted['asg_nonce']) ? (string) $posted['asg_nonce'] : '',
                'asg_js' => isset($posted['asg_js']) ? (string) $posted['asg_js'] : '',
            ),
        );

        $decision = $this->engine->evaluate($event);

        if ($decision['action'] === 'block') {
            if (is_object($result) && method_exists($result, 'invalidate')) {
                $first_tag = null;
                if (is_array($tags) && isset($tags[0])) { $first_tag = $tags[0]; }
                $result->invalidate($first_tag, esc_html__('Spam detected. Please try again.', 'aegis-spam-guard'));
            }
        } elseif ($decision['action'] === 'hold' || $decision['action'] === 'challenge') {
            if (is_object($result) && method_exists($result, 'invalidate')) {
                $first_tag = null;
                if (is_array($tags) && isset($tags[0])) { $first_tag = $tags[0]; }
                $result->invalidate($first_tag, esc_html__('Please retry in a moment (verification required).', 'aegis-spam-guard'));
            }
        }

        return $result;
    }

    public function woo_checkout_validate($data, $errors) {
        if (!is_object($errors) || !method_exists($errors, 'add')) { return; }

        $email = isset($data['billing_email']) ? sanitize_email($data['billing_email']) : '';
        $name  = trim((string) (isset($data['billing_first_name']) ? $data['billing_first_name'] : '') . ' ' . (isset($data['billing_last_name']) ? $data['billing_last_name'] : ''));
        $note  = isset($data['order_comments']) ? (string) $data['order_comments'] : '';

        $event = array(
            'type' => 'woo',
            'email' => $email,
            'name' => $name,
            'content' => $note,
            'meta' => array(
                'fields' => array(
                    'billing_email' => $email,
                    'billing_first_name' => isset($data['billing_first_name']) ? (string) $data['billing_first_name'] : '',
                    'billing_last_name' => isset($data['billing_last_name']) ? (string) $data['billing_last_name'] : '',
                ),
            ),
        );

        $decision = $this->engine->evaluate($event);
        if (!empty($decision['action']) && in_array($decision['action'], array('block','hold','challenge'), true)) {
            $msg = 'Your checkout was flagged as potential spam. Please review your info and try again.';
            $errors->add('asg_spam', $msg);
        }
    }

    public function action_wp_insert_comment($comment_id, $comment) {
        if (empty($comment_id)) { return; }
        $content = is_object($comment) ? (string) $comment->comment_content : '';
        $email   = is_object($comment) ? (string) $comment->comment_author_email : '';
        $author  = is_object($comment) ? (string) $comment->comment_author : '';
        $event = array(
            'type' => 'comment',
            'actor' => array(
                'ip' => ASG_Engine::get_client_ip(),
                'user_id' => is_object($comment) ? intval($comment->user_id) : 0,
                'email_hash' => ASG_Engine::hash_email($email),
                'session_id' => '',
            ),
            'payload' => array(
                'fields' => array('author' => $author),
                'message' => $content,
                'urls' => array(),
                'metadata' => array('stage' => 'post_insert', 'comment_id' => intval($comment_id)),
            ),
        );
        try { $this->engine->evaluate($event, array('enforce' => false)); } catch (Exception $e) { /* ignore */ }
    }

    public function action_user_register($user_id) {
        $user_id = intval($user_id);
        if ($user_id <= 0) { return; }
        $u = get_userdata($user_id);
        if (!$u) { return; }
        $email = (string) $u->user_email;
        $name  = trim((string) $u->display_name);
        $event = array(
            'type' => 'registration',
            'actor' => array(
                'ip' => ASG_Engine::get_client_ip(),
                'user_id' => $user_id,
                'email_hash' => ASG_Engine::hash_email($email),
                'session_id' => '',
            ),
            'payload' => array(
                'fields' => array('name' => $name),
                'message' => '',
                'urls' => array(),
                'metadata' => array('stage' => 'post_register'),
            ),
        );
        try { $this->engine->evaluate($event, array('enforce' => false)); } catch (Exception $e) { /* ignore */ }
    }

    public function cf7_before_send_mail($contact_form) {
        if (!is_object($contact_form)) { return; }
        $submission = function_exists('WPCF7_Submission::get_instance') ? WPCF7_Submission::get_instance() : null;
        if (!$submission) { return; }
        $data = $submission->get_posted_data();
        $email = '';
        $msg = '';
        foreach ($data as $k=>$v) {
            if ($email === '' && is_string($v) && preg_match('/@/', $v)) { $email = (string)$v; }
            if ($msg === '' && is_string($v) && strlen($v) > 20) { $msg = (string)$v; }
        }
        $event = array(
            'type' => 'form',
            'actor' => array(
                'ip' => ASG_Engine::get_client_ip(),
                'user_id' => get_current_user_id(),
                'email_hash' => ASG_Engine::hash_email($email),
                'session_id' => '',
            ),
            'payload' => array(
                'fields' => $data,
                'message' => $msg,
                'urls' => array(),
                'metadata' => array('form' => 'cf7', 'stage' => 'before_send_mail'),
            ),
        );
        try { $this->engine->evaluate($event, array('enforce' => false)); } catch (Exception $e) { /* ignore */ }
    }

    public function woo_after_checkout_validation($data, $errors) {
        if (!is_array($data) || !is_object($errors)) { return; }
        $email = isset($data['billing_email']) ? (string)$data['billing_email'] : '';
        $name  = trim((string)(isset($data['billing_first_name']) ? $data['billing_first_name'] : '') . ' ' . (isset($data['billing_last_name']) ? $data['billing_last_name'] : ''));
        $note  = isset($data['order_comments']) ? (string)$data['order_comments'] : '';
        $event = array(
            'type' => 'checkout',
            'actor' => array(
                'ip' => ASG_Engine::get_client_ip(),
                'user_id' => get_current_user_id(),
                'email_hash' => ASG_Engine::hash_email($email),
                'session_id' => '',
            ),
            'payload' => array(
                'fields' => array('name' => $name, 'email' => $email),
                'message' => $note,
                'urls' => array(),
                'metadata' => array('stage' => 'checkout_validation'),
            ),
        );

        try {
            $res = $this->engine->evaluate($event, array('enforce' => false));
            if (is_array($res) && isset($res['action']) && $res['action'] === 'block') {
                $errors->add('asg_spam', esc_html__('Checkout blocked by AegisSpamGuard (spam risk).', 'aegis-spam-guard'));
            }
        } catch (Exception $e) { /* ignore */ }
    }


	private function build_jsproof_bundle_html() {
		$nonce = wp_generate_password(20, false, false);
		$ts    = time();
		$key   = 'asg_nonce_' . md5($nonce);

		set_transient($key, array('ts' => $ts), 20 * MINUTE_IN_SECONDS);

		$hp = '<span class="asg-hp" style="position:absolute;left:-9999px;top:-9999px;height:0;width:0;overflow:hidden;" aria-hidden="true">'
			. '<label>Leave this field empty'
			. '<input type="text" name="asg_hp" value="" autocomplete="off" tabindex="-1" />'
			. '</label>'
			. '</span>';

		$hidden = sprintf(
			'<input type="hidden" name="asg_ts" value="%d" />'
			. '<input type="hidden" name="asg_nonce" value="%s" />'
			. '<input type="hidden" name="asg_js" value="" />',
			intval($ts),
			esc_attr($nonce)
		);

		$js = '<script>(function(){try{'
			. 'var s=document.currentScript;'
			. 'var root=s&&s.parentNode?s.parentNode:document;'
			. 'var nonceEl=root.querySelector(\'input[name="asg_nonce"]\');'
			. 'var outEl=root.querySelector(\'input[name="asg_js"]\');'
			. 'if(!nonceEl||!outEl) return;'
			. 'var ua=(navigator.userAgent||"");'
			. 'var data=(nonceEl.value+"|"+ua);'
			. 'if(window.crypto&&crypto.subtle&&window.TextEncoder){'
			. 'var enc=new TextEncoder();'
			. 'crypto.subtle.digest("SHA-256", enc.encode(data)).then(function(buf){'
			. 'var arr=Array.from(new Uint8Array(buf));'
			. 'outEl.value=arr.map(function(b){return("00"+b.toString(16)).slice(-2);}).join("");'
			. '}).catch(function(){});'
			. '} else {'
			. 'var h=0; for (var j=0;j<data.length;j++){h=((h<<5)-h)+data.charCodeAt(j); h|=0;}'
			. 'outEl.value=(""+h);'
			. '}'
			. '}catch(e){} })();</script>';

		return $hp . $hidden . $js;
	}

    public function wpforms_inject_fields($output, $form_data = array(), $context = null) {
        if (!is_string($output) || $output === '') { return $output; }
        if (strpos($output, 'name="asg_nonce"') !== false) { return $output; }

        $bundle = $this->build_jsproof_bundle_html();
        if (stripos($output, '</form>') !== false) {
            return preg_replace('~</form>~i', $bundle . '</form>', $output, 1);
        }
        return $output . $bundle;
    }

    public function gravity_inject_fields($form_string, $form = null) {
        if (!is_string($form_string) || $form_string === '') { return $form_string; }
        if (strpos($form_string, 'name="asg_nonce"') !== false) { return $form_string; }

        $bundle = $this->build_jsproof_bundle_html();
        if (stripos($form_string, '</form>') !== false) {
            return preg_replace('~</form>~i', $bundle . '</form>', $form_string, 1);
        }
        return $form_string . $bundle;
    }

    public function ajax_mint_nonce() {

        $nonce = wp_generate_password(20, false, false);
        $ts = time();
        $key = 'asg_nonce_' . md5($nonce);
        set_transient($key, array('ts'=>$ts), 20 * MINUTE_IN_SECONDS);

        wp_send_json_success(array(
            'nonce' => $nonce,
            'ts'    => $ts,
        ));
    }

    public function aegisform_footer_injector() {

        if (is_admin()) { return; }
        $ajax = admin_url('admin-ajax.php');
        ?>
<script>
(function(){
  try{
    var forms = document.querySelectorAll('form.aegisform, form.aegis-form, form[data-aegisform="1"], form[data-aegisform="true"]');
    if(!forms || !forms.length) return;
    var ajax = <?php echo json_encode($ajax); ?>;
    function shaFallback(data){
      var h=0; for (var j=0;j<data.length;j++){h=((h<<5)-h)+data.charCodeAt(j); h|=0;} return (""+h);
    }
    function inject(form, bundle){
      if(form.querySelector('input[name="asg_nonce"]')) return;
      var hp=document.createElement('span');
      hp.className='asg-hp';
      hp.style.cssText='position:absolute;left:-9999px;top:-9999px;height:0;width:0;overflow:hidden;';
      hp.innerHTML='<label>Leave this field empty<input type="text" name="asg_hp" value="" autocomplete="off" tabindex="-1"></label>';
      form.appendChild(hp);

      var ts=document.createElement('input'); ts.type='hidden'; ts.name='asg_ts'; ts.value=bundle.ts;
      var n=document.createElement('input'); n.type='hidden'; n.name='asg_nonce'; n.value=bundle.nonce;
      var js=document.createElement('input'); js.type='hidden'; js.name='asg_js'; js.value='';
      form.appendChild(ts); form.appendChild(n); form.appendChild(js);

      var ua=(navigator.userAgent||"");
      var data=(bundle.nonce+"|"+ua);
      if(window.crypto && crypto.subtle && window.TextEncoder){
        crypto.subtle.digest("SHA-256", (new TextEncoder()).encode(data)).then(function(buf){
          var arr=Array.from(new Uint8Array(buf));
          js.value=arr.map(function(b){return ("00"+b.toString(16)).slice(-2);}).join("");
        }).catch(function(){ js.value=shaFallback(data); });
      } else {
        js.value=shaFallback(data);
      }
    }
    function mintThenInject(form){
      var xhr=new XMLHttpRequest();
      xhr.open('POST', ajax, true);
      xhr.setRequestHeader('Content-Type','application/x-www-form-urlencoded; charset=UTF-8');
      xhr.onreadystatechange=function(){
        if(xhr.readyState!==4) return;
        try{
          var res=JSON.parse(xhr.responseText||'{}');
          if(res && res.success && res.data){ inject(form, res.data); }
        }catch(e){}
      };
      xhr.send('action=asg_mint_nonce');
    }
    for(var i=0;i<forms.length;i++){ mintThenInject(forms[i]); }
  }catch(e){}
})();
</script>
<?php
    }

public function generic_post_guard() {

	$s = get_option('asg_settings', array());
	if (!is_array($s)) { $s = array(); }

	if (is_multisite()) {
		$net = function_exists('get_site_option') ? get_site_option('asg_network_settings', array()) : array();
		if (is_array($net) && !empty($net)) {
			$s = array_merge($net, $s);
		}
	}

    if (empty($s['enable_generic_post_protection'])) { return; }
    if (!isset($_SERVER['REQUEST_METHOD']) || strtoupper((string) $_SERVER['REQUEST_METHOD']) !== 'POST') { return; }

    if (!empty($GLOBALS['asg_request_guard_ran'])) { return; }
    $GLOBALS['asg_request_guard_ran'] = true;

    if ((defined('DOING_CRON') && DOING_CRON) || (defined('REST_REQUEST') && REST_REQUEST)) { return; }

    $uri  = isset($_SERVER['REQUEST_URI']) ? (string) $_SERVER['REQUEST_URI'] : '';
    $path = (string) wp_parse_url($uri, PHP_URL_PATH);

    $core_skip = array('wp-login.php', 'wp-signup.php', 'wp-activate.php', 'xmlrpc.php', 'wp-comments-post.php');
    foreach ($core_skip as $needle) {
        if (strpos($path, $needle) !== false) { return; }
    }

    if (!empty($_POST['_wpcf7']) || !empty($_POST['wpforms']) || !empty($_POST['gform_submit'])) { return; }

    if (!empty($s['generic_post_allow_paths']) && $this->asg_path_is_allowlisted($path, (string) $s['generic_post_allow_paths'])) {
        return;
    }

    $is_ajax = (function_exists('wp_doing_ajax') && wp_doing_ajax()) || (strpos($path, 'admin-ajax.php') !== false);
    $is_admin_post = (strpos($path, 'admin-post.php') !== false);

    if ($is_ajax && empty($s['enable_generic_post_include_ajax'])) { return; }
    if ($is_admin_post && empty($s['enable_generic_post_include_admin_post'])) { return; }

    if ($is_ajax || $is_admin_post) {
        $action = '';
        if (isset($_REQUEST['action']) && is_string($_REQUEST['action'])) { $action = trim((string) $_REQUEST['action']); }
        if ($action === '') { return; }

        if ($is_ajax && $this->asg_action_is_allowlisted($action, (string) $s['generic_post_allow_ajax_actions'])) { return; }
        if ($is_admin_post && $this->asg_action_is_allowlisted($action, (string) $s['generic_post_allow_admin_post_actions'])) { return; }
    }

    $email = '';
    $name  = '';
    $msg   = '';
    if (isset($_POST['email']) && is_string($_POST['email'])) { $email = (string) $_POST['email']; }
    if (isset($_POST['name']) && is_string($_POST['name'])) { $name = (string) $_POST['name']; }

    foreach (array('message','comment','content','body','note','text','details') as $k) {
        if (isset($_POST[$k]) && is_string($_POST[$k]) && trim((string)$_POST[$k]) !== '') {
            $msg = (string) $_POST[$k];
            break;
        }
    }

    $meta = array(
        'adapter' => 'generic_post',
        'path' => $path,
        'is_ajax' => $is_ajax ? 1 : 0,
        'is_admin_post' => $is_admin_post ? 1 : 0,
    );

    foreach (array('asg_ts','asg_nonce','asg_js','asg_hp') as $k) {
        if (isset($_POST[$k]) && is_string($_POST[$k])) { $meta[$k] = (string) $_POST[$k]; }
    }

    $event = array(
        'type' => 'form',
        'actor' => array(
            'ip' => isset($_SERVER['REMOTE_ADDR']) ? (string) $_SERVER['REMOTE_ADDR'] : '',
            'user_id' => is_user_logged_in() ? get_current_user_id() : 0,
            'session_id' => '',
        ),
        'payload' => array(
            'fields' => $_POST,
            'message' => $msg,
            'metadata' => $meta,
            'email' => $email,
            'name' => $name,
        ),
    );

    $res = $this->engine->score_event($event);
    if (!is_array($res) || empty($res['action'])) { return; }

    $action = (string) $res['action'];

    $block_on_challenge = !empty($s['generic_post_block_on_challenge']);

    if ($action === 'block' || $action === 'hold' || ($block_on_challenge && $action === 'challenge')) {
        wp_die(
            esc_html__('Submission blocked by AegisSpamGuard.', 'aegis-spam-guard'),
            esc_html__('Blocked', 'aegis-spam-guard'),
            array('response' => 403)
        );
    }
}

private function asg_action_is_allowlisted($action, $raw_lines) {
    $action = trim((string) $action);
    if ($action === '') { return false; }
    $lines = preg_split('/\r\n|\r|\n/', (string) $raw_lines);
    if (!is_array($lines)) { return false; }
    foreach ($lines as $ln) {
        $ln = trim((string) $ln);
        if ($ln === '' || $ln[0] === '#') { continue; }
        if (strcasecmp($ln, $action) === 0) { return true; }
    }
    return false;
}

private function asg_path_is_allowlisted($path, $raw_lines) {
    $path = (string) $path;
    $lines = preg_split('/\r\n|\r|\n/', (string) $raw_lines);
    if (!is_array($lines)) { return false; }
    foreach ($lines as $ln) {
        $ln = trim((string) $ln);
        if ($ln === '' || $ln[0] === '#') { continue; }
        if (strpos($path, $ln) !== false) { return true; }
    }
    return false;
}

public function protect_search_query($query) {
    if (!is_a($query, 'WP_Query')) { return; }
    if (is_admin() || !$query->is_main_query()) { return; }
    if (!$query->is_search()) { return; }

	$s = get_option('asg_settings', array());
	if (!is_array($s)) { $s = array(); }

	if (is_multisite()) {
		$net = function_exists('get_site_option') ? get_site_option('asg_network_settings', array()) : array();
		if (is_array($net) && !empty($net)) {
			$s = array_merge($net, $s);
		}
	}

    if (empty($s['enable_search_protection'])) { return; }

    $term = $query->get('s');
    if (!is_string($term) || $term === '') { return; }

    $event = array(
        'type' => 'search',
        'actor' => array(
            'ip' => isset($_SERVER['REMOTE_ADDR']) ? (string) $_SERVER['REMOTE_ADDR'] : '',
            'user_id' => get_current_user_id(),
            'session_id' => '',
        ),
        'payload' => array(
            'fields' => array('s' => $term),
            'message' => $term,
            'metadata' => array('adapter' => 'search'),
        ),
    );

    $res = $this->engine->evaluate($event);
    $action = isset($res['action']) ? (string)$res['action'] : 'allow';

    if ($action === 'block' || $action === 'hold') {
        status_header(429);
        nocache_headers();
        wp_die(
            esc_html__('Search rate limited by AegisSpamGuard.', 'aegis-spam-guard'),
            esc_html__('Please slow down', 'aegis-spam-guard'),
            array('response' => 429)
        );
    }
    if ($action === 'challenge') {
        $ms = isset($res['meta']['progressive_delay_ms']) ? intval($res['meta']['progressive_delay_ms']) : 250;
        if ($ms > 0 && $ms <= 3000) { usleep($ms * 1000); }
    }
}

public function firewall_precheck() {

    if (is_admin()) { return; }
    if ((defined('DOING_CRON') && DOING_CRON) || (defined('REST_REQUEST') && REST_REQUEST)) { return; }
    if (function_exists('wp_doing_ajax') && wp_doing_ajax()) { return; }
    if (is_user_logged_in()) { return; }

	$s = get_option('asg_settings', array());
	if (!is_array($s)) { $s = array(); }

	if (is_multisite()) {
		$net = function_exists('get_site_option') ? get_site_option('asg_network_settings', array()) : array();
		if (is_array($net) && !empty($net)) {
			$s = array_merge($net, $s);
		}
	}

    if (empty($s['enable_firewall_precheck'])) { return; }

    $ctx = array(
        'ip' => isset($_SERVER['REMOTE_ADDR']) ? (string) $_SERVER['REMOTE_ADDR'] : '',
        'ua' => isset($_SERVER['HTTP_USER_AGENT']) ? (string) $_SERVER['HTTP_USER_AGENT'] : '',
        'uri' => isset($_SERVER['REQUEST_URI']) ? (string) $_SERVER['REQUEST_URI'] : '',
        'referer' => isset($_SERVER['HTTP_REFERER']) ? (string) $_SERVER['HTTP_REFERER'] : '',
    );

    $fw = $this->engine->firewall_check($ctx);
    if (!is_array($fw) || empty($fw['action'])) { return; }

    if (!empty($s['enable_firewall_log_stream'])) {
        $this->engine->log_firewall_event($ctx, $fw);
    }

    $action = (string) $fw['action'];
    if ($action === 'block') {
        $blank = !empty($s['firewall_blank_page']);
        if ($blank) {
            status_header(403);
            header('Content-Type: text/plain; charset=utf-8');
            echo '';
            exit;
        }
        wp_die(
            esc_html__('Access blocked by AegisSpamGuard firewall.', 'aegis-spam-guard'),
            esc_html__('Blocked', 'aegis-spam-guard'),
            array('response' => 403)
        );
    }

    if ($action === 'challenge') {
        $ms = isset($fw['meta']['delay_ms']) ? intval($fw['meta']['delay_ms']) : 250;
        if ($ms > 0 && $ms <= 3000) { usleep($ms * 1000); }
    }
}

    private static function clm_register_free_install() : void {
        if (!defined('ASG_CLM_FREE_REG_ENDPOINT')) { return; }

        if (get_transient('asg_clm_free_reg_sent')) { return; }

        $site_url = home_url();
        $domain   = wp_parse_url($site_url, PHP_URL_HOST);

        $payload = array(
            'product'        => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
            'email'          => (string) get_option('admin_email'),
            'site_url'       => (string) $site_url,
            'domain'         => (string) $domain,
            'plugin_version' => defined('ASG_VERSION') ? (string) ASG_VERSION : '',
            'wp_version'     => (string) get_bloginfo('version'),
            'php_version'    => (string) PHP_VERSION,
        );

        $args = array(
            'timeout' => 15,
            'headers' => array('Content-Type' => 'application/json'),
            'body'    => wp_json_encode($payload),
        );

        $resp = wp_remote_post(ASG_CLM_FREE_REG_ENDPOINT, $args);

        set_transient('asg_clm_free_reg_sent', 1, DAY_IN_SECONDS * 7);

        if (is_wp_error($resp)) { return; }
    }

    public static function clm_free_unregister_install(string $license_key, string $license_email) : void {
        if (!defined('ASG_CLM_FREE_UNREG_ENDPOINT')) { return; }
        $site_url = home_url();
        $domain   = wp_parse_url($site_url, PHP_URL_HOST);

        $payload = array(
            'slug'          => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
            'product'       => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
            'license_key'   => $license_key,
            'email'         => $license_email,
            'license_email' => $license_email,
            'domain'        => (string) $domain,
            'site_url'      => (string) $site_url,
        );

        wp_remote_post(ASG_CLM_FREE_UNREG_ENDPOINT, array(
            'timeout' => 12,
            'headers' => array('Content-Type' => 'application/json'),
            'body'    => wp_json_encode($payload),
        ));
    }



    /**
     * CLM: Ask update server for latest version + download URL. Includes license_key (PRO).
     */
    private function clm_direct_update_check(string $current_version) : array {
        if (!defined('ASG_CLM_UPDATE_ENDPOINT')) { return array(); }

        $site_url = home_url();
        $domain   = wp_parse_url($site_url, PHP_URL_HOST);

        $license_key = (string) get_option('asg_license_key', '');

        $payload = array(
            'slug'            => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
            'product'         => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
            'version'         => $current_version,
            'domain'          => (string) $domain,
            'site_url'        => (string) $site_url,
            'license_key'     => $license_key,
            'license_email'   => (string) get_option('asg_license_email', ''),
            'wp_version'      => (string) get_bloginfo('version'),
            'php_version'     => (string) PHP_VERSION,
            'plugin_basename' => defined('ASG_PLUGIN_BASENAME') ? (string) ASG_PLUGIN_BASENAME : '',
        );

        $args = array(
            'timeout' => 20,
            'headers' => array('Content-Type' => 'application/json'),
            'body'    => wp_json_encode($payload),
        );

        $resp = wp_remote_post(ASG_CLM_UPDATE_ENDPOINT, $args);
        if (is_wp_error($resp)) { return array(); }

        $code = (int) wp_remote_retrieve_response_code($resp);
        $body = (string) wp_remote_retrieve_body($resp);
        if ($code < 200 || $code >= 300 || $body === '') { return array(); }

        $data = json_decode($body, true);
        return is_array($data) ? $data : array();
    }

private static function clm_post_json(string $url, array $payload, int $timeout = 15) : array {
    if (empty($url)) { return array('success' => false, 'error' => 'empty_url'); }

    $args = array(
        'timeout' => $timeout,
        'headers' => array(
            'Content-Type' => 'application/json; charset=utf-8',
            'Accept'       => 'application/json',
        ),
        'body'    => wp_json_encode($payload),
    );

    $resp = wp_remote_post($url, $args);
    if (is_wp_error($resp)) {
        return array('success' => false, 'error' => 'request_failed', 'message' => $resp->get_error_message());
    }

    $code = (int) wp_remote_retrieve_response_code($resp);
    $body = (string) wp_remote_retrieve_body($resp);

    $data = json_decode($body, true);

    if (!is_array($data)) {
        $data = array('success' => false, 'error' => 'invalid_json', 'http_code' => $code, 'raw' => $body);
    } else {
        if (!isset($data['http_code'])) { $data['http_code'] = $code; }
        if (!isset($data['success'])) { $data['success'] = ($code >= 200 && $code < 300); }
    }

    return $data;
}

public static function clm_activate_license(string $license_key, string $license_email) : array {
    if (!defined('ASG_CLM_ACTIVATE_ENDPOINT')) { return array(); }

    $site_url = home_url();
    $domain   = wp_parse_url($site_url, PHP_URL_HOST);

    $payload = array(
        'slug'            => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
        'product'         => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
        'license_key'     => $license_key,
        'email'           => $license_email,
        'license_email'   => $license_email,
        'domain'          => (string) $domain,
        'site_url'        => (string) $site_url,
        'plugin_version'  => defined('ASG_VERSION') ? (string) ASG_VERSION : '',
        'wp_version'      => (string) get_bloginfo('version'),
        'php_version'     => (string) PHP_VERSION,
    );

    $data = self::clm_post_json(ASG_CLM_ACTIVATE_ENDPOINT, $payload);
    if (!is_array($data)) { $data = array(); }

    // If activate did not return expiry, ask status endpoint (CLM is the source of truth for expires).
    if (empty($data['expires_at']) && empty($data['expires']) && !empty($license_key) && !empty($license_email)) {
        $st = self::clm_status_license($license_key, $license_email);
        if (is_array($st)) {
            foreach (array('expires_at','expires','expiry','expiresAt') as $k) {
                if (!empty($st[$k])) { $data[$k] = $st[$k]; break; }
            }
            // also merge common fields if activate response was sparse
            foreach (array('status','license_status','plan','tier','license_plan','email','registered_email','customer_email') as $k) {
                if (empty($data[$k]) && !empty($st[$k])) { $data[$k] = $st[$k]; }
            }
        }
    }

    if (defined('ASG_CLM_FREE_UNREG_ENDPOINT') && $license_key !== '' && $license_email !== '') {
        self::clm_post_json(ASG_CLM_FREE_UNREG_ENDPOINT, array(
            'license_key' => $license_key,
            'email'       => $license_email,
            'domain'      => (string) $domain,
            'site_url'    => (string) $site_url,
        ));
    }

    return $data;
}

public static function clm_status_license(string $license_key, string $license_email) : array {
    if (!defined('ASG_CLM_STATUS_ENDPOINT')) { return array(); }

    $site_url = home_url();
    $domain   = wp_parse_url($site_url, PHP_URL_HOST);

    $payload = array(
        'slug'            => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
        'product'         => defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam',
        'license_key'     => $license_key,
        'email'           => $license_email,
        'license_email'   => $license_email,
        'domain'          => (string) $domain,
        'site_url'        => (string) $site_url,
        'plugin_version'  => defined('ASG_VERSION') ? (string) ASG_VERSION : '',
        'wp_version'      => (string) get_bloginfo('version'),
        'php_version'     => (string) PHP_VERSION,
    );

    $data = self::clm_post_json(ASG_CLM_STATUS_ENDPOINT, $payload);
    return is_array($data) ? $data : array();
}


public static function clm_extract_email(array $data) : string {
    foreach (array('email','registered_email','customer_email','license_email') as $k) {
        if (!empty($data[$k]) && is_string($data[$k])) { return (string) $data[$k]; }
    }
    return '';
}


public static function clm_extract_meta(array $data) : array {
    $status = '';
    $plan   = '';
    $expires = '';
    foreach (array('status','license_status') as $k) {
        if (!empty($data[$k]) && is_string($data[$k])) { $status = (string) $data[$k]; break; }
    }
    foreach (array('plan','tier','license_plan') as $k) {
        if (!empty($data[$k]) && is_string($data[$k])) { $plan = (string) $data[$k]; break; }
    }
    foreach (array('expires_at','expires','expiry','expiresAt') as $k) {
        if (!empty($data[$k])) { $expires = (string) $data[$k]; break; }
    }
    return array('status'=>$status,'plan'=>$plan,'expires'=>$expires,'expires_at'=>$expires);
}

public static function clm_maybe_refresh_license() : void {
    $key   = (string) get_option('asg_license_key', '');
    $email = (string) get_option('asg_license_email', '');
    if ($key === '' || $email === '') { return; }

    $last = (int) get_option('asg_license_last_check', 0);
    if ($last > 0 && (time() - $last) < 12 * HOUR_IN_SECONDS) { return; }

    $data = self::clm_status_license($key, $email);
    update_option('asg_license_last_check', time());

    $clm_email = self::clm_extract_email($data);
    $meta      = self::clm_extract_meta($data);

    if ($clm_email !== '') {
        update_option('asg_license_registered_email', $clm_email);
    }

    if ($clm_email !== '' && strtolower(trim($clm_email)) !== strtolower(trim($email))) {
        update_option('asg_license_status', 'email_mismatch');
        update_option('asg_license_plan', '');
        update_option('asg_license_expires_at', '');
        update_option('asg_license_error', 'Email does not match CLM record');
        return;
    }

    if (!empty($meta['status'])) { update_option('asg_license_status', (string) $meta['status']); }
    if (isset($meta['plan'])) { update_option('asg_license_plan', (string) $meta['plan']); }
    if (!empty($meta['expires_at'])) { update_option('asg_license_expires_at', (string) $meta['expires_at']); }
}


    public function clm_update_transient($transient) {
        if (!is_object($transient) || empty($transient->checked) || !is_array($transient->checked)) {
            return $transient;
        }

        if (!defined('ASG_PLUGIN_BASENAME') || empty($transient->checked[ ASG_PLUGIN_BASENAME ])) {
            return $transient;
        }

        $cache_key = 'asg_clm_update_' . md5((string) ASG_PLUGIN_BASENAME . '|' . home_url('/'));
        $cached = get_site_transient($cache_key);
        if (is_object($cached)) {
            $transient->response[ ASG_PLUGIN_BASENAME ] = $cached;
            return $transient;
        }

        $current_version = (string) $transient->checked[ ASG_PLUGIN_BASENAME ];
        $data = $this->clm_direct_update_check($current_version);

        $remote_version = '';
        if (!empty($data['version'])) { $remote_version = (string) $data['version']; }
        elseif (!empty($data['latest_version'])) { $remote_version = (string) $data['latest_version']; }

        if ($remote_version === '' || version_compare($remote_version, $current_version, '<=')) {
            return $transient;
        }

        if (empty($data['update_available']) && empty($data['download_url'])) {
        }

        $package = !empty($data['download_url']) ? (string) $data['download_url'] : '';
        if ($package === '') { return $transient; }

        $update              = new stdClass();
        $update->slug        = defined('ASG_SLUG') ? ASG_SLUG : 'aegisspam';
        $update->plugin      = (string) ASG_PLUGIN_BASENAME;
        $update->new_version = $remote_version;
        $update->package     = $package;
        $update->url         = 'https://aegisify.com/aegisspam';

        $transient->response[ ASG_PLUGIN_BASENAME ] = $update;
        set_site_transient($cache_key, $update, DAY_IN_SECONDS);

        return $transient;
    }

}
