<?php
defined('ABSPATH') || exit;

/**
 * Frontend Admin Dashboard Shortcode
 * Shortcode: [aegisticket_admin_dashboard]
 *
 * Features:
 * - List tickets (open + closed) with assignment to WordPress users
 * - Actions: View | Edit | Delete
 * - Inline View/Edit below list using ?admin-ticket=###&mode=edit
 * - Reply thread with attachments for both admin and ticket owner (chat style)
 * - Close ticket (admin/support agent)
 * - Shows closed tickets with closed-by user (meta)
 */

function aegisticket_reply_handle_uploads(string $field_name = 'reply_attachments'): array {
    if (!function_exists('aegisticket_handle_uploads')) {
        return [];
    }
    return aegisticket_handle_uploads($field_name);
}

add_shortcode('aegisticket_admin_dashboard', function () {

    if (!is_user_logged_in()) {
        auth_redirect();
        return '';
    }

    if (!current_user_can('view_support_tickets')) {
        return '<p>Access denied.</p>';
    }

    
    // Delete own reply (comment) in admin dashboard
    if (
        isset($_GET['aegisticket_action'], $_GET['comment_id'], $_GET['_wpnonce']) &&
        $_GET['aegisticket_action'] === 'admin_delete_reply' &&
        wp_verify_nonce($_GET['_wpnonce'], 'aegisticket_admin_delete_reply')
    ) {
        $cid = (int) $_GET['comment_id'];
        $c = get_comment($cid);
        if ($c && (int)$c->user_id === (int)get_current_user_id() && $c->comment_type === 'ticket_reply') {
            wp_delete_comment($cid, true);
        }
        wp_safe_redirect(remove_query_arg(['aegisticket_action','comment_id','_wpnonce','edit-reply']));
        exit;
    }

$active_ticket_id = isset($_GET['admin-ticket']) ? (int) $_GET['admin-ticket'] : 0;
$force_open = $active_ticket_id > 0;
    $mode = sanitize_text_field($_GET['mode'] ?? '');

    // Handle delete (admin only)
    if (
        isset($_GET['aegisticket_action'], $_GET['ticket_id'], $_GET['_wpnonce']) &&
        $_GET['aegisticket_action'] === 'delete' &&
        wp_verify_nonce($_GET['_wpnonce'], 'aegisticket_admin_delete_ticket')
    ) {
        $ticket_id = (int) $_GET['ticket_id'];
        if ($ticket_id && current_user_can('delete_support_ticket', $ticket_id)) {
            wp_trash_post($ticket_id);
        }
        wp_safe_redirect(remove_query_arg(['aegisticket_action','ticket_id','_wpnonce']));
        exit;
    }

    // Query tickets (both open and closed)
    $tickets = get_posts([
        'post_type' => 'support_ticket',
        'post_status' => ['open', 'closed'],
        'posts_per_page' => 100,
        'orderby' => 'date',
        'order' => 'DESC',
    ]);

    ob_start();
    ?>
    <div class="aegisticket">
        <h2>Admin Ticket Dashboard</h2>

        <table class="aegisticket-table">
            <thead>
                <tr>
                    <th>Ticket #</th>
                    <th>User</th>
                    <th>Subject</th>
                    <th>Priority</th>
                    <th>Status</th>
                    <th>Date Open</th>
                    <th>Date Closed</th>
                    <th>Assigned To</th>
                    <th>Admin Replied</th>
                    <th>Closed By</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
            <?php if (empty($tickets)) : ?>
                <tr><td colspan="9">No tickets found.</td></tr>
            <?php else : foreach ($tickets as $t) :
                $num = (int) get_post_meta($t->ID, '_ticket_number', true);
                $pri = (string) get_post_meta($t->ID, '_ticket_priority', true);
                $adm = (int) get_post_meta($t->ID, '_admin_replied', true);
                $assigned = (int) get_post_meta($t->ID, '_assigned_user_id', true);
                $closed_by = (string) get_post_meta($t->ID, '_closed_by', true);

                // Compute admin replied reliably (meta can be stale)
                $admin_replied = $adm ? 1 : 0;
                if (!$admin_replied) {
                    $maybe = get_comments([
                        'post_id' => $t->ID,
                        'status'  => 'approve',
                        'type'    => 'ticket_reply',
                        'number'  => 1,
                        'orderby' => 'comment_date_gmt',
                        'order'   => 'DESC',
                    ]);
                    if (!empty($maybe)) {
                        $c = $maybe[0];
                        $is_agent = $c->user_id && (user_can($c->user_id, 'manage_options') || user_can($c->user_id, 'reply_support_tickets'));
                        if ($is_agent) {
                            $admin_replied = 1;
                            update_post_meta($t->ID, '_admin_replied', 1);
                        }
                    }
                }

                $date_open = $t->post_date ? mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $t->post_date) : '—';
                $date_closed_meta = (string) get_post_meta($t->ID, '_closed_date_gmt', true);
                $date_closed = '—';
                if ($t->post_status === 'closed') {
                    if ($date_closed_meta) {
                        $date_closed = mysql2date(get_option('date_format') . ' ' . get_option('time_format'), get_date_from_gmt($date_closed_meta));
                    } else {
                        // fallback to last modified if meta missing
                        $date_closed = $t->post_modified ? mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $t->post_modified) : '—';
                    }
                }

                $author_id = (int) $t->post_author;
                $author = get_userdata($author_id);
                $assigned_user = $assigned ? get_userdata($assigned) : null;

                $view_url = add_query_arg(['admin-ticket' => $t->ID], get_permalink());
                $edit_url = add_query_arg(['admin-ticket' => $t->ID, 'mode' => 'edit'], get_permalink());
                $del_url  = wp_nonce_url(add_query_arg(['aegisticket_action' => 'delete', 'ticket_id' => $t->ID], get_permalink()), 'aegisticket_admin_delete_ticket');

                $row_class = ($active_ticket_id === (int)$t->ID) ? ' style="background:#fffbe6;"' : '';
            ?>
                <tr<?php echo $row_class; ?>>
                    <td><?php echo esc_html($num ?: $t->ID); ?></td>
                    <td><?php echo esc_html($author ? $author->user_login : 'user'); ?></td>
                    <td><?php echo esc_html($t->post_title); ?></td>
                    <td><?php echo esc_html($pri ?: 'medium'); ?></td>
                    <td><span class="aegisticket-badge <?php echo esc_attr($t->post_status); ?>"><?php echo esc_html($t->post_status); ?></span></td>
                    <td><?php echo esc_html($date_open); ?></td>
                    <td><?php echo esc_html($date_closed); ?></td>
                    <td><?php echo esc_html($assigned_user ? $assigned_user->user_login : '—'); ?></td>
                    <td><?php echo esc_html($admin_replied ? 'Yes' : 'No'); ?></td>
                    <td><?php echo esc_html($t->post_status === 'closed' ? ($closed_by ?: '—') : '—'); ?></td>
                    <td>
                        <a href="<?php echo esc_url($view_url); ?>">Open</a>
                        | <a href="<?php echo esc_url($edit_url); ?>">Edit</a>
                        | <a href="<?php echo esc_url($del_url); ?>" onclick="return confirm('Delete this ticket?');">Delete</a>
                    </td>
                </tr>
            <?php endforeach; endif; ?>
            </tbody>
        </table>

        <?php
        // Inline view/edit panel
        if ($force_open) :
            if (!aegisticket_ticket_can_view($active_ticket_id)) {
                echo '<p>Access denied.</p>';
            } else {
                $ticket = get_post($active_ticket_id);
                if ($ticket && $ticket->post_type === 'support_ticket') {

                    $errors = [];
                    $success = '';

                    $num  = (int) get_post_meta($ticket->ID, '_ticket_number', true);
                    $pri  = (string) get_post_meta($ticket->ID, '_ticket_priority', true);
                    $adm  = (int) get_post_meta($ticket->ID, '_admin_replied', true);
                    $atts = (array) get_post_meta($ticket->ID, '_ticket_attachments', true);
                    $atts = is_array($atts) ? $atts : [];

                    $assigned = (int) get_post_meta($ticket->ID, '_assigned_user_id', true);

                    // Admin edit (assignment + subject + priority + description + attach)
                    if (
                        $mode === 'edit' &&
                        isset($_POST['aegisticket_admin_update']) &&
                        isset($_POST['_wpnonce']) &&
                        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_admin_update_ticket')
                    ) {
                        $subject = sanitize_text_field($_POST['subject'] ?? '');
                        $priority = sanitize_text_field($_POST['priority'] ?? 'medium');
                        $description = sanitize_textarea_field($_POST['description'] ?? '');
                        $assign_to = (int) ($_POST['assigned_user_id'] ?? 0);

                        if ($subject === '') $errors[] = 'Subject is required.';
                        if ($description === '') $errors[] = 'Description is required.';

                        $valid_pri = ['low','medium','high','production alert'];
                        if (!in_array($priority, $valid_pri, true)) $priority = 'medium';

                        if (empty($errors)) {
                            wp_update_post([
                                'ID' => $ticket->ID,
                                'post_title' => $subject,
                                'post_content' => $description,
                            ]);

                            update_post_meta($ticket->ID, '_ticket_priority', $priority);

                            // Assign
                            if ($assign_to > 0) {
                                update_post_meta($ticket->ID, '_assigned_user_id', $assign_to);
                                $assigned = $assign_to;
                            } else {
                                delete_post_meta($ticket->ID, '_assigned_user_id');
                                $assigned = 0;
                            }

                            // Append ticket-level uploads
                            $new_ids = aegisticket_handle_uploads('attachments');
                            if (!empty($new_ids)) {
                                $merged = array_values(array_unique(array_merge($atts, $new_ids)));
                                update_post_meta($ticket->ID, '_ticket_attachments', $merged);
                                $atts = $merged;
                            }

                            $success = 'Ticket updated.';
                            $ticket = get_post($ticket->ID);
                            $pri = (string) get_post_meta($ticket->ID, '_ticket_priority', true);
                        }
                    }

                    
                    // Edit own reply (comment) (admin/support agent)
                    if (
                        isset($_POST['aegisticket_admin_edit_reply']) &&
                        isset($_POST['_wpnonce']) &&
                        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_admin_edit_reply')
                    ) {
                        $comment_id = (int) ($_POST['comment_id'] ?? 0);
                        $new_text = sanitize_textarea_field($_POST['reply_edit_text'] ?? '');
                        $c = $comment_id ? get_comment($comment_id) : null;
                        if (!$c || (int)$c->comment_post_ID !== (int)$ticket->ID || $c->comment_type !== 'ticket_reply' || (int)$c->user_id !== (int)get_current_user_id()) {
                            $errors[] = 'Invalid reply.';
                        } elseif ($new_text === '') {
                            $errors[] = 'Reply cannot be empty.';
                        } else {
                            wp_update_comment([
                                'comment_ID' => $comment_id,
                                'comment_content' => $new_text,
                            ]);
                            $success = 'Reply updated.';
                        }
                    }

// Reply (admin/support agent) with attachments
                    if (
                        isset($_POST['aegisticket_admin_reply']) &&
                        isset($_POST['_wpnonce']) &&
                        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_admin_reply_ticket') &&
                        aegisticket_ticket_is_open($ticket->ID)
                    ) {
                        $reply = sanitize_textarea_field($_POST['reply'] ?? '');
                        if ($reply === '') {
                            $errors[] = 'Reply cannot be empty.';
                        } else {
                            // Insert comment
                            $comment_id = wp_insert_comment([
                                'comment_post_ID' => $ticket->ID,
                                'comment_content' => $reply,
                                'comment_type' => 'ticket_reply',
                                'user_id' => get_current_user_id(),
                                'comment_approved' => 1,
                            ]);

                            if ($comment_id) {
                                // Reply attachments
                                $reply_ids = aegisticket_reply_handle_uploads('reply_attachments');
                                if (!empty($reply_ids)) {
                                    delete_comment_meta($comment_id, '_aegisticket_reply_attachments');
                                add_comment_meta($comment_id, '_aegisticket_reply_attachments', array_values($reply_ids), true);
                                }
                                $success = 'Reply posted.';
                            } else {
                                $errors[] = 'Could not post reply.';
                            }
                        }
                    }

                    // Close ticket (stores closed-by username)
                    if (
                        isset($_POST['aegisticket_admin_close']) &&
                        isset($_POST['_wpnonce']) &&
                        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_admin_close_ticket') &&
                        aegisticket_ticket_is_open($ticket->ID)
                    ) {
                        wp_update_post(['ID' => $ticket->ID, 'post_status' => 'closed']);
                        $u = wp_get_current_user();
                        update_post_meta($ticket->ID, '_closed_by', $u ? $u->user_login : 'admin');
                        update_post_meta($ticket->ID, '_closed_date_gmt', current_time('mysql', true));
                        $ticket = get_post($ticket->ID);
                        $success = 'Ticket closed.';
                    }

                    // Replies
                    $replies = get_comments([
                        'post_id' => $ticket->ID,
                        'status'  => 'approve',
                        'type'    => 'ticket_reply',
                        'orderby' => 'comment_date_gmt',
                        'order'   => 'ASC',
                    ]);

                    // Assigned user options (all users)
                    $users = get_users(['fields' => ['ID','user_login']]);
                    ?>
                    <hr>
                    <div class="aegisticket-card">
                        <?php if (!empty($success)) : ?>
                            <div class="aegisticket-notice success"><?php echo esc_html($success); ?></div>
                        <?php endif; ?>
                        <?php if (!empty($errors)) : ?>
                            <div class="aegisticket-notice error"><ul>
                                <?php foreach ($errors as $e) : ?><li><?php echo esc_html($e); ?></li><?php endforeach; ?>
                            </ul></div>
                        <?php endif; ?>

                        <h3>Ticket #<?php echo esc_html($num ?: $ticket->ID); ?> — <?php echo esc_html($ticket->post_title); ?></h3>

                        <p>
                            <strong>Status:</strong>
                            <span class="aegisticket-badge <?php echo esc_attr($ticket->post_status); ?>"><?php echo esc_html($ticket->post_status); ?></span>
                            &nbsp;
                            <?php
            $prod = get_post_meta($ticket->ID, '_ticket_product', true);
            $ver  = get_post_meta($ticket->ID, '_ticket_version', true);
            ?>
            <strong>Product:</strong> <?php echo esc_html($prod ?: '—'); ?>
            &nbsp;
            <strong>Version:</strong> <?php echo esc_html($ver ?: '—'); ?>
            &nbsp;
            <strong>Priority:</strong> <?php echo esc_html($pri ?: 'medium'); ?>
                            &nbsp;
                            <strong>Admin Replied:</strong> <?php echo esc_html($adm ? 'Yes' : 'No'); ?>
                            &nbsp;
                            <strong>Date Open:</strong> <?php echo esc_html($ticket->post_date ? mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $ticket->post_date) : '—'); ?>
                            &nbsp;
                            <strong>Date Closed:</strong> <?php echo esc_html(($ticket->post_status === 'closed') ? ( ( ($dc = (string) get_post_meta($ticket->ID, '_closed_date_gmt', true)) ? mysql2date(get_option('date_format') . ' ' . get_option('time_format'), get_date_from_gmt($dc)) : ($ticket->post_modified ? mysql2date(get_option('date_format') . ' ' . get_option('time_format'), $ticket->post_modified) : '—') ) ) : '—'); ?>
                        </p>

                        <?php if ($mode === 'edit') : ?>
                            <form method="post" enctype="multipart/form-data" class="aegisticket-form">
                                <?php wp_nonce_field('aegisticket_admin_update_ticket'); ?>

                                <label>Subject</label>
                                <input type="text" name="subject" value="<?php echo esc_attr($ticket->post_title); ?>" required>

                                <label>Priority</label>
                                <select name="priority">
                                    <?php foreach (['low','medium','high','production alert'] as $o) : ?>
                                        <option value="<?php echo esc_attr($o); ?>" <?php selected($pri, $o); ?>><?php echo esc_html($o); ?></option>
                                    <?php endforeach; ?>
                                </select>

                                <label>Description</label>
                                <textarea name="description" rows="4" style="width:100%;" required><?php echo esc_textarea($ticket->post_content); ?></textarea>

                                <label>Assign to User</label>
                                <select name="assigned_user_id">
                                    <option value="0">— Unassigned —</option>
                                    <?php foreach ($users as $usr) : ?>
                                        <option value="<?php echo (int)$usr->ID; ?>" <?php selected($assigned, (int)$usr->ID); ?>>
                                            <?php echo esc_html($usr->user_login); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>

                                <label>Upload additional files</label>
                                <input type="file" name="attachments[]" multiple>

                                <button type="submit" name="aegisticket_admin_update" class="aegisticket-btn">Save Changes</button>
                            </form>
                        <?php else : ?>
                            <div class="aegisticket-desc"><?php echo wpautop(esc_html($ticket->post_content)); ?></div>
                        <?php endif; ?>

                        <?php if (!empty($atts)) : ?>
                            <h4>Attachments</h4>
                            <ul>
                                <?php foreach ($atts as $aid) :
                                    $url = wp_get_attachment_url((int)$aid);
                                    if (!$url) continue;
                                ?>
                                    <li><a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener"><?php echo esc_html(basename($url)); ?></a></li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                    </div>

                    <div class="aegisticket-chat">
                        <h3>Conversation</h3>
                        <?php if (empty($replies)) : ?>
                            <p>No replies yet.</p>
                        <?php else : foreach ($replies as $c) :
                            $u = $c->user_id ? get_userdata($c->user_id) : null;
                            $who = 'User';
                            if ($c->user_id && (user_can($c->user_id, 'manage_options') || user_can($c->user_id, 'reply_support_tickets'))) {
                                $who = 'Admin';
                            } elseif ($u) {
                                $who = $u->user_login;
                            }

                            $reply_atts = get_comment_meta($c->comment_ID, '_aegisticket_reply_attachments', true);
                            // Normalize attachments per message
                            if (is_string($reply_atts)) {
                                $maybe = @unserialize($reply_atts);
                                if ($maybe !== false || $reply_atts === 'b:0;') {
                                    $reply_atts = $maybe;
                                }
                            }
                            $reply_atts = is_array($reply_atts) ? $reply_atts : [];
                        ?>
                            <div class="aegisticket-msg <?php echo ($who === 'Admin') ? 'agent' : 'user'; ?>">
                                <div class="aegisticket-msg-meta">
                                    <strong><?php echo esc_html($who); ?>:</strong>
                                    <?php if ((int)$c->user_id === (int)get_current_user_id()) :
                                        $edit_link = add_query_arg(['admin-ticket' => $ticket->ID, 'edit-reply' => $c->comment_ID], get_permalink());
                                        $del_link  = wp_nonce_url(add_query_arg(['aegisticket_action' => 'admin_delete_reply', 'comment_id' => $c->comment_ID], get_permalink()), 'aegisticket_admin_delete_reply');
                                    ?>
                                        <a href="<?php echo esc_url($edit_link); ?>" title="Edit" style="margin-left:8px; text-decoration:none;">
                                            <span class="dashicons dashicons-edit"></span>
                                        </a>
                                        <a href="<?php echo esc_url($del_link); ?>" title="Delete" onclick="return confirm('Delete this message?');" style="margin-left:4px; text-decoration:none;">
                                            <span class="dashicons dashicons-trash"></span>
                                        </a>
                                    <?php endif; ?>
                                    <span><?php echo esc_html(get_date_from_gmt($c->comment_date_gmt)); ?></span>
                                </div>
                                <div class="aegisticket-msg-body"><?php echo nl2br(esc_html($c->comment_content)); ?></div>

                                <?php if (!empty($reply_atts)) : ?>
                                    <div class="aegisticket-attachments">
                                        <strong>Attachments:</strong>
                                        <ul>
                                            <?php foreach ($reply_atts as $aid) :
                                                $url = wp_get_attachment_url((int)$aid);
                                                if (!$url) continue;
                                            ?>
                                                <li><a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener"><?php echo esc_html(basename($url)); ?></a></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; endif; ?>
                    </div>

                    <?php if ($ticket->post_status === 'open') : ?>
                        <div class="aegisticket-reply">
                            <h3>Post a reply (admin)</h3>
                            <form method="post" enctype="multipart/form-data" class="aegisticket-form">
                                <?php wp_nonce_field('aegisticket_admin_reply_ticket'); ?>
                                <textarea name="reply" rows="4" style="width:100%;" required></textarea>
                                <label>Attach files</label>
                                <input type="file" name="reply_attachments[]" multiple accept=".jpg,.jpeg,.png,.pdf,.doc,.docx,.zip">
                                <button type="submit" name="aegisticket_admin_reply" class="aegisticket-btn">Send Reply</button>
                            </form>

                            <form method="post" class="aegisticket-close">
                                <?php wp_nonce_field('aegisticket_admin_close_ticket'); ?>
                                <button type="submit" name="aegisticket_admin_close" class="aegisticket-btn danger" onclick="return confirm('Close this ticket?');">Close Ticket</button>
                            </form>
                        </div>
                    <?php else : ?>
                        <p><em>This ticket is closed.</em></p>
                    <?php endif; ?>

                    <?php
                }
            }
        endif;
        ?>
    </div>
    <?php

    return ob_get_clean();
});
