<?php
defined('ABSPATH') || exit;

function aegisticket_allowed_mimes(): array {
    // Safe common file types
    return [
        'jpg|jpeg|jpe' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif',
        'webp' => 'image/webp',
        'pdf' => 'application/pdf',
        'txt' => 'text/plain',
        'log' => 'text/plain',
        'zip' => 'application/zip',
    ];
}

function aegisticket_max_upload_bytes(): int {
    // 5MB default
    return (int) apply_filters('aegisticket_max_upload_bytes', 5 * 1024 * 1024);
}

/**
 * Handle attachments with validation + returns attachment IDs.
 */
function aegisticket_handle_uploads(string $field_name = 'attachments'): array {
    if (empty($_FILES[$field_name]) || empty($_FILES[$field_name]['name'])) return [];
    if (!is_array($_FILES[$field_name]['name'])) return [];

    $max = aegisticket_max_upload_bytes();
    $mimes = aegisticket_allowed_mimes();

    $ids = [];
    $count = count($_FILES[$field_name]['name']);

    require_once ABSPATH . 'wp-admin/includes/file.php';
    require_once ABSPATH . 'wp-admin/includes/media.php';
    require_once ABSPATH . 'wp-admin/includes/image.php';

    for ($i = 0; $i < $count; $i++) {
        if (empty($_FILES[$field_name]['name'][$i])) continue;

        $file = [
            'name'     => $_FILES[$field_name]['name'][$i],
            'type'     => $_FILES[$field_name]['type'][$i] ?? '',
            'tmp_name' => $_FILES[$field_name]['tmp_name'][$i] ?? '',
            'error'    => $_FILES[$field_name]['error'][$i] ?? 0,
            'size'     => (int) ($_FILES[$field_name]['size'][$i] ?? 0),
        ];

        if ($file['error'] !== 0) continue;
        if ($file['size'] <= 0 || $file['size'] > $max) continue;

        // Validate by extension + mime map
        $ft = wp_check_filetype($file['name'], $mimes);
        if (empty($ft['type'])) continue;

        // Temporarily override $_FILES for media_handle_sideload
        $overrides = [
            'test_form' => false,
            'mimes'     => $mimes,
        ];

        $upload = wp_handle_upload($file, $overrides);
        if (!empty($upload['error']) || empty($upload['file'])) {
            continue;
        }

        $attachment = [
            'post_mime_type' => $upload['type'],
            'post_title'     => sanitize_file_name(basename($upload['file'])),
            'post_content'   => '',
            'post_status'    => 'inherit',
        ];

        $attach_id = wp_insert_attachment($attachment, $upload['file']);
        if (is_wp_error($attach_id) || !$attach_id) continue;

        $meta = wp_generate_attachment_metadata($attach_id, $upload['file']);
        if (is_array($meta)) {
            wp_update_attachment_metadata($attach_id, $meta);
        }

        $ids[] = (int) $attach_id;
    }

    return $ids;
}

/**
 * Shortcode: [support_ticket_form]
 */
add_shortcode('support_ticket_form', function () {

    if (!is_user_logged_in()) {
        auth_redirect();
        return '';
    }

    $errors = [];
    $success = '';

    if (
        isset($_POST['aegisticket_create']) &&
        isset($_POST['_wpnonce']) &&
        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_create_ticket')
    ) {
        $subject = sanitize_text_field($_POST['subject'] ?? '');
        $priority = sanitize_text_field($_POST['priority'] ?? 'medium');
        $description = sanitize_textarea_field($_POST['description'] ?? '');

        if ($subject === '') $errors[] = 'Subject is required.';
        if ($description === '') $errors[] = 'Description is required.';

        $valid_pri = ['low','medium','high','production alert'];
        if (!in_array($priority, $valid_pri, true)) $priority = 'medium';

        if (empty($errors)) {
            $ticket_number = sts_generate_ticket_number();

            $ticket_id = wp_insert_post([
                'post_type'    => 'support_ticket',
                'post_title'   => $subject,
                'post_content' => $description,
                'post_status'  => 'open',
                'post_author'  => get_current_user_id(),
            ], true);

            if (is_wp_error($ticket_id)) {
                $errors[] = 'Could not create ticket.';
            } else {
                update_post_meta($ticket_id, '_ticket_number', $ticket_number);
                update_post_meta($ticket_id, '_ticket_priority', $priority);
    update_post_meta($ticket_id, '_ticket_product', sanitize_text_field($_POST['ticket_product'] ?? ''));
    update_post_meta($ticket_id, '_ticket_version', sanitize_text_field($_POST['ticket_version'] ?? ''));
                update_post_meta($ticket_id, '_admin_replied', 0);

                $attach_ids = aegisticket_handle_uploads('attachments');
                update_post_meta($ticket_id, '_ticket_attachments', $attach_ids);

                // Email admin on submission
                $admin_email = get_option('admin_email');
                $ticket_url = add_query_arg(['id' => $ticket_id], get_permalink());
                wp_mail(
                    $admin_email,
                    'New Support Ticket #' . $ticket_number,
                    "A new ticket has been submitted.\n\nSubject: {$subject}\nPriority: {$priority}\n\n(Frontend view depends on your page setup)\nTicket ID: {$ticket_id}\n"
                );

                $success = 'Ticket submitted successfully. Ticket #' . (int)$ticket_number;
            }
        }
    }

    ob_start();
    ?>
    <div class="aegisticket">
        <?php if (!empty($success)) : ?>
            <div class="aegisticket-notice success"><?php echo esc_html($success); ?></div>
        <?php endif; ?>
        <?php if (!empty($errors)) : ?>
            <div class="aegisticket-notice error">
                <ul>
                    <?php foreach ($errors as $e) : ?>
                        <li><?php echo esc_html($e); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <form method="post" enctype="multipart/form-data" class="aegisticket-form">
            <?php wp_nonce_field('aegisticket_create_ticket'); ?>

            <label>Subject</label>
            <input type="text" name="subject" required />

            <?php $products = get_option('aegisticket_products', []); ?>
        <label>Product</label>
        <select name="ticket_product" required>
            <option value="">Select Product</option>
            <?php foreach ($products as $p => $vs) : ?>
                <option value="<?php echo esc_attr($p); ?>"><?php echo esc_html($p); ?></option>
            <?php endforeach; ?>
        </select>
        <label>Product Version</label>
        <select name="ticket_version" required><option value="">Select Version</option></select>
        <script>
        document.addEventListener('DOMContentLoaded', function () {
            const data = <?php echo json_encode($products); ?>;
            const p = document.querySelector('[name="ticket_product"]');
            const v = document.querySelector('[name="ticket_version"]');
            p.addEventListener('change', function () {
                v.innerHTML = '<option value="">Select Version</option>';
                (data[this.value] || []).forEach(x => {
                    let o = document.createElement('option'); o.value=x; o.textContent=x; v.appendChild(o);
                });
            });
        });
        </script>
        <label>Priority</label>
        
            <select name="priority">
                <option value="low">low</option>
                <option value="medium" selected>medium</option>
                <option value="high">high</option>
                <option value="production alert">production alert</option>
            </select>

            <label>Description</label>
            <textarea name="description" rows="4" style="width:100%;" required></textarea>

            <label>Upload files / images</label>
            <input type="file" name="attachments[]" multiple />

            <button type="submit" name="aegisticket_create" class="aegisticket-btn">Submit Ticket</button>
        </form>
    </div>
    <?php
    return ob_get_clean();
});

/**
 * Frontend actions for edit/delete are handled in ticket-list.php and ticket-view.php
 */
