<?php
defined('ABSPATH') || exit;

/**
 * Shortcode: [aegisticket_profile]
 * Displays current user's profile details with view/edit toggle.
 *
 * Rules:
 * - Must be logged in (auth_redirect).
 * - View mode shows details + Edit button.
 * - Edit mode shows fields editable + Submit.
 */
add_shortcode('aegisticket_profile', function () {

    if (!is_user_logged_in()) {
        auth_redirect();
        return '';
    }

    $user_id = get_current_user_id();
    $user = get_userdata($user_id);
    if (!$user) {
        return '<p>User not found.</p>';
    }

    $is_edit = isset($_GET['profile-edit']) && (string)$_GET['profile-edit'] === '1';
    $notice = '';
    $errors = [];

    // Handle submit
    if (
        isset($_POST['aegisticket_profile_submit']) &&
        isset($_POST['_wpnonce']) &&
        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_profile_update')
    ) {
        $first_name = sanitize_text_field($_POST['first_name'] ?? '');
        $last_name  = sanitize_text_field($_POST['last_name'] ?? '');
        $email      = sanitize_email($_POST['email'] ?? '');
        $password   = (string) ($_POST['password'] ?? '');

        if ($email === '' || !is_email($email)) {
            $errors[] = 'A valid email address is required.';
        }

        if (empty($errors)) {
            $userdata = [
                'ID' => $user_id,
                'user_email' => $email,
            ];

            // Only update password if provided
            $password = trim($password);
            $password_changed = false;
            if ($password !== '') {
                $userdata['user_pass'] = $password;
                $password_changed = true;
            }

            $updated = wp_update_user($userdata);
            if (is_wp_error($updated)) {
                $errors[] = $updated->get_error_message();
            } else {
                // Names are stored in usermeta; wp_update_user can accept them but we update meta explicitly.
                update_user_meta($user_id, 'first_name', $first_name);
                update_user_meta($user_id, 'last_name', $last_name);

                $notice = $password_changed
                    ? 'Profile updated. If you changed your password, you may need to log in again.'
                    : 'Profile updated.';
                $is_edit = false;

                // Refresh user data
                $user = get_userdata($user_id);
            }
        }
    }

    $first = (string) get_user_meta($user_id, 'first_name', true);
    $last  = (string) get_user_meta($user_id, 'last_name', true);

    // Fallbacks (some sites store/display these via the WP_User object)
    if ($first === '' && isset($user->first_name)) { $first = (string) $user->first_name; }
    if ($last === '' && isset($user->last_name)) { $last = (string) $user->last_name; }

    $base_url = remove_query_arg(['profile-edit']);
    $edit_url = add_query_arg(['profile-edit' => '1'], $base_url);

    ob_start();
    ?>
    <div class="aegisticket">
        <?php if ($notice) : ?>
            <div class="aegisticket-notice success"><?php echo esc_html($notice); ?></div>
        <?php endif; ?>

        <?php if (!empty($errors)) : ?>
            <div class="aegisticket-notice error">
                <ul>
                    <?php foreach ($errors as $e) : ?>
                        <li><?php echo esc_html($e); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <?php if (!$is_edit) : ?>
            <div class="aegisticket-card">
                <h3>Profile Details</h3>
                <p><strong>First Name / Last Name:</strong> <?php echo esc_html(trim($first . ' ' . $last)); ?></p>
                <p><strong>Username:</strong> <?php echo esc_html($user->user_login); ?></p>
                <p><strong>Email Address:</strong> <?php echo esc_html($user->user_email); ?></p>
                <p><strong>Password:</strong> ••••••••</p>

                <p>
                    <a class="aegisticket-btn" href="<?php echo esc_url($edit_url); ?>">Edit</a>
                </p>
            </div>
        <?php else : ?>
            <div class="aegisticket-card">
                <h3>Edit Profile</h3>

                <form method="post" class="aegisticket-form">
                    <?php wp_nonce_field('aegisticket_profile_update'); ?>

                    <label>First Name</label>
                    <input type="text" name="first_name" value="<?php echo esc_attr($first); ?>" />

                    <label>Last Name</label>
                    <input type="text" name="last_name" value="<?php echo esc_attr($last); ?>" />

                    <label>Username (read-only)</label>
                    <input type="text" value="<?php echo esc_attr($user->user_login); ?>" disabled />

                    <label>Email Address</label>
                    <input type="text" name="email" value="<?php echo esc_attr($user->user_email); ?>" required />

                    <label>Password (leave blank to keep current)</label>
                    <input type="password" name="password" value="" autocomplete="new-password" />

                    <button type="submit" name="aegisticket_profile_submit" class="aegisticket-btn">Submit</button>
                </form>
            </div>
        <?php endif; ?>
    </div>
    <?php
    return ob_get_clean();
});
