<?php
defined('ABSPATH') || exit;

/**
 * Shortcode: [view_support_ticket id="123"]
 * If id not passed, uses ?id=
 */
add_shortcode('view_support_ticket', function ($atts) {

    if (!is_user_logged_in()) {
        auth_redirect();
        return '';
    }

    $atts = shortcode_atts(['id' => 0], $atts);
    $ticket_id = (int) ($atts['id'] ?: ($_GET['id'] ?? 0));
    $mode = sanitize_text_field($_GET['mode'] ?? '');

    if (!$ticket_id) return '<p>Ticket not found.</p>';
    if (!aegisticket_ticket_can_view($ticket_id)) return '<p>Access denied.</p>';

    $ticket = get_post($ticket_id);
    if (!$ticket || $ticket->post_type !== 'support_ticket') return '<p>Ticket not found.</p>';

    $num = (int) get_post_meta($ticket_id, '_ticket_number', true);
    $pri = (string) get_post_meta($ticket_id, '_ticket_priority', true);
    $adm = (int) get_post_meta($ticket_id, '_admin_replied', true);
    $atts_ids = get_post_meta($ticket_id, '_ticket_attachments', true);
    $atts_ids = is_array($atts_ids) ? $atts_ids : [];

    $errors = [];
    $success = '';

    // Handle edit (owner only, open only)
    if (
        $mode === 'edit' &&
        isset($_POST['aegisticket_update']) &&
        isset($_POST['_wpnonce']) &&
        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_update_ticket') &&
        aegisticket_is_ticket_owner($ticket_id) &&
        aegisticket_ticket_is_open($ticket_id)
    ) {
        $subject = sanitize_text_field($_POST['subject'] ?? '');
        $priority = sanitize_text_field($_POST['priority'] ?? 'medium');
        $description = sanitize_textarea_field($_POST['description'] ?? '');

        if ($subject === '') $errors[] = 'Subject is required.';
        if ($description === '') $errors[] = 'Description is required.';

        $valid_pri = ['low','medium','high','production alert'];
        if (!in_array($priority, $valid_pri, true)) $priority = 'medium';

        if (empty($errors)) {
            wp_update_post([
                'ID' => $ticket_id,
                'post_title' => $subject,
                'post_content' => $description,
            ]);

            update_post_meta($ticket_id, '_ticket_priority', $priority);

            // Append new uploads
            $new_ids = aegisticket_handle_uploads('attachments');
            if (!empty($new_ids)) {
                $merged = array_values(array_unique(array_merge($atts_ids, $new_ids)));
                update_post_meta($ticket_id, '_ticket_attachments', $merged);
                $atts_ids = $merged;
            }

            $success = 'Ticket updated.';
            // Refresh ticket object
            $ticket = get_post($ticket_id);
        }
    }

    // Handle reply (owner or agent; open only)
    if (
        isset($_POST['aegisticket_reply']) &&
        isset($_POST['_wpnonce']) &&
        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_reply_ticket') &&
        aegisticket_ticket_is_open($ticket_id)
    ) {
        $reply = sanitize_textarea_field($_POST['reply'] ?? '');
        if ($reply === '') {
            $errors[] = 'Reply cannot be empty.';
        } else {
            // Owner can reply only if owner; agent can reply if agent
            if (!aegisticket_ticket_can_view($ticket_id)) {
                $errors[] = 'Access denied.';
            } else {
                $commentdata = [
                    'comment_post_ID' => $ticket_id,
                    'comment_content' => $reply,
                    'comment_type' => 'ticket_reply',
                    'user_id' => get_current_user_id(),
                    'comment_approved' => 1,
                ];
                wp_insert_comment($commentdata);
                $success = 'Reply posted.';
            }
        }
    }

    // Admin/agent close ticket
    if (
        isset($_POST['aegisticket_close']) &&
        isset($_POST['_wpnonce']) &&
        wp_verify_nonce($_POST['_wpnonce'], 'aegisticket_close_ticket') &&
        aegisticket_is_agent() &&
        aegisticket_ticket_is_open($ticket_id)
    ) {
        wp_update_post(['ID' => $ticket_id, 'post_status' => 'closed']);
        $ticket = get_post($ticket_id);
        $success = 'Ticket closed.';
    }

    // Load replies (comments)
    $replies = get_comments([
        'post_id' => $ticket_id,
        'status' => 'approve',
        'type' => 'ticket_reply',
        'orderby' => 'comment_date_gmt',
        'order' => 'ASC',
        'number' => 0,
    ]);

    ob_start();
    ?>
    <div class="aegisticket">
        <?php if (!empty($success)) : ?>
            <div class="aegisticket-notice success"><?php echo esc_html($success); ?></div>
        <?php endif; ?>
        <?php if (!empty($errors)) : ?>
            <div class="aegisticket-notice error"><ul>
                <?php foreach ($errors as $e) : ?><li><?php echo esc_html($e); ?></li><?php endforeach; ?>
            </ul></div>
        <?php endif; ?>

        <div class="aegisticket-card">
            <div class="aegisticket-header">
                <div class="aegisticket-title">
                    Ticket #<?php echo esc_html($num ?: $ticket_id); ?> — <?php echo esc_html($ticket->post_title); ?>
                </div>
                <div class="aegisticket-meta">
                    <span class="aegisticket-badge <?php echo esc_attr($ticket->post_status); ?>"><?php echo esc_html($ticket->post_status); ?></span>
                    <span class="aegisticket-badge pri"><?php echo esc_html($pri ?: 'medium'); ?></span>
                    <?php if ($adm) : ?><span class="aegisticket-badge replied">admin replied</span><?php endif; ?>
                </div>
            </div>

            <?php if ($mode === 'edit' && aegisticket_is_ticket_owner($ticket_id) && $ticket->post_status === 'open') : ?>
                <form method="post" enctype="multipart/form-data" class="aegisticket-form">
                    <?php wp_nonce_field('aegisticket_update_ticket'); ?>
                    <label>Subject</label>
                    <input type="text" name="subject" value="<?php echo esc_attr($ticket->post_title); ?>" required />

                    <label>Priority</label>
                    <select name="priority">
                        <?php
                        $opts = ['low','medium','high','production alert'];
                        foreach ($opts as $o) {
                            echo '<option value="'.esc_attr($o).'" '.selected($pri, $o, false).'>'.esc_html($o).'</option>';
                        }
                        ?>
                    </select>

                    <label>Description</label>
                    <textarea name="description" rows="4" style="width:100%;" required><?php echo esc_textarea($ticket->post_content); ?></textarea>

                    <label>Upload additional files / images</label>
                    <input type="file" name="attachments[]" multiple />

                    <button type="submit" name="aegisticket_update" class="aegisticket-btn">Save Changes</button>
                </form>
            <?php else : ?>
                <div class="aegisticket-body">
                    <div class="aegisticket-desc"><?php echo wpautop(esc_html($ticket->post_content)); ?></div>
                </div>
            <?php endif; ?>

            <?php if (!empty($atts_ids)) : ?>
                <div class="aegisticket-attachments">
                    <strong>Attachments:</strong>
                    <ul>
                        <?php foreach ($atts_ids as $aid) :
                            $url = wp_get_attachment_url((int)$aid);
                            if (!$url) continue;
                        ?>
                            <li><a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener"><?php echo esc_html(basename($url)); ?></a></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
        </div>

        <div class="aegisticket-chat">
            <h3>Responses</h3>
            <?php if (empty($replies)) : ?>
                <p>No responses yet.</p>
            <?php else : ?>
                <?php foreach ($replies as $c) :
                    $u = $c->user_id ? get_userdata($c->user_id) : null;
                    $is_agent = $c->user_id ? user_can($c->user_id, 'reply_support_tickets') || user_can($c->user_id, 'manage_options') : false;
                ?>
                    <div class="aegisticket-msg <?php echo $is_agent ? 'agent' : 'user'; ?>">
                        <div class="aegisticket-msg-meta">
                            <strong><?php echo esc_html($u ? $u->display_name : 'User'); ?></strong>
                            <span><?php echo esc_html(get_date_from_gmt($c->comment_date_gmt)); ?></span>
                        </div>
                        <div class="aegisticket-msg-body"><?php echo nl2br(esc_html($c->comment_content)); ?></div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <?php if ($ticket->post_status === 'open') : ?>
            <div class="aegisticket-reply">
                <h3>Post a reply</h3>
                <form method="post" class="aegisticket-form">
                    <?php wp_nonce_field('aegisticket_reply_ticket'); ?>
                    <textarea name="reply" rows="4" style="width:100%;" required></textarea>
                    <button type="submit" name="aegisticket_reply" class="aegisticket-btn">Send Reply</button>
                </form>

                <?php if (aegisticket_is_agent()) : ?>
                    <form method="post" class="aegisticket-close">
                        <?php wp_nonce_field('aegisticket_close_ticket'); ?>
                        <button type="submit" name="aegisticket_close" class="aegisticket-btn danger" onclick="return confirm('Close this ticket?');">Close Ticket</button>
                    </form>
                <?php endif; ?>
            </div>
        <?php else : ?>
            <p><em>This ticket is closed.</em></p>
        <?php endif; ?>

    </div>
    <?php
    return ob_get_clean();
});
