<?php
/**
 * Plugin Name: AegisTicket
 * Plugin URI: https://aegisify.com/
 * Description: A basic WordPress-native support ticketing system.
 * Version: 1.8.1
 * Requires at least: 6.8
 * Requires PHP: 8.2
 * Author:      Aegisify | AegisShield | AegisWAF | AegisSEO
 * Author URI:  https://aegisify.com/
 * Text Domain: aegisticket
 */

defined('ABSPATH') || exit;

define('AEGISTICKET_VERSION', '1.8.1');
define('AEGISTICKET_PATH', plugin_dir_path(__FILE__));
define('AEGISTICKET_URL', plugin_dir_url(__FILE__));
define('AEGISTICKET_BASENAME', plugin_basename(__FILE__));
define('AEGISTICKET_SLUG', 'aegisticket');
define('AEGISTICKET_CLM_BASE', 'https://aegisify.com');
define('AEGISTICKET_CLM_FREE_REG_ENDPOINT', AEGISTICKET_CLM_BASE . '/wp-json/chue-license/v1/' . AEGISTICKET_SLUG . '/free-register');
define('AEGISTICKET_CLM_UPDATE_ENDPOINT',    AEGISTICKET_CLM_BASE . '/wp-json/chue-license/v1/' . AEGISTICKET_SLUG . '/update-check');
define('AEGISTICKET_CLM_ACTIVATE_ENDPOINT',   AEGISTICKET_CLM_BASE . '/wp-json/chue-license/v1/' . AEGISTICKET_SLUG . '/activate');
define('AEGISTICKET_CLM_DEACTIVATE_ENDPOINT', AEGISTICKET_CLM_BASE . '/wp-json/chue-license/v1/' . AEGISTICKET_SLUG . '/deactivate');
define('AEGISTICKET_CLM_STATUS_ENDPOINT',     AEGISTICKET_CLM_BASE . '/wp-json/chue-license/v1/' . AEGISTICKET_SLUG . '/status');

if ( ! function_exists( 'aegisticket_update_log' ) ) {
    function aegisticket_update_log( string $msg, array $ctx = array() ) : void {
        if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
            error_log( '[AegisTicket License] ' . $msg . ( $ctx ? ' | ' . wp_json_encode( $ctx ) : '' ) );
        }
    }
}

if ( ! function_exists( 'aegisticket_clm_register_free_install' ) ) {
    function aegisticket_clm_register_free_install() : void {

        if ( get_transient( 'aegisticket_clm_free_reg_sent' ) ) {
            return;
        }

        $site_url = home_url();
        $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

        $payload = array(
            'product'        => AEGISTICKET_SLUG,
            'email'          => (string) get_option( 'admin_email' ),
            'site_url'       => (string) $site_url,
            'domain'         => (string) ( $domain ?: '' ),
            'plugin_version' => defined( 'AEGISTICKET_VERSION' ) ? (string) AEGISTICKET_VERSION : 'unknown',
            'wp_version'     => (string) get_bloginfo( 'version' ),
            'php_version'    => (string) PHP_VERSION,
        );

        $args = array(
            'timeout' => 15,
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body'    => wp_json_encode( $payload ),
        );

        $resp = wp_remote_post( AEGISTICKET_CLM_FREE_REG_ENDPOINT, $args );

        if ( is_wp_error( $resp ) ) {
            aegisticket_update_log( 'CLM free-register failed (WP_Error).', array( 'error' => $resp->get_error_message() ) );
            return;
        }

        $code = (int) wp_remote_retrieve_response_code( $resp );
        $body = (string) wp_remote_retrieve_body( $resp );

        $backoff = ( $code >= 500 ) ? HOUR_IN_SECONDS : DAY_IN_SECONDS;
        set_transient( 'aegisticket_clm_free_reg_sent', 1, $backoff );

        aegisticket_update_log( 'CLM free-register response.', array(
            'endpoint'  => AEGISTICKET_CLM_FREE_REG_ENDPOINT,
            'http_code' => $code,
            'body'      => $body,
            'backoff_s' => $backoff,
        ) );
    }
}

if ( ! function_exists( 'aegisticket_clm_direct_update_check' ) ) {
    function aegisticket_clm_direct_update_check( string $current_version ) : array {

        $site_url = home_url();
        $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

        $license_key = (string) get_option( 'aegisticket_license_key', '' );

        $payload = array(
            'slug'        => AEGISTICKET_SLUG,
            'product'     => AEGISTICKET_SLUG,
            'version'     => $current_version,
            'domain'      => $domain ? (string) $domain : '',
            'site_url'    => (string) $site_url,
            'license_key' => $license_key,
        );

        $args = array(
            'timeout' => 15,
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body'    => wp_json_encode( $payload ),
        );

        $resp = wp_remote_post( AEGISTICKET_CLM_UPDATE_ENDPOINT, $args );

        if ( is_wp_error( $resp ) ) {
            return array(
                'success' => false,
                'status'  => 'wp_error',
                'message' => $resp->get_error_message(),
            );
        }

        $code = (int) wp_remote_retrieve_response_code( $resp );
        $body = (string) wp_remote_retrieve_body( $resp );

        if ( $code < 200 || $code >= 300 ) {
            return array(
                'success' => false,
                'status'  => 'http_' . $code,
                'message' => 'Non-2xx from CLM',
                'raw'     => $body,
            );
        }

        $data = json_decode( $body, true );
        if ( ! is_array( $data ) ) {
            return array(
                'success' => false,
                'status'  => 'bad_json',
                'message' => 'CLM returned invalid JSON',
                'raw'     => $body,
            );
        }

        return $data;
    }
}

if ( ! function_exists( 'aegisticket_check_for_updates_once_daily' ) ) {
    function aegisticket_check_for_updates_once_daily( $transient ) {

        if ( ! is_object( $transient ) || empty( $transient->checked ) || ! is_array( $transient->checked ) ) {
            return $transient;
        }

        if ( empty( $transient->checked[ AEGISTICKET_BASENAME ] ) ) {
            return $transient;
        }

        $cache_key = 'aegisticket_clm_update_' . md5( home_url() . '|' . AEGISTICKET_VERSION );
        $cached    = get_site_transient( $cache_key );
        if ( $cached && is_object( $cached ) ) {
            $transient->response[ AEGISTICKET_BASENAME ] = $cached;
            return $transient;
        }

        $current_version = (string) $transient->checked[ AEGISTICKET_BASENAME ];
        $data            = aegisticket_clm_direct_update_check( $current_version );

        if ( empty( $data['success'] ) || empty( $data['update_available'] ) ) {
            return $transient;
        }

        $remote_version = ! empty( $data['version'] ) ? (string) $data['version'] : '';
        $package        = ! empty( $data['download_url'] ) ? (string) $data['download_url'] : '';

        if ( empty( $remote_version ) || empty( $package ) ) {
            return $transient;
        }

        if ( $current_version && version_compare( $remote_version, $current_version, '<=' ) ) {
            return $transient;
        }

        $update              = new stdClass();
        $update->slug        = AEGISTICKET_SLUG;
        $update->plugin      = AEGISTICKET_BASENAME;
        $update->new_version = $remote_version;
        $update->package     = $package;
        $update->url         = ! empty( $data['homepage'] ) ? (string) $data['homepage'] : AEGISTICKET_CLM_BASE;

        $transient->response[ AEGISTICKET_BASENAME ] = $update;
        set_site_transient( $cache_key, $update, DAY_IN_SECONDS );

        return $transient;
    }
}

add_filter( 'pre_set_site_transient_update_plugins', 'aegisticket_check_for_updates_once_daily', 25, 1 );

add_action( 'plugins_loaded', 'aegisticket_clm_register_free_install', 5 );


require_once AEGISTICKET_PATH . 'includes/cpt-tickets.php';
require_once AEGISTICKET_PATH . 'includes/ticket-meta.php';
require_once AEGISTICKET_PATH . 'includes/frontend-forms.php';
require_once AEGISTICKET_PATH . 'includes/ticket-list.php';
require_once AEGISTICKET_PATH . 'includes/ticket-view.php';
require_once AEGISTICKET_PATH . 'includes/profile-shortcode.php';
require_once AEGISTICKET_PATH . 'includes/comments-replies.php';
require_once AEGISTICKET_PATH . 'includes/email-notifications.php';
require_once AEGISTICKET_PATH . 'includes/class-aegisify-bundled-plugins-page.php';
require_once AEGISTICKET_PATH . 'includes/admin-pages.php';
require_once AEGISTICKET_PATH . 'includes/admin-dashboard-shortcode.php';


add_action('wp_enqueue_scripts', function () {
    wp_register_style('aegisticket-css', AEGISTICKET_URL . 'assets/css/tickets.css', [], AEGISTICKET_VERSION);
    wp_enqueue_style('aegisticket-css');
});

register_activation_hook(__FILE__, function () {
    $admin = get_role('administrator');
    if ($admin) {
        $caps = [
            'view_support_tickets',
            'reply_support_tickets',
            'close_support_tickets',
        ];
        foreach ($caps as $cap) {
            $admin->add_cap($cap);
        }
    }

    if (!get_role('support_agent')) {
        add_role('support_agent', 'Support Agent', [
            'read' => true,
            'upload_files' => true,
            'view_support_tickets' => true,
            'reply_support_tickets' => true,
            'close_support_tickets' => true,
        ]);
    }

    if (get_option('sts_last_ticket_number', null) === null) {
        add_option('sts_last_ticket_number', 999);
    }

    if (function_exists('flush_rewrite_rules')) {
        flush_rewrite_rules();
    }
});

register_deactivation_hook(__FILE__, function () {
    if (function_exists('flush_rewrite_rules')) {
        flush_rewrite_rules();
    }
});