<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Page_API_Shield {

    public static function render() : void {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegiswaf' ) );
        }

        $s = AegisWAF_Storage::get_settings();
        $wp = is_array( $s['wp_protect'] ?? null ) ? $s['wp_protect'] : [];

		$is_pro = false;

		if ( class_exists( 'AegisWAF_Features' ) && method_exists( 'AegisWAF_Features', 'is_pro' ) ) {
			$is_pro = (bool) AegisWAF_Features::is_pro();
		}

		if ( ! $is_pro && function_exists( 'aegiswaf_is_pro' ) ) {
			$is_pro = (bool) aegiswaf_is_pro();
		} elseif ( ! $is_pro && class_exists( 'AegisWAF_Pro' ) && method_exists( 'AegisWAF_Pro', 'is_active' ) ) {
			$is_pro = (bool) AegisWAF_Pro::is_active();
		}

		$is_pro = (bool) apply_filters( 'aegiswaf_is_pro', $is_pro );

        if ( isset( $_POST['aegiswaf_save_api_shield'] ) ) {
            check_admin_referer( 'aegiswaf_save_api_shield' );

            $wp['enabled'] = ! empty( $_POST['wp_enabled'] );
            $wp['rest_protect'] = ! empty( $_POST['wp_rest_protect'] );
            $wp['rest_only_unauth'] = ! empty( $_POST['wp_rest_only_unauth'] );
            $wp['rest_block_user_enum'] = ! empty( $_POST['wp_rest_block_user_enum'] );
            $wp['rest_require_write_auth'] = ! empty( $_POST['wp_rest_require_write_auth'] );
            $wp['rest_require_json'] = ! empty( $_POST['wp_rest_require_json'] );
            $wp['rest_max_body_bytes'] = max( 0, (int) ( $_POST['wp_rest_max_body_bytes'] ?? 0 ) );

            $wp['rest_cors_allowlist'] = isset( $_POST['wp_rest_cors_allowlist'] ) ? (string) wp_unslash( $_POST['wp_rest_cors_allowlist'] ) : '';

            $wp['rest_api_key_enabled'] = ! empty( $_POST['wp_rest_api_key_enabled'] );
            $wp['rest_api_key_header']  = isset( $_POST['wp_rest_api_key_header'] ) ? sanitize_key( (string) wp_unslash( $_POST['wp_rest_api_key_header'] ) ) : 'x-aegis-key';
            $wp['rest_api_key_value']   = isset( $_POST['wp_rest_api_key_value'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['wp_rest_api_key_value'] ) ) : '';

            $wp['rest_allowlist'] = isset( $_POST['wp_rest_allowlist'] ) ? (string) wp_unslash( $_POST['wp_rest_allowlist'] ) : '';
            $wp['window_seconds_rest'] = max( 15, (int) ( $_POST['wp_window_rest'] ?? 60 ) );
            $wp['thresholds_rest'] = is_array( $wp['thresholds_rest'] ?? null ) ? $wp['thresholds_rest'] : [];
            $wp['thresholds_rest']['challenge_at']  = isset( $_POST['wp_thr_rest_challenge'] ) ? max( 0, (int) $_POST['wp_thr_rest_challenge'] ) : (int) ( $wp['thresholds_rest']['challenge_at'] ?? 0 );
            $wp['thresholds_rest']['rate_limit_at'] = isset( $_POST['wp_thr_rest_rate'] ) ? max( 0, (int) $_POST['wp_thr_rest_rate'] ) : (int) ( $wp['thresholds_rest']['rate_limit_at'] ?? 0 );
            $wp['thresholds_rest']['block_at']      = isset( $_POST['wp_thr_rest_block'] ) ? max( 0, (int) $_POST['wp_thr_rest_block'] ) : (int) ( $wp['thresholds_rest']['block_at'] ?? 0 );

			if ( $is_pro ) {
				$list = [];
				if ( ! empty( $wp['rest_route_overrides_list'] ) && is_array( $wp['rest_route_overrides_list'] ) ) {
					$list = $wp['rest_route_overrides_list'];
				} else {
					$legacy = isset( $wp['rest_route_overrides'] ) ? (string) $wp['rest_route_overrides'] : '';
					$lines = preg_split( "/\r\n|\n|\r/", $legacy );
					foreach ( $lines as $ln ) {
						$ln = trim( (string) $ln );
						if ( $ln === '' ) { continue; }
						$parsed = self::parse_route_override_line( $ln );
						if ( is_wp_error( $parsed ) ) { continue; }
						$list[] = $parsed;
					}
				}

				$action = isset( $_POST['wp_rest_override_action'] ) ? sanitize_key( (string) $_POST['wp_rest_override_action'] ) : '';
				$index  = isset( $_POST['wp_rest_override_index'] ) ? (int) $_POST['wp_rest_override_index'] : -1;

				if ( $action === 'delete' && $index >= 0 && isset( $list[ $index ] ) ) {
					unset( $list[ $index ] );
					$list = array_values( $list );
				}

				if ( ( $action === 'add' || $action === 'update' ) ) {
					$raw = isset( $_POST['wp_rest_override_line'] ) ? trim( (string) wp_unslash( $_POST['wp_rest_override_line'] ) ) : '';
					if ( $raw !== '' ) {
						$parsed = self::parse_route_override_line( $raw );
						if ( is_wp_error( $parsed ) ) {
							add_settings_error( 'aegiswaf_api_shield', 'route_override_parse', $parsed->get_error_message(), 'error' );
						} else {
							if ( $action === 'update' && $index >= 0 && isset( $list[ $index ] ) ) {
								$list[ $index ] = $parsed;
							} else {
								$list[] = $parsed;
							}
						}
					}
				}

				$wp['rest_route_overrides_list'] = $list;

				$lines = [];
				foreach ( $list as $row ) {
					if ( ! empty( $row['raw'] ) ) { $lines[] = (string) $row['raw']; }
				}
				$wp['rest_route_overrides'] = implode( "\n", $lines );

				$wp['rest_route_auto_profiles'] = ! empty( $_POST['wp_rest_route_auto_profiles'] );

			} else {
				unset( $wp['rest_route_overrides'], $wp['rest_route_auto_profiles'], $wp['rest_route_overrides_list'] );
			}

            $s['wp_protect'] = $wp;

            $ok = AegisWAF_Storage::update_settings( $s );

            if ( ! $ok ) {
                if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                    AegisWAF_Logger::log( 'wp_admin', 'POST', 'api_shield', 'save_failed', [
                        'reason' => 'update_settings returned false',
                        'wp_protect' => $wp,
                    ] );
                }
                error_log( '[AegisWAF] API Shield settings save failed (update_settings returned false).' );

                echo '<div class="notice notice-error"><p>' . esc_html__( 'API Shield settings failed to save. Check Logs and debug.log.', 'aegiswaf' ) . '</p></div>';
            } else {
                if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                    AegisWAF_Logger::log( 'wp_admin', 'POST', 'api_shield', 'settings_saved', [
                        'wp_protect' => $wp,
                    ] );
                }

                echo '<div class="notice notice-success"><p>' . esc_html__( 'API Shield settings saved.', 'aegiswaf' ) . '</p></div>';
            }
        }

        $s = AegisWAF_Storage::get_settings();
        $wp = is_array( $s['wp_protect'] ?? null ) ? $s['wp_protect'] : [];

        echo '<div class="wrap">';
        echo '<h1>API Shield</h1>';
        AegisWAF_Pro_Notice::render();
        echo '<p class="description">The API Shield tab secures REST and custom endpoints by enforcing request methods, validating access behavior, and applying rate limits to prevent abuse. It helps protect APIs from enumeration, brute force attempts, and automated misuse. In Pro, API Shield adds per-endpoint policies, header and token enforcement, and advanced API-specific logging, giving administrators precise control over how each endpoint can be accessed.</p>';

		$viz = self::get_api_shield_visuals( 24 ); // last 24 hours

		echo '<style>
		.aegiswaf-viz-grid{display:grid;grid-template-columns:repeat(4,minmax(220px,1fr));gap:12px;margin:14px 0 18px 0;}
		.aegiswaf-viz-card{background:#fff;border:1px solid #e5e5e5;border-radius:10px;padding:12px;}
		.aegiswaf-viz-title{font-size:13px;font-weight:600;margin:0 0 6px 0;}
		.aegiswaf-viz-sub{font-size:12px;color:#646970;margin:0 0 10px 0;}
		.aegiswaf-viz-wrap{position:relative;height:210px;}
		@media (max-width:1200px){.aegiswaf-viz-grid{grid-template-columns:repeat(2,minmax(220px,1fr));}}
		@media (max-width:640px){.aegiswaf-viz-grid{grid-template-columns:1fr;}}
		</style>';

		echo '<div class="aegiswaf-viz-grid">';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Actions Breakdown</p>';
		echo '<p class="aegiswaf-viz-sub">Allow vs Challenge vs Rate-limit vs Block (last 24h)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_api_actions_donut"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">API Events Over Time</p>';
		echo '<p class="aegiswaf-viz-sub">Hourly trend (last 24h)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_api_timeline_line"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Top Targeted Routes</p>';
		echo '<p class="aegiswaf-viz-sub">Most-hit REST paths (last 24h)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_api_top_routes_bar"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Top Offending IPs</p>';
		echo '<p class="aegiswaf-viz-sub">Bubble = total events (last 24h)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_api_offenders_bubble"></canvas></div>';
		echo '</div>';

		echo '</div>';

		$chart_js_url = plugins_url( 'assets/js/chart.umd.min.js', dirname( __FILE__, 4 ) . '/aegiswaf.php' );
		echo '<script src="' . esc_url( $chart_js_url ) . '"></script>';

		echo '<script>
		window.AEGISWAF_API_VIZ = ' . wp_json_encode( $viz ) . ';
		(function(){
		  function safeNum(n){ n = parseInt(n,10); return isFinite(n) ? n : 0; }

		  function init(){
			if(!window.Chart || !window.AEGISWAF_API_VIZ){ return; }
			const v = window.AEGISWAF_API_VIZ;

			// 1) Donut: actions breakdown
			const d1 = document.getElementById("aegiswaf_api_actions_donut");
			if (d1) {
			  new Chart(d1, {
				type: "doughnut",
				data: {
				  labels: v.actions.labels,
				  datasets: [{ data: v.actions.values }]
				},
				options: {
				  responsive: true,
				  maintainAspectRatio: false,
				  plugins: { legend: { position: "bottom" } }
				}
			  });
			}

			// 2) Line: timeline (hourly)
			const d2 = document.getElementById("aegiswaf_api_timeline_line");
			if (d2) {
			  new Chart(d2, {
				type: "line",
				data: {
				  labels: v.timeline.labels,
				  datasets: [{
					label: "Events",
					data: v.timeline.values,
					tension: 0.25
				  }]
				},
				options: {
				  responsive: true,
				  maintainAspectRatio: false,
				  plugins: { legend: { display: false } },
				  scales: { y: { beginAtZero: true } }
				}
			  });
			}

			// 3) Bar: top routes
			const d3 = document.getElementById("aegiswaf_api_top_routes_bar");
			if (d3) {
			  new Chart(d3, {
				type: "bar",
				data: {
				  labels: v.routes.labels,
				  datasets: [{ label: "Hits", data: v.routes.values }]
				},
				options: {
				  responsive: true,
				  maintainAspectRatio: false,
				  plugins: { legend: { display: false } },
				  scales: { y: { beginAtZero: true } }
				}
			  });
			}

			// 4) Bubble: top offenders (x=blocks, y=rate-limits, r=total)
			const d4 = document.getElementById("aegiswaf_api_offenders_bubble");
			if (d4) {
			  new Chart(d4, {
				type: "bubble",
				data: {
				  datasets: [{
					label: "IPs",
					data: v.offenders.points
				  }]
				},
				options: {
				  responsive: true,
				  maintainAspectRatio: false,
				  parsing: false,
				  plugins: {
					tooltip: {
					  callbacks: {
						label: function(ctx){
						  const p = ctx.raw || {};
						  return (p.label || "IP") + " — blocks:" + safeNum(p.x) + ", rate:" + safeNum(p.y) + ", total:" + safeNum(p.r);
						}
					  }
					},
					legend: { display: false }
				  },
				  scales: {
					x: { beginAtZero: true, title: { display: true, text: "Blocks" } },
					y: { beginAtZero: true, title: { display: true, text: "Rate-limits" } }
				  }
				}
			  });
			}
		  }

		  if (document.readyState === "loading") {
			document.addEventListener("DOMContentLoaded", init);
		  } else {
			init();
		  }
		})();
		</script>';

        echo '<div class="aegiswaf-card">';
        echo '<h2>REST API Protection</h2>';
        echo '<form method="post">';
        wp_nonce_field( 'aegiswaf_save_api_shield' );

        echo '<p><label><input type="checkbox" name="wp_enabled" value="1" ' . checked( ! empty( $wp['enabled'] ), true, false ) . '> Enable WordPress protection layer</label></p>';
        echo '<p><label><input type="checkbox" name="wp_rest_protect" value="1" ' . checked( ! empty( $wp['rest_protect'] ), true, false ) . '> Enable REST API protection</label></p>';

        echo '<p><label><input type="checkbox" name="wp_rest_only_unauth" value="1" ' . checked( ! empty( $wp['rest_only_unauth'] ), true, false ) . '> Only enforce on unauthenticated REST requests</label></p>';


        echo '<h3>REST Hardening</h3>';
        echo '<p class="description">Extra controls that apply before rate limiting. All denials are logged in Logs → WAF.</p>';

        echo '<p><label><input type="checkbox" name="wp_rest_block_user_enum" value="1" ' . checked( ! empty( $wp['rest_block_user_enum'] ), true, false ) . '> Block unauthenticated access to /wp/v2/users (user enumeration)</label></p>';

        echo '<p><label><input type="checkbox" name="wp_rest_require_write_auth" value="1" ' . checked( ! empty( $wp['rest_require_write_auth'] ), true, false ) . '> Require authentication for non-GET REST requests (POST/PUT/PATCH/DELETE) unless allowlisted</label></p>';

        echo '<p><label><input type="checkbox" name="wp_rest_require_json" value="1" ' . checked( ! empty( $wp['rest_require_json'] ), true, false ) . '> Require application/json Content-Type on REST requests with a body</label></p>';

        echo '<p><label>Max REST body size (bytes, 0 = unlimited) <input type="number" name="wp_rest_max_body_bytes" min="0" step="1" value="' . esc_attr( (string) ( $wp['rest_max_body_bytes'] ?? 0 ) ) . '"></label></p>';

        echo '<p><label>CORS allowlist (one Origin per line; supports * wildcards). Leave blank to allow any Origin.</label><br>';
        echo '<textarea name="wp_rest_cors_allowlist" rows="4" style="width:100%;">' . esc_textarea( (string) ( $wp['rest_cors_allowlist'] ?? '' ) ) . '</textarea></p>';

        echo '<p><label><input type="checkbox" name="wp_rest_api_key_enabled" value="1" ' . checked( ! empty( $wp['rest_api_key_enabled'] ), true, false ) . '> Require API key header for unauthenticated REST requests</label></p>';
        echo '<p style="display:flex;gap:10px;align-items:center;flex-wrap:wrap;"></p>';
        echo '<label>Header <input type="text" name="wp_rest_api_key_header" value="' . esc_attr( (string) ( $wp['rest_api_key_header'] ?? 'x-aegis-key' ) ) . '"></label>';
        echo '<label>Value <input type="text" name="wp_rest_api_key_value" value="' . esc_attr( (string) ( $wp['rest_api_key_value'] ?? '' ) ) . '"></label>';

        echo '<p><label>REST window (seconds) <input type="number" name="wp_window_rest" min="15" max="600" value="' . esc_attr( (string) ( $wp['window_seconds_rest'] ?? 60 ) ) . '"></label></p>';

        echo '<h3>Progressive Enforcement (REST)</h3>';
        echo '<p class="description">Counts within the REST window that trigger actions. Set 0 to disable a step.</p>';
        $thr = is_array( $wp['thresholds_rest'] ?? null ) ? $wp['thresholds_rest'] : [];
        $c = (int) ( $thr['challenge_at'] ?? 60 );
        $r = (int) ( $thr['rate_limit_at'] ?? 120 );
        $b = (int) ( $thr['block_at'] ?? 220 );

        echo '<table class="widefat striped" style="max-width:700px"><thead><tr><th>Challenge at</th><th>Rate-limit at</th><th>Block at</th></tr></thead><tbody><tr>';
        echo '<td><input type="number" name="wp_thr_rest_challenge" value="' . esc_attr( (string) $c ) . '" min="0" max="100000" style="width:140px"></td>';
        echo '<td><input type="number" name="wp_thr_rest_rate" value="' . esc_attr( (string) $r ) . '" min="0" max="100000" style="width:140px"></td>';
        echo '<td><input type="number" name="wp_thr_rest_block" value="' . esc_attr( (string) $b ) . '" min="0" max="100000" style="width:140px"></td>';
        echo '</tr></tbody></table>';

        echo '<h3>REST Allowlist</h3>';
        echo '<p class="description">One route per line. Wildcards supported, e.g. <code>/wp/v2/*</code> or <code>/myplugin/v1/*</code>.</p>';
        echo '<p><textarea name="wp_rest_allowlist" rows="6" style="width:100%">' . esc_textarea( (string) ( $wp['rest_allowlist'] ?? '' ) ) . '</textarea></p>';
        echo '</div>'; // wrap

		echo '<div style="margin-top:18px; padding:12px; border:1px solid #e5e5e5; background:#fff;">';
		echo '<h2>Advanced Per-Route Controls (PRO)</h2>';

		echo '<p class="description">Define per-route thresholds, per-method enforcement, and category/profile tagging. One rule per line.</p>';

		echo '<p style="margin:10px 0 6px 0;"><strong>Format (one per line)</strong></p>';
		echo '<code style="display:block; padding:8px; background:#f6f7f7; border:1px solid #e5e5e5;">';
		echo esc_html( 'pattern | category | profile | METHOD:challenge,rate,block | METHOD:challenge,rate,block ...' );
		echo '</code>';

		echo '<p style="margin:10px 0 6px 0;"><strong>Example</strong></p>';
		echo '<code style="display:block; padding:8px; background:#f6f7f7; border:1px solid #e5e5e5;">';
		echo esc_html( '/wp/v2/users* | user_enum | sensitive | GET:10,20,30' ) . '<br>';
		echo esc_html( '/wc/v3/orders* | ecommerce | write_heavy | GET:60,120,220 | POST:15,30,45 | PUT:10,20,30' ) . '<br>';
		echo esc_html( '/jwt-auth/* | auth | auth | POST:5,10,15' );
		echo '</code>';	

		if ( ! $is_pro ) {
			$license_url = admin_url( 'admin.php?page=aegiswaf&tab=license' );
		} else {
			$list = ! empty( $wp['rest_route_overrides_list'] ) && is_array( $wp['rest_route_overrides_list'] )
				? $wp['rest_route_overrides_list']
				: [];

			echo '<p style="margin-top:10px;"><label>';
			echo '<input type="checkbox" name="wp_rest_route_auto_profiles" value="1" ' . checked( ! empty( $wp['rest_route_auto_profiles'] ), true, false ) . '>';
			echo ' Enable auto-profiles (recommended)</label></p>';

			echo '<h3 style="margin:14px 0 8px;">Add / Edit Rule</h3>';
			echo '<input type="hidden" id="wp_rest_override_action" name="wp_rest_override_action" value="add">';
			echo '<input type="hidden" id="wp_rest_override_index" name="wp_rest_override_index" value="-1">';

			echo '<textarea id="wp_rest_override_line" name="wp_rest_override_line" rows="3" style="width:100%; font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, monospace;" placeholder="/wp/v2/users* | user_enum | sensitive | GET:10,20,30"></textarea>';
			echo '<p style="margin:8px 0 0 0;">';
			echo '<button type="button" class="button" id="wp_rest_override_reset">Reset</button> ';
			echo '<button type="submit" class="button button-primary" onclick="document.getElementById(\'wp_rest_override_action\').value = (document.getElementById(\'wp_rest_override_index\').value >= 0 ? \'update\' : \'add\');">Save Rule</button>';
			echo '</p>';

			echo '<h3 style="margin:16px 0 8px;">Saved Rules</h3>';
			echo '<table class="widefat striped" style="width:100%;">';
			echo '<thead><tr>';
			echo '<th style="width:55%;">Rule</th>';
			echo '<th style="width:15%;">Category</th>';
			echo '<th style="width:15%;">Profile</th>';
			echo '<th style="width:15%;">Actions</th>';
			echo '</tr></thead><tbody>';

			if ( empty( $list ) ) {
				echo '<tr><td colspan="4"><em>No rules saved yet.</em></td></tr>';
			} else {
				foreach ( $list as $i => $row ) {
					$raw = (string) ( $row['raw'] ?? '' );
					$cat = (string) ( $row['category'] ?? '' );
					$pro = (string) ( $row['profile'] ?? '' );

					echo '<tr>';
					echo '<td><code>' . esc_html( $raw ) . '</code></td>';
					echo '<td>' . esc_html( $cat ) . '</td>';
					echo '<td>' . esc_html( $pro ) . '</td>';
					echo '<td>';
					echo '<a href="#" class="aegiswaf-edit-rule" data-index="' . esc_attr( (string) $i ) . '" data-raw="' . esc_attr( $raw ) . '">Edit</a> | ';
					echo '<a href="#" class="aegiswaf-del-rule" data-index="' . esc_attr( (string) $i ) . '">Delete</a>';
					echo '</td>';
					echo '</tr>';
				}
			}
			echo '</tbody></table>';

			echo '<script>
			(function(){
			  var line = document.getElementById("wp_rest_override_line");
			  var idx  = document.getElementById("wp_rest_override_index");
			  var act  = document.getElementById("wp_rest_override_action");
			  var reset = document.getElementById("wp_rest_override_reset");

			  document.querySelectorAll(".aegiswaf-edit-rule").forEach(function(a){
				a.addEventListener("click", function(e){
				  e.preventDefault();
				  line.value = this.getAttribute("data-raw") || "";
				  idx.value = this.getAttribute("data-index") || "-1";
				  act.value = "update";
				  line.scrollIntoView({behavior:"smooth",block:"center"});
				  line.focus();
				});
			  });

			  document.querySelectorAll(".aegiswaf-del-rule").forEach(function(a){
				a.addEventListener("click", function(e){
				  e.preventDefault();
				  if(!confirm("Delete this rule?")) return;
				  idx.value = this.getAttribute("data-index") || "-1";
				  act.value = "delete";
				  // submit the main form
				  this.closest("form").submit();
				});
			  });

			  reset.addEventListener("click", function(e){
				e.preventDefault();
				line.value = "";
				idx.value = "-1";
				act.value = "add";
			  });
			})();
			</script>';

			echo '<p class="description" style="margin-top:8px;">Wildcards: use <code>*</code> in the pattern. Categories/profiles are used for reporting + presets. Methods supported: GET, POST, PUT, PATCH, DELETE, HEAD, OPTIONS.</p>';
		}

		echo '</div>';
		
        echo '<p style="margin-top:14px;"><button class="button button-primary" name="aegiswaf_save_api_shield" value="1">Save API Shield</button></p>';
        echo '</form>';
		echo '<h2>Definition of each part in:</h2>';
		echo '<p style="margin:10px 0 6px 0;"><strong>Format (one per line)</strong></p>';
		echo '<code style="display:block; padding:8px; background:#f6f7f7; border:1px solid #e5e5e5;">';
		echo esc_html( 'pattern | category | profile | METHOD:challenge,rate,block | METHOD:challenge,rate,block ...' );
		echo '</code>';
		echo '<p class="description"><b>Pattern</b> - The REST route match pattern (wildcards allowed with *). AegisWAF compares the request route (example: /wp/v2/users) to this pattern. If it matches, the rule is considered for enforcement.
			Example: /wp/v2/users* matches /wp/v2/users, /wp/v2/users/1, etc.</p>';

		echo '<p class="description"><b>Category</b> - A human-readable label used for grouping/reporting (and for explaining events in logs). Think of it as “what kind of endpoint is this?”
			Examples: user_enum, ecommerce, auth.</p>';

		echo '<p class="description"><b>Profile</b> - A second label describing the risk/behavior style of the route (used for reporting and “auto-profile” logic). Think of it as “how should this typically behave?”
			Examples: sensitive, write_heavy, auth.</p>';

		echo '<p class="description"><b>METHOD:challenge,rate,block (repeatable per method)</b> - Per-HTTP-method thresholds for that route pattern. When the request method matches (GET/POST/PUT/etc.), AegisWAF uses these numbers within the configured REST time window:</p>';
			echo '<p class="description">- challenge: at this count, start returning a “challenge” action (your plugin’s challenge response)</p>';
			echo '<p class="description">- rate: at this count, start rate-limiting responses</p>';
			echo '<p class="description">- block: at this count, block the requests</p>';
			echo '<p class="description">Example: GET:10,20,30 means within the window: challenge at 10, rate-limit at 20, block at 30.</p>';
		echo '<p class="description"><b>Why set this up / benefit:</b> it’s the difference between “one blunt rule for all APIs” and “smart API protection”—tight where attackers hit (auth/users/orders), relaxed where real traffic is expected, with better logs that show the matched pattern/category/profile when something is challenged/rate-limited/blocked.</p>';
		
    }

	private static function get_api_shield_visuals( int $hours = 24 ) : array {
		global $wpdb;

		$hours = max( 1, min( 168, $hours ) ); // 1h..7d
		$cache_key = 'aegiswaf_api_shield_viz_' . $hours;

		$cached = get_transient( $cache_key );
		if ( is_array( $cached ) ) {
			return $cached;
		}

		$table = class_exists( 'AegisWAF_Logger' ) ? AegisWAF_Logger::table_name() : ( $wpdb->prefix . 'aegiswaf_logs' );
		$since = gmdate( 'Y-m-d H:i:s', time() - ( $hours * HOUR_IN_SECONDS ) );

		$rows = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT action_taken, route, method, ip, created_at FROM {$table}
				 WHERE category = %s AND created_at >= %s
				 ORDER BY id DESC
				 LIMIT 2000",
				'api_shield',
				$since
			),
			ARRAY_A
		);

		if ( ! is_array( $rows ) ) { $rows = []; }

		$action_map = [
			'allow'      => 'Allow',
			'challenge'  => 'Challenge',
			'rate_limit' => 'Rate-limit',
			'block'      => 'Block',
		];
		$action_counts = array_fill_keys( array_keys( $action_map ), 0 );

		$timeline = [];
		$now = time();
		for ( $i = $hours - 1; $i >= 0; $i-- ) {
			$t = $now - ( $i * HOUR_IN_SECONDS );
			$k = gmdate( 'Y-m-d H:00', $t );
			$timeline[ $k ] = 0;
		}

		$route_counts = [];

		$ip_stats = []; // ip => ['block'=>0,'rate_limit'=>0,'total'=>0]

		foreach ( $rows as $r ) {
			$act = strtolower( (string) ( $r['action_taken'] ?? '' ) );
			$act = $act === 'ratelimit' ? 'rate_limit' : $act;

			if ( isset( $action_counts[ $act ] ) ) {
				$action_counts[ $act ]++;
			} else {
				$action_counts['allow']++;
			}

			$created = (string) ( $r['created_at'] ?? '' );
			if ( $created !== '' ) {
				$h = substr( $created, 0, 13 ) . ':00';
				if ( isset( $timeline[ $h ] ) ) {
					$timeline[ $h ]++;
				}
			}

			$route = (string) ( $r['route'] ?? '' );
			if ( $route !== '' ) {
				if ( ! isset( $route_counts[ $route ] ) ) { $route_counts[ $route ] = 0; }
				$route_counts[ $route ]++;
			}

			$ip = (string) ( $r['ip'] ?? '' );
			if ( $ip !== '' ) {
				if ( ! isset( $ip_stats[ $ip ] ) ) {
					$ip_stats[ $ip ] = [ 'block' => 0, 'rate_limit' => 0, 'total' => 0 ];
				}
				$ip_stats[ $ip ]['total']++;

				if ( $act === 'block' ) {
					$ip_stats[ $ip ]['block']++;
				} elseif ( $act === 'rate_limit' ) {
					$ip_stats[ $ip ]['rate_limit']++;
				}
			}
		}

		arsort( $route_counts );
		$top_routes = array_slice( $route_counts, 0, 7, true );

		uasort( $ip_stats, function( $a, $b ) {
			return (int) ( $b['total'] ?? 0 ) <=> (int) ( $a['total'] ?? 0 );
		} );
		$top_ips = array_slice( $ip_stats, 0, 10, true );

		$points = [];
		foreach ( $top_ips as $ip => $st ) {
			$blocks = (int) ( $st['block'] ?? 0 );
			$rate   = (int) ( $st['rate_limit'] ?? 0 );
			$total  = (int) ( $st['total'] ?? 0 );

			$points[] = [
				'x' => $blocks,
				'y' => $rate,
				'r' => max( 3, min( 30, $total ) ),
				'label' => $ip,
			];
		}

		$payload = [
			'actions' => [
				'labels' => array_values( $action_map ),
				'values' => [
					(int) $action_counts['allow'],
					(int) $action_counts['challenge'],
					(int) $action_counts['rate_limit'],
					(int) $action_counts['block'],
				],
			],
			'timeline' => [
				'labels' => array_keys( $timeline ),
				'values' => array_values( $timeline ),
			],
			'routes' => [
				'labels' => array_keys( $top_routes ),
				'values' => array_values( $top_routes ),
			],
			'offenders' => [
				'points' => $points,
			],
			'meta' => [
				'hours' => $hours,
				'since_gmt' => $since,
				'rows_used' => count( $rows ),
			],
		];

		set_transient( $cache_key, $payload, 60 );

		return $payload;
	}
		private static function parse_route_override_line( string $line ) {
		$line = trim( $line );
		if ( $line === '' ) {
			return new WP_Error( 'empty', 'Rule line is empty.' );
		}

		$parts = array_map( 'trim', explode( '|', $line ) );
		if ( count( $parts ) < 4 ) {
			return new WP_Error( 'format', 'Invalid format. Expected: pattern | category | profile | METHOD:challenge,rate,block ...' );
		}

		$pattern  = trim( $parts[0] );
		$category = trim( $parts[1] );
		$profile  = trim( $parts[2] );

		if ( $pattern === '' || $category === '' || $profile === '' ) {
			return new WP_Error( 'required', 'Pattern, category, and profile are required.' );
		}

		$methods = [];
		for ( $i = 3; $i < count( $parts ); $i++ ) {
			$seg = trim( $parts[$i] );
			if ( $seg === '' ) { continue; }

			if ( ! preg_match( '/^([A-Z]+)\s*:\s*([0-9]+)\s*,\s*([0-9]+)\s*,\s*([0-9]+)\s*$/', strtoupper( $seg ), $m ) ) {
				return new WP_Error( 'method', 'Bad method segment: ' . $seg . ' (expected METHOD:challenge,rate,block)' );
			}
			$methods[ $m[1] ] = [
				'challenge' => (int) $m[2],
				'rate'      => (int) $m[3],
				'block'     => (int) $m[4],
			];
		}

		if ( empty( $methods ) ) {
			return new WP_Error( 'methods', 'At least one METHOD:challenge,rate,block segment is required.' );
		}

		return [
			'raw'      => $line,
			'pattern'  => $pattern,
			'category' => $category,
			'profile'  => $profile,
			'methods'  => $methods,
		];
	}

}
