<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Page_License {

    protected function log_email_event( $to, $subject, $success, $context = 'test' ) : void {
        $logs = get_option( 'aegiswaf_email_log', array() );
        if ( ! is_array( $logs ) ) {
            $logs = array();
        }

        $logs[] = array(
            'time'    => current_time( 'mysql' ),
            'to'      => sanitize_text_field( (string) $to ),
            'subject' => sanitize_text_field( (string) $subject ),
            'status'  => $success ? 'sent' : 'failed',
            'context' => sanitize_key( (string) $context ),
        );

        if ( count( $logs ) > 50 ) {
            $logs = array_slice( $logs, -50 );
        }

        update_option( 'aegiswaf_email_log', $logs, false );
    }

    public function render() : void {
        if ( ! current_user_can( 'manage_options' ) ) { return; }

        $lm   = new AegisWAF_License_Manager();
        $data = $lm->get_license_data();

        $notice = '';

        if ( isset( $_POST['aegiswaf_activate_license'] ) ) {
            check_admin_referer( 'aegiswaf_license_action' );
            $key   = isset( $_POST['license_key'] ) ? (string) wp_unslash( $_POST['license_key'] ) : '';
            $email = isset( $_POST['license_email'] ) ? (string) wp_unslash( $_POST['license_email'] ) : '';
            $res   = $lm->activate_license( $key, $email );
            if ( is_array( $res ) ) {
                $notice = (string) ( $res['message'] ?? '' );
            }
        } elseif ( isset( $_POST['aegiswaf_deactivate_license'] ) ) {
            check_admin_referer( 'aegiswaf_license_action' );
            $res = $lm->deactivate_license();
            if ( is_array( $res ) ) {
                $notice = (string) ( $res['message'] ?? '' );
            }
        }

        $is_pro = AegisWAF_Features::is_pro();
        $status = strtoupper( (string) ( $data['status'] ?? 'inactive' ) );
        $plan   = (string) ( $data['plan'] ?? '—' );
        $exp    = (string) ( $data['expires_at'] ?? '—' );
		if ( (string) ( $data['status'] ?? '' ) === 'email_mismatch' ) {
			echo '<div style="margin:12px 0;padding:12px;border:1px solid #d63638;background:#fff5f5;border-radius:8px;">';
			echo '<strong style="color:#d63638;">Email mismatch</strong><br>';
			echo 'The email returned by the license server does not match the email you entered. PRO features remain disabled.';
			if ( ! empty( $data['server_email'] ) ) {
				echo '<br><code>Server email:</code> ' . esc_html( (string) $data['server_email'] );
			}
			echo '</div>';
		}

        $current_tab = isset( $_GET['license_tab'] ) ? sanitize_key( wp_unslash( $_GET['license_tab'] ) ) : 'license';

		$current_page = isset($_GET['page']) ? sanitize_key( wp_unslash($_GET['page']) ) : 'aegiswaf-license';

		$page_url = admin_url( 'admin.php?page=' . $current_page );

		if ( isset($_GET['tab']) ) {
			$main_tab = sanitize_key( wp_unslash($_GET['tab']) );
			if ( $main_tab !== '' ) {
				$page_url = add_query_arg( 'tab', $main_tab, $page_url );
			}
		}

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__( 'AegisWAF License & Upgrades', 'aegiswaf' ) . '</h1>';

        ?>
        <h2 class="nav-tab-wrapper">
			<a href="<?php echo esc_url( add_query_arg( 'license_tab', 'license', $page_url ) ); ?>" class="nav-tab <?php echo ( 'license' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
				<?php esc_html_e( 'License', 'aegiswaf' ); ?>
			</a>
			<a href="<?php echo esc_url( add_query_arg( 'license_tab', 'comparison', $page_url ) ); ?>" class="nav-tab <?php echo ( 'comparison' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
				<?php esc_html_e( 'Free vs Pro Comparison', 'aegiswaf' ); ?>
			</a>
			<a href="<?php echo esc_url( add_query_arg( 'license_tab', 'settings', $page_url ) ); ?>" class="nav-tab <?php echo ( 'settings' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
				<?php esc_html_e( 'Email Settings', 'aegiswaf' ); ?>
			</a>
        </h2>
        <?php

        if ( ! empty( $notice ) ) {
            echo '<div class="notice notice-info is-dismissible"><p>' . esc_html( $notice ) . '</p></div>';
        }

        if ( 'license' === $current_tab ) {
            $this->render_tab_license( $lm, $data, $status, $plan, $exp, $is_pro );
        } elseif ( 'settings' === $current_tab ) {
            $this->render_tab_email_settings();
        } else { // comparison
            $this->render_tab_comparison( $is_pro );
        }

        echo '</div>';
    }

    protected function render_tab_license( AegisWAF_License_Manager $lm, array $data, string $status, string $plan, string $exp, bool $is_pro ) : void {

        AegisWAF_Pro_Notice::render();

        echo '<p class="description">' . esc_html__( 'The Licensing tab manages your Pro activation. The comparison tab shows what Pro unlocks, and Email Settings configures reliable alert delivery.', 'aegiswaf' ) . '</p>';

        echo '<div class="aegiswaf-card"><h3>' . esc_html__( 'License & Upgrades', 'aegiswaf' ) . '</h3>';
        echo '<div class="aegiswaf-kv">';
        echo '<div class="item"><strong>' . esc_html__( 'Status', 'aegiswaf' ) . '</strong><div>' . esc_html( $status ) . '</div></div>';
        echo '<div class="item"><strong>' . esc_html__( 'Plan', 'aegiswaf' ) . '</strong><div>' . esc_html( $plan ) . '</div></div>';
        echo '<div class="item"><strong>' . esc_html__( 'Expires', 'aegiswaf' ) . '</strong><div>' . esc_html( $exp ) . '</div></div>';
        echo '</div>';

        echo '<form method="post" action="">';
        wp_nonce_field( 'aegiswaf_license_action' );

        $license_key   = isset( $data['license_key'] ) ? (string) $data['license_key'] : '';
        $license_email = isset( $data['license_email'] ) ? (string) $data['license_email'] : '';

        echo '<table class="form-table" style="max-width:900px;">';
        echo '<tr><th scope="row"><label for="license_key">' . esc_html__( 'License Key', 'aegiswaf' ) . '</label></th>';
        echo '<td><input type="text" id="license_key" name="license_key" class="regular-text" value="' . esc_attr( $license_key ) . '" placeholder="XXXX-XXXX-XXXX-XXXX" /></td></tr>';

        echo '<tr><th scope="row"><label for="license_email">' . esc_html__( 'Email', 'aegiswaf' ) . '</label></th>';
        echo '<td><input type="email" id="license_email" name="license_email" class="regular-text" value="' . esc_attr( $license_email ) . '" placeholder="you@example.com" /></td></tr>';
        echo '</table>';

        echo '<p class="submit">';
        echo '<button type="submit" class="button button-primary" name="aegiswaf_activate_license" value="1">' . esc_html__( 'Activate / Save License', 'aegiswaf' ) . '</button> ';
        if ( ! empty( $license_key ) ) {
            echo '<button type="submit" class="button" name="aegiswaf_deactivate_license" value="1">' . esc_html__( 'Deactivate License', 'aegiswaf' ) . '</button>';
        }
        echo '</p>';

        echo '</form>';
        echo '</div>';
    }

    protected function render_tab_comparison( bool $is_pro ) : void {

        $matrix = [
            'Logging & Evidence' => [
				'Core event logging (engine / managed_rule / heuristic / api_shield / bot_control / ddos_shield / endpoint_policy)' => [true, true],
				'Log viewer: search + filters (route contains, method, event type/category, action, IP)' => [true, true],
				'Date range filtering (From / To)' => [true, true],
				'Rows-per-page setting' => [true, true],

				'Alert Settings: create alerts (title, keyword string, recipients)' => [false, true],
				'Alert Settings: edit/delete existing alerts' => [false, true],
				'“Alerts only” filtering' => [true, true],

				'Retention days setting (auto-delete older logs) + manual cleanup button' => [false, true],

				'Attack Story (Last 24h): inspected requests, events, unique IPs/routes, top detected types/rules' => [true, true],
				'Attack Story Deep Dive Reports' => [false, true],
				'Attack Story Narrative analysis generation (incident-style explanation)' => [false, true],

				'Visual Intelligence: 8 charts across WAF, API Shield, Bot, DDoS, Attack Story' => [true, true],
            ],
            'WAF Rules' => [

				'Custom Rules engine (path pattern matching with wildcards * and ?)' => [true, true],
				'Rule methods: ANY / GET / POST / PUT / PATCH / DELETE / HEAD / OPTIONS' => [true, true],
				'Rule actions: Log / Block / Allow' => [true, true],
				'Rule “Contains (all)” matching (one token per line; ALL must match)' => [true, true],
				'Regex conditions (one regex per line) — enforced only in PRO' => [false, true],
				'FREE enforcement cap: only first 5 enabled rules are evaluated (hard cap in engine)' => [true, true],

				'Managed Rules engine enabled toggle' => [true, true],
				'Inspection scope controls: Inspect headers / cookies / body' => [true, true],
				'Max request body bytes for inspection (body is skipped if too large)' => [true, true],
				'Managed Rule categories (always available): SQLi / XSS / Path Traversal' => [true, true],
				'Managed Rule categories (PRO-only enforcement): RCE / Command Injection / SSRF / File Upload' => [false, true],
				'Sensitivity mode: balanced / strict (signature pack selection)' => [true, true],
				'Action mode: FREE forced to LOG; PRO can use Log / Block / Challenge / Rate Limit' => [true, true],

				'Heuristic layer evaluation (request scoring layer runs in engine)' => [true, true],
				'Heuristic tuning (score threshold, encoding threshold, meta threshold, token length)' => [false, true],
				'JS Challenge Token support (challenge + TTL settings)' => [true, true],

				'Endpoint Policies framework (per-path policy match + applied logging)' => [true, true],
				'Endpoint Policies: override action + categories + thresholds per path (UI uses PRO Mode)' => [false, true],

				'Rule Sets presets: WP Login Protection / XML-RPC Hard Block / WP-JSON Throttle' => [false, true],
				'WP Default “Revert” snapshot behavior after applying presets' => [false, true],
				'Import/Export WAF Rules JSON (move rules between sites)' => [false, true],

				'Visual Intelligence charts (7-day trend, category breakdown, top routes, action outcomes)' => [true, true],
            ],
            'Bot Control' => [
				'Enable Bot Control (front requests + REST requests enforcement)' => [true, true],
				'Default rate limit per minute (enforced per IP + path + minute bucket)' => [true, true],
				'Bad User-Agent blocking (token contains match; blocks)' => [true, true],
				'Per-path thresholds (wildcards supported; per-path limit/min override)' => [true, true],
				'Bot Control logging (allow + rate_limit events with matched pattern and scope)' => [true, true],

				'Behavioral scoring (PRO-only scoring + threshold enforcement)' => [false, true],
				'IP / CIDR blocklist enforcement (PRO-only)' => [false, true],
				'Country blocking enforcement (PRO-only; requires provider)' => [false, true],
				'ASN blocking enforcement (PRO-only; requires provider)' => [false, true],
				'Geo/ASN provider framework (hook/provider selection support)' => [false, true],
				'MaxMind MMDB path setting (framework for provider usage)' => [false, true],

				'Bot analytics charts (24h trend, 7d trend, radar, UA, etc.)' => [true, true],
            ],
            'API Shield' => [

				'Enable WordPress protection layer (master switch)' => [true, true],
				'Enable REST API protection layer (REST inspection enforcement)' => [true, true],
				'Only enforce on unauthenticated REST requests (logged-in bypass)' => [true, true],

				'Block unauthenticated GET access to /wp/v2/users* (user enumeration control)' => [true, true],
				'Require authentication for non-GET REST requests (POST/PUT/PATCH/DELETE) unless allowlisted' => [true, true],
				'Require application/json Content-Type for REST requests with body (POST/PUT/PATCH)' => [true, true],
				'Max REST body size (bytes) enforcement (413 if exceeded; 0 disables)' => [true, true],
				'CORS allowlist enforcement (Origin must match an allowlisted line)' => [true, true],

				'API key header enforcement (header name + expected value; blocks if missing/mismatch)' => [true, true],

				'REST window (seconds)' => [true, true],
				'Progressive enforcement thresholds: Challenge-at / Rate-limit-at / Block-at' => [true, true],
				'REST allowlist (route prefix match; bypasses enforcement)' => [true, true],

				'Advanced Per-Route Controls (pattern | category | profile | per-method thresholds)' => [false, true],
				'Add / Edit / Delete saved per-route rules' => [false, true],
				'Auto-profiles for tagging/reporting (when enabled)' => [false, true],

				'API Shield logging (request_seen, allow, block, override_matched metadata)' => [true, true],
            ],
            'Geo & ASN Protection' => [
                'IP / CIDR Blocklist' => [ false, true ],
                'Country Blocking' => [ false, true ],
                'ASN Blocking' => [ false, true ],
                'TOR / Proxy Signals' => [ false, true ],
            ],
        ];

        echo '<h2>' . esc_html__( 'Free vs Pro Comparison', 'aegiswaf' ) . '</h2>';

        if ( ! $is_pro ) {
            echo '<div class="notice notice-info" style="margin-top:10px;border-left:4px solid #ff9800;">'
                . '<p style="margin:8px 0;"><strong>' . esc_html__( 'Unlock everything in AegisWAF Pro.', 'aegiswaf' ) . '</strong> '
                . esc_html__( 'Get advanced rules, tuning controls, geo/ASN controls, virtual patching, and richer alerting.', 'aegiswaf' )
                . '</p>'
                . '<p style="margin:8px 0;">'
                . '<a href="https://aitechave.com/aegiswaf-pro" target="_blank" class="button button-primary">'
                . esc_html__( 'Upgrade to AegisWAF Pro', 'aegiswaf' )
                . '</a>'
                . '</p>'
                . '</div>';
        }

        echo '<table class="widefat striped" style="max-width:1100px;">';
        echo '<thead><tr>';
        echo '<th style="width:70%">' . esc_html__( 'Feature', 'aegiswaf' ) . '</th>';
        echo '<th style="width:15%;text-align:center;">' . esc_html__( 'Free', 'aegiswaf' ) . '</th>';
        echo '<th style="width:15%;text-align:center;">' . esc_html__( 'Pro', 'aegiswaf' ) . '</th>';
        echo '</tr></thead><tbody>';

        foreach ( $matrix as $section => $features ) {
            echo '<tr><th colspan="3" style="background:#f6f7f7;">' . esc_html( $section ) . '</th></tr>';

            foreach ( $features as $label => $flags ) {
                $free = ! empty( $flags[0] );
                $pro  = ! empty( $flags[1] );

                echo '<tr>';
                echo '<td><strong>' . esc_html( $label ) . '</strong></td>';
                echo '<td style="text-align:center;">' . ( $free ? '<span style="color:#1d7f2e;font-weight:700;">✔</span>' : '<span style="color:#b32d2e;font-weight:700;">✖</span>' ) . '</td>';
                echo '<td style="text-align:center;">' . ( $pro ? '<span style="color:#1d7f2e;font-weight:700;">✔</span>' : '<span style="color:#b32d2e;font-weight:700;">✖</span>' ) . '</td>';
                echo '</tr>';
            }
        }

        echo '</tbody></table>';
    }

    protected function render_tab_email_settings() : void {

        if (
            isset( $_POST['aegiswaf_send_test_email'] )
            && check_admin_referer( 'aegiswaf_mail_settings_save', 'aegiswaf_mail_settings_nonce' )
        ) {
            $to      = get_option( 'admin_email' );
            $subject = __( 'AegisWAF Test Email', 'aegiswaf' );
            $body    = __( 'This is a test email sent using your configured SMTP settings.', 'aegiswaf' );

            $sent = false;

            if ( class_exists( 'AegisWAF_Notifier' ) ) {
                $notifier = new AegisWAF_Notifier();
                if ( method_exists( $notifier, 'send_email_to' ) ) {
                    $sent = (bool) $notifier->send_email_to( $to, $subject, $body );
                } else {
                    $sent = (bool) wp_mail( $to, $subject, $body );
                }
            } else {
                $sent = (bool) wp_mail( $to, $subject, $body );
            }

            $this->log_email_event( $to, $subject, (bool) $sent, 'test' );

            if ( $sent ) {
                echo '<div class="notice notice-success is-dismissible"><p>'
                    . esc_html__( 'Test email sent successfully.', 'aegiswaf' )
                    . '</p></div>';
            } else {
                echo '<div class="notice notice-error is-dismissible"><p>'
                    . esc_html__( 'Test email failed. Check SMTP settings and server connectivity.', 'aegiswaf' )
                    . '</p></div>';
            }
        }

        if (
            isset( $_POST['aegiswaf_mail_settings_save'] )
            && check_admin_referer( 'aegiswaf_mail_settings_save', 'aegiswaf_mail_settings_nonce' )
        ) {
            $settings = array(
                'alert_recipients' => isset( $_POST['as_alert_recipients'] ) ? sanitize_textarea_field( wp_unslash( $_POST['as_alert_recipients'] ) ) : '',

                'smtp_enabled'     => isset( $_POST['as_smtp_enabled'] ) ? 1 : 0,
                'smtp_host'        => isset( $_POST['as_smtp_host'] ) ? sanitize_text_field( wp_unslash( $_POST['as_smtp_host'] ) ) : '',
                'smtp_port'        => isset( $_POST['as_smtp_port'] ) ? absint( wp_unslash( $_POST['as_smtp_port'] ) ) : 0,
                'smtp_username'    => isset( $_POST['as_smtp_username'] ) ? sanitize_text_field( wp_unslash( $_POST['as_smtp_username'] ) ) : '',
                'smtp_password'    => isset( $_POST['as_smtp_password'] ) ? sanitize_text_field( wp_unslash( $_POST['as_smtp_password'] ) ) : '',
                'smtp_encryption'  => isset( $_POST['as_smtp_encryption'] ) ? sanitize_key( wp_unslash( $_POST['as_smtp_encryption'] ) ) : '',
                'smtp_from_email'  => isset( $_POST['as_smtp_from_email'] ) ? sanitize_email( wp_unslash( $_POST['as_smtp_from_email'] ) ) : '',
                'smtp_from_name'   => isset( $_POST['as_smtp_from_name'] ) ? sanitize_text_field( wp_unslash( $_POST['as_smtp_from_name'] ) ) : '',
                'smtp_reply_to'    => isset( $_POST['as_smtp_reply_to'] ) ? sanitize_email( wp_unslash( $_POST['as_smtp_reply_to'] ) ) : '',
                'smtp_auth'        => isset( $_POST['as_smtp_auth'] ) ? 1 : 0,
            );

            update_option( 'aegiswaf_mail_settings', $settings, false );

            echo '<div class="notice notice-success is-dismissible"><p>'
                . esc_html__( 'Mail settings saved.', 'aegiswaf' )
                . '</p></div>';
        }

        $mail_settings    = get_option( 'aegiswaf_mail_settings', array() );
        $alert_recipients = $mail_settings['alert_recipients'] ?? '';
        ?>
        <h2><?php esc_html_e( 'Mail Delivery Settings (SMTP)', 'aegiswaf' ); ?></h2>

        <p style="max-width:900px; line-height:1.6;">
            <?php esc_html_e( 'Shared hosting often blocks or throttles outbound mail. Configure SMTP so AegisWAF can deliver security alerts reliably.', 'aegiswaf' ); ?>
        </p>

        <form method="post" action="">
            <?php wp_nonce_field( 'aegiswaf_mail_settings_save', 'aegiswaf_mail_settings_nonce' ); ?>

            <h3><?php esc_html_e( 'Global Alert Recipients', 'aegiswaf' ); ?></h3>

            <table class="form-table" style="max-width:900px;">
                <tr>
                    <th scope="row">
                        <label for="as_alert_recipients"><?php esc_html_e( 'Alert Recipient Emails', 'aegiswaf' ); ?></label>
                    </th>
                    <td>
                        <textarea id="as_alert_recipients" name="as_alert_recipients" rows="3" class="large-text"
                                  placeholder="security@yourdomain.com&#10;it@yourdomain.com"><?php echo esc_textarea( (string) $alert_recipients ); ?></textarea>
                        <p class="description"><?php esc_html_e( 'One email per line. These recipients receive alert notifications (where applicable).', 'aegiswaf' ); ?></p>
                    </td>
                </tr>
            </table>

            <hr style="margin:25px 0;" />

            <h3 style="margin-top:25px;"><?php esc_html_e( 'SMTP (Outgoing Mail)', 'aegiswaf' ); ?></h3>

            <table class="form-table" style="max-width:900px;">
                <tr>
                    <th scope="row"><?php esc_html_e( 'Enable SMTP', 'aegiswaf' ); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="as_smtp_enabled" value="1" <?php checked( (int) ( $mail_settings['smtp_enabled'] ?? 0 ), 1 ); ?> />
                            <?php esc_html_e( 'Use an external SMTP server for sending alerts', 'aegiswaf' ); ?>
                        </label>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="as_smtp_host"><?php esc_html_e( 'SMTP Host', 'aegiswaf' ); ?></label></th>
                    <td><input type="text" id="as_smtp_host" name="as_smtp_host" class="regular-text"
                               value="<?php echo esc_attr( (string) ( $mail_settings['smtp_host'] ?? '' ) ); ?>"
                               placeholder="smtp.yourdomain.com" /></td>
                </tr>

                <tr>
                    <th scope="row"><label for="as_smtp_port"><?php esc_html_e( 'SMTP Port', 'aegiswaf' ); ?></label></th>
                    <td><input type="number" id="as_smtp_port" name="as_smtp_port" class="small-text"
                               value="<?php echo esc_attr( (string) ( $mail_settings['smtp_port'] ?? '' ) ); ?>"
                               placeholder="587" /></td>
                </tr>

                <tr>
                    <th scope="row"><label for="as_smtp_encryption"><?php esc_html_e( 'Encryption', 'aegiswaf' ); ?></label></th>
                    <td>
                        <select id="as_smtp_encryption" name="as_smtp_encryption">
                            <?php
                            $enc  = (string) ( $mail_settings['smtp_encryption'] ?? '' );
                            $opts = array(
                                ''         => __( 'None', 'aegiswaf' ),
                                'ssl'      => __( 'SSL', 'aegiswaf' ),
                                'tls'      => __( 'TLS', 'aegiswaf' ),
                                'starttls' => __( 'STARTTLS', 'aegiswaf' ),
                            );
                            foreach ( $opts as $val => $label ) {
                                echo '<option value="' . esc_attr( $val ) . '" ' . selected( $enc, $val, false ) . '>' . esc_html( $label ) . '</option>';
                            }
                            ?>
                        </select>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e( 'SMTP Auth', 'aegiswaf' ); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="as_smtp_auth" value="1" <?php checked( (int) ( $mail_settings['smtp_auth'] ?? 0 ), 1 ); ?> />
                            <?php esc_html_e( 'Use SMTP authentication (recommended)', 'aegiswaf' ); ?>
                        </label>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><label for="as_smtp_username"><?php esc_html_e( 'SMTP Username', 'aegiswaf' ); ?></label></th>
                    <td><input type="text" id="as_smtp_username" name="as_smtp_username" class="regular-text"
                               value="<?php echo esc_attr( (string) ( $mail_settings['smtp_username'] ?? '' ) ); ?>" /></td>
                </tr>

                <tr>
                    <th scope="row"><label for="as_smtp_password"><?php esc_html_e( 'SMTP Password', 'aegiswaf' ); ?></label></th>
                    <td><input type="password" id="as_smtp_password" name="as_smtp_password" class="regular-text"
                               value="<?php echo esc_attr( (string) ( $mail_settings['smtp_password'] ?? '' ) ); ?>" autocomplete="new-password" /></td>
                </tr>

                <tr>
                    <th scope="row"><label for="as_smtp_from_name"><?php esc_html_e( 'From Name', 'aegiswaf' ); ?></label></th>
                    <td><input type="text" id="as_smtp_from_name" name="as_smtp_from_name" class="regular-text"
                               value="<?php echo esc_attr( (string) ( $mail_settings['smtp_from_name'] ?? '' ) ); ?>" placeholder="AegisWAF" /></td>
                </tr>

                <tr>
                    <th scope="row"><label for="as_smtp_from_email"><?php esc_html_e( 'From Email', 'aegiswaf' ); ?></label></th>
                    <td><input type="email" id="as_smtp_from_email" name="as_smtp_from_email" class="regular-text"
                               value="<?php echo esc_attr( (string) ( $mail_settings['smtp_from_email'] ?? '' ) ); ?>" placeholder="no-reply@yourdomain.com" /></td>
                </tr>

                <tr>
                    <th scope="row"><label for="as_smtp_reply_to"><?php esc_html_e( 'Reply-To Email', 'aegiswaf' ); ?></label></th>
                    <td><input type="email" id="as_smtp_reply_to" name="as_smtp_reply_to" class="regular-text"
                               value="<?php echo esc_attr( (string) ( $mail_settings['smtp_reply_to'] ?? '' ) ); ?>" placeholder="support@yourdomain.com" /></td>
                </tr>
            </table>

            <p class="submit">
                <button type="submit" class="button button-primary" name="aegiswaf_mail_settings_save" value="1">
                    <?php esc_html_e( 'Save Mail Settings', 'aegiswaf' ); ?>
                </button>

                <button type="submit" class="button" name="aegiswaf_send_test_email" value="1">
                    <?php esc_html_e( 'Send Test Email', 'aegiswaf' ); ?>
                </button>
            </p>
        </form>

        <hr style="margin:30px 0;" />

        <h3><?php esc_html_e( 'Email Delivery Log (Last 50)', 'aegiswaf' ); ?></h3>

        <?php
        $email_logs = get_option( 'aegiswaf_email_log', array() );
        if ( ! is_array( $email_logs ) ) {
            $email_logs = array();
        }

        if ( empty( $email_logs ) ) : ?>
            <p class="description"><?php esc_html_e( 'No email delivery logs yet. Use "Send Test Email" to generate a log entry.', 'aegiswaf' ); ?></p>
        <?php else : ?>
            <table class="widefat striped" style="max-width:900px;">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Time', 'aegiswaf' ); ?></th>
                        <th><?php esc_html_e( 'Recipient', 'aegiswaf' ); ?></th>
                        <th><?php esc_html_e( 'Subject', 'aegiswaf' ); ?></th>
                        <th><?php esc_html_e( 'Status', 'aegiswaf' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ( array_reverse( $email_logs ) as $row ) : ?>
                        <tr>
                            <td><?php echo esc_html( (string) ( $row['time'] ?? '' ) ); ?></td>
                            <td><?php echo esc_html( (string) ( $row['to'] ?? '' ) ); ?></td>
                            <td><?php echo esc_html( (string) ( $row['subject'] ?? '' ) ); ?></td>
                            <td>
                                <?php
                                $st = (string) ( $row['status'] ?? '' );
                                echo ( 'sent' === $st )
                                    ? '<span style="color:#1d7f2e;font-weight:700;">' . esc_html__( 'Sent', 'aegiswaf' ) . '</span>'
                                    : '<span style="color:#b32d2e;font-weight:700;">' . esc_html__( 'Failed', 'aegiswaf' ) . '</span>';
                                ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif;
    }

}