<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Challenge {

    const COOKIE = 'aegiswaf_chal';

    public static function enabled() : bool {
        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];
        $c  = is_array( $mr['challenge'] ?? null ) ? $mr['challenge'] : [];
        return ! empty( $c['enabled'] );
    }

    public static function ttl() : int {
        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];
        $c  = is_array( $mr['challenge'] ?? null ) ? $mr['challenge'] : [];
        $ttl = isset( $c['ttl_seconds'] ) ? (int) $c['ttl_seconds'] : 1800;
        if ( $ttl < 300 ) { $ttl = 300; }
        if ( $ttl > 86400 ) { $ttl = 86400; }
        return $ttl;
    }

    public static function token_for( string $ip ) : string {
        $ts = time();
        $ttl = self::ttl();
        $bucket = (int) floor( $ts / $ttl );
        $salt = wp_salt( 'aegiswaf_challenge' );
        $sig = hash_hmac( 'sha256', $ip . '|' . $bucket, $salt );
        return $bucket . '.' . $sig;
    }

    public static function verify( string $ip ) : bool {
        $cookie = isset( $_COOKIE[ self::COOKIE ] ) ? (string) $_COOKIE[ self::COOKIE ] : '';
        if ( $cookie === '' ) { return false; }
        $parts = explode( '.', $cookie, 2 );
        if ( count( $parts ) !== 2 ) {
            if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                AegisWAF_Logger::log( '(challenge)', 'GET', 'challenge', 'cookie_invalid', [
                    'ip'     => $ip,
                    'cookie' => $cookie,
                ] );
            }
            error_log( '[AegisWAF] Challenge cookie invalid format — cannot verify.' );
            return false;
        }		
        $parts = explode( '.', $cookie, 2 );
        if ( count( $parts ) !== 2 ) { return false; }
        $bucket = (int) $parts[0];
        $sig = (string) $parts[1];
        $ttl = self::ttl();
        $salt = wp_salt( 'aegiswaf_challenge' );
        $cur = (int) floor( time() / $ttl );
        foreach ( [ $cur, $cur - 1 ] as $b ) {
            $exp = hash_hmac( 'sha256', $ip . '|' . $b, $salt );
            if ( hash_equals( $exp, $sig ) && $bucket === $b ) {
                return true;
            }
        }
        return false;
    }
    public static function render_page( string $message = 'Challenge required' ) : void {
        $ip  = AegisWAF_Utils::client_ip();
        $ttl = self::ttl();
        $tok = self::token_for( $ip );

        if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
            AegisWAF_Logger::log( (string) ( $_SERVER['REQUEST_URI'] ?? '' ), (string) ( $_SERVER['REQUEST_METHOD'] ?? 'GET' ), 'challenge', 'issued', [
                'ip'  => $ip,
                'ttl' => $ttl,
            ] );
        }

        header( 'Content-Type: text/html; charset=utf-8' );
        status_header( 403 );

        $cookie_str = self::COOKIE . '=' . $tok . '; path=/; max-age=' . $ttl . '; samesite=lax';
        $cookie_js  = json_encode( $cookie_str );

        echo '<!doctype html><html><head><meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">';
        echo '<title>AegisWAF Challenge</title></head><body style="font-family:system-ui;margin:40px">';
        echo '<h2>AegisWAF</h2><p>' . esc_html( $message ) . '</p>';
        echo '<p>Verifying your browser…</p>';
        echo '<script>';
        echo 'document.cookie = ' . $cookie_js . ';';
        echo 'setTimeout(function(){ location.reload(); }, 400);';
        echo '</script>';
        echo '<noscript><p>Please enable JavaScript and refresh.</p></noscript>';
        echo '</body></html>';
        exit;
    }

}
