<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Heuristics {

    public static function evaluate( array $payload, array $ctx ) : ?array {
        $hay = (string) ( $payload['norm'] ?? '' );
        if ( $hay === '' ) {
            return null;
        }

        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];
        $h  = is_array( $mr['heuristics'] ?? null ) ? $mr['heuristics'] : [];

        if ( empty( $h['enabled'] ) ) {
            return null;
        }
        if ( ! isset( $h['score_threshold'] ) ) {
            if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                AegisWAF_Logger::log( 'engine', 'SCAN', 'heuristics', 'config_defaulting', [
                    'missing' => 'score_threshold',
                    'default' => 40,
                ] );
            }
            error_log( '[AegisWAF] Heuristics enabled but score_threshold missing — defaulting to 40.' );
        }

        $score   = 0;
        $reasons = [];

        $pct = substr_count( $hay, '%' );
        if ( $pct >= (int) ( $h['pct_threshold'] ?? 8 ) ) {
            $score += 15;
            $reasons[] = 'high_percent_encoding';
        }

        if ( preg_match_all( '/[<>"\'`$;|&]/', $hay ) >= (int) ( $h['meta_threshold'] ?? 12 ) ) {
            $score += 20;
            $reasons[] = 'many_meta_chars';
        }

        if ( substr_count( $hay, '../' ) + substr_count( $hay, '..\\' ) >= 2 ) {
            $score += 20;
            $reasons[] = 'traversal_repeat';
        }

        if ( preg_match( '/\b(file|gopher|dict):\/\//i', $hay ) ) {
            $score += 20;
            $reasons[] = 'dangerous_scheme';
        }

        if ( preg_match( '/\b(or|and)\b\s+\d+=\d+/i', $hay ) ) {
            $score += 25;
            $reasons[] = 'tautology';
        }

        $maxlen = (int) ( $h['token_len_threshold'] ?? 900 );
        if ( $maxlen > 0 ) {
            foreach ( preg_split( '/\s+/', $hay ) as $tok ) {
                if ( strlen( $tok ) >= $maxlen ) {
                    $score += 20;
                    $reasons[] = 'very_long_token';
                    break;
                }
            }
        }

        $min = (int) ( $h['score_threshold'] ?? 40 );
        if ( $score < $min ) {
            return null;
        }

        return [
            'type'     => 'heuristic',
            'category' => 'heuristic',
            'rule_id'  => 'heur_01',
            'score'    => $score,
            'reasons'  => $reasons,
            'action'   => 'investigate',
            'message'  => 'Heuristic anomaly detected',
        ];
    }
}
