<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Logger {

    public static function table_name() : string {
        global $wpdb;
        return $wpdb->prefix . 'aegiswaf_logs';
    }

    public static function activate() : void {
        global $wpdb;
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $charset = $wpdb->get_charset_collate();
        $table = self::table_name();

        $sql = "CREATE TABLE $table (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            event_time DATETIME NOT NULL,
            route VARCHAR(255) NOT NULL,
            method VARCHAR(16) NOT NULL,
            category VARCHAR(80) NOT NULL,
            action_taken VARCHAR(40) NOT NULL,
            details LONGTEXT NULL,
            ip VARCHAR(64) NULL,
            ua VARCHAR(255) NULL,
            PRIMARY KEY  (id),
            KEY event_time (event_time),
            KEY route (route),
            KEY category (category),
            KEY action_taken (action_taken),
            KEY ip (ip)
        ) $charset;";

        dbDelta( $sql );
    }

	public static function create_table() : void {
		self::activate();
	}


	public static function ensure_table() : void {
		global $wpdb;

		$table = self::table_name();

		$exists = $wpdb->get_var(
			$wpdb->prepare(
				'SHOW TABLES LIKE %s',
				$table
			)
		);

		if ( $exists !== $table ) {
			self::activate();
		}
	}

    public static function log( string $route, string $method, string $category, string $action_taken, array $details = [] ) : void {
        global $wpdb;

        $table = self::table_name();

        if ( method_exists( __CLASS__, 'ensure_table' ) ) {
            self::ensure_table();
        }

        $ip = '';
        if ( isset( $details['ip'] ) && is_string( $details['ip'] ) ) {
            $ip = $details['ip'];
        } elseif ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
            $ip = (string) $_SERVER['REMOTE_ADDR'];
        }

        $ua = '';
        if ( isset( $details['ua'] ) && is_string( $details['ua'] ) ) {
            $ua = $details['ua'];
        } elseif ( isset( $_SERVER['HTTP_USER_AGENT'] ) ) {
            $ua = (string) $_SERVER['HTTP_USER_AGENT'];
        }

        $ok = $wpdb->insert(
            $table,
            [
                'event_time'   => gmdate( 'Y-m-d H:i:s' ),
                'route'        => substr( (string) $route, 0, 255 ),
                'method'       => substr( (string) $method, 0, 16 ),
                'category'     => substr( (string) $category, 0, 80 ),
                'action_taken' => substr( (string) $action_taken, 0, 40 ),
                'details'      => wp_json_encode( $details ),
                'ip'           => substr( (string) $ip, 0, 64 ),
                'ua'           => substr( (string) $ua, 0, 255 ),
            ],
            [ '%s','%s','%s','%s','%s','%s','%s','%s' ]
        );

		if ( $ok ) {
			try {
				$settings = AegisWAF_Storage::get_settings();
				$alerts   = $settings['logs']['alerts'] ?? [];

				if ( ! empty( $alerts ) ) {
					$details_str = wp_json_encode( $details );

					$full_line = implode( ' | ', [
						'Time(UTC): ' . gmdate( 'Y-m-d H:i:s' ),
						'Route: ' . $route,
						'Method: ' . $method,
						'Category: ' . $category,
						'Action: ' . $action_taken,
						'IP: ' . (string) $ip,
						'Details: ' . $details_str,
					] );

					foreach ( $alerts as $alert ) {
						$kw_raw = (string) ( $alert['keywords'] ?? '' );
						if ( $kw_raw === '' ) { continue; }

						$keywords = preg_split( '/[\r\n,]+/', $kw_raw );
						$matched  = false;

						foreach ( $keywords as $kw ) {
							$kw = trim( (string) $kw );
							if ( $kw === '' ) { continue; }

							if ( stripos( $full_line, $kw ) !== false ) {
								$matched = true;
								break;
							}
						}

						if ( ! $matched ) { continue; }

						$emails_raw = trim( (string) ( $alert['emails'] ?? '' ) );
						if ( $emails_raw === '' ) {
							$to = get_option( 'admin_email' );
						} else {
							// allow comma or semicolon separated
							$parts = preg_split( '/[;,]+/', $emails_raw );
							$parts = array_filter( array_map( 'trim', (array) $parts ) );
							$parts = array_map( 'sanitize_email', $parts );
							$to    = $parts ? $parts : get_option( 'admin_email' );
						}

						wp_mail(
							$to,
							'[AegisWAF Alert] Keyword Matched',
							$full_line
						);

						break;
					}
				}
			} catch ( Throwable $e ) {
				error_log( '[AegisWAF] Alert email failed: ' . $e->getMessage() );
			}
		}

		if ( ! $ok ) {
			$err = isset( $wpdb->last_error ) ? (string) $wpdb->last_error : '';
			$qry = isset( $wpdb->last_query ) ? (string) $wpdb->last_query : '';
			error_log( '[AegisWAF] Logger insert failed: ' . $err . ' | last_query=' . $qry );
		}
    }

    public static function count_all() : int {
        global $wpdb;
        $table = self::table_name();
        return (int) $wpdb->get_var( "SELECT COUNT(1) FROM $table" );
    }

    public static function count_filtered( array $args ) : int {
        global $wpdb;
        $table = self::table_name();

        $where = self::build_where_sql( $args, $params );
        $sql = "SELECT COUNT(1) FROM $table $where";
        return (int) $wpdb->get_var( $wpdb->prepare( $sql, $params ) );
    }

    public static function fetch_filtered( int $limit, int $offset, array $args ) : array {
        global $wpdb;
        $table = self::table_name();

        $limit  = max( 1, min( 200, $limit ) );
        $offset = max( 0, $offset );

        $where = self::build_where_sql( $args, $params );
        $sql = "SELECT * FROM $table $where ORDER BY id DESC LIMIT %d OFFSET %d";
        $params[] = $limit;
        $params[] = $offset;

        return $wpdb->get_results( $wpdb->prepare( $sql, $params ), ARRAY_A ) ?: [];
    }

    public static function fetch( int $limit, int $offset ) : array {
        return self::fetch_filtered( $limit, $offset, [] );
    }

    public static function delete_older_than_days( int $days ) : int {
        global $wpdb;
        $table = self::table_name();
        $days = max( 1, min( 3650, $days ) );
        return (int) $wpdb->query(
            $wpdb->prepare( "DELETE FROM $table WHERE event_time < (UTC_TIMESTAMP() - INTERVAL %d DAY)", $days )
        );
    }

    public static function is_alert_row( array $row ) : bool {
        $action = strtolower( (string) ( $row['action_taken'] ?? '' ) );
        $cat    = strtolower( (string) ( $row['category'] ?? '' ) );

        if ( in_array( $action, [ 'block', 'challenge', 'rate_limit', 'error' ], true ) ) { return true; }
        if ( in_array( $cat, [ 'error', 'warning', 'alert', 'attack', 'malware' ], true ) ) { return true; }

        return false;
    }

    public static function alert_level( array $row ) : string {
        $action = strtolower( (string) ( $row['action_taken'] ?? '' ) );
        $cat    = strtolower( (string) ( $row['category'] ?? '' ) );

        if ( $action === 'error' || $cat === 'error' ) { return 'critical'; }
        if ( $action === 'block' ) { return 'high'; }
		if ( $cat === 'heuristic' ) { return 'high'; }
        if ( $action === 'challenge' || $action === 'rate_limit' ) { return 'medium'; }
        if ( $cat === 'warning' ) { return 'medium'; }
        return 'info';
    }

    private static function build_where_sql( array $args, &$params ) : string {
        global $wpdb;
        $table = self::table_name();

        $params = [];
        $clauses = [ '1=1' ];

        $q = isset( $args['q'] ) ? trim( (string) $args['q'] ) : '';
        if ( $q !== '' ) {
            $like = '%' . $wpdb->esc_like( $q ) . '%';
            $clauses[] = "(route LIKE %s OR category LIKE %s OR action_taken LIKE %s OR ip LIKE %s OR ua LIKE %s OR details LIKE %s)";
            $params = array_merge( $params, [ $like, $like, $like, $like, $like, $like ] );
        }

        foreach ( [ 'route' => 'route', 'method' => 'method', 'category' => 'category', 'action' => 'action_taken', 'ip' => 'ip' ] as $k => $col ) {
            $v = isset( $args[ $k ] ) ? trim( (string) $args[ $k ] ) : '';
            if ( $v !== '' ) {
                if ( $k === 'route' ) {
                    $like = '%' . $wpdb->esc_like( $v ) . '%';
                    $clauses[] = "$col LIKE %s";
                    $params[] = $like;
                } else {
                    $clauses[] = "$col = %s";
                    $params[] = $v;
                }
            }
        }

        $df = isset( $args['date_from'] ) ? trim( (string) $args['date_from'] ) : '';
        if ( $df !== '' ) {
            $clauses[] = "event_time >= %s";
            $params[] = $df . ' 00:00:00';
        }
        $dt = isset( $args['date_to'] ) ? trim( (string) $args['date_to'] ) : '';
        if ( $dt !== '' ) {
            $clauses[] = "event_time <= %s";
            $params[] = $dt . ' 23:59:59';
        }

        $alerts_only = ! empty( $args['alerts_only'] );
        if ( $alerts_only ) {
            $clauses[] = "(action_taken IN ('block','challenge','rate_limit','error') OR category IN ('error','warning','alert','attack','malware'))";
        }

        return 'WHERE ' . implode( ' AND ', $clauses );
    }
}

add_action( 'plugins_loaded', function () {
    if ( class_exists( 'AegisWAF_Logger' ) ) {
        if ( method_exists( 'AegisWAF_Logger', 'ensure_table' ) ) {
            AegisWAF_Logger::ensure_table();
        } else {
            AegisWAF_Logger::activate();
        }
    }
}, 1 );


