<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Normalizer {

    public static function normalize_front() : array {
        $path = '';
        if ( isset( $_SERVER['REQUEST_URI'] ) ) {
            $path = (string) wp_parse_url( (string) $_SERVER['REQUEST_URI'], PHP_URL_PATH );
        }

        $method = isset( $_SERVER['REQUEST_METHOD'] ) ? (string) $_SERVER['REQUEST_METHOD'] : 'GET';
        $ua     = (string) AegisWAF_Utils::user_agent();
        $ip     = (string) AegisWAF_Utils::client_ip();

        $headers = self::get_headers();
        $cookies = is_array( $_COOKIE ?? null ) ? $_COOKIE : [];
        $query   = is_array( $_GET ?? null ) ? $_GET : [];
        $body    = self::get_body_safe();

        return self::build_payload( $path, $method, $headers, $query, $cookies, $body, $ua, $ip );
    }

    public static function normalize_rest( $request ) : array {
        $route = '';
        if ( is_object( $request ) && method_exists( $request, 'get_route' ) ) {
            $route = (string) $request->get_route();
        }
        if ( $route === '' && isset( $_SERVER['REQUEST_URI'] ) ) {
            $route = (string) wp_parse_url( (string) $_SERVER['REQUEST_URI'], PHP_URL_PATH );
        }

        $method = (string) ( is_object( $request ) && method_exists( $request, 'get_method' ) ? $request->get_method() : ( $_SERVER['REQUEST_METHOD'] ?? 'GET' ) );
        $ua     = (string) AegisWAF_Utils::user_agent();
        $ip     = (string) AegisWAF_Utils::client_ip();

        $headers = self::get_headers();
        $cookies = is_array( $_COOKIE ?? null ) ? $_COOKIE : [];
        $query   = is_array( $_GET ?? null ) ? $_GET : [];

        $body = '';
        if ( is_object( $request ) && method_exists( $request, 'get_body' ) ) {
            $body = (string) $request->get_body();
        } else {
            $body = self::get_body_safe();
        }

        return self::build_payload( $route, $method, $headers, $query, $cookies, $body, $ua, $ip );
    }

    protected static function build_payload( string $path, string $method, array $headers, array $query, array $cookies, string $body, string $ua, string $ip ) : array {
        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];

        $inspect_headers = ! empty( $mr['inspect_headers'] );
        $inspect_cookies = ! empty( $mr['inspect_cookies'] );
        $inspect_body    = ! empty( $mr['inspect_body'] );

        $parts = [];
        $parts[] = 'path=' . $path;
        $parts[] = 'method=' . $method;

        // Query
        if ( ! empty( $query ) ) {
            $parts[] = 'query=' . wp_json_encode( $query );
        }

        // Headers (all)
        if ( $inspect_headers && ! empty( $headers ) ) {
            $parts[] = 'headers=' . wp_json_encode( $headers );
        }

        // Cookies
        if ( $inspect_cookies && ! empty( $cookies ) ) {
            $parts[] = 'cookies=' . wp_json_encode( $cookies );
        }

        // Body (limit size, skip binary)
        if ( $inspect_body && $body !== '' ) {
            $parts[] = 'body=' . $body;
        }

        $raw = implode( ' | ', $parts );

        return [
            'path'    => $path,
            'method'  => $method,
            'headers' => $headers,
            'query'   => $query,
            'cookies' => $cookies,
            'body'    => $body,
            'ua'      => $ua,
            'ip'      => $ip,
            'raw'     => $raw,
            'norm'    => self::normalize_string( $raw ),
        ];
    }

    public static function normalize_string( string $s ) : string {
        $s = (string) $s;
        $s = wp_check_invalid_utf8( $s, true );
        $s = str_replace( "\0", '', $s );

        $s = rawurldecode( $s );
        $s = rawurldecode( $s );
        $s = html_entity_decode( $s, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
        $s = strtolower( $s );

        $s = preg_replace( '/\s+/', ' ', $s );
        return (string) $s;
    }

    protected static function get_headers() : array {
        $headers = [];
        if ( function_exists( 'getallheaders' ) ) {
            $h = getallheaders();
            if ( is_array( $h ) ) {
                foreach ( $h as $k => $v ) {
                    $headers[ (string) $k ] = (string) $v;
                }
            }
        } else {
            foreach ( $_SERVER as $k => $v ) {
                if ( strpos( (string) $k, 'HTTP_' ) === 0 ) {
                    $name = str_replace( '_', '-', substr( (string) $k, 5 ) );
                    $headers[ $name ] = (string) $v;
                }
            }
        }
        return $headers;
    }

    protected static function get_body_safe() : string {
        $settings = AegisWAF_Storage::get_settings();
        $mr = is_array( $settings['managed_rules'] ?? null ) ? $settings['managed_rules'] : [];
        $max = isset( $mr['max_body_bytes'] ) ? (int) $mr['max_body_bytes'] : 65536;
        if ( $max < 1024 ) { $max = 1024; }

        $ct = isset( $_SERVER['CONTENT_TYPE'] ) ? strtolower( (string) $_SERVER['CONTENT_TYPE'] ) : '';

        if ( $ct !== '' && strpos( $ct, 'multipart/form-data' ) !== false ) {
            return '';
        }

        if ( ! empty( $_POST ) && is_array( $_POST ) ) {
            return (string) wp_json_encode( $_POST );
        }

        $len = isset( $_SERVER['CONTENT_LENGTH'] ) ? (int) $_SERVER['CONTENT_LENGTH'] : 0;
        if ( $len > 0 && $len > $max ) { return ''; }

        $raw = '';
        $stream = @file_get_contents( 'php://input' );
        if ( is_string( $stream ) ) {
            $raw = $stream;
        }
        if ( $raw === '' ) { return ''; }
        if ( strlen( $raw ) > $max ) {
            $raw = substr( $raw, 0, $max );
        }
        return (string) $raw;
    }
}
