<?php
/**
 * Plugin Name: AegisWAF
 * Plugin URI: https://aegisify.com/aegiswaf
 * Description: AegisWAF provides a rule-driven Web Application Firewall for WordPress, inspecting incoming requests to block malicious traffic, bots, and common web attacks.
 * Version: 1.8.17
 * Requires at least: 6.8
 * Requires PHP: 8.2
 * Author:      Aegisify, LLC
 * Author URI:  https://aegisify.com/
 * Text Domain: aegiswaf
 * License: GPLv2 or later
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'AEGISWAF_VERSION', '1.8.17' );
define( 'AEGISWAF_PATH', plugin_dir_path( __FILE__ ) );
define( 'AEGISWAF_URL', plugin_dir_url( __FILE__ ) );
define( 'AEGISWAF_SLUG', 'aegiswaf' );
define( 'AEGISWAF_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

define( 'AEGISWAF_CLM_BASE', 'https://aegisify.com' );
define( 'AEGISWAF_CLM_FREE_REG_ENDPOINT', AEGISWAF_CLM_BASE . '/wp-json/chue-license/v1/aegiswaf/free-register' );
define( 'AEGISWAF_CLM_UPDATE_ENDPOINT',    AEGISWAF_CLM_BASE . '/wp-json/chue-license/v1/aegiswaf/update-check' );

require_once AEGISWAF_PATH . 'includes/class-aegiswaf-features.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-utils.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-storage.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-overrides.php';
require_once AEGISWAF_PATH . 'includes/ddos/class-aegiswaf-ddos-storage.php';
require_once AEGISWAF_PATH . 'includes/ddos/class-aegiswaf-ddos-rules.php';
require_once AEGISWAF_PATH . 'includes/ddos/class-aegiswaf-ddos-engine.php';
require_once AEGISWAF_PATH . 'includes/ddos/class-aegiswaf-ddos-admin.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-logger.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-engine.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-normalizer.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-managed-rules.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-heuristics.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-endpoint-policy.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-challenge.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-wp-protect.php';
require_once AEGISWAF_PATH . 'includes/licensing/class-aegiswaf-license-manager.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-rules.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-intel.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-behavioral-timeline.php';
require_once AEGISWAF_PATH . 'includes/class-aegiswaf-rule-sets.php';
require_once AEGISWAF_PATH . 'includes/providers/class-aegiswaf-geoasn-provider.php';
require_once AEGISWAF_PATH . 'includes/admin/class-aegiswaf-admin.php';

function aegiswaf_update_log( string $msg, array $ctx = array() ) : void {
    if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log( '[AegisWAF Update] ' . $msg . ( $ctx ? ' | ' . wp_json_encode( $ctx ) : '' ) ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
    }
}

function aegiswaf_clm_register_free_install( bool $force = false ) : array {

    if ( ! $force && get_transient( 'aegiswaf_clm_free_reg_sent' ) ) {
        return array(
            'success' => false,
            'message' => 'Backoff active (already attempted recently).',
        );
    }

    $site_url = home_url();
    $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

    $payload = array(
        'product'        => AEGISWAF_SLUG,
        'email'          => (string) get_option( 'admin_email' ),
        'site_url'       => (string) $site_url,
        'domain'         => (string) $domain,
        'plugin_version' => (string) AEGISWAF_VERSION,
        'wp_version'     => (string) get_bloginfo( 'version' ),
        'php_version'    => (string) PHP_VERSION,
    );

    $args = array(
        'timeout' => 15,
        'headers' => array( 'Content-Type' => 'application/json' ),
        'body'    => wp_json_encode( $payload ),
    );

    $resp = wp_remote_post( AEGISWAF_CLM_FREE_REG_ENDPOINT, $args );

    // Backoff for 24h to avoid spam (same intent as your original)
    set_transient( 'aegiswaf_clm_free_reg_sent', 1, DAY_IN_SECONDS );

    if ( is_wp_error( $resp ) ) {
        aegiswaf_update_log(
            'CLM free-register HTTP error (WP_Error).',
            array( 'error' => $resp->get_error_message() ),
            'ERROR'
        );
        return array(
            'success' => false,
            'message' => $resp->get_error_message(),
        );
    }

    $code = (int) wp_remote_retrieve_response_code( $resp );
    $body = (string) wp_remote_retrieve_body( $resp );

    $decoded = null;
    if ( ! empty($body) ) {
        $decoded = json_decode( $body, true );
    }

    // Decide success:
    // - HTTP 200/201 are typically OK
    // - AND response JSON may contain success=true (depends on your CLM)
    $ok = false;
    if ( $code >= 200 && $code < 300 ) {
        $ok = true;
        if ( is_array($decoded) && array_key_exists('success', $decoded) ) {
            $ok = (bool) $decoded['success'];
        }
    }

    if ( ! $ok ) {
        aegiswaf_update_log(
            'CLM free-register failed response.',
            array( 'http_code' => $code, 'body' => $body ),
            'ERROR'
        );
        return array(
            'success'   => false,
            'http_code' => $code,
            'body'      => $body,
            'message'   => 'CLM rejected the request.',
        );
    }

    aegiswaf_update_log(
        'CLM free-register success.',
        array( 'http_code' => $code ),
        'INFO'
    );

    return array(
        'success'   => true,
        'http_code' => $code,
        'body'      => $body,
        'message'   => 'Registered successfully.',
    );
}

function aegiswaf_clm_direct_update_check( string $current_version ) : array {

    $site_url = home_url();
    $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

    $license_key = (string) get_option( 'aegiswaf_license_key', '' );

    $payload = array(
        'slug'            => AEGISWAF_SLUG,
        'product'         => AEGISWAF_SLUG,
        'version'         => $current_version,
        'domain'          => (string) $domain,
        'site_url'        => (string) $site_url,
        'license_key'     => $license_key,
        'wp_version'      => (string) get_bloginfo( 'version' ),
        'php_version'     => (string) PHP_VERSION,
        'plugin_basename' => (string) AEGISWAF_PLUGIN_BASENAME,
    );

    $args = array(
        'timeout' => 20,
        'headers' => array( 'Content-Type' => 'application/json' ),
        'body'    => wp_json_encode( $payload ),
    );

    $resp = wp_remote_post( AEGISWAF_CLM_UPDATE_ENDPOINT, $args );

    if ( is_wp_error( $resp ) ) {
        return array(
            'success' => false,
            'status'  => 'wp_error',
            'message' => $resp->get_error_message(),
        );
    }

    $code = (int) wp_remote_retrieve_response_code( $resp );
    $body = (string) wp_remote_retrieve_body( $resp );

    if ( $code < 200 || $code >= 300 ) {
        return array(
            'success' => false,
            'status'  => 'http_' . $code,
            'message' => 'CLM HTTP ' . $code,
            'raw'     => $body,
        );
    }

    $data = json_decode( $body, true );
    if ( ! is_array( $data ) ) {
        return array(
            'success' => false,
            'status'  => 'bad_json',
            'message' => 'CLM returned invalid JSON',
            'raw'     => $body,
        );
    }

    if ( empty( $data['version'] ) && ! empty( $data['latest_version'] ) ) {
        $data['version'] = (string) $data['latest_version'];
    }

    return $data;
}

register_activation_hook( __FILE__, [ 'AegisWAF_Logger', 'activate' ] );

add_action( 'plugins_loaded', function() {

    ( new AegisWAF_License_Manager() )->init();AegisWAF_GeoASN_Provider::register_filter();

    require_once AEGISWAF_PATH . 'includes/admin/class-aegiswaf-notices.php';
    AegisWAF_Admin_Notices::init();

}, 10 );

add_action( 'admin_init', function() {
    if ( defined('DOING_AJAX') && DOING_AJAX ) {
        return;
    }
    if ( wp_doing_cron() ) {
        return;
    }

	if ( isset( $_GET['action'] ) && sanitize_key( wp_unslash( $_GET['action'] ) ) === 'activate' ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return;
	}
    aegiswaf_clm_register_free_install();
}, 20 );

add_action( 'init', function() {
    AegisWAF_Engine::init();
    AegisWAF_WP_Protect::init();
}, 20 );