<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

if ( ! function_exists( 'aegiswaf_bot_control_get_visual_data' ) ) {
	function aegiswaf_bot_control_get_visual_data() : array {
		global $wpdb;

		if ( ! class_exists( 'AegisWAF_Logger' ) || ! method_exists( 'AegisWAF_Logger', 'table_name' ) ) {
			return [
				'totals' => [ 'last_24h' => 0, 'last_7d' => 0 ],
				'action_labels' => [], 'action_counts' => [],
				'scope_labels'  => [], 'scope_counts'  => [],
				'ua_labels'     => [], 'ua_counts'     => [],
				'bubbles'       => [],
				'since_gmt'     => gmdate( 'Y-m-d H:i:s', time() - ( 7 * DAY_IN_SECONDS ) ),
			];
		}

		$table = AegisWAF_Logger::table_name();
		$since_7d_gmt  = gmdate( 'Y-m-d H:i:s', time() - ( 7 * DAY_IN_SECONDS ) );
		$since_24h_gmt = gmdate( 'Y-m-d H:i:s', time() - DAY_IN_SECONDS );

		$total_7d = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT COUNT(*) FROM %i
				  WHERE event_time >= %s
				    AND category IN ('bot_control',  'bot_bad_ua')", 
				$table,
				$since_7d_gmt
			)
		);
		$total_24h = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT COUNT(*) FROM %i
				  WHERE event_time >= %s
				    AND category IN ('bot_control',  'bot_bad_ua')", 
				$table,
				$since_24h_gmt
			)
		);

		$action_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT action_taken,  COUNT(*) AS c
				   FROM %i
				  WHERE event_time >= %s
				    AND category IN ('bot_control','bot_bad_ua')
				  GROUP BY action_taken
				  ORDER BY c DESC", 
				$table,
				$since_7d_gmt
			),
			ARRAY_A
		);

		$action_map = [];
		foreach ( (array) $action_rows as $r ) {
			$k = sanitize_key( (string) ( $r['action_taken'] ?? 'unknown' ) );
			if ( $k === '' ) { $k = 'unknown'; }
			$action_map[ $k ] = (int) ( $r['c'] ?? 0 );
		}

		foreach ( [ 'rate_limit', 'blocked', 'config_missing', 'other' ] as $k ) {
			if ( ! isset( $action_map[ $k ] ) ) { $action_map[ $k ] = 0; }
		}

		$main = [ 'rate_limit', 'blocked', 'config_missing' ];
		$other_sum = 0;
		foreach ( $action_map as $k => $v ) {
			if ( ! in_array( $k, $main, true ) && $k !== 'other' ) {
				$other_sum += (int) $v;
				unset( $action_map[ $k ] );
			}
		}
		$action_map['other'] += $other_sum;

		$action_labels = array_keys( $action_map );
		$action_counts = array_values( $action_map );

		$scope_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT details
				   FROM %i
				  WHERE event_time >= %s
				    AND category IN ('bot_control',  'bot_bad_ua')
				  ORDER BY id DESC
				  LIMIT 2000", 
				$table,
				$since_7d_gmt
			),
			ARRAY_A
		);

		$scope_map = [ 'front' => 0, 'rest' => 0, 'unknown' => 0 ];
		foreach ( (array) $scope_rows as $r ) {
			$details = [];
			if ( ! empty( $r['details'] ) ) {
				$decoded = json_decode( (string) $r['details'], true );
				if ( is_array( $decoded ) ) { $details = $decoded; }
			}
			$scope = sanitize_key( (string) ( $details['scope'] ?? 'unknown' ) );
			if ( $scope === '' ) { $scope = 'unknown'; }
			if ( ! isset( $scope_map[ $scope ] ) ) { $scope_map[ $scope ] = 0; }
			$scope_map[ $scope ]++;
		}

		$scope_labels = array_keys( $scope_map );
		$scope_counts = array_values( $scope_map );

		$ua_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT ua,  COUNT(*) AS c
				   FROM %i
				  WHERE event_time >= %s
				    AND category IN ('bot_control','bot_bad_ua')
				    AND ua IS NOT NULL AND ua <> ''
				  GROUP BY ua
				  ORDER BY c DESC
				  LIMIT 7", 
				$table,
				$since_7d_gmt
			),
			ARRAY_A
		);

		$ua_labels = [];
		$ua_counts = [];
		foreach ( (array) $ua_rows as $r ) {
			$ua = (string) ( $r['ua'] ?? '' );
			if ( strlen( $ua ) > 38 ) {
				$ua = substr( $ua, 0, 38 ) . '…';
			}
			$ua_labels[] = $ua;
			$ua_counts[] = (int) ( $r['c'] ?? 0 );
		}

		$ip_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT ip,  COUNT(*) AS total,
				        SUM(CASE WHEN action_taken='rate_limit' THEN 1 ELSE 0 END) AS rl,
				        SUM(CASE WHEN action_taken='blocked' THEN 1 ELSE 0 END) AS bl
				   FROM %i
				  WHERE event_time >= %s
				    AND category IN ('bot_control','bot_bad_ua')
				    AND ip IS NOT NULL AND ip <> ''
				  GROUP BY ip
				  ORDER BY total DESC
				  LIMIT 10", 
				$table,
				$since_24h_gmt
			),
			ARRAY_A
		);

		$bubbles = [];
		foreach ( (array) $ip_rows as $r ) {
			$ip = (string) ( $r['ip'] ?? '' );
			$total = (int) ( $r['total'] ?? 0 );
			$rl = (int) ( $r['rl'] ?? 0 );
			$bl = (int) ( $r['bl'] ?? 0 );

			$radius = max( 4, min( 18, 4 + (int) floor( $bl / 2 ) ) );

			$bubbles[] = [
				'label' => $ip,
				'x' => $total,
				'y' => $rl,
				'r' => $radius,
				'blocked' => $bl,
			];
		}

		return [
			'totals' => [ 'last_24h' => $total_24h, 'last_7d' => $total_7d ],
			'action_labels' => $action_labels,
			'action_counts' => $action_counts,
			'scope_labels'  => $scope_labels,
			'scope_counts'  => $scope_counts,
			'ua_labels'     => $ua_labels,
			'ua_counts'     => $ua_counts,
			'bubbles'       => $bubbles,
			'since_gmt'     => $since_7d_gmt,
		];
	}
}

add_action( 'admin_enqueue_scripts', function() {
	$page = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	if ( $page !== 'aegiswaf' ) { return; }

	$tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'overview'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	if ( $tab !== 'bot_control' ) { return; }

	$base_url = plugin_dir_url( dirname( __FILE__, 3 ) ); // plugin root

	wp_enqueue_script(
		'aegiswaf-chartjs',
		$base_url . 'assets/js/chart.umd.min.js',
		[],
		'1.6.9',
		true
	);

	$data = aegiswaf_bot_control_get_visual_data();
	$json = wp_json_encode( $data );

	$inline = "
	(function(){
		if (typeof Chart === 'undefined') { return; }
		var data = {$json};
		function el(id){ return document.getElementById(id); }

		// Totals
		var t24 = el('aegiswaf_bot_total_24h');
		var t7d = el('aegiswaf_bot_total_7d');
		if (t24) { t24.textContent = String((data.totals && data.totals.last_24h) ? data.totals.last_24h : 0); }
		if (t7d) { t7d.textContent = String((data.totals && data.totals.last_7d) ? data.totals.last_7d : 0); }

		// 1) Radar — enforcement signals
		var c1 = el('aegiswaf_bot_chart_radar');
		if (c1) {
			new Chart(c1.getContext('2d'), {
				type: 'radar',
				data: {
					labels: data.action_labels || [],
					datasets: [{
						label: 'Bot enforcement signals (7d)',
						data: data.action_counts || []
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { display: true } },
					scales: { r: { beginAtZero: true } }
				}
			});
		}

		// 2) Pie — scope breakdown
		var c2 = el('aegiswaf_bot_chart_scope');
		if (c2) {
			new Chart(c2.getContext('2d'), {
				type: 'pie',
				data: {
					labels: data.scope_labels || [],
					datasets: [{ data: data.scope_counts || [] }]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { position: 'bottom' } }
				}
			});
		}

		// 3) Polar Area — top UA strings
		var c3 = el('aegiswaf_bot_chart_ua');
		if (c3) {
			new Chart(c3.getContext('2d'), {
				type: 'polarArea',
				data: {
					labels: data.ua_labels || [],
					datasets: [{ data: data.ua_counts || [] }]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: { legend: { position: 'bottom' } }
				}
			});
		}

		// 4) Bubble — top IP offenders
		var c4 = el('aegiswaf_bot_chart_bubble');
		if (c4) {
			var pts = (data.bubbles || []).map(function(p){
				return { x: p.x, y: p.y, r: p.r, _label: p.label, _blocked: p.blocked };
			});
			new Chart(c4.getContext('2d'), {
				type: 'bubble',
				data: {
					datasets: [{
						label: 'Top offending IPs (24h)',
						data: pts
					}]
				},
				options: {
					responsive: true,
					maintainAspectRatio: false,
					plugins: {
						tooltip: {
							callbacks: {
								label: function(ctx){
									var p = ctx.raw || {};
									return (p._label || 'ip') + ' — total:' + p.x + ', rate_limit:' + p.y + ', blocked:' + (p._blocked || 0);
								}
							}
						},
						legend: { display: true }
					},
					scales: {
						x: { beginAtZero: true, title: { display: true, text: 'Total bot events (24h)' } },
						y: { beginAtZero: true, title: { display: true, text: 'Rate-limit events (24h)' } }
					}
				}
			});
		}
	})();";

	wp_add_inline_script( 'aegiswaf-chartjs', $inline, 'after' );
}, 21 );


class AegisWAF_Page_Bot_Control {

    public function render() : void {
        if ( ! current_user_can( 'manage_options' ) ) { return; }

        $is_pro = AegisWAF_Features::is_pro();
        $s = AegisWAF_Storage::get_settings();

        
        if ( isset( $_POST['aegiswaf_save_wp_protect'] ) ) {
            check_admin_referer( 'aegiswaf_save_wp_protect' );
            $s['wp_protect']['enabled'] = ! empty( $_POST['wp_enabled'] );
            $s['wp_protect']['login_protect'] = ! empty( $_POST['wp_login_protect'] );
            $s['wp_protect']['admin_protect'] = ! empty( $_POST['wp_admin_protect'] );
            $s['wp_protect']['ajax_protect'] = ! empty( $_POST['wp_ajax_protect'] );
            $s['wp_protect']['xmlrpc_enabled'] = ! empty( $_POST['wp_xmlrpc_enabled'] );
            $s['wp_protect']['rest_protect'] = ! empty( $_POST['wp_rest_protect'] );
            $s['wp_protect']['rest_only_unauth'] = ! empty( $_POST['wp_rest_only_unauth'] );
            $s['wp_protect']['rest_allowlist'] = isset( $_POST['wp_rest_allowlist'] ) ? sanitize_textarea_field( wp_unslash( $_POST['wp_rest_allowlist'] ) ) : '';
            $s['wp_protect']['window_seconds_rest'] = max( 15, absint( wp_unslash( $_POST['wp_window_rest'] ?? 60 ) ) );
            $s['wp_protect']['xmlrpc_mode'] = isset( $_POST['wp_xmlrpc_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['wp_xmlrpc_mode'] ) ) : 'restrict';
            $s['wp_protect']['xmlrpc_allowlist'] = isset( $_POST['wp_xmlrpc_allowlist'] ) ? sanitize_textarea_field( wp_unslash( $_POST['wp_xmlrpc_allowlist'] ) ) : '';
            $s['wp_protect']['window_seconds'] = max( 60, absint( wp_unslash( $_POST['wp_window'] ?? 300 ) ) );
            $s['wp_protect']['window_seconds_admin'] = max( 60, absint( wp_unslash( $_POST['wp_window_admin'] ?? 300 ) ) );
            $s['wp_protect']['window_seconds_xmlrpc'] = max( 60, absint( wp_unslash( $_POST['wp_window_xmlrpc'] ?? 300 ) ) );
            $s['wp_protect']['distributed_enabled'] = ! empty( $_POST['wp_distributed'] );
            $s['wp_protect']['bruteforce_enabled'] = ! empty( $_POST['wp_bruteforce'] );
            $s['wp_protect']['reset_user_on_success'] = ! empty( $_POST['wp_reset_user'] );
            // Per-surface thresholds (challenge/rate/block)
$surfaces = [
    'thresholds_login' => 'login',
    'thresholds_admin' => 'admin',
    'thresholds_ajax'  => 'ajax',
    'thresholds_xmlrpc'=> 'xmlrpc',
    'thresholds_rest'  => 'rest',
    'thresholds_login_ip' => 'login_ip',
    'thresholds_login_user' => 'login_user',
];
foreach ( $surfaces as $k => $prefix ) {
    $s['wp_protect'][ $k ] = is_array( $s['wp_protect'][ $k ] ?? null ) ? $s['wp_protect'][ $k ] : [];
    $s['wp_protect'][ $k ]['challenge_at']   = isset( $_POST[ 'wp_thr_' . $prefix . '_challenge' ] ) ? max( 0, absint( wp_unslash( $_POST[ 'wp_thr_' . $prefix . '_challenge' ] ) ) ) : (int) ( $s['wp_protect'][ $k ]['challenge_at'] ?? 0 );
    $s['wp_protect'][ $k ]['rate_limit_at']  = isset( $_POST[ 'wp_thr_' . $prefix . '_rate' ] ) ? max( 0, absint( wp_unslash( $_POST[ 'wp_thr_' . $prefix . '_rate' ] ) ) ) : (int) ( $s['wp_protect'][ $k ]['rate_limit_at'] ?? 0 );
    $s['wp_protect'][ $k ]['block_at']       = isset( $_POST[ 'wp_thr_' . $prefix . '_block' ] ) ? max( 0, absint( wp_unslash( $_POST[ 'wp_thr_' . $prefix . '_block' ] ) ) ) : (int) ( $s['wp_protect'][ $k ]['block_at'] ?? 0 );
}

            AegisWAF_Storage::update_settings( $s );
            echo '<div class="notice notice-success"><p>WordPress protection settings saved.</p></div>';
        }


        if ( isset( $_POST['aegiswaf_save_bot'] ) ) {
            check_admin_referer( 'aegiswaf_save_bot' );

            $s['bot']['enabled'] = ! empty( $_POST['bot_enabled'] );
            $s['bot']['block_bad_ua'] = ! empty( $_POST['bot_block_bad_ua'] );
            $s['bot']['default_limit_per_min'] = max( 10, min( 10000, absint( wp_unslash( $_POST['bot_default_limit_per_min'] ?? 180 ) ) ) );

            $ua_list = isset( $_POST['bot_bad_ua_list'] ) ? sanitize_textarea_field( wp_unslash( $_POST['bot_bad_ua_list'] ) ) : '';
            $s['bot']['bad_ua_list'] = $ua_list;
            $patterns = [];
            $raw_patterns = [];
            if ( isset( $_POST['path_pattern'] ) && is_array( $_POST['path_pattern'] ) ) {
                $raw_patterns = array_map( 'sanitize_text_field', (array) wp_unslash( $_POST['path_pattern'] ) );
            }
            foreach ( $raw_patterns as $val ) {
                $patterns[] = sanitize_text_field( (string) $val );
            }

            $limits = [];
            $raw_limits = [];
            if ( isset( $_POST['path_limit'] ) && is_array( $_POST['path_limit'] ) ) {
                $raw_limits = array_map( 'absint', (array) wp_unslash( $_POST['path_limit'] ) );
            }
            foreach ( $raw_limits as $val ) {
                $limits[] = absint( $val );
            }
            $rows = [];
            $count = min( count( $patterns ), count( $limits ) );
            for ( $i = 0; $i < $count; $i++ ) {
                $p = sanitize_text_field( $patterns[ $i ] );
                $l = absint( $limits[ $i ] );
                if ( $p === '' || $l <= 0 ) { continue; }
                $rows[] = [ 'pattern' => $p, 'limit_per_min' => max( 1, min( 10000, $l ) ) ];
            }
            $s['bot']['per_path_limits'] = $rows;

$surfaces = [
    'thresholds_login' => 'login',
    'thresholds_admin' => 'admin',
    'thresholds_ajax'  => 'ajax',
    'thresholds_xmlrpc'=> 'xmlrpc',
    'thresholds_rest'  => 'rest',
    'thresholds_login_ip' => 'login_ip',
    'thresholds_login_user' => 'login_user',
];
foreach ( $surfaces as $k => $prefix ) {
    $s['wp_protect'][ $k ] = is_array( $s['wp_protect'][ $k ] ?? null ) ? $s['wp_protect'][ $k ] : [];
    $s['wp_protect'][ $k ]['challenge_at']   = isset( $_POST[ 'wp_thr_' . $prefix . '_challenge' ] ) ? max( 0, absint( wp_unslash( $_POST[ 'wp_thr_' . $prefix . '_challenge' ] ) ) ) : (int) ( $s['wp_protect'][ $k ]['challenge_at'] ?? 0 );
    $s['wp_protect'][ $k ]['rate_limit_at']  = isset( $_POST[ 'wp_thr_' . $prefix . '_rate' ] ) ? max( 0, absint( wp_unslash( $_POST[ 'wp_thr_' . $prefix . '_rate' ] ) ) ) : (int) ( $s['wp_protect'][ $k ]['rate_limit_at'] ?? 0 );
    $s['wp_protect'][ $k ]['block_at']       = isset( $_POST[ 'wp_thr_' . $prefix . '_block' ] ) ? max( 0, absint( wp_unslash( $_POST[ 'wp_thr_' . $prefix . '_block' ] ) ) ) : (int) ( $s['wp_protect'][ $k ]['block_at'] ?? 0 );
}

		$ok = AegisWAF_Storage::update_settings( $s );

		if ( ! $ok ) {
			if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				AegisWAF_Logger::log( 'admin:bot_control', 'POST', 'bot_control', 'save_failed', [
					'reason' => 'update_option(aegiswaf_settings) returned false',
					'user_id' => get_current_user_id(),
				] );
			}
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( '[AegisWAF] Bot Control save failed: update_option(aegiswaf_settings) returned false.' );

			echo '<div class="notice notice-error"><p><strong>Bot Control save failed.</strong> WordPress did not persist the settings. Check permissions/object cache, and see debug.log for details.</p></div>';
		} else {
			echo '<div class="notice notice-success"><p>Bot Control settings saved.</p></div>';
		}
        }
        echo '<div class="wrap">';
        echo '<h1>BOT Control</h1>';
        AegisWAF_Pro_Notice::render();
        echo '<p class="description">Bot Control protects your site from abusive and automated traffic by identifying and blocking known bad bots, suspicious user agents, and abnormal request patterns. The module applies rate-based detection and basic behavioral checks to reduce scraping, brute force activity, and resource abuse. Pro enhances bot defense with behavioral bot scoring, good-bot allowlisting (such as search engine crawlers), and deeper bot analytics for improved accuracy and reduced false positives.</p>';

		echo '<style>
		.aegiswaf-botviz-grid{
			display:grid;
			grid-template-columns: repeat(4, minmax(220px, 1fr));
			gap:12px;
			margin:12px 0 14px;
		}
		@media (max-width: 1300px){ .aegiswaf-botviz-grid{ grid-template-columns: repeat(2, minmax(240px, 1fr)); } }
		@media (max-width: 782px){ .aegiswaf-botviz-grid{ grid-template-columns: 1fr; } }

		.aegiswaf-botviz-panel{
			border:1px solid #e5e7eb;
			border-radius:10px;
			padding:12px;
			background:#fff;
		}
		.aegiswaf-botviz-title{ font-weight:700; margin:0 0 4px; }
		.aegiswaf-botviz-sub{ color:#646970; font-size:12px; margin:0 0 10px; }
		.aegiswaf-botviz-canvas{ height:220px; position:relative; }
		</style>';

		echo '<div class="aegiswaf-card" style="margin-top:12px;">';
		echo '<h3>Bot Control Visual Intelligence</h3>';
		echo '<p class="description" style="margin:6px 0 10px;">Quick clarity on bot enforcement and abnormal traffic patterns (Bot Control logs only).</p>';
		echo '<p style="margin:0 0 10px;font-weight:600;">Last 24h: <span id="aegiswaf_bot_total_24h">0</span> &nbsp; | &nbsp; Last 7d: <span id="aegiswaf_bot_total_7d">0</span></p>';

		echo '<div class="aegiswaf-botviz-grid">';

			echo '<div class="aegiswaf-botviz-panel">';
				echo '<div class="aegiswaf-botviz-title">Enforcement Signals</div>';
				echo '<div class="aegiswaf-botviz-sub">Radar (7 days): rate_limit / blocked / config</div>';
				echo '<div class="aegiswaf-botviz-canvas"><canvas id="aegiswaf_bot_chart_radar"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-botviz-panel">';
				echo '<div class="aegiswaf-botviz-title">Surface Scope</div>';
				echo '<div class="aegiswaf-botviz-sub">Pie: front vs REST enforcement</div>';
				echo '<div class="aegiswaf-botviz-canvas"><canvas id="aegiswaf_bot_chart_scope"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-botviz-panel">';
				echo '<div class="aegiswaf-botviz-title">Top User-Agents</div>';
				echo '<div class="aegiswaf-botviz-sub">Polar area: most frequent bot UA strings</div>';
				echo '<div class="aegiswaf-botviz-canvas"><canvas id="aegiswaf_bot_chart_ua"></canvas></div>';
			echo '</div>';

			echo '<div class="aegiswaf-botviz-panel">';
				echo '<div class="aegiswaf-botviz-title">Top Offending IPs</div>';
				echo '<div class="aegiswaf-botviz-sub">Bubble (24h): total vs rate-limit, size = blocked</div>';
				echo '<div class="aegiswaf-botviz-canvas"><canvas id="aegiswaf_bot_chart_bubble"></canvas></div>';
			echo '</div>';

		echo '</div>'; // grid
		echo '</div>'; // card
		echo '<div class="aegiswaf-card" style="width:100%;max-width:100%;box-sizing:border-box;margin-top:12px;">';
		echo '<div class="aegiswaf-cardhead"><h3>Bot Control (Wired)</h3></div>';
        echo '<p>Enforcement is active for front requests and REST API requests.</p>';
        echo '<form method="post">';
        wp_nonce_field( 'aegiswaf_save_bot' );

        echo '<table class="form-table" role="presentation"><tbody>';
        echo '<tr><th scope="row">Enable Bot Control</th><td><label><input type="checkbox" name="bot_enabled" value="1" ' . checked( ! empty( $s['bot']['enabled'] ), true, false ) . '> Enabled</label></td></tr>';

        echo '<tr><th scope="row">Default Rate Limit (per minute)</th><td>';
        echo '<input type="number" min="10" max="10000" name="bot_default_limit_per_min" value="' . esc_attr( (int) ( $s['bot']['default_limit_per_min'] ?? 180 ) ) . '">';
        echo '<p class="description">Applied per IP + path + minute bucket (conservative).</p>';
        echo '</td></tr>';

        echo '<tr><th scope="row">Bad User-Agent Blocking</th><td>';
        echo '<label><input type="checkbox" name="bot_block_bad_ua" value="1" ' . checked( ! empty( $s['bot']['block_bad_ua'] ), true, false ) . '> Enable UA blocking</label>';
        echo '<p class="description">One token per line. If the request UA contains a token, it is blocked.</p>';
        echo '<textarea name="bot_bad_ua_list" rows="6" style="width:420px;">' . esc_textarea( (string) ( $s['bot']['bad_ua_list'] ?? '' ) ) . '</textarea>';
        echo '</td></tr>';
        echo '</tbody></table>';

        echo '<h3>Per-Path Thresholds</h3>';
        echo '<p class="description">Use * wildcard. Example: <code>/wp-login.php</code> or <code>/wp-json/*</code></p>';

        $rows = is_array( $s['bot']['per_path_limits'] ?? null ) ? $s['bot']['per_path_limits'] : [];
        $rows = array_values( $rows );
        $max_rows = 5;

        echo '<table class="widefat striped"><thead><tr><th>Path Pattern</th><th>Limit / min</th></tr></thead><tbody>';
        for ( $i = 0; $i < $max_rows; $i++ ) {
            $pat = $rows[ $i ]['pattern'] ?? '';
            $lim = $rows[ $i ]['limit_per_min'] ?? '';
            echo '<tr>';
            echo '<td><input type="text" name="path_pattern[]" value="' . esc_attr( (string) $pat ) . '" placeholder="/wp-login.php" style="width:95%;"></td>';
            echo '<td style="width:140px;"><input type="number" name="path_limit[]" value="' . esc_attr( (string) $lim ) . '" min="1" max="10000" style="width:120px;"></td>';
            echo '</tr>';
        }
        echo '</tbody></table>';

        submit_button( 'Save Bot Control', 'primary', 'aegiswaf_save_bot' );
        echo '</form></div>';
		echo '<p></p>';

        echo '<div class="aegiswaf-card"><h3>Bot Intelligence Settings';
        echo '<p class="description">Behavioral scoring and IP/CIDR blocklist are enforced when PRO is active. Geo/ASN requires a provider hook.</p>';
        if ( ! $is_pro ) {
            AegisWAF_Utils::pro_gate_box(
                'Unlock PRO Intelligence & Automation',
                'PRO unlocks advanced BOT Intelligence Settings'
            );
        }
        $intel = AegisWAF_Intel::get();

        if ( isset( $_POST['aegiswaf_save_intel'] ) ) {
            check_admin_referer( 'aegiswaf_save_intel' );
            if ( $is_pro ) {
                $intel['ip_blocklist'] = isset( $_POST['intel_ip_blocklist'] ) ? sanitize_textarea_field( wp_unslash( $_POST['intel_ip_blocklist'] ) ) : '';
                $intel['country_block'] = isset( $_POST['intel_country_block'] ) ? sanitize_text_field( wp_unslash( $_POST['intel_country_block'] ) ) : '';
                $intel['asn_block'] = isset( $_POST['intel_asn_block'] ) ? sanitize_textarea_field( wp_unslash( $_POST['intel_asn_block'] ) ) : '';
                $intel['geo_provider'] = isset($_POST['intel_geo_provider']) ? sanitize_text_field( wp_unslash($_POST['intel_geo_provider']) ) : 'none';
                $intel['maxmind_mmdb_path'] = isset($_POST['intel_maxmind_mmdb_path']) ? sanitize_text_field( wp_unslash($_POST['intel_maxmind_mmdb_path']) ) : '';
                $intel['behavioral_threshold'] = max( 1, min( 100, absint( wp_unslash( $_POST['intel_behavioral_threshold'] ?? 80 ) ) ) );
                AegisWAF_Storage::update_intel( $intel );
                echo '<div class="notice notice-success"><p>Bot Intelligence settings saved.</p></div>';
            } else {
                echo '<div class="notice notice-warning"><p>Bot Intelligence settings are PRO.</p></div>';
            }
        }

        AegisWAF_Utils::dim_wrap_open( ! $is_pro );
        echo '<form method="post">';
        wp_nonce_field( 'aegiswaf_save_intel' );

        echo '<table class="form-table"><tbody>';
        echo '<tr><th>Behavioral block threshold</th><td><input type="number" min="1" max="100" name="intel_behavioral_threshold" value="' . esc_attr( (string) (int) $intel['behavioral_threshold'] ) . '"><p class="description">Higher = less aggressive. Default 80.</p></td></tr>';
        echo '<tr><th>IP / CIDR blocklist</th><td><textarea name="intel_ip_blocklist" rows="4" style="width:420px;">' . esc_textarea( (string) $intel['ip_blocklist'] ) . '</textarea><p class="description">One per line. Supports IPv4 CIDR (e.g., 1.2.3.0/24).</p></td></tr>';
        echo '<tr><th>Country block (framework)</th><td><input type="text" name="intel_country_block" value="' . esc_attr( (string) $intel['country_block'] ) . '" style="width:420px;"><p class="description">Comma-separated country codes (e.g., RU,CN). Requires provider.</p></td></tr>';
        echo '<tr><th>ASN block (framework)</th><td><textarea name="intel_asn_block" rows="3" style="width:420px;">' . esc_textarea( (string) $intel['asn_block'] ) . '</textarea><p class="description">One per line (e.g., AS13335). Requires provider.</p></td></tr>';
echo '<tr><th>Geo/ASN provider</th><td><select name="intel_geo_provider">'
    . '<option value="none" ' . selected( (string) ( $intel["geo_provider"] ?? "none" ), "none", false ) . '>None</option>'
    . '<option value="cloudflare" ' . selected( (string) ( $intel["geo_provider"] ?? "none" ), "cloudflare", false ) . '>Cloudflare headers (country only)</option>'
    . '<option value="maxmind_ext" ' . selected( (string) ( $intel["geo_provider"] ?? "none" ), "maxmind_ext", false ) . '>MaxMind MMDB (PHP ext)</option>'
    . '</select>'
    . '<p class="description">Cloudflare uses CF-IPCountry. MaxMind requires the <code>maxminddb</code> PHP extension and a local MMDB file path.</p>'
    . '</td></tr>';
echo '<tr><th>MaxMind MMDB path</th><td><input type="text" name="intel_maxmind_mmdb_path" value="' . esc_attr( (string) ( $intel["maxmind_mmdb_path"] ?? "" ) ) . '" style="width:420px;">'
    . '<p class="description">Example: <code>/home/USER/geo/GeoLite2-ASN.mmdb</code> or <code>GeoLite2-Country.mmdb</code></p>'
    . '</td></tr>';
        echo '</tbody></table>';

        submit_button( 'Save Bot Intelligence', 'secondary', 'aegiswaf_save_intel' );
        echo '</form>';
        echo '<p class="description">Geo/ASN provider hook: <code>add_filter(\'aegiswaf_geoasn_provider\', function() { return function($ip){ return [\'country\'=>\'US\',\'asn\'=>\'AS0\']; }; });</code></p>';
        AegisWAF_Utils::dim_wrap_close();
        echo '</div>';

    }
}