<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

$aegiswaf_has_pro = (
	class_exists( 'AegisWAF_License' ) &&
	method_exists( 'AegisWAF_License', 'is_pro_active' ) &&
	AegisWAF_License::is_pro_active()
);

if ( ! function_exists( 'aegiswaf_logs_get_visual_data' ) ) {
	function aegiswaf_logs_get_visual_data() : array {
		global $wpdb;

		if ( ! class_exists( 'AegisWAF_Logger' ) || ! method_exists( 'AegisWAF_Logger', 'table_name' ) ) {
			return [
				'totals' => [ 'last_24h' => 0, 'last_7d' => 0 ],
				'trend_labels' => [], 'trend_block' => [], 'trend_challenge' => [], 'trend_rate' => [],
				'threat_labels' => [], 'threat_counts' => [],
				'action_labels' => [], 'action_counts' => [],
				'module_labels' => [], 'module_counts' => [],
				'method_labels' => [], 'method_counts' => [],
				'route_labels' => [], 'route_counts' => [],
				'bubbles' => [],
				'scatter' => [],
			];
		}

		$table = AegisWAF_Logger::table_name();

		
		// Hard-sanitize identifier for static analysis.
		$table = preg_replace( '/[^A-Za-z0-9_]/', '', (string) $table );
$since_14d_gmt = gmdate( 'Y-m-d H:i:s', time() - ( 14 * DAY_IN_SECONDS ) );
		$since_7d_gmt  = gmdate( 'Y-m-d H:i:s', time() - ( 7 * DAY_IN_SECONDS ) );
		$since_24h_gmt = gmdate( 'Y-m-d H:i:s', time() - DAY_IN_SECONDS );

		$sec_categories = [ 'managed_rule', 'heuristic', 'api_shield', 'bot_control', 'bot_bad_ua', 'ddos_shield' ];
		$sec_categories = array_values( array_filter( array_map( 'sanitize_key', $sec_categories ) ) );
		$sec_categories = array_pad( $sec_categories, 6, '' );
		list( $cat1, $cat2, $cat3, $cat4, $cat5, $cat6 ) = $sec_categories;


		$total_7d = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT COUNT(*) FROM %i WHERE event_time >= %s AND category IN (%s,%s,%s,%s,%s,%s)",
				$table, $since_7d_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			)
		);

		$total_24h = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT COUNT(*) FROM %i WHERE event_time >= %s AND category IN (%s,%s,%s,%s,%s,%s)",
				$table, $since_24h_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			)
		);
		$trend_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT DATE(event_time) AS d, SUM(CASE WHEN action_taken='block' THEN 1 ELSE 0 END) AS c_bl,
				        SUM(CASE WHEN action_taken='challenge' THEN 1 ELSE 0 END) AS c_ch,
				        SUM(CASE WHEN action_taken='rate_limit' THEN 1 ELSE 0 END) AS c_rl
				   FROM %i
				  WHERE event_time >= %s
				    AND category IN (%s,%s,%s,%s,%s,%s)
				  GROUP BY DATE(event_time)
				  ORDER BY d ASC",
				$table, $since_14d_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			),
			ARRAY_A
		);
		$trend_labels = [];
		$trend_bl = [];
		$trend_ch = [];
		$trend_rl = [];
		foreach ( (array) $trend_rows as $r ) {
			$trend_labels[] = (string) ( $r['d'] ?? '' );
			$trend_bl[] = (int) ( $r['c_bl'] ?? 0 );
			$trend_ch[] = (int) ( $r['c_ch'] ?? 0 );
			$trend_rl[] = (int) ( $r['c_rl'] ?? 0 );
		}

		$mr_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT details
				   FROM %i
				  WHERE event_time >= %s
				    AND category='managed_rule'
				  ORDER BY id DESC
				  LIMIT 4000", $table, $since_7d_gmt
			),
			ARRAY_A
		);


		$threat_map = [];
		foreach ( (array) $mr_rows as $r ) {
			$details = [];
			if ( ! empty( $r['details'] ) ) {
				$d = json_decode( (string) $r['details'], true );
				if ( is_array( $d ) ) { $details = $d; }
			}
			$k = sanitize_key( (string) ( $details['category'] ?? 'unknown' ) );
			if ( $k === '' ) { $k = 'unknown'; }
			$threat_map[ $k ] = ( $threat_map[ $k ] ?? 0 ) + 1;
		}
		arsort( $threat_map );
		$top_threats = array_slice( $threat_map, 0, 10, true );

		$threat_labels = array_keys( $top_threats );
		$threat_counts = array_values( $top_threats );

		$action_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT action_taken, COUNT(*) AS c FROM %i WHERE event_time >= %s AND category IN (%s,%s,%s,%s,%s,%s) GROUP BY action_taken ORDER BY c DESC",
				$table, $since_7d_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			),
			ARRAY_A
		);
		$action_labels = [];
		$action_counts = [];
		foreach ( (array) $action_rows as $r ) {
			$action_labels[] = (string) ( $r['action_taken'] ?? '' );
			$action_counts[] = (int) ( $r['c'] ?? 0 );
		}

		$module_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT category, COUNT(*) AS c FROM %i WHERE event_time >= %s AND category IN (%s,%s,%s,%s,%s,%s) GROUP BY category ORDER BY c DESC",
				$table, $since_7d_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			),
			ARRAY_A
		);
		$module_map = [
			'WAF (rules+heuristics)' => 0,
			'API Shield'            => 0,
			'Bot Control'           => 0,
			'DDoS'                  => 0,
		];

		foreach ( (array) $module_rows as $r ) {
			$cat = (string) ( $r['category'] ?? '' );
			$c   = (int) ( $r['c'] ?? 0 );

			if ( $cat === 'managed_rule' || $cat === 'heuristic' ) { $module_map['WAF (rules+heuristics)'] += $c; }
			elseif ( $cat === 'api_shield' ) { $module_map['API Shield'] += $c; }
			elseif ( $cat === 'bot_control' || $cat === 'bot_bad_ua' ) { $module_map['Bot Control'] += $c; }
			elseif ( $cat === 'ddos_shield' ) { $module_map['DDoS'] += $c; }
		}

		$module_labels = array_keys( $module_map );
		$module_counts = array_values( $module_map );

		$method_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT method, COUNT(*) AS c FROM %i WHERE event_time >= %s AND category IN (%s,%s,%s,%s,%s,%s) AND method IS NOT NULL AND method <> '' GROUP BY method ORDER BY c DESC LIMIT 10",
				$table, $since_7d_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			),
			ARRAY_A
		);
		$method_labels = [];
		$method_counts = [];
		foreach ( (array) $method_rows as $r ) {
			$method_labels[] = (string) ( $r['method'] ?? '' );
			$method_counts[] = (int) ( $r['c'] ?? 0 );
		}

		$route_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT route, COUNT(*) AS c FROM %i WHERE event_time >= %s AND category IN (%s,%s,%s,%s,%s,%s) AND route IS NOT NULL AND route <> '' GROUP BY route ORDER BY c DESC LIMIT 10",
				$table, $since_7d_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			),
			ARRAY_A
		);
		$route_labels = [];
		$route_counts = [];
		foreach ( (array) $route_rows as $r ) {
			$route = (string) ( $r['route'] ?? '' );
			if ( strlen( $route ) > 38 ) { $route = substr( $route, 0, 38 ) . '…'; }
			$route_labels[] = $route;
			$route_counts[] = (int) ( $r['c'] ?? 0 );
		}

		$ip_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT ip, COUNT(*) AS c FROM %i WHERE event_time >= %s AND category IN (%s,%s,%s,%s,%s,%s) AND ip IS NOT NULL AND ip <> '' GROUP BY ip ORDER BY c DESC LIMIT 10",
				$table, $since_7d_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			),
			ARRAY_A
		);
		$bubbles = [];
		foreach ( (array) $ip_rows as $r ) {
			$blocks = (int) ( $r['blocks'] ?? 0 );
			$radius = max( 4, min( 18, 4 + (int) floor( $blocks / 2 ) ) );

			$bubbles[] = [
				'label' => (string) ( $r['ip'] ?? '' ),
				'x' => (int) ( $r['total'] ?? 0 ),
				'y' => (int) ( $r['enforced'] ?? 0 ),
				'r' => $radius,
				'blocks' => $blocks,
			];
		}

		$hot_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT url, COUNT(*) AS c FROM %i WHERE event_time >= %s AND category IN (%s,%s,%s,%s,%s,%s) AND url IS NOT NULL AND url <> '' GROUP BY url ORDER BY c DESC LIMIT 10",
				$table, $since_7d_gmt, $cat1, $cat2, $cat3, $cat4, $cat5, $cat6
			),
			ARRAY_A
		);
		$scatter = [];
		foreach ( (array) $hot_rows as $r ) {
			$route = (string) ( $r['route'] ?? '' );
			if ( strlen( $route ) > 32 ) { $route = substr( $route, 0, 32 ) . '…'; }

			$scatter[] = [
				'label' => $route,
				'x' => (int) ( $r['total'] ?? 0 ),
				'y' => (int) ( $r['enforced'] ?? 0 ),
			];
		}

		return [
			'totals' => [ 'last_24h' => $total_24h, 'last_7d' => $total_7d ],
			'trend_labels' => $trend_labels,
			'trend_block' => $trend_bl,
			'trend_challenge' => $trend_ch,
			'trend_rate' => $trend_rl,
			'threat_labels' => $threat_labels,
			'threat_counts' => $threat_counts,
			'action_labels' => $action_labels,
			'action_counts' => $action_counts,
			'module_labels' => $module_labels,
			'module_counts' => $module_counts,
			'method_labels' => $method_labels,
			'method_counts' => $method_counts,
			'route_labels' => $route_labels,
			'route_counts' => $route_counts,
			'bubbles' => $bubbles,
			'scatter' => $scatter,
		];
	}
}

add_action( 'admin_enqueue_scripts', function() {
	$page = isset( $_GET['page'] ) ? sanitize_key( (string) wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	if ( '' === $page || 'aegiswaf' !== $page ) { return; }

	$tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'overview'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	if ( $tab !== 'logs' ) { return; }

	$base_url = plugin_dir_url( dirname( __FILE__, 3 ) ); // plugin root

	wp_enqueue_script(
		'aegiswaf-chartjs',
		$base_url . 'assets/js/chart.umd.min.js',
		[],
		'1.6.9',
		true
	);

	$data = aegiswaf_logs_get_visual_data();
	$json = wp_json_encode( $data );

	$inline = "
	(function(){
		if (typeof Chart === 'undefined') { return; }
		var data = {$json};
		function el(id){ return document.getElementById(id); }

		// Totals
		var t24 = el('aegiswaf_logs_total_24h');
		var t7d = el('aegiswaf_logs_total_7d');
		if (t24) { t24.textContent = String((data.totals && data.totals.last_24h) ? data.totals.last_24h : 0); }
		if (t7d) { t7d.textContent = String((data.totals && data.totals.last_7d) ? data.totals.last_7d : 0); }

		// 1) Area (line w/ fill): enforcement trend
		var c1 = el('aegiswaf_logs_chart_area');
		if (c1) {
			new Chart(c1.getContext('2d'), {
				type: 'line',
				data: {
					labels: data.trend_labels || [],
					datasets: [
						{ label:'Block', data: data.trend_block || [], fill:true, tension:0.3, pointRadius:1 },
						{ label:'Challenge', data: data.trend_challenge || [], fill:true, tension:0.3, pointRadius:1 },
						{ label:'Rate-limit', data: data.trend_rate || [], fill:true, tension:0.3, pointRadius:1 }
					]
				},
				options: { responsive:true, maintainAspectRatio:false, scales:{ y:{ beginAtZero:true } } }
			});
		}

		// 2) Bar: top WAF threat categories
		var c2 = el('aegiswaf_logs_chart_bar');
		if (c2) {
			new Chart(c2.getContext('2d'), {
				type: 'bar',
				data: { labels: data.threat_labels || [], datasets: [{ label:'Top WAF threats (7d)', data: data.threat_counts || [] }] },
				options: { responsive:true, maintainAspectRatio:false, scales:{ y:{ beginAtZero:true } }, plugins:{ legend:{ display:true } } }
			});
		}

		// 3) Doughnut: action outcomes
		var c3 = el('aegiswaf_logs_chart_doughnut');
		if (c3) {
			new Chart(c3.getContext('2d'), {
				type: 'doughnut',
				data: { labels: data.action_labels || [], datasets: [{ data: data.action_counts || [] }] },
				options: { responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:'bottom' } } }
			});
		}

		// 4) Pie: module share
		var c4 = el('aegiswaf_logs_chart_pie');
		if (c4) {
			new Chart(c4.getContext('2d'), {
				type: 'pie',
				data: { labels: data.module_labels || [], datasets: [{ data: data.module_counts || [] }] },
				options: { responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:'bottom' } } }
			});
		}

		// 5) Radar: method mix
		var c5 = el('aegiswaf_logs_chart_radar');
		if (c5) {
			new Chart(c5.getContext('2d'), {
				type: 'radar',
				data: { labels: data.method_labels || [], datasets: [{ label:'HTTP method mix (7d)', data: data.method_counts || [] }] },
				options: { responsive:true, maintainAspectRatio:false, scales:{ r:{ beginAtZero:true } } }
			});
		}

		// 6) Polar area: top routes
		var c6 = el('aegiswaf_logs_chart_polar');
		if (c6) {
			new Chart(c6.getContext('2d'), {
				type: 'polarArea',
				data: { labels: data.route_labels || [], datasets: [{ data: data.route_counts || [] }] },
				options: { responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:'bottom' } } }
			});
		}

		// 7) Bubble: top IPs
		var c7 = el('aegiswaf_logs_chart_bubble');
		if (c7) {
			var pts = (data.bubbles || []).map(function(p){
				return { x:p.x, y:p.y, r:p.r, _label:p.label, _blocks:(p.blocks||0) };
			});
			new Chart(c7.getContext('2d'), {
				type: 'bubble',
				data: { datasets: [{ label:'Top IPs (24h)', data: pts }] },
				options: {
					responsive:true,
					maintainAspectRatio:false,
					plugins: {
						tooltip: { callbacks: { label: function(ctx){
							var p = ctx.raw || {};
							return (p._label||'ip') + ' — total:' + p.x + ', enforced:' + p.y + ', blocks:' + (p._blocks||0);
						}}}
					},
					scales: {
						x:{ beginAtZero:true, title:{ display:true, text:'Total security events (24h)' } },
						y:{ beginAtZero:true, title:{ display:true, text:'Enforced events (24h)' } }
					}
				}
			});
		}

		// 8) Scatter: route hotspot map
		var c8 = el('aegiswaf_logs_chart_scatter');
		if (c8) {
			var pts2 = (data.scatter || []).map(function(p){
				return { x:p.x, y:p.y, _label:p.label };
			});
			new Chart(c8.getContext('2d'), {
				type: 'scatter',
				data: { datasets: [{ label:'Route hotspots (24h)', data: pts2 }] },
				options: {
					responsive:true,
					maintainAspectRatio:false,
					plugins: {
						tooltip: { callbacks: { label: function(ctx){
							var p = ctx.raw || {};
							return (p._label||'route') + ' — total:' + p.x + ', enforced:' + p.y;
						}}}
					},
					scales: {
						x:{ beginAtZero:true, title:{ display:true, text:'Total hits (24h)' } },
						y:{ beginAtZero:true, title:{ display:true, text:'Enforced hits (24h)' } }
					}
				}
			});
		}
	})();";

	wp_add_inline_script( 'aegiswaf-chartjs', $inline, 'after' );
}, 21 );

add_action( 'admin_post_aegiswaf_blocked_action', function() {

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( 'Forbidden' );
	}

	check_admin_referer( 'aegiswaf_blocked_action' );

	$do = isset( $_POST['do'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['do'] ) ) : '';
	$id = isset( $_POST['log_id'] ) ? absint( wp_unslash( $_POST['log_id'] ) ) : 0; // phpcs

	$ip     = isset( $_POST['ip'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['ip'] ) ) : '';
	$route  = isset( $_POST['route'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['route'] ) ) : '';
	$method = isset( $_POST['method'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['method'] ) ) : '';

	$redirect = wp_get_referer();
	if ( ! $redirect ) {
		$redirect = admin_url( 'admin.php?page=aegiswaf&tab=logs' );
	}

	if ( $do === 'unblock' ) {

		$append_line = function( string $list, string $token ) : string {
			$list  = (string) $list;
			$token = trim( $token );
			if ( $token === '' ) { return $list; }

			$lines = preg_split( "/\r\n|\n|\r/", $list );
			$lines = is_array( $lines ) ? $lines : [];

			foreach ( $lines as $ln ) {
				if ( trim( (string) $ln ) === $token ) {
					return $list; // already present
				}
			}

			$lines[] = $token;
			return trim( implode( "\n", array_filter( array_map( 'trim', $lines ) ) ) );
		};

		$did_ddos = false;
		$did_api  = false;

		if ( class_exists( 'AegisWAF_DDoS_Storage' ) ) {

			if ( method_exists( 'AegisWAF_DDoS_Storage', 'get' ) && method_exists( 'AegisWAF_DDoS_Storage', 'update' ) ) {
				$ddos = (array) AegisWAF_DDoS_Storage::get();
				$ddos['allowlist'] = $append_line( (string) ( $ddos['allowlist'] ?? '' ), $ip );
				AegisWAF_DDoS_Storage::update( $ddos );
				$did_ddos = true;
			}

			if ( ! $did_ddos ) {
				foreach ( [ 'aegiswaf_ddos', 'aegiswaf_ddos_settings', 'aegiswaf_ddos_options' ] as $opt ) {
					$val = get_option( $opt, null );
					if ( is_array( $val ) ) {
						$val['allowlist'] = $append_line( (string) ( $val['allowlist'] ?? '' ), $ip );
						update_option( $opt, $val, false );
						$did_ddos = true;
						break;
					}
				}
			}
		}

		if ( class_exists( 'AegisWAF_Storage' ) ) {

			if ( method_exists( 'AegisWAF_Storage', 'get' ) && method_exists( 'AegisWAF_Storage', 'set' ) ) {
				$api = (array) AegisWAF_Storage::get( 'api_shield' );
				if ( isset( $api['allowlist'] ) || ! isset( $api['ip_allowlist'] ) ) {
					$api['allowlist'] = $append_line( (string) ( $api['allowlist'] ?? '' ), $ip );
				} else {
					$api['ip_allowlist'] = $append_line( (string) ( $api['ip_allowlist'] ?? '' ), $ip );
				}
				AegisWAF_Storage::set( 'api_shield', $api );
				$did_api = true;
			}
		}

		if ( ! $did_api ) {
			foreach ( [ 'aegiswaf_api_shield', 'aegiswaf_api_shield_settings', 'aegiswaf_api_shield_options' ] as $opt ) {
				$val = get_option( $opt, null );
				if ( is_array( $val ) ) {
					if ( isset( $val['allowlist'] ) || ! isset( $val['ip_allowlist'] ) ) {
						$val['allowlist'] = $append_line( (string) ( $val['allowlist'] ?? '' ), $ip );
					} else {
						$val['ip_allowlist'] = $append_line( (string) ( $val['ip_allowlist'] ?? '' ), $ip );
					}
					update_option( $opt, $val, false );
					$did_api = true;
					break;
				}
			}
		}

		if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
			AegisWAF_Logger::log(
				$route !== '' ? $route : 'blocked_list',
				$method !== '' ? $method : 'POST',
				'admin_action',
				'unblock',
				[
					'ip'        => $ip,
					'route'     => $route,
					'did_ddos'  => $did_ddos ? 1 : 0,
					'did_api'   => $did_api ? 1 : 0,
					'note'      => 'Unblock added to DDoS allowlist + API Shield allowlist',
				]
			);
		}

		wp_safe_redirect( add_query_arg( [ 'blocked_msg' => 'unblocked' ], $redirect ) );
		exit;
	}

	if ( $do === 'delete' ) {

		if ( $id > 0 && class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'table_name' ) ) {
			global $wpdb;
			$table = AegisWAF_Logger::table_name();
			$wpdb->delete( $table, [ 'id' => $id ], [ '%d' ] ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		}

		wp_safe_redirect( add_query_arg( [ 'blocked_msg' => 'deleted' ], $redirect ) );
		exit;
	}

	if ( $do === 'escalate' ) {

		if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
			AegisWAF_Logger::log( $route !== '' ? $route : 'blocked_list', $method !== '' ? $method : 'POST', 'admin_action', 'escalate', [
				'ip'     => $ip,
				'route'  => $route,
				'log_id' => $id,
				'note'   => 'Escalated from Blocked list for investigation',
			] );
		}

		wp_safe_redirect( add_query_arg( [ 'blocked_msg' => 'escalated' ], $redirect ) );
		exit;
	}

	wp_safe_redirect( $redirect );
	exit;

} );

class AegisWAF_Page_Logs {

    public function render() : void {
        if ( ! current_user_can( 'manage_options' ) ) { return; }

        $is_pro = AegisWAF_Features::is_pro();
        $s = AegisWAF_Storage::get_settings();

        if ( empty( $s['logs'] ) || ! is_array( $s['logs'] ) ) { $s['logs'] = []; }
        $saved_per_page = (int) ( $s['logs']['per_page'] ?? 50 );
        if ( ! in_array( $saved_per_page, [ 50, 100, 200 ], true ) ) { $saved_per_page = 50; }
        $saved_retention = (int) ( $s['logs']['retention_days'] ?? 30 );
        if ( $saved_retention <= 0 ) { $saved_retention = 30; }

        if ( isset( $_POST['aegiswaf_save_logs'] ) ) {
            check_admin_referer( 'aegiswaf_save_logs' );

            $per_page = absint( wp_unslash( $_POST['logs_per_page'] ?? 50 ) );
            if ( ! in_array( $per_page, [ 50, 100, 200 ], true ) ) { $per_page = 50; }

            $days = absint( wp_unslash( $_POST['logs_retention_days'] ?? 30 ) );
            $days = max( 1, min( 3650, $days ) );

            $s['logs']['per_page'] = $per_page;
            $s['logs']['retention_days'] = $days;

            AegisWAF_Storage::update_settings( $s );

            if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
				check_admin_referer( 'aegiswaf_logs_settings' );
                AegisWAF_Logger::log( 'admin', 'POST', 'settings', 'logs_settings_saved', [
                    'per_page' => $per_page,
                    'retention_days' => $days,
                ] );
            }

            $saved_per_page = $per_page;
            $saved_retention = $days;

            echo '<div class="notice notice-success"><p>Log settings saved.</p></div>';
			// Alerts save
			if ( $is_pro && isset( $_POST['aegiswaf_alerts'] ) && is_array( $_POST['aegiswaf_alerts'] ) ) {
				$alerts_in = map_deep( wp_unslash( $_POST['aegiswaf_alerts'] ), 'sanitize_text_field' );
				if ( ! is_array( $alerts_in ) ) { $alerts_in = []; }

				$alerts = [];

				foreach ( $alerts_in as $alert ) {
					if ( empty( $alert['keywords'] ) ) { continue; }

					$alerts[] = [
						'id'       => sanitize_text_field( $alert['id'] ?? uniqid( 'alert_', true ) ),
						'keywords' => sanitize_text_field( $alert['keywords'] ),
						'emails'   => sanitize_text_field( $alert['emails'] ?? '' ),
					];
				}

				$s['logs']['alerts'] = $alerts;
				AegisWAF_Storage::update_settings( $s );
			}
        }

			if ( isset( $_POST['aegiswaf_alert_action'] ) ) {
				check_admin_referer( 'aegiswaf_save_logs' );

				if ( ! $is_pro ) {
					echo '<div class="notice notice-warning"><p>Alert Settings is a PRO feature. Upgrade to enable alerts.</p></div>';
				} else {

					$settings = AegisWAF_Storage::get_settings();
			if ( empty( $settings['logs'] ) || ! is_array( $settings['logs'] ) ) { $settings['logs'] = []; }
			$alerts = $settings['logs']['alerts'] ?? [];
			if ( ! is_array( $alerts ) ) { $alerts = []; }

			$action = sanitize_key( (string) wp_unslash( $_POST['aegiswaf_alert_action'] ) );

			if ( $action === 'delete' && ! empty( $_POST['alert_id'] ) ) {
				$del_id = sanitize_text_field( (string) wp_unslash( $_POST['alert_id'] ) );
				$alerts = array_values( array_filter( $alerts, function( $a ) use ( $del_id ) {
					return (string)($a['id'] ?? '') !== $del_id;
				}));
			}

			if ( $action === 'create' && ! empty( $_POST['aegiswaf_new_alert'] ) && is_array( $_POST['aegiswaf_new_alert'] ) ) {
				$na = map_deep( wp_unslash( $_POST['aegiswaf_new_alert'] ), 'sanitize_text_field' );

				$title    = sanitize_text_field( (string) ( $na['title'] ?? '' ) );
				$keywords = sanitize_text_field( (string) ( $na['keywords'] ?? '' ) );
				$emails   = sanitize_text_field( (string) ( $na['emails'] ?? '' ) );

				if ( $title !== '' && $keywords !== '' ) {
					$alerts[] = [
						'id'       => uniqid( 'alert_', true ),
						'title'    => $title,
						'keywords' => $keywords,
						'emails'   => $emails,
					];
				}
			}

			if ( $action === 'update' && ! empty( $_POST['alert_id'] ) && ! empty( $_POST['aegiswaf_new_alert'] ) && is_array( $_POST['aegiswaf_new_alert'] ) ) {
				$edit_id = sanitize_text_field( (string) wp_unslash( $_POST['alert_id'] ) );
				$na = map_deep( wp_unslash( $_POST['aegiswaf_new_alert'] ), 'sanitize_text_field' );

				$title    = sanitize_text_field( (string) ( $na['title'] ?? '' ) );
				$keywords = sanitize_text_field( (string) ( $na['keywords'] ?? '' ) );
				$emails   = sanitize_text_field( (string) ( $na['emails'] ?? '' ) );

				foreach ( $alerts as $idx => $a ) {
					if ( (string) ( $a['id'] ?? '' ) === $edit_id ) {
						$alerts[$idx]['title']    = $title;
						$alerts[$idx]['keywords'] = $keywords;
						$alerts[$idx]['emails']   = $emails;
						break;
					}
				}
			}

			$settings['logs']['alerts'] = $alerts;
			AegisWAF_Storage::update_settings( $settings );

			$s = $settings;
				}
		}

        if ( isset( $_POST['aegiswaf_run_retention'] ) ) {
            check_admin_referer( 'aegiswaf_save_logs' );

            if ( ! $is_pro ) {
                echo '<div class="notice notice-warning"><p>Retention cleanup is PRO.</p></div>';
            } else {
                $days = $saved_retention;
                $deleted = AegisWAF_Logger::delete_older_than_days( $days );

                AegisWAF_Logger::log( 'admin', 'POST', 'maintenance', 'logs_retention_ran', [
                    'deleted' => (int) $deleted,
                    'days' => (int) $days,
                ] );

                echo '<div class="notice notice-success"><p>' .esc_html( 'Deleted ' . (int) $deleted . ' rows older than ' . (int) $days . ' days.' ) . '</p></div>';

            }
        }

        if ( isset( $_POST['aegiswaf_update_event_action'] ) ) {
            check_admin_referer( 'aegiswaf_update_event_action', 'aegiswaf_update_event_action_nonce' );

            $event_id = isset( $_POST['aegiswaf_event_id'] ) ? absint( wp_unslash( $_POST['aegiswaf_event_id'] ) ) : 0;
            $new_action = isset( $_POST['aegiswaf_event_action'] ) ? sanitize_key( (string) wp_unslash( $_POST['aegiswaf_event_action'] ) ) : '';

            $allowed = [ 'block', 'allow', 'investigate' ];
            if ( $event_id <= 0 || ! in_array( $new_action, $allowed, true ) ) {
                echo '<div class="notice notice-error"><p>Invalid action update.</p></div>';
            } else {
                global $wpdb;

                if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'table_name' ) ) {
                    $table = AegisWAF_Logger::table_name();
                    $old_action = (string) $wpdb->get_var( $wpdb->prepare( "SELECT action_taken FROM %i WHERE id = %d", $table, $event_id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

                    $updated = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $table,
                        [ 'action_taken' => $new_action ],
                        [ 'id' => $event_id ],
                        [ '%s' ],
                        [ '%d' ]
                    );
					if ( $new_action === 'allow' && class_exists( 'AegisWAF_Overrides' ) ) {
						$row = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
							$wpdb->prepare( "SELECT route, method, ip FROM %i WHERE id = %d", $table, $event_id ),
							ARRAY_A
						);

						if ( is_array( $row ) ) {
							$rip = (string) ( $row['ip'] ?? '' );
							$rroute = (string) ( $row['route'] ?? '' );
							$rmethod = (string) ( $row['method'] ?? '' );

							// 24 hours is a safe default; you can raise to 7 days if you want.
							AegisWAF_Overrides::add_allow( $rip, $rroute, $rmethod, 86400 );
						}
					}
					if ( $new_action === 'block' && class_exists( 'AegisWAF_Overrides' ) ) {
						$row = $wpdb->get_row( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
							$wpdb->prepare( "SELECT route, method, ip FROM %i WHERE id = %d", $table, $event_id ),
							ARRAY_A
						);

						if ( is_array( $row ) ) {
							AegisWAF_Overrides::remove_allow(
								(string) ( $row['ip'] ?? '' ),
								(string) ( $row['route'] ?? '' ),
								(string) ( $row['method'] ?? '' )
							);
						}
					}
                    if ( false === $updated ) {
                        echo '<div class="notice notice-error"><p>Failed to update event action.</p></div>';
                    } else {
                        // Log the administrative override for auditability.
                        if ( method_exists( 'AegisWAF_Logger', 'log' ) ) {
                            AegisWAF_Logger::log( 'admin', 'POST', 'logs', 'event_action_updated', [
                                'event_id' => (int) $event_id,
                                'old_action' => (string) $old_action,
                                'new_action' => (string) $new_action,
                            ] );
                        }

                        echo '<div class="notice notice-success"><p>Event action updated.</p></div>';
                    }
                } else {
                    echo '<div class="notice notice-error"><p>Logger not available.</p></div>';
                }
            }
        }


        $paged = isset( $_GET['paged'] ) ? max( 1, (int) $_GET['paged'] ) : 1;
        $per_page = $saved_per_page;

        $filters = [
            'q' => isset( $_GET['q'] ) ? sanitize_text_field( wp_unslash( $_GET['q'] ) ) : '',
            'route' => isset( $_GET['route'] ) ? sanitize_text_field( wp_unslash( $_GET['route'] ) ) : '',
            'method' => isset( $_GET['method'] ) ? sanitize_text_field( wp_unslash( $_GET['method'] ) ) : '',
            'category' => isset( $_GET['category'] ) ? sanitize_text_field( wp_unslash( $_GET['category'] ) ) : '',
            'action' => isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : '',
            'ip' => isset( $_GET['ip'] ) ? sanitize_text_field( wp_unslash( $_GET['ip'] ) ) : '',
            'date_from' => isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : '',
            'date_to' => isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : '',
            'alerts_only' => ! empty( $_GET['alerts_only'] ) ? 1 : 0,
        ];

        $offset = ( $paged - 1 ) * $per_page;

        try {
            $total = AegisWAF_Logger::count_filtered( $filters );
            $rows  = AegisWAF_Logger::fetch_filtered( $per_page, $offset, $filters );
        } catch ( Throwable $e ) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log( '[AegisWAF] Logs UI query failed: ' . $e->getMessage() );
            echo '<div class="notice notice-error"><p>Logs query failed. Check debug.log.</p></div>';
            $total = 0;
            $rows = [];
        }

        $total_pages = max( 1, (int) ceil( $total / $per_page ) );

        echo '<div class="wrap">';
        echo '<h1>Log Management / Attack Story</h1>';
        AegisWAF_Pro_Notice::render();
        echo '<p class="description">The Activity Logs Management & Attack Story tab records all meaningful WAF-related events, including blocked requests, allowed traffic, rule triggers, and enforcement decisions. Logs include critical context such as IP address, endpoint, rule source, and timestamps, enabling administrators to audit behavior and investigate incidents. Free provides core logging and basic filtering, while Pro adds advanced filters, extended retention, and export options (CSV/JSON) for compliance, forensics, and long-term analysis.</p>';

		echo '<style>
		.aegiswaf-logs-viz-wrap{
			display:flex;
			flex-direction:column;
			gap:12px;
			margin:12px 0 14px;
		}
		.aegiswaf-logs-viz-header{
			display:flex;
			align-items:flex-start;
			justify-content:space-between;
			gap:12px;
		}
		.aegiswaf-logs-viz-kpis{
			font-weight:700;
			margin-top:6px;
		}
		.aegiswaf-logs-viz-grid{
			display:grid;
			grid-template-columns: repeat(4, minmax(220px, 1fr));
			gap:12px;
		}
		@media (max-width: 1300px){ .aegiswaf-logs-viz-grid{ grid-template-columns: repeat(2, minmax(240px, 1fr)); } }
		@media (max-width: 782px){ .aegiswaf-logs-viz-grid{ grid-template-columns: 1fr; } }

		.aegiswaf-logs-viz-panel{
			border:1px solid #e5e7eb;
			border-radius:12px;
			padding:12px;
			background:#fff;
		}
		.aegiswaf-logs-viz-title{ font-weight:800; margin:0 0 4px; }
		.aegiswaf-logs-viz-sub{ color:#646970; font-size:12px; margin:0 0 10px; }
		.aegiswaf-logs-viz-canvas{ height:220px; position:relative; }
		.aegiswaf-logs-viz-section{
			border:1px solid #e5e7eb;
			border-radius:12px;
			background:#fff;
			padding:14px;
		}
		</style>';

		echo '<div class="aegiswaf-logs-viz-wrap">';
			echo '<div class="aegiswaf-logs-viz-section">';
				echo '<div class="aegiswaf-logs-viz-header">';
					echo '<div>';
						echo '<h2 style="margin:0;">Enterprise Visual Intelligence</h2>';
						echo '<p class="description" style="margin:6px 0 0;">8 high-signal charts across WAF Rules, API Shield, Bot Control, DDoS, and Attack Story logs.</p>';
					echo '</div>';
					echo '<div class="aegiswaf-logs-viz-kpis">Last 24h: <span id="aegiswaf_logs_total_24h">0</span> &nbsp; | &nbsp; Last 7d: <span id="aegiswaf_logs_total_7d">0</span></div>';
				echo '</div>';

				echo '<div class="aegiswaf-logs-viz-grid" style="margin-top:12px;">';

					echo '<div class="aegiswaf-logs-viz-panel">';
						echo '<div class="aegiswaf-logs-viz-title">Enforcement Trend</div>';
						echo '<div class="aegiswaf-logs-viz-sub">Area (14d): block / challenge / rate-limit</div>';
						echo '<div class="aegiswaf-logs-viz-canvas"><canvas id="aegiswaf_logs_chart_area"></canvas></div>';
					echo '</div>';

					echo '<div class="aegiswaf-logs-viz-panel">';
						echo '<div class="aegiswaf-logs-viz-title">Top WAF Threats</div>';
						echo '<div class="aegiswaf-logs-viz-sub">Bar (7d): SQLi / XSS / RCE / etc.</div>';
						echo '<div class="aegiswaf-logs-viz-canvas"><canvas id="aegiswaf_logs_chart_bar"></canvas></div>';
					echo '</div>';

					echo '<div class="aegiswaf-logs-viz-panel">';
						echo '<div class="aegiswaf-logs-viz-title">Action Outcomes</div>';
						echo '<div class="aegiswaf-logs-viz-sub">Doughnut (7d): what AegisWAF is doing</div>';
						echo '<div class="aegiswaf-logs-viz-canvas"><canvas id="aegiswaf_logs_chart_doughnut"></canvas></div>';
					echo '</div>';

					echo '<div class="aegiswaf-logs-viz-panel">';
						echo '<div class="aegiswaf-logs-viz-title">Module Share</div>';
						echo '<div class="aegiswaf-logs-viz-sub">Pie (7d): WAF / API / Bot / DDoS</div>';
						echo '<div class="aegiswaf-logs-viz-canvas"><canvas id="aegiswaf_logs_chart_pie"></canvas></div>';
					echo '</div>';

				echo '</div>'; // grid
			echo '</div>'; // section 1

			echo '<div class="aegiswaf-logs-viz-section">';
				echo '<h2 style="margin:0 0 6px;">Attack Story Deep Dive</h2>';
				echo '<p class="description" style="margin:0 0 12px;">Where the attacks are landing, how they arrive, and which actors are producing the highest enforcement pressure.</p>';

				echo '<div class="aegiswaf-logs-viz-grid">';

					echo '<div class="aegiswaf-logs-viz-panel">';
						echo '<div class="aegiswaf-logs-viz-title">Method Mix</div>';
						echo '<div class="aegiswaf-logs-viz-sub">Radar (7d): HTTP method distribution</div>';
						echo '<div class="aegiswaf-logs-viz-canvas"><canvas id="aegiswaf_logs_chart_radar"></canvas></div>';
					echo '</div>';

					echo '<div class="aegiswaf-logs-viz-panel">';
						echo '<div class="aegiswaf-logs-viz-title">Top Routes</div>';
						echo '<div class="aegiswaf-logs-viz-sub">Polar Area (7d): most targeted endpoints</div>';
						echo '<div class="aegiswaf-logs-viz-canvas"><canvas id="aegiswaf_logs_chart_polar"></canvas></div>';
					echo '</div>';

					echo '<div class="aegiswaf-logs-viz-panel">';
						echo '<div class="aegiswaf-logs-viz-title">Top IP Offenders</div>';
						echo '<div class="aegiswaf-logs-viz-sub">Bubble (24h): total vs enforced, size=blocks</div>';
						echo '<div class="aegiswaf-logs-viz-canvas"><canvas id="aegiswaf_logs_chart_bubble"></canvas></div>';
					echo '</div>';

					echo '<div class="aegiswaf-logs-viz-panel">';
						echo '<div class="aegiswaf-logs-viz-title">Route Hotspots</div>';
						echo '<div class="aegiswaf-logs-viz-sub">Scatter (24h): total hits vs enforced hits</div>';
						echo '<div class="aegiswaf-logs-viz-canvas"><canvas id="aegiswaf_logs_chart_scatter"></canvas></div>';
					echo '</div>';

				echo '</div>'; // grid
			echo '</div>'; // section 2

		echo '</div>'; // wrap

		$edit_alert_id = isset( $_GET['edit_alert'] ) ? sanitize_text_field( (string) wp_unslash( $_GET['edit_alert'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$edit_alert = null;
		if ( $edit_alert_id !== '' ) {
			$existing_alerts = $s['logs']['alerts'] ?? [];
			if ( is_array( $existing_alerts ) ) {
				foreach ( $existing_alerts as $a ) {
					if ( (string) ( $a['id'] ?? '' ) === $edit_alert_id ) {
						$edit_alert = $a;
						break;
					}
				}
			}
		}

		echo '<div style="display:flex; gap:20px;">';

		echo '<form method="post" style="display:flex; gap:20px; width:100%;">';
		wp_nonce_field( 'aegiswaf_save_logs' );

		echo '<div class="aegiswaf-card" style="flex:0 0 360px;"><h2>Log Settings</h2>';

		echo '<table class="form-table" role="presentation"><tbody>';
		echo '<tr><th scope="row">Rows per page</th><td><select name="logs_per_page">';
		foreach ( [ 50, 100, 200 ] as $n ) {
			echo '<option value="' . esc_attr( (string) $n ) . '" ' . selected( $saved_per_page, $n, false ) . '>' . esc_html( (string) $n ) . '</option>';
		}
		echo '</select></td></tr>';

		echo '<tr><th scope="row">Retention days (PRO)</th><td>';
		echo '<input type="number" name="logs_retention_days" min="1" max="3650" value="' . esc_attr( (string) $saved_retention ) . '"> ';
		echo '<span class="description">Auto-delete older logs.</span></td></tr>';
		echo '</tbody></table>';

		echo '<p>';
		echo '<button class="button button-primary" type="submit" name="aegiswaf_save_logs" value="1">Save Log Settings</button> ';
		echo '<button class="button" type="submit" name="aegiswaf_run_retention" value="1">Run Retention Cleanup Now</button>';
		echo '</p>';

		echo '</div>'; // end Log Settings card

		echo '<div class="aegiswaf-card" id="aegiswaf-alert-settings" style="flex:1;border:1px solid #ccd0d4;background:#fff;padding:14px;border-radius:10px;">';
		echo '<h3>Alert Settings</h3>';
		echo '<p class="description">Create alerts that trigger when log entries match keywords (route/action/details).</p>';
        if ( ! $is_pro ) {
            AegisWAF_Utils::pro_gate_box(
                'Unlock PRO Intelligence & Automation',
                'PRO unlocks creating alerts, notifications and alert management.'
            );
        }
		if ( ! $is_pro ) { echo '<div class="aegiswaf-pro-dim">'; }
		echo '<input type="hidden" name="alert_id" value="' . esc_attr( $edit_alert_id ) . '" />';

		$is_editing = ( $edit_alert && is_array( $edit_alert ) );
		$form_title = $is_editing ? 'Edit Alert' : 'Create Alert';
		$btn_label  = $is_editing ? 'Update Alert' : 'Create Alert';
		$btn_action = $is_editing ? 'update' : 'create';

		$val_title    = $is_editing ? (string) ( $edit_alert['title'] ?? '' ) : '';
		$val_keywords = $is_editing ? (string) ( $edit_alert['keywords'] ?? '' ) : '';
		$val_emails   = $is_editing ? (string) ( $edit_alert['emails'] ?? '' ) : '';

		echo '<h4 style="margin-top:10px;">' . esc_html( $form_title ) . '</h4>';
		echo '<table class="form-table" role="presentation"><tbody>';

		echo '<tr>';
		echo '<th scope="row"><label for="alert_title">Alert Title</label></th>';
		echo '<td><input type="text" id="alert_title" name="aegiswaf_new_alert[title]" class="regular-text" value="' . esc_attr( $val_title ) . '" placeholder="e.g. SQL Injection Attempts" /></td>';
		echo '</tr>';

		echo '<tr>';
		echo '<th scope="row"><label for="alert_keywords">Keyword String</label></th>';
		echo '<td>';
		echo '<input type="text" id="alert_keywords" name="aegiswaf_new_alert[keywords]" class="regular-text" value="' . esc_attr( $val_keywords ) . '" placeholder="sql,union,select" />';
		echo '<p class="description">Comma-separated. Matched against route/action/details.</p>';
		echo '</td>';
		echo '</tr>';

		echo '<tr>';
		echo '<th scope="row"><label for="alert_emails">Send alerts to</label></th>';
		echo '<td>';
		echo '<input type="text" id="alert_emails" name="aegiswaf_new_alert[emails]" class="regular-text" value="' . esc_attr( $val_emails ) . '" placeholder="admin@site.com, security@site.com" />';
		echo '<p class="description">Leave blank to send to the WordPress admin email.</p>';
		echo '</td>';
		echo '</tr>';

		echo '</tbody></table>';

		echo '<p>';
		echo '<button type="submit" class="button button-primary" name="aegiswaf_alert_action" value="' . esc_attr( $btn_action ) . '">' . esc_html( $btn_label ) . '</button> ';

		if ( $is_editing ) {
			$cancel_url = remove_query_arg( [ 'edit_alert' ] );
			echo '<a class="button" href="' . esc_url( $cancel_url ) . '">Cancel</a>';
		}
		echo '</p>';

		echo '<hr />';
		echo '<h4>Existing Alerts</h4>';

		$alerts = $s['logs']['alerts'] ?? [];
		echo '<table class="widefat fixed striped">';
		echo '<thead><tr><th>Title</th><th>Keyword String</th><th>Send alerts to</th><th style="width:160px;">Actions</th></tr></thead><tbody>';

		if ( empty( $alerts ) ) {
			echo '<tr><td colspan="4"><em>No alerts configured.</em></td></tr>';
		} else {
			foreach ( $alerts as $a ) {
				$id = (string) ( $a['id'] ?? '' );

				$edit_url = add_query_arg( [ 'edit_alert' => $id ] );
				echo '<tr>';
				echo '<td>' . esc_html( (string) ( $a['title'] ?? '' ) ) . '</td>';
				echo '<td>' . esc_html( (string) ( $a['keywords'] ?? '' ) ) . '</td>';
				echo '<td>' . esc_html( (string) ( $a['emails'] ?? '' ) ) . '</td>';

				echo '<td>';
				echo '<a button class="button" href="' . esc_url( $edit_url ) . '">Edit</a> ';

				echo '<button class="button button-primary"
					class="button-link-delete"
					name="aegiswaf_alert_action"
					value="delete"
					onclick="this.form.alert_id.value=\'' . esc_js( $id ) . '\';">
					Delete
				</button>';
				echo '</td>';

				echo '</tr>';
			}
		}

		echo '</tbody></table>';
		if ( ! $is_pro ) { echo '</div>'; }
		echo '</div>'; // end Alert Settings card

		echo '</form>'; // end unified form
		echo '</div>'; // end flex wrapper
		echo '<p></p>';

        echo '<div class="aegiswaf-card">';
        echo '<form method="get" style="display:flex; gap:10px; flex-wrap:wrap; align-items:flex-end;">';
        echo '<input type="hidden" name="page" value="aegiswaf">';
        echo '<input type="hidden" name="tab" value="logs">';

        echo '<div><label>Search<br><input type="text" name="q" value="' . esc_attr( $filters['q'] ) . '" placeholder="route, action, IP, details..." style="width:260px"></label></div>';
        echo '<div><label>Route contains<br><input type="text" name="route" value="' . esc_attr( $filters['route'] ) . '" placeholder="/wp-json/" style="width:220px"></label></div>';

        echo '<div><label>Method<br><select name="method"><option value="">Any</option>';
        foreach ( [ 'GET','POST','PUT','PATCH','DELETE','HEAD','OPTIONS' ] as $m ) {
            echo '<option value="' . esc_attr( $m ) . '" ' . selected( $filters['method'], $m, false ) . '>' . esc_html( $m ) . '</option>';
        }
        echo '</select></label></div>';

        echo '<div><label>Event Type (Category)<br><input type="text" name="category" value="' . esc_attr( $filters['category'] ) . '" placeholder="managed_rules / heuristic / error" style="width:190px"></label></div>';

        echo '<div><label>Action<br><select name="action"><option value="">Any</option>';
        foreach ( [ 'log'=>'log', 'allow'=>'allow', 'block'=>'block', 'investigate'=>'investigate', 'challenge'=>'challenge', 'rate_limit'=>'rate_limit', 'settings'=>'settings', 'warning'=>'warning', 'error'=>'error' ] as $k => $lab ) {
            echo '<option value="' . esc_attr( $k ) . '" ' . selected( $filters['action'], $k, false ) . '>' . esc_html( $lab ) . '</option>';
        }
        echo '</select></label></div>';

        echo '<div><label>IP<br><input type="text" name="ip" value="' . esc_attr( $filters['ip'] ) . '" placeholder="1.2.3.4" style="width:150px"></label></div>';
        echo '<div><label>From<br><input type="date" name="date_from" value="' . esc_attr( $filters['date_from'] ) . '"></label></div>';
        echo '<div><label>To<br><input type="date" name="date_to" value="' . esc_attr( $filters['date_to'] ) . '"></label></div>';

        echo '<div><label><input type="checkbox" name="alerts_only" value="1" ' . checked( $filters['alerts_only'], 1, false ) . '> Alerts only</label></div>';

        echo '<div><button class="button button-primary" type="submit">Filter</button> '; 
        $reset_url = add_query_arg( [ 'page' => 'aegiswaf', 'tab' => 'logs' ], admin_url( 'admin.php' ) );
        echo '<a class="button" href="' . esc_url( $reset_url ) . '">Reset</a></div>';

        echo '</form>';
        echo '<p class="description">Showing ' . count( $rows ) . ' of ' . (int) $total . ' events. Page size: ' . (int) $per_page . '.</p>';

        echo '<div class="aegiswaf-card"><h2>Events</h2>';
        echo '<table class="widefat fixed striped" style="table-layout:fixed;width:100%;"><thead><tr>';
        echo '<th style="width:160px">Time (UTC)</th>';
        echo '<th style="width:260px">Route</th>';
        echo '<th style="width:80px">Method</th>';
        echo '<th style="width:140px">Event Type</th>';
		echo '<th style="width:140px">Action</th>';
		echo '<th style="width:140px; padding-left:12px;">IP</th>';
		echo '<th style="padding-left:12px;">Details</th>';
        echo '</tr></thead><tbody>';

        if ( empty( $rows ) ) {
            echo '<tr><td colspan="7">No events yet. Generate traffic (front + /wp-json/*) and ensure logging is enabled.</td></tr>';
        } else {
            foreach ( $rows as $r ) {
                $details = [];
                if ( ! empty( $r['details'] ) ) {
                    $decoded = json_decode( (string) $r['details'], true );
                    if ( is_array( $decoded ) ) { $details = $decoded; }
                }

                $level = method_exists( 'AegisWAF_Logger', 'alert_level' ) ? AegisWAF_Logger::alert_level( $r ) : 'info';
                $badge = '<span class="badge badge-' . esc_attr( $level ) . '">' . esc_html( strtoupper( $level ) ) . '</span>';

                $route = (string) ( $r['route'] ?? '' );
                $method = (string) ( $r['method'] ?? '' );
                $category = (string) ( $r['category'] ?? '' );
                $action = (string) ( $r['action_taken'] ?? '' );
                $ip = (string) ( $r['ip'] ?? '' );

                $short = '';
                if ( isset( $details['reason'] ) ) { $short = (string) $details['reason']; }
                elseif ( isset( $details['msg'] ) ) { $short = (string) $details['msg']; }
                elseif ( isset( $details['hit'] ) ) { $short = (string) $details['hit']; }
                elseif ( isset( $details['key'] ) ) { $short = (string) $details['key']; }

                if ( $short === '' ) {
                    $short = wp_json_encode( $details );
                }
                $short = is_string( $short ) ? $short : '';

                echo '<tr>';
                echo '<td>' . esc_html( (string) ( $r['event_time'] ?? '' ) ) . '</td>';
                echo '<td style="width:260px;overflow:hidden;text-overflow:ellipsis;"><code>' . esc_html( $route ) . '</code></td>';
                echo '<td>' . esc_html( $method ) . '</td>';
                echo '<td>' . wp_kses_post( $badge ) . ' <code>' . esc_html( $category ) . '</code></td>';
                $id = (int) ( $r['id'] ?? 0 );

                echo '<td>';
                // Only allow triage override for the actionable outcomes (false-positive block, etc.)
                if ( $id > 0 && in_array( $action, [ 'block', 'allow', 'investigate' ], true ) ) {
                    echo '<form method="post" style="margin:0;display:flex;gap:6px;align-items:center;">';
                    wp_nonce_field( 'aegiswaf_update_event_action', 'aegiswaf_update_event_action_nonce' );
                    echo '<input type="hidden" name="aegiswaf_update_event_action" value="1">';
                    echo '<input type="hidden" name="aegiswaf_event_id" value="' . (int) $id . '">';

                    echo '<select name="aegiswaf_event_action" style="min-width:95px">';
                    foreach ( [ 'block' => 'Block', 'allow' => 'Allow', 'investigate' => 'Investigate' ] as $k => $lab ) {
                        echo '<option value="' . esc_attr( $k ) . '" ' . selected( $action, $k, false ) . '>' . esc_html( $lab ) . '</option>';
                    }
                    echo '</select>';

                    echo '<button type="submit" class="button button-small">Save</button>';
                    echo '</form>';
                } else {
                    // Non-triage actions (challenge/rate_limit/log/settings/etc.) remain read-only labels.
                    echo '<code>' . esc_html( $action ) . '</code>';
                }
                echo '</td>';
                echo '<td><code>' . esc_html( $ip ) . '</code></td>';

                $full = ! empty( $details ) ? wp_json_encode( $details, JSON_PRETTY_PRINT ) : (string) ( $r['details'] ?? '' );
                $full = (string) $full;

                echo '<td style="overflow-wrap:anywhere;word-break:break-word;"><details><summary>' . esc_html( mb_substr( $short, 0, 140 ) ) . '</summary><pre style="white-space:pre-wrap; margin:8px 0 0;">' . esc_html( $full ) . '</pre></details></td>';
                echo '</tr>';
            }
        }
		echo '</div>';
		echo '<p></p>';
        echo '</tbody></table>';

        $base = remove_query_arg( 'paged' );
        $links = paginate_links( [
            'base'      => esc_url_raw( add_query_arg( 'paged', '%#%', $base ) ),
            'format'    => '',
            'prev_text' => '&laquo;',
            'next_text' => '&raquo;',
            'total'     => $total_pages,
            'current'   => $paged,
        ] );

        echo '<div style="margin-top:12px;">' . ( $links ? wp_kses_post( $links ) : '' ) . '</div>';

		$window_label = 'Last 24h';
		$story_filters = [
			'q'          => '',
			'route'      => '',
			'method'     => '',
			'category'   => '',
			'action'     => '',
			'ip'         => '',
			'date_from'  => '',
			'date_to'    => '',
			'alerts_only'=> 0,
		];

		if ( ! empty( $filters['ip'] ) )    { $story_filters['ip'] = $filters['ip']; }
		if ( ! empty( $filters['route'] ) ) { $story_filters['route'] = $filters['route']; }

		if ( ! empty( $filters['date_from'] ) || ! empty( $filters['date_to'] ) ) {
			$story_filters['date_from'] = (string) ( $filters['date_from'] ?? '' );
			$story_filters['date_to']   = (string) ( $filters['date_to'] ?? '' );
			$window_label = 'Filtered date range';
		} else {
			$story_filters['date_from'] = gmdate( 'Y-m-d', time() - 24 * 3600 );
			$story_filters['date_to']   = gmdate( 'Y-m-d' );
		}

		$story_limit = 1200;
		$story_logs  = AegisWAF_Logger::fetch_filtered( $story_limit, 0, $story_filters );

		echo '<div class="aegiswaf-card" style="margin-top:16px;">';
		echo '<h2>Attack Story <span class="description">(' . esc_html( $window_label ) . ')</span></h2>';
		echo '<p class="description">Proof of protection: what AegisWAF inspected, what it blocked, and what threats were detected.</p>';

		if ( ! $is_pro ) {
			AegisWAF_Utils::pro_gate_box(
				'Unlock PRO Intelligence & Automation',
				'PRO unlocks the Attack Story details and reports.'
			);
			echo '</div>'; // end card
		} else {

			if ( empty( $story_logs ) ) {
				echo '<p>No events found for this window.</p>';
				echo '</div>'; // end card
			} else {

			$inc = function( array &$arr, string $k, int $n = 1 ) {
				if ( $k === '' ) { $k = '(unknown)'; }
				$arr[ $k ] = ( $arr[ $k ] ?? 0 ) + $n;
			};

			$parse_rule = function( $details ) : string {
				if ( is_string( $details ) ) {
					$decoded = json_decode( $details, true );
					if ( is_array( $decoded ) ) { $details = $decoded; }
				}
				if ( is_array( $details ) ) {
					foreach ( [ 'rule_id', 'id', 'rule', 'signature', 'sig', 'tag', 'name', 'code' ] as $k ) {
						if ( ! empty( $details[ $k ] ) && is_string( $details[ $k ] ) ) {
							return $details[ $k ];
						}
					}
					if ( ! empty( $details['category'] ) && ! empty( $details['path'] ) ) {
						return (string) $details['category'];
					}
				}
				return '';
			};

			$is_block = function( string $action_taken ) : bool {
				$a = strtoupper( $action_taken );
				return ( strpos( $a, 'BLOCK' ) !== false || strpos( $a, 'DENY' ) !== false );
			};
			$is_challenge = function( string $action_taken ) : bool {
				$a = strtoupper( $action_taken );
				return ( strpos( $a, 'CHALLENGE' ) !== false || strpos( $a, 'CAPTCHA' ) !== false );
			};
			$is_rate = function( string $action_taken ) : bool {
				$a = strtoupper( $action_taken );
				return ( strpos( $a, 'RATE' ) !== false || strpos( $a, 'LIMIT' ) !== false );
			};

			$total = 0;
			$blocks = 0;
			$challenges = 0;
			$ratelimits = 0;
			$allows = 0;

			$ips = [];
			$routes = [];
			$cats = [];
			$actions = [];
			$rules = [];

			$hour_counts = array_fill( 0, 24, 0 );
			$now = time();

			foreach ( $story_logs as $r ) {
				$total++;

				$ip = (string) ( $r['ip'] ?? '' );
				$route = (string) ( $r['route'] ?? '' );
				$cat = (string) ( $r['category'] ?? '' );
				$action_taken = (string) ( $r['action_taken'] ?? '' );

				$inc( $ips, $ip );
				$inc( $routes, $route );
				$inc( $cats, $cat );
				$inc( $actions, strtoupper( $action_taken ) );

				if ( $is_block( $action_taken ) ) { $blocks++; }
				elseif ( $is_challenge( $action_taken ) ) { $challenges++; }
				elseif ( $is_rate( $action_taken ) ) { $ratelimits++; }
				else { $allows++; }

				$rule = $parse_rule( $r['details'] ?? '' );
				if ( $rule !== '' ) { $inc( $rules, $rule ); }

				$time_raw = (string) ( $r['event_time'] ?? ( $r['created_at'] ?? '' ) );
				$ts = $time_raw !== '' ? strtotime( $time_raw . ' UTC' ) : 0;

				if ( $ts ) {
					$age = $now - $ts;
					if ( $age >= 0 && $age <= 24 * 3600 ) {
						$bucket = 23 - (int) floor( $age / 3600 ); // oldest→0, newest→23
						if ( $bucket >= 0 && $bucket <= 23 ) {
							$hour_counts[ $bucket ]++;
						}
					}
				}
			}

			$unique_ips = count( $ips );
			$unique_routes = count( $routes );

			arsort( $ips ); arsort( $routes ); arsort( $cats ); arsort( $rules ); arsort( $actions );

			$top_ips = array_slice( $ips, 0, 5, true );
			$top_routes = array_slice( $routes, 0, 5, true );
			$top_cats = array_slice( $cats, 0, 5, true );
			$top_rules = array_slice( $rules, 0, 5, true );

			$max_hour = max( 1, max( $hour_counts ) );

			echo '<div class="aegiswaf-story-cards">';
				echo '<div class="aegiswaf-story-card"><div class="k">Requests inspected</div><div class="v">' . esc_html( number_format_i18n( $total ) ) . '</div><div class="s">Logged events captured</div></div>';
				echo '<div class="aegiswaf-story-card"><div class="k">Blocked</div><div class="v aegiswaf-v-bad">' . esc_html( number_format_i18n( $blocks ) ) . '</div><div class="s">Threats stopped</div></div>';
				echo '<div class="aegiswaf-story-card"><div class="k">Challenges</div><div class="v">' . esc_html( number_format_i18n( $challenges ) ) . '</div><div class="s">Bots slowed down</div></div>';
				echo '<div class="aegiswaf-story-card"><div class="k">Rate-limited</div><div class="v">' . esc_html( number_format_i18n( $ratelimits ) ) . '</div><div class="s">Abuse throttled</div></div>';
				echo '<div class="aegiswaf-story-card"><div class="k">Unique IPs</div><div class="v">' . esc_html( number_format_i18n( $unique_ips ) ) . '</div><div class="s">Active sources</div></div>';
				echo '<div class="aegiswaf-story-card"><div class="k">Unique routes</div><div class="v">' . esc_html( number_format_i18n( $unique_routes ) ) . '</div><div class="s">Targeted endpoints</div></div>';
			echo '</div>';

			echo '<div class="aegiswaf-story-grid">';

				// Timeline spark bars
				echo '<div class="aegiswaf-story-panel">';
				echo '<h3>Activity Timeline <span class="description">(hourly, UTC)</span></h3>';
				echo '<div class="aegiswaf-spark">';
				foreach ( $hour_counts as $c ) {
					$h = (int) round( ( $c / $max_hour ) * 44 ); // px
					echo '<span class="bar" title="' . esc_attr( (string) $c ) . '" style="height:' . esc_attr( (string) max( 2, $h ) ) . 'px"></span>';
				}
				echo '</div>';
				echo '<div class="description">Higher bars = more WAF activity captured.</div>';
				echo '</div>';

				// Top Sources
				echo '<div class="aegiswaf-story-panel">';
				echo '<h3>Top Sources (IPs)</h3>';
				$max = max( 1, (int) ( reset( $top_ips ) ?: 1 ) );
				foreach ( $top_ips as $k => $v ) {
					$pct = (int) round( ( $v / $max ) * 100 );
					echo '<div class="aegiswaf-barrow"><div class="lbl">' . esc_html( $k ) . '</div><div class="barwrap"><div class="fill" style="width:' . esc_attr( (string) $pct ) . '%"></div></div><div class="val">' . esc_html( (string) $v ) . '</div></div>';
				}
				echo '</div>';

				// Top Targeted Routes
				echo '<div class="aegiswaf-story-panel">';
				echo '<h3>Top Targeted Routes</h3>';
				$max = max( 1, (int) ( reset( $top_routes ) ?: 1 ) );
				foreach ( $top_routes as $k => $v ) {
					$pct = (int) round( ( $v / $max ) * 100 );
					echo '<div class="aegiswaf-barrow"><div class="lbl">' . esc_html( $k ) . '</div><div class="barwrap"><div class="fill" style="width:' . esc_attr( (string) $pct ) . '%"></div></div><div class="val">' . esc_html( (string) $v ) . '</div></div>';
				}
				echo '</div>';

				echo '<div class="aegiswaf-story-panel">';
				echo '<h3>Top Detected Types</h3>';
				$max = max( 1, (int) ( reset( $top_cats ) ?: 1 ) );
				foreach ( $top_cats as $k => $v ) {
					$pct = (int) round( ( $v / $max ) * 100 );
					echo '<div class="aegiswaf-barrow"><div class="lbl">' . esc_html( $k ) . '</div><div class="barwrap"><div class="fill" style="width:' . esc_attr( (string) $pct ) . '%"></div></div><div class="val">' . esc_html( (string) $v ) . '</div></div>';
				}

				if ( ! empty( $top_rules ) ) {
					echo '<hr style="margin:12px 0;border:none;border-top:1px solid #eee;">';
					echo '<h3>Top Triggered Rules</h3>';
					$max = max( 1, (int) ( reset( $top_rules ) ?: 1 ) );
					foreach ( $top_rules as $k => $v ) {
						$pct = (int) round( ( $v / $max ) * 100 );
						echo '<div class="aegiswaf-barrow"><div class="lbl">' . esc_html( $k ) . '</div><div class="barwrap"><div class="fill" style="width:' . esc_attr( (string) $pct ) . '%"></div></div><div class="val">' . esc_html( (string) $v ) . '</div></div>';
					}
				}
				echo '</div>';

			echo '</div>'; // grid

			$proof = sprintf(
				'In this window, AegisWAF inspected %s requests and enforced %s blocks, %s challenges, and %s rate-limit actions.',
				number_format_i18n( $total ),
				number_format_i18n( $blocks ),
				number_format_i18n( $challenges ),
				number_format_i18n( $ratelimits )
			);
			echo '<p class="aegiswaf-proof"><span class="dashicons dashicons-shield"></span> ' . esc_html( $proof ) . '</p>';

			echo '</div>'; // end card
			}
		}

		if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'table_name' ) ) {

			global $wpdb;
			$table = AegisWAF_Logger::table_name();

			$blocked_pp = isset( $_GET['blocked_pp'] ) ? absint( wp_unslash( $_GET['blocked_pp'] ) ) : 25; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( ! in_array( $blocked_pp, [ 25, 50 ], true ) ) { $blocked_pp = 25; }

			$blocked_page = isset( $_GET['blocked_page'] ) ? max( 1, absint( wp_unslash( $_GET['blocked_page'] ) ) ) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$blocked_offset = ( $blocked_page - 1 ) * $blocked_pp;

			$like_block = '%block%';
			$like_deny  = '%deny%';
			$total_blocked = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->prepare(
					"SELECT COUNT(*) FROM %i
					 WHERE (action_taken LIKE %s OR action_taken LIKE %s)",
					$table,
					$like_block,
					$like_deny
				)
			);
$blocked_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->prepare(
					"SELECT * FROM %i
					 WHERE (action_taken LIKE %s OR action_taken LIKE %s)
					 ORDER BY event_time DESC
					 LIMIT %d OFFSET %d",
					$table,
					$like_block,
					$like_deny,
					$blocked_pp,
					$blocked_offset
				),
				ARRAY_A
			);

			$blocked_total_pages = max( 1, (int) ceil( $total_blocked / $blocked_pp ) );

			echo '<div class="aegiswaf-card" style="margin-top:16px;">';
			echo '<h2>Blocked <span class="description">(recent)</span></h2>';
			echo '<p class="description">A focused list of blocked sources pulled from your logs. Use this to Unblock (allow override), Delete log rows, or Escalate for investigation.</p>';

			if ( isset( $_GET['blocked_msg'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$msg = sanitize_text_field( (string) wp_unslash( $_GET['blocked_msg'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				if ( $msg === 'unblocked' ) {
					echo '<div class="notice notice-success inline"><p>Unblock applied (allow override added).</p></div>';
				} elseif ( $msg === 'deleted' ) {
					echo '<div class="notice notice-success inline"><p>Blocked log row deleted.</p></div>';
				} elseif ( $msg === 'escalated' ) {
					echo '<div class="notice notice-success inline"><p>Escalated (admin_action log created).</p></div>';
				}
			}

			echo '<form method="get" style="margin:10px 0 12px; display:flex; gap:10px; align-items:center;">';
			echo '<input type="hidden" name="page" value="aegiswaf">';
			echo '<input type="hidden" name="tab" value="logs">';
			echo '<label for="blocked_pp"><strong>Rows:</strong></label> ';
			echo '<select name="blocked_pp" id="blocked_pp">';
			foreach ( [ 25, 50 ] as $n ) {
				echo '<option value="' . esc_attr( (string) $n ) . '" ' . selected( $blocked_pp, $n, false ) . '>' . esc_html( (string) $n ) . '</option>';
			}
			echo '</select>';
			echo '<input type="hidden" name="blocked_page" value="1">';
			echo '<button class="button">Apply</button>';
			echo '</form>';

			if ( empty( $blocked_rows ) ) {
				echo '<p>No blocked events found.</p>';
				echo '</div>'; // card
			} else {

				$extract_domain = function( $details ) : string {
					if ( is_string( $details ) ) {
						$decoded = json_decode( $details, true );
						if ( is_array( $decoded ) ) { $details = $decoded; }
					}
					if ( is_array( $details ) ) {
						foreach ( [ 'domain', 'host', 'hostname', 'remote_domain', 'remote_host' ] as $k ) {
							if ( ! empty( $details[ $k ] ) && is_string( $details[ $k ] ) ) {
								return (string) $details[ $k ];
							}
						}
						if ( ! empty( $details['url'] ) && is_string( $details['url'] ) ) {
							$h = wp_parse_url( $details['url'], PHP_URL_HOST );
							if ( is_string( $h ) && $h !== '' ) { return $h; }
						}
					}
					return '';
				};

				$parse_why = function( $details, string $category, string $action_taken ) : string {
					$rule = '';
					if ( is_string( $details ) ) {
						$decoded = json_decode( $details, true );
						if ( is_array( $decoded ) ) { $details = $decoded; }
					}
					if ( is_array( $details ) ) {
						foreach ( [ 'rule_id', 'id', 'rule', 'signature', 'sig', 'tag', 'name', 'code', 'reason' ] as $k ) {
							if ( ! empty( $details[ $k ] ) && is_string( $details[ $k ] ) ) {
								$rule = (string) $details[ $k ];
								break;
							}
						}
					}
					$out = strtoupper( $action_taken ) . ' • ' . $category;
					if ( $rule !== '' ) { $out .= ' • ' . $rule; }
					return $out;
				};

				echo '<div style="overflow:auto;">';
				echo '<table class="widefat striped">';
				echo '<thead><tr>';
				echo '<th style="min-width:180px;">Source IP / Domain</th>';
				echo '<th style="min-width:220px;">Purpose (Endpoint)</th>';
				echo '<th style="width:110px;">Date</th>';
				echo '<th style="width:90px;">Time (UTC)</th>';
				echo '<th style="min-width:220px;">Why it was blocked</th>';
				echo '<th style="min-width:260px;">Notes</th>';
				echo '<th style="width:260px;">Actions</th>';
				echo '</tr></thead><tbody>';

				foreach ( $blocked_rows as $r ) {

					$ip = (string) ( $r['ip'] ?? '' );
					$route = (string) ( $r['route'] ?? '' );
					$method = (string) ( $r['method'] ?? '' );
					$category = (string) ( $r['category'] ?? '' );
					$action_taken = (string) ( $r['action_taken'] ?? '' );
					$details = $r['details'] ?? '';

					$domain = $extract_domain( $details );
					$source = $domain !== '' ? ($domain . ' / ' . $ip) : $ip;

					$when = (string) ( $r['event_time'] ?? '' );
					$dt = $when !== '' ? strtotime( $when . ' UTC' ) : 0;
					$date = $dt ? gmdate( 'Y-m-d', $dt ) : '';
					$time = $dt ? gmdate( 'H:i:s', $dt ) : '';

					$why = $parse_why( $details, $category, $action_taken );

					$notes = '';
					if ( is_string( $details ) ) {
						$notes = $details;
					} else {
						$notes = wp_json_encode( $details );
					}
					$notes = (string) $notes;
					if ( strlen( $notes ) > 220 ) {
						$notes = substr( $notes, 0, 220 ) . '…';
					}

					echo '<tr>';
					echo '<td><code>' . esc_html( $source ) . '</code></td>';
					echo '<td><code>' . esc_html( $route ) . '</code> <span class="description">(' . esc_html( strtoupper( $method ) ) . ')</span></td>';
					echo '<td>' . esc_html( $date ) . '</td>';
					echo '<td>' . esc_html( $time ) . '</td>';
					echo '<td>' . esc_html( $why ) . '</td>';
					echo '<td><small>' . esc_html( $notes ) . '</small></td>';
					echo '<td>';
					echo '<div style="display:flex; gap:8px; flex-wrap:wrap;">';
					echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '">';
					wp_nonce_field( 'aegiswaf_blocked_action' );
					echo '<input type="hidden" name="action" value="aegiswaf_blocked_action">';
					echo '<input type="hidden" name="do" value="unblock">';
					echo '<input type="hidden" name="log_id" value="' . esc_attr( (string) (int) $r['id'] ) . '">';
					echo '<input type="hidden" name="ip" value="' . esc_attr( $ip ) . '">';
					echo '<input type="hidden" name="route" value="' . esc_attr( $route ) . '">';
					echo '<input type="hidden" name="method" value="' . esc_attr( $method ) . '">';
					echo '<button class="button">Unblock</button>';
					echo '</form>';

					echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '" onsubmit="return confirm(\'Delete this blocked log row?\');">';
					wp_nonce_field( 'aegiswaf_blocked_action' );
					echo '<input type="hidden" name="action" value="aegiswaf_blocked_action">';
					echo '<input type="hidden" name="do" value="delete">';
					echo '<input type="hidden" name="log_id" value="' . esc_attr( (string) (int) $r['id'] ) . '">';
					echo '<button class="button">Delete</button>';
					echo '</form>';

					echo '<form method="post" action="' . esc_url( admin_url( 'admin-post.php' ) ) . '">';
					wp_nonce_field( 'aegiswaf_blocked_action' );
					echo '<input type="hidden" name="action" value="aegiswaf_blocked_action">';
					echo '<input type="hidden" name="do" value="escalate">';
					echo '<input type="hidden" name="log_id" value="' . esc_attr( (string) (int) $r['id'] ) . '">';
					echo '<input type="hidden" name="ip" value="' . esc_attr( $ip ) . '">';
					echo '<input type="hidden" name="route" value="' . esc_attr( $route ) . '">';
					echo '<input type="hidden" name="method" value="' . esc_attr( $method ) . '">';
					echo '<button class="button">Escalate</button>';
					echo '</form>';

					echo '</div>';

					echo '</td>';
					echo '</tr>';
				}

				echo '</tbody></table>';
				echo '</div>'; // overflow

				if ( $blocked_total_pages > 1 ) {
					echo '<div style="margin-top:12px; display:flex; gap:8px; align-items:center; flex-wrap:wrap;">';
					for ( $p = 1; $p <= $blocked_total_pages; $p++ ) {
						$url = add_query_arg( [
							'page'         => 'aegiswaf',
							'tab'          => 'logs',
							'blocked_pp'   => $blocked_pp,
							'blocked_page' => $p,
						], admin_url( 'admin.php' ) );

						if ( $p === $blocked_page ) {
							echo '<span class="button button-primary" style="pointer-events:none;">' . esc_html( (string) $p ) . '</span>';
						} else {
							echo '<a class="button" href="' . esc_url( $url ) . '">' . esc_html( (string) $p ) . '</a>';
						}
					}
					echo '</div>';
				}

				echo '</div>'; 
			}

		}

        echo '<style>
            .badge{display:inline-block;padding:2px 6px;border-radius:10px;font-size:11px;line-height:1;background:#eee;margin-right:6px;}
            .badge-critical{background:#d63638;color:#fff;}
            .badge-high{background:#b32d2e;color:#fff;}
            .badge-medium{background:#dba617;color:#111;}
            .badge-info{background:#2271b1;color:#fff;}
			
			/* Attack Story Visual Insights */
			.aegiswaf-story-cards{
			  display:grid;grid-template-columns:repeat(6,minmax(0,1fr));gap:10px;margin:12px 0 14px;
			}
			.aegiswaf-story-card{
			  background:#fff;border:1px solid #e5e7eb;border-radius:10px;padding:10px;
			}
			.aegiswaf-story-card .k{font-size:12px;color:#6b7280;margin-bottom:4px;}
			.aegiswaf-story-card .v{font-size:22px;font-weight:700;color:#111827;line-height:1.1;}
			.aegiswaf-story-card .v.aegiswaf-v-bad{color:#b91c1c;}
			.aegiswaf-story-card .s{font-size:11px;color:#6b7280;margin-top:4px;}
			.aegiswaf-story-grid{
			  display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:12px;margin-top:10px;
			}
			.aegiswaf-story-panel{
			  background:#fff;border:1px solid #e5e7eb;border-radius:10px;padding:12px;
			}
			.aegiswaf-story-panel h3{margin:0 0 10px;font-size:13px;}
			.aegiswaf-barrow{display:grid;grid-template-columns:1fr 160px 50px;gap:10px;align-items:center;margin:6px 0;}
			.aegiswaf-barrow .lbl{font-size:12px;color:#111827;overflow:hidden;text-overflow:ellipsis;white-space:nowrap;}
			.aegiswaf-barrow .barwrap{height:10px;background:#eef2f7;border-radius:999px;overflow:hidden;}
			.aegiswaf-barrow .barwrap .fill{height:10px;background:#2271b1;border-radius:999px;}
			.aegiswaf-barrow .val{font-size:12px;color:#111827;text-align:right;}
			.aegiswaf-spark{display:flex;gap:2px;align-items:flex-end;height:46px;margin:10px 0 6px;}
			.aegiswaf-spark .bar{width:6px;background:#2271b1;border-radius:3px;}
			.aegiswaf-proof{margin-top:12px;background:#f6f7f7;border:1px solid #e5e7eb;padding:10px;border-radius:10px;}
			.aegiswaf-proof .dashicons{vertical-align:middle;margin-right:6px;}
			.aegiswaf-pro-dim{
			  opacity:0.45;
			  pointer-events:none;
			  filter:grayscale(100%);
			}
			.aegiswaf-pro-placeholder{
			  height:220px;
			  display:flex;
			  align-items:center;
			  justify-content:center;
			  border:1px dashed #ccd0d4;
			  border-radius:10px;
			  background:#f6f7f7;
			  color:#646970;
			  font-weight:700;
			}
        </style>';

        echo '</div>'; // card
        echo '</div>'; // wrap
    }
}
