<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

add_action( 'admin_enqueue_scripts', function() {
    if ( ! is_admin() ) { return; }
    $page = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if ( $page !== 'aegiswaf' ) { return; }

    $base_url = plugin_dir_url( dirname( __FILE__, 3 ) ); // plugin root
    wp_enqueue_style(
        'aegiswaf-admin',
        $base_url . 'assets/admin.css',
        [],
        '1.6.4'
    );
	wp_enqueue_script(
		'aegiswaf-chartjs',
		$base_url . 'assets/js/chart.umd.min.js',
		[],
		(string) ( defined('AEGISWAF_VERSION') ? AEGISWAF_VERSION : time() ),
		true
	);
}, 20 );

class AegisWAF_Page_Overview {

    public function render() : void {
        if ( ! current_user_can( 'manage_options' ) ) { return; }

        $is_pro = AegisWAF_Features::is_pro();
        $s      = AegisWAF_Storage::get_settings();
        $status = 'inactive';
        try {
            $lm = new AegisWAF_License_Manager();
            $lic = $lm->get_license_data();
            $status = (string) ( $lic['status'] ?? 'inactive' );
        } catch ( Throwable $e ) {
            $status = 'error';
            if ( class_exists( 'AegisWAF_Logger' ) && method_exists( 'AegisWAF_Logger', 'log' ) ) {
                AegisWAF_Logger::log( 'admin', 'GET', 'error', 'overview_license_read_failed', [
                    'msg' => $e->getMessage(),
                ] );
            } else {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                    error_log( '[AegisWAF] overview_license_read_failed: ' . $e->getMessage() );
                }
            }
        }

        $ddos = [];
        if ( class_exists( 'AegisWAF_DDoS_Storage' ) && method_exists( 'AegisWAF_DDoS_Storage', 'get' ) ) {
            $ddos = AegisWAF_DDoS_Storage::get();
        }

        $intel = $this->get_dashboard_intel();

        $logs_url   = esc_url( add_query_arg( [ 'page' => 'aegiswaf', 'tab' => 'logs' ], admin_url( 'admin.php' ) ) );
        $alerts_url = esc_url( add_query_arg( [ 'page' => 'aegiswaf', 'tab' => 'logs', 'alerts_only' => 1 ], admin_url( 'admin.php' ) ) );

		echo '<div class="aegiswaf-dashboard" style="width:100%;max-width:100%;box-sizing:border-box;">';
		
		echo '<div class="wrap">';
        echo '<h1>Overview Dashboard</h1>';
        AegisWAF_Pro_Notice::render();
        echo '<p class="description">The Overview Dashboard provides a real-time snapshot of your site’s security posture, summarizing WAF activity, active protections, and recent traffic behavior in one place. It displays high-level metrics such as blocked versus allowed requests, module status indicators, and recent threat activity, allowing administrators to quickly assess whether protections are functioning as expected. In Pro, the dashboard expands with enhanced threat metrics, contextual insights, and visual indicators that help identify attack patterns and emerging risks at a glance.</p>';

        echo '<div class="aegiswaf-kv aegiswaf-top-strip" style="display:grid;grid-template-columns:repeat(6,minmax(0,1fr));gap:12px;width:100%;max-width:100%;box-sizing:border-box;">';
        echo '<div class="item"><strong>Mode</strong><br>' . ( $is_pro ? '<span class="aegiswaf-badge">PRO</span>' : '<span class="aegiswaf-badge">FREE</span>' ) . '</div>';
        echo '<div class="item"><strong>License</strong><br><span class="' . ( $is_pro ? 'aegiswaf-status-active' : 'aegiswaf-status-inactive' ) . '">' . esc_html( strtoupper( $status ) ) . '</span></div>';
        echo '<div class="item"><strong>Alerts (24h)</strong><br>' . esc_html( (string) ( $intel['alerts_24h'] ?? 0 ) ) . '</div>';
        echo '<div class="item"><strong>Blocks (24h)</strong><br>' . esc_html( (string) ( $intel['blocks_24h'] ?? 0 ) ) . '</div>';
        echo '<div class="item"><strong>DDoS Actions (24h)</strong><br>' . esc_html( (string) ( $intel['ddos_actions_24h'] ?? 0 ) ) . '</div>';
        echo '<div class="item"><strong>Last Event (UTC)</strong><br>' . esc_html( (string) ( $intel['last_event_utc'] ?? '—' ) ) . '</div>';
        echo '</div>';

		$risk = $this->risk_score( $intel );
		echo '<div style="display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:12px;width:100%;max-width:100%;box-sizing:border-box;align-items:stretch;">';
		echo '<div class="aegiswaf-card" style="min-width:0;box-sizing:border-box;height:100%;display:flex;flex-direction:column;">';
		echo '<div class="aegiswaf-hero-left">';
		echo '<div class="aegiswaf-cardhead"><span class="dashicons dashicons-dashboard"></span><h2>Security Overview</h2></div>';
		echo '<p class="description">Posture + 24h performance. Green = stable, Red = attention needed.</p>';

		echo '<div class="aegiswaf-hero-right">';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- SVG markup generated internally
		echo '<div class="aegiswaf-gaugewrap">' . $this->gauge_svg( $risk ) . '</div>';
		echo '</div>';

		echo '<div class="aegiswaf-hero-kpis">';
		echo '<div class="kpi"><div class="label">Alerts (24h)</div><div class="val">' . intval( $intel['alerts_24h'] ?? 0 ) . '</div></div>';
		echo '<div class="kpi"><div class="label">Blocks</div><div class="val">' . intval( $intel['blocks_24h'] ?? 0 ) . '</div></div>';
		echo '<div class="kpi"><div class="label">Rate Limits</div><div class="val">' . intval( $intel['rate_limits_24h'] ?? 0 ) . '</div></div>';
		echo '<div class="kpi"><div class="label">Errors</div><div class="val">' . intval( $intel['errors_24h'] ?? 0 ) . '</div></div>';
		echo '</div>';

		echo '</div>'; 

		echo '<div class="aegiswaf-sparks">';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- SVG markup generated internally
		echo '<div class="spark"><div class="label">Blocks</div>' . $this->sparkline_svg( (array) ( $intel['hourly_blocks'] ?? [] ) ) . '</div>';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- SVG markup generated internally
		echo '<div class="spark"><div class="label">Challenges</div>' . $this->sparkline_svg( (array) ( $intel['hourly_challenges'] ?? [] ) ) . '</div>';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- SVG markup generated internally
		echo '<div class="spark"><div class="label">Rate Limits</div>' . $this->sparkline_svg( (array) ( $intel['hourly_rate_limits'] ?? [] ) ) . '</div>';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- SVG markup generated internally
		echo '<div class="spark"><div class="label">Errors</div>' . $this->sparkline_svg( (array) ( $intel['hourly_errors'] ?? [] ) ) . '</div>';
		echo '</div>';

		echo '</div>'; 

		echo '<div style="min-width:0;display:flex;flex-direction:column;gap:12px;width:100%;max-width:100%;box-sizing:border-box;">';
		echo '<div class="aegiswaf-card" style="min-width:0;box-sizing:border-box;height:100%;display:flex;flex-direction:column;">';
		echo '<div class="aegiswaf-cardhead"><span class="dashicons dashicons-shield"></span><h2>Security Health</h2></div>';
		echo '<p class="description">At-a-glance threat activity derived from the AegisWAF log stream.</p>';

		echo '<div class="aegiswaf-metrics">';
		echo wp_kses_post( $this->metric_tile( 'Critical Errors (24h)', (int) ( $intel['errors_24h'] ?? 0 ), 'critical'  ) );
		echo wp_kses_post( $this->metric_tile( 'Blocks (24h)', (int) ( $intel['blocks_24h'] ?? 0 ), 'high'  ) );
		echo wp_kses_post( $this->metric_tile( 'Challenges (24h)', (int) ( $intel['challenges_24h'] ?? 0 ), 'medium'  ) );
		echo wp_kses_post( $this->metric_tile( 'Rate Limits (24h)', (int) ( $intel['rate_limits_24h'] ?? 0 ), 'medium'  ) );
		echo '</div>';

		if ( ! empty( $intel['top_categories_24h'] ) && is_array( $intel['top_categories_24h'] ) ) {
			echo '<h4 style="margin:14px 0 8px;">Top Attack Categories (24h)</h4>';
			echo '<div class="aegiswaf-mini-table">';
			foreach ( $intel['top_categories_24h'] as $row ) {
				$cat = (string) ( $row['category'] ?? '' );
				$cnt = (int) ( $row['cnt'] ?? 0 );
				if ( $cat === '' ) { continue; }
				echo '<div class="row"><code>' . esc_html( $cat ) . '</code><span>' . esc_html( (string) $cnt ) . '</span></div>';
			}
			echo '</div>';
		} else {
			echo '<p class="description" style="margin-top:12px;">No category intel yet. Generate traffic (front + REST) and review <a href="' . esc_url( $logs_url ) . '">Logs</a>.</p>';
		}

		echo '<p style="margin:14px 0 0;"><a class="button button-primary" href="' . esc_url( $alerts_url ) . '">View Alerts</a> <a class="button" href="' . esc_url( $logs_url ) . '">View All Logs</a></p>';
		echo '</div>'; // end Security Health card

		echo '</div>'; // end right column

		echo '</div>'; // end 50/50 row

        echo '<div class="aegiswaf-card aegiswaf-span-6" style="grid-column:span 6;margin-top:12px;min-width:0;">';
		$viz = $this->get_overview_visuals( 24 );

		$chart_js_url = plugins_url( 'assets/js/chart.umd.min.js', dirname( __FILE__, 3 ) . '/aegiswaf.php' );

		echo '<style>
		.aegiswaf-viz8-grid{display:grid;grid-template-columns:repeat(4,minmax(220px,1fr));gap:12px;margin:14px 0 14px 0;}
		.aegiswaf-viz-card{background:#fff;border:1px solid #e5e5e5;border-radius:10px;padding:12px;min-width:0;box-sizing:border-box;}
		.aegiswaf-viz-title{font-size:13px;font-weight:600;margin:0 0 6px 0;}
		.aegiswaf-viz-sub{font-size:12px;color:#646970;margin:0 0 10px 0;}
		.aegiswaf-viz-wrap{position:relative;height:210px;width:100%;}
		@media (max-width:1200px){.aegiswaf-viz8-grid{grid-template-columns:repeat(2,minmax(220px,1fr));}}
		@media (max-width:640px){.aegiswaf-viz8-grid{grid-template-columns:1fr;}}
		</style>';

		echo '<div class="aegiswaf-viz8-grid">';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Threat Volume by Module</p>';
		echo '<p class="aegiswaf-viz-sub">Stacked area (hourly, last 24h)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_ov_area_modules"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Attack Categories</p>';
		echo '<p class="aegiswaf-viz-sub">Distribution across WAF/API/Bot/DDoS/Story</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_ov_polar_categories"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Module Pressure Score</p>';
		echo '<p class="aegiswaf-viz-sub">Weighted impact (block>rate>challenge)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_ov_radar_pressure"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Top Targeted Endpoints</p>';
		echo '<p class="aegiswaf-viz-sub">Most-hit routes/paths (last 24h)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_ov_bar_endpoints"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">HTTP Method Mix</p>';
		echo '<p class="aegiswaf-viz-sub">GET vs POST vs PUT/PATCH/DELETE</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_ov_pie_methods"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Enforcement Trend</p>';
		echo '<p class="aegiswaf-viz-sub">Blocks / Rate-limits / Challenges (hourly)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_ov_line_actions"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">IP Risk Scatter</p>';
		echo '<p class="aegiswaf-viz-sub">Each dot = an IP (total events vs block%)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_ov_scatter_iprisk"></canvas></div>';
		echo '</div>';

		echo '<div class="aegiswaf-viz-card">';
		echo '<p class="aegiswaf-viz-title">Top Offenders</p>';
		echo '<p class="aegiswaf-viz-sub">Bubble = total events (blocks vs rate-limits)</p>';
		echo '<div class="aegiswaf-viz-wrap"><canvas id="aegiswaf_ov_bubble_offenders"></canvas></div>';
		echo '</div>';

		echo '</div>'; // grid


		echo '<script>
		window.AEGISWAF_OV_VIZ = ' . wp_json_encode( $viz ) . ';
		(function(){
		  function init(){
			if(!window.Chart || !window.AEGISWAF_OV_VIZ){ return; }
			const v = window.AEGISWAF_OV_VIZ;

			// 1) Stacked Area: module volume
			const c1 = document.getElementById("aegiswaf_ov_area_modules");
			if(c1){
			  new Chart(c1,{
				type:"line",
				data:{ labels:v.area.labels, datasets:v.area.datasets },
				options:{
				  responsive:true, maintainAspectRatio:false,
				  plugins:{ legend:{ position:"bottom" } },
				  elements:{ point:{ radius:0 } },
				  scales:{ x:{ stacked:true }, y:{ stacked:true, beginAtZero:true } }
				}
			  });
			}

			// 2) PolarArea: categories
			const c2 = document.getElementById("aegiswaf_ov_polar_categories");
			if(c2){
			  new Chart(c2,{
				type:"polarArea",
				data:{ labels:v.categories.labels, datasets:[{ data:v.categories.values }] },
				options:{ responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:"bottom" } } }
			  });
			}

			// 3) Radar: pressure score
			const c3 = document.getElementById("aegiswaf_ov_radar_pressure");
			if(c3){
			  new Chart(c3,{
				type:"radar",
				data:{ labels:v.pressure.labels, datasets:[{ label:"Pressure", data:v.pressure.values }] },
				options:{ responsive:true, maintainAspectRatio:false, plugins:{ legend:{ display:false } } }
			  });
			}

			// 4) Bar (horizontal): endpoints
			const c4 = document.getElementById("aegiswaf_ov_bar_endpoints");
			if(c4){
			  new Chart(c4,{
				type:"bar",
				data:{ labels:v.endpoints.labels, datasets:[{ label:"Hits", data:v.endpoints.values }] },
				options:{
				  responsive:true, maintainAspectRatio:false,
				  indexAxis:"y",
				  plugins:{ legend:{ display:false } },
				  scales:{ x:{ beginAtZero:true } }
				}
			  });
			}

			// 5) Pie: methods
			const c5 = document.getElementById("aegiswaf_ov_pie_methods");
			if(c5){
			  new Chart(c5,{
				type:"pie",
				data:{ labels:v.methods.labels, datasets:[{ data:v.methods.values }] },
				options:{ responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:"bottom" } } }
			  });
			}

			// 6) Line: actions trend
			const c6 = document.getElementById("aegiswaf_ov_line_actions");
			if(c6){
			  new Chart(c6,{
				type:"line",
				data:{ labels:v.actions_line.labels, datasets:v.actions_line.datasets },
				options:{
				  responsive:true, maintainAspectRatio:false,
				  plugins:{ legend:{ position:"bottom" } },
				  scales:{ y:{ beginAtZero:true } }
				}
			  });
			}

			// 7) Scatter: ip risk
			const c7 = document.getElementById("aegiswaf_ov_scatter_iprisk");
			if(c7){
			  new Chart(c7,{
				type:"scatter",
				data:{ datasets:[{ label:"IPs", data:v.ip_scatter.points }] },
				options:{
				  responsive:true, maintainAspectRatio:false,
				  parsing:false,
				  plugins:{ legend:{ display:false } },
				  scales:{
					x:{ beginAtZero:true, title:{ display:true, text:"Total Events" } },
					y:{ beginAtZero:true, max:100, title:{ display:true, text:"Block %"} }
				  }
				}
			  });
			}

			// 8) Bubble: offenders
			const c8 = document.getElementById("aegiswaf_ov_bubble_offenders");
			if(c8){
			  new Chart(c8,{
				type:"bubble",
				data:{ datasets:[{ label:"IPs", data:v.offenders.points }] },
				options:{
				  responsive:true, maintainAspectRatio:false,
				  parsing:false,
				  plugins:{
					legend:{ display:false },
					tooltip:{ callbacks:{ label:function(ctx){
					  const p = ctx.raw||{};
					  return (p.label||"IP") + " — blocks:" + (p.x||0) + ", rate:" + (p.y||0) + ", total:" + (p.r||0);
					}}}
				  },
				  scales:{
					x:{ beginAtZero:true, title:{ display:true, text:"Blocks" } },
					y:{ beginAtZero:true, title:{ display:true, text:"Rate-limits" } }
				  }
				}
			  });
			}
		  }
		  if(document.readyState==="loading"){ document.addEventListener("DOMContentLoaded", init); } else { init(); }
		})();
		</script>';

        echo '<div class="aegiswaf-cardhead"><span class="dashicons dashicons-admin-generic"></span><h2>Service Status</h2></div>';
        echo '<p class="description">Configured modules + basic health checks (missing class = not enforcing).</p>';

		echo '<div class="aegis-posture">';
		echo '<div><strong>WAF</strong>' . wp_kses_post( $this->severity_light( !empty($s['rules']['enabled']), class_exists('AegisWAF_Engine') ) ) . '</div>';
		echo '<div><strong>Bot Control</strong>' . wp_kses_post( $this->severity_light( !empty($s['bot']['enabled']), class_exists('AegisWAF_Page_Bot_Control') ) ) . '</div>';
		echo '<div><strong>API Shield</strong>' . wp_kses_post( $this->severity_light( !empty($s['wp_protect']['enabled']), class_exists('AegisWAF_Endpoint_Policy') ) ) . '</div>';
		echo '<div><strong>DDoS</strong>' . wp_kses_post( $this->severity_light( !empty($ddos['enabled']), class_exists('AegisWAF_DDoS_Engine') ) ) . '</div>';
		echo '</div>';

        echo '<div class="aegiswaf-statuslist">';
        echo wp_kses_post( $this->status_row( 'Managed Rules', ! empty( $s['managed_rules']['enabled'] ), class_exists( 'AegisWAF_Engine' ) ) );
        echo wp_kses_post( $this->status_row( 'WAF Rules', ! empty( $s['rules']['enabled'] ), class_exists( 'AegisWAF_Page_WAF_Rules' ) ) );
        echo wp_kses_post( $this->status_row( 'API Shield', ! empty( $s['wp_protect']['enabled'] ), class_exists( 'AegisWAF_Endpoint_Policy' ) ) );
        echo wp_kses_post( $this->status_row( 'Bot Control', ! empty( $s['bot']['enabled'] ), class_exists( 'AegisWAF_Page_Bot_Control' ) ) );
        echo wp_kses_post( $this->status_row( 'DDoS Shield', ! empty( $ddos['enabled'] ), class_exists( 'AegisWAF_DDoS_Engine' ) ) );
        echo wp_kses_post( $this->status_row( 'Logging', true, $intel['log_table_ok'] === true ) );
        echo '</div>';

        // Quick config hints
        $hints = [];
        if ( empty( $s['managed_rules']['enabled'] ) ) { $hints[] = 'Managed Rules are OFF (reduces detection coverage).'; }
        if ( empty( $s['wp_protect']['enabled'] ) ) { $hints[] = 'API Shield is OFF (REST endpoints are not inspected).'; }
        if ( ! empty( $ddos['enabled'] ) && empty( $ddos['ignore_logged_in'] ) ) { $hints[] = 'DDoS applies to logged-in users (can be noisy on admin workflows).'; }
        if ( $intel['log_table_ok'] !== true ) { $hints[] = 'Log table check failed (verify DB permissions / wp-config table prefix).'; }

        if ( ! empty( $hints ) ) {
            echo '<div class="notice notice-warning" style="margin:14px 0 0;"><p><strong>Attention:</strong><br>' . esc_html( implode( ' ', $hints ) ) . '</p></div>';
        } else {
            echo '<div class="notice notice-success" style="margin:14px 0 0;"><p><strong>All core services look healthy.</strong></p></div>';
        }

        echo '</div>';
		echo '<div class="aegiswaf-card" style="width:100%;max-width:100%;box-sizing:border-box;margin-top:12px;">';
		echo '<div class="aegiswaf-cardhead"><span class="dashicons dashicons-warning"></span><h2>Recent Alerts</h2></div>';
		echo '<p class="description">Most recent high-signal events (block / challenge / rate-limit / errors).</p>';
        if ( ! $is_pro ) {
            AegisWAF_Utils::pro_gate_box(
                'Unlock PRO Intelligence & Automation',
                'PRO unlocks virtual patching shields, retention + export, and advanced bot intelligence.'
            );
        }
		if ( empty( $intel['recent_alerts'] ) ) {
			echo '<p>No alerts yet. When a request is blocked / challenged / rate-limited, it will appear here.</p>';
		} else {
			echo '<div class="aegiswaf-alertlist">';
			foreach ( $intel['recent_alerts'] as $r ) {
				$level = (string) ( $r['level'] ?? 'info' );
				$time  = (string) ( $r['event_time'] ?? '' );
				$cat   = (string) ( $r['category'] ?? '' );
				$act   = (string) ( $r['action_taken'] ?? '' );
				$ip    = (string) ( $r['ip'] ?? '' );
				$route = (string) ( $r['route'] ?? '' );

				echo '<div class="aegiswaf-alert">';
				echo '<span class="aegiswaf-pill aegiswaf-pill-' . esc_attr( $level ) . '">' . esc_html( strtoupper( $level ) ) . '</span> ';
				echo '<span class="aegiswaf-alert-meta"><code>' . esc_html( $cat ) . '</code> <code>' . esc_html( $act ) . '</code> <code>' . esc_html( $ip ) . '</code></span>';
				echo '<div class="aegiswaf-alert-line"><span class="aegiswaf-alert-time">' . esc_html( $time ) . ' UTC</span> <code>' . esc_html( $route ) . '</code></div>';
				echo '</div>';
			}
			echo '</div>';
		}

		echo '<p style="margin:14px 0 0;"><a class="button" href="' . esc_url( $alerts_url ) . '">Open Alerts in Logs</a></p>';
		echo '</div>';

		echo '<div class="aegiswaf-card" style="min-width:0;margin-top:12px;box-sizing:border-box;">';
		echo '<div class="aegiswaf-cardhead"><span class="dashicons dashicons-networking"></span><h2>Top Sources (24h)</h2></div>';
		echo '<p class="description">Most active IPs from alert traffic.</p>';
        if ( ! $is_pro ) {
            AegisWAF_Utils::pro_gate_box(
                'Unlock PRO Intelligence & Automation',
                'PRO unlocks virtual patching shields, retention + export, and advanced bot intelligence.'
            );
        }
		if ( ! empty( $intel['top_ips_24h'] ) && is_array( $intel['top_ips_24h'] ) ) {
			echo '<div class="aegiswaf-mini-table">';
			foreach ( $intel['top_ips_24h'] as $row ) {
				$ip  = (string) ( $row['ip'] ?? '' );
				$cnt = (int) ( $row['cnt'] ?? 0 );
				if ( $ip === '' ) { continue; }
				echo '<div class="row"><code>' . esc_html( $ip ) . '</code><span>' . esc_html( (string) $cnt ) . '</span></div>';
			}
			echo '</div>';
		} else {
			echo '<p class="description">No source data yet.</p>';
		}

		echo '</div>'; // end Top Sources card

        echo '</div>'; // dashboard grid
        echo '</div>'; // dashboard wrap

    }

    private function get_dashboard_intel() : array {
        global $wpdb;

        $out = [
            'log_table_ok' => false,
            'last_event_utc' => '—',
            'alerts_24h' => 0,
            'blocks_24h' => 0,
            'challenges_24h' => 0,
            'rate_limits_24h' => 0,
            'errors_24h' => 0,
            'ddos_actions_24h' => 0,
            'top_categories_24h' => [],
            'top_ips_24h' => [],
            'recent_alerts' => [],
			'hourly_blocks' => [],
			'hourly_challenges' => [],
			'hourly_rate_limits' => [],
			'hourly_errors' => [],
        ];

        if ( ! class_exists( 'AegisWAF_Logger' ) ) { return $out; }

        if ( method_exists( 'AegisWAF_Logger', 'ensure_table' ) ) {
            try { AegisWAF_Logger::ensure_table(); } catch ( Throwable $e ) {}
        }

        $table = AegisWAF_Logger::table_name();

        try {
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $out['log_table_ok'] = ( $exists === $table );
        } catch ( Throwable $e ) {
            $out['log_table_ok'] = false;
        }

        if ( $out['log_table_ok'] !== true ) {
            return $out;
        }

        $since = gmdate( 'Y-m-d H:i:s', time() - DAY_IN_SECONDS );

		$out['hourly_blocks']      = array_fill( 0, 24, 0 );
		$out['hourly_challenges']  = array_fill( 0, 24, 0 );
		$out['hourly_rate_limits'] = array_fill( 0, 24, 0 );
		$out['hourly_errors']      = array_fill( 0, 24, 0 );

		$hour_rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT DATE_FORMAT(event_time,'%%Y-%%m-%%d %%H:00:00') AS hr, action_taken, COUNT(1) AS cnt
				 FROM %i
				 WHERE event_time >= %s
				 AND action_taken IN ('block','challenge','rate_limit','error')
				 GROUP BY hr, action_taken
				 ORDER BY hr ASC",
				array( $table, $since )
			),
			ARRAY_A
		) ?: [];

		$hours = [];
		for ( $i = 23; $i >= 0; $i-- ) {
			$hours[] = gmdate( 'Y-m-d H:00:00', time() - ( $i * HOUR_IN_SECONDS ) );
		}
		$index = array_flip( $hours );

		foreach ( $hour_rows as $r ) {
			$hr = (string) ( $r['hr'] ?? '' );
			if ( $hr === '' || ! isset( $index[ $hr ] ) ) { continue; }

			$i = (int) $index[ $hr ];
			$a = strtolower( (string) ( $r['action_taken'] ?? '' ) );
			$c = (int) ( $r['cnt'] ?? 0 );

			if ( $a === 'block' )      { $out['hourly_blocks'][ $i ] = $c; }
			if ( $a === 'challenge' )  { $out['hourly_challenges'][ $i ] = $c; }
			if ( $a === 'rate_limit' ) { $out['hourly_rate_limits'][ $i ] = $c; }
			if ( $a === 'error' )      { $out['hourly_errors'][ $i ] = $c; }
		}

        $last = $wpdb->get_var( $wpdb->prepare( 'SELECT MAX(event_time) FROM %i', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        if ( is_string( $last ) && $last !== '' ) {
            $out['last_event_utc'] = $last;
        }

        $rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT action_taken, COUNT(1) AS cnt
                 FROM %i
                 WHERE event_time >= %s
                 AND action_taken IN ('block','challenge','rate_limit','error')
                 GROUP BY action_taken",
                array( $table, $since )
            ),
            ARRAY_A
        ) ?: [];

        foreach ( $rows as $r ) {
            $a = strtolower( (string) ( $r['action_taken'] ?? '' ) );
            $c = (int) ( $r['cnt'] ?? 0 );
            if ( $a === 'block' ) { $out['blocks_24h'] = $c; }
            if ( $a === 'challenge' ) { $out['challenges_24h'] = $c; }
            if ( $a === 'rate_limit' ) { $out['rate_limits_24h'] = $c; }
            if ( $a === 'error' ) { $out['errors_24h'] = $c; }
        }

        $out['alerts_24h'] = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT COUNT(1)
                 FROM %i
                 WHERE event_time >= %s
                 AND (action_taken IN ('block','challenge','rate_limit','error')
                      OR category IN ('error','warning','alert','attack','malware'))",
                array( $table, $since )
            )
        );

        $out['ddos_actions_24h'] = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT COUNT(1)
                 FROM %i
                 WHERE event_time >= %s
                 AND category = 'ddos_shield'
                 AND action_taken IN ('block','challenge','rate_limit','error','engine_missing','engine_missing_rest','engine_missing_front')",
                array( $table, $since )
            )
        );

        $out['top_categories_24h'] = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT category, COUNT(1) AS cnt
                 FROM %i
                 WHERE event_time >= %s
                 AND (action_taken IN ('block','challenge','rate_limit','error')
                      OR category IN ('error','warning','alert','attack','malware'))
                 GROUP BY category
                 ORDER BY cnt DESC
                 LIMIT 6",
                array( $table, $since )
            ),
            ARRAY_A
        ) ?: [];

        // Top IPs (24h) for alerts-only traffic
        $out['top_ips_24h'] = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT ip, COUNT(1) AS cnt
                 FROM %i
                 WHERE event_time >= %s
                 AND ip IS NOT NULL AND ip <> ''
                 AND (action_taken IN ('block','challenge','rate_limit','error')
                      OR category IN ('error','warning','alert','attack','malware'))
                 GROUP BY ip
                 ORDER BY cnt DESC
                 LIMIT 6",
                array( $table, $since )
            ),
            ARRAY_A
        ) ?: [];

        $recent = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $wpdb->prepare(
                "SELECT event_time, route, method, category, action_taken, ip
                 FROM %i
                 WHERE (action_taken IN ('block','challenge','rate_limit','error')
                      OR category IN ('error','warning','alert','attack','malware'))
                 ORDER BY id DESC
                 LIMIT %d",
                $table,
                10
            ),
            ARRAY_A
        ) ?: [];

        $out['recent_alerts'] = [];
        foreach ( $recent as $r ) {
            $level = 'info';
            if ( method_exists( 'AegisWAF_Logger', 'alert_level' ) ) {
                try { $level = (string) AegisWAF_Logger::alert_level( $r ); } catch ( Throwable $e ) { $level = 'info'; }
            }
            $r['level'] = $level;
            $out['recent_alerts'][] = $r;
        }

        return $out;

    }
	
		private function severity_light( bool $enabled, bool $healthy ) : string {
		if ( ! $enabled ) {
			return '<span class="aegis-light gray" title="Disabled"></span>';
		}
		if ( $healthy ) {
			return '<span class="aegis-light green" title="Operational"></span>';
		}
		return '<span class="aegis-light red" title="Error"></span>';
	}

	private function percent_bar( int $value, int $max = 100 ) : string {
		$pct = min( 100, max( 0, intval( ( $value / max(1,$max) ) * 100 ) ) );
		return '<div class="aegis-bar"><div class="fill" style="width:' . $pct . '%"></div></div>';
	}


    private function metric_tile( string $label, int $value, string $level ) : string {
        $value = max( 0, $value );
        return '<div class="aegiswaf-metric"><div class="aegiswaf-metric__label">' . esc_html( $label ) . '</div><div class="aegiswaf-metric__value"><span class="aegiswaf-pill aegiswaf-pill-' . esc_attr( $level ) . '">' . esc_html( (string) $value ) . '</span></div></div>';
    }

    private function status_row( string $name, bool $enabled, bool $healthy ) : string {
        $state = $enabled ? 'ON' : 'OFF';
        $pill  = $enabled ? 'info' : 'medium';

        if ( $enabled && ! $healthy ) {
            $pill = 'critical';
            $state = 'ERROR';
        }

        $health_txt = $healthy ? 'OK' : 'Missing/Unavailable';
        if ( ! $enabled ) { $health_txt = 'Disabled'; }

        return '<div class="aegiswaf-statusrow"><div class="left"><strong>' . esc_html( $name ) . '</strong><br><span class="description">' . esc_html( $health_txt ) . '</span></div><div class="right"><span class="aegiswaf-pill aegiswaf-pill-' . esc_attr( $pill ) . '">' . esc_html( $state ) . '</span></div></div>';
    }
	private function clamp_int( int $v, int $min, int $max ) : int {
    return max( $min, min( $max, $v ) );
	}

	private function risk_score( array $intel ) : int {
		$blocks = (int) ( $intel['blocks_24h'] ?? 0 );
		$chall  = (int) ( $intel['challenges_24h'] ?? 0 );
		$rates  = (int) ( $intel['rate_limits_24h'] ?? 0 );
		$errs   = (int) ( $intel['errors_24h'] ?? 0 );

		$raw = ( $errs * 8 ) + ( $blocks * 4 ) + ( $rates * 2 ) + ( $chall * 1 );

		$score = (int) round( min( 100, ( $raw / 50 ) * 100 ) );

		return $this->clamp_int( $score, 0, 100 );
	}

	private function gauge_svg( int $score ) : string {
		$score = $this->clamp_int( $score, 0, 100 );

		$pct = $score / 100;
		$dash = (int) round( 157 * $pct ); // 157 ~= half circumference for r=50
		$color = ( $score >= 70 ) ? '#ef4444' : ( ( $score >= 35 ) ? '#f59e0b' : '#22c55e' );

		return '<svg class="aegiswaf-gauge" viewBox="0 0 120 70" aria-hidden="true">'
			. '<path d="M10,60 A50,50 0 0 1 110,60" fill="none" stroke="#e5e7eb" stroke-width="10" stroke-linecap="round"></path>'
			. '<path d="M10,60 A50,50 0 0 1 110,60" fill="none" stroke="' . esc_attr( $color ) . '" stroke-width="10" stroke-linecap="round"'
			. ' stroke-dasharray="' . esc_attr( (string) $dash ) . ' 157"></path>'
			. '<text x="60" y="48" text-anchor="middle" font-size="18" font-weight="700" fill="#111827">' . esc_html( (string) $score ) . '</text>'
			. '<text x="60" y="64" text-anchor="middle" font-size="10" fill="#6b7280">Risk Score</text>'
			. '</svg>';
	}

	private function sparkline_svg( array $series ) : string {
		$points = [];
		$series = array_values( $series );
		$max = 1;
		foreach ( $series as $v ) { $max = max( $max, (int) $v ); }

		$w = 220; $h = 40;
		$n = max( 1, count( $series ) );
		for ( $i = 0; $i < $n; $i++ ) {
			$x = ( $n === 1 ) ? 0 : ( $i * ( $w / ( $n - 1 ) ) );
			$y = $h - ( ( (int) $series[$i] / $max ) * $h );
			$points[] = round( $x, 2 ) . ',' . round( $y, 2 );
		}

		return '<svg class="aegiswaf-spark" viewBox="0 0 ' . $w . ' ' . $h . '" aria-hidden="true">'
			. '<polyline fill="none" stroke="#2271b1" stroke-width="2" points="' . esc_attr( implode( ' ', $points ) ) . '"></polyline>'
			. '</svg>';
	}

	private function get_overview_visuals( int $hours = 24 ) : array {
		global $wpdb;

		$hours = max( 1, min( 168, $hours ) ); // 1h..7d
		$cache_key = 'aegiswaf_overview_viz_' . $hours;

		$cached = get_transient( $cache_key );
		if ( is_array( $cached ) ) { return $cached; }

		if ( ! class_exists( 'AegisWAF_Logger' ) ) {
			return [
				'area' => [ 'labels' => [], 'datasets' => [] ],
				'categories' => [ 'labels' => [], 'values' => [] ],
				'pressure' => [ 'labels' => [], 'values' => [] ],
				'endpoints' => [ 'labels' => [], 'values' => [] ],
				'methods' => [ 'labels' => [], 'values' => [] ],
				'actions_line' => [ 'labels' => [], 'blocks' => [], 'rate_limits' => [], 'challenges' => [] ],
				'ip_scatter' => [ 'points' => [] ],
				'offenders' => [ 'points' => [] ],
				'meta' => [ 'hours' => $hours, 'rows_used' => 0 ],
			];
		}

		if ( method_exists( 'AegisWAF_Logger', 'ensure_table' ) ) {
			try { AegisWAF_Logger::ensure_table(); } catch ( Throwable $e ) {}
		}

		$table = AegisWAF_Logger::table_name();
		$since = gmdate( 'Y-m-d H:i:s', time() - ( $hours * HOUR_IN_SECONDS ) );

		$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				"SELECT action_taken, category, method, route, ip, event_time
				 FROM %i
				 WHERE event_time >= %s
				 ORDER BY id DESC
				 LIMIT 8000",
				$table,
				$since
			),
			ARRAY_A
		);
		if ( ! is_array( $rows ) ) { $rows = []; }

		$hours_map = [];
		$now = time();
		for ( $i = $hours - 1; $i >= 0; $i-- ) {
			$t = $now - ( $i * HOUR_IN_SECONDS );
			$k = gmdate( 'Y-m-d H:00', $t );
			$hours_map[ $k ] = [
				'blocks' => 0,
				'rate_limits' => 0,
				'challenges' => 0,
				'WAF' => 0,
				'API Shield' => 0,
				'Bot Control' => 0,
				'DDoS' => 0,
				'Attack Story' => 0,
			];
		}

		$cat_counts = [];
		$method_counts = [];
		$route_counts = [];

		$module_score = [
			'WAF' => 0,
			'API Shield' => 0,
			'Bot Control' => 0,
			'DDoS' => 0,
			'Attack Story' => 0,
		];

		$ip_stats = []; // ip => ['total'=>0,'blocks'=>0,'rate_limits'=>0,'challenges'=>0]

		foreach ( $rows as $r ) {
			$act = strtolower( (string) ( $r['action_taken'] ?? '' ) );
			if ( $act === 'ratelimit' ) { $act = 'rate_limit'; }

			$cat = (string) ( $r['category'] ?? '' );
			$method = strtoupper( (string) ( $r['method'] ?? '' ) );
			$route = (string) ( $r['route'] ?? '' );
			$ip = (string) ( $r['ip'] ?? '' );
			$et = (string) ( $r['event_time'] ?? '' );

			$module = 'WAF';
			if ( $cat === 'api_shield' || strpos( $cat, 'api' ) !== false ) { $module = 'API Shield'; }
			elseif ( $cat === 'bot_control' || strpos( $cat, 'bot' ) !== false ) { $module = 'Bot Control'; }
			elseif ( $cat === 'ddos' || strpos( $cat, 'ddos' ) !== false ) { $module = 'DDoS'; }
			elseif ( $cat === 'attack_story' || strpos( $cat, 'story' ) !== false ) { $module = 'Attack Story'; }

			$hk = '';
			if ( $et !== '' && strlen( $et ) >= 13 ) {
				$hk = substr( $et, 0, 13 ) . ':00';
			}

			if ( $cat !== '' ) {
				if ( ! isset( $cat_counts[ $cat ] ) ) { $cat_counts[ $cat ] = 0; }
				$cat_counts[ $cat ]++;
			}

			if ( $method !== '' ) {
				if ( ! isset( $method_counts[ $method ] ) ) { $method_counts[ $method ] = 0; }
				$method_counts[ $method ]++;
			}

			if ( $route !== '' ) {
				if ( ! isset( $route_counts[ $route ] ) ) { $route_counts[ $route ] = 0; }
				$route_counts[ $route ]++;
			}

			$is_block = ( $act === 'block' );
			$is_rate  = ( $act === 'rate_limit' );
			$is_chal  = ( $act === 'challenge' );

			if ( $hk !== '' && isset( $hours_map[ $hk ] ) ) {
				$hours_map[ $hk ][ $module ]++;

				if ( $is_block ) { $hours_map[ $hk ]['blocks']++; }
				if ( $is_rate )  { $hours_map[ $hk ]['rate_limits']++; }
				if ( $is_chal )  { $hours_map[ $hk ]['challenges']++; }
			}

			if ( $is_block ) { $module_score[ $module ] += 3; }
			elseif ( $is_rate ) { $module_score[ $module ] += 2; }
			elseif ( $is_chal ) { $module_score[ $module ] += 1; }

			if ( $ip !== '' ) {
				if ( ! isset( $ip_stats[ $ip ] ) ) {
					$ip_stats[ $ip ] = [ 'total' => 0, 'blocks' => 0, 'rate_limits' => 0, 'challenges' => 0 ];
				}
				$ip_stats[ $ip ]['total']++;
				if ( $is_block ) { $ip_stats[ $ip ]['blocks']++; }
				if ( $is_rate )  { $ip_stats[ $ip ]['rate_limits']++; }
				if ( $is_chal )  { $ip_stats[ $ip ]['challenges']++; }
			}
		}

		arsort( $cat_counts );
		arsort( $route_counts );
		arsort( $method_counts );

		$top_endpoints = array_slice( $route_counts, 0, 8, true );

		$m_top = array_slice( $method_counts, 0, 6, true );
		$m_other = array_sum( $method_counts ) - array_sum( $m_top );
		if ( $m_other > 0 ) { $m_top['OTHER'] = $m_other; }

		$top_cats = array_slice( $cat_counts, 0, 8, true );

		$time_labels = array_keys( $hours_map );
		$line_blocks = [];
		$line_rates  = [];
		$line_chals  = [];

		$mod_waf = [];
		$mod_api = [];
		$mod_bot = [];
		$mod_ddos = [];
		$mod_story = [];

		foreach ( $hours_map as $k => $v ) {
			$line_blocks[] = (int) $v['blocks'];
			$line_rates[]  = (int) $v['rate_limits'];
			$line_chals[]  = (int) $v['challenges'];

			$mod_waf[]   = (int) $v['WAF'];
			$mod_api[]   = (int) $v['API Shield'];
			$mod_bot[]   = (int) $v['Bot Control'];
			$mod_ddos[]  = (int) $v['DDoS'];
			$mod_story[] = (int) $v['Attack Story'];
		}

		uasort( $ip_stats, function( $a, $b ) {
			return (int) ( $b['total'] ?? 0 ) <=> (int) ( $a['total'] ?? 0 );
		} );
		$ip_top = array_slice( $ip_stats, 0, 60, true );

		$scatter = [];
		$bubbles = [];
		$top_offenders = array_slice( $ip_stats, 0, 12, true );

		foreach ( $ip_top as $ip => $st ) {
			$total = (int) ( $st['total'] ?? 0 );
			$blocks = (int) ( $st['blocks'] ?? 0 );
			$rate   = (int) ( $st['rate_limits'] ?? 0 );

			$block_pct = $total > 0 ? round( ( $blocks / $total ) * 100, 1 ) : 0.0;

			$scatter[] = [
				'x' => $total,
				'y' => $block_pct,
				'label' => $ip,
			];
		}

		foreach ( $top_offenders as $ip => $st ) {
			$total = (int) ( $st['total'] ?? 0 );
			$blocks = (int) ( $st['blocks'] ?? 0 );
			$rate   = (int) ( $st['rate_limits'] ?? 0 );

			$bubbles[] = [
				'x' => $blocks,
				'y' => $rate,
				'r' => max( 3, min( 30, $total ) ),
				'label' => $ip,
			];
		}

		$payload = [
			'area' => [
				'labels' => $time_labels,
				'datasets' => [
					[ 'label' => 'WAF',         'data' => $mod_waf ],
					[ 'label' => 'API Shield',  'data' => $mod_api ],
					[ 'label' => 'Bot Control', 'data' => $mod_bot ],
					[ 'label' => 'DDoS',        'data' => $mod_ddos ],
					[ 'label' => 'Attack Story','data' => $mod_story ],
				],
			],

			'categories' => [
				'labels' => array_keys( $top_cats ),
				'values' => array_values( $top_cats ),
			],

			'pressure' => [
				'labels' => array_keys( $module_score ),
				'values' => array_values( $module_score ),
			],

			'endpoints' => [
				'labels' => array_keys( $top_endpoints ),
				'values' => array_values( $top_endpoints ),
			],

			'methods' => [
				'labels' => array_keys( $m_top ),
				'values' => array_values( $m_top ),
			],

			'actions_line' => [
				'labels' => $time_labels,
				'blocks' => $line_blocks,
				'rate_limits' => $line_rates,
				'challenges' => $line_chals,
			],

			'ip_scatter' => [
				'points' => $scatter,
			],

			'offenders' => [
				'points' => $bubbles,
			],

			'meta' => [
				'hours' => $hours,
				'since_gmt' => $since,
				'rows_used' => count( $rows ),
			],
		];

		set_transient( $cache_key, $payload, 60 );
		return $payload;
	}

}