<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Logger {

    public static function table_name() : string {
        global $wpdb;
        return $wpdb->prefix . 'aegiswaf_logs';
    }

    public static function activate() : void {
        global $wpdb;
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $charset = $wpdb->get_charset_collate();
        $table = self::table_name();

        $sql = "CREATE TABLE $table (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            event_time DATETIME NOT NULL,
            route VARCHAR(255) NOT NULL,
            method VARCHAR(16) NOT NULL,
            category VARCHAR(80) NOT NULL,
            action_taken VARCHAR(40) NOT NULL,
            details LONGTEXT NULL,
            ip VARCHAR(64) NULL,
            ua VARCHAR(255) NULL,
            PRIMARY KEY  (id),
            KEY event_time (event_time),
            KEY route (route),
            KEY category (category),
            KEY action_taken (action_taken),
            KEY ip (ip)
        ) $charset;";

        dbDelta( $sql );
    }

	public static function create_table() : void {
		self::activate();
	}


	public static function ensure_table() : void {
		global $wpdb;

		$table = self::table_name();

		$exists = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->prepare(
				'SHOW TABLES LIKE %s',
				$table
			)
		);
		if ( $exists !== $table ) {
			self::activate();
		}
	}

    public static function log( string $route, string $method, string $category, string $action_taken, array $details = [] ) : void {
        global $wpdb;

        $table = self::table_name();

        if ( method_exists( __CLASS__, 'ensure_table' ) ) {
            self::ensure_table();
        }

        $ip = '';
        if ( isset( $details['ip'] ) && is_string( $details['ip'] ) ) {
            $ip = $details['ip'];
        } elseif ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
            $ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
        }

        $ua = '';
        if ( isset( $details['ua'] ) && is_string( $details['ua'] ) ) {
            $ua = $details['ua'];
        } elseif ( isset( $_SERVER['HTTP_USER_AGENT'] ) ) {
            $ua = sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
        }

        $ok = $wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $table,
            [
                'event_time'   => gmdate( 'Y-m-d H:i:s' ),
                'route'        => substr( (string) $route, 0, 255 ),
                'method'       => substr( (string) $method, 0, 16 ),
                'category'     => substr( (string) $category, 0, 80 ),
                'action_taken' => substr( (string) $action_taken, 0, 40 ),
                'details'      => wp_json_encode( $details ),
                'ip'           => substr( (string) $ip, 0, 64 ),
                'ua'           => substr( (string) $ua, 0, 255 ),
            ],
            [ '%s','%s','%s','%s','%s','%s','%s','%s' ]
        );

		if ( $ok ) {
            // bust cached log counts/lists
            wp_cache_delete( 'aegiswaf_logs_count_all', 'aegiswaf' );
            // filtered caches are keyed by args hash; easiest is to bump group via delete_group-like; not available.
            // keep it minimal: short TTL already (30s).

			try {
				$settings = AegisWAF_Storage::get_settings();
				$alerts   = $settings['logs']['alerts'] ?? [];

				if ( ! empty( $alerts ) ) {
					$details_str = wp_json_encode( $details );

					$full_line = implode( ' | ', [
						'Time(UTC): ' . gmdate( 'Y-m-d H:i:s' ),
						'Route: ' . $route,
						'Method: ' . $method,
						'Category: ' . $category,
						'Action: ' . $action_taken,
						'IP: ' . (string) $ip,
						'Details: ' . $details_str,
					] );

					foreach ( $alerts as $alert ) {
						$kw_raw = (string) ( $alert['keywords'] ?? '' );
						if ( $kw_raw === '' ) { continue; }

						$keywords = preg_split( '/[\r\n,]+/', $kw_raw );
						$matched  = false;

						foreach ( $keywords as $kw ) {
							$kw = trim( (string) $kw );
							if ( $kw === '' ) { continue; }

							if ( stripos( $full_line, $kw ) !== false ) {
								$matched = true;
								break;
							}
						}

						if ( ! $matched ) { continue; }

						$emails_raw = trim( (string) ( $alert['emails'] ?? '' ) );
						if ( $emails_raw === '' ) {
							$to = get_option( 'admin_email' );
						} else {
							// allow comma or semicolon separated
							$parts = preg_split( '/[;,]+/', $emails_raw );
							$parts = array_filter( array_map( 'trim', (array) $parts ) );
							$parts = array_map( 'sanitize_email', $parts );
							$to    = $parts ? $parts : get_option( 'admin_email' );
						}

						wp_mail(
							$to,
							'[AegisWAF Alert] Keyword Matched',
							$full_line
						);

						break;
					}
				}
			} catch ( Throwable $e ) {
				if ( defined('WP_DEBUG') && WP_DEBUG ) { // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log( '[AegisWAF] Alert email failed: ' . $e->getMessage() ); }
			}
		}

		if ( ! $ok ) {
			$err = isset( $wpdb->last_error ) ? (string) $wpdb->last_error : '';
			$qry = isset( $wpdb->last_query ) ? (string) $wpdb->last_query : '';
			if ( defined('WP_DEBUG') && WP_DEBUG ) { // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log( '[AegisWAF] Logger insert failed: ' . $err . ' | last_query=' . $qry ); }
		}
    }

    public static function count_all() : int {
        global $wpdb;
        $table = self::table_name();
        $cache_key = 'aegiswaf_logs_count_all';
        $cached = wp_cache_get( $cache_key, 'aegiswaf' );
        if ( false !== $cached ) { return (int) $cached; }
        $count = (int) $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(1) FROM %i', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        wp_cache_set( $cache_key, $count, 'aegiswaf', 30 );
        return $count;
    }
public static function count_filtered( array $args ) : int {
	global $wpdb;

	self::ensure_table();
	$table = self::table_name();

	$q        = isset( $args['q'] ) ? trim( (string) $args['q'] ) : '';
	$route    = isset( $args['route'] ) ? trim( (string) $args['route'] ) : '';
	$method   = isset( $args['method'] ) ? trim( (string) $args['method'] ) : '';
	$category = isset( $args['category'] ) ? trim( (string) $args['category'] ) : '';
	$action   = isset( $args['action'] ) ? trim( (string) $args['action'] ) : '';
	$ip       = isset( $args['ip'] ) ? trim( (string) $args['ip'] ) : '';
	$df       = isset( $args['date_from'] ) ? trim( (string) $args['date_from'] ) : '';
	$dt       = isset( $args['date_to'] ) ? trim( (string) $args['date_to'] ) : '';
	$alerts   = ! empty( $args['alerts_only'] ) ? 1 : 0;

	$q_like     = $q !== '' ? '%' . $wpdb->esc_like( $q ) . '%' : '%';
	$route_like = $route !== '' ? '%' . $wpdb->esc_like( $route ) . '%' : '%';
	$df_dt      = $df !== '' ? $df . ' 00:00:00' : '';
	$dt_dt      = $dt !== '' ? $dt . ' 23:59:59' : '';

	$count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->prepare(
			"SELECT COUNT(1)
			 FROM %i
			 WHERE
				( %s = '' OR (route LIKE %s OR category LIKE %s OR action_taken LIKE %s OR ip LIKE %s OR ua LIKE %s OR details LIKE %s) )
				AND ( %s = '' OR route LIKE %s )
				AND ( %s = '' OR method = %s )
				AND ( %s = '' OR category = %s )
				AND ( %s = '' OR action_taken = %s )
				AND ( %s = '' OR ip = %s )
				AND ( %s = '' OR event_time >= %s )
				AND ( %s = '' OR event_time <= %s )
				AND ( %d = 0 OR (action_taken IN ('block','challenge','rate_limit','error') OR category IN ('error','warning','alert','attack','malware')) )",
			$table,
			$q,
			$q_like, $q_like, $q_like, $q_like, $q_like, $q_like,
			$route, $route_like,
			$method, $method,
			$category, $category,
			$action, $action,
			$ip, $ip,
			$df, $df_dt,
			$dt, $dt_dt,
			$alerts
		)
	);

	return max( 0, $count );
}

public static function fetch_filtered( int $limit, int $offset, array $args ) : array {
	global $wpdb;

	self::ensure_table();
	$table = self::table_name();

	$limit  = max( 1, min( 500, $limit ) );
	$offset = max( 0, $offset );

	$q        = isset( $args['q'] ) ? trim( (string) $args['q'] ) : '';
	$route    = isset( $args['route'] ) ? trim( (string) $args['route'] ) : '';
	$method   = isset( $args['method'] ) ? trim( (string) $args['method'] ) : '';
	$category = isset( $args['category'] ) ? trim( (string) $args['category'] ) : '';
	$action   = isset( $args['action'] ) ? trim( (string) $args['action'] ) : '';
	$ip       = isset( $args['ip'] ) ? trim( (string) $args['ip'] ) : '';
	$df       = isset( $args['date_from'] ) ? trim( (string) $args['date_from'] ) : '';
	$dt       = isset( $args['date_to'] ) ? trim( (string) $args['date_to'] ) : '';
	$alerts   = ! empty( $args['alerts_only'] ) ? 1 : 0;

	$q_like     = $q !== '' ? '%' . $wpdb->esc_like( $q ) . '%' : '%';
	$route_like = $route !== '' ? '%' . $wpdb->esc_like( $route ) . '%' : '%';
	$df_dt      = $df !== '' ? $df . ' 00:00:00' : '';
	$dt_dt      = $dt !== '' ? $dt . ' 23:59:59' : '';

	$cache_key = 'aegiswaf_logs_' . md5( wp_json_encode( [ $limit, $offset, $q, $route, $method, $category, $action, $ip, $df, $dt, $alerts ] ) );
	$cached    = wp_cache_get( $cache_key, 'aegiswaf' );
	if ( is_array( $cached ) ) {
		return $cached;
	}

	$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->prepare(
			"SELECT *
			 FROM %i
			 WHERE
				( %s = '' OR (route LIKE %s OR category LIKE %s OR action_taken LIKE %s OR ip LIKE %s OR ua LIKE %s OR details LIKE %s) )
				AND ( %s = '' OR route LIKE %s )
				AND ( %s = '' OR method = %s )
				AND ( %s = '' OR category = %s )
				AND ( %s = '' OR action_taken = %s )
				AND ( %s = '' OR ip = %s )
				AND ( %s = '' OR event_time >= %s )
				AND ( %s = '' OR event_time <= %s )
				AND ( %d = 0 OR (action_taken IN ('block','challenge','rate_limit','error') OR category IN ('error','warning','alert','attack','malware')) )
			 ORDER BY id DESC
			 LIMIT %d OFFSET %d",
			$table,
			$q,
			$q_like, $q_like, $q_like, $q_like, $q_like, $q_like,
			$route, $route_like,
			$method, $method,
			$category, $category,
			$action, $action,
			$ip, $ip,
			$df, $df_dt,
			$dt, $dt_dt,
			$alerts,
			$limit,
			$offset
		),
		ARRAY_A
	);

	if ( ! is_array( $rows ) ) {
		$rows = [];
	}

	wp_cache_set( $cache_key, $rows, 'aegiswaf', 30 );
	return $rows;
}

public static function fetch( int $limit, int $offset ) : array {
    return self::fetch_filtered( $limit, $offset, [] );
}

/**
 * Build a safe WHERE clause (no dynamic identifier interpolation).
 *
 * @param array $args Filter args.
 * @return array{0:string,1:array} Tuple of WHERE SQL and args.
 */
private static function build_filters_where( array $args ) : array {
    global $wpdb;

    $where_sql  = 'WHERE 1=1';
    $where_args = [];

    $q = isset( $args['q'] ) ? trim( (string) $args['q'] ) : '';
    if ( $q !== '' ) {
        $like = '%' . $wpdb->esc_like( $q ) . '%';
        $where_sql .= " AND (route LIKE %s OR category LIKE %s OR action_taken LIKE %s OR ip LIKE %s OR ua LIKE %s OR details LIKE %s)";
        $where_args = array_merge( $where_args, [ $like, $like, $like, $like, $like, $like ] );
    }

    $route = isset( $args['route'] ) ? trim( (string) $args['route'] ) : '';
    if ( $route !== '' ) {
        $like = '%' . $wpdb->esc_like( $route ) . '%';
        $where_sql .= ' AND route LIKE %s';
        $where_args[] = $like;
    }

    $method = isset( $args['method'] ) ? trim( (string) $args['method'] ) : '';
    if ( $method !== '' ) {
        $where_sql .= ' AND method = %s';
        $where_args[] = $method;
    }

    $category = isset( $args['category'] ) ? trim( (string) $args['category'] ) : '';
    if ( $category !== '' ) {
        $where_sql .= ' AND category = %s';
        $where_args[] = $category;
    }

    $action = isset( $args['action'] ) ? trim( (string) $args['action'] ) : '';
    if ( $action !== '' ) {
        $where_sql .= ' AND action_taken = %s';
        $where_args[] = $action;
    }

    $ip = isset( $args['ip'] ) ? trim( (string) $args['ip'] ) : '';
    if ( $ip !== '' ) {
        $where_sql .= ' AND ip = %s';
        $where_args[] = $ip;
    }

    $df = isset( $args['date_from'] ) ? trim( (string) $args['date_from'] ) : '';
    if ( $df !== '' ) {
        $where_sql .= ' AND event_time >= %s';
        $where_args[] = $df . ' 00:00:00';
    }

    $dt = isset( $args['date_to'] ) ? trim( (string) $args['date_to'] ) : '';
    if ( $dt !== '' ) {
        $where_sql .= ' AND event_time <= %s';
        $where_args[] = $dt . ' 23:59:59';
    }

    if ( ! empty( $args['alerts_only'] ) ) {
        $where_sql .= " AND (action_taken IN ('block','challenge','rate_limit','error') OR category IN ('error','warning','alert','attack','malware'))";
    }

    return [ $where_sql, $where_args ];
}
}

add_action( 'plugins_loaded', function () {
    if ( class_exists( 'AegisWAF_Logger' ) ) {
        if ( method_exists( 'AegisWAF_Logger', 'ensure_table' ) ) {
            AegisWAF_Logger::ensure_table();
        } else {
            AegisWAF_Logger::activate();
        }
    }
}, 1 );


