<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_Storage {

    private const OPT_SETTINGS = 'aegiswaf_settings';
    private const OPT_API_POLICIES = 'aegiswaf_api_policies';
    private const OPT_WAF_RULES = 'aegiswaf_waf_rules';
    private const OPT_INTEL = 'aegiswaf_intel';

    public static function default_settings() : array {
        return [
            'waf' => [
                'enabled' => true,
                'mode' => 'log',
                'enable_owasp' => true,
                'enable_normalization' => true,
            ],
            'api' => [
                'enabled' => true,
                'default_action' => 'inspect',
                'max_body_kb' => 512,
            ],

            'wp_protect' => [
                'enabled' => true,
                'rest_protect' => true,
                'rest_only_unauth' => true,

                'rest_block_user_enum' => true,
                'rest_require_write_auth' => false,
                'rest_require_json' => false,
                'rest_max_body_bytes' => 0,

                'rest_cors_allowlist' => '',

                'rest_api_key_enabled' => false,
                'rest_api_key_header'  => 'x-aegis-key',
                'rest_api_key_value'   => '',

                'rest_allowlist' => '',

                'window_seconds_rest' => 60,
                'thresholds_rest' => [
                    'challenge_at'  => 0,
                    'rate_limit_at' => 0,
                    'block_at'      => 0,
                ],

				'rest_route_overrides'      => '',   // one rule per line
				'rest_route_auto_profiles'  => true, // enable tagging presets when category/profile not provided
            ],
            'bot' => [
                'enabled' => true,
                'block_bad_ua' => true,
                'bad_ua_list' => "curl
wget
python-requests
httpclient
",
                'per_path_limits' => [],
                'default_limit_per_min' => 180,
            ],
            'logs' => [
                'per_page' => 50,
                'retention_days' => 30, // PRO
            ],
            'rules' => [
                'enabled' => true,
            ],
            'managed_rules' => [
                'enabled' => true,
                'mode_free' => 'log',
                'mode_pro'  => 'block',
                'sensitivity' => 'balanced', // balanced|strict
                'inspect_headers' => true,
                'inspect_cookies' => true,
                'inspect_body' => true,
                'max_body_bytes' => 65536,
                'thresholds' => [
                    'sqli' => 1,
                    'xss' => 1,
                    'path_traversal' => 1,
                    'rce' => 1,
                    'cmd_injection' => 1,
                    'ssrf' => 1,
                    'file_upload' => 1,
                    'heuristic' => 50,
                ],
                'challenge' => [
                    'enabled' => true,
                    'ttl_seconds' => 1800,
                ],
                'heuristics' => [
                    'enabled' => true,
                    'score_threshold' => 40,
                    'pct_threshold' => 8,
                    'meta_threshold' => 12,
                    'token_len_threshold' => 900,
                ],
                'categories' => [
                    'sqli' => true,
                    'xss'  => true,
                    'rce'  => false,
                    'path_traversal' => true,
                    'file_upload' => false,
                    'cmd_injection' => false,
                    'ssrf' => false,
                ],
            ],
            'intel' => [
                'enabled' => true,
                'behavioral_threshold' => 80, // PRO
                'geo_provider' => 'none', // none|cloudflare|maxmind_ext
                'maxmind_mmdb_path' => '' // PRO
            
            ],
        ];
    }

    public static function get_settings() : array {
        $settings = get_option( self::OPT_SETTINGS, [] );
        if ( ! is_array( $settings ) ) { $settings = []; }
        return array_replace_recursive( self::default_settings(), $settings );
    }

    public static function update_settings( array $settings ) : bool {
        return update_option( self::OPT_SETTINGS, $settings, false );
    }

    public static function get_api_policies() : array {
        $policies = get_option( self::OPT_API_POLICIES, [] );
        return is_array( $policies ) ? $policies : [];
    }

    public static function update_api_policies( array $policies ) : bool {
        return update_option( self::OPT_API_POLICIES, $policies, false );
    }

    public static function new_policy_id() : string {
        return 'p_' . wp_generate_password( 10, false, false );
    }

    public static function get_waf_rules() : array {
        $rules = get_option( self::OPT_WAF_RULES, [] );
        return is_array( $rules ) ? $rules : [];
    }

    public static function update_waf_rules( array $rules ) : bool {
        return update_option( self::OPT_WAF_RULES, $rules, false );
    }

    public static function get_intel() : array {
        $intel = get_option( self::OPT_INTEL, [] );
        return is_array( $intel ) ? $intel : [];
    }

    public static function update_intel( array $intel ) : bool {
        return update_option( self::OPT_INTEL, $intel, false );
    }
}

