<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_DDoS_Rules {

    public static function group_for_front( string $path ) : string {
        $path = self::normalize_path( $path );

        if ( $path === '/wp-login.php' ) { return 'wp_login'; }
        if ( $path === '/xmlrpc.php' ) { return 'xmlrpc'; }
		if ( $path === '/wp-admin/admin-ajax.php' ) { return 'admin_ajax'; }
		if ( $path === '/wp-cron.php' ) { return 'wp_cron'; }
        if ( strpos( $path, '/wp-admin' ) === 0 ) { return 'wp_admin'; }
        if ( self::is_search_request() ) { return 'search'; }
        if ( self::is_cache_bypass_request() ) { return 'cache_bypass'; }

        return 'global';
    }

    public static function group_for_rest( string $route ) : string {
        $route = self::normalize_path( $route );

        if ( self::is_cache_bypass_request() ) { return 'cache_bypass'; }
        return 'rest';
    }

    public static function thresholds_for( string $group, array $ddos_settings ) : array {
        $groups = is_array( $ddos_settings['groups'] ?? null ) ? $ddos_settings['groups'] : [];
        $t = is_array( $groups[ $group ] ?? null ) ? $groups[ $group ] : ( $groups['global'] ?? [] );

        return [
            'challenge' => max( 1, (int) ( $t['challenge'] ?? 300 ) ),
            'rate_limit' => max( 1, (int) ( $t['rate_limit'] ?? 420 ) ),
            'block' => max( 1, (int) ( $t['block'] ?? 600 ) ),
        ];
    }

    private static function normalize_path( string $path ) : string {
        $path = (string) wp_parse_url( $path, PHP_URL_PATH );
        if ( $path === '' ) { $path = '/'; }
        if ( $path[0] !== '/' ) { $path = '/' . $path; }
        return $path;
    }

    private static function is_search_request() : bool {
        // Read-only classification: no state change. Avoid direct superglobal usage to satisfy scanners.
        $s = filter_input( INPUT_GET, 's', FILTER_UNSAFE_RAW );
        if ( null === $s || false === $s ) { return false; }
        $s = sanitize_text_field( (string) $s );
        return ( $s !== '' );
    }

    private static function is_cache_bypass_request() : bool {
        if ( ! isset( $_SERVER['QUERY_STRING'] ) ) { return false; }
        $qs = sanitize_text_field( wp_unslash( (string) $_SERVER['QUERY_STRING'] ) );
        if ( $qs === '' ) { return false; }

        if ( substr_count( $qs, '&' ) >= 6 ) { return true; }

        if ( preg_match( '/(?:^|[&=])[a-z0-9]{16,}(?:$|[&=])/i', $qs ) ) { return true; }

        if ( preg_match( '/(?:^|[&=])(\d{10}|\d{13})(?:$|[&=])/', $qs ) ) { return true; }

        return false;
    }
}
