<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_DDoS_Storage {

    public static function defaults() : array {
        return [
            'enabled' => false,
            'ignore_logged_in' => true,
            'cooldown_seconds' => 300,

            'allowlist' => '',
			'endpoint_allowlist' => "/wp-json/chue-license/*\n/chue-license/*",

            'groups' => [
                'global' => [ 'challenge' => 300, 'rate_limit' => 420, 'block' => 600 ],
                'wp_login' => [ 'challenge' => 20, 'rate_limit' => 35, 'block' => 60 ],
                'xmlrpc' => [ 'challenge' => 5, 'rate_limit' => 8, 'block' => 12 ],
                'wp_admin' => [ 'challenge' => 80, 'rate_limit' => 120, 'block' => 180 ],
				'admin_ajax' => [ 'challenge' => 120, 'rate_limit' => 180, 'block' => 260 ],
				'wp_cron' => [ 'challenge' => 25, 'rate_limit' => 40, 'block' => 70 ],
                'rest' => [ 'challenge' => 160, 'rate_limit' => 220, 'block' => 320 ],
                'search' => [ 'challenge' => 35, 'rate_limit' => 55, 'block' => 90 ],
                'cache_bypass' => [ 'challenge' => 40, 'rate_limit' => 60, 'block' => 90 ],
            ],

				'enable_cache_bypass_detection' => true,
				'enable_search_detection' => true,

				'emergency_mode' => false,
				'emergency_multiplier' => 0.50,
				'emergency_minimums' => [ 'challenge' => 3, 'rate_limit' => 5, 'block' => 8 ],

        ];
    }

    public static function get() : array {
        $s = AegisWAF_Storage::get_settings();
        $ddos = is_array( $s['ddos'] ?? null ) ? $s['ddos'] : [];
        return array_replace_recursive( self::defaults(), $ddos );
    }

    public static function update( array $ddos ) : bool {
        $s = AegisWAF_Storage::get_settings();
        $s['ddos'] = $ddos;
        return AegisWAF_Storage::update_settings( $s );
    }

    public static function allowlist_lines( string $raw ) : array {
        $raw = str_replace( "\r", "\n", $raw );
        $lines = array_filter( array_map( 'trim', explode( "\n", $raw ) ) );
        $out = [];

        foreach ( $lines as $line ) {
            if ( $line === '' ) { continue; }
            $line = preg_replace( '/\s*[#;].*$/', '', $line );
            $line = trim( (string) $line );
            if ( $line === '' ) { continue; }
            $out[] = $line;
        }

        return array_values( array_unique( $out ) );
    }

    public static function ip_allowlisted( string $ip, string $allowlist_raw ) : bool {
        if ( $ip === '' ) { return false; }

        $items = self::allowlist_lines( $allowlist_raw );
        if ( empty( $items ) ) { return false; }

        foreach ( $items as $item ) {
            if ( filter_var( $item, FILTER_VALIDATE_IP ) && $item === $ip ) {
                return true;
            }

            if ( strpos( $item, '/' ) !== false ) {
                if ( self::ip_in_cidr( $ip, $item ) ) {
                    return true;
                }
            }
        }

        return false;
    }

    private static function ip_in_cidr( string $ip, string $cidr ) : bool {
        $cidr = trim( $cidr );
        if ( $cidr === '' || strpos( $cidr, '/' ) === false ) { return false; }

        [ $subnet, $bits ] = array_pad( explode( '/', $cidr, 2 ), 2, '' );
        $subnet = trim( (string) $subnet );
        $bits = (int) trim( (string) $bits );

        if ( ! filter_var( $subnet, FILTER_VALIDATE_IP ) ) { return false; }
        if ( ! filter_var( $ip, FILTER_VALIDATE_IP ) ) { return false; }

        $ip_bin = @inet_pton( $ip );
        $subnet_bin = @inet_pton( $subnet );
        if ( $ip_bin === false || $subnet_bin === false ) { return false; }

        $len = strlen( $ip_bin ); // 4 for v4, 16 for v6
        $max_bits = $len * 8;

        if ( $bits < 0 || $bits > $max_bits ) { return false; }

        $bytes = intdiv( $bits, 8 );
        $remainder = $bits % 8;

        if ( $bytes > 0 ) {
            if ( substr( $ip_bin, 0, $bytes ) !== substr( $subnet_bin, 0, $bytes ) ) {
                return false;
            }
        }

        if ( $remainder === 0 ) { return true; }

        $mask = chr( ( 0xFF << ( 8 - $remainder ) ) & 0xFF );
        return ( $ip_bin[$bytes] & $mask ) === ( $subnet_bin[$bytes] & $mask );
    }
	public static function endpoint_allowlisted( string $path, string $raw_patterns ) : bool {
		$path = self::normalize_path_only( $path );
		$patterns = self::allowlist_lines( $raw_patterns );
		if ( empty( $patterns ) ) { return false; }

		foreach ( $patterns as $pat ) {
			$pat = trim( (string) $pat );
			if ( $pat === '' ) { continue; }

			// wildcard support
			if ( strpos( $pat, '*' ) !== false ) {
				$re = '#^' . str_replace( '\*', '.*', preg_quote( $pat, '#' ) ) . '$#i';
				if ( preg_match( $re, $path ) ) { return true; }
			} else {
				if ( strtolower( $pat ) === strtolower( $path ) ) { return true; }
			}
		}
		return false;
	}

	private static function normalize_path_only( string $path ) : string {
		$path = (string) wp_parse_url( $path, PHP_URL_PATH );
		if ( $path === '' ) { $path = '/'; }
		if ( $path[0] !== '/' ) { $path = '/' . $path; }
		return $path;
	}

}
