<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

class AegisWAF_License_Manager {

    const OPTION_KEY       = 'aegiswaf_license';
    const TRANSIENT_STATUS = 'aegiswaf_license_status_checked';
    const CHECK_INTERVAL   = 4 * HOUR_IN_SECONDS;

    protected string $api_base_url = 'https://aegisify.com/wp-json/chue-license/v1';

    protected string $product_slug = 'aegiswaf';

    public function init() : void {
        add_action( 'admin_init', [ $this, 'maybe_check_license_status' ] );
        add_filter( 'aegiswaf_is_pro', [ $this, 'filter_is_pro_active' ] );
    }

    public function get_license_data() : array {
        $data = get_option( self::OPTION_KEY, [] );
        if ( ! is_array( $data ) ) { $data = []; }

		return wp_parse_args( $data, [
			'license_key'   => '',
			'license_email' => '',
			'server_email'  => '', // canonical email returned by CLM
			'status'        => 'inactive',
			'plan'          => '',
			'expires_at'    => '',
			'last_checked'  => 0,
			'message'       => '',
		] );
    }

    public function update_license_data( array $data ) : void {
        update_option( self::OPTION_KEY, array_merge( $this->get_license_data(), $data ), false );
    }

	function is_pro_active() : bool {

			static $memo = null;
			if ( $memo !== null ) {
				return (bool) $memo;
			}

			if ( function_exists( 'aegisify_is_pro' ) && did_action( 'init' ) ) {
				$suite_is_pro = (bool) aegisify_is_pro( false ); // IMPORTANT: false (no force)
				if ( $suite_is_pro ) {
					$memo = true;
					return true;
				}
			}

			$data = $this->get_license_data();
			$status = (string) ( $data['status'] ?? 'inactive' );

			if ( $status !== 'active' ) {
				$memo = false;
				return false;
			}

			if ( $status === 'email_mismatch' ) {
				$memo = false;
				return false;
			}

			if ( ! empty( $data['expires_at'] ) ) {
				$ts = strtotime( (string) $data['expires_at'] );
				if ( $ts && $ts < time() ) {
					$memo = false;
					return false;
				}
			}

			$memo = true;
			return true;
		}

    public function filter_is_pro_active( $active ) : bool {
        return $this->is_pro_active();
    }

    public function activate_license( string $license_key, string $email ) : array {
        $license_key = sanitize_text_field( $license_key );
        $email       = sanitize_email( $email );

        if ( empty( $license_key ) || empty( $email ) ) {
            return [ 'success' => false, 'message' => 'License key and email are required.' ];
        }

        $body = [
            'license_key' => $license_key,
            'email'       => $email,
            'site_url'    => home_url(),
            'plugin_slug' => 'aegiswaf',
            'version'     => defined( 'AEGISWAF_VERSION' ) ? AEGISWAF_VERSION : '',
        ];

        $response = $this->remote_post( '/' . $this->product_slug . '/activate', $body );

        if ( is_wp_error( $response ) ) {
            return [ 'success' => false, 'message' => $response->get_error_message() ];
        }

        $data = $this->parse_api_response( $response );

		if ( isset( $data['email'] ) && is_string( $data['email'] ) ) {
			$server_email = sanitize_email( $data['email'] );
			if ( $server_email !== '' && strtolower( $server_email ) !== strtolower( $email ) ) {
				$this->update_license_data( [
					'license_key'   => $license_key,
					'license_email' => $email,
					'server_email'  => $server_email,
					'status'        => 'email_mismatch',
					'plan'          => '',
					'expires_at'    => '',
					'message'       => 'Email mismatch: the email returned by the license server does not match the email you entered.',
					'last_checked'  => time(),
				] );

				delete_transient( self::TRANSIENT_STATUS );

				return [
					'success' => false,
					'message' => 'Email mismatch: please enter the email address that is registered to this license key.',
				];
			}
		}

        $this->update_license_data( [
            'license_key'   => $license_key,
            'license_email' => $email,
            'status'        => $data['status'] ?? 'inactive',
            'plan'          => $data['plan'] ?? '',
            'expires_at'    => $data['expires_at'] ?? '',
            'message'       => $data['message'] ?? '',
            'last_checked'  => time(),
        ] );

        delete_transient( self::TRANSIENT_STATUS );

        return [
            'success' => ( $data['status'] ?? '' ) === 'active',
            'message' => $data['message'] ?? 'License activation completed.',
        ];
    }

    public function deactivate_license() : array {
        $data = $this->get_license_data();

        if ( empty( $data['license_key'] ) ) {
            return [ 'success' => false, 'message' => 'No license key to deactivate.' ];
        }

        $response = $this->remote_post(
            '/' . $this->product_slug . '/deactivate',
            [
                'license_key' => $data['license_key'],
                'email'       => $data['license_email'],
                'site_url'    => home_url(),
            ]
        );

        if ( is_wp_error( $response ) ) {
            return [ 'success' => false, 'message' => $response->get_error_message() ];
        }

        $this->update_license_data( [
            'status'       => 'inactive',
            'plan'         => '',
            'expires_at'   => '',
            'last_checked' => time(),
            'message'      => 'License deactivated.',
			'server_email' => isset( $data['server_email'] ) ? sanitize_email( (string) $data['server_email'] ) : '',
        ] );

        delete_transient( self::TRANSIENT_STATUS );

        return [ 'success' => true, 'message' => 'License deactivated.' ];
    }

    public function maybe_check_license_status() : void {
        if ( get_transient( self::TRANSIENT_STATUS ) ) {
            return;
        }

        $data = $this->get_license_data();
        if ( empty( $data['license_key'] ) || empty( $data['license_email'] ) ) {
            return;
        }

        $response = $this->remote_post(
            '/' . $this->product_slug . '/status',
            [
                'license_key' => $data['license_key'],
                'email'       => $data['license_email'],
                'site_url'    => home_url(),
            ]
        );

        if ( ! is_wp_error( $response ) ) {
            $api = $this->parse_api_response( $response );

		if ( isset( $api['email'] ) && is_string( $api['email'] ) ) {
			$server_email = sanitize_email( $api['email'] );
			$local_email  = sanitize_email( (string) ( $data['license_email'] ?? '' ) );

			if ( $server_email !== '' && $local_email !== '' && strtolower( $server_email ) !== strtolower( $local_email ) ) {
				$this->update_license_data( [
					'server_email'  => $server_email,
					'status'        => 'email_mismatch',
					'plan'          => '',
					'expires_at'    => '',
					'message'       => 'Email mismatch detected during license validation.',
					'last_checked'  => time(),
				] );

				set_transient( self::TRANSIENT_STATUS, 1, self::CHECK_INTERVAL );
				return;
			}
		}

            $this->update_license_data( [
                'status'       => $api['status'] ?? $data['status'],
                'plan'         => $api['plan'] ?? $data['plan'],
                'expires_at'   => $api['expires_at'] ?? $data['expires_at'],
                'message'      => $api['message'] ?? '',
                'last_checked' => time(),
            ] );
        }

        set_transient( self::TRANSIENT_STATUS, 1, self::CHECK_INTERVAL );
    }

    protected function remote_post( string $path, array $body ) {
        return wp_remote_post(
            trailingslashit( $this->api_base_url ) . ltrim( $path, '/' ),
            [ 'timeout' => 15, 'body' => $body ]
        );
    }

    protected function parse_api_response( array $response ) : array {
        $code = (int) wp_remote_retrieve_response_code( $response );
        $body = (string) wp_remote_retrieve_body( $response );

        if ( 200 !== $code || $body === '' ) {
            return [ 'status' => 'error', 'message' => 'Invalid license server response.' ];
        }

        $data = json_decode( $body, true );
        return is_array( $data ) ? $data : [ 'status' => 'error', 'message' => 'Malformed response.' ];
    }
}
